/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 *   Gnome Apt frontend
 *
 *   Copyright (C) 1998 Havoc Pennington <hp@pobox.com>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "index.h"
#include "pkgutil.h"

static void
clist_select_row(GtkWidget* clist, 
                 gint row,
                 gint col,
                 GdkEvent* event,
                 gpointer data)
{
  GAptPkgTree* tree = static_cast<GAptPkgTree*>(data);

  g_return_if_fail(tree != 0);

  g_return_if_fail(row >= 0);

  pkgCache::Package* p = 
    static_cast<pkgCache::Package*>(gtk_clist_get_row_data(GTK_CLIST(clist),
                                                           row));

  g_return_if_fail(p != 0);

  tree->move_selection(p);

  return;
}

Index::Index(Indexes* indexes, GAptPkgTree* tree, Filter* filter)
  : indexes_(indexes), tree_(tree), 
    cache_(tree_->cache()), filter_(filter)
{
  g_return_if_fail(cache_ != 0);
  g_return_if_fail(filter_ != 0);
  g_return_if_fail(indexes_ != 0);

  indexes_->add(this);

  gnome_apt_cache_file()->add_view(this); // results in a set_cache
  filter_->add_view(this); 

  // not using GnomeDialog because this is a persistent thing,
  //  not a transient dialog
  dialog_ = gtk_window_new(GTK_WINDOW_TOPLEVEL);

  gtk_window_set_title(GTK_WINDOW(dialog_), _("Index"));

  gtk_window_set_policy(GTK_WINDOW(dialog_), TRUE, TRUE, FALSE);
  
  gtk_window_set_default_size(GTK_WINDOW(dialog_), 180, 300);

  gtk_signal_connect(GTK_OBJECT(dialog_),
                     "destroy",
                     GTK_SIGNAL_FUNC(dialog_close),
                     this);

  GtkWidget* vbox = gtk_vbox_new(0, FALSE);

  gtk_container_set_border_width(GTK_CONTAINER(vbox),
                                 GNOME_PAD);

  gtk_container_add(GTK_CONTAINER(dialog_), vbox);

  clist_ = gtk_clist_new(1);

  gtk_clist_set_selection_mode(GTK_CLIST(clist_),
                               GTK_SELECTION_SINGLE);

  gtk_signal_connect(GTK_OBJECT(clist_),
                     "select_row",
                     GTK_SIGNAL_FUNC(clist_select_row),
                     tree_);

  GtkWidget* sw = gtk_scrolled_window_new(NULL,NULL);

  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
                                 GTK_POLICY_NEVER,
                                 GTK_POLICY_AUTOMATIC);

  gtk_container_add(GTK_CONTAINER(sw), clist_);

  gtk_box_pack_start(GTK_BOX(vbox),
                     sw,
                     TRUE, TRUE, GNOME_PAD_SMALL);

  gtk_widget_show_all(dialog_);
}

Index::~Index()
{
  // Indexes will remove us when it deletes us

  if (dialog_)
    {
      gtk_widget_destroy(dialog_);
      dialog_ = 0;
      clist_ = 0;
    }

  GAptCacheFile* cf = gnome_apt_cache_file();
  if (cf != 0) // cf is 0 when the app closes.
    cf->remove_view(this);

  if (filter_) 
    filter_->remove_view(this);
}

void 
Index::add(pkgCache::Package* package)
{
#ifdef GNOME_ENABLE_DEBUG
  vector<pkgCache::Package*>::iterator i = packages_.begin();
  while (i != packages_.end())
    {
      if (*i == package) g_warning("Package added twice!");
      ++i;
    }
#endif
  g_return_if_fail(cache_ != 0);
  g_return_if_fail(dialog_ != 0);

  packages_.push_back(package);

  add_to_clist(package);
}

void
Index::add(vector<pkgCache::Package*> & packages)
{
  if (clist_ != 0) gtk_clist_freeze(GTK_CLIST(clist_));

  vector<pkgCache::Package*>::iterator i = packages.begin();
  while (i != packages.end())
    {
      add(*i);
      ++i;
    }

  if (clist_ != 0) gtk_clist_thaw(GTK_CLIST(clist_));
}

void
Index::remove(pkgCache::Package* package)
{
  g_return_if_fail(dialog_ != 0);

  vector<pkgCache::Package*>::iterator i = packages_.begin();
  while (i != packages_.end())
    {
      if (*i == package) break;
      ++i;
    }
  
  g_return_if_fail(i != packages_.end());

  packages_.erase(i);

  remove_from_clist(package);
}

void 
Index::add_to_clist(pkgCache::Package* package)
{
  g_return_if_fail(clist_ != 0);
  g_return_if_fail(cache_ != 0);

  pkgCache::PkgIterator pi(*cache_, package);

  if (filter_)
    {
      if (filter_->include_package(pi, cache_) == false)
        return;
    }

  const char* text[1];
  text[0] = pi.Name();  
  
  gint row = gtk_clist_append(GTK_CLIST(clist_),
                              const_cast<char**>(text));
  
  gtk_clist_set_row_data(GTK_CLIST(clist_), row,
                         package);
}

void 
Index::remove_from_clist(pkgCache::Package* package)
{
  g_return_if_fail(clist_ != 0);

  gint row = gtk_clist_find_row_from_data(GTK_CLIST(clist_),
                                          package);

  g_return_if_fail (row >= 0);

  gtk_clist_remove(GTK_CLIST(clist_),
                   row);
}


void
Index::set_cache(GAptCache* cache)
{
  if (cache == 0) 
    {
      g_return_if_fail(package_names_.empty());

      vector<pkgCache::Package*>::iterator i = packages_.begin();
      while (i != packages_.end())
        {
          pkgCache::PkgIterator pi(*cache_, *i);
          package_names_.push_back(pi.Name());
          ++i;
        }
      
      packages_.clear(); // Package* are now invalid.

      cache_ = 0;
      return;
    }
  else 
    {
      // this happens when we first add ourselves as a cache view
      //  Terrible hack
      if (cache_ != 0) 
        return; 
      
      vector<string>::iterator i = package_names_.begin();
      while (i != package_names_.end())
        {
          pkgCache::PkgIterator pi = cache->FindPkg(*i);

          if (!pi.end()) // make sure it's still in the new cache
            {
              packages_.push_back(const_cast<pkgCache::Package*>(&*pi));
            }
          
          ++i;
        }

      package_names_.clear(); // no longer care about names

      cache_ = cache;

      refill();
    }
}

void 
Index::refill()
{
  g_return_if_fail(filter_ != 0);
  g_return_if_fail(clist_ != 0);

  gtk_clist_freeze(GTK_CLIST(clist_));

  gtk_clist_clear(GTK_CLIST(clist_));

  vector<pkgCache::Package*>::iterator i = packages_.begin();
  while (i != packages_.end())
    {
      pkgCache::Package* package = *i;
      
      pkgCache::PkgIterator pi(*cache_, package);
      
      if (filter_->include_package(pi, cache_))
        {
          add_to_clist(package);
        }
      ++i;
    }

  gtk_clist_thaw(GTK_CLIST(clist_));
}

void
Index::filter_changed()
{
  refill();
}

gint
Index::dialog_close(GtkWidget* d, gpointer data)
{
  Index* i = static_cast<Index*>(data);

  i->detach_dialog();

  return FALSE;
}

void 
Index::detach_dialog()
{
  dialog_ = 0;
  clist_ = 0;

  indexes_->schedule_for_deletion(this);
}

//////////////////////////////


Indexes::Indexes()
  : deletion_pending_(false)
{
  
  
}


Indexes::~Indexes()
{
  set<Index*>::iterator i = indexes_.begin();
  while (i != indexes_.end())
    {
      delete *i;
      ++i;
    }

  if (deletion_pending_)
    {
      do_deletion();

      gtk_idle_remove(idle_tag_);
    }
}

void 
Indexes::add(Index* i)
{
#ifdef GNOME_ENABLE_DEBUG
  set<Index*>::iterator j = indexes_.find(i);
  g_return_if_fail (j == indexes_.end());
#endif
  
  indexes_.insert(i);
}

void 
Indexes::schedule_for_deletion(Index* i)
{
  set<Index*>::iterator j = indexes_.find(i);
  g_return_if_fail (j != indexes_.end());
  
  indexes_.erase(j);
  to_be_deleted_.push_back(i);

  if (!deletion_pending_)
    {
      idle_tag_ = gtk_idle_add(deletion_idle, this);
      deletion_pending_ = true;
    }
}
  
gint 
Indexes::deletion_idle(gpointer data)
{
  Indexes* i = static_cast<Indexes*>(data);

  i->do_deletion();

  return FALSE; // remove ourselves
}

void 
Indexes::do_deletion()
{
  g_return_if_fail(deletion_pending_);

  vector<Index*>::iterator j = to_be_deleted_.begin();
  while (j != to_be_deleted_.end())
    {
      delete *j;
      ++j;
    }
  
  to_be_deleted_.clear();

  deletion_pending_ = false;
}

static Indexes* indexes = 0;

Indexes*
gnome_apt_indexes()
{
  // Eventually we might bother to delete this on exit
  if (indexes == 0)
    indexes = new Indexes;

  return indexes;
}

void     
gnome_apt_indexes_shutdown()
{
  if (indexes != 0)
    delete indexes;
  
  indexes = 0;
}
