/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 *   Gnome Apt frontend
 *
 *   Copyright (C) 1998 Havoc Pennington <hp@pobox.com>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnome-apt.h"
#include "app.h"
#include "cache.h"

#include <signal.h>

#include <vector>
#include <iostream>

/* Popt Stuff */

static const gchar* cmdline_geometry = 0;

static const struct poptOption options[] = {
  {"geometry", 'g', POPT_ARG_STRING, &cmdline_geometry, 0, 
   N_("Set initial main window size and position\n"
      "GEOMETRY is <width>x<height>+<x>+<y>.\n"), 
   N_("GEOMETRY")},
  {NULL, '\0', 0, NULL, 0}
};

// so we don't have to include these headers
extern void gnome_apt_sources_shutdown();
extern void gnome_apt_indexes_shutdown();
extern void gnome_apt_preferences_shutdown();

static GtkWindow* main_window = 0;
static set<GtkWidget*> pending_dialogs;

static GtkWidget* gnome_apt_splash_new();
static void gnome_apt_splash_destroy(GtkWidget* splash);

int main ( int argc, char ** argv )
{
  // Ensure FILE* and iostreams don't get confused
  // I think this should maybe be in ios_base:: according to the standard...
  ios::sync_with_stdio();

  // i18n
  bindtextdomain (PACKAGE, GNOMELOCALEDIR);
  textdomain (PACKAGE);

#ifdef GNOME_ENABLE_DEBUG
  int x = 1;
  ga_debug("Got arguments:");
  while (x < argc)
    {
      ga_debug("%d %s", x, argv[x]);
      ++x;
    }
#endif

  poptContext ctx;

  gnome_init_with_popt_table (APPNAME, VERSION, argc, argv, options, 0, &ctx);

  // I wonder why I had this here... probably will find out in a bug
  // report.
  // poptGetArgs (ctx);

#ifdef GNOME_ENABLE_DEBUG
  // so we don't have to install
  gtk_rc_parse("./gnome-aptrc");
  gtk_rc_parse("./gdeb/gdebrc");
#endif

  // FIXME This is somewhat wrong; it will override the user version  

  // Debian does not allow the default Gnome location
  gtk_rc_parse(GDEBRC); // hang on; we also include this in gnome-aptrc?
  gtk_rc_parse(GNOMEAPTRC);

  gchar* geometry = 0;
  if (cmdline_geometry != 0)
    {
      geometry = g_strdup(cmdline_geometry);
      cmdline_geometry = 0;
    }

  poptFreeContext(ctx);

  CommandLine::Args Args[] = {
    {0,0,0,0}};
  
  // Parse the command line and initialize the package library
  CommandLine CmdL(Args,_config);
  if (pkgInitialize(*_config) == false ||
      CmdL.Parse(argc,const_cast<const char**>(argv)) == false)
    {
      _error->DumpErrors();
      return 100;
    }
  
  // Print any errors or warnings found during parsing
  if (_error->empty() == false)
    {
      bool Errors = _error->PendingError();
      _error->DumpErrors();
      if (Errors == true)
        exit(EXIT_FAILURE);
    }
  
  GtkWidget* splash = gnome_apt_splash_new();

  gnome_apt_cache_file_init();
  GAptCacheFile* cf = gnome_apt_cache_file();

  if (cf) {
    GAptApp* app = new GAptApp(geometry);

    if (geometry) 
      {
        g_free(geometry);
        geometry = 0;
      }

    main_window = GTK_WINDOW(app->widget());

    if (gnome_config_get_bool("/gnome-apt/testing/alpha_warning=true"))
      {
        GtkWidget* warning = gnome_warning_dialog(_("This is alpha software, and may damage your system.\nUse at your own risk.\nThere is ABSOLUTELY NO WARRANTY; see the GPL for details."));
        gnome_apt_setup_dialog(warning);
        gtk_window_set_modal(GTK_WINDOW(warning), TRUE);
        gnome_config_set_bool("/gnome-apt/testing/alpha_warning", FALSE);
      }

    set<GtkWidget*>::iterator i = pending_dialogs.begin();
    while (i != pending_dialogs.end())
      {
        gnome_apt_setup_dialog(*i);
        ++i;
      }
 
    pending_dialogs.clear();

	gnome_apt_splash_destroy(splash);

    gtk_main();

    gnome_apt_cache_file_shutdown();    

    delete app;
    app = 0;

    gnome_apt_sources_shutdown();
    gnome_apt_indexes_shutdown();
    gnome_apt_preferences_shutdown();

    return 0;
  }
  else {
    // show the error dialog
    gtk_main(); 
    return 1;
  }  
}

void 
gnome_apt_quit(void)
{
  g_warning("Quitting without asking first");
  
  g_return_if_fail(gtk_main_level() == 1);

  gtk_main_quit();
}

static GtkWidget*
error_dialog(const char* message)
{
  g_return_val_if_fail(message != 0, 0);

  string all_errors;

  all_errors += message;

  all_errors += _("\nThe following errors may help:\n");

  while (!_error->empty()) {
    string next;
    _error->PopMessage(next);

    // Chill the line length - bad hack for now.
    size_t i = 80;
    while (i < next.size())
      {
        next.insert(i, "\n");
        
        i += 80;
      }

    all_errors += next;
    all_errors += "\n";
  }
  
  cerr << all_errors << endl; // be sure it goes somewhere visible

  return gnome_error_dialog(all_errors.c_str());
}

void 
gnome_apt_error_dialog(const char* message)
{
  GtkWidget* dialog = error_dialog(message);
  gnome_apt_setup_dialog(dialog);
}

static gint 
dialog_close_cb(GtkWidget* w, gpointer data)
{
  gnome_apt_quit();
  return FALSE;
}

void 
gnome_apt_fatal_dialog(const char* message)
{
  GtkWidget* dialog = error_dialog(message);
  
  gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);
  gnome_apt_setup_dialog(dialog);  

  gtk_signal_connect(GTK_OBJECT(dialog), "close",
                     GTK_SIGNAL_FUNC(dialog_close_cb),
                     NULL);
}

static void
dialog_destroy_cb(GtkWidget* dialog, gpointer data)
{
  set<GtkWidget*>::iterator i = pending_dialogs.find(dialog);

  if (i != pending_dialogs.end())
    pending_dialogs.erase(i);
}

void 
gnome_apt_setup_dialog(GtkWidget* dialog)
{
  g_return_if_fail(dialog != 0);

  if (main_window == 0)
    {
      pending_dialogs.insert(dialog);
      // arrange to forget about closed dialogs
      gtk_signal_connect(GTK_OBJECT(dialog),
                         "destroy",
                         GTK_SIGNAL_FUNC(dialog_destroy_cb),
                         NULL);
    }
  else
    gnome_dialog_set_parent(GNOME_DIALOG(dialog), main_window);
}

GtkWidget* 
gnome_apt_labelled_widget(const gchar* label, GtkWidget* widget)
{
  GtkWidget* l = gtk_label_new(label);

  GtkWidget* hbox = gtk_hbox_new(FALSE, GNOME_PAD_SMALL);
  gtk_container_set_border_width(GTK_CONTAINER(hbox), GNOME_PAD_SMALL);

  gtk_misc_set_alignment(GTK_MISC(l), 0.0, 0.5);

  gtk_box_pack_start(GTK_BOX(hbox), l, FALSE, FALSE, 0);
  gtk_box_pack_end(GTK_BOX(hbox), widget, TRUE, TRUE, 0);

  return hbox;
}

GtkWidget*
gnome_apt_splash_new()
{
  GtkWidget *pix;
  GtkWidget *box;
  GtkWidget* spl;
  int       x,y;
  gchar* pixfile;

  spl = gtk_window_new(GTK_WINDOW_POPUP);

  x = (gdk_screen_width() - 400) / 2;
  y = (gdk_screen_height() - 200) / 2;

  gtk_widget_set_uposition(spl,x,y);

  box = gtk_hbox_new(0,TRUE);
  gtk_widget_show(box);
  gtk_container_add(GTK_CONTAINER(spl),box);

  pixfile = gnome_pixmap_file("gnome-apt-splash.png");

  if (pixfile)
    {
      pix = gnome_pixmap_new_from_file(pixfile);
      if (pix)
        {
          gtk_widget_show(pix);
          gtk_container_add(GTK_CONTAINER(box),pix);
        }
      g_free(pixfile);
    }

  gtk_widget_show_now(spl);

  return spl;
}

void
gnome_apt_splash_destroy(GtkWidget* splash)
{
  gtk_widget_destroy(splash);
}


