
#include <std.h>
#include <strings.h>

#include <func/apt.h>

#include <parsers/strutl.h>
#include <parsers/color.h>

#include <list/packages.h>
#include <list/screens.h>
#include <list/colors.h>
#include <list/sections.h>

#include "coreui.h"
#include "formatting.h"
#include "format_strings.h"

void mvwblankl(WINDOW * win, int y, int x_len)
{
	int x;
	for (x = 0; x < x_len; x++)
		mvwaddch(win, y, x, ' ');
}

void my_werase(WINDOW * win)
{
	register short x, y;

	for (y = 0; y <= win->_maxy; y++)
		for (x = 0; x <= win->_maxx; x++)
			mvwaddch(win, y, x, ' ');
}

void mv_print_line(PackageList::Package * p, int row)
{
	/*
	 * We must determine what TYPE of line this is and call the appropriate
	 * wrapper function for it.
	 */

	bool cursor = (p == screen->Cursor);

	switch (p->type)
	{
	case PackageList::Div_State:
	case PackageList::Div_Sect:
	case PackageList::Div_Pri:
	case PackageList::Div_Custom:
		format_divider(p, cursor, row);
		break;

	case PackageList::Pkg:
		format_package(p, cursor, row);
		break;

	default:
		/* in theory this can't happen... ? */
		break;
	}
}

/*
 * Properly format the sort method dividers in the package list
 */
void format_divider(PackageList::Package * P, bool cursor, int row)
{
	int back_color, attrib;
	int Inc = 0;
	int i;
	char *ptr;

	// Defaults
	int StartPoint = 0;
	int Factor = 1;

	wmove(wmain, row, 0);
	wclrtoeol(wmain);

	// Set up color for the divider

	if (cursor == true)
	{
		back_color = CAPT_COLOR_CURSOR;
		attrib = A_REVERSE | A_BOLD;
	}
	else
	{
		back_color = CAPT_COLOR_DIVIDER_BACK;
		attrib = A_BOLD;
	}

	wattrset(wmain, attrib);

	string Format = FMT_DIV_EXPN;

	switch (P->type)
	{
	case PackageList::Div_State:
		Inc = screen->Sorting->FindIncrement(sortorderlist::Status) - 1;

		if ((*Cache)[P->pkg].Status == 1)
			Format += FMT_DIV_STAT_INST;
		else if ((*Cache)[P->pkg].Status == 0)
			Format += FMT_DIV_STAT_UPGR;
		else
			Format += FMT_DIV_STAT_NOTINST;
		break;

	case PackageList::Div_Sect:
		Inc = screen->Sorting->FindIncrement(sortorderlist::Section) - 1;

		if (P->pkg.Section() == NULL)
			Format += FMT_DIV_SECT_NULL;
		else
			Format += Section_Lookup((char *) P->pkg.Section());

		break;

	case PackageList::Div_Pri:
		Inc = screen->Sorting->FindIncrement(sortorderlist::Priority) - 1;

		ptr = (char *) (*Cache)[P->pkg].CandidateVerIter(*Cache).PriorityType();

		if (*ptr == 0)
		{
			Format += FMT_DIV_PRI_NULL;
		}
		else
		{
			Format += ptr;
			Format += " Packages";
		}

		break;

	case PackageList::Div_Custom:
		Inc = P->increment;

		Format += (char *) P->Extra;

		break;

	default:
		break;
	}

	int len_left = ColorCountStrLen((char *) Format.c_str());

	scrollok(wmain, false);

	ColorMVPrintW(wmain, row, StartPoint + (Factor * Inc), back_color, attrib & ~A_BOLD, "%s", Format.c_str());

	for (i = StartPoint + len_left + (Factor * Inc); i < COLS; i++)
		mvwaddch(wmain, row, i, ' ');

	scrollok(wmain, true);

	wrefresh(wmain);
}

/*
 * Display one package on one line on one screen [Capisch?]
 */
void format_package(PackageList::Package * P, bool cursor, int row)
{
	pkgCache::PkgIterator I = P->pkg;
	pkgCache::VerIterator V, InstV;

	V = (*Cache)[I].CandidateVerIter(*Cache);
	InstV = (*Cache)[I].InstVerIter(*Cache);

	char left[256], right[256];
	int len_left = 0, len_right = 0;

	bool MarkInstall = (*Cache)[I].Install();
	bool MarkDelete = (*Cache)[I].Delete();

	wattrset(wmain, Pair(COLOR_WHITE, COLOR_BLACK));

	for (int X = 0; X < (P->increment); X++)
		mvwaddch(wmain, row, X, ' ');

	int back_color, attrib;

	if (cursor == true)
	{
		back_color = CAPT_COLOR_CURSOR;
		attrib = A_REVERSE;
	}
	else
	{
		back_color = CAPT_COLOR_BACK;
		attrib = 0;
	}

	// Normal packages colors are chosen here

	if ((*Cache)[I].Upgrade() == true && (*Cache)[I].NewInstall() == false)
	{
		wattrset(wmain, Pair(CAPT_COLOR_UPGRADE_MARK, back_color) | attrib | A_BOLD);
	}
	else if (MarkInstall == true)
	{
		wattrset(wmain, Pair(CAPT_COLOR_INSTALL_MARK, back_color) | attrib | A_BOLD);
	}
	else if (MarkDelete == true)
	{
		bool Purge = (*Cache)[I].iFlags & pkgDepCache::Purge;
		if (Purge == true)
			wattrset(wmain, Pair(CAPT_COLOR_REMOVE_MARK, back_color) | attrib | A_BOLD);
		else
			wattrset(wmain, Pair(CAPT_COLOR_REMOVE_MARK, back_color) | attrib);
	}
	else if (screen->Packages->IsMatched(P))
	{
		wattrset(wmain, Pair(CAPT_COLOR_SEARCH_MATCH, back_color) | attrib);
	}
	else if ((*Cache)[I].Status == 1)
	{
		wattrset(wmain, Pair(CAPT_COLOR_NOT_UPGRADED, back_color) | attrib | A_BOLD);
	}
	else if (InstV.end() == false)
	{
		wattrset(wmain, Pair(CAPT_COLOR_INSTALLED, back_color) | attrib);
	}
	else
	{
		wattrset(wmain, Pair(CAPT_COLOR_DEFAULT, back_color) | attrib);
	}

	// Make sure the strings are not too big ..

	char *Priority = NULL, *Name = NULL, *CandVer = NULL, *InstVer = NULL, *Section = NULL;
	unsigned short PriorityLen = 0, NameLen = 0, CandVerLen = 0, InstVerLen = 0, SectionLen = 0;

	// Lengths are based on which view mode we are in

	switch (screen->mode)
	{
	case ScreenList::Normal:
		PriorityLen = 9;
		NameLen = 17;
		CandVerLen = 18;
		InstVerLen = 18;
		SectionLen = 10;
		break;

	case ScreenList::Verbose:
		PriorityLen = 3;
		NameLen = 12;
		CandVerLen = 10;
		InstVerLen = 10;
		SectionLen = 8;
		break;
	}

	/*
	 * Chomp all the strings to the desired lengths
	 */

	Name = period_string(I.Name(), NameLen);

	if (V.end() == false)
	{
		CandVer = period_string(V.VerStr(), CandVerLen);

		if (strcmp(V.PriorityType(), "") != 0)
			Priority = period_string(V.PriorityType(), PriorityLen);
		else
			Priority = period_string(FMT_PKG_PRI_NULL, PriorityLen);
	}
	else
		CandVer = FMT_PKG_CANDVER_NULL;

	if (InstV.end() == false)
		InstVer = period_string(InstV.VerStr(), InstVerLen);
	else
		InstVer = FMT_PKG_CANDVER_NULL;

	if (I.Section() && V.end() == false)
	{
		char *x;

		for (x = (char *) I.Section(); *x != '/' && *x != 0; x++);

		if (*x == '/')
			x++;
		else
			x = (char *) I.Section();

		Section = period_string(x, SectionLen);
	}

	char *leftptr = NULL;

	if (screen->mode == ScreenList::Normal)
	{
		len_left = snprintf(left, 255, FMT_PKG_VWNORM_LEFT, Priority, Name, CandVer);
		leftptr = strdup(left);
		len_right = snprintf(right, 255, FMT_PKG_VWNORM_RIGHT, InstVer, V->Size / 1024, V->InstalledSize / 1024, Section ? Section : "<none>");

		if (screen->Packages->IsExpanded(P))
			len_left += ColorCountStrLen(FMT_PKG_EXPN);
		else
			len_left += ColorCountStrLen(FMT_PKG_NEXPN);
	}
	else
	{
		string s = Recs->Lookup(V.FileList()).ShortDesc();

		len_left = sprintf(left, FMT_PKG_VWVERB_LEFT, Priority, Name, Section ? Section : FMT_PKG_SECT_NULL, CandVer, InstVer, V->Size / 1024, V->InstalledSize / 1024, s.c_str());

		if (screen->scroll >= (unsigned) len_left)
		{
			leftptr = strdup("");
			len_left = 0;
		}
		else
		{
			leftptr = period_string(left + screen->scroll, COLS - P->increment - 4);
			len_left = strlen(leftptr);
		}
	}


	free(Name);

	if (InstV.end() == false)
		free(InstVer);

	if (V.end() == false)
	{
		free(Priority);
		free(CandVer);
		free(Section);
	}

	scrollok(wmain, false);

/*	ColorMVPrintW(wmain, row, 0, back_color, attrib, " $5%c$6%c$3%c$! ",
		( InstV.end() == false ? '*' : ' ' ),
		( MarkInstall == true ? '*' : ( MarkDelete == true ? '!' : ' ' ) ),
		( (*Cache)[I].Status == 1 ? '*' : ' ' )); */

	if (screen->Packages->IsExpanded(P))
		ColorMVPrintW(wmain, row, P->increment, back_color, attrib, "%s%s", FMT_PKG_EXPN, leftptr);
	else
		ColorMVPrintW(wmain, row, P->increment, back_color, attrib, "%s%s", FMT_PKG_NEXPN, leftptr);

	free(leftptr);

	if (screen->mode == ScreenList::Normal)
	{
		mvwprintw(wmain, row, COLS - len_right, "%s", right);
		for (register int i = 0; i <= (COLS - len_right - len_left - 1); i++)
			mvwaddch(wmain, row, len_left + i, ' ');
	}
	else
	{
		for (register int i = COLS; i > (len_left + P->increment + 3); i--)
			mvwaddch(wmain, row, i, ' ');
	}

	scrollok(wmain, true);
}

void paint_description_top(void)
{
	/* Should put something here, in theory. */

	werase(wdtop);

	switch (screen->Cursor->type)
	{
	case PackageList::Pkg:
		mvwprintw(wdtop, 0, 0, "%-15s %s, %s; %s",
			  screen->Cursor->pkg.Name(),
			  (*Cache)[screen->Cursor->pkg].InstVerIter(*Cache).end() == false ? "installed" : "not installed",
			  (*Cache)[screen->Cursor->pkg].Status == 0 ? "upgraded" : "not upgraded",
			  (*Cache)[screen->Cursor->pkg].Install() == true ? "install" : ((*Cache)[screen->Cursor->pkg].Delete() == true ? "delete" : "unchanged"));
		break;

	default:
		break;
	}

	wrefresh(wdtop);
}

void paint_description()
{
	if (wdesc == NULL)
		return;

	werase(wdesc);
	wmove(wdesc, 0, 0);

	paint_description_top();

	switch (screen->Cursor->type)
	{
	case PackageList::Pkg:
		break;
	case PackageList::Div_Sect:
		mvwprintw(wdesc, DESC_LINES / 2, (COLS - strlen(FMT_DWIN_DIV_SECT)) / 2, "%s", FMT_DWIN_DIV_SECT);
		wrefresh(wdesc);
		return;
	case PackageList::Div_Pri:
		mvwprintw(wdesc, DESC_LINES / 2, (COLS - strlen(FMT_DWIN_DIV_PRI)) / 2, "%s", FMT_DWIN_DIV_PRI);
		wrefresh(wdesc);
		return;
	case PackageList::Div_State:
		mvwprintw(wdesc, DESC_LINES / 2, (COLS - strlen(FMT_DWIN_DIV_STATE)) / 2, "%s", FMT_DWIN_DIV_STATE);
		wrefresh(wdesc);
		return;

	case PackageList::Div_Custom:
		mvwprintw(wdesc, DESC_LINES / 2, (COLS - strlen(FMT_DWIN_DIV_CUST)) / 2, FMT_DWIN_DIV_CUST, (char *) screen->Cursor->Extra);
		wrefresh(wdesc);
		return;
	}

	delete curDescLines;
	curDescLines = new vector < string > (5);
	desc_width = 0;

	string s = Recs->Lookup(((*Cache)[screen->Cursor->pkg].CandidateVerIter(*Cache)).FileList()).LongDesc();

	char *scpy = (char *) malloc(s.length() + 1);

	memcpy(scpy, s.c_str(), s.length() + 1);

	int I = 0;

	for (char *P = strtok(scpy, "\n"); P != NULL; P = strtok(NULL, "\n"))
	{
		++I;

		if (I == 1)
			continue;

		++desc_width;

		if (strncmp(P, " .", 2) == 0)
			P = "";

		if (strlen(P) >= (unsigned) COLS)
			P[COLS - 1] = 0;

		if (curDescLines->size() == (unsigned) desc_width)
			curDescLines->resize(curDescLines->size() + 5);

		(*curDescLines)[desc_width - 1] = (string) P;

		if (desc_width <= DESC_LINES)
			if (strlen(P) >= screen->scroll)
				mvwprintw(wdesc, desc_width - 1, 0, "%s", P + screen->scroll);
	}

	free(scpy);

	desc_bound_start = 0;

	if (desc_width < DESC_LINES)
		desc_bound_end = desc_width;
	else
		desc_bound_end = DESC_LINES;

	wrefresh(wdesc);
}

void paint_status(void)
{
	werase(wstat);

	ColorMVPrintW(wstat, 0, 0, COLOR_BLUE, 0, FMT_STAT_LEFT, long ((double (screen->_cursor)) /double (screen->Packages->Length() - 1) * 100.0));

	char buf[COLS];
	int len;

	sprintf(buf, FMT_STAT_CENTER, Cache->InstCount(), Cache->DelCount(), Cache->KeepCount(), Cache->BrokenCount());

	len = ColorCountStrLen(buf);

	ColorMVPrintW(wstat, 0, (COLS - len) / 2, COLOR_BLUE, 0, "%s", buf);

	// Display the version, and other info.

	if (Filters.Length() > 0)
		if (screen.Current() == screen.Head()->next->next)
			sprintf(buf, FMT_STAT_FILTINV FMT_STAT_FILT FMT_STAT_RIGHT, Filters.Length(), INTERNAL_VERSION);
		else
			sprintf(buf, FMT_STAT_FILT FMT_STAT_RIGHT, Filters.Length(), INTERNAL_VERSION);
	else
		sprintf(buf, FMT_STAT_RIGHT, INTERNAL_VERSION);

	len = ColorCountStrLen(buf);

	ColorMVPrintW(wstat, 0, COLS - len, COLOR_BLUE, 0, buf);

	wrefresh(wstat);
}

void paint_top(void)
{
	char left[256], right[256];
	int len_left = 0, len_right = 0;

	// Draw the menu

	wattrset(wtop, Pair(COLOR_WHITE, COLOR_BLUE) | A_BOLD);

	for (register int i = 0; i < COLS; i++)
		mvwaddch(wtop, 0, i, ' ');

	wattron(wtop, A_UNDERLINE);

	unsigned alg = ((COLS) / 6);

	ColorMVPrintW(wtop, 0, (alg * 0), COLOR_BLUE, A_UNDERLINE, "| O$3#peration$! ");
	ColorMVPrintW(wtop, 0, (alg * 1), COLOR_BLUE, A_UNDERLINE, "| M$3#ovement$!  ");
	ColorMVPrintW(wtop, 0, (alg * 2), COLOR_BLUE, A_UNDERLINE, "| T$3#oggles$!   ");
	ColorMVPrintW(wtop, 0, (alg * 3), COLOR_BLUE, A_UNDERLINE, "| A$3#lter$!     ");
	ColorMVPrintW(wtop, 0, (alg * 4), COLOR_BLUE, A_UNDERLINE, "| F$3#ilters$!   ");
	ColorMVPrintW(wtop, 0, (alg * 5), COLOR_BLUE, A_UNDERLINE, "| S$3#orting$!   ");

	// Draw the information for the main window

	wattrset(wtop, Pair(COLOR_WHITE, COLOR_RED) | A_BOLD);

	string whitespace;

	for (int I = 0; I <= ((screen->Sorting->DividerCount() + 1) - 2); I++)
		whitespace += " ";

	for (register int i = 0; i < COLS; i++)
		mvwaddch(wtop, 1, i, ' ');

	if (screen->mode == ScreenList::Normal)
	{

		len_left = snprintf(left, 255, "%s" FMT_TOP_VWNORM_LEFT, whitespace.c_str(), "Priority", "Package", "Avail.ver");
		len_right = snprintf(right, 255, FMT_TOP_VWNORM_RIGHT, "Inst.ver", "Size", "Instsize", "Section");

		mvwprintw(wtop, 1, 0, left);

		mvwprintw(wtop, 1, COLS - len_right, "%s", right);
	}
	else
	{
		len_left = snprintf(left, COLS, "%s" FMT_TOP_VWVERB_LEFT, whitespace.c_str(), "Pri", "Package", "Section", "Avail.ver", "Inst.ver", "PkgSZ", "InstSZ", "Description");

		for (register int i = len_left; i < COLS; i++)
			mvwaddch(wtop, 1, i, ' ');

		char *ptr;

		if (screen->scroll >= (unsigned) len_left)
			ptr = "";
		else
			ptr = left + screen->scroll;

		mvwprintw(wtop, 1, 0, "%s", ptr);
	}

	wrefresh(wtop);
}
