#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/klog.h>
#ifndef _TESTING_
# include <sys/reboot.h>
#endif
#ifdef _TESTING_
# define RB_AUTOBOOT 1
reboot(int blah){}
#endif
#include <sys/types.h>
#include <fcntl.h>
#include <ctype.h>
#include <asm/types.h>
#include <linux/serial.h>
#include <syslog.h>

#ifdef USE_LANGUAGE_CHOOSER
#include <locale.h>
#include <wchar.h>
#endif

#include "dbootstrap.h"
#include "lang.h"
#include "util.h"

char *kver;
char *append_opts;
char *Arch2;
char kernel_image_path[PATH_MAX + 1];
char drivers_path[PATH_MAX + 1];

/* Defaults to no serial console present */
int serialConsole = -1;

#if #cpu(alpha)
int srm_boot = 1;
char milo_binary_path[PATH_MAX + 1];
#endif

#ifdef USE_LANGUAGE_CHOOSER
const struct language_item *lang = NULL;
#endif

struct Arg {
  const char *name;
  int isflag;
  void *value;
};

int parsefile(const char *filename, struct Arg *options, 
	      int hasinterspace, int toeol) {
  int fd, bytes, lineend=0;
  char *start= prtbuf, *end;
  struct Arg *a;

  if ((fd = open(filename, O_RDONLY, 0)) < 0) {
    ERRMSG(_("Cannot open %s: %s\n"), filename, strerror(errno) );
    return 1;
  }
  while ( (bytes = read(fd, prtbuf, PRTBUFSIZE - 1)) != 0) {
    prtbuf[bytes]= '\0';

    /* parse line */
    while (! lineend) {
      /* skip space at the beginning of the line */
      while (isspace(*start)) start++;
      if (!(*start)) break;

      /* take the first word on the string as the option name */
      end = start;
      while (*end && !isspace(*end)) end++;
      if (!*end) lineend = 1;
      *end = '\0';

      a = options;
      while ( a->name != NULL ) {
	if (a->isflag) {
	  if (! strcmp(start, a->name)) {
	    /* the option is valid. It is a flag, switch it on */
	    *((int *)a->value) = 1;
	  }
	} else {
	  int namelen=strlen(a->name);
	  if (!strncmp(start, a->name, namelen)) {
	    /* the option is valid. It expects a value */
	    if (hasinterspace) {
	      if (lineend) {
		/* If the file syntax is "name value" (hasinterspace==1) 
		 * and lineend == 1, we have found a name with no value. */
		break;
	      }
	      /* skip whitespace after the option name */
	      namelen = 0;
	      start = end + 1;
	      /* MDP: Skip a colon since PPC uses a different format */
	      while (isspace(*start) || *start == ':') start++;
	      if (!(*start)) {
		/* We have reached the end of the line, that means we have
		 * found a name with no value. */
		break;
	      }
	      end = start;
	      if (toeol)  /* include embedded spaces in value */
		while (*end >= ' ') end++;
	      else
		while (*end && !isspace(*end)) end++;
	      *end = '\0';
	      /* Skip the rest of the line */
	      lineend = 1;
	    }
	    *((char **)a->value)=strdup(start+namelen);
	  }
	}
	a++;
      }
      start = end + 1;
    }
  }
  close(fd);
  return 0;
}

#if #cpu(alpha) || #cpu(sparc)
static int
parse_cpuinfo(const char *key, char *val, int len)
{
    FILE *cpuinfo;
    char format[256];
    int found = 0;

    snprintf(format, sizeof(format), "%s : %%%ds", key, len-1);

#if defined(_EXETESTING_) || defined(_TESTING_)
    if ((cpuinfo = fopen ("testcpuinfo", "r")) != NULL)
#else
    if ((cpuinfo = fopen ("/proc/cpuinfo", "r")) != NULL)
#endif
      {
	while (!feof(cpuinfo)) {
	    if (fscanf (cpuinfo, format, val) == 1) {
	      found = 1;
	      break;
	    }
	    /* Grab the rest of the line */
	    fgets (val, len, cpuinfo);
	}
	fclose(cpuinfo);
      }
    return found;
}
#endif

/* Get arguments from boot commmand line */
int
readcmdline (void)
{
  int  status;

  struct Arg bootoptions[] = {
    { "root=",     0, &bootargs.root     },
    { "disksize=", 0, &bootargs.disksize },
    { "bootkbd=",  0, &bootargs.kbd      },
    { "mouse=",    0, &bootargs.mouse    },
    { "flavor=",   0, &bootargs.flavor   },
    { "mono",      1, &bootargs.ismono   },
    { "debug",     1, &bootargs.isdebug  },
    { "verbose",   1, &bootargs.isverbose}, /* chatty mode */
    { "quiet",     1, &bootargs.isquiet  }, /* quiet mode */
    { "cdrom",     1, &bootargs.cdrom    }, /* cdrom install */
    { NULL,        0, NULL               }
  };

  memset ((void *) &bootargs, 0, sizeof(struct BootArgs));
#if defined(_EXETESTING_) || defined(_TESTING_)
  status = parsefile("testcmdline", bootoptions, 0, 0);
#else
  status = parsefile("/proc/cmdline", bootoptions, 0, 0);
#endif

  return status;
}

#if #cpu (m68k)
static char *get_m68k_model( void )
{
  char *model = NULL;
  int status;
  struct Arg archoptions[] = {
    { "Model:", 0, &model },
    { NULL, 0, NULL}
  };
  status = parsefile("/proc/hardware", archoptions, 1, 0);
  if(status) {
    fprintf( stderr, _("m68k machine type unknown!\n") );
    return NULL;
  }
  if (!model || !*model) {
     fprintf( stderr, _("Unable to determine m68k machine type!\n") );
     return NULL;
  }

  if (strcmp(model, "Motorola") == 0)
  {
    /* reparse to get MVME model number as well */
    parsefile("/proc/hardware", archoptions, 1, 1);
  }
 
  return model;
}
#endif

#if #cpu (powerpc)
static char *get_powerpc_model( void )
{
  char *model = NULL;
  int status;
  struct Arg archoptions[] = {
    { "machine", 0, &model },
    { NULL, 0, NULL}
  };
  status = parsefile("/proc/cpuinfo", archoptions, 2, 0);
  if (status) {
    fprintf( stderr, _("PowerPC machine type unknown!\n") );
    return NULL;
  }
  if (!model || !*model)
    fprintf( stderr, _("Unable to determine PowerPC machine type!\n") );
  if ((strncasecmp(model, "Power", 5) == 0)
     || (strncasecmp(model, "iMac",  4) == 0)) model = "PowerMac";
  if (strcmp(model,"Amiga") == 0)
    model = "apus";
  return model;
}
#endif

#if #cpu(alpha)
static int
is_in_cpuinfo(const char *key, const char *str)
{
  char buf[256];

  if (parse_cpuinfo(key, buf, sizeof(buf))) {
    if (strncmp(buf, str, strlen(str)) == 0)
      return 1;
  }
  return 0;
}

int get_alpha_boot_method(void)
{
  if (is_in_cpuinfo("system serial number", "MILO"))
    srm_boot = 0;
  else
    srm_boot = 1;
  return srm_boot;
}

/* These are ones that can't be resolved simply by downcasing */
static struct alpha_model {
  const char *systype, *model;
} alpha_models[] = {
  {"Nautilus", "nautilus"},
  {"Jensen", "jensen"},
  {"AlphaBook1", "book1"},
  {"EB64+", "eb64p"},
  {"EB66+", "eb66p"},
  {"Platform2000", "p2k"},
};
#define ALPHA_MODEL_COUNT (sizeof(alpha_models) / sizeof(alpha_models[0]))

char* get_alpha_model(void)
{
  char buf[256];
  if (parse_cpuinfo("system type", buf, sizeof(buf))) {
    int i;
    char *c;
    for (i = 0; i < ALPHA_MODEL_COUNT; i++) {
      if (strcmp(alpha_models[i].systype, buf) == 0)
	return strdup(alpha_models[i].model);
    }
    /* else, take the one we got and downcase it */
    for (c = buf; *c; c++)
      *c = tolower(*c);
    return strdup(buf);
  }
  return strdup(""); /* generic */
}
#endif

#if #cpu(sparc)
/* FIXME: what do we do about sun4e?! */
static char* get_sparc_model(void)
{
  char buf[16];

  if (parse_cpuinfo("type", buf, sizeof(buf))) {
    if (! strncmp(buf, "sun4", 4)) {
      if (buf[4] == 'c' || buf[4] == 'd' || buf[4] == 'm')
	return strdup("sun4cdm");
    }
    return strdup(buf);
  }
  return NULL;
}
#endif

#if #cpu(arm)
char *Arch3;

static char* get_arm_model(void)
{
  char *model = NULL;
  int status;
  struct Arg archoptions[] = {
    { "Hardware", 0, &model },
    { NULL, 0, NULL}
  };
  status = parsefile("/proc/cpuinfo", archoptions, 2, 0);
  if (status || !model || !*model) {
    fprintf( stderr, _("ARM machine type unknown!\n") );
    model = "";
  }
  Arch3 = model;
  if (strncasecmp(model, "acorn-riscpc", 6) == 0)
    model = "riscpc";
  if (strcasecmp(model,"chalice-cats") == 0
      || strcasecmp(model, "rebel-netwinder") == 0
      || strcasecmp(model, "ebsa285") == 0)
    model = "netwinder";
  return model;
}
#endif

void get_subarch_name(void)
{
#if #cpu(m68k)
  Arch2 = get_m68k_model();
#elif #cpu(powerpc)
  Arch2 = get_powerpc_model();
#elif #cpu(alpha)
  Arch2    = get_alpha_model();
#elif #cpu(sparc)
  Arch2 = get_sparc_model();
#elif #cpu(arm)
  Arch2 = get_arm_model();
#else
  Arch2 = strdup(""); /* prevent errors */
#endif
}

void get_kver(void)
{
  FILE *f;
  char osrelease[20];
  if ((f = fopen("/proc/sys/kernel/osrelease", "r"))) {
    fscanf(f, "%19s", osrelease);
    fclose(f);
    kver = strdup(osrelease);
  } else {
    /* panic! */
    kver = strdup("2.2.14"); /* default for now */
  }
}

/* try to eject root floppy */
#if #cpu(sparc)
void
try_to_eject_root_floppy(void)
{
  if (bootargs.root && 0==strncmp(bootargs.root,"/dev/fd",7)) {
      /* extract floppy device name */
      eject_floppy(bootargs.root);
  }
}
#endif

#ifdef SCSI_FLOPPY
/* scan /proc/scsi/scsi for TEAC FC-1 floppy drive
 * and create /dev/sfd[0-7] symlinks if found
 */
void
find_scsi_floppy()
{
  FILE *fp;
  char buf[100];
  char dev[8];
  int  i = -1;
  int  n = 0;
 
  if ((fp = fopen("/proc/scsi/scsi", "r")) != NULL) {
    while (fgets(buf, sizeof(buf), fp)) {
      if ((strncmp(buf, "Host:", 5) == 0) && strstr(buf, "Channel:"))
        i++;
      else {
        if (strstr(buf, "Vendor:")
        &&  strstr(buf, "TEAC"   )
        &&  strstr(buf, "Model:" )
        &&  strstr(buf, "FC-1"   )) {
          sprintf(dev, "sd%c", 'a' + i);
          sprintf(buf, "/dev/sfd%d", n++);
          unlink(buf);
          symlink(dev, buf);
        }
      }
    }
    fclose(fp);
  }
}
#endif

/*
 * check for supported filesystem in kernel
 */
int
supported_filesystem(const char* fsname)
{
  char buf[12];
  int found=0;
  FILE *f = fopen("/proc/filesystems", "r");
  if (f) {
    while (fscanf(f, "%11s", buf) > 0) {
      if (!strcmp(buf, fsname)) {
	found=1;
	break;
      }
    }
    fclose(f);
  }
  return found;
}

void
setup_image_names(void)
{
  /* Keep track of lengths - start with trailing zero */
  int kplen = 1, dplen = 1;
  kernel_image_path[0] = 0;
  drivers_path[0] = 0;
 
  /* Subarchitecture */
  if (Arch2) {
    int len = strlen(Arch2);
    kplen += len + 1;
    dplen += len + 1;
    if (kplen > sizeof(kernel_image_path)
	|| dplen > sizeof(drivers_path)) {
      ERRMSG("kernel_image_path or drivers_path too long (%d %d), punting",
	     kplen, dplen);
      kernel_image_path[0] = 0;
      drivers_path[0] = 0;
      goto append_names;
    }
      
    if (len > 0) {
      /* default */
      const char *arch_path = Arch2;

#if #cpu(powerpc)
      if (!strcmp(Arch2, "Amiga"))
	 arch_path = "apus";
      else if (!strcmp(Arch2, "CHRP"))
	 arch_path = "chrp";
      else if (!strcmp(Arch2, "PowerMac"))
	 arch_path = "powermac";
      else if (!strcmp(Arch2, "PReP"))
	 arch_path =  "prep";

#elif #cpu(m68k)
      if (!strncmp(Arch2, "BVME", 4))
	 arch_path = "bvme6000";
      else if (!strncmp(Arch2, "Motorola", 8)) {
      	 if (!strncmp(Arch2 + 8, " MVME147", 8))
      	   arch_path = "mvme147";
      	 else
	   arch_path = "mvme16x";
      }
      else if (!strcmp(Arch2, "Atari"))
	 arch_path = "atari";
      else if (!strcmp(Arch2, "Amiga"))
      	 arch_path = "amiga";
      else if (!strcmp(Arch2, "Macintosh"))
      	 arch_path = "mac";

#elif #cpu(alpha)
      /* The rescue disk is generic for most Alphas, but MILO is not.
         So we have to fix up here :P */
      if (strcmp(Arch2, "nautilus") && strcmp(Arch2, "jensen"))
	arch_path = "";
      /* Furthermore, choose_medium() is really broken, so we have to
         cook up a MILO path here while we're at it */
      sprintf(milo_binary_path, "MILO/%s", Arch2);
#endif

      strcat(kernel_image_path, arch_path);
      strcat(kernel_image_path, "/");

      strcat(drivers_path, arch_path);
      strcat(drivers_path, "/");
    }
  }

  /*
   * Some architectures have several sizes of diskette images.
   */
  if (bootargs.disksize) {
    int len = strlen(bootargs.disksize);
    kplen += len + 8;
    if (kplen > sizeof(kernel_image_path)
	|| dplen > sizeof(drivers_path)) {
      ERRMSG("kernel_image_path or drivers_path too long (%d %d), punting",
	     kplen, dplen);
      kernel_image_path[0] = 0;
      drivers_path[0] = 0;
      goto append_names;
    }
      
    if (len > 0) {
      char part[len + 9];
      sprintf(part, "images-%s/", bootargs.disksize);
      strcat(kernel_image_path, part);
    }
  }

  /*
   * `flavor' is something like "compact", "safe", or "standard".  For
   *   some flavors, `kver' (aka `uname -r`) will be the same as the
   *   flavor, but for others it won't be. (eg: The "safe" flavor on
   *   i386 is the standard kernel with special options given to the
   *   boot loader installer to make it work with old BIOSes.  Its
   *   modules path and module set is the same as for the vanilla
   *   "2.2.xx" flavor kernel, but its flavor is different because it
   *   uses a different rescue disk image.
   */
  if (bootargs.flavor) {
    int len = strlen(bootargs.flavor);
    kplen += len + 1;
    dplen += len + 1;
    if (kplen > sizeof(kernel_image_path)
	|| dplen > sizeof(drivers_path)) {
      ERRMSG("kernel_image_path or drivers_path too long (%d %d), punting",
	     kplen, dplen);
      kernel_image_path[0] = 0;
      drivers_path[0] = 0;
      goto append_names;
    }

    if (len > 0) {
      /* special case ... the safe flavor doesn't have its own kernel
	 and drivers, so just use the standard ones */
      if (strcmp(bootargs.flavor, "safe") != 0) {
	strcat(kernel_image_path, bootargs.flavor);
	strcat(kernel_image_path, "/");
	strcat(drivers_path, bootargs.flavor);
	strcat(drivers_path, "/");
      }
    }
  }

 append_names:
  strcat(kernel_image_path, KERDISKFILE);
  DEBUGMSG("kernel_image_path is '%s'", kernel_image_path);
  strcat(drivers_path, DRVTGZFILE);
  DEBUGMSG("drivers_path is '%s'", drivers_path);
}

/* This simply checks that /proc/version exists, which should give us a
 * good indicator that /proc is in fact mounted - BenC
 */
static void check_proc (void) {
    struct stat statbuf;
    if (!NAME_ISREG("/proc/version", &statbuf)) {
	fprintf(stderr, "E: /proc does not appear to be mounted (%s)\n",
		strerror(errno));
    	exit(1);
    }
}

#ifndef _TESTING_
int main (void) {
    char *term;
    struct serial_struct ser;

    /* make sure that /proc is mounted */
    check_proc();
    /* Downgrade kernel verbosity */
    klogctl(8, NULL, 4);

    readcmdline();

    /* Default disksize, in case disksize= is not set in bootloader config. */
    if (bootargs.disksize == NULL) {
      bootargs.disksize = "1.44";
    }

    if ( bootargs.isverbose && bootargs.isquiet ) {
      ERRMSG("both verbose and quiet set; turning off quiet");
      bootargs.isquiet = 0;
    }

    /* get kernel version */
    get_kver();

    /* get subarchitecture name */
    get_subarch_name();

    /* initialize boot parameters */
    append_opts = strdup ("");

    /* Alpha is weird, we act totally differently based on what kind
       of firmware we were booted from */
#if #cpu(alpha)
    get_alpha_boot_method();
#endif

    /* setup rescue & drivers disks images filenames
     * (should be done before the first call to stderrToTTY) */
    setup_image_names();

    DEBUGMSG("dbootstrap debugging mode started, $Revision: 1.113 $");

    /* Are we on a serial console? */
    if (ioctl(0, TIOCGSERIAL, &ser) < 0)
      /* Not on a serial console */
      serialConsole = -1;
    else {
      /* Serial console on ttySx */
      serialConsole = ser.line;
      DEBUGMSG("serial console found on line %d", serialConsole);
    }
    /* Set term type and initialize it. */
    term = getenv("TERM");
    if (!term || (term && ! strncmp(term, "linux", 5))) {
      setenv("TERM", "linux", 1);
      setMono();
    }
#ifdef USE_LANGUAGE_CHOOSER
    setlocale (LC_CTYPE, "");   /* Language chooser relies on this call! */
#endif
    boxInit();

#ifdef USE_LANGUAGE_CHOOSER
    if ((lang = boxChooseLanguageVariant (available_languages ())) != NULL)
    {
        char *msgcat = (char *)malloc (PATH_MAX);

        snprintf (msgcat, PATH_MAX, "/etc/messages.%s", lang->msgcat);
        msgcat[PATH_MAX - 1] = '\0';

        DEBUGMSG ("loading message catalog %s", msgcat);

        if (LOAD_TRMFILE (msgcat) == 0)         /* Failed to load localized strings? */
        {
            if (LOAD_TRMFILE (TRMBACKUP) == 0)  /* Failed to load English strings as well? */
            {
                problemBoxEn ("An error occured while loading application messages.", "Problem");

                reboot (RB_AUTOBOOT);

                /* when not root and debugging */
                boxFinished();
                exit(1); 
            }
            else
            {
                char message[255];

                snprintf (message, sizeof (message), "An error occured while loading localized application messages from '%s'. English messages will be used instead.", msgcat);
                message[254] = '\0'; /* Just in case */

                problemBoxEn (message, "Problem");
            }
	}
        free (msgcat);
    }
    else
    {
        problemBoxEn ("You failed to choose a language.  Rebooting...", "Problem");

        reboot (RB_AUTOBOOT);

        /* when not root and debugging */
        boxFinished();
        exit (1);
    }
#else
    /* We can use problemBox only AFTER boxInit */
    /* So it's important that no (NO!!!) function before this call (LOAD_TRMFILE) uses gettext function */
    if (LOAD_TRMFILE(TRMFILE) == 0 && LOAD_TRMFILE(TRMBACKUP) == 0) {
	char message[255];
	snprintf(message, sizeof(message), 
		 "An error occured while loading application messages from '%s'.", TRMFILE);
    message[254] = '\0'; /* Just in case */
	problemBoxEn (message, "Problem");
	reboot(RB_AUTOBOOT);
	/* when not root and debugging */
	boxFinished();
	exit(1); 
    }
#endif

#if #cpu (m68k)
    if ((strncmp(Arch2, "BVME",     4) == 0)
    ||  (strncmp(Arch2, "Motorola", 8) == 0)) Arch2 = "VME";
    else
    if ( (strcmp(Arch2,"Atari")) && (strcmp(Arch2,"Amiga")) &&
	 (strcmp(Arch2,"Macintosh"))) {
      problemBox(_("Your m68k architecture is not supported yet."), _("Problem"));
      reboot(RB_AUTOBOOT);
    }

    /* stop kernel complaining to console about scsi floppy with no disk */
    if (strcmp(Arch2, "VME") == 0)
      klogctl(6, NULL, 0);
#elif #cpu (powerpc)
    if ((strncasecmp(Arch2, "Power", 5) == 0)
     || (strncasecmp(Arch2, "iMac",  4) == 0)) Arch2 = "PowerMac";
    else
    if ( (strcmp(Arch2,"CHRP")) && (strcmp(Arch2,"PReP"))
     && (strcasecmp(Arch2,"apus"))) {
      problemBox(_("Your PowerPC architecture is not supported yet."), _("Problem"));
      reboot(RB_AUTOBOOT);
    }
#endif

#ifdef SCSI_FLOPPY
    find_scsi_floppy();
#endif

#if #cpu(sparc)
    try_to_eject_root_floppy();
#endif

    Boot = Root = NULL;
    noSwap = 0;
    notCreatedBootFloppy = 1;
    notInstalledLILO = 1;
    Archive_Dir=NULL;

    InstallationRootDevice=block_device("/");
    if (!InstallationRootDevice) {
      /* net installation with no /dev/root device on nfsroot */
      /* leave time to see any error message */
      sleep(8);
      problemBox(_("I had trouble checking the choice of root device. I don't know what to use as the root fs."),_("Problem"));
#if #cpu(alpha)
      exit(1);
#else
      reboot(RB_AUTOBOOT);
      boxFinished();
      exit(1); /* when not root and debugging */
#endif
    }
   
#ifdef USE_LANGUAGE_CHOOSER
    release_notes (lang->msgcat);
#else
    release_notes (NULL);
#endif
    is_root_a_floppy ();

    main_menu ();

    boxFinished();

#if #cpu (m68k)
    /* restart kernel logging to console */
    if (strcmp(Arch2,"VME") == 0)
      klogctl(7,NULL,0);
#endif

    closelog();

    return 0;
}
#endif /* _TESTING_ */
