#include <assert.h>
#include <stdio.h>
#include <glob.h>
#include <sys/mount.h>
#include <sys/wait.h>
#include <syslog.h>

#include "lang.h"
#include "dbootstrap.h"
#include "util.h"
#include "../libfdisk/fdisk.h"	/* for fdisk_partitions #### Move this code to another file??? */

#define MAX_IDE_FLOPPIES 4
#define CM_MOUNTPOINT_DIR "/instmnt"
#ifdef _TESTING_
#define is_network_configured()   1
#define configure_network()       1
#define supported_filesystem(x)   1
char kernel_image_path[PATH_MAX+1] = "images-1.44/rescue.bin";
char drivers_path[PATH_MAX+1] = "drivers.tgz";
#endif /* _TESTING_ */

int dirc = 0;
char **dirv;
char *pattern;

/* #### This and several other globals probably belong in a header also once more of this is librified. */
char *cdrom_device = NULL;

#include <string.h>
#include <dirent.h>

/* ED: floppy filesystem type is not the same for all architectures */
#if #cpu(sparc) || #cpu(powerpc)
const char *fs_type_tab[] = { "ext2", NULL };
#else
const char *fs_type_tab[] = { "msdos", "ext2", NULL };
#endif

/* Return Values:
 *   0           successful
 *  DLG_CANCEL   Cancel (10
 *  -1           Error
 */
int mount_and_check_floppy(char *device, const char *type,
			   const char *text)
{
    int status;
    FILE *tmpstream;
    struct stat statbuf;
    const char **fs_type;
    char *actualType = NULL;

#if #cpu (m68k)
    if (strcmp(Arch2, "VME") == 0) {
	fs_type_tab[0] = "ext2";
	fs_type_tab[1] = NULL;
    }
#endif

    for (;;) {
	umount("/floppy");

	if (!device) {
	    struct d_choices choices[3];
	    char *dlist[3];
	    int rs, items;

	    items = 0;
	    if (strcmp(InstallationRootDevice, "/dev/fd0")) {
		choices[items].string = _("/dev/fd0  : first floppy drive");
		choices[items].tag = NULL;
		choices[items].state = 0;
		dlist[items] = "/dev/fd0";
		items++;
	    }
	    if (strcmp(InstallationRootDevice, "/dev/fd1")) {
		choices[items].string = _("/dev/fd1  : second floppy drive");
		choices[items].tag = NULL;
		choices[items].state = 0;
		dlist[items] = "/dev/fd1";
		items++;
	    }
#ifdef SCSI_FLOPPY
	    if (strcmp(InstallationRootDevice, "/dev/sfd0")) {
		choices[items].string = _("/dev/sfd0 : SCSI floppy drive");
		choices[items].tag = NULL;
		choices[items].state = 0;
		dlist[items] = "/dev/sfd0";
		items++;
	    }
#endif
	    snprintf(prtbuf, sizeof(prtbuf),
		     _("Please select the floppy drive you will use to read the %s."), text);
	    rs = menuBox(prtbuf, _("Select Floppy Disk Drive"), choices,
			items, 1);
	    if (rs == -1)
		return -1;
	    device = dlist[rs];
	}

	if (!strncmp(device, "/dev/fd", 7)
#ifdef SCSI_FLOPPY
	    || !strncmp(device, "/dev/sfd", 8)
#endif
	    ) {
	    snprintf(prtbuf, sizeof(prtbuf),
		     _("Please place the %s in the %s floppy drive."), text,
#ifdef SCSI_FLOPPY
		     (device[5] == 's') ? "SCSI" :
#endif
		     ((device[7] == '0') ? _("first") : _("second")));
	    if (twoButtonBox(prtbuf, _("Please insert disk"),
			     _("Continue"), _("Cancel")) == DLG_NO) {
		return DLG_CANCEL;
	    }
	}

	for (fs_type = fs_type_tab; *fs_type; fs_type++) {
	    snprintf(prtbuf, sizeof(prtbuf), "mount -r -t %s %s /floppy",
		     *fs_type, device);
	    status = execlog(prtbuf, LOG_INFO);
	    if (status == 0)	/* success */
		break;
	}
	if (!*fs_type) {	// all mounts failed
	    sleep(2);		// let see error messages given by mount
	    snprintf(prtbuf, sizeof(prtbuf), 
		     _("Unable to mount the %s.  You may have inserted the wrong floppy.  Please try again."),
		     text);
	    problemBox(prtbuf, _("Mount Failed"));
	    device = NULL;
	    continue;
	}

	if (! NAME_ISREG("/floppy/type.txt", &statbuf)) {
	    snprintf(prtbuf, sizeof(prtbuf),
		     _("This is not the %s. Please place the %s in the %s floppy drive and try again."), text, text,
#ifdef SCSI_FLOPPY
		     (device[5] == 's') ? "SCSI" :
#endif
		     ((device[7] == '0') ? _("first") : _("second")));
	    problemBox(prtbuf, _("Mount Failed"));
	    device = NULL;
	    continue;
	}
	if ((tmpstream = fopen("/floppy/type.txt", "r")) == NULL) {
	    perrorBox(_("Cannot open /floppy/type.txt "));
	    continue;
	}
	actualType = (char *) malloc(10);
	status = fscanf(tmpstream, "%10s", actualType) - 1;
	fclose(tmpstream);
	if (status) {
	    perrorBox(_("Cannot read /floppy/type.txt "));
	} else {
	    status = strcmp(type, actualType);
	    if (status) {
		if (!strcmp(type, "rescue") && !strcmp(actualType, "boot"))
		    snprintf(prtbuf, sizeof(prtbuf),
			    _("This floppy is for booting only. You need the standard %s disk for installation."), text);
		else
		    snprintf(prtbuf, sizeof(prtbuf),
			    _("This is not the %s. Please place the %s in the %s floppy drive and try again."), text, text,
#ifdef SCSI_FLOPPY
			     (device[5] == 's') ? "SCSI" :
#endif
			     ((device[7] == '0') ? _("first") : _("second")));

		problemBox(prtbuf, _("Wrong Floppy!"));
	    }
	}
	free(actualType);
	if (! status) {
	    return 0;
	}
    }
}

int match_find(const char *fileName, struct stat *statbuf)
{
    char *dir, *matched;

    dir = strdup(fileName);
    matched = strstr(dir, pattern);
    if ( matched && matched != dir && *(matched - 1) == '/' ) {
        *(matched - 1) = '\0';	/* now dir is the remainder */
	snprintf(prtbuf, sizeof(prtbuf), "%s/%s", dir, drivers_path);
	if (!NAME_ISREG(prtbuf, statbuf))
	  snprintf(prtbuf, sizeof(prtbuf), "%s/" DRVTGZFILE, dir);
	if ((disqtype != kernel) || (NAME_ISREG(prtbuf, statbuf))) {
	  DEBUGMSG("found pattern %s in dir %s", pattern, dir);
	  if ((dirc) && (!(dirc % 5)))
	    dirv = (char **) realloc(dirv, (dirc + 5) * sizeof(char *));
	  dirv[dirc] = strdup(dir);
	  dirc++;
	} else {
	  INFOMSG("found %s in %s, but can't find drivers %s", pattern, dir, drivers_path);
	}
    }
    return 0;
}

				/* note: if you pass 'text', we assume you use CM_MOUNTPOINT_DIR */
static int choose_archive_dir(char *text)
{
    char buffer[PATH_MAX+1];
    char buffer2[PATH_MAX+1];
    char *prefix;
    char *mountpoint, *descr = "", *def;
    static char *preventry = NULL;
    unsigned int checked_archive;
/*  glob_t globbuf; */
    struct stat statbuf;
#define ARC_default     100
#define ARC_list        101
#define ARC_manually    102
    struct d_choices ochoices[3] = {
	{NULL, _("default  : The default Debian archive"), 0},
	{NULL, _("list     : Choose from a list of all likely directories"), 0},
	
	{NULL, _("manually : Enter the directory manually"),
	     0}
    }, *choices;
    int olist[3] = { ARC_default, ARC_list, ARC_manually }, *ilist;
    int status, items = 2;
    choices = &ochoices[1];
    ilist = &olist[1];

    if (text == NULL) {
	text = _("Please choose the directory where the Debian archive resides.");
	mountpoint = "/";
	prefix = strdup ("");
    } else {			
	/* FIXME: why is the assumption that the mountpoint is
           CM_MOUNTPOINT_DIR tied to whether or not we pass the 'text'
           arg?  */
	mountpoint = CM_MOUNTPOINT_DIR;
	prefix = strdup(CM_MOUNTPOINT_DIR);
    }
    /* keep user entry between calls to choose_archive_dir */
    if (!preventry)
	preventry = strdup("");
    for (checked_archive = 0 ;; checked_archive++) {
	if ( ! checked_archive && ! bootargs.isverbose ) {
	    /* this is our first time around -- try to avoid the enterDirBox below */
	    if (is_cdrom_image() == 1)
		/* we've boot off a CD-ROM */
		buffer2[0] = '\0';
	    else if (is_cdrom_image() == 2)
		/* we've boot off a CD-ROM, live style */
		strcpy(buffer2, "/");
	    else
		/* what the hell, use our prefix anyway, it's worth a try */
		buffer2[0] = '\0';
	} else {
	    /* FIXME: unclear and confusing and nightmarish!  */
	    status = enterDirBox(_("Choose Debian Archive Location"),
				 text, prefix, preventry, buffer2, PATH_MAX);
	    if (status == DLG_CANCEL)
		return DLG_CANCEL;
	    else if (status != DLG_OKAY)
		return 1;
	}
	checked_archive++;
	free(preventry);
	preventry = strdup (buffer2);
	snprintf (buffer, sizeof (buffer), "%s%s", prefix, buffer2);
	if (NAME_ISDIR(buffer, &statbuf))
	    break;
	else {
	    snprintf (prtbuf, sizeof (prtbuf),
		      _("The supplied directory %s does not exist. Please enter a new one."), buffer2);
	    problemBox(prtbuf, _("Directory Error"));
	    if (checked_archive == 1 && is_cdrom_image()) { /* first time here, and the last */
		free(preventry);
		preventry = strdup(""); /* start over as normal */
	    }
	}
    }
    mountpoint = strdup(buffer);

    switch (disqtype) {
    case kernel:
	/* this could be a wildcard pattern but no multiple match */
	pattern = kernel_image_path;
	descr = _("the kernel and modules");
	break;
    case base:
	pattern = BASETGZ;
	descr = _("the Base system");
	break;
#if #cpu(alpha)
    case milo:
	pattern = milo_binary_path;
	descr = _("MILO and linload.exe");
	break;
    case apb:
	pattern = "APB/apb.exe";
	descr = _("APB and PALcode");
	break;
#endif
    }

    def = NULL;

    snprintf(prtbuf, sizeof(prtbuf), "%s/" ARCHIVE_LOCATION "/%s",
	     mountpoint, ARCHNAME, pattern);
    if (NAME_ISREG(prtbuf, &statbuf)) {
      DEBUGMSG("located file at %s", prtbuf);
      /* do an additional test when installing kernel and modules,
         since we need drivers disk */
      if (disqtype == kernel) {
	snprintf(prtbuf, sizeof(prtbuf),"%s/" ARCHIVE_LOCATION "/%s",
		 mountpoint, ARCHNAME, drivers_path);
	if (NAME_ISREG(prtbuf, &statbuf)) {
	  /* success */
	  snprintf(prtbuf, sizeof(prtbuf), "%s/" ARCHIVE_LOCATION, 
		   mountpoint, ARCHNAME);
	  def = strdup(prtbuf);
	} else {
	  INFOMSG("found kernel, but no drivers at %s", prtbuf);
	}
      } else {
	/* success */
	snprintf(prtbuf, sizeof(prtbuf), "%s/" ARCHIVE_LOCATION,
		 mountpoint, ARCHNAME);
	def = strdup(prtbuf);
      }
    } else if (disqtype == kernel) { /* try just the file name */
      INFOMSG(_("no file '%s' found, kernel not in default location"), prtbuf);
      snprintf(prtbuf, sizeof(prtbuf), "%s/" ARCHIVE_LOCATION "/" KERDISKFILE,
	       mountpoint, ARCHNAME);
      if (NAME_ISREG(prtbuf, &statbuf)) { /* ok, let's try for a flat setup */
	DEBUGMSG("checking for flat dir space for kernel and modules");
	snprintf(prtbuf, sizeof(prtbuf),"%s/" ARCHIVE_LOCATION "/" DRVTGZFILE,
		 mountpoint, ARCHNAME);
	if (NAME_ISREG(prtbuf, &statbuf)) {
	  DEBUGMSG("found kernel and modules in flat dir structure");
	  /* good, we have it now */
	  snprintf(prtbuf, sizeof(prtbuf), "%s/" ARCHIVE_LOCATION,
		   mountpoint, ARCHNAME);
	  def = strdup(prtbuf);
	  pattern = KERDISKFILE;
	} else {
	  INFOMSG("found kernel, but no drivers at %s", prtbuf);
	}
      }
    } else {
      INFOMSG(_("no file '%s' found, Debian archive not in default location"), 
	      prtbuf);
    }
    if (def) {
	snprintf(prtbuf, sizeof(prtbuf),
		 _("Please select the directory containing a file %s that you will use to install %s.\nUnless you really know what you need choose the default."),
		 pattern, descr);
	choices--;
	ilist--;
	items++;
    } else
	snprintf(prtbuf, sizeof(prtbuf),
		 _("Please select the directory containing a file %s that you will use to install %s."),
		 pattern, descr);

    if (def && ! bootargs.isverbose) /* if there's a default, lets go with it */
      status = 0;
    else
      status = menuBox(prtbuf, _("Select Debian Archive path"), 
		       choices, items, 1);

    if (status != -1) {
	int data;
	data = ilist[status];
	status = 0;

	switch (data) {
	case ARC_default:
	    Archive_Dir = strdup(def);
	    free(def);
	    break;
	case ARC_list:
	    /*
	     * Build the list.
	     */
	    if (NAME_ISLNK(mountpoint, &statbuf) && mountpoint[strlen (mountpoint)-1] != '/') {
		snprintf (buffer, sizeof (buffer), "%s/", mountpoint);
		free (mountpoint);
		mountpoint = strdup (buffer);
	    }
	    snprintf(prtbuf, sizeof(prtbuf),
		     _("The installation program is building a list of all directories containing the file '%s' that can be used to install %s."), 
		     pattern, descr);
	    pleaseWaitBox(prtbuf);
	    dirc = 0;
	    dirv = (char **) malloc(5 * sizeof(char *));
	    DEBUGMSG(_("looking for %s under %s"), pattern, mountpoint);
	    recursiveAction(mountpoint, TRUE, FALSE, FALSE, match_find, NULL);
	    
	    /* If nothing found and we are looking for kernel, check for flat dir */
	    if (dirc < 1 && disqtype == kernel) {
		pattern = KERDISKFILE;
		DEBUGMSG(_("looking for %s under %s"), pattern, mountpoint);
		recursiveAction(mountpoint, TRUE, FALSE, FALSE, match_find, NULL);
	    }
#if #cpu(alpha)
	    /* Similar stupid kludges for MILO and APB (gack) */
	    if (dirc < 1 && disqtype == milo) {
		pattern = Arch2;
		DEBUGMSG(_("looking for %s under %s"), pattern, mountpoint);
		recursiveAction(mountpoint, TRUE, FALSE, FALSE, match_find, NULL);
	    }
	    if (dirc < 1 && disqtype == apb) {
		pattern = "apb.exe";
		DEBUGMSG(_("looking for %s under %s"), pattern, mountpoint);
		recursiveAction(mountpoint, TRUE, FALSE, FALSE, match_find, NULL);
	    }
#endif
	    
	    boxPopWindow();

	    /*
	     * List menu
	     */
	    if (dirc > 0) {
		int i;
		struct d_choices *nchoices;
		nchoices = calloc(dirc, sizeof(struct d_choices));
		for (i = 0; i < dirc; i++) {
		    /* shorten paths that are too long to be displayed */
		    if (strlen(dirv[i]) > 56 - 3 - 2) {	/* text - left - 2 */
			strcpy(prtbuf, "...");
			strcat(prtbuf + 3,
			       dirv[i] + strlen(dirv[i]) - 56 + 3 + 3);
		    } else
			strcpy(prtbuf, dirv[i]);
		    nchoices[i].string = strdup(prtbuf);
		}
		snprintf(prtbuf, sizeof(prtbuf),
			 _("Please select the directory that you will use to install %s from."), descr);
		if ( dirc == 1 && bootargs.isquiet )
		  status = 0;	/* quiet: use the only match automatically */
		else 
		  status = menuBox(prtbuf, _("Select Archive path"), nchoices,
				   dirc, 1);
		if (status != -1) {
		    Archive_Dir = strdup(dirv[status]);
		    status = 0;
		}
		while (dirc) {
		    dirc--;
		    free(dirv[dirc]);
		    free(nchoices[dirc].string);
		}
		free(dirv);
		free(nchoices);
		if (status != 0)
		    return DLG_CANCEL;
	    } else {
		if (disqtype == kernel) {
		    snprintf (buffer, sizeof (buffer), "%s, %s", pattern, drivers_path);
		    snprintf(prtbuf, sizeof(prtbuf),
			     _("The installation program couldn't find any directory containing the files %s."), buffer);
		} else {
		    snprintf(prtbuf, sizeof(prtbuf),
			     _("The installation program couldn't find any directory containing the file %s."), pattern);
		}
		problemBox(prtbuf, _("File not found!"));
		status = -1;
	    }
	    break;
	case ARC_manually:
	    snprintf(prtbuf, sizeof(prtbuf),
		     _("Please enter the name of the directory that contains the Archive files.\nThe installation medium is mounted below %s/ ."),
		     mountpoint);
	    ;
	    status = enterDirBox (_("Enter the Archive directory"), prtbuf, prefix, mountpoint, buffer, PATH_MAX);
	    if (status == DLG_CANCEL)
		return DLG_CANCEL;
	    else if (status != DLG_OKAY) {
		status = -1;
	    } else {
		Archive_Dir = strdup(buffer);
		snprintf(prtbuf, sizeof(prtbuf), "%s/%s", Archive_Dir,
			 pattern);
		if (!NAME_ISREG(prtbuf, &statbuf)) {
		    INFOMSG("manual dir specified for %s, but %s not found", descr, prtbuf);
		    snprintf(prtbuf, sizeof(prtbuf),
			     _("%s does not contain the file %s that is needed to install %s"), 
			     Archive_Dir, pattern, descr);
		    problemBox(prtbuf, _("File not found!"));
		    status = -1;
		} else {
		    snprintf(prtbuf, sizeof(prtbuf), "%s/%s", Archive_Dir,
			     drivers_path);
		    if ((disqtype == kernel)
			&& (!NAME_ISREG(prtbuf, &statbuf))) {
			snprintf(prtbuf, sizeof(prtbuf),
				 _("%s does not contain the file %s that is needed to install %s"), Archive_Dir, drivers_path, descr);
			problemBox(prtbuf, _("File not found!"));
			status = -1;
		    }
		}
	    }
	    break;
	}
    } else {	/* It's the Cancel button again */
	return DLG_CANCEL;
    }
    free(mountpoint);
    return status;
}


/*
 * "/proc/sys/dev/cdrom/info", on bittersweet, contains:
 * 8<------------------------------------------------->8
 * CD-ROM information, Id: cdrom.c 2.56 1999/09/09
 *
 * drive name:             sr0     hdd
 * drive speed:            6       4
 * drive # of slots:       1       0
 * Can close tray:         1       1
 * Can open tray:          1       1
 * Can lock tray:          1       1
 * Can change speed:       1       1
 * Can select disk:        0       0
 * Can read multisession:  1       1
 * Can read MCN:           1       1
 * Reports media changed:  1       1
 * Can play audio:         1       1
 *
 *
 * 8<------------------------------------------------->8
 *
 * Note that there are actually tab characters in the file; I've
 * untabified so it stays lined up in this comment.  I have verified
 * that this file does not exist on a machine with no CDROM attached.
 * - karlheg 2000.03.20
 * Linux bittersweet 2.2.14 #9 Fri Mar 17 00:50:19 PST 2000 i586 unknown
 */

int
have_cdrom ()
{
    struct stat statbuf;
    int status;
    if ((status = stat("/proc/sys/dev/cdrom/info", &statbuf)) == 0)
	return 1;
    return 0;
}

static
char *
get_cd_type_description (const char *dev)
{
    struct cd_type_alist_struct {
	char *dev;
	char *description;
    } cd_type_alist[] = {
	{"sr0",    _("SCSI (first CD-ROM in chain)")},
	{"sr1",    _("SCSI (second CD-ROM in chain)")},
	{"hda",    _("ATAPI (IDE), first drive on the primary controller")},
	{"hdb",    _("ATAPI (IDE), second drive on the primary controller")},
	{"hdc",    _("ATAPI (IDE), first drive on the secondary controller")},
	{"hdd",    _("ATAPI (IDE), second drive on the secondary controller")},
	{"hde",    _("ATAPI (IDE), first drive on the third controller")},
	{"hdf",    _("ATAPI (IDE), second drive on the third controller")},
	{"hdg",    _("ATAPI (IDE), first drive on the fourth controller")},
	{"hdh",    _("ATAPI (IDE), second drive on the fourth controller")},
	{"cm206",  _("Philips/LMS CM206 proprietary CDROM drive")},
	{"mcd",    _("Mitsumi proprietary CDROM drive")},
	{"mcdx",   _("Mitsumi proprietary CDROM drive")},
	{"sbpcd",  _("Matsushita/Panasonic/Creative/Longshine/TEAC proprietary CDROM drive")},
	{"cdu31a", _("Sony CDU31A/CDU33A proprietary CDROM drive")},
	{"wtf0",   "What are some other devices?"}, /* #### */
	{NULL, NULL}
    };
    int i;
    for (i = 0; cd_type_alist[i].dev != NULL; i++) {
	if (!strncmp(dev, cd_type_alist[i].dev, strlen (cd_type_alist[i].dev))) {
	    return cd_type_alist[i].description;
	}
    }
    return "No description defined for this type of CD-ROM.";
}

static int
choose_cdrom (void)
{
    FILE             *cd_info;
    char             *p_line, *line       = NULL;
    size_t            line_size           = 0;
    int               status;
    struct d_choices *choices;
    int               i_choices           = 0;
    int               n_choices_allocated = 1;

    /* Should be guaranteed on entry that the cd_info file exists - use have_cdrom() */
    /* #### I get a warning here about assignment makes pointer from integer without a cast.  Why?? */
    if ((cd_info = fopen ("/proc/sys/dev/cdrom/info", "r")) == NULL) {
	return -1;		/*  Shouldn't happen if I can stat it, right ? */
    }

    /* #### Should check for malloc error; must decide what/where to throw */
    choices = (struct d_choices *) malloc (n_choices_allocated * sizeof (struct d_choices));

    /* Read the file and build a list of available CD devices. */
    getline (&line, &line_size, cd_info);
    getline (&line, &line_size, cd_info);
    getline (&line, &line_size, cd_info);
    p_line = line + strlen(line) - 1; /* Point at \n at end of string. */
    *p_line-- = '\0';		/* chop \n */
    while (*p_line != ':') {

	while (!isspace(*(p_line - 1))) --p_line; /* scan backwards until char-before isspace() */

	{ /* new local scope; not a while loop */
	    char dev[5 + strlen (p_line) + 1];
	    strcpy (dev, "/dev/");
	    strcpy (dev + 5, p_line);
	    /* #### Should check for malloc error; must decide what/where to throw. */
	    choices[i_choices].tag    = strdup (dev);
	    choices[i_choices].string = get_cd_type_description (p_line);
	    choices[i_choices].state  = ((cdrom_device != NULL) && !strcmp (choices[i_choices].tag,
									    cdrom_device));
	    i_choices++;
	    if (i_choices + 1 > n_choices_allocated) {
		/* #### Should check for malloc error; must decide what/where to throw */
		void *p = realloc (choices, ((n_choices_allocated + 2) * sizeof (struct d_choices)));
		choices = (struct d_choices *) p;
		n_choices_allocated += 2;
	    }
	} /* end scope block */

	while (isspace(*(--p_line))) /* Scan backwards for end of previous non whitespace. */ ;
	*(p_line + 1) = '\0';

    } /* end of "while not looking-at ':'" */
    fclose (cd_info);

    if (i_choices > 1) {
	if ((status = menuBox (_("You have more than one CD-ROM drive.  Please select "
				 "the drive from which you want to install Debian"),
			       _("Select CD-ROM drive"), choices, i_choices, 1)) < 0) {
	    /* Exit value from old code... #### should be standardized across all of (lib)dbootstrap. */
	    status = 1;		/* #### DB_EXIT_CANCEL */
	    goto cleanup_and_exit;
	}
	cdrom_device = strdup (choices[status].tag);
    }
    else {			/* There must be at least one CD-ROM - use has_cdrom() */
	cdrom_device = strdup (choices[0].tag);
    }

    { /* Create the /dev/cdrom -> cdrom_device symlinks. */
	char *not_dir = cdrom_device + 5; /* "/dev/" is 5 characters */
	char pwd[PATH_MAX + 1];

	INFOMSG("making symlink from /dev/%s to /dev/cdrom", cdrom_device);
	getcwd (pwd, PATH_MAX);
	chdir ("/dev");
	unlink ("cdrom");
	symlink (not_dir, "cdrom");
	chdir (pwd);
    }

    umount(CM_MOUNTPOINT_DIR);

    problemBox(_("Please place the first Debian CD-ROM in the CD-ROM drive."),
	       _("Please insert the CD-ROM"));

    if (execlog("mount -t iso9660 -o nojoliet -r /dev/cdrom " CM_MOUNTPOINT_DIR, LOG_INFO)) {
	problemBox(_("The CD-ROM was not mounted successfully."), _("Mount Failed"));
	/* db_errno = DB_CANNOT_MOUNT or something */
	status = -1;		/* #### DB_EXIT_ERROR */
	goto cleanup_and_exit;
    }


    if ((status = choose_archive_dir(_("Please choose the path inside the CD-ROM where "
				       "the Debian Archive resides.")))) {
	/* failure, umount */
	umount(CM_MOUNTPOINT_DIR);
    }

 cleanup_and_exit:
    {
	int i;
	for (i = i_choices - 1; i >= 0; i--) {
	    free (choices[i].tag);
	}
	free (choices);		/* Choose Debian! */
    }
    return status;
}

struct ide_floppy_ident_t {
    enum {LS_120 = 1, ZIP_100, ZIP_250} type;
    const char *ident_string;
    const char *symlink_name;
    const char *partition;
} ide_floppy_ident[] = {
    { LS_120,  "LS-120",         "ls120",    "" },
    { ZIP_100, "IOMEGA ZIP 100", "zip",      "4" },
    { ZIP_250, "IOMEGA ZIP 250", "zip",      "4" },
    { 0, 0 }
};
    
struct ide_floppy_t {
    int type;
    char device;
} ide_floppies[MAX_IDE_FLOPPIES];

int have_idefloppy(void)
{
    FILE *fp = NULL;
    glob_t gl;
    char buf[1024];
    char *p;
    int i, j;
    int found = 0;
    
    if (glob("/proc/ide/hd?/media", 0, NULL, &gl) == 0) {
        for (i = 0; i < gl.gl_pathc && found < MAX_IDE_FLOPPIES; i++) {
	    fp = fopen(gl.gl_pathv[i], "r");
	    if (fp) {
	        fgets(buf, sizeof(buf), fp);
		fclose(fp);
		if (strncmp(buf, "floppy", 6) == 0) {
		    strncpy(buf, gl.gl_pathv[i], sizeof(buf));
	            p = strrchr(buf, '/');
		    assert(p != NULL);
		    *(p+1) = 0;
		    strncat(buf, "model", sizeof(buf));
		    if ((fp = fopen(buf, "r"))) {
		        fgets(buf, sizeof(buf), fp);
			for (j = 0; ide_floppy_ident[j].type != 0; j++) {
			    if (strncmp(buf, ide_floppy_ident[j].ident_string,
			                strlen(ide_floppy_ident[j].ident_string))
					== 0) {
			        ide_floppies[found].type = ide_floppy_ident[j].type;
			        ide_floppies[found].device = gl.gl_pathv[i][12];
			        found++;
			    }
			} 
	                fclose(fp);
		    }
		}
	    }
	}
	globfree(&gl);
    }
    return found;
}

/* Return Values:
 *   0           successful
 *  DLG_CANCEL   Cancel (10)
 *  -1           Error
 */
static int choose_idefloppy(void)
{
    int status = 0;
    int i;
    struct d_choices *choices;
    int devcount = have_idefloppy();
    char *device = NULL;

    choices = malloc(sizeof(struct d_choices) * (devcount+1));
    
    for (i = 0; i < devcount; i++) {
       asprintf(&choices[i].tag, "/dev/hd%c%s",
	        ide_floppies[i].device,
                ide_floppy_ident[ide_floppies[i].type-1].partition);

       choices[i].string = strdup(ide_floppy_ident[ide_floppies[i].type-1].ident_string);
    } 
    choices[devcount].tag = "custom";
    choices[devcount].string = _(" none of the above; prompt for device");
    if ((status = menuBox (_("Please select the proper IDE floppy device to use"),
			       _("Select IDE floppy drive"), choices, devcount+1, 1)) < 0) {
        return DLG_CANCEL;
    } else {
        if (strcmp(choices[status].tag, "custom") == 0) {
            device = inputBox(_("IDE floppy device: "),
			      _("Please enter IDE floppy device name"),
			      "/dev/hd");
	    if (device == NULL)
		return DLG_CANCEL;
	} else {
	    device = strdup (choices[status].tag);
	}
    }

    if (status >= 0) {
       problemBox(_("Please place the disk in your IDE floppy drive"),
	          _("Please insert the disk"));

        if (!system("cat /proc/mounts | grep -q " CM_MOUNTPOINT_DIR))
  	    execlog("umount " CM_MOUNTPOINT_DIR, LOG_DEBUG);

        snprintf(prtbuf, sizeof(prtbuf), "mount -t auto %s %s",
                 device, CM_MOUNTPOINT_DIR);
	syslog(LOG_CRIT, prtbuf);
        if (execlog(prtbuf, LOG_INFO)) {
	    problemBox(_("The disk was not mounted successfully."), _("Mount Failed"));
	    status = -1;
        } else {
            status = choose_archive_dir (_("Please choose the path on the disk"
				           " where the Debian Archive resides."));
        }
    }
    
    free(device);
    return status;
}

static int choose_harddisk(void)
{
    struct fdisk_partition *p;
    int status;
    char *type;

    if (!system("cat /proc/mounts | grep -q " CM_MOUNTPOINT_DIR))
      execlog("umount " CM_MOUNTPOINT_DIR, LOG_DEBUG);

    p = select_not_mounted(_("Please select the partition where your Debian Archive resides."), 
			   _("Select Partition"), 5,
			   FSTYPE_EXT2, FSTYPE_MINIX, FSTYPE_MSDOS, 
			   FSTYPE_AFFS, FSTYPE_HFS);
    if (!p)
	return DLG_CANCEL;

    type = fdisk_fstype_name_of(p->type);
    if (! type) {
	snprintf(prtbuf, sizeof(prtbuf),
		 _("Don't know which filesystem type to apply to a %s partition."), 
		 fdisk_sysname(p->type));
	problemBox(prtbuf, _("Cannot mount"));
	return 1;
    }
#ifndef _TESTING_
    snprintf(prtbuf, sizeof(prtbuf),
	     "mount -t %s -r %s " CM_MOUNTPOINT_DIR, type, p->name);
    if (execlog(prtbuf, LOG_INFO)) {
      problemBox(_("The partition was not mounted successfully."),
		 _("Mount Failed"));
      return 1;
    }
#endif
    status = choose_archive_dir(_("Please choose the path inside the harddisk filesystem where the Debian Archive resides."));
    if (status && status != DLG_CANCEL)
    {
      if (!system("cat /proc/mounts | grep -q " CM_MOUNTPOINT_DIR))
	execlog("umount " CM_MOUNTPOINT_DIR, LOG_DEBUG);
    }
    return status;
}

struct execlog_env_s {
    char *cmd_buf;
    int loglevel;
    int status;
};

static void
choose_NFS_server_execlog (void *result)
{
    struct execlog_env_s *res = (struct execlog_env_s *) result;
    res->status = execlog (res->cmd_buf, res->loglevel);
    return;
}

static int choose_NFS_server(void)
{
    char *nfsmountpath;
    static char *preventry = NULL;
    int status;

    /* PowerPC and SPARC have nfs built in the kernel. */
    /* try to load nfs module only if not already in kernel */
    if (!supported_filesystem("nfs")) {
	snprintf(prtbuf, sizeof(prtbuf), "modprobe nfs");
	if (execlog(prtbuf, LOG_INFO)) {
            snprintf(prtbuf, sizeof(prtbuf),
		_("Error loading NFS module. Perhaps you need to install the drivers first?"));
	    problemBox(prtbuf, _("Cannot attempt NFS mount"));
	}
	return 1;
    }

    for (;;) {
	/* keep user entry between calls to choose_NFS_server */
	if (!preventry)
	    preventry = strdup("");
	nfsmountpath = inputBox(_("Please choose the NFS server and the mount path of the NFS filesystem that contains the Debian archive.\nEnter them in this way:  server:/ftp/debian"), _("Choose Debian NFS filesystem"), preventry);
	if (!nfsmountpath)
	    return DLG_CANCEL;
	free(preventry);
	preventry = nfsmountpath;
	{
	    struct execlog_env_s result;
	    int threw;
	    if (! system ("cat /proc/mounts | grep -q " CM_MOUNTPOINT_DIR)) {
		result.cmd_buf = "umount " CM_MOUNTPOINT_DIR;
		result.loglevel = LOG_DEBUG;
		snprintf (prtbuf, sizeof (prtbuf),
			  _("Unmounting %s."), CM_MOUNTPOINT_DIR);
		threw = do_interruptable (prtbuf,
					  &choose_NFS_server_execlog,
					  (void *) &result);
		if (threw) {
		    int status;
		    system ("killall -KILL umount");
		    sleep(1);
		    waitpid (WAIT_MYPGRP, &status, WNOHANG);
		}
	    }
	    {
		char msg_buf[ sizeof (prtbuf) ];
		snprintf (msg_buf, sizeof (msg_buf),
			  _("Mounting %s on %s via NFS."),
			  nfsmountpath, CM_MOUNTPOINT_DIR);
		snprintf (prtbuf, sizeof(prtbuf),
			  "mount -r -t nfs -o nolock %s " CM_MOUNTPOINT_DIR,
			  nfsmountpath);
		result.cmd_buf = prtbuf;
		result.loglevel = LOG_INFO;
		threw = do_interruptable (msg_buf,
					  &choose_NFS_server_execlog,
					  (void *) &result);
		if (threw) {
		    int status;
		    system ("killall -KILL mount");
		    sleep(1);
		    waitpid (WAIT_MYPGRP, &status, WNOHANG);
		}
	    }
	    if (threw) {
		problemBox (_("Interrupted by operator."), _("NFS mount failed"));
		return 1;
	    }
	    else if (result.status) {
		snprintf (prtbuf, sizeof (prtbuf),
			  _("Error mounting NFS filesystem `%s', please try again."),
			  nfsmountpath);
		problemBox(prtbuf, _("NFS mount failed"));
	    }
	    else {
		break;
	    }
	}
    }
    if ((status = choose_archive_dir(_("Please choose the path inside the mounted NFS filesystem where the Debian Archive resides."))))
    {
      if (!system("cat /proc/mounts | grep -q " CM_MOUNTPOINT_DIR))
	execlog("umount " CM_MOUNTPOINT_DIR, LOG_DEBUG);
    }
    return status;
}

/* Maybe this belongs elsewhere, but I'll define it right here and now since this is where I need it first. */
int have_unused_partitions (void)
{
    struct fdisk_partition * partition;
    int ret = 0;
    for (partition = fdisk_partitions; partition != NULL; partition = partition->next) {
	if (!partition->in_use && fdisk_fstype_of (partition->type) != FSTYPE_EXTPART) {
	    ret = 1;
	    break;
	}
    }
    return ret;
}

#if 0				/* Test program for the record. */
/* My attempt at floppy drive detection.
 * gcc -o check_for_floppy check_for_floppy.c
 * time sudo ./check_for_floppy
 */

#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <linux/hdreg.h>
#include <errno.h>

void try_dev (const char *dev)
{
  struct hd_geometry geom;
  int fd, status;

  printf ("Trying %s ...\n", dev);
  if ((fd = open (dev, O_RDONLY)) < 0) {
    printf ("Error: open %s: %d, %s\n", dev, errno, strerror(errno));
  }
  printf ("Open returned, fd=%i\n", fd);
  if (fd >= 0) {
    printf ("Trying ioctl ...\n");
    if ((status = ioctl (fd, HDIO_GETGEO, &geom)) < 0) {
      printf ("Error: ioctl %s: %d, %s\n", dev, errno, strerror(errno));
    }
    else {
      printf ("heads=%i, sectors=%i, cylinders=%i\n",
	      (int) geom.heads, (int) geom.sectors, geom.cylinders);
    }
    close(fd);
  }
}

int
main (int argc, char **argv)
{
  try_dev("/dev/fd0");
  try_dev("/dev/fd1");
}
#endif

#if 0
int
has_floppy (dev_t dev)
{
    STUB!!
}
#endif

/* Return Values:
 *   0           successful
 *  DLG_CANCEL   Cancel (10)
 *  -1           Error
 */
int choose_medium(void)
{
    struct stat statbuf;
#define MED_none          100
#define MED_default       101
#define MED_fd0           102
#define MED_fd1           103
#define MED_cdrom         104
#define MED_harddisk      105
#define MED_mounted       106
#define MED_nfs           107
#define MED_sfd0          108
#define MED_netfetch      109
#define MED_idefloppy     110
#define NITEMS 10
    int old_medium_device, offer_default = 0, items = 0, status =
	0, ilist[NITEMS];
    static int Medium_Device = MED_none;
    struct d_choices choices[NITEMS];
    char *old_archive_dir;
    
    memset(choices, 0, NITEMS * sizeof(struct d_choices));

    if (Archive_Dir) {
	switch (disqtype) {
	case kernel:
	    snprintf(prtbuf, sizeof(prtbuf), "%s/%s", Archive_Dir,
		     kernel_image_path);
	    if (NAME_ISREG(prtbuf, &statbuf)) {
		snprintf(prtbuf, sizeof(prtbuf), "%s/%s", Archive_Dir,
			 drivers_path);
		if (NAME_ISREG(prtbuf, &statbuf))
		    offer_default = 1;
	    } else {
		snprintf(prtbuf, sizeof(prtbuf), "%s/" KERDISKFILE, Archive_Dir);
		if (NAME_ISREG(prtbuf, &statbuf)) {
		    snprintf(prtbuf, sizeof(prtbuf), "%s/" DRVTGZFILE, Archive_Dir);
		    if (NAME_ISREG(prtbuf, &statbuf))
			offer_default = 1;
		}
	    }
	    break;
	case base:
	    snprintf(prtbuf, sizeof(prtbuf), "%s/%s", Archive_Dir,
		     BASETGZ);
	    if (NAME_ISREG(prtbuf, &statbuf))
		offer_default = 1;
	    break;
#if #cpu(alpha)
	case milo:
	    snprintf(prtbuf, sizeof(prtbuf), "%s/%s", Archive_Dir,
		     milo_binary_path);
	    if (NAME_ISREG(prtbuf, &statbuf))
		offer_default = 1;
	    break;
	case apb:
	    snprintf(prtbuf, sizeof(prtbuf), "%s/APB/apb.exe",
		     Archive_Dir);
	    if (NAME_ISREG(prtbuf, &statbuf))
		offer_default = 1;
	    break;
#endif
	}
	if (offer_default) {
	    choices[items].string = _("default    : Previous selection");
	    ilist[items] = MED_default;
	    items++;
	}
    }
    /* put CD-ROM at the top of the list, since this is what most people use */
    if (have_cdrom()) {
	choices[items].string = _("cdrom      : CD-ROM drive");
	ilist[items] = MED_cdrom;
	items++;
    }
    if (strcmp(InstallationRootDevice, "/dev/fd0")) {
	choices[items].string = _("/dev/fd0   : first floppy drive");
	ilist[items] = MED_fd0;
	items++;
    }
    if (strcmp(InstallationRootDevice, "/dev/fd1")) {
	choices[items].string = _("/dev/fd1   : second floppy drive");
	ilist[items] = MED_fd1;
	items++;
    }
#ifdef SCSI_FLOPPY
    if (strcmp(InstallationRootDevice, "/dev/sfd0")) {
	choices[items].string = _("/dev/sfd0  : SCSI floppy drive");
	ilist[items] = MED_sfd0;
	items++;
    }
#endif
    if (have_idefloppy()) {
	choices[items].string = _("IDE floppy : LS/120 or IOMEGA ZIP drive");
	ilist[items] = MED_idefloppy;
	items++;
    }
    
    if (have_unused_partitions()) {
	choices[items].string = _("harddisk   : partition on the hard disk");
	ilist[items] = MED_harddisk;
	items++;
    }

    choices[items].string = _("mounted    : already mounted filesystem");
    ilist[items] = MED_mounted;
    items++;

    if (is_network_up(1) && supported_filesystem("nfs")) {
	choices[items].string = _("nfs        : NFS (network filesystem)");
	ilist[items] = MED_nfs;
	items++;
    }

    if (is_network_up(1)) {
	choices[items].string = _("network    : retrieve from network");
	ilist[items] = MED_netfetch;
	items++;
    }

    if (is_cdrom_image() == 1) {
	if (bootargs.isverbose)
	    status = menuBox(_("Please select the medium you will use to install the system.\nSince you are installing from CDROM, you should choose the `cdrom' option unless you have a specific reason not to."),
		_("Select Installation Medium"), choices, items, 1);
	else {
	    int i;
	    for (i = 0 ; ilist[i] != MED_cdrom; i++); /* find our item */
	    status = i;
	}
    } else if (is_cdrom_image() == 2) {
	if (bootargs.isverbose)
	    status = menuBox(_("Please select the medium you will use to install the system.\nSince you are installing from CDROM, you should choose the `mounted' option unless you have a specific reason not to."),
		_("Select Installation Medium"), choices, items, 1);
	else {
	    int i;
	    for (i = 0 ; ilist[i] != MED_mounted; i++); /* find our item */
	    status = i;
	}
    } else
	status = menuBox(_("Please select the medium you will use to install the system."),
		_("Select Installation Medium"), choices, items, 1);
    if (status == -1) {		/* It's the Cancel button */
	return DLG_CANCEL;
    }
    old_medium_device = Medium_Device;
    old_archive_dir = Archive_Dir;
    Archive_Dir = NULL; /* we are going to reallocate it here */
    Medium_Device = ilist[status];

    status = 0;
    switch (Medium_Device) {
    case MED_default:
	Medium_Device = old_medium_device;
	Archive_Dir = strdup(old_archive_dir);
	break;
    case MED_fd0:
	Archive_Dir = strdup("/dev/fd0");
	break;
    case MED_fd1:
	Archive_Dir = strdup("/dev/fd1");
	break;
#ifdef SCSI_FLOPPY
    case MED_sfd0:
	Archive_Dir = strdup("/dev/sfd0");
	break;
#endif
    case MED_mounted:
	status = choose_archive_dir(NULL);
	if (status == DLG_CANCEL)
	    return DLG_CANCEL;
	break;
    case MED_cdrom:
	status = choose_cdrom();
	break;
    case MED_idefloppy:
	status = choose_idefloppy();
	if (status == DLG_CANCEL)
	    return DLG_CANCEL;
	break;
    case MED_harddisk:
	status = choose_harddisk();
	break;
    case MED_nfs:
	if (! is_network_up(0) ) {
	    wideMessageBox(_("\nYou must first configure your network. Please do so now.\n"),
			   _("Network Not Yet Configured"));
	    status = configure_network();
	    if (status == DLG_CANCEL)
		return DLG_CANCEL;
	}
	if (status == 0) {
	    status = choose_NFS_server();
	    if (status == DLG_CANCEL)
		return DLG_CANCEL;
	}
	break;
    case MED_netfetch:
        if (! is_network_up(0)) {
            wideMessageBox(_("\nYou must first configure your network. Please do so now.\n"), 
			   _("Network Not Yet Configured"));
            status = configure_network();
	    if (status == DLG_CANCEL)
		return DLG_CANCEL;
	}
        if (status == 0) {
            Archive_Dir = strdup("netfetch");
	}
	break;
    }
    if (status == 0) {
	/* success, out with the old one */
	if (old_archive_dir)
	    free(old_archive_dir);
    } else {
	/* failure, restore the old one */
	if (Archive_Dir)
	    free(Archive_Dir);
	Archive_Dir = old_archive_dir;
    }
    return status;
}

#ifdef DO_EJECT
int do_eject(void)
{
#define EJ_none          -1
#define EJ_fd0           0
#define EJ_fd1           1
    struct d_choices ch[2] = { 
	    {NULL, _("/dev/fd0  : First floppy drive"), 0}, 
	    {NULL, _("/dev/fd1  : Second floppy drive"), 0} };
    static int Eject_Device = EJ_none;

    Eject_Device =
	menuBox(_("Please select the disk you want to eject."),
		_("Select disk to eject"), ch, 2, 1);
    if (Eject_Device == -1) {
	return -1;
    }

    switch (Eject_Device) {
    case EJ_fd0:
	eject_floppy("/dev/fd0");
	break;
    case EJ_fd1:
	eject_floppy("/dev/fd1");
	break;
    }
    return 0;
}
#endif

#ifdef _TESTING_
/*
 * To test, compile using: make choose_medium_test
 */
int main(void)
{
    int status;
    LOAD_TRMFILE("test.trm");
    get_kver();
    Archive_Dir = NULL;
    disqtype = kernel;
    InstallationRootDevice = strdup("/dev/sdc");
    fdisk_reread ();
    boxInit();
    /*
    status = choose_archive_dir (_("Please choose the path inside the CD-ROM where the Debian Archive resides."));
    status = choose_archive_dir (NULL);
    status = choose_cdrom();
    */
    status = choose_medium();
    boxFinished();
    if (status == DLG_CANCEL) {    
	fprintf(stderr, "\nCancelled!\n");
	INFOMSG("failed");
    } else if (status) {    
	fprintf(stderr, "\nFailed!\n");
	INFOMSG("failed");
    } else {
	fprintf(stderr, "\nArchive_Dir = %s\n", Archive_Dir);
	INFOMSG("Archive_Dir = %s", Archive_Dir);
    }
    free(Archive_Dir);
    return 0;
}
#endif

/*
 * Local variables:
 *   c-basic-offset: 4
 *   indent-tabs-mode: t
 * End:
 */
