

/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * Modified by the GTK+ Team and others 1997-1999.  See the AUTHORS
 * file for a list of people on the GTK+ Team.  See the ChangeLog
 * files for a list of changes.  These files are distributed with
 * GTK+ at ftp://ftp.gtk.org/pub/gtk/. 
 */

/* Note: this is basically the gtkbutton code, hacked a bit to do
   needed it to do.  As of 11/17/99 I've contributed a total of
   two comment marks - just so no one thinks I'm trying to take
   credit for *anyone's* work here. :)  - Rob
*/

#include <string.h>
#include "gtkmovie.h"
#include <gtk/gtklabel.h>
#include <gtk/gtkmain.h>
#include <gtk/gtksignal.h>


#define CHILD_SPACING     1
#define DEFAULT_LEFT_POS  4
#define DEFAULT_TOP_POS   4
#define DEFAULT_SPACING   7


enum {
  PRESSED,
  RELEASED,
  CLICKED,
  ENTER,
  LEAVE,
  LAST_SIGNAL
};
enum {
  ARG_0,
  ARG_LABEL,
  ARG_RELIEF
};



static void gtk_movie_class_init     (GtkMovieClass   *klass);
static void gtk_movie_init           (GtkMovie        *movie);
static void gtk_movie_set_arg        (GtkObject        *object,
				       GtkArg           *arg,
				       guint		 arg_id);

static void gtk_movie_get_arg        (GtkObject        *object,
				       GtkArg           *arg,
				       guint		 arg_id);

static void gtk_movie_realize        (GtkWidget        *widget);
static void gtk_movie_size_request   (GtkWidget        *widget,
				       GtkRequisition   *requisition);
static void gtk_movie_size_allocate  (GtkWidget        *widget,
				       GtkAllocation    *allocation);
static void gtk_movie_paint          (GtkWidget        *widget,
				       GdkRectangle     *area);
static void gtk_movie_draw           (GtkWidget        *widget,
				       GdkRectangle     *area);
static void gtk_movie_draw_focus     (GtkWidget        *widget);
static void gtk_movie_draw_default   (GtkWidget        *widget);
static gint gtk_movie_expose         (GtkWidget        *widget,
				       GdkEventExpose   *event);
static gint gtk_movie_button_press   (GtkWidget        *widget,
				       GdkEventButton   *event);
static gint gtk_movie_button_release (GtkWidget        *widget,
				       GdkEventButton   *event);
static gint gtk_movie_enter_notify   (GtkWidget        *widget,
				       GdkEventCrossing *event);
static gint gtk_movie_leave_notify   (GtkWidget        *widget,
				       GdkEventCrossing *event);
static gint gtk_movie_focus_in       (GtkWidget        *widget,
				       GdkEventFocus    *event);
static gint gtk_movie_focus_out      (GtkWidget        *widget,
				       GdkEventFocus    *event);
static void gtk_movie_add            (GtkContainer     *container,
				       GtkWidget        *widget);
static void gtk_movie_remove         (GtkContainer     *container,
				       GtkWidget        *widget);

static void gtk_real_movie_pressed   (GtkMovie        *movie);
static void gtk_real_movie_released  (GtkMovie        *movie);
static void gtk_real_movie_enter     (GtkMovie        *movie);
static void gtk_real_movie_leave     (GtkMovie        *movie);

static GtkType gtk_movie_child_type  (GtkContainer     *container);


static GtkBinClass *parent_class = NULL;
static guint movie_signals[LAST_SIGNAL] = { 0 };


GtkType
gtk_movie_get_type (void)
{
  static GtkType movie_type = 0;

  if (!movie_type)
    {
      static const GtkTypeInfo movie_info =
      {
	"GtkMovie",
	sizeof (GtkMovie),
	sizeof (GtkMovieClass),
	(GtkClassInitFunc) gtk_movie_class_init,
	(GtkObjectInitFunc) gtk_movie_init,
        /* reserved_1 */ NULL,
	/* reserved_2 */ NULL,
	(GtkClassInitFunc) NULL,
      };

      movie_type = gtk_type_unique (GTK_TYPE_BIN, &movie_info);
      gtk_type_set_chunk_alloc (movie_type, 16);
    }

  return movie_type;
}

static void
gtk_movie_class_init (GtkMovieClass *klass)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;

  object_class = (GtkObjectClass*) klass;
  widget_class = (GtkWidgetClass*) klass;
  container_class = (GtkContainerClass*) klass;

  parent_class = gtk_type_class (GTK_TYPE_BIN);

  gtk_object_add_arg_type ("GtkMovie::label", GTK_TYPE_STRING, GTK_ARG_READWRITE, ARG_LABEL);
  gtk_object_add_arg_type ("GtkMovie::relief", GTK_TYPE_RELIEF_STYLE, GTK_ARG_READWRITE, ARG_RELIEF);

  movie_signals[PRESSED] =
    gtk_signal_new ("pressed",
                    GTK_RUN_FIRST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (GtkMovieClass, pressed),
                    gtk_marshal_NONE__NONE,
		    GTK_TYPE_NONE, 0);

  movie_signals[RELEASED] =
    gtk_signal_new ("released",
                    GTK_RUN_FIRST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (GtkMovieClass, released),
                    gtk_marshal_NONE__NONE,
		    GTK_TYPE_NONE, 0);

  movie_signals[CLICKED] =
    gtk_signal_new ("clicked",
                    GTK_RUN_FIRST | GTK_RUN_ACTION,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (GtkMovieClass, clicked),
                    gtk_marshal_NONE__NONE,
		    GTK_TYPE_NONE, 0);

  movie_signals[ENTER] =
    gtk_signal_new ("enter",
                    GTK_RUN_FIRST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (GtkMovieClass, enter),
                    gtk_marshal_NONE__NONE,
		    GTK_TYPE_NONE, 0);

  movie_signals[LEAVE] =
    gtk_signal_new ("leave",
                    GTK_RUN_FIRST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (GtkMovieClass, leave),
                    gtk_marshal_NONE__NONE,
		    GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals (object_class, movie_signals, LAST_SIGNAL);

  object_class->set_arg = gtk_movie_set_arg;
  object_class->get_arg = gtk_movie_get_arg;

  widget_class->activate_signal = movie_signals[CLICKED];
  widget_class->realize = gtk_movie_realize;
  widget_class->draw = gtk_movie_draw;
  widget_class->draw_focus = gtk_movie_draw_focus;
  widget_class->draw_default = gtk_movie_draw_default;
  widget_class->size_request = gtk_movie_size_request;
  widget_class->size_allocate = gtk_movie_size_allocate;
  widget_class->expose_event = gtk_movie_expose;
  widget_class->button_press_event = gtk_movie_button_press;
  widget_class->button_release_event = gtk_movie_button_release;
  widget_class->enter_notify_event = gtk_movie_enter_notify;
  widget_class->leave_notify_event = gtk_movie_leave_notify;
  widget_class->focus_in_event = gtk_movie_focus_in;
  widget_class->focus_out_event = gtk_movie_focus_out;

  container_class->add = gtk_movie_add;
  container_class->remove = gtk_movie_remove;
  container_class->child_type = gtk_movie_child_type;

  klass->pressed = gtk_real_movie_pressed;
  klass->released = gtk_real_movie_released;
  klass->clicked = NULL;
  klass->enter = gtk_real_movie_enter;
  klass->leave = gtk_real_movie_leave;
}

static void
gtk_movie_init (GtkMovie *movie)
{
  GTK_WIDGET_SET_FLAGS (movie, GTK_CAN_FOCUS | GTK_RECEIVES_DEFAULT);
  GTK_WIDGET_UNSET_FLAGS (movie, GTK_NO_WINDOW);

  movie->child = NULL;
  movie->in_button = FALSE;
  movie->button_down = FALSE;
  movie->relief = GTK_RELIEF_NORMAL;
}

static GtkType
gtk_movie_child_type  (GtkContainer     *container)
{
  if (!GTK_BIN (container)->child)
    return GTK_TYPE_WIDGET;
  else
    return GTK_TYPE_NONE;
}

static void
gtk_movie_set_arg (GtkObject *object,
		    GtkArg    *arg,
		    guint      arg_id)
{
  GtkMovie *movie;

  movie = GTK_MOVIE (object);

  switch (arg_id)
    {
      GtkWidget *child;

    case ARG_LABEL:
      child = GTK_BIN (movie)->child;
      if (!child)
	child = gtk_widget_new (GTK_TYPE_LABEL,
				"visible", TRUE,
				"parent", movie,
				NULL);
      if (GTK_IS_LABEL (child))
	gtk_label_set_text (GTK_LABEL (child),
		       GTK_VALUE_STRING (*arg) ? GTK_VALUE_STRING (*arg) : "");
      break;
    case ARG_RELIEF:
      gtk_movie_set_relief (movie, GTK_VALUE_ENUM (*arg));
      break;
    default:
      break;
    }
}

static void
gtk_movie_get_arg (GtkObject *object,
		    GtkArg    *arg,
		    guint      arg_id)
{
  GtkMovie *movie;

  movie = GTK_MOVIE (object);

  switch (arg_id)
    {
    case ARG_LABEL:
      if (GTK_BIN (movie)->child && GTK_IS_LABEL (GTK_BIN (movie)->child))
	GTK_VALUE_STRING (*arg) = g_strdup (GTK_LABEL (GTK_BIN (movie)->child)->label);
      else
	GTK_VALUE_STRING (*arg) = NULL;
      break;
    case ARG_RELIEF:
      GTK_VALUE_ENUM (*arg) = gtk_movie_get_relief (movie);
      break;
    default:
      arg->type = GTK_TYPE_INVALID;
      break;
    }
}

GtkWidget*
gtk_movie_new (void)
{
  return GTK_WIDGET (gtk_type_new (gtk_movie_get_type ()));
}

GtkWidget*
gtk_movie_new_with_label (const gchar *label)
{
  GtkWidget *movie;
  GtkWidget *label_widget;

  movie = gtk_movie_new ();
  label_widget = gtk_label_new (label);
  gtk_misc_set_alignment (GTK_MISC (label_widget), 0.5, 0.5);

  gtk_container_add (GTK_CONTAINER (movie), label_widget);
  gtk_widget_show (label_widget);

  return movie;
}

void
gtk_movie_pressed (GtkMovie *movie)
{
  g_return_if_fail (movie != NULL);
  g_return_if_fail (GTK_IS_MOVIE (movie));

  gtk_signal_emit (GTK_OBJECT (movie), movie_signals[PRESSED]);
}

void
gtk_movie_released (GtkMovie *movie)
{
  g_return_if_fail (movie != NULL);
  g_return_if_fail (GTK_IS_MOVIE (movie));

  gtk_signal_emit (GTK_OBJECT (movie), movie_signals[RELEASED]);
}

void
gtk_movie_clicked (GtkMovie *movie)
{
  g_return_if_fail (movie != NULL);
  g_return_if_fail (GTK_IS_MOVIE (movie));

  gtk_signal_emit (GTK_OBJECT (movie), movie_signals[CLICKED]);
}

void
gtk_movie_enter (GtkMovie *movie)
{
  g_return_if_fail (movie != NULL);
  g_return_if_fail (GTK_IS_MOVIE (movie));

  gtk_signal_emit (GTK_OBJECT (movie), movie_signals[ENTER]);
}

void
gtk_movie_leave (GtkMovie *movie)
{
  g_return_if_fail (movie != NULL);
  g_return_if_fail (GTK_IS_MOVIE (movie));

  gtk_signal_emit (GTK_OBJECT (movie), movie_signals[LEAVE]);
}

void
gtk_movie_set_relief (GtkMovie *movie,
		       GtkReliefStyle newrelief)
{
  g_return_if_fail (movie != NULL);
  g_return_if_fail (GTK_IS_MOVIE (movie));

  movie->relief = newrelief;
  gtk_widget_queue_draw (GTK_WIDGET (movie));
}

GtkReliefStyle
gtk_movie_get_relief (GtkMovie *movie)
{
  g_return_val_if_fail (movie != NULL, GTK_RELIEF_NORMAL);
  g_return_val_if_fail (GTK_IS_MOVIE (movie), GTK_RELIEF_NORMAL);

  return movie->relief;
}

static void
gtk_movie_realize (GtkWidget *widget)
{
  GtkMovie *movie;
  GdkWindowAttr attributes;
  gint attributes_mask;
  gint border_width;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_MOVIE (widget));

  movie = GTK_MOVIE (widget);
  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);

  border_width = GTK_CONTAINER (widget)->border_width;

  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.x = widget->allocation.x + border_width;
  attributes.y = widget->allocation.y + border_width;
  attributes.width = widget->allocation.width - border_width * 2;
  attributes.height = widget->allocation.height - border_width * 2;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);
  attributes.event_mask = gtk_widget_get_events (widget);
  attributes.event_mask |= (GDK_EXPOSURE_MASK |
			    GDK_BUTTON_PRESS_MASK |
			    GDK_BUTTON_RELEASE_MASK |
			    GDK_ENTER_NOTIFY_MASK |
			    GDK_LEAVE_NOTIFY_MASK);

  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

  widget->window = gdk_window_new (gtk_widget_get_parent_window (widget), &attributes, attributes_mask);
  gdk_window_set_user_data (widget->window, movie);

  widget->style = gtk_style_attach (widget->style, widget->window);
  gtk_style_set_background (widget->style, widget->window, GTK_STATE_NORMAL);
}

static void
gtk_movie_size_request (GtkWidget      *widget,
			 GtkRequisition *requisition)
{
  GtkMovie *movie;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_MOVIE (widget));
  g_return_if_fail (requisition != NULL);

  movie = GTK_MOVIE (widget);

  requisition->width = (GTK_CONTAINER (widget)->border_width + CHILD_SPACING +
			GTK_WIDGET (widget)->style->klass->xthickness) * 2;
  requisition->height = (GTK_CONTAINER (widget)->border_width + CHILD_SPACING +
			 GTK_WIDGET (widget)->style->klass->ythickness) * 2;

  if (GTK_WIDGET_CAN_DEFAULT (widget))
    {
      requisition->width += (GTK_WIDGET (widget)->style->klass->xthickness * 2 +
			     DEFAULT_SPACING);
      requisition->height += (GTK_WIDGET (widget)->style->klass->ythickness * 2 +
			      DEFAULT_SPACING);
    }

  if (GTK_BIN (movie)->child && GTK_WIDGET_VISIBLE (GTK_BIN (movie)->child))
    {
      GtkRequisition child_requisition;

      gtk_widget_size_request (GTK_BIN (movie)->child, &child_requisition);

      requisition->width += child_requisition.width;
      requisition->height += child_requisition.height;
    }
}

static void
gtk_movie_size_allocate (GtkWidget     *widget,
			  GtkAllocation *allocation)
{
  GtkMovie *movie;
  GtkAllocation child_allocation;
  gint border_width;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_MOVIE (widget));
  g_return_if_fail (allocation != NULL);

  widget->allocation = *allocation;
  border_width = GTK_CONTAINER (widget)->border_width;

  if (GTK_WIDGET_REALIZED (widget))
    gdk_window_move_resize (widget->window,
			    widget->allocation.x + border_width,
			    widget->allocation.y + border_width,
			    widget->allocation.width - border_width * 2,
			    widget->allocation.height - border_width * 2);

  movie = GTK_MOVIE (widget);

  if (GTK_BIN (movie)->child && GTK_WIDGET_VISIBLE (GTK_BIN (movie)->child))
    {
      child_allocation.x = (CHILD_SPACING + GTK_WIDGET (widget)->style->klass->xthickness);
      child_allocation.y = (CHILD_SPACING + GTK_WIDGET (widget)->style->klass->ythickness);

      child_allocation.width = MAX (1, (gint)widget->allocation.width - child_allocation.x * 2 -
	                         border_width * 2);
      child_allocation.height = MAX (1, (gint)widget->allocation.height - child_allocation.y * 2 -
	                          border_width * 2);

      if (GTK_WIDGET_CAN_DEFAULT (movie))
	{
	  child_allocation.x += (GTK_WIDGET (widget)->style->klass->xthickness +
				 DEFAULT_LEFT_POS);
	  child_allocation.y += (GTK_WIDGET (widget)->style->klass->ythickness +
				 DEFAULT_TOP_POS);
	  child_allocation.width =  MAX (1, (gint)child_allocation.width -
					 (gint)(GTK_WIDGET (widget)->style->klass->xthickness * 2 + DEFAULT_SPACING));
	  child_allocation.height = MAX (1, (gint)child_allocation.height -
					 (gint)(GTK_WIDGET (widget)->style->klass->xthickness * 2 + DEFAULT_SPACING));
	}

      gtk_widget_size_allocate (GTK_BIN (movie)->child, &child_allocation);
    }
}

/*
 * +------------------------------------------------+
 * |                   BORDER                       |
 * |  +------------------------------------------+  |
 * |  |\\\\\\\\\\\\\\\\DEFAULT\\\\\\\\\\\\\\\\\  |  |
 * |  |\\+------------------------------------+  |  |
 * |  |\\| |           SPACING       3      | |  |  |
 * |  |\\| +--------------------------------+ |  |  |
 * |  |\\| |########## FOCUS ###############| |  |  |
 * |  |\\| |#+----------------------------+#| |  |  |
 * |  |\\| |#|         RELIEF            \|#| |  |  |
 * |  |\\| |#|  +-----------------------+\|#| |  |  |
 * |  |\\|1|#|  +     THE TEXT          +\|#|2|  |  |
 * |  |\\| |#|  +-----------------------+\|#| |  |  |
 * |  |\\| |#| \\\\\ ythickness \\\\\\\\\\|#| |  |  |
 * |  |\\| |#+----------------------------+#| |  |  |
 * |  |\\| |########### 1 ##################| |  |  |
 * |  |\\| +--------------------------------+ |  |  |
 * |  |\\| |        default spacing   4     | |  |  |
 * |  |\\+------------------------------------+  |  |
 * |  |\            ythickness                   |  |
 * |  +------------------------------------------+  |
 * |                border_width                    |
 * +------------------------------------------------+
 */

static void
gtk_movie_paint (GtkWidget    *widget,
		  GdkRectangle *area)
{
  GtkMovie *movie;
  GtkShadowType shadow_type;
  gint width, height;
  gint x, y;
   
  return ;

  /*
  if (GTK_WIDGET_DRAWABLE (widget))
    {
      movie = GTK_MOVIE (widget);
	
      x = 0;
      y = 0;
      width = widget->allocation.width - GTK_CONTAINER (widget)->border_width * 2;
      height = widget->allocation.height - GTK_CONTAINER (widget)->border_width * 2;

      gdk_window_set_back_pixmap (widget->window, NULL, TRUE);
      gdk_window_clear_area (widget->window, area->x, area->y, area->width, area->height);

      if (GTK_WIDGET_HAS_DEFAULT (widget) &&
	  GTK_MOVIE (widget)->relief == GTK_RELIEF_NORMAL)
	{
	  gtk_paint_box (widget->style, widget->window,
			 GTK_STATE_NORMAL, GTK_SHADOW_IN,
			 area, widget, "moviedefault",
			 x, y, width, height);
	}

      if (GTK_WIDGET_CAN_DEFAULT (widget))
	{
	  x += widget->style->klass->xthickness;
	  y += widget->style->klass->ythickness;
	  width -= 2 * x + DEFAULT_SPACING;
	  height -= 2 * y + DEFAULT_SPACING;
	  x += DEFAULT_LEFT_POS;
	  y += DEFAULT_TOP_POS;
	}
       
      if (GTK_WIDGET_HAS_FOCUS (widget))
	{
	  x += 1;
	  y += 1;
	  width -= 2;
	  height -= 2;
	}
	
	
      if (GTK_WIDGET_STATE (widget) == GTK_STATE_ACTIVE)
	shadow_type = GTK_SHADOW_IN;
      else
	shadow_type = GTK_SHADOW_OUT;
	

	
      if ((movie->relief != GTK_RELIEF_NONE) ||
	  ((GTK_WIDGET_STATE(widget) != GTK_STATE_NORMAL) &&
	   (GTK_WIDGET_STATE(widget) != GTK_STATE_INSENSITIVE)))
	gtk_paint_box (widget->style, widget->window,
		       GTK_WIDGET_STATE (widget),
		       shadow_type, area, widget, "movie",
		       x, y, width, height);
	 
    
      if (GTK_WIDGET_HAS_FOCUS (widget))
	{
	  x -= 1;
	  y -= 1;
	  width += 2;
	  height += 2;
	     
	  gtk_paint_focus (widget->style, widget->window,
			   area, widget, "movie",
			   x, y, width - 1, height - 1);
	} 
    }
	*/

}


static void
gtk_movie_draw (GtkWidget    *widget,
		 GdkRectangle *area)
{
  GtkMovie *movie;
  GdkRectangle child_area;
  GdkRectangle tmp_area;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_MOVIE (widget));
  g_return_if_fail (area != NULL);

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      movie = GTK_MOVIE (widget);

      tmp_area = *area;
      tmp_area.x -= GTK_CONTAINER (movie)->border_width;
      tmp_area.y -= GTK_CONTAINER (movie)->border_width;

      gtk_movie_paint (widget, &tmp_area);

      if (GTK_BIN (movie)->child && gtk_widget_intersect (GTK_BIN (movie)->child, &tmp_area, &child_area))
	gtk_widget_draw (GTK_BIN (movie)->child, &child_area);
    }
}

static void
gtk_movie_draw_focus (GtkWidget *widget)
{
  gtk_widget_draw (widget, NULL);
}

static void
gtk_movie_draw_default (GtkWidget *widget)
{
  gtk_widget_draw (widget, NULL);
}

static gint
gtk_movie_expose (GtkWidget      *widget,
		   GdkEventExpose *event)
{
  GtkBin *bin;
  GdkEventExpose child_event;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_MOVIE (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      bin = GTK_BIN (widget);
      
      gtk_movie_paint (widget, &event->area);

      child_event = *event;
      if (bin->child && GTK_WIDGET_NO_WINDOW (bin->child) &&
	  gtk_widget_intersect (bin->child, &event->area, &child_event.area))
	gtk_widget_event (bin->child, (GdkEvent*) &child_event);
    }

  return FALSE;
}

static gint
gtk_movie_button_press (GtkWidget      *widget,
			 GdkEventButton *event)
{
  GtkMovie *movie;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_MOVIE (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  if (event->type == GDK_BUTTON_PRESS)
    {
      movie = GTK_MOVIE (widget);

      if (!GTK_WIDGET_HAS_FOCUS (widget))
	gtk_widget_grab_focus (widget);

      if (event->button == 1)
	{
	  gtk_grab_add (GTK_WIDGET (movie));
	  gtk_movie_pressed (movie);
	}
    }

  return TRUE;
}

static gint
gtk_movie_button_release (GtkWidget      *widget,
			   GdkEventButton *event)
{
  GtkMovie *movie;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_MOVIE (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  if (event->button == 1)
    {
      movie = GTK_MOVIE (widget);
      gtk_grab_remove (GTK_WIDGET (movie));
      gtk_movie_released (movie);
    }

  return TRUE;
}

static gint
gtk_movie_enter_notify (GtkWidget        *widget,
			 GdkEventCrossing *event)
{
  GtkMovie *movie;
  GtkWidget *event_widget;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_MOVIE (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  movie = GTK_MOVIE (widget);
  event_widget = gtk_get_event_widget ((GdkEvent*) event);

  if ((event_widget == widget) &&
      (event->detail != GDK_NOTIFY_INFERIOR))
    {
      movie->in_button = TRUE;
      gtk_movie_enter (movie);
    }

  return FALSE;
}

static gint
gtk_movie_leave_notify (GtkWidget        *widget,
			 GdkEventCrossing *event)
{
  GtkMovie *movie;
  GtkWidget *event_widget;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_MOVIE (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  movie = GTK_MOVIE (widget);
  event_widget = gtk_get_event_widget ((GdkEvent*) event);

  if ((event_widget == widget) &&
      (event->detail != GDK_NOTIFY_INFERIOR))
    {
      movie->in_button = FALSE;
      gtk_movie_leave (movie);
    }

  return FALSE;
}

static gint
gtk_movie_focus_in (GtkWidget     *widget,
		     GdkEventFocus *event)
{
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_MOVIE (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  GTK_WIDGET_SET_FLAGS (widget, GTK_HAS_FOCUS);
  gtk_widget_draw_focus (widget);

  return FALSE;
}

static gint
gtk_movie_focus_out (GtkWidget     *widget,
		      GdkEventFocus *event)
{
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_MOVIE (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  GTK_WIDGET_UNSET_FLAGS (widget, GTK_HAS_FOCUS);
  gtk_widget_draw_focus (widget);

  return FALSE;
}

static void
gtk_movie_add (GtkContainer *container,
		GtkWidget    *widget)
{
  g_return_if_fail (container != NULL);
  g_return_if_fail (widget != NULL);

  if (GTK_CONTAINER_CLASS (parent_class)->add)
    GTK_CONTAINER_CLASS (parent_class)->add (container, widget);

  GTK_MOVIE (container)->child = GTK_BIN (container)->child;
}

static void
gtk_movie_remove (GtkContainer *container,
		   GtkWidget    *widget)
{
  g_return_if_fail (container != NULL);
  g_return_if_fail (widget != NULL);

  if (GTK_CONTAINER_CLASS (parent_class)->remove)
    GTK_CONTAINER_CLASS (parent_class)->remove (container, widget);

  GTK_MOVIE (container)->child = GTK_BIN (container)->child;
}

static void
gtk_real_movie_pressed (GtkMovie *movie)
{
  GtkStateType new_state;

  g_return_if_fail (movie != NULL);
  g_return_if_fail (GTK_IS_MOVIE (movie));

  movie->button_down = TRUE;

  new_state = (movie->in_button ? GTK_STATE_ACTIVE : GTK_STATE_NORMAL);

  if (GTK_WIDGET_STATE (movie) != new_state)
    {
      gtk_widget_set_state (GTK_WIDGET (movie), new_state);
      gtk_widget_queue_draw (GTK_WIDGET (movie));
    }
}

static void
gtk_real_movie_released (GtkMovie *movie)
{
  GtkStateType new_state;

  g_return_if_fail (movie != NULL);
  g_return_if_fail (GTK_IS_MOVIE (movie));

  if (movie->button_down)
    {
      movie->button_down = FALSE;

      if (movie->in_button)
	gtk_movie_clicked (movie);

      new_state = (movie->in_button ? GTK_STATE_PRELIGHT : GTK_STATE_NORMAL);

      if (GTK_WIDGET_STATE (movie) != new_state)
	{
	  gtk_widget_set_state (GTK_WIDGET (movie), new_state);
	  /* We _draw () instead of queue_draw so that if the operation
	   * blocks, the label doesn't vanish.
	   */
	  gtk_widget_draw (GTK_WIDGET (movie), NULL);
	}
    }
}

static void
gtk_real_movie_enter (GtkMovie *movie)
{
  GtkStateType new_state;

  g_return_if_fail (movie != NULL);
  g_return_if_fail (GTK_IS_MOVIE (movie));

  new_state = (movie->button_down ? GTK_STATE_ACTIVE : GTK_STATE_PRELIGHT);

  if (GTK_WIDGET_STATE (movie) != new_state)
    {
      gtk_widget_set_state (GTK_WIDGET (movie), new_state);
      gtk_widget_queue_draw (GTK_WIDGET (movie));
    }
}

static void
gtk_real_movie_leave (GtkMovie *movie)
{
  g_return_if_fail (movie != NULL);
  g_return_if_fail (GTK_IS_MOVIE (movie));

  if (GTK_WIDGET_STATE (movie) != GTK_STATE_NORMAL)
    {
      gtk_widget_set_state (GTK_WIDGET (movie), GTK_STATE_NORMAL);
      gtk_widget_queue_draw (GTK_WIDGET (movie));
    }
}
