 /*************************************************************************/
 /*                                                                       */
 /*                Centre for Speech Technology Research                  */
 /*                     University of Edinburgh, UK                       */
 /*                      Copyright (c) 1995,1996                          */
 /*                        All Rights Reserved.                           */
 /*  Permission to use, copy, modify, distribute this software and its    */
 /*  documentation for research, educational and individual use only, is  */
 /*  hereby granted without fee, subject to the following conditions:     */
 /*   1. The code must retain the above copyright notice, this list of    */
 /*      conditions and the following disclaimer.                         */
 /*   2. Any modifications must be clearly marked as such.                */
 /*   3. Original authors' names are not deleted.                         */
 /*  This software may not be used for commercial purposes without        */
 /*  specific prior written permission from the authors.                  */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
 /*  THIS SOFTWARE.                                                       */
 /*                                                                       */
 /*************************************************************************/
 /*                                                                       */
 /*                 Author: Richard Caley (rjc@cstr.ed.ac.uk)             */
 /*                   Date: Mon Jul 21 1997                               */
 /* --------------------------------------------------------------------  */
 /* Untyped list used as the basis of the TList class                     */
 /*                                                                       */
 /*************************************************************************/

#include <EST_UList.h>


void EST_UList::clear_and_free(void (*item_free)(EST_UItem *p))
{
    EST_UItem *p, *np;

    for (p=head(); p != 0; p = np)
      {
	np=next(p);
	if (item_free)
	  item_free(p);
	else
	  delete p;
      }
    h = p = NULL;
}

int EST_UList::length() const
{
    EST_UItem *ptr;
    int n = 0;

    ptr = head();
    ptr = next(ptr);
    
    for (ptr = head(); ptr != 0; ptr = next(ptr))
	++n;
    return n;
}

int EST_UList::index(EST_UItem *item) const
{
    EST_UItem *ptr;
    int n = 0;
    
    for (ptr = head(); ptr != 0; ptr = next(ptr), ++n)
	if (item == ptr)
	    return n;

    return -1;
}

EST_UItem *EST_UList::nth_pointer(int n) const
{
    EST_UItem *ptr;
    int i;
    
    for (i = 0, ptr = head(); ptr != 0; ptr = next(ptr), ++i)
      if (i == n)
	return ptr;

    cerr << "Requested item #" << n << " off end of list" << endl;
    return head();
}

EST_UItem * EST_UList::remove(EST_UItem *item,
			      void (*free_item)(EST_UItem *item))
{

    EST_UItem *prev_item, *next_item;

    if(item == 0)
	return 0;

    prev_item = item->p;
    next_item = item->n;
    
    if ( (next_item == 0) && (prev_item == 0)){ // removing only item in list
	h = 0;
	t = 0;

    }else if (prev_item == 0){ // removing head of list
	h = next_item;
	next_item->p = 0;
	// return 0 when removing head of list
	// calling fn must test for this !
	prev_item = 0; 
	
    }else if (next_item == 0){ // removing tail of list
	t = prev_item;
	prev_item->n = 0;
	
    }else{ // removing from middle of list
	next_item->p = prev_item;
	prev_item->n = next_item;
    }

    if (free_item)
      free_item(item);
    else
      delete item;

    return prev_item;
}

EST_UItem * EST_UList::remove(int n,
			      void (*item_free)(EST_UItem *item))
{
    int i;
    EST_UItem *p;

    for (i = 0, p = head(); p != 0; p = next(p), ++i)
	 if (i == n)
	     return remove(p, item_free);
    
    cerr << "Couldn't remove item " << n << "in list\n";
    return head();
}


// This should check if the incoming prev_item actually is in the list

EST_UItem *EST_UList::insert_after(EST_UItem *prev_item, EST_UItem *new_item)
{

    // fixed by simonk, 28.11.96

    if(prev_item == t){

	t = new_item;
	new_item->n = 0;
	new_item->p = prev_item;
	prev_item->n = new_item;

    }else{

	EST_UItem *next_item;
	next_item = prev_item->n;
    
	prev_item->n = new_item;
	new_item->p = prev_item;
	new_item->n = next_item;
	next_item->p = new_item;

    }

    return (EST_UItem*)new_item;
}

EST_UItem *EST_UList::insert_before(EST_UItem *next_item, EST_UItem *new_item)
{

    // fixed by simonk, 28.11.96

    if(next_item == 0)
	next_item=h; // well what else can you do ?

    if (next_item == h){
	h = new_item;
	new_item->p = 0;
	new_item->n = next_item;
	next_item->p = new_item;

    }else{

	EST_UItem *prev_item;
	prev_item = next_item->p;

	prev_item->n = new_item;
	new_item->p = prev_item;
	new_item->n = next_item;
	next_item->p = new_item;

    }

    return next_item;
}

 void EST_UList::exchange(EST_UItem *a,EST_UItem *b)
{

    // this seems WAY too longwinded

    if(a==b)
	return;

    if((a==NULL) || (b==NULL)){
	cerr << "Can't exchange NULL items !" << endl;
	return;
    }

    EST_UItem *a_prev_item, *a_next_item;
    EST_UItem *b_prev_item, *b_next_item;
    

    if(a->n == b){

	a_prev_item = a->p;
	b_next_item = b->n;

	b->p = a_prev_item;
	b->n = a;
	a->p = b;
	a->n = b_next_item;

	if(a_prev_item == NULL)
	    h = b;
	else
	    a_prev_item->n = b;
	
	if(b_next_item == NULL)
	    t = a;
	else
	    b_next_item->p = a;

	return;
    }

    if(b->n == a)
	exchange(b,a);

    a_prev_item = a->p;
    a_next_item = a->n;
    b_prev_item = b->p;
    b_next_item = b->n;

    a->p = b_prev_item;
    a->n = b_next_item;
    b->p = a_prev_item;
    b->n = a_next_item;


    if(a_prev_item == NULL)
	h = b;
    else
	a_prev_item->n = b;
    
    if(b_prev_item == NULL)
	h = a;
    else
	b_prev_item->n = a;
    
    if(a_next_item == NULL)
	t = b;
    else
	a_next_item->p = b;

    if(b_next_item == NULL)
	t = a;
    else
	b_next_item->p = a;

}

void EST_UList::exchange(int i, int j)
{

    EST_UItem *p;
    EST_UItem *a=NULL,*b=NULL;
    int k;

    for(k=0,p = head(); p != 0; p = next(p),k++){
	if(i==k)
	    a = p;
	if(j==k)
	    b = p;
    }

    if((a==NULL) || (b==NULL)){
	cerr << "Can't exchange items " << i << " and " << j 
	     << " (off end of list)" << endl;
    }

    exchange(a,b);
}



void EST_UList::reverse()
{

    // not tested yet ...

    // easy : swap all previous/next pointers, swap head/tail pointers
    EST_UItem *p,*q,*r;
    p=h;
    h=t;
    t=p;
    while(p!=NULL){

	q=p;
	p=next(p);

	r    = q->p;
	q->p = q->n;
	q->n = r;

    }

}


void EST_UList::append(EST_UItem *new_item)
{

    if (h == 0)
    {
	h = new_item;
	new_item->p = 0;
    }
    else
    {
	t->n = new_item;
	new_item->p = t;
	new_item->n = 0;
    }
    t = new_item;
}

bool EST_UList::operator_eq(const EST_UList &a, 
		 const EST_UList &b, 
		 bool (*eq)(const EST_UItem *item1, const EST_UItem *item2))
{
  EST_UItem *p,*q;
  q=b.head();
  for (p = a.head(); p != NULL; p = next(p)){
    if(q == NULL)
      return false;
    if(eq(q, p))
      q=next(q);
    else
      return false;
  }

  if(q == NULL)
    return true;
  else
    return false;
}

void EST_UList::sort(EST_UList &l,
		      bool (*gt)(const EST_UItem *item1, const EST_UItem *item2))
{

    // just bubble sort for now
    // use EST_String::operator > for comparisons

    EST_UItem *l_ptr,*m_ptr;
    bool sorted=false;

    while(!sorted){
	sorted=true;

	for(l_ptr=l.head(); l_ptr != 0; l_ptr=next(l_ptr)){
	    
	    m_ptr=next(l_ptr);
	    if(m_ptr != 0)
		if(gt(l_ptr, m_ptr)){
		    l.exchange(l_ptr,m_ptr);
		    sorted=false;
		}
	}
    }
    
}

// quicksort from 'Algorithms'
// by Cormen, Leiserson & Rivest

static EST_UItem *partition(EST_UItem *p, EST_UItem *r,
			    bool (*gt)(const EST_UItem *item1, const EST_UItem *item2),
			    void (*exchange)(EST_UItem *item1, EST_UItem *item2))
{
    // this can be tidied up / speeded up

    EST_UItem *i,*j,*i2,*j2;
    EST_UItem *x = p;

    i = p;
    j = r;

    while(true){

	while(gt(j, x) )
	    j = prev(j);
	
	while(gt(x, i))
	    i = next(i);

	if((i != j) && (prev(i) != j)){
	    i2=i;
	    j2=j;
	    i=next(i);
	    j=prev(j);
	    exchange(i2,j2);

	}else
	    return j;

    }
    return NULL;
}


static void qsort_sub(EST_UList &l, EST_UItem *p, EST_UItem *r,
		      bool (*gt)(const EST_UItem *item1, const EST_UItem *item2),
		      void (*exchange)(EST_UItem *item1, EST_UItem *item2))
{
    EST_UItem *q;
    if(p != r){
	q = partition(p,r, gt, exchange);
	qsort_sub(l,p,q, gt, exchange);
	qsort_sub(l,next(q),r, gt, exchange);
    }
}

void EST_UList::qsort(EST_UList &l,
		      bool (*gt)(const EST_UItem *item1, const EST_UItem *item2),
		      void (*exchange)(EST_UItem *item1, EST_UItem *item2))
{
    qsort_sub(l,l.head(),l.tail(), gt, exchange);
}


void EST_UList::sort_unique(EST_UList &l,
			    bool (*eq)(const EST_UItem *item1, const EST_UItem *item2),
			    bool (*gt)(const EST_UItem *item1, const EST_UItem *item2),
			    void (*item_free)(EST_UItem *item))
{
    // as sort(..) but delete any repeated items

    EST_UItem *l_ptr,*m_ptr;
    bool sorted=false;

    while(!sorted){
	sorted=true;

	for(l_ptr=l.head(); l_ptr != 0; l_ptr=next(l_ptr)){
	    
	    m_ptr=next(l_ptr);
	    if(m_ptr != 0)
	      if(gt(l_ptr, m_ptr)){
		    l.exchange(l_ptr,m_ptr);
		    sorted=false;
		} else if(eq(l_ptr,  m_ptr)){
		  l.remove(m_ptr, item_free);
		  sorted=false;
		}
	}
    }
}

void EST_UList::merge_sort_unique(EST_UList &l, EST_UList &m,
		bool (*eq)(const EST_UItem *item1, const EST_UItem *item2),
		bool (*gt)(const EST_UItem *item1, const EST_UItem *item2),
		void (*item_free)(EST_UItem *item))
{
    // keep all unique items in l, and add any new items from m to l

    EST_UItem *l_ptr,*m_ptr;
    bool flag;

    // make sure
    sort_unique(l, eq, gt, item_free);

    for(m_ptr=m.head(); m_ptr != 0; m_ptr=next(m_ptr)){

	// try and put item from m in list 
	flag=false;
	for(l_ptr=l.head(); l_ptr != 0; l_ptr=next(l_ptr)){
	  if( gt(l_ptr, m_ptr) ){
		l.insert_before(l_ptr, m_ptr);
		flag=true;
		break;
	    }else if( eq(m_ptr, l_ptr) ){
		flag=true;
		break;
	    }
	}
	// or try and append it
	if(!flag && ( gt(m_ptr, l.tail()) ) )
	    l.append(m_ptr);
    }
}
