/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                Author :  Alan W Black                                 */
/*                Date   :  July 1997                                    */
/*-----------------------------------------------------------------------*/
/*  Optional Sun 16bit linear support for /dev/audio                     */
/*  This only works when compiled under Solaris or SunOS as it requires  */
/*  Sun's headers, and much more importantly Sun's /dev/audio.  This     */
/*  of course will access the *local* machine's /dev/audio definite not  */
/*  the "network is the computer" maxim but sometimes you might want     */
/*  this                                                                 */
/*                                                                       */
/*=======================================================================*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/stat.h>
#include <sys/types.h>
#include "EST_cutils.h"
#include "EST_Wave.h"
#include "EST_Option.h"
#include "audioP.h"
#include "EST_io_aux.h"
#include "EST_unix.h"

#ifdef SUPPORT_SUN16
#if defined(__svr4__) || defined(SYSV) || defined(SVR4)
/* Solaris */
#include <sys/audioio.h>
#else
/* Sunos */
#include <sun/audioio.h>
#include <sys/ioccom.h>
#endif


static int sun16_check_device(int audio);
static int sun16_set_info(int audio, int sample_rate);

int sun16_supported = TRUE;

#define AUDIOBUFFSIZE 256

int play_sun16_wave(EST_Wave &inwave, EST_Option &al)
{
    /* supported sampling frequencies for Sun dbri device */
    int dev_sr[] = {8000, 9600, 11025, 16000, 18900, 22050, 32000, 
		    37800, 44100, 48000, -1};
    int sample_rate;
    short *waveform;
    FILE *fdaudio;
    int num_samples;
    int audio;
    int i,r,n;
    (void)al;

    sample_rate = inwave.sample_rate();
    int samp_rate_ok = FALSE;
    for (i=0; dev_sr[i] != -1; i++)
	if (sample_rate == dev_sr[i])
	    samp_rate_ok = TRUE;
    if (samp_rate_ok == FALSE)
    {
	if (sample_rate == 10000)
	    inwave.resample(9600);  // just sounds much better than 16000
	else 
	    inwave.resample(16000);
    }

    if ((fdaudio = fopen("/dev/audio","wb")) == NULL)
    {
	cerr << "SUN16: can't open /dev/audio" << endl;
	return -1;
    }
    // As I can't find open in an Sun CC include file I'll avoid it
    audio = fileno(fdaudio);

    waveform = inwave.data();
    num_samples = inwave.num_samples();
    sample_rate = inwave.sample_rate();

    if (sun16_check_device(audio) == FALSE)
    {
	cerr << "SUN16: device doesn't support 16bit linear." << endl;
	fclose(fdaudio);
	return -1;
    }

    if (sun16_set_info(audio,sample_rate) == FALSE)
    {
	cerr << "SUN16: unable to set sample rate " <<
	    sample_rate << endl;
	fclose(fdaudio);
	return -1;
    }

    for (i=0; i < num_samples; i += r/2)
    {
	if (num_samples > i+AUDIOBUFFSIZE)
	    n = AUDIOBUFFSIZE;
	else
	    n = num_samples-i;
	r = write(audio,&waveform[i], n*2);
	if (r == 0)
	{
	    cerr << "SUN16: failed to write to buffer" <<
	    sample_rate << endl;
	    fclose(fdaudio);
	    return -1;
	}
    }

    fclose(fdaudio);
    return 1;
}

static int sun16_check_device(int audio)
{
#ifdef __svr4__
/* Solaris */
    audio_device_t    type;
    
    ioctl(audio, AUDIO_DRAIN, 0);	       /* drain everything out */

    if ((ioctl(audio, AUDIO_GETDEV, &type) != -1) &&
	((streq("SUNW,CS4231",type.name)) ||   /* Newer Suns (ultras)      */
	 (streq("SUNW,dbri",type.name)) ||     /* Older Suns (SS10s SS20s) */
	 (streq("SUNW,sb16",type.name))))      /* i386 machines            */
	return TRUE;
    else
	return FALSE;
#else
/* SunOS */
    int type;

    ioctl(audio, AUDIO_DRAIN, 0);	       /* drain everything out */

    if ((ioctl(audio, AUDIO_GETDEV, &type) != -1) &&
	((type == AUDIO_DEV_SPEAKERBOX) || (type == AUDIO_DEV_CODEC)))
	return TRUE;
    else
	return FALSE;
#endif

}
	
static int sun16_set_info(int audio, int sample_rate)
{
    audio_info_t    info;
    
    ioctl(audio, AUDIO_GETINFO, &info);

    info.play.sample_rate = sample_rate;
    info.play.encoding = AUDIO_ENCODING_LINEAR;
    info.play.precision = 16;
    info.play.channels = 1;

    ioctl(audio, AUDIO_SETINFO, &info);

    return 1;
}

#else
int sun16_supported = FALSE;

int play_sun16_wave(EST_Wave &inwave, EST_Option &al)
{
    (void)inwave;
    (void)al;
    cerr << "Sun 16bit linear not supported" << endl;
    return -1;
}

#endif
