/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1996,1997                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                Author :  Alan W Black and Paul Taylor                 */
/*                Date   :  April 1996                                   */
/*-----------------------------------------------------------------------*/
/*                                                                       */
/* Top level file for synthesizer                                        */
/*                                                                       */
/*=======================================================================*/
#include <stdio.h>
#include "EST_unix.h"
#include <stdlib.h>
#include "festival.h"
#include "festivalP.h"
#include "ModuleDescription.h"
#include "DummyObject.h"
#include "Unit.h"
#include "UnitDatabase.h"
#include "UnitIndex.h"
#include "Join.h"
#include "SchemeModuleDescription.h"
#include "siod.h"

static void festival_lisp_funcs(void);
static void festival_lisp_vars(void);
static void festival_types(void);
static void festival_banner(void);
static void festival_load_default_files(void);

int ft_endian_loc = 1;

char *festival_version = "1.2.1 September 1997";

// Allow the path to be passed in without quotes because Windoze command line
// is stupid
// Extra level of indirection needed to get an extra macro expansion. Yeuch.

#define _S_S_S(S) #S
#define STRINGIZE(S) _S_S_S(S)
#ifdef FTLIBDIRC
#    define FTLIBDIR STRINGIZE(FTLIBDIRC)
#endif


#ifndef FTLIBDIR
#define FTLIBDIR "/projects/festival/lib/"
#endif
#ifndef FTOSTYPE
#define FTOSTYPE ""
#endif

char *festival_libdir = FTLIBDIR;
ostream *cdebug;
static int festival_server_port = 1314;
static EST_StrList sub_copyrights;

#if 0
LISP describe_module(LISP lname, LISP lstream);
LISP describe_all_modules(void);
#endif

void festival_initialize(int load_init_files,int heap_size)
{
    // all initialisation

    siod_init(heap_size);
    festival_lisp_vars();
    festival_lisp_funcs();
    festival_types();
    if (load_init_files)
	festival_load_default_files();

    cdebug = new ofstream("/dev/null");  // This wont work on Win/NT
    stddebug = fopen("/dev/null","w");

    return;
}

void festival_repl(int interactive)
{
    // top level read eval print loop

    if (interactive)
	festival_banner();
    siod_repl(interactive);  

}

void festival_server_mode(void)
{
    // Go into server mode 
    LISP lport;

    lport = siod_get_lval("server_port",NULL);

    if (lport != NULL)
	festival_server_port = get_c_long(lport);

    festival_start_server(festival_server_port);

}

void festival_init_lang(const EST_String &language)
{
    // Call the lisp function to setup for names language 
    
    leval(cons(rintern("select_language"),
	       cons(siod_quote(rintern(language)),NIL)),
	  NIL);
}

void festival_load_file(const EST_String &fname)
{
    /* Load and evaluate named file */

    vload((const char *)fname,FALSE);

}

static void festival_banner(void)
{
    /* Print out a banner, copyright and version */
    
    if (siod_get_lval("hush_startup",NULL) == NIL)
    {
	EST_TBI *t;
	cout << "Festival Speech Synthesis System " << 
	    festival_version << endl;
	cout << "Copyright (C) University of Edinburgh, 1996,1997. " <<
	    "All rights reserved." << endl;
	cout << "For details type `(festival_warranty)'" << endl;
	if (sub_copyrights.length() > 0)
	    for (t = sub_copyrights.head(); t != 0; t = next(t))
		cout << sub_copyrights.item(t);
    }
}

void festival_def_utt_module(char *name,LISP (*fcn)(LISP),char *docstring)
{
    // Define an utterance module to the LISP system
    init_subr_1(name,fcn,docstring);
}

void festival_say_file(const EST_String &fname)
{
    /* Say this file as text */
    LISP tts_command;

    tts_command = cons(rintern("tts"),
		       cons(siod_quote(rintern(fname)),
			    cons(NIL,NIL)));
    leval(tts_command,NIL);

}

void festival_wait_for_spooler(void)
{
    leval(cons(rintern("audio_mode"),
	       cons(siod_quote(rintern("close")),NIL)),NIL);
}

static LISP lisp_debug_output(LISP arg)
{
    // switch debug output stream

    if (cdebug != &cerr)
	delete cdebug;
    if (stddebug != stderr)
	fclose(stddebug);

    if (arg == NIL)
    {   // this might be a problem on non-Unix machines
	cdebug = new ofstream("/dev/null");
	stddebug = fopen("/dev/null","w");
    }
    else
    {
	cdebug = &cerr;
	stddebug = stderr;
    }

    return NIL;
}

static void festival_load_default_files(void)
{
    // Load in default files, init.scm. Users ~/.festivalrc 
    // (or whatever you wish to call it) is loaded by init.scm
    EST_String userinitfile, home_str, initfile;

    // Load library init first
    initfile = EST_Pathname(festival_libdir).as_directory() + "init.scm";
    if (access((const char *)initfile,R_OK) == 0)
	vload(initfile,FALSE);
    else
	cerr << "Initialization file " << initfile << " not found" << endl;

}

static void festival_types(void)
{
    // set up types which siod needs to know about.
    utterance_lisp_declare();
    SchemeObject::lisp_declare();
    DummyObject::lisp_declare();
    Unit::lisp_declare();
    UnitDatabase::lisp_declare();
    UnitIndex::lisp_declare();
    Join::lisp_declare();
    SchemeModuleDescription::lisp_declare();
}

static void festival_lisp_vars(void)
{
    // set up specific lisp variables 
    EST_TokenStream ts;
    int major,minor,subminor;
    
    siod_set_lval("libdir",strintern(festival_libdir));
    if (!streq(FTOSTYPE,""))
	siod_set_lval("*ostype*",cintern(FTOSTYPE));
    siod_set_lval("festival_version",
		  strcons(strlen(festival_version),festival_version));
    ts.open_string(festival_version);
    ts.set_WhiteSpaceChars(". ");
    major = atoi(ts.get().string());
    minor = atoi(ts.get().string());
    subminor = atoi(ts.get().string());
    ts.close();
    siod_set_lval("festival_version_number",
		  cons(flocons(major),
		       cons(flocons(minor),
			    cons(flocons(subminor),NIL))));
    siod_set_lval("*modules*",NIL);
    siod_set_lval("*module-descriptions*",NIL);
    if (nas_supported)
	proclaim_module("nas");
    if (sun16_supported)
	proclaim_module("sun16audio");
    if (freebsd16_supported)
	proclaim_module("freebsd16audio");
    if (linux16_supported)
	proclaim_module("linux16audio");
    if (win32audio_supported)
	proclaim_module("win32audio");
    if (mplayer_supported)
	proclaim_module("mplayeraudio");
    
    // Add etc-dir path and machine specific directory etc/$OSTYPE
    char *etcdir = walloc(char,strlen(festival_libdir)+strlen("etc/")+
			  strlen(FTOSTYPE)+3);
    sprintf(etcdir,"%s/etc/%s/",festival_libdir,FTOSTYPE);
    char *etcdircommon = walloc(char,strlen(festival_libdir)+strlen("etc/")+3);
    sprintf(etcdircommon,"%s/etc/",festival_libdir);
    
    //  Modify my PATH to include these directories
    siod_set_lval("etc-path",cons(rintern(etcdir),
				  cons(rintern(etcdircommon),NIL)));
    char *path = getenv("PATH");
    char *newpath = walloc(char,1024+strlen(path)+strlen(etcdir)+
			   strlen(etcdircommon));
    sprintf(newpath,"PATH=%s:%s:%s",path,etcdir,etcdircommon);
    putenv(newpath);
    
    wfree(etcdir);
    wfree(etcdircommon);
    return;
}

static LISP lmake_tmp_filename(void)
{
    EST_String tfile = make_tmp_filename();
    
    return strintern(tfile);
}

LISP l_wagon(LISP utt, LISP si, LISP tree);

static void festival_lisp_funcs(void)
{
    // declare festival specific Lisp functions 
    
    // Standard functions
    festival_phoneset_funcs();
    festival_utt_wave_funcs();
    festival_feature_funcs();
    festival_tcl_init();	// does nothing if TCL not selected
    
    //  General ones
    festival_init_modules();
    
    // Some other ones that aren't anywhere else
    init_subr_1("parse_url", lisp_parse_url,
 "(parse_url URL)\n\
  Split URL into a list (protocol host port path) suitable\n\
  for giving to fopen.");
    init_subr_0("make_tmp_filename",lmake_tmp_filename,
 "(make_tmp_filename)\n\
  Return name of temporary file.");
    init_subr_1("debug_output",lisp_debug_output,
 "(debug_output ARG)\n\
  If ARG is non-nil cause all future debug output to be sent to cerr,\n\
  otherwise discard it (send it to /dev/null).");
    
    init_subr_2("ngram.load",lisp_load_ngram,
 "(ngram.load NAME FILENAME)\n\
  Load an ngram from FILENAME and store it named NAME for later access.");
    init_subr_3("ngram.build",lisp_build_ngram,
 "(ngram.build NAME FILENAME PARAMS)\n\
  Build an ngram from the data in FILENAME and store it called NAME.\n\
  PARAMS is an assoc list of parameters.  The list of defaults\n\
  for these parameters are: (order 2) (vocab NIL)  (pred_vocab vocab)\n\
  (representation dense) (prev_prev_tag \"\") (prev_tag \"\") (last_tag \"\")");
    init_subr_3("ngram.test",lisp_test_ngram,
 "(ngram.test NAME FILENAME PARAMS)\n\
  Finds the perplexity of ngram NAME with respect to the data in FILENAME.\n\
  Respects PARAMS of prev_prev_tag, prev_tag and last_tag.");
    init_subr_3("ngram.save",lisp_save_ngram,
 "(ngram.save NAME FILENAME TYPE)\n\
  Save ngram named NAME in FILENAME using format TYPE.  Where TYPE may be\n\
  one of cstr_bin, cstr_ascii, ...");
    init_subr_2("ngram.smooth",lisp_smooth_ngram,
 "(ngram.smooth NAME SMOOTHCOUNT)\n\
  Smooth named ngrammar to freq limit of SMOOTHCOUNT.");
    
    init_subr_3("wagon",l_wagon,
 "(wagon UTT STREAMITEM TREE)\n\
  Apply the CART tree TREE to STREAMITEM in UTT.  This returns the full\n\
  predicted form, you need to extract the value from the returned form\n\
  itself. [see CART trees]");

#if 0
    init_subr_2("describe_module", describe_module,
    "(describe_module NAME STREAM)\n\
  Print a description of the named module.");
    init_subr_0("describe_all_modules", describe_all_modules,
    "(describe_all_modules)\n\
  Print descriptions of all modules.");
#endif
    
}

void proclaim_module(const EST_String &name,
		     const EST_String &banner_copyright,
		     const ModuleDescription *description)
{
    // Add this name to the variable *modules*, so people can test for
    // it in the lisp world 
    LISP mods = siod_get_lval("*modules*",NULL);
    LISP name_sym = rintern(name);
    
    siod_set_lval("*modules*",cons(name_sym,mods));
    
    if (banner_copyright != "")
	sub_copyrights.append(name + ": " + banner_copyright);

    if (description != NULL)
      {
	LISP module_descriptions = siod_get_lval("*module-descriptions*",NULL);
	LISP scheme_desc = siod_make_typed_cell(tc_festival_schememoduledescription,
						new SchemeModuleDescription(description));

	siod_set_lval("*module-descriptions*",
		      cons(cons(name_sym, 
				cons(scheme_desc, NIL)),
			   module_descriptions));
      }
    
}

void proclaim_module(const EST_String &name,
		     const ModuleDescription *description)
{ 
  proclaim_module(name, "", description); 
}

void init_module_subr(const char *name, LISP (*fcn)(LISP), const ModuleDescription *description)
{
  char * desc_string = NULL;

  if (description)
    {
      EST_String desc(to_string(*description));

      desc_string = wstrdup(desc);
    }

  init_lsubr((char *)name, fcn, desc_string);

  // delete desc_string;
}

LISP ft_get_param(const EST_String &pname)
{
    LISP params,lpair;
    
    params = siod_get_lval("Parameter","no parameters");
    
    lpair = siod_assoc_str(pname,params);
    
    if (lpair == NIL)
	return NIL;
    else
	return car(cdr(lpair));
}

void print_string(EST_String s)
{
    cout << s << endl;
}

#if defined(INSTANTIATE_TEMPLATES)
#endif
