//============================================================================
//
//    SSSS    tt          lll  lll              
//   SS  SS   tt           ll   ll                
//   SS     tttttt  eeee   ll   ll   aaaa    "An Atari 2600 VCS Emulator"
//    SSSS    tt   ee  ee  ll   ll      aa      
//       SS   tt   eeeeee  ll   ll   aaaaa   Copyright (c) 1995,1996,1997
//   SS  SS   tt   ee      ll   ll  aa  aa         Bradford W. Mott
//    SSSS     ttt  eeeee llll llll  aaaaa    
//
//============================================================================

#ifndef M6507_HXX
#define M6507_HXX

#include <fstream.h>

class M6507;

#include "machine.hxx"
#include "System.hxx"

/**
  This is an abstract base class for classes that emulate the
  6507 microprocessor.  For the most part the 6507 is just like 
  the 6502 except it has an 8K address space.

  @author  Bradford W. Mott
  @version $Id: M6507.hxx,v 1.2 1997/05/17 19:00:06 bwmott Exp $
*/
class M6507
{
  public:
    /// Enumeration of the 6507 addressing modes
    enum AddressingMode 
    {
      Implied, Immediate, Absolute, AbsoluteX, AbsoluteY,
      Zero, ZeroX, ZeroY, Indirect, IndirectX, IndirectY, Invalid 
    };

  public:
    /// Constructor
    M6507(System& system);

    /// Destructor
    virtual ~M6507();

  public:
    /// Reset myself to power-on state
    virtual void reset();

    /**
      Execute instructions until at least the given number of 
      cycles have passed or execution is halted by someone
    */
    virtual void execute(uLong numberOfCycles) = 0;

  public:
    /// Indicate that execution should be stopped
    void halt() { myHaltExecution = true; }

    /// Answer the number of CPU cycles executed since the last reset
    uLong cycles() const { return myCycles; }

    /// Waste the given number of CPU cycles
    void wasteCycles(uLong cycles) { myCycles += cycles; }

  public:
#ifdef DEBUG
    // Output stream used for debugging trace
    ofstream debugStream;
#endif

  protected:
    // Called if an illegal instruction is executed
    void invalidInstruction();

  protected:
    // Answers value of the Processor Status register 
    uByte PS() const;

    // Change the processor flags to correspond to the given value
    void PS(uByte ps);

  protected:
    // Output operator for addressing modes
    friend ostream& operator<<(ostream& out, const AddressingMode& mode);

  protected:
    uByte A;    // Accumulator                 8 bits
    uByte X;    // X index register            8 bits
    uByte Y;    // Y index register            8 bits
    uByte SP;   // Stack Pointer               8 bits
    uByte IR;   // Instruction register        8 bits
    uWord PC;   // Program Counter            16 bits

    bool N;     // N flag for processor status register
    bool V;     // V flag for processor status register
    bool B;     // B flag for processor status register
    bool D;     // D flag for processor status register
    bool I;     // I flag for processor status register
    bool Z;     // Z flag for processor status register
    bool C;     // C flag for processor status register
   
    // Indicates if execution should be halted
    bool myHaltExecution;

    // Reference to the system I belong to 
    System& mySystem;
 
    // Indicates how many cycles I've executed
    uLong myCycles;

  protected:
    // Addressing mode for each of the 256 opcodes
    static AddressingMode ourAddressingModeTable[256];

    // Lookup table used for binary-code-decimal math
    static uByte ourBCDTable[2][256];

    // Table of instruction cycle times.  In some cases additional
    // cycles will be added during the execution of an instruction.
    static uLong ourInstructionCycleTable[256];

    // Table of instruction mnemonics
    static const char* ourInstructionMnemonicTable[256];
};
#endif

