/* $Id: util.c,v 3.0 1992/02/23 21:25:39 davison Trn $
 */
/* This software is Copyright 1991 by Stan Barber. 
 *
 * Permission is hereby granted to copy, reproduce, redistribute or otherwise
 * use this software as long as: there is no monetary profit gained
 * specifically from the use or reproduction of this software, it is not
 * sold, rented, traded or otherwise marketed, and this copyright notice is
 * included prominently in any copy made. 
 *
 * The authors make no claims as to the fitness or correctness of this software
 * for any use whatsoever, and it is provided as is. Any use of this software
 * is at the user's own risk. 
 */

#include "EXTERN.h"
#include "common.h"
#include "final.h"
#include "term.h"
#include "intrp.h"
#ifdef SCAN
#include "scan.h"
#include "smisc.h"	/* s_default_cmd */
#endif
#include "sconfig.h"
#include "INTERN.h"
#include "util.h"

static char **rnrc_names INIT(Null(char**));
static char **rnrc_values INIT(Null(char**));
static int rnrc_num INIT(0);	/* also number of allocated values */
/* hack.  This variable points to the first *user* .strnrc value, so that
 * user environment variables can override site-wide .rnrc values.
 */
static int user_rnrc_num INIT(0);

void
util_init()
{
    ;
}
    
/* fork and exec a shell command */

int
doshell(shl,s)
char *s, *shl;
{
    int status, pid, w;
    char *shell;

#ifdef SIGTSTP
    sigset(SIGTSTP,SIG_DFL);
    sigset(SIGTTOU,SIG_DFL);
    sigset(SIGTTIN,SIG_DFL);
#endif
    if (shl != Nullch)
	shell = shl;
    else if ((shell = getval("SHELL",Nullch)) == Nullch || !*shell)
	shell = PREFSHELL;
    if ((pid = vfork()) == 0) {
#ifdef USE_NNTP
        int i;

	/* This is necessary to keep bourne shell from puking */

        for (i = 3; i < 10; ++i)
                close(i);
#endif /* USE_NNTP */

	if (*s)
	    execl(shell, shell, "-c", s, Nullch);
	else
	    execl(shell, shell, Nullch, Nullch, Nullch);
	_exit(127);
    }
    signal(SIGINT, SIG_IGN);
#ifdef SIGQUIT
    signal(SIGQUIT, SIG_IGN);
#endif 
    termlib_reset();
    waiting = TRUE;
    while ((w = wait(&status)) != pid)
	if (w == -1 && errno != EINTR)
	    break;
    if (w == -1)
	status = -1;
    termlib_init();
    waiting = FALSE;
    sigset(SIGINT, int_catcher);	/* always catch interrupts */
#ifdef SIGQUIT
    signal(SIGQUIT, SIG_DFL);
#endif 
#ifdef SIGTSTP
    sigset(SIGTSTP,stop_catcher);
    sigset(SIGTTOU,stop_catcher);
    sigset(SIGTTIN,stop_catcher);
#endif
    return status;
}

#ifndef USE_DEBUGGING_MALLOC
static char nomem[] = "trn: out of memory!\n";

/* paranoid version of malloc */

char *
safemalloc(size)
MEM_SIZE size;
{
    char *ptr;

    ptr = malloc(size ? size : (MEM_SIZE)1);
    if (ptr == Nullch) {
	fputs(nomem,stdout) FLUSH;
	sig_catcher(0);
    }
    return ptr;
}

/* paranoid version of realloc.  If where is NULL, call malloc */

char *
saferealloc(where,size)
char *where;
MEM_SIZE size;
{
    char *ptr;

    if (!where)
	ptr = malloc(size ? size : (MEM_SIZE)1);
    else
	ptr = realloc(where, size ? size : (MEM_SIZE)1);
    if (!ptr) {
	fputs(nomem,stdout) FLUSH;
	sig_catcher(0);
    }
    return ptr;
}
#endif /* !USE_DEBUGGING_MALLOC */

/* safe version of string copy */

char *
safecpy(to,from,len)
char *to;
register char *from;
register int len;
{
    register char *dest = to;

    if (from != Nullch) 
	for (len--; len && (*dest++ = *from++); len--) ;
    *dest = '\0';
    return to;
}

/* safe version of string concatenate, with \n deletion and space padding */

char *
safecat(to,from,len)
char *to;
register char *from;
register int len;
{
    register char *dest = to;

    len--;				/* leave room for null */
    if (*dest) {
	while (len && *dest++) len--;
	if (len) {
	    len--;
	    *(dest-1) = ' ';
	}
    }
    if (from != Nullch)
	while (len && (*dest++ = *from++)) len--;
    if (len)
	dest--;
    if (*(dest-1) == '\n')
	dest--;
    *dest = '\0';
    return to;
}

/* copy a string up to some (non-backslashed) delimiter, if any */

char *
cpytill(to,from,delim)
register char *to, *from;
register int delim;
{
    for (; *from; from++,to++) {
	if (*from == '\\' && from[1] == delim)
	    from++;
	else if (*from == delim)
	    break;
	*to = *from;
    }
    *to = '\0';
    return from;
}

/* return ptr to little string in big string, NULL if not found */

char *
instr(big, little, case_matters)
char *big, *little;
bool_int case_matters;
{
    register char *t, *s, *x;

    for (t = big; *t; t++) {
	for (x=t,s=little; *s; x++,s++) {
	    if (!*x)
		return Nullch;
	    if (case_matters == TRUE) {
		if(*s != *x)
		    break;
	    } else {
		register char c,d;
		if (isupper(*s)) 
		    c = tolower(*s);
		else
		    c = *s;
		if (isupper(*x)) 
		    d = tolower(*x);
		else
		    d = *x;
		if ( c != d )
		    break;
	   }
	}
	if (!*s)
	    return t;
    }
    return Nullch;
}

/* effective access */

#ifdef SETUIDGID
int
eaccess(filename, mod)
char *filename;
int mod;
{
    int protection, euid;
    
    mod &= 7;				/* remove extraneous garbage */
    if (stat(filename, &filestat) < 0)
	return -1;
    euid = geteuid();
    if (euid == ROOTID)
	return 0;
    protection = 7 & (filestat.st_mode >>
      (filestat.st_uid == euid ? 6 :
        (filestat.st_gid == getegid() ? 3 : 0)
      ));
    if ((mod & protection) == mod)
	return 0;
    errno = EACCES;
    return -1;
}
#endif

/*
 * Get working directory
 */
#ifndef HAS_GETWD
#ifdef HAS_GETCWD
char *
getwd(np)
char *np;
{
    char *ret;
    extern char *getcwd();

    if ((ret = getcwd(np,512)) == Nullch) {
	printf("Cannot determine current working directory!\n") FLUSH;
	finalize(1);
    }
    return ret;
}
#else
char *
getwd(np)
char *np;
{
    FILE *popen();
    FILE *pipefp;

    if ((pipefp = popen("/bin/pwd","r")) == Nullfp) {
	printf("Can't run /bin/pwd\n") FLUSH;
	finalize(1);
    }
    fgets(np,512,pipefp);
    np[strlen(np)-1] = '\0';	/* wipe out newline */
    if (pclose(pipefp) == EOF) {
	printf("Failed to run /bin/pwd\n") FLUSH;
	finalize(1);
    }
    return np;
}
#endif
#endif

/* just like fgets but will make bigger buffer as necessary */

char *
get_a_line(original_buffer,buffer_length,fp)
char *original_buffer;
register int buffer_length;
FILE *fp;
{
    register int bufix = 0;
    register int nextch;
    register char *some_buffer_or_other = original_buffer;

    do {
	if (bufix >= buffer_length) {
	    buffer_length *= 2;
	    if (some_buffer_or_other == original_buffer) {
					/* currently static? */
		some_buffer_or_other = safemalloc((MEM_SIZE)buffer_length+1);
		strncpy(some_buffer_or_other,original_buffer,buffer_length/2);
					/* so we must copy it */
	    }
	    else {			/* just grow in place, if possible */
		some_buffer_or_other = saferealloc(some_buffer_or_other,
		    (MEM_SIZE)buffer_length+1);
	    }
	}
	if ((nextch = getc(fp)) == EOF)
	    return Nullch;
	some_buffer_or_other[bufix++] = (char) nextch;
    } while (nextch && nextch != '\n');
    some_buffer_or_other[bufix] = '\0';
    len_last_line_got = bufix;
    buflen_last_line_got = buffer_length;
    return some_buffer_or_other;
}

/* copy a string to a safe spot */

char *
savestr(str)
char *str;
{
    register char *newaddr = safemalloc((MEM_SIZE)(strlen(str)+1));

    strcpy(newaddr,str);
    return newaddr;
}

int
makedir(dirname,nametype)
register char *dirname;
int nametype;
{
#ifdef MAKEDIR
    register char *end;
    register char *s;
    char tmpbuf[1024];
    register char *tbptr = tmpbuf+5;

    for (end = dirname; *end; end++) ;	/* find the end */
    if (nametype == MD_FILE) {		/* not to create last component? */
	for (--end; end != dirname && *end != '/'; --end) ;
	if (*end != '/')
	    return 0;			/* nothing to make */
	*end = '\0';			/* isolate file name */
    }
    strcpy(tmpbuf,"mkdir");

    s = end;
    for (;;) {
	if (stat(dirname,&filestat) >= 0 && (filestat.st_mode & S_IFDIR)) {
					/* does this much exist as a dir? */
	    *s = '/';			/* mark this as existing */
	    break;
	}
	s = rindex(dirname,'/');	/* shorten name */
	if (!s)				/* relative path! */
	    break;			/* hope they know what they are doing */
	*s = '\0';			/* mark as not existing */
    }
    
    for (s=dirname; s <= end; s++) {	/* this is grody but efficient */
	if (!*s) {			/* something to make? */
	    sprintf(tbptr," %s",dirname);
	    tbptr += strlen(tbptr);	/* make it, sort of */
	    *s = '/';			/* mark it made */
	}
    }
    if (nametype == MD_DIR)		/* don't need final slash unless */
	*end = '\0';			/*  a filename follows the dir name */

    return (tbptr==tmpbuf+5 ? 0 : doshell(sh,tmpbuf));
					/* exercise our faith */
#else
    sprintf(cmd_buf,"%s %s %d", filexp(DIRMAKER), dirname, nametype);
    return doshell(sh,cmd_buf);
#endif
}

#ifdef SETENV
static bool firstexport = TRUE;
extern char **environ;

void
export(nam,val)
char *nam, *val;
{
    register int i=envix(nam);		/* where does it go? */

    if (!environ[i]) {			/* does not exist yet */
	if (firstexport) {		/* need we copy environment? */
	    int j;
#ifndef lint
	    char **tmpenv = (char**)	/* point our wand at memory */
		safemalloc((MEM_SIZE) (i+2) * sizeof(char*));
#else
	    char **tmpenv = Null(char **);
#endif /* lint */
    
	    firstexport = FALSE;
	    for (j=0; j<i; j++)		/* copy environment */
		tmpenv[j] = environ[j];
	    environ = tmpenv;		/* tell exec where it is now */
	}
#ifndef lint
	else
	    environ = (char**) saferealloc((char*) environ,
		(MEM_SIZE) (i+2) * sizeof(char*));
					/* just expand it a bit */
#endif /* lint */
	environ[i+1] = Nullch;	/* make sure it's null terminated */
    }
    environ[i] = safemalloc((MEM_SIZE) strlen(nam) + strlen(val) + 2);
					/* this may or may not be in */
					/* the old environ structure */
    sprintf(environ[i],"%s=%s",nam,val);/* all that work just for this */
}

int
envix(nam)
char *nam;
{
    register int i, len = strlen(nam);

    for (i = 0; environ[i]; i++) {
	if (strnEQ(environ[i],nam,len) && environ[i][len] == '=')
	    break;			/* strnEQ must come first to avoid */
    }					/* potential SEGV's */
    return i;
}
#endif

void
notincl(feature)
char *feature;
{
    printf("\nNo room for feature \"%s\" on this machine.\n",feature) FLUSH;
}

char *
getval(nam,def)
char *nam,*def;
{
    char *val;

    int i;
    /* first check to see if there are any user rnrc values */
    for (i=user_rnrc_num;i<rnrc_num;i++)
	if (strEQ(nam,rnrc_names[i]))
	    return(rnrc_values[i]);
    /* if not, then check the environment */
    if ((val = getenv(nam)) == Nullch || !*val) {
	/* check the .strnconfig file */
	if ((val = scf_getval(nam,3)))	/* user or system value */
	    return(val);
	/* check site-wide definitions now */
	for (i=0;i<user_rnrc_num;i++)
	    if (strEQ(nam,rnrc_names[i]))
		return(rnrc_values[i]);
	val = def;	/* just do the default */
    }
    return val;
}

/* grow a static string to at least a certain length */

void
growstr(strptr,curlen,newlen)
char **strptr;
int *curlen;
int newlen;
{
    if (newlen > *curlen) {		/* need more room? */
	if (*curlen)
	    *strptr = saferealloc(*strptr,(MEM_SIZE)newlen);
	else
	    *strptr = safemalloc((MEM_SIZE)newlen);
	*curlen = newlen;
    }
}

void
setdef(buffer,dflt)
char *buffer,*dflt;
{
#ifdef SCAN
    s_default_cmd = FALSE;
#endif
#ifdef STRICTCR
    if (*buffer == ' ')
#else
    if (*buffer == ' ' || *buffer == '\n')
#endif
    {
#ifdef SCAN
	s_default_cmd = TRUE;
#endif
	if (*dflt == '^' && isupper(dflt[1]))
	    *buffer = Ctl(dflt[1]);
	else
	    *buffer = *dflt;
	lastchar = *buffer;
    }
}

void
safelink(old, new)
char *old, *new;
{
#if 0
    extern int sys_nerr;
    extern char *sys_errlist[];
#endif

    if (link(old,new)) {
	printf("Can't link backup (%s) to .newsrc (%s)\n", old, new) FLUSH;
#if 0
	if (errno>0 && errno<sys_nerr)
	    printf("%s\n", sys_errlist[errno]);
#endif
	finalize(1);
    }
}

/* CAA 1/8/93 */
#ifndef HAVE_STRSTR
char *
trn_strstr (s1, s2)
char *s1, *s2;
{
    register char *p = s1;
    register int len = strlen (s2);

    for (;(p = index (p, *s2)) != Nullch;p++)
	if (strncmp(p, s2, len) == 0)
	    return(p);
    return (Nullch);
}
#endif /* !HAVE_STRSTR */

/* yet more buffers... */
static char *rnrc_buf = Nullch;
static char *rnrc_buf2 = Nullch;	/* for easy continuation lines */

void
rn_rc_init()
{
    char *s,*s2;
    char *fname;
    int len;
    FILE *fp;
    bool new_rnrc;

    rnrc_names = (char**)NULL;
    rnrc_values = (char**)NULL;
    rnrc_num = 0;
    user_rnrc_num = 0;

    if (!rnrc_buf) {
	rnrc_buf = (char *)safemalloc(LBUFLEN);
	rnrc_buf2 = (char *)safemalloc(LBUFLEN);
    }

/* system-wide strnrc not supported now */
#if 0
    /* read in the system-wide strnrc */
    fname = filexp("%X/STRNRC");
    fp = fopen(fname,"r");
    if (fp) {
	while (s = fgets(rnrc_buf,LBUFLEN-4,fp)) {
	    /* do continuation lines soon */
	    (void)rn_rc_do_line(s);
	}
	fclose(fp);
    }
#endif

    /* now read in the user's .rnrc */
    user_rnrc_num = rnrc_num;
    new_rnrc = TRUE;
    fname = filexp(getval("STRNRC","%./.strnrc"));
    if (!(fp = fopen(fname,"r"))) {
#if 0
/* this code section can interfere with an ordinary .rnrc */
	/* OK--try the old names (for compatibilty) */
	fname = filexp(getval("RNRC","%./.rnrc"));
	fp = fopen(fname,"r");
	new_rnrc = FALSE;
#endif
    }
    if (!fp)
	return;
    printf("\nWARNING!\nThe .strnrc file is no longer supported.\n");
    printf("Please use the online configuration editor instead.\n");
    printf("To use this editor, type the 'e' key at the old newsgroup");
    printf(" selector\nor the group scan mode.\n") FLUSH;
    printf("To remove this warning, remove the file %s\n",fname);
    get_anything();
    eat_typeahead();
    if (!atoi(getval("USESTRNRC","0"))) {
	fclose(fp);
	return;
    }
    while (s = fgets(rnrc_buf,LBUFLEN-4,fp)) {	/* consider buffer size */
	/* do continuation lines here soon */
	(void)rn_rc_do_line(s);
    }
    fclose(fp);
}

void
rn_rc_do_line(line)
char *line;
{
    char *s;
    char *s1,*s2;
    char ch;
    int i;

    if (!line || !*line)
	return;
    if (line[strlen(line)-1] == '\n')
	line[strlen(line)-1] = '\0';	/* kill the newline */

    for (s=line;isspace(*s);s++)
	; /* EMPTY */
    if (!*s)
	return;
    /* later expand this, but for now just handle "value" requests. */
    if strnEQ(s,"value",5) {
	s = s + 5;
	for (;isspace(*s);s++)
	    ; /* EMPTY */
	if (!*s) {
	    printf(".strnrc: value line without arguments\n") FLUSH;
	    return;
	}
	s1 = s;
	for (;(*s && !isspace(*s));s++)
	    ; /* EMPTY */
	s2 = s;
	for (;isspace(*s);s++)
	    ; /* EMPTY */
	if (!*s) {
	    printf(".strnrc: value line with one argument:\n%s\n",line) FLUSH;
	    return;
	}
	i = rnrc_num;
	if (rnrc_num==0) {
	    rnrc_names = (char**)safemalloc(sizeof(char*));
	    rnrc_values = (char**)safemalloc(sizeof(char*));
	} else {
	    rnrc_names = (char**)saferealloc((char*)rnrc_names,
				(i+1)*sizeof(char*));
	    rnrc_values = (char**)saferealloc((char*)rnrc_values,
				(i+1)*sizeof(char*));
	}
	rnrc_num++;
	ch = *s2;
	*s2 = '\0';
	rnrc_names[i] = savestr(s1);
	*s2 = ch;
	s1 = s;
#ifdef UNDEF
	/* do this code if you prefer trailing space deletion to allowing
	 * multi-word values
	 */
	/* later allow nice things like quotes */
	for (;(*s && !isspace(*s));s++)
	    ; /* EMPTY */
	*s = '\0';	/* just in case there was trailing space */
#endif
	rnrc_values[i] = savestr(s1);
#ifdef UNDEF
	printf("Value keyword parsed successfully:\n|%s| |%s|\n",
		rnrc_names[i],rnrc_values[i]) FLUSH;
#endif
    }
}

void
runback_add_groups(groups)
char *groups;
{
    char *s;

    while (groups && *groups) {
	if (*groups == '(') {
	    groups++;
	    for (s=groups;*s && *s != ')';s++)
		;	/* EMPTY */
	    if (!*s)
		return;	/* complain later about unbalanced parens? */
	    *s = '\0';
	    runback_add_groups(getval(groups,Null(char*)));
	    *s++ = ')';
	    while((*s == ' ') || (*s == '\n'))
		s++;
	    groups = s;
	    continue;
	}
	for (s=groups;*s && (*s != ' ') && (*s != '\n');s++)
	    ;	/* EMPTY */
	*s = '\0';
	setngtodo(groups);
	*s++ = ' ';
	while (*s == ' ')
	    s++;
	groups = s;
    }
}

void
runback_init()
{
    char *s;

    use_runback = TRUE;
    /* savestr used so the copy can be written to */
    s = savestr("(BG_GROUPS)");
    runback_add_groups(s);
    free(s);
}

/* attempts to verify a cryptographic signature. */
void
verify_sig()
{
    int i;
    char *s;
    char lbuf [LBUFLEN];	/* big overkill */

    printf("\n");
    /* RIPEM */
    sprintf(lbuf,"grep -s \"BEGIN PRIVACY-ENHANCED MESSAGE\" %s",filexp("%A"));
    i = doshell(sh,lbuf);
    if (!i) {	/* found RIPEM */
	if (strEQ((s = getval("VERIFY_RIPEM","///")),"///"))
	    sprintf(lbuf,"ripem -d -Y fgs -i %s",filexp("%A"));
	else
	    sprintf(lbuf,"%s",filexp(s));
	i = doshell(sh,lbuf);
	printf("\nReturned value: %d\n",i) FLUSH;
	return;
    }
    /* PGP */
    sprintf(lbuf,"grep -s \"BEGIN PGP SIGNED\" %s",filexp("%A"));
    i = doshell(sh,lbuf);
    if (!i) {	/* found PGP */
	if (strEQ((s = getval("VERIFY_PGP","///")),"///"))
	    sprintf(lbuf,"pgp +batchmode -m %s",filexp("%A"));
	else
	    sprintf(lbuf,"%s",filexp(s));
	i = doshell(sh,lbuf);
	printf("\nReturned value: %d\n",i) FLUSH;
	return;
    }
    printf("No signatures detected.\n") FLUSH;
}

/* Ask for a single character (improve the prompt?) */
char
menu_get_char()
{
    printf("Enter your choice: ");
    fflush(stdout);
    eat_typeahead();
    getcmd(buf);
    printf("%c\n",*buf) FLUSH;
    return(*buf);
}

/* returns pointer to text, or Nullch if no text found */
char *
menu_get_text()
{
    buf[0] = '>';
    buf[1] = FINISHCMD;
    if (!finish_command(TRUE))
	return(Nullch);
    return(buf+1);
}


/* waits for a key to be pressed, and eats the key */
void
menu_pause_key()
{
    eat_typeahead();
    (void)get_anything();
    eat_typeahead();
}

static int tmpfile_num = 0;

/* returns a saved string representing a unique temporary filename */
char *
temp_filename()
{
    char buf[100];
    sprintf(buf,"/tmp/strn%s.%d",filexp("%$"),tmpfile_num++);
    return(savestr(buf));
}
