/*
 * Routines to insert into a news reader to use newsfilter pipe mode
 * By: Brad Templeton
 * This protocol is documented in 'filter.man' in the NewsClip doc
 * directory.
 *
 * Unlike the other files in the NewsClip distribution, this file is
 * released to the public domain.
 *
 * Modified by Clifford A. Adams (caadams@access.digex.net)
 * for inclusion into STRN.  Modifications include ANSI compatability
 * and a new call to return the score of the last article filtered.
 *
 */

#include "EXTERN.h"
#include "common.h"
#ifdef NEWSFILTER
#include "INTERN.h"
#include "filtpipe.h"

#define AS_SIZE 3
#define SEQ_SIZE 6			/* adjust to make struct even bound */
#ifdef MAX_LLEN
# define MAX_ARGLEN MAX_LLEN
#else
# define MAX_ARGLEN 255
#endif

	/* struct for start of message */

struct command {
	char comtype;			/* message type, usually 'C' */
	char comcode;			/* command code */
	char space;			/* the delimiting space */
	char seq_num[SEQ_SIZE];		/* sequence number plus 0 */
	char space2;			/* extra space */
	char arg_size[AS_SIZE];		/* argument size */
	char zerob;			/* a zero byte */
	};

#define MAX_ARGS 40			/* max arguments to a command */

#define ABASE 1				/* first real argument */

/* The pipes we use to talk to the reader (filter end) */

#define F_COMPIPE 0		/* stdin */
#define F_ANSPIPE 1		/* stdout */

static int clip_running = 0;		/* is a filter running? */
static long last_score;			/* score of last filter query */
static long last_art = -1;		/* last article filtered */

static int compipe;			/* device for the command pipe */
static int anspipe;			/* device for the answer pipe */

FILE *pipelog = 0;			/* debugging log */

static void	filter_process	_((char *,int,int));
/* static SIGRET	deadkid		_((int)); /* */
static void	deadkid		_((int)); /* */
static int	parent_init	_((int));
static void	fork_failed	_((int));
static int gen_send _((char_int,char_int,char*,char*,char*,char*,char*));
static void	pipe_abort	_((void));
static int	read_answer	_((char *,char **,int));
static void	bugerr		_((char *));

int
filter_init( dotdir )
char *dotdir;			/* the directory for user news programs */
{
	int com_pipe[2];	/* descriptor pair for command pipe */
	int ans_pipe[2];	/* descriptor pair for answer pipe */
	int child;		/* child pid */


	if( pipe( com_pipe ) ) 
		return -1;		/* failure opening pipes */
	if( pipe( ans_pipe ) ) {
		close( com_pipe[0] );
		close( com_pipe[1] );
		return -1;
		}
	/* the two pipes are open, now fork to run news filter */
	if( child = fork() ) {
		/* this is the parent */

		/* close the unused ends of the pipes */
		close( com_pipe[0] );	/* reading end of command pipe */
		close( ans_pipe[1] );	/* writing end of answer pipe */
		compipe = com_pipe[1];
		anspipe = ans_pipe[0];
		if( child == -1 ) {
			bugerr( "Fork Failure" );
			close( compipe );
			close( anspipe );
			return -1;
			}
		/* await the answer of the child */
		return parent_init(child);
		}
	 else {
		close( com_pipe[1] );	/* writing end of command pipe */
		close( ans_pipe[0] );	/* reading end of answer pipe */
		
		filter_process(dotdir, com_pipe[0], ans_pipe[1]);
		}

}

/* the code to execute the news filter */

static void
filter_process(dotdir, reader, writer)
char *dotdir;				/* directory to run program in */
int reader, writer;			/* pipe descriptors */
{
	char *cliploc;
	char ncdir[255];		/* enough to hold a directory name? */
	char *getenv();
	/* prepare stdin and stdout */

	close( 1 );
	if( dup( writer ) != 1 ) {
		bugerr( "dup writer failure" );
		fork_failed( writer );
		}

	close( 0 );
	if( dup( reader ) != 0 ) {
		bugerr( "Dup reader failure" );
		fork_failed( writer );
		}

	close( reader );
	close( writer );

	/* we now read commands from stdin, write answers to stdout */

	cliploc = getenv( "NCLIP" );
	if( !cliploc ) {
		if( chdir( dotdir ) ) {
			bugerr( "Chdir failure" );
			/* failed to chdir */
			fork_failed(1);
			}
		cliploc = "nclip";
		}
	sprintf( ncdir, "dot=%s", dotdir );
	/* You may want to filter off signals the parent is handling
	   at this point */
	execl( cliploc, "nclip", "mode=pipe", ncdir, 0 );
	/* did that fail?  try execlp of nclip */
	execlp( "nclip", "nclip", "mode=pipe", ncdir, 0 );
	/* oh no, that failed too.  I guess we blew it */
	/* bugerr( "Exec failure" ); */
	fork_failed(1);
}

/* This is called if we find the end of the pipe broke off.  We assume
   that the filter process is dead or funny */

static void
deadkid(dummy)
int dummy;
{
	clip_running = FALSE;		/* disable filtering */
	if( compipe > 0 ) {
		close( compipe );
		compipe = -1;
		}
	if( anspipe >= 0 ) {
		close( anspipe );
		anspipe = -1;
		}
	fprintf( stderr, "\nNews Filter Process Aborted.\n" );
}

static int filter_pid;

/* Initialize the parent.
 * We use the ALARM signal to deal with a news filter process that never
 * starts up or otherwise acts funny.  This way the newsreader does not
 * block forever waiting for it.
 * If the newsreader uses the alarm, it must adapt this code a bit.
 */

static int
parent_init(child)
int child;
{

	char repargs[MAX_ARGLEN];		/* reply arguments */
	int argc;			/* count of arguments to command */
	char *argv[MAX_ARGS];		/* argument pointer vector */

	/* debug */
	/* pipelog = fopen( "/tmp/rpipelog", "w" ); */
	if( pipelog )
		setbuf( pipelog, NULL );
	/* set up to ignore alarm clock, as we only want it to halt the
	   pipe read if it fails */
	sigset( SIGALRM, SIG_IGN );
	/* in case child dies, avoid this signal */
	sigset( SIGPIPE, deadkid );

	alarm( 20 );			/* give process 20 seconds to start */

	argc = read_answer( repargs, argv, MAX_ARGS );

	alarm( 0 );

	if( argc > 0 && argv[0][1] == 'O' ) {
		/* ok, we are in communication */
		/* send version command */
		gen_send( 'C', 'V', "V100", NULL, NULL, NULL, NULL );
		argc = read_answer( repargs, argv, MAX_ARGS );
		if( argc > 1 && argv[0][1] == 'V' && atoi(argv[1]+1) >= 100 ) {
			/* we are in business! */
			clip_running = TRUE;
			filter_pid = child;
			return 0;
			}
		bugerr( "Version response failure" );
		}
	 else {
		/* I think the original test (argc>0) was an error */
		if( argc < 1 ) {
			printf( "\nNo News Filter Present\n" );
			fflush(stdout);
			}
		 else
			bugerr( "initial response failure -- no response" );
		}
	close( compipe );
	close( anspipe );
	/* something failed */
	kill( child, SIGTERM );
	return -1;
}

/* Routine to tell that the fork failed */

/* a message of error */
static struct command error_buf = { 'R', 'E', ' ', "-1", ' ', "0", 0 };

static void
fork_failed(desc)
int desc;		/* descriptor to describe failure with */
{
	write( desc, (char *)&error_buf, sizeof(error_buf) );
	exit(1);
}


 /* low level reply routine */
static int
gen_send( ctype, code, a,b,c,d,e )
char ctype;			/* command type */
char code;			/* command code */
char *a,*b,*c,*d,*e;		/* args */
{
	struct command cmd_buf;
	char comargs[MAX_ARGLEN];
	char *vector[6];
	int i, pos;		/* loop counter and position in comargs */

	if( compipe < 0 )
		return;

	cmd_buf.comtype = ctype;
	cmd_buf.comcode = code;
	cmd_buf.space = ' ';

	/* make a vector out of the various args, up to 5 of them */

	vector[0] = a;
	vector[1] = b;
	vector[2] = c;
	vector[3] = d;
	vector[4] = e;

	pos = 0;
	for( i = 0; i < 5 && vector[i]; i++ ) {
		strcpy( comargs+pos, vector[i] );
		/* check overflow? */
		pos += strlen(vector[i]) + 1;
		}
	sprintf( cmd_buf.arg_size, "%03d", pos );
	cmd_buf.zerob =0;
	/* sequence number is unimportant */
	sprintf( cmd_buf.seq_num, "%5.5s", "" );
	cmd_buf.space2 = ' ';

	if( pipelog ) {
		int ic;
		fprintf( pipelog, "R:Send %s\n", &cmd_buf.comtype );
		for( ic = 0; vector[ic]; ic++ )
			fprintf( pipelog, ":%s:", vector[ic] );
		fprintf( pipelog, "\n" );
		}
	if( write(compipe, (char*)&cmd_buf, sizeof(cmd_buf))
	    == sizeof(cmd_buf)){
		if( pos > 0 && write( compipe, (char *) comargs, pos )
			       != pos ) {
			pipe_abort();
			return -1;
			}
		}
	 else {
		pipe_abort();
		return -1;
		}
	return 0;
}

static void
pipe_abort()
{
	if( pipelog )
		fprintf( pipelog, "Pipe abort\n" );
	kill( filter_pid, SIGTERM );
	deadkid(0);
}

static struct command resp_buf;		/* command input buffer */

static int
read_answer( argument_buf, argv, avsize )
char *argument_buf;	/* buffer to store arguments in */
char **argv;		/* pointer to array to store arg pointers in */
int avsize;		/* number of elements in argv */
{
	int argsize;		/* size of argument buffer */
	int argc;		/* argument count */
	int scanloc;		/* where in the arg list we are */
	int foo;

	argc = 0;

	if( anspipe < 0 )
		return 0;

	foo = read( anspipe, &resp_buf, sizeof(struct command) );
	if( pipelog )
		fprintf( pipelog, "R:Got %d bytes - %s\n", foo,
					&resp_buf.comtype );
	if( foo == (int)sizeof(struct command) ) {
		argv[argc++] = &resp_buf.comtype;
		resp_buf.space = 0;
		argsize = atoi( resp_buf.arg_size );
		if( argsize > 0 ) {
			if( argsize > MAX_ARGLEN ) {
				int i;
				char c;
				/* read away the too long arguments */
				for( i = 0; i < argsize; i++ )
					read( anspipe, &c, 1 );
				return -1;
				}
			 else {
				if( read( anspipe, argument_buf, argsize )
							!= argsize ) {
					return 0;
					}
				/* if not null terminated, give an error */
				if( argument_buf[argsize-1] != 0 )
					return -1;
				}
			/* now scan the arguments into argv */
			for( scanloc = 0; scanloc < argsize;
				     scanloc += strlen(argument_buf+scanloc)+1 )
				if( argc < avsize ) {
					argv[argc++] = argument_buf+scanloc;
					}
			if( pipelog ) {
				int i;
				for( i = 1; i < argc; i++ )
					fprintf(pipelog, "R:Got Arg %d: %s\n",
							i, argv[i] );
				}
			}
		}
	 else {
		bugerr( "End of file" );
		deadkid(0);
		return 0;		/* end of file */
		}

	return argc;
}

/* Called when the newsreader terminates */

void
filter_quit()
{
	char repargs[MAX_ARGLEN];		/* reply arguments */
	int argc;			/* count of arguments to command */
	char *argv[MAX_ARGS];		/* argument pointer vector */

	if( clip_running ) {
		gen_send( 'C', 'Q', NULL, NULL, NULL, NULL, NULL );
		argc = read_answer( repargs, argv, MAX_ARGS );
		/* if not OK, what to do? */
		close( compipe );
		close( anspipe );
		clip_running = FALSE;
		if( argc != ABASE && argv[0][1] != 'O' ) {
			bugerr( "Quit failure" );
			kill( filter_pid, SIGTERM );
			}
		}
}

/* Now the routines that are used within the RN newsreader to query articles */

/* Returns true if the article is accepted, false otherwise */

int
filter_art( spool, ngdir, ngname, art )
char *spool;		/* article spool dir or NULL */
char *ngdir;		/* newsgroup directory name */
char *ngname;		/* newsgroup name */
long art;		/* article number */
{
	char repargs[MAX_ARGLEN];		/* reply arguments */
	char artnum[20];			/* article number string */
	int argc;			/* count of arguments to command */
	char *argv[MAX_ARGS];		/* argument pointer vector */

	if( !clip_running )
		return TRUE;

	sprintf( artnum, "%ld", art );
	/* build article filename */
	if( ngdir )
		sprintf( repargs, "%s/%s/%s", spool, ngdir, artnum );
	 else
		strncpy( repargs, spool, sizeof(repargs) );

	/* send off request on article */
	/* We use the 'full' article mode */
	gen_send( 'C', 'A', ngname, artnum, "F", repargs, NULL );

	/* await response */
	argc = read_answer( repargs, argv, MAX_ARGS );

	last_score = 0;		/* just in case no score exists */
	last_art = art;		/* for the article # */
	/* any kind of error or message other than Reject means accept */
	if( argc < 1 || argv[0][1] != 'R' ) {
		if ((argc>1) && (argv[0][1] == 'A'))	/* explicit accept */
			last_score = atoi(argv[1]);
		return TRUE;
	 } else {
		if ((argc>1) && (argv[0][1] == 'R'))	/* explicit accept */
			last_score = atoi(argv[1]);
		return FALSE;
	}
	/* should we shut down on errors? */
}

/* indicate if we should filter this group or not */

int
no_filter_group( groupname )
char *groupname;
{
	char repargs[MAX_ARGLEN];		/* reply arguments */
	int argc;			/* count of arguments to command */
	char *argv[MAX_ARGS];		/* argument pointer vector */

	if( clip_running ) {
		gen_send( 'C', 'N', groupname, NULL, NULL, NULL, NULL );

		/* await response */
		argc = read_answer( repargs, argv, MAX_ARGS );
	
		return argc < 1 || argv[0][1] != 'O';
		}
	 else
		return TRUE;		/* do not filter */
}

static void
bugerr( str )
char *str;
{
	fprintf( stderr, "BUG: %s\n", str );
}

/* accept a command for the filter.  Return true if the command was
   valid, false if it was not a valid command */

int
filter_command( com )
char *com;
{
	char repargs[MAX_ARGLEN];		/* reply arguments */
	int argc;			/* count of arguments to command */
	char *argv[MAX_ARGS];		/* argument pointer vector */

	if( clip_running ) {
		/* truncate command to safe length */
		strncpy( repargs, com, MAX_ARGLEN-1 );
		repargs[MAX_ARGLEN-1] = 0;
		gen_send( 'C', 'P', repargs, NULL, NULL, NULL, NULL );

		/* await response */
		argc = read_answer( repargs, argv, MAX_ARGS );
	
		/* return true if we got OK, false otherwise */
		return argc > 0 && argv[0][1] == 'O';
		}
	 else
		return FALSE;		/* command not accepted */
}

long
filter_score_art(artnum)
long artnum;	/* last article #, used as a check */
{
	if (last_art == artnum)
		return(last_score);
	return((long)0);
}
#endif /* NEWSFILTER */
