
/*
 * DNEWSLINK.C
 *
 * dnewslink -b batchfile -B srcipaddress -h host -t timeout -l log-after 
 * 	-c close-reopen-after -pipe ...other options
 *
 * This program will attempt to run one or more batch files containing
 * article references of the form:
 *
 * Relative-Path <message-id>
 * Relative-Path <message-id>
 *
 * For example:
 *
 * de/comm/chatsystems/911 <4mle9s$mef@nz12.rz.uni-karlsruhe.de>
 * alt/binaries/multimedia/d/3643 <4mme9r$plc@nntp1.best.com>
 *
 * This program also supports multi-article batch files.  Each article in the
 * multi-article batch file must be terminated with a \0 and batch file lines
 * must include and offset and byte count (non-inclusive of the \0):
 *
 * de/comm/chatsystems/911 <4mle9s$mef@nz12.rz.uni-karlsruhe.de> off,size
 * alt/binaries/multimedia/d/3643 <4mme9r$plc@nntp1.best.com> off,size
 *
 * The primary methodology of using this program is to have another
 * program, SPOOLOUT, generate and maintain numerically indexed spool files.
 * That program forks and runs this program to actually process the spool
 * files.
 *
 * This program's job is to process one or more spool files (performing locking
 * and skipping locked spool files), spool the articles in question to the
 * remote hosts, then delete the spool files as process is completed, or
 * rewrite the spool files if a failure occurs.
 *
 * (c)Copyright 1997, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution 
 *    for specific rights granted.
 */

#include "defs.h"

#define DIABLO

#define T_ACCEPTED	1
#define T_REFUSED	2
#define T_REJECTED	3
#define T_FAILED	4
#define T_STREAMING	5
#define T_FAILEDEXIT	6

#define INF_HELP        100     /* Help text on way */
#define INF_AUTH        180     /* Authorization capabilities */
#define INF_DEBUG       199     /* Debug output */

#ifndef SPOOL
#define SPOOL		"/news/spool/news"
#endif
#ifndef OUTGOING
#ifdef DIABLO
#define OUTGOING	"/news/dqueue"
#else
#define OUTGOING	"/news/spool/out.going"
#endif
#endif

#define MAXCLINE	8192
#define MAXFILEDES	32

#define OUR_DELAY	1
#define THEIR_DELAY	2
#define DDTIME		30

#define OK_CANPOST      200     /* Hello; you can post */
#define OK_NOPOST       201     /* Hello; you can't post */
#define OK_SLAVE        202     /* Slave status noted */
#define OK_STREAMOK	203	/* Can-do streaming	*/
#define OK_GOODBYE      205     /* Closing connection */
#define OK_GROUP        211     /* Group selected */
#define OK_GROUPS       215     /* Newsgroups follow */
#define OK_ARTICLE      220     /* Article (head & body) follows */
#define OK_HEAD         221     /* Head follows */
#define OK_BODY         222     /* Body follows */
#define OK_NOTEXT       223     /* No text sent -- stat, next, last */
#define OK_NEWNEWS      230     /* New articles by message-id follow */
#define OK_NEWGROUPS    231     /* New newsgroups follow */
#define OK_XFERED       235     /* Article transferred successfully */
#define OK_STRMCHECK	238	/* check response / want article	*/
#define OK_STRMTAKE	239	/* takeit response / article received	*/
#define OK_POSTED       240     /* Article posted successfully */
#define OK_AUTHSYS      280     /* Authorization system ok */
#define OK_AUTH         281     /* Authorization (user/pass) ok */

#define CONT_XFER       335     /* Continue to send article */
#define CONT_POST       340     /* Continue to post article */
#define NEED_AUTHINFO   380     /* authorization is required */
#define NEED_AUTHDATA   381     /* <type> authorization data required */

#define ERR_GOODBYE     400     /* Have to hang up for some reason */
#define ERR_NOGROUP     411     /* No such newsgroup */
#define ERR_NCING       412     /* Not currently in newsgroup */
#define ERR_NOCRNT      420     /* No current article selected */
#define ERR_NONEXT      421     /* No next article in this group */
#define ERR_NOPREV      422     /* No previous article in this group */
#define ERR_NOARTIG     423     /* No such article in this group */
#define ERR_NOART       430     /* No such article at all */
#define ERR_RESEND	431	/* please resend the article	*/
#define ERR_GOTIT       435     /* Already got that article, don't send */
#define ERR_XFERFAIL    436     /* Transfer failed */
#define ERR_XFERRJCT    437     /* ihave, Article rejected, don't resend */
#define ERR_STRMCHECK	438	/* check response / do not want article */
#define ERR_STRMTAKE	439	/* takeit response / article failed	*/
#define ERR_NOPOST      440     /* Posting not allowed */
#define ERR_POSTFAIL    441     /* Posting failed */
#define ERR_NOAUTH      480     /* authorization required for command */
#define ERR_AUTHSYS     481     /* Authorization system invalid */
#define ERR_AUTHREJ     482     /* Authorization data rejected */

#define ERR_COMMAND     500     /* Command not recognized */
#define ERR_CMDSYN      501     /* Command syntax error */
#define ERR_ACCESS      502     /* Access to server denied */
#define ERR_FAULT       503     /* Program fault, command not performed */
#define ERR_AUTHBAD     580     /* Authorization Failed */

#define ERR_UNKNOWN	990

/*
 * Streaming parameters
 */

#define STREAM_OFF	0
#define STREAM_RELOAD	1
#define STREAM_ON	2

#define STATE_EMPTY	0	/* empty slot, must be 0		*/
#define STATE_CHECK	1	/* check transmitted			*/
#define STATE_POSTED	2	/* takethis + article transmitted	*/
#define STATE_RETRY	3	/* retry after connection failure	*/

#define MAXSTREAM	16	/* Minimum is 2				*/
#define STREAMFRAC	10	/* number of check responses before 	*/
				/* MaxStream is incremented		*/
#define MAXPENDBYTES	1024 - (MAXSTREAM * 8)	/* maximum pending bytes*/

typedef struct Stream {
    int	st_State;		/* state	*/
    int st_DumpRCode;
    char *st_RelPath;		/* path		*/
    char *st_MsgId;		/* message id	*/
    int32 st_Off;		/* file offset	*/
    int32 st_Size;		/* file size	*/
} Stream;

int connectTo(const char *hostName, const char *serviceName, int defPort);
int Transact(int cfd, const char *relPath, const char *msgId, int o, int s);
int DumpArticle(int cfd, const char *relPath, int o, int s);
int StreamTransact(int cfd, const char *relPath, const char *msgId, int o, int s);
void StreamReload(int cfd);
Stream *LocateStream(const char *msgId, int state);
int RefilePendingStreams(FILE *fo);
int commandResponse(int cfd, char **rptr, const char *ctl, ...);
void clearResponseBuf(void);
void readreset(int fd);
int readretry(char *buf, int size);
int readline(int fd, char *buf, int size);
void logit(const char *ctl, ...);
void logStats(const char *description);
int ValidMsgId(char *msgid);
void AttemptRemoveRenamedFile(struct stat *st, const char *spoolFile);
char *cdmap(const char *path, int off, int *psize, int *multiArtFile);
void cdunmap(char *ptr, int bytes, int multiArtFile);
int extractFeedLine(const char *buf, char *relPath, char *msgId, int *poff, int *psize);
const char *tstamp(void);

#ifdef CREDTIME
void credtime(int whos);
void credreset(void);
#else
#define credtime(whos)
#define credreset()
#endif

int LogAfter = 1000;
int CloseReopenAfter = 1000;
int Timeout = 600;
int DeleteDetectOpt = 0;
int WaitTime = 10;
int Port = 119;
int TailOpt = 0;
int TxBufSize = 0;
int RxBufSize = 0;
struct stat CurSt;

Stream StreamAry[MAXSTREAM];
int MaxStream = MAXSTREAM;
#ifdef NOTDEF
int MaxStreamFrac = 0;
#endif
int StreamMode = 0;		/* set by stream negotiation	*/
int StreamPend = 0;		/* pending streaming requests	*/
int StreamRetry = 0;		/* retry after connection failure */
int TryStreaming = 1;
int BytesPend = 0;

int LogAfterCount = -1;
int CloseReopenCount = -1;
int KillFd = -1;

char *HostName;
char *OutboundIpName;
char *BatchFileCtl;
char CurrentBatchFile[1024];
char LastErrBuf[256];
int BatchSeq = -1;
int NumBatches = 1;
int32 AcceptedBytes;

int AcceptedTotal;
int RefusedTotal;
int RejectedTotal;
int CountTotal;
int PipeOpt;
int RealTimeOpt;
int AcceptedBytesTotal;
time_t TimeStart;

int AcceptedCnt;
int RefusedCnt;
int RejectedCnt;
int Count;
time_t DeltaStart;

int TermFlag;
int StatSize;
char *StatBuf;

int OurMs;
int TheirMs;
int MsCount;

int OurMsTotal;
int TheirMsTotal;
int MsCountTotal;
int WouldHaveRefiled;

void sigTerm(int sigNo);
void bsprintf(const char *ctl, ...);

int
main(int ac, char **av)
{
    int i;
    int eNoBat = 0;

    openlog("newslink", LOG_PERROR | LOG_NDELAY | LOG_PID, LOG_NEWS);

    for (i = 1; i < ac; ++i) {
	char *ptr = av[i];

	ptr += 2;
	switch(ptr[-1]) {
	case 's':
	    ptr = (*ptr) ? ptr : av[++i];
	    if (strlen(ptr) > 23) {
		StatBuf = ptr;
		StatSize = strlen(ptr);
	    }
	    break;
	case 'T':
	    TxBufSize = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
	    if (TxBufSize < 512)
	        TxBufSize = 512;
	    break;
	case 'R':
	    /*
	     * note: must be large enough to hold check responses that
	     * may become pending while we are transmitting an article.
	     */
	    RxBufSize = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
	    if (RxBufSize < MAXSTREAM * (MAXMSGIDLEN + 64))
	        RxBufSize = MAXSTREAM * (MAXMSGIDLEN + 64);
	    break;
	case 'n':
	    /* NOP */
	    break;
	case 'i':
	    TryStreaming = 0;
	    break;
	case 'D':
	    DeleteDetectOpt = 1;
	    break;
	case 'f':
	    TailOpt = 1;	/* sit on tailable file XXX	*/
	    break;
	case 'p':
	    PipeOpt = 1;	/* input from pipe 	*/
	    break;
	case 'P':
	    Port = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
	    break;
	case 'd':
	    DebugOpt = (*ptr) ? strtol(ptr, NULL, 0) : 1;
	    break;
	case 'b':
	    BatchFileCtl = (*ptr) ? ptr : av[++i];
	    break;
	case 'r':
	    RealTimeOpt = (*ptr) ? strtol(ptr, NULL, 0) : 1;
	    break;
	case 'S':
	    BatchSeq = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
	    break;
	case 'N':
	    NumBatches = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
	    break;
	case 'h':
	    HostName = (*ptr) ? ptr : av[++i];
	    break;
	case 'B':
	    OutboundIpName = (*ptr) ? ptr : av[++i];
	    break;
	case 'w':
	    WaitTime = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
	    break;
	case 't':
	    Timeout = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
	    break;
	case 'l':
	    LogAfter = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
	    break;
	case 'c':
	    CloseReopenAfter = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
	    break;
	}
    }
    LogAfterCount += LogAfter;
    CloseReopenCount += CloseReopenAfter;

    if (BatchFileCtl == NULL || HostName == NULL || i > ac) {
	printf("newslink %4.2f\n", VERS);
	puts(av[0]);
	puts(
	    "-b batchfile    - specify batchfile\n"
	    "-b template%d   - template containing %[xx]d\n"
	    "-h host         - specify remote INND host\n"
	    "-t #            - specify timeout, default 600s\n"
	    "-l #            - log-after-count, default 1000\n"
	    "-c #            - close-reopen-after, default 1000\n"
	    "-s \"<24chars>\"  - /bin/ps argv space for status\n"
	    "-D              - detect delete-out-from-under\n"
	    "-p              - input on stdin rather then file\n"
	    "-d[#]           - debug option\n"
	    "-S #            - starting sequence #, template mode\n"
	    "-N #            - number of batchfiles to process, template mode\n"
	    "-P #            - specify destination tcp port\n"
	    "-w #            - reconnect-after-failure delay\n"
	    "-i              - disable streaming & streaming check\n"
	    "-T #            - set transmit buffer size\n"
	    "-R #            - set receive buffer size\n"
	    "-B ip           - set source ip address for outbound connections\n"
	    "-nop            - NOP"
	);
	exit(1);
    }

    rsignal(SIGPIPE, SIG_IGN);
    rsignal(SIGHUP, sigTerm);
    rsignal(SIGINT, sigTerm);
    rsignal(SIGALRM, sigTerm);

    if (*BatchFileCtl == 0) {
	logit("batchfile is null!\n");
	exit(0);
    }

    while (--NumBatches >= 0) {
	time_t ddTime;
	int fd;
	int cfd = -1;

	AcceptedBytes = 0;
	AcceptedBytesTotal = 0;

	AcceptedTotal = 0;
	RefusedTotal = 0;
	RejectedTotal = 0;
	CountTotal = 0;

	AcceptedCnt = 0;
	RefusedCnt = 0;
	RejectedCnt = 0;
	Count = 0;

	OurMs = 0;
	TheirMs = 0;
	MsCount = 0;

	OurMsTotal = 0;
	TheirMsTotal = 0;
	MsCountTotal = 0;

	sprintf(CurrentBatchFile, "%s/", OUTGOING);

	if (BatchSeq < 0) {
	    sprintf(CurrentBatchFile + strlen(CurrentBatchFile), "%s", BatchFileCtl);
	} else {
	    sprintf(CurrentBatchFile + strlen(CurrentBatchFile), BatchFileCtl, BatchSeq);
	    ++BatchSeq;
	}

	/*
	 * Open next batchfile.  If the open fails, we go onto the next
	 * batchfile.  However, if we are doing a realtime feed, we must
	 * deal with a race condition with diablo where the next realtime
	 * queue file may not yet exist.
	 */

	if (PipeOpt) {
	    fd = 0;
	} else {
	    fd = open(CurrentBatchFile, O_RDWR | (RealTimeOpt ? O_CREAT : 0), 0600);
	    if (fd < 0) {
		if (eNoBat == 0) {
		    eNoBat = 1;
		    logit("no batchfile\n");
		}
		continue;
	    }

	    /*
	     * Lock batchfile
	     */
	    {
		struct stat st;

		if (xflock(fd, XLOCK_EX|XLOCK_NB) != 0) {
		    if (eNoBat == 0) {
			eNoBat = 1;
			logit("batchfile already locked\n");
		    }
		    close(fd);

		    /*
		     * If we cannot get the lock and RealTimeOpt is
		     * set, we abort - someone else has a lock on the
		     * realtime file.  Otherwise we retry (w/ the next
		     * sequence number)
		     */

		    if (RealTimeOpt) {
			logit("realtime batchfile already locked\n");
			break;
		    }
		    continue;
		}
		if (fstat(fd, &st) != 0) {
		    logit("fstat failed: %s\n", strerror(errno));
		    exit(1);
		}
		if (st.st_nlink == 0) {
		    close(fd);
		    continue;
		}
		eNoBat = 0;
	    }
	    fprintf(stderr, "%s\n", CurrentBatchFile);
	} /* pipeopt */

	fstat(fd, &CurSt);
	WouldHaveRefiled = 0;

	TimeStart = DeltaStart = ddTime = time(NULL);

	/*
	 * Connect to remote, send news
	 */

	if ((cfd = connectTo(HostName, NULL, Port)) >= 0) {
	    char buf[1024];
	    char relPath[1024];
	    char msgId[1024];
	    char path[1024];
	    int fileOff;
	    int fileSize;
	    FILE *fo = NULL;
	    int connectCount = 0;
	    int bufInval = 1;
	    int loggedMark = 0;

	    credtime(0);

	    if (PipeOpt)
		sprintf(path, "%s", CurrentBatchFile);
	    else
		sprintf(path, "%s.tmp", CurrentBatchFile);

	    bsprintf("wait/in %d", connectCount);

	    /*
	     * Loop if need to reconnect, there are pending streaming commands,
	     * or we can get a valid buffer.  Note that this allows us to halt
	     * readline() calls when we reach our streaming limit as well as to
	     * do a final-drain when we reach the end of the buffer.
	     */

	    for (;;) {
		/*
		 * If there is no active buffer and we have not reached
		 * our streaming limit, read in another article control line.
		 */
		if (bufInval && 
		    StreamPend < MaxStream && 
		    BytesPend < MAXPENDBYTES
		) {
		    if ((bufInval = readretry(buf, sizeof(buf))) != 0)
			bufInval = readline(fd, buf, sizeof(buf));
		}

		/*
		 * If there is nothing left to do, break out of the loop
		 */

		if (bufInval && StreamPend == 0)
		    break;

		/*
		 * Go.
		 */

		if (DebugOpt > 1)
		    printf("%s bufInval %d StreamPend %d/%d (%s)\n", tstamp(), bufInval, StreamPend, MaxStream, (bufInval) ? "?" : buf);

		if (DeleteDetectOpt && 
		    !PipeOpt && 
		    (int32)(time(NULL) - ddTime) > DDTIME
		) {
		    struct stat st;

		    if (stat(CurrentBatchFile, &st) != 0)
			break;
		    ddTime = time(NULL);
		}

		if (bufInval || extractFeedLine(buf, relPath, msgId, &fileOff, &fileSize) == 0) {
		    int t;

		    bsprintf("process %d", connectCount);

		    if (bufInval)
			t = Transact(cfd, NULL, NULL, 0, 0);
		    else
			t = Transact(cfd, relPath, msgId, fileOff, fileSize);

		    if (DebugOpt > 1)
			printf("%s Transaction result: %d %s", tstamp(), t, (bufInval) ? "<ibuf-empty>\n" : buf);

		    bsprintf("wait/in %d", connectCount);

		    switch(t) {
		    case T_STREAMING:
			/*
			 * operation in progress, will return the real status
			 * later.
			 */
			break;
		    case T_ACCEPTED:
			++AcceptedCnt;
			++AcceptedTotal;
			++connectCount;
			AcceptedBytes += (int32)fileSize;
			AcceptedBytesTotal += (int32)fileSize;
			break;
		    case T_REFUSED:
			++RefusedCnt;
			++RefusedTotal;
			++connectCount;
			break;
		    case T_REJECTED:
			++RejectedCnt;
			++RejectedTotal;
			++connectCount;
			break;
		    case T_FAILED:
		    case T_FAILEDEXIT:
			/*
			 * If failed, attempt to reconnect to remote.
			 * if failed+exit, exit out
			 */
			break;
		    default:
			/* should not occur */
			break;
		    }

		    /*
		     * Many INND's are setup to close a connection after a
		     * certain period of time.  Thus, a failure could be a
		     * normal occurance.  If we have already had at least one
		     * successfull transaction, we attempt to reconnect when
		     * this case occurs.
		     */

		    if (t == T_FAILED || t == T_FAILEDEXIT) {
			if (connectCount > 0 && t != T_FAILEDEXIT && TermFlag == 0) {
			    logit("Remote EOF, attempting to reconnect\n");
			    logStats("mark");
			    loggedMark = 1;
			    close(cfd);
			    KillFd = -1;
			    credtime(OUR_DELAY);
			    (void)RefilePendingStreams(NULL); /* retry */
			    if ((cfd = connectTo(HostName, NULL, Port)) >= 0) {
				connectCount = 0;
				if (CloseReopenCount > 0)
				    CloseReopenCount = CountTotal+CloseReopenAfter;
				credtime(0);
				continue;
			    }
			    credtime(0);
			}

			/*
			 * reopen failed: termination, cleanup
			 */

			NumBatches = 0;

			if (RealTimeOpt == 0) {
			    if (fo == NULL)
				fo = fopen(path, "w");
			    if (fo != NULL) {
				(void)RefilePendingStreams(fo);
				if (!bufInval)
				    fputs(buf, fo);
			    }
			} else {
			    WouldHaveRefiled += RefilePendingStreams(NULL);
			    if (!bufInval)
				++WouldHaveRefiled;
			}
			break; /* break out of for(EVER) */
		    }

		    /*
		     * buffer successfully dealt with, buffer is now invalid
		     */

		    bufInval = 1;

		    /*
		     * Bump transaction count
		     */

		    if (t != T_STREAMING) {
			++Count;
			++CountTotal;
		    }

		    /*
		     * Log stats after specified count
		     */

		    if (LogAfterCount > 0 && CountTotal >= LogAfterCount) {
			loggedMark = 1;
			logStats("mark");
			LogAfterCount += LogAfter;
		    }

		    /*
		     * Close remote to allow remote logging to occur after
		     * specified count (doesn't work with streaming yet)
		     */

		    if (
			StreamMode == STREAM_OFF &&
			CloseReopenCount > 0 && 
			CountTotal >= CloseReopenCount
		    ) {
			char *ptr;

			(void)RefilePendingStreams(NULL); /* retry */
			commandResponse(cfd, &ptr, "quit\r\n");
			close(cfd);
			KillFd = -1;
			credtime(OUR_DELAY);
			if ((cfd = connectTo(HostName, NULL, Port)) < 0) {
			    credtime(0);
			    break;
			}
			credtime(0);
			connectCount = 0;
			CloseReopenCount += CloseReopenAfter;
		    }
		} else {
		    logit("Buffer syntax error: %s\n", buf);
		    bufInval = 1;
		}

		/*
		 * If a signal occured during the transaction, break out of
		 * our loop.
		 */

		if (TermFlag) {
		    logit("Terminated with signal\n");
		    break;
		}
	    } /* for(EVER) */

	    /*
	     * Final delta stats, only if we had
	     * previously logged marks, otherwise the
	     * mark stats will be the same as the final
	     * stats and we do not bother logging the mark.
	     */

	    if (loggedMark != 0)
		logStats("mark");

	    /*
	     * Rewrite batchfile, but only if not a realtime dnewslink.
	     */

	    if (RealTimeOpt == 0) {
		if (StreamPend) {
		    if (fo == NULL)
			fo = fopen(path, "w");
		    if (fo)
			(void)RefilePendingStreams(fo);
		}

		if (readline(fd, buf, sizeof(buf)) == 0 || fo != NULL) {
		    struct stat st;

		    if (fo == NULL)
			fo = fopen(path, "w");
		    if (fo != NULL) {
			do {
			    fputs(buf, fo);
			} while (readline(fd, buf, sizeof(buf)) == 0);
			fflush(fo);
			fclose(fo);
			fo = NULL;
		    }
		    if (DeleteDetectOpt && 
			!PipeOpt && 
			!RealTimeOpt &&
			stat(CurrentBatchFile, &st) != 0
		    ) {
			remove(path);
		    } else if (PipeOpt == 0 && RealTimeOpt == 0) {
			rename(path, CurrentBatchFile);
		    }
		} else {
		    if (PipeOpt == 0 && RealTimeOpt == 0)
			remove(CurrentBatchFile);
		}
	    } else {
		/*
		 * refiling does not work if we are in realtime mode
		 */
		if (StreamPend)
		    WouldHaveRefiled += RefilePendingStreams(NULL);
		if (readline(fd, buf, sizeof(buf)) == 0)
		    WouldHaveRefiled = 1;
	    }

	    /*
	     * Final overall stats
	     */

	    AcceptedBytes = AcceptedBytesTotal;
	    AcceptedCnt = AcceptedTotal;
	    RefusedCnt = RefusedTotal;
	    RejectedCnt = RejectedTotal;
	    Count = CountTotal;
	    DeltaStart = TimeStart;
	    OurMs += OurMsTotal;
	    TheirMs += TheirMsTotal;
	    MsCount += MsCountTotal;

	    logStats("final");

	    /*
	     * be nice
	     */

	    if (cfd >= 0) {
		char *ptr;

		commandResponse(cfd, &ptr, "quit\r\n");
		close(cfd);
		KillFd = -1;
		/* normal quit, do not set cfd to -1 	     */
		/* this allows us to go on to the next batch */
	    }
	} /* cfd=connectTo */

	if (fd > 0) {
	    readreset(fd);
	    close(fd);
	}
	if (cfd < 0 || TermFlag)
	    break;
    }

    exit(0);
}

int
connectTo(const char *hostName, const char *serviceName, int defPort)
{
    struct sockaddr_in sin;
    int fd = socket(AF_INET, SOCK_STREAM, 0);
    int cfd = -1;
    static int ConnectCount;

    clearResponseBuf();

    memset(&sin, 0, sizeof(sin));
    ++ConnectCount;

    bsprintf("connect: %d", ConnectCount);

    /*
     * Make sure keepalive is turned on to prevent infinite hangs
     */

    {
	int on = 1;
	setsockopt(fd, SOL_SOCKET, SO_KEEPALIVE, (void *)&on, sizeof(on));
    }

    /*
     * Set the transmit and receive buffer size
     */

    if (TxBufSize) {
	setsockopt(fd, SOL_SOCKET, SO_SNDBUF, (void *)&TxBufSize, sizeof(TxBufSize));
    }
    if (RxBufSize) {
	setsockopt(fd, SOL_SOCKET, SO_RCVBUF, (void *)&RxBufSize, sizeof(RxBufSize));
    }

    /*
     * figure out the ip address and port
     */

    {
	struct hostent *host = gethostbyname(hostName);

	if (host) {
	    sin.sin_family = host->h_addrtype;
	    memmove(&sin.sin_addr, host->h_addr, host->h_length);
	} else if (strtol(hostName, NULL, 0) != 0) {
	    sin.sin_family = AF_INET;
	    sin.sin_addr.s_addr = inet_addr(hostName);
	} else {
	    logit("hostname lookup failure: %s\n", strerror(errno));
	    fd = -1;
	}
    }
    {
	struct servent *serv = serviceName ? getservbyname(serviceName, "tcp") : NULL;
	if (serv) {
	    sin.sin_port = serv->s_port;
	} else {
	    sin.sin_port = htons(defPort);
	    if (serviceName) {
		logit("unable to lookup service '%s', using port %d\n", 
		    serviceName, 
		    defPort
		);
	    }
	}
    }

    /*
     * bind outbound socket if requested
     */

    if (OutboundIpName != NULL) {
	struct hostent *host = gethostbyname(OutboundIpName);
	struct sockaddr_in bsin;

	bzero(&bsin, sizeof(bsin));

	if (host) {
	    bsin.sin_family = host->h_addrtype;
	    memmove(&bsin.sin_addr, host->h_addr, host->h_length);
	} else if (strtol(OutboundIpName, NULL, 0) != 0) {
	    bsin.sin_family = AF_INET;
	    bsin.sin_addr.s_addr = inet_addr(OutboundIpName);
	} else {
	    logit("outbound -B address lookup failure: %s\n", strerror(errno));
	    close(fd);
	    fd = -1;
	}
	if (bind(fd, (struct sockaddr *)&bsin, sizeof(bsin)) < 0) {
	    logit("bind() failed on -B address: %s\n", strerror(errno));
	    close(fd);
	    fd = -1;
	}
    }

    /*
     * connect to the destination
     */

    if (fd >= 0 && connect(fd, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
	logit("connect: %s\n", strerror(errno));
	close(fd);
	fd = -1;
	bsprintf("connect: fail");
	sleep(5);
    }

    /*
     * Sometimes setting the transmit and receive buffer sizes prior to
     * the connect does not work, because the connect() overrides the 
     * parameters based on the destination route.  So, we do it here as well.
     */

    if (TxBufSize) {
	setsockopt(fd, SOL_SOCKET, SO_SNDBUF, (void *)&TxBufSize, sizeof(TxBufSize));
    }
    if (RxBufSize) {
	setsockopt(fd, SOL_SOCKET, SO_RCVBUF, (void *)&RxBufSize, sizeof(RxBufSize));
    }

    LastErrBuf[0] = 0;

    /*
     * get initial message from remote
     */

    if (fd >= 0) {
	char *ptr;

	cfd = fd;

	switch(commandResponse(cfd, &ptr, NULL)) {
	case OK_CANPOST:	/* innd	*/
	case OK_NOPOST:		/* nntpd may still allow news transfers */
	    break;
	default:
	    logit("connect: %s\n", (ptr ? ptr : "(unknown error)"));
	    bsprintf("connect: err resp");
	    close(cfd);
	    cfd = -1;
	    fd = -1;
	    break;
	}
	credreset();
    }

    /*
     * Unless streaming is disabled, we attempt to turn on streaming.  If
     * it succeeds, we set the StreamMode to STREAM_RELOAD to resend any
     * queued streaming requests from a prior connection.
     */

    if (cfd >= 0) {
	char *ptr;

	if (TryStreaming) {
	    switch(commandResponse(cfd, &ptr, "mode stream\r\n")) {
	    case OK_STREAMOK:
		StreamMode = STREAM_ON;
		logit("connect: %s (streaming)\n", ptr);
		bsprintf("connect: stream");
		break;
	    default:
		StreamMode = STREAM_OFF;
		logit("connect: %s (nostreaming)\n", ptr);
		bsprintf("connect: nostream");
		break;
	    }
	} else {
	    logit("connect (streaming disabled)\n");
	    bsprintf("connect (streaming disabled)\n");
	    StreamMode = STREAM_OFF;
	}
    }

    /*
     * clean up
     */

    if (cfd < 0) {
	sprintf(LastErrBuf, "connect-timeout");
	sleep(WaitTime);
    }
    KillFd = fd;
    return(cfd);
}

/*
 * Transact() - begin a streaming or non-streaming transaction, or
 *		complete a streaming transaction.
 */

int
Transact(int cfd, const char *relPath, const char *msgId, int off, int size)
{
    int r = 0;
    char *ptr = NULL;

    /*
     * Handle streaming.  If r returns zero, we revert to the
     * old operation (this is only occurs if streaming is disabled).
     */

#ifdef NOTDEF
    if (StreamMode == STREAM_RELOAD) {
	StreamReload(cfd);
	StreamMode = STREAM_ON;
    }
#endif
    if (StreamMode == STREAM_ON) {
	r = StreamTransact(cfd, relPath, msgId, off, size);
	if (r)
	    return(r);
    }

    if (relPath == NULL)
	return(T_STREAMING);

    /*
     * This is necessary to guarentee sufficient receive buffer
     * space.
     */
    if (strlen(msgId) > MAXMSGIDLEN)
	return(T_REJECTED);

    switch(commandResponse(cfd, &ptr, "ihave %s\r\n", msgId)) {
    case CONT_XFER:
	break;
    case ERR_XFERRJCT:
	r = T_REJECTED;
	break;
    case ERR_GOTIT:
	r = T_REFUSED;
	break;
    case ERR_ACCESS:
    case ERR_FAULT:
    case ERR_AUTHBAD:
    case ERR_GOODBYE:
	r = T_FAILEDEXIT;
	break;
    case ERR_NOAUTH:
    default:
	r = T_FAILED;
	break;
    }
    if (r == 0) {
	r = DumpArticle(cfd, relPath, off, size);
	switch(commandResponse(cfd, &ptr, NULL)) {
	case OK_XFERED:
	    if (r == 0)
		r = T_ACCEPTED;
	    break;
	case ERR_GOTIT:
	    if (r == 0)
		r = T_REFUSED;
	    break;
	case ERR_XFERRJCT:
	    if (r == 0)
		r = T_REJECTED;
	    break;
	case ERR_XFERFAIL:
	default:
	    if (r == 0)
		r = T_FAILED;
	    break;
	}

	/*
	 * If we couldn't send the article, we ignore the
	 * response code because the remote might send the
	 * wrong response to the null article.
	 */
	if (r < 0)
	    r = T_REJECTED;
    }
    return(r);
}

int
DumpArticle(int cfd, const char *relPath, int off, int size)
{
    char path[4096];
    char *ptr = NULL;
    FILE *fo = NULL;
    int r = -1;
    int ccfd = dup(cfd);
    int multiArtFile = 0;

    sprintf(path, "%s/%s", SPOOL, relPath);

    if (cfd >= 0 &&
	(ptr = cdmap(path, off, &size, &multiArtFile)) != NULL &&
	(fo = fdopen(ccfd, "w")) != NULL
    ) {
	int i;
	int b;

	r = 0;

	if (DebugOpt > 1) {
	    printf("%s >> (article, %d bytes)\n", tstamp(), size);
	}

	/*
	 * In terms of credtime, we assume that our reading
	 * of the file is instantanious.  I don't want to
	 * call gettimeofday() for each line!
	 */

	credtime(OUR_DELAY);

	for (i = b = 0; i < size; b = i) {
	    /*
	     * if first character is a '.', escape it
	     */

	    if (ptr[i] == '.')
		fputc('.', fo);

	    /*
	     * dump the article.
	     */

	    while (i < size && ptr[i] != '\n')
		++i;
	    fwrite(ptr + b, i - b, 1, fo);
	    fputc('\r', fo);
	    fputc('\n', fo);
	    ++i;

	    /*
	     * if i > fsize, we hit the end of the file without
	     * a terminating LF.  We don't have to do anything
	     * since we've already terminated the last line.
	     */
	}
	fflush(fo);
	credtime(THEIR_DELAY);
    } else {
	if (DebugOpt > 1)
	    printf("%s >> (file not found or other error)\n", tstamp());
    }
    if (ptr)
	cdunmap(ptr, size, multiArtFile);

    if (fo)
	fclose(fo);
    else if (ccfd >= 0)
	close(ccfd);

    commandResponse(cfd, NULL, ".\r\n");

    return(r);
}
	
#ifdef NOTDEF

/*
 * Handle streaming protocol
 *
 * StreamReload() - retransmit pending stream after disconnect/reconnect
 * StreamTransact() - normal stream state machine
 */

void
StreamReload(int cfd)
{
    int i;

    for (i = 0; i < MAXSTREAM; ++i) {
	Stream *s = &StreamAry[i];

	if (s->st_State != STATE_EMPTY) {
	    commandResponse(cfd, NULL, "check %s\r\n", s->st_MsgId);
	    s->st_State = STATE_CHECK;
	}
    }
}

#endif

int
StreamTransact(int cfd, const char *relPath, const char *msgId, int off, int size)
{
    Stream *s;
    int r = T_STREAMING;

    /*
     * Another article to stream?
     */

    if (relPath) {
	if ((s = LocateStream(msgId, 0)) != NULL) {
	    return(T_REFUSED);
	}
	s = &StreamAry[StreamPend];
	s->st_MsgId  = strcpy(malloc(strlen(msgId) + 1), msgId);
	s->st_RelPath   = strcpy(malloc(strlen(relPath) + 1), relPath);
	s->st_Off = off;
	s->st_Size = size;
	commandResponse(cfd, NULL, "check %s\r\n", s->st_MsgId);
	s->st_State = STATE_CHECK;
	BytesPend += strlen(s->st_MsgId);
	++StreamPend;
    }

    /*
     * Should we wait for a response ?
     *
     * note: code has been written to allow for non-blocking reads in the 
     * future.  For now, we simply enforce the pipeline by ensuring the
     * receive buffer is large enough.
     *
     * check command response:	  OK_STRMCHECK, ERR_STRMCHECK, ERR_RESEND?
     * takethis command response: OK_STRMTAKE,  ERR_STRMTAKE, ERR_RESEND
     * misc:			  ERR_GOODBYE
     */

    if ((relPath == NULL && StreamPend > 0) || StreamPend > MaxStream * 2 / 3 - 1) {
	char *ptr;
	Stream *s = NULL;
	int delMe = 0;

	switch(commandResponse(cfd, &ptr, NULL)) {
	case ERR_RESEND:
	    if ((s = LocateStream(ptr, STATE_POSTED)) != NULL) {
		delMe = 1;
		r = T_REJECTED;	/* XXX */
		if (s->st_DumpRCode < 0)
		   r = T_REJECTED;
		break;
	    }
	    /*
	     * fall through.  Consider a resend-later response to a check,
	     * which is really a completely illegal response, to be the same
	     * as OK_STRMCHECK and let the 'takethis' statemachine deal with it.
	     */
	case OK_STRMCHECK:
	    if ((s = LocateStream(ptr, STATE_CHECK)) != NULL) {
		int r;

		commandResponse(cfd, NULL, "takethis %s\r\n", s->st_MsgId);
		r = DumpArticle(cfd, s->st_RelPath, s->st_Off, s->st_Size);
		s->st_State = STATE_POSTED;
		s->st_DumpRCode = r;
	    }
#ifdef NOTDEF
	    if (MaxStream < MAXSTREAM && ++MaxStreamFrac == STREAMFRAC) {
		++MaxStream;
		MaxStreamFrac = 0;
	    }
#endif
	    break;
	case ERR_STRMCHECK:
	    if ((s = LocateStream(ptr, STATE_CHECK)) != NULL) {
		delMe = 1;
		r = T_REFUSED;
	    }
#ifdef NOTDEF
	    if (MaxStream < MAXSTREAM && ++MaxStreamFrac == STREAMFRAC) {
		++MaxStream;
		MaxStreamFrac = 0;
	    }
#endif
	    break;
	case OK_STRMTAKE:
	    if ((s = LocateStream(ptr, STATE_POSTED)) != NULL) {
		delMe = 1;
		r = T_ACCEPTED;
		if (s->st_DumpRCode < 0)
		   r = T_REJECTED;
	    }
#ifdef NOTDEF
	    if (MaxStream < MAXSTREAM && ++MaxStreamFrac == STREAMFRAC) {
		MaxStreamFrac = 0;
		++MaxStream;
	    }
#endif
	    break;
	case ERR_STRMTAKE:
	case ERR_XFERRJCT:
	    /* 
	     * NOTE: ERR_XFERRJCT is an illegal response to a streaming
	     * takethis, but some news systems return it so...
	     */
	    if ((s = LocateStream(ptr, STATE_POSTED)) != NULL) {
		delMe = 1;
		/*r = T_REFUSED; actually, this is a reject */
	        r = T_REJECTED;
		if (s->st_DumpRCode < 0)
		   r = T_REJECTED;
	    }
	    break;
	case ERR_GOODBYE:
	case ERR_ACCESS:
	case ERR_FAULT:
	case ERR_AUTHBAD:
	    r = T_FAILEDEXIT;
	    break;
	default:
	    r = T_FAILED;
	    break;
	}
	if (delMe) {
	    --StreamPend;
	    BytesPend -= strlen(s->st_MsgId);
	    free(s->st_RelPath);
	    free(s->st_MsgId);
	    if (s != &StreamAry[StreamPend]) {
		*s = StreamAry[StreamPend];
		memset(&StreamAry[StreamPend], 0, sizeof(Stream));
	    } else {
		memset(s, 0, sizeof(Stream));
	    }
	}
    }
    if (DebugOpt > 1)
	printf("%s StreamTransact: return %d\n", tstamp(), r);
    return(r);
}

/*
 * Locate id.  Applies only to active id's.  STATE_RETRY id's are
 * not (yet) active.
 */

Stream *
LocateStream(const char *msgId, int state)
{
    int i;
    int idLen;

    while (*msgId && *msgId != '<')
	++msgId;
    for (idLen = 0; msgId[idLen] && msgId[idLen] != '>'; ++idLen)
	;
    if (msgId[idLen] == '>')
	++idLen;

    if (DebugOpt > 1)
	printf("%s MsgId(%*.*s,%d):", tstamp(), idLen, idLen, msgId, state);

    for (i = 0; i < MAXSTREAM; ++i) {
	Stream *s = &StreamAry[i];

	if (s->st_State == STATE_RETRY)
	    continue;

	if ((state == 0 || s->st_State == state) && 
	    s->st_State &&
	    strlen(s->st_MsgId) == idLen &&
	    strncmp(msgId, s->st_MsgId, idLen) == 0
	) {
	    if (DebugOpt > 1)
		printf(" found\n");
	    return(s);
	}
    }
    if (DebugOpt > 1)
	printf(" not found\n");
    return(NULL);
}

/*
 * Refile pending streams.  If we have an output file, the
 * pending streams are refiled to the file.  If we do not,
 * we leave them in the stream array and mark them for RETRY.
 * This will cause them to be regenerated as file input
 * later.
 *
 * It is possible for a previous Refile to mark as retry what
 * we now wish to write to a file.
 */

int
RefilePendingStreams(FILE *fo)
{
    int i;
    int n = 0;

    for (i = 0; i < MAXSTREAM; ++i) {
	Stream *s = &StreamAry[i];

	if (s->st_State != STATE_EMPTY) {
	    /*
	     * If an output file is available, drain pending/retry entries
	     * to it and clear the stream entirely.  Otherwise change all
	     * entries to STREAM_RETRY.
	     */

	    if (fo) {
		if (s->st_Off || s->st_Size) {
		    fprintf(fo, "%s\t%s\t%d,%d\n", 
			s->st_RelPath, 
			s->st_MsgId,
			(int)s->st_Off, 
			(int)s->st_Size
		    );
		} else {
		    fprintf(fo, "%s\t%s\n", s->st_RelPath, s->st_MsgId);
		}
		free(s->st_MsgId);
		free(s->st_RelPath);
		if (s->st_State != STATE_RETRY) {
		    --StreamPend;
		} else {
		    --StreamRetry;
		}
		memset(s, 0, sizeof(Stream));
	    } else {
		if (s->st_State != STATE_RETRY) {
		    s->st_State = STATE_RETRY;
		    ++StreamRetry;
		    --StreamPend;
		}
	    }
	    ++n;
	}
    }
    if (StreamPend != 0) {
	syslog(LOG_NOTICE, "stream array corrupt");
	StreamPend = 0;
    }
    BytesPend = 0;
    return(n);
}

/*
 * Misc subroutines
 */

void 
logit(const char *ctl, ...)
{
    va_list va;
    char buf[1024];

    sprintf(buf, "%s:%s ", 
	((HostName) ? HostName : "<unknown>"),
	((CurrentBatchFile) ? CurrentBatchFile : "<unknown>")
    );
    va_start(va, ctl);
    vsprintf(buf + strlen(buf), ctl, va);
    va_end(va);
    syslog(LOG_NOTICE, "%s", buf);
    if (DebugOpt > 1)
	printf("%s %s%s", tstamp(), buf, ((buf[0] && buf[strlen(buf)-1] == '\n') ? "" : "\n"));
}


void 
logStats(const char *description)
{
    int secs = time(NULL) - DeltaStart;

    /*
     * If we got something through, cut off LastErrBuf
     */

    if (Count)
	LastErrBuf[9] = 0;

    credtime(OUR_DELAY);

    /*
     * Log!
     */

    logit("%s secs=%-4d acc=%-4d dup=%-4d rej=%-4d tot=%-4d bytes=%-4ld (%d/min"
#ifdef CREDTIME
    " %d/%d mS" 
#endif
    ") %s\n",
	description,
	secs,
	AcceptedCnt,
	RefusedCnt,
	RejectedCnt,
	Count,
	AcceptedBytes,
	((secs) ? Count * 60 / secs : 0),
#ifdef CREDTIME
	((MsCount) ? OurMs / MsCount : -1),
	((MsCount) ?  TheirMs / MsCount : -1),
#endif
	LastErrBuf
    );
    DeltaStart = time(NULL);
    AcceptedCnt = 0;
    AcceptedBytes = 0;
    RefusedCnt = 0;
    RejectedCnt = 0;
    Count = 0;

    MsCountTotal += MsCount;
    OurMsTotal += OurMs;
    TheirMsTotal += TheirMs;

    MsCount = 0;
    OurMs = 0;
    TheirMs = 0;
}

/*
 * commandResponse() optionally send command, optionally read a response
 */

static char Buf[4096];
static int Bufb;
static int Bufe;

void
clearResponseBuf(void)
{
    Bufb = Bufe = 0;
}

int
commandResponse(int cfd, char **rptr, const char *ctl, ...)
{
    va_list va;
    int r = ERR_UNKNOWN;
    int n;

    if (ctl) {
	char tmp[4096];

	va_start(va, ctl);
	vsprintf(tmp, ctl, va);
	va_end(va);
	credtime(OUR_DELAY);
	write(cfd, tmp, strlen(tmp));
	credtime(THEIR_DELAY);

	if (DebugOpt > 1) {
	    int i;

	    printf("%s >> ", tstamp());
	    for (i = 0; tmp[i]; ++i) {
		if (isprint(tmp[i]))
		    printf("%c", tmp[i]);
		else
		    printf("[%02x]", tmp[i]);
	    }
	    puts("");
	}
    }
    if (rptr) {
	if (Timeout > 0)
	    alarm(Timeout);

	*rptr = NULL;

	/*
	 * Find next complete line in buffer, read more data if
	 * necessary.
	 */

	n = 0;

	credtime(OUR_DELAY);

	do {
	    int i;

	    credtime(THEIR_DELAY);
	    Bufe += n;
	    for (i = Bufb; i < Bufe; ++i) {
		if (Buf[i] == '\n') {
		    Buf[i] = 0;
		    if (i != Bufb && Buf[i-1] == '\r')
			Buf[i-1] = 0;
		    *rptr = Buf + Bufb;
		    Bufb = i + 1;
		    r = strtol(*rptr, NULL, 10);
		    break;
		}
	    }
	    if (i != Bufe)
		break;
	    /*
	     * We may have run out of space in the
	     * buffer, try to left-justify the data
	     * to make more room.  If we can't, the line
	     * is just too long.
	     */
	    if (Bufe == sizeof(Buf) - 1) {
		if (Bufb == 0)
		    break;
		memcpy(Buf, Buf + Bufb, Bufe - Bufb);
		Bufe -= Bufb;
		Bufb = 0;
	    }
	    credtime(OUR_DELAY);
	} while ((n = read(cfd, Buf + Bufe, sizeof(Buf) - Bufe - 1)) > 0);
	credtime(THEIR_DELAY);
	++MsCount;
	if (Timeout > 0)
	    alarm(0);
	if (DebugOpt > 1)
	    printf("%s << %s\n", tstamp(), (*rptr) ? *rptr : "<interrupt/error>");
	if (*rptr) {
	    strncpy(LastErrBuf, *rptr, 32);
	    LastErrBuf[32] = 0;
	} else {
	    strcpy(LastErrBuf, "<interrupt/error>");
	}
    } else {
	r = 0;
    }
    return(r);
}

void
sigTerm(int sigNo)
{
    TermFlag = 1;
    if (sigNo == SIGALRM) {
	if (KillFd >= 0)
	    close(KillFd);
    }
}

void 
bsprintf(const char *ctl, ...)
{
    if (StatBuf) {
	va_list va;
	int i;

	StatBuf[0] = 0;
 	if (MsCount > 4)
	    sprintf(StatBuf, "%d/%dmS ", OurMs / MsCount, TheirMs / MsCount);
	i = strlen(StatBuf);
	va_start(va, ctl);
	i += vsprintf(StatBuf + i, ctl, va);
	va_end(va);
	memset(StatBuf + i, ' ', StatSize - i - 1);
    }
}

#ifdef CREDTIME

void
credtime(int whos)
{
    struct timeval tv;
    static struct timeval Tv;

    gettimeofday(&tv, NULL);

    if (whos) {
	int ms = (tv.tv_usec + 1000000 - Tv.tv_usec) / 1000 +
	     (tv.tv_sec - Tv.tv_sec - 1) * 1000;

	switch(whos) {
	case OUR_DELAY:
	    OurMs += ms;
	    break;
	case THEIR_DELAY:
	    TheirMs += ms;
	    break;
	}
    }
    Tv = tv;
}

void
credreset(void)
{
    OurMs = TheirMs = 0;
}

#endif

typedef struct {
    char rb_Buf[MAXCLINE-sizeof(int)*3];
    int  rb_Base;
    int  rb_Index;
    int	 rb_Len;
} RBlock;

RBlock *RBlockAry[MAXFILEDES];

void
readreset(int fd)
{
    RBlock *rbs;

    if (fd < 0 || fd >= MAXFILEDES) {
	logit("readreset: bad fd: %d\n", fd);
	return;
    }
    if ((rbs = RBlockAry[fd]) != NULL) {
	free(rbs);
	RBlockAry[fd] = NULL;
    }
}

/*
 * readretry() - read 
 *
 */

int
readretry(char *buf, int size)
{
    int inval = 1;

    if (StreamRetry) {
	int i;

	for (i = 0; i < MAXSTREAM; ++i) {
	    Stream *s = &StreamAry[i];

	    if (s->st_State == STATE_RETRY) {
		if (s->st_Off || s->st_Size) {
		    sprintf(buf, 
			"%s\t%s\t%d,%d",
			s->st_RelPath, 
			s->st_MsgId, 
			s->st_Off,
			s->st_Size
		    );
		} else {
		    sprintf(buf, "%s\t%s", s->st_RelPath, s->st_MsgId);
		}
		memset(s, 0, sizeof(Stream));
		inval = 0;
		--StreamRetry;
		if (DebugOpt > 1)
		    printf("%s (retry from stream): %s\n", tstamp(), buf);
		break;
	    }
	}
    }
    return(inval);
}

/*
 * readline() - read next line from input.  This routine may block if we 
 *		are in realtime mode (-r), but only if StreamPend == 0
 */

int
readline(int fd, char *buf, int size)
{
    RBlock *rbs;

    if (fd < 0 || fd >= MAXFILEDES) {
	logit("readline: bad fd: %d\n", fd);
	return(-1);
    }
    if ((rbs = RBlockAry[fd]) == NULL) {
	rbs = calloc(sizeof(RBlock), 1);
	RBlockAry[fd] = rbs;
    }
    
    /*
     * Ensure there is enough room for a full line
     */

    if (rbs->rb_Base > 0 && rbs->rb_Len > MAXCLINE / 2) {
	memmove(rbs->rb_Buf, rbs->rb_Buf + rbs->rb_Base, rbs->rb_Len - rbs->rb_Base);
	rbs->rb_Len -= rbs->rb_Base;
	rbs->rb_Index -= rbs->rb_Base;
	rbs->rb_Base = 0;
    }

    for (;;) {
	int n;

	/*
	 * Look for newline
	 */

	for (n = rbs->rb_Index; n < rbs->rb_Len; ++n) {
	    if (rbs->rb_Buf[n] == '\n') {
		int s = ++n - rbs->rb_Base;

		if (s >= size)
		    s = size - 1;
		memmove(buf, rbs->rb_Buf + rbs->rb_Base, s);
		buf[s] = 0;

		if (n == rbs->rb_Len) {
		    rbs->rb_Base = 0;
		    rbs->rb_Index = 0;
		    rbs->rb_Len = 0;
		} else {
		    rbs->rb_Base = n;
		    rbs->rb_Index = n;
		}
		return(0);
	    }
	}
	rbs->rb_Index = n;

	/*
	 * None found, read and loop
	 */

	if (rbs->rb_Len == sizeof(rbs->rb_Buf)) {
	    logit("Line too long in batchfile!\n");
	    return(-1);
	}
	n = read(fd, rbs->rb_Buf + rbs->rb_Len, sizeof(rbs->rb_Buf) - rbs->rb_Len);
	if (n <= 0) {
	    struct stat st;

	    /*
	     * If a read error occurs or we are not in realtime mode or the
	     * termination flag has been set, break out.  If we ARE in realtime
	     * mode and StreamPend is not 0, breakout.  But if StreamPend
	     * is 0 and we are in realtime mode, we block.
	     */

	    if (n < 0 || RealTimeOpt == 0 || StreamPend != 0 || TermFlag)
		break;
	    /*
	     * sleep and retry the read.  Effectively a tail -f.  When the file
	     * is renamed AND a new realtime batchfile exists, we switch to the
	     * new realtime batch and attempt to remove the renamed one that
	     * we still have a lock on.  This only works if the batch file
	     * is named after the label / sequence file.
	     *
	     * If the file is renamed, we stay with the file until a new 
	     * realtime batch is available.  
	     */
	    if (stat(CurrentBatchFile,&st) == 0 && st.st_ino != CurSt.st_ino) {
		/*
		 * do not attempt to remove the batchfile if we would have
		 * had to refile some of our entries.
		 */
		if (WouldHaveRefiled == 0)
		    AttemptRemoveRenamedFile(&CurSt, BatchFileCtl);
		break;
	    }
	    sleep(RealTimeOpt);
	}
	rbs->rb_Len += n;
    }
    return(-1);
}

int
ValidMsgId(char *msgid)
{
    int l = strlen(msgid);

    if (msgid[0] == '<' && msgid[l-1] == '>')
	return(1);
    return(0);
}

void
AttemptRemoveRenamedFile(struct stat *st, const char *spoolFile)
{
    char buf[256];
    int fd;
    int begSeq = -1;

    sprintf(buf, "%s/.%s.seq", OUTGOING, spoolFile);
    if ((fd = open(buf, O_RDONLY)) >= 0) {
	int n;
	if ((n = read(fd, buf, sizeof(buf) - 1)) > 0) {
	    buf[n] = 0;
	    sscanf(buf, "%d", &begSeq);
	}
	close(fd);
    }

    /*
     * Attempt to locate the renamed file.  It is no big deal if we can't
     * find it.  We still have our lock on the file, so we can safely remove
     * it.
     */

    if (begSeq >= 0) {
	int n;

	for (n = 3; n >= 0; --n, ++begSeq) {
	    struct stat nst;

	    sprintf(buf, "%s/%s.S%05d", OUTGOING, spoolFile, begSeq);
	    if (stat(buf, &nst) == 0 && st->st_ino == nst.st_ino) {
		remove(buf);
		break;
	    }
	}
    }
}

/*
 * CDMAP() - map part of a file.  If off & *psize are 0, we map the whole
 *	     file.
 */

#define MAXMCACHE	16

typedef struct XCache {
    char *mc_Path;
    int  mc_Fd;
    int  mc_Size;	/* only if off/psize are not known */
} XCache;

XCache	XCacheAry[MAXMCACHE];

char *
cdmap(const char *path, int off, int *psize, int *multiArtFile)
{
    int i;
    XCache *mc;
    char *ptr = NULL;

    *multiArtFile = 0;
    if (*psize || off) {
	*multiArtFile = 1;
    }

    for (i = 0; i < MAXMCACHE; ++i) {
	if (XCacheAry[i].mc_Path == NULL)
	    break;
	if (strcmp(path, XCacheAry[i].mc_Path) == 0)
	    break;
    }
    if (i == MAXMCACHE) {
	i = random() % MAXMCACHE;

	mc = &XCacheAry[i];
	close(mc->mc_Fd);
	free(mc->mc_Path);
	mc->mc_Fd = -1;
	mc->mc_Path = NULL;
	mc->mc_Size = 0;
    }
    mc = &XCacheAry[i];

    if (mc->mc_Path == NULL) {
	if ((mc->mc_Fd = cdopen(path, O_RDONLY, 0)) >= 0) {
	    struct stat st;

	    st.st_size = 0;
	    fstat(mc->mc_Fd, &st);
	    mc->mc_Size = st.st_size;
	    mc->mc_Path = strcpy(malloc(strlen(path) + 1), path);
	}
    }

    /*
     * When mapping multi-article files, a \0 terminator must also be mapped.
     * There is no \0 terminator for regular files.
     */

    if (mc->mc_Fd >= 0) {
	if (!*multiArtFile)
	    *psize = mc->mc_Size;

	ptr = xmap(NULL, *psize + *multiArtFile, PROT_READ, MAP_SHARED, mc->mc_Fd, off);
	if (*multiArtFile && ptr && ptr[*psize] != 0) {
	    syslog(LOG_CRIT, "article batch corrupted: %s @ %d,%d", path, off, *psize);
	    xunmap(ptr, *psize + *multiArtFile);
	    ptr = NULL;
	}
    }
    return(ptr);
}

void
cdunmap(char *ptr, int bytes, int multiArtFile)
{
    xunmap((caddr_t)ptr, bytes + multiArtFile);
}

#define ISWHITE(c)	((c)=='\n' || (c) == '\r' || (c) == ' ' || (c) == '\t')

int
extractFeedLine(const char *buf, char *relPath, char *msgId, int *poff, int *psize)
{
    const char *s = buf;
    const char *b;
    int l1 = 0;
    int l2 = 0;

    /*
     * RELPATH
     */
    for (b = s; *s && !ISWHITE(*s); ++s)
	;
    l1 = s - b;
    bcopy(b, relPath, l1);
    relPath[l1] = 0;

    while (ISWHITE(*s))
	++s;

    /*
     * MSGID
     */

    if (*s == '<') {
	for (b = s; *s && *s != '>'; ++s)
	    ;
	if (*s == '>') {
	    ++s;
	    l2 = s - b;
	    bcopy(b, msgId, l2);
	    msgId[l2] = 0;
	}
	while (ISWHITE(*s))
	    ++s;
    }

    /*
     * offset/size
     */

    b = s;
    if (isdigit(*s)) {
	sscanf(s, "%d,%d", poff, psize);
    }
    if (l1 && l2 && ValidMsgId(msgId))
	return(0);
    return(-1);
}

const char *
tstamp(void)
{
    struct timeval tv;
    static struct timeval stv;
    static char buf[64];
    int dt = 0;

    gettimeofday(&tv, NULL);
    if (stv.tv_sec) {
	dt = (tv.tv_sec - stv.tv_sec - 1) * 1000 + (tv.tv_usec + 1000000 - stv.tv_usec) / 1000;
    }
    stv = tv;
    sprintf(buf, "%4d.%03d ", dt / 1000, dt % 1000);
    return(buf);
}

