;;;
;;; tm-image.el --- a tm-view content filter to display images in
;;;                 XEmacs or Mule buffers
;;;
;;; Copyright (C) 1995,1996 MORIOKA Tomohiko
;;;
;;; Author: MORIOKA Tomohiko <morioka@jaist.ac.jp>
;;; Maintainer: MORIOKA Tomohiko <morioka@jaist.ac.jp>
;;; Created: 1995/12/15
;;; Version:
;;;	$Id: tm-image.el,v 4.6 1996/02/13 17:56:43 morioka Exp $
;;;
;;; Keywords: mail, news, MIME, multimedia, image, picture
;;;
;;; This file is part of tm (Tools for MIME).
;;;
;;; This program is free software; you can redistribute it and/or
;;; modify it under the terms of the GNU General Public License as
;;; published by the Free Software Foundation; either version 2, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with This program.  If not, write to the Free Software
;;; Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
;;;
;;; Commentary:
;;;   If you use this program with Mule, please install bitmap
;;;   extension package. It is available from
;;;	ftp://etlport.etl.go.jp/pub/mule/contrib/bitmap.tar.gz
;;;
;;; Code:

(require 'tm-view)

(cond ((string-match "XEmacs" emacs-version)
       (require 'xpm)
       (require 'annotations)
       
       (set-alist 'mime-viewer/content-filter-alist
		  "image/jpeg" (function mime-preview/filter-for-image))
       
       (set-alist 'mime-viewer/content-filter-alist
		  "image/gif" (function mime-preview/filter-for-image))
       
       (set-alist 'mime-viewer/content-filter-alist
		  "image/tiff" (function mime-preview/filter-for-image))
       (set-alist 'mime-viewer/content-filter-alist
		  "image/x-tiff" (function mime-preview/filter-for-image))
       
       (set-alist 'mime-viewer/content-filter-alist
		  "image/x-pic" (function mime-preview/filter-for-image))
       
       (set-alist 'mime-viewer/content-filter-alist
		  "image/x-mag" (function mime-preview/filter-for-image))
       
       (defvar tm-image/inline-image-types
	 '("image/jpeg" "image/gif" "image/tiff"
	   "image/x-tiff" "image/x-pic" "image/x-mag" "image/x-xbm"))

       (defun bitmap-read-xbm (file)
	 (let (gl)
	   (while (progn
		    (setq gl (make-glyph file))
		    (eq (image-instance-type (glyph-image-instance gl))
			'text)
		    ))
	   (make-annotation gl (point) 'text)
	   ))
       
       (defvar mime-preview/x-face-function
	 (function mime-preview/x-face-function-for-xemacs))
       )
      ((boundp 'MULE)
       (require 'bitmap)
       
       (defvar tm-image/inline-image-types '("image/x-mag" "image/x-xbm"))
       
       (defvar mime-preview/x-face-function
	 (function mime-preview/x-face-function-for-mule))
       ))

(defvar mime-viewer/shell-command "/bin/sh")
(defvar mime-viewer/shell-arguments '("-c"))

(defvar mime-viewer/graphic-converter-alist
  '(("image/jpeg"   . "djpeg -color 256 < %s | ppmtoxpm > %s")
    ("image/gif"    . "giftopnm < %s | ppmtoxpm > %s")
    ("image/tiff"   . "tifftopnm < %s | ppmquant 256 | ppmtoxpm > %s")
    ("image/x-tiff" . "tifftopnm < %s | ppmquant 256 | ppmtoxpm > %s")
    ("image/x-pic"  . "pictoppm < %s | ppmquant 256 | ppmtoxpm > %s")
    ("image/x-mag"  . "magtoppm < %s | ppmtoxpm > %s")
    ))

(defvar mime-viewer/x-face-to-xbm-command
  (concat mime-viewer/x-face-to-pbm-command " | pbmtoxbm"))

(add-hook 'mime-viewer/content-header-filter-hook
	  mime-preview/x-face-function)

(defun mime-preview/x-face-function-for-xemacs ()
  (save-restriction
    (narrow-to-region (point-min) (re-search-forward "^$" nil t))
    (goto-char (point-min))
    (if (re-search-forward "^X-Face:[ \t]*" nil t)
	(let ((beg (match-end 0))
	      (end (rfc822/field-end))
	      (xbm-file
	       (make-temp-name (expand-file-name "tmxf" mime/tmp-dir))
	       ))
	  (call-process-region
	   beg end "sh" nil 0 nil
	   "-c"
	   (format "%s > %s"
		   mime-viewer/x-face-to-xbm-command
		   xbm-file))
	  (while (not (file-exists-p xbm-file)))
	  (delete-region beg end)
	  (bitmap-read-xbm xbm-file)
	  (condition-case nil
	      (delete-file xbm-file)
	    (error nil))
	  ))))

(defun mime-preview/x-face-function-for-mule ()
  (save-restriction
    (narrow-to-region (point-min) (re-search-forward "^$" nil t))
    (goto-char (point-min))
    (if (re-search-forward "^X-Face:[ \t]*" nil t)
	(let ((p (match-beginning 0))
	      (beg (match-end 0))
	      (end (rfc822/field-end))
	      (xbm-file
	       (concat
		(make-temp-name (expand-file-name "tmxf" mime/tmp-dir))
		".xbm")
	       ))
	  (if (< end (point-max))
	      (setq end (1+ end))
	    )
	  (while (progn
		   (call-process-region
		    beg end "sh" nil 0 nil
		    "-c"
		    (format "%s > %s"
			    mime-viewer/x-face-to-xbm-command
			    xbm-file))
		   (not (file-exists-p xbm-file))
		   ))
	  (save-restriction
	    (narrow-to-region p end)
	    (delete-region p end)
	    (goto-char p)
	    (while (progn
		     (condition-case nil
			 (bitmap-read-xbm xbm-file)
		       (error nil))
		     (kill-buffer 
		      (some-element
		       (function
			(lambda (buf)
			  (string-equal (buffer-file-name buf) xbm-file)
			  ))
		       (buffer-list)))
		     (= (point-min) (point-max))
		     ))
	    (goto-char p)
	    (insert "X-Face: ")
	    (while (re-search-forward "^." nil t)
	      (goto-char (match-beginning 0))
	      (insert "        ")
	      ))
	  (condition-case nil
	      (delete-file xbm-file)
	    (error nil))
	  ))))


;;; @ content filter for images
;;;

(defun mime-preview/filter-for-image (ctype params encoding)
  (let* ((mode mime::preview/original-major-mode)
	 (m (assq mode mime-viewer/code-converter-alist))
	 (filter (assoc-value ctype mime-viewer/graphic-converter-alist))
	 )
    (if filter
	(let* ((beg (point-min)) (end (point-max))
	       (orig-file
		(make-temp-name (expand-file-name "tm" mime/tmp-dir)))
	       (xbm-file (concat orig-file ".xbm"))
	       gl annot)
	  ;;(remove-text-properties beg end '(face nil))
	  (mime/decode-region encoding beg end)
	  (write-region (point-min)(point-max) orig-file)
	  (delete-region (point-min)(point-max))
	  (message "Now translating, please wait...")
	  (apply (function call-process)
		 mime-viewer/shell-command nil nil nil
		 (append mime-viewer/shell-arguments
			 (list (format filter orig-file xbm-file)))
		 )
	  (setq gl (make-glyph xbm-file))
	  (setq annot (make-annotation gl (point) 'text))
	  (unwind-protect
	      (delete-file orig-file)
	    (condition-case nil
		(delete-file xbm-file)
	      (error nil)))
	  (goto-char (point-max))
	  (insert "\n")
	  (message "Translation done.")
	  )
      (message (format "%s is not supported." ctype))
      )))


;;; @ content filter for xbm
;;;

(defun mime-preview/filter-for-image/xbm (ctype params encoding)
  (let* ((mode mime::preview/original-major-mode)
	 (m (assq mode mime-viewer/code-converter-alist))
	 (charset (assoc "charset" params))
	 (beg (point-min)) (end (point-max))
	 (xbm-file (make-temp-name (expand-file-name "tm" mime/tmp-dir)))
	 )
    (remove-text-properties beg end '(face nil))
    (mime/decode-region encoding beg end)
    (write-region (point-min)(point-max) xbm-file)
    (delete-region (point-min)(point-max))
    (bitmap-read-xbm xbm-file)
    (delete-file xbm-file)
    ))

(set-alist 'mime-viewer/content-filter-alist
	   "image/xbm" (function mime-preview/filter-for-image/xbm))

(set-alist 'mime-viewer/content-filter-alist
	   "image/x-xbm" (function mime-preview/filter-for-image/xbm))


;;; @ setting
;;;

(mapcar
 (lambda (ctype)
   (or (member ctype mime-viewer/default-showing-Content-Type-list)
       (setq mime-viewer/default-showing-Content-Type-list
	     (cons ctype
		   mime-viewer/default-showing-Content-Type-list))
       ))
 tm-image/inline-image-types)


;;; @ end
;;;

(provide 'tm-image)

;;; tm-image.el ends here
