;; ---------------------------------------------------------------------- ;;
;! @file     output.scm                                                   !;
;! @created  Fri Jun 20 13:43:15 1997                                     !;
;! @modified Thu Mar  5 15:33:24 1998                                     !;
;; ---------------------------------------------------------------------- ;;
;! @copyright Dominique Boucher                                           !;
;; ---------------------------------------------------------------------- ;;
;; HTML output functions                                                  ;;
;; ---------------------------------------------------------------------- ;;

(module output

	(export 
	 (output-file-header port)
	 (output-file-footer port)
	 (output-section-header port str)
	 (output-text port lst)
	 (output-variable port name docs)
	 (output-function port name args docs)
	 (output-macro port name args docs)
	 (output-structure port name fields docs))

	(import
	 (globals    "globals.scm")
	 (utils      "utils.scm")
	 (predicates "preds.scm")
	 (format     "format.scm"))

	(extern
	 (include "time.c")
	 (time-string::string () "scmGetTimeAndDate")))

(define *index* '())


;; ---------------------------------------------------------------------- ;;
;; Output the file header                                                 ;;
;; ---------------------------------------------------------------------- ;;
(define (output-file-header oport)
  (format oport "<H1 ALIGN=center>File <CODE>~a</CODE></H1>~%" 
	  (or file "Untitled"))
  (format oport "<TABLE BORDER=0 WIDTH=\"100%\">~%")
  (format oport "<TR><TD BGCOLOR=#999999 WIDTH=\"20%\"><B>VERSION</B></TD>")
  (format oport "    <TD BGCOLOR=#CCCCCC>~a</TD></TR>~%" (or version "unknown"))
  (format oport "<TR><TD BGCOLOR=#999999><B>CREATED</B></TD>")
  (format oport "    <TD BGCOLOR=#CCCCCC>~a</TD></TR>~%" (or created "-"))
  (format oport "<TR><TD BGCOLOR=#999999><B>MODIFIED</B></TD>")
  (format oport "    <TD BGCOLOR=#CCCCCC>~a</TD></TR>~%" (or modified "-"))
  (format oport "<TR><TD BGCOLOR=#999999><B>COPYRIGHT</B></TD>")
  (format oport "    <TD BGCOLOR=#CCCCCC>~a</TD></TR>~%" (or copyright "public domain"))
  (format oport "</TABLE>~%~%")
  (if (and included (file-exists? included))
      (begin
	(format oport "<BR>~%")
	(call-with-input-file included
	  (lambda (port)
	    (let loop ((c (read-char port)))
	      (if (not (eof-object? c))
		  (begin (write-char c oport) (loop (read-char port)))))))
	(format oport "~%")))

  (if overview            ;***
      (output-text oport overview))
    
  (format oport "~%~%<HR>~%~%")
  (format oport "<H1 ALIGN=CENTER>INTERFACE</H1>~%~%"))

;; ---------------------------------------------------------------------- ;;
;; Output the file footer                                                 ;;
;; ---------------------------------------------------------------------- ;;
(define (output-file-footer oport)
  (format oport "~%~%<HR>~%~%")
  (output-section-header oport "INDEX")
  (format oport "<UL>~%")
  (set! *index* 
	(sort *index* (lambda (elt1 elt2) (string<? (car elt1) (car elt2)))))
  (for-each 
   (lambda (descr)
     (format oport "<LI> ")
     (output-generic-xref oport (car descr) (cadr descr)))
   *index*)
  (format oport "</UL>~%")
  
  (format oport
	  "<HR>~%File created automatically by ~a on ~a~%"
	  progname
	  (time-string)))


(define (output-section-header port str)
  (format port "<H2 ALIGN=center>~a</H2>~%~%" str))

(define (output-text oport lst)        ;***
  (format oport "~%<BR>~%")
  (for-each
    (lambda (str) (format oport "~a~%" str))
    lst)
  (format oport "~%<BR>~%"))

;; ---------------------------------------------------------------------- ;;
;; Output a variable definition                                           ;;
;; ---------------------------------------------------------------------- ;;
(define (output-variable port name docs)
  (let ((fn-doc (filter function-comment? docs)))
    (if (pair? fn-doc)
        (let* ((str   (cadr (car fn-doc)))
               (proto (let* ((port (open-input-string str))
			     (exp  (read port)))
			(close-input-port port)
			exp)))
          (output-function port (car proto) (cdr proto) docs))
        
        (begin
          (output-generic-header port name "Variable")
          (let ((descr-doc (filter descr-comment? docs)))
            (if (pair? descr-doc)
                (begin
                  (format port "<LI><B>Description:</B>~%")
                  (for-each
                    (lambda (str) (format port "~a~%" str))
                    (cdr (car descr-doc))))))
          (output-generic-footer port)))))

  
;; ---------------------------------------------------------------------- ;;
;; Output a function definition                                           ;;
;; ---------------------------------------------------------------------- ;;
(define (output-function port name args docs)
  (output-generic-header port name "Function")
  (output-proto port name args)
  (output-descriptions port docs param-comment? "Arguments")
  (output-return-val port (filter return-comment? docs))
  (output-generic-footer port))

;; ---------------------------------------------------------------------- ;;
;; Output a macro definition                                              ;;
;; ---------------------------------------------------------------------- ;;
(define (output-macro port name args docs)
  (output-generic-header port name "Macro")
  (output-proto port name args)
  (output-descriptions port docs param-comment? "Arguments")
  (output-generic-footer port))

;; ---------------------------------------------------------------------- ;;
;; Output a structure definition                                          ;;
;; ---------------------------------------------------------------------- ;;
(define (output-structure port name fields docs)
  (output-generic-header port name "Structure")
  (let ((flds (flatten fields))
	(creator (string-append "make-" (symbol->string name))))
    (output-proto port creator flds)
    (output-descriptions port docs field-comment? "Fields"))
  (output-generic-footer port))

;; ---------------------------------------------------------------------- ;;
;; Output a general item header                                           ;;
;; ---------------------------------------------------------------------- ;;
(define (output-generic-header port name type)
  (format port "<H3>~a <A NAME=\"~a\"><CODE>~a</CODE></A></H3>~%~%" 
	  type name name)
  (set! *index* 
	(cons (list (symbol->string name) type) *index*))
  (format port "<UL>~%"))

(define (output-generic-xref port name type)
  (format port "<CODE><A HREF=\"#~a\">~a</A></CODE>: ~a~%" 
	  name name type))

;; ---------------------------------------------------------------------- ;;
;; Output a generic item footer                                           ;;
;; ---------------------------------------------------------------------- ;;
(define (output-generic-footer port)
  (format port "</UL>~%")
  (format port "~%~%"))

;; ---------------------------------------------------------------------- ;;
;; Output parameters descriptions                                         ;;
;; ---------------------------------------------------------------------- ;;
(define (output-descriptions port docs type-test type-str)
  
  (let ((descr-doc (filter descr-comment? docs)))
    (if (pair? descr-doc)
	(begin
	  (format port "<LI><B>Description:</B>~%")
	  (for-each 
	   (lambda (str) (format port "~a~%" str))
	   (cdr (car descr-doc))))))
  
  (let ((args-doc (filter type-test docs)))
    (if (pair? args-doc)
	(begin
	  (format port "<LI><B>~a:</B>~%" type-str)
	
	  (format port "<UL>~%")
	  (let loop ((l args-doc))
	    (if (pair? l)
		(let ((arg (car l)))
		  (format port "<LI><CODE>~a</CODE> : ~%" (cadr arg))
		  (for-each
		   (lambda (str)
		     (format port "   ~a~%" str))
		   (cddr arg))
		  (loop (cdr l)))))
	  (format port "</UL>~%")))))
  
  
;; ---------------------------------------------------------------------- ;;
;; Output a the function/macro prototype                                  ;;
;; ---------------------------------------------------------------------- ;;
(define (output-proto port name args)
  (format port "<LI><B>Prototype:</B> ~%")
  (format port "<CODE>(~a" name)
  (let loop ((l args))
    (cond
     ((null? l)
      (format port ")"))
     ((pair? l)
      (let ((arg (car l)))
	(format port " ~a" arg))
      (loop (cdr l)))
     (else
      (format port " . ~a)" l))))
  (format port "</CODE>~%"))
  
;; ---------------------------------------------------------------------- ;;
;; Output the returned value                                              ;;
;; ---------------------------------------------------------------------- ;;
(define (output-return-val port doc)
  (if (pair? doc)
      (begin
        (format port "<LI><B>Return:</B>~%")
        (for-each
          (lambda (str) (format port "~a~%" str))
          (cdr (car doc))))))

;; ---------------------------------------------------------------------- ;;
;; Return the current date and time in a string                           ;;
;; ---------------------------------------------------------------------- ;;
;; (define (time-string)                                                  ;;
;;   (pragma::string "({time_t ___tmp; time(&___tmp); ctime(&___tmp);});")) ;;
