/*
 * int13.S		
 * 
 * Interface to BIOS int 13h functions.
 *
 * Copyright (c) Tuomo Valkonen 1996-1997.
 */

#ifdef STAGE2
#include "debug.S"
#endif

// Read sectors at linear disk address
//  Input:
//	AX,DX = linear address ( sectors )
//      ES:BX = where to load
//      CL    = drive
////////////////////////////////////////////
read_linear:
	pusha

#ifdef STAGE2
// Check for hole and end-of-file
///////////////////////////////////
	cmp	dx,#MF_NULLEND_HI
	jne	rl_not_hole
	cmp	ax,#MF_END_LOW
	jne	rl_not_end
	popa
	stc			// Carry-set means end-of-file
	ret
rl_not_end:
	cmp	ax,#MF_NULL_LOW
	jne	rl_not_hole
	cld
	mov	di,bx
	mov	cx,#SECTORSIZE
	xor	al,al
	rep
	stosb
	popa
	clc	// make sure...
	ret
rl_not_hole:
#endif
	
	// Get geometry if last drive used wasn't this one
	////////////////////////////////////////////////////
	cmp	byte ptr last_drive,cl
	je	do_lin_read
	
	mov	byte ptr last_drive,cl
	
	call	get_geometry

	popa	// Restore registers
	pusha

do_lin_read:
	
	push	ax
	push	dx
	push	cx			! Save stuff
	push	bx
#if 0
	// May cause divide overflow on high addresses
	// (linear sector address>sects per cyl*65535)
	mov	cx,tmp_sects		! Count sector
	div	cx
	inc	dx
	mov	cx,dx

	mov	bl,tmp_heads		! Count head & cylinder
	xor	dx,dx
	xor	bh,bh  
	div	bx

#else
	// Calculate sector
	rol	eax,#16			! ax:dx -> eax
	mov	ax,dx
	rol	eax,#16
	xor	edx,edx
	xor	ecx,ecx
	mov	cx,tmp_sects
	div	ecx
	inc	dx			! sector = 1 to sect/cyl
	mov	cx,dx			! sect/cyl<256, so it fits in dx
	
	// Calculate cylinder and head
	xor	edx,edx
	xor	ebx,ebx
	mov	bl,tmp_heads
	div	ebx
	
	// cylinder number should fit in ax and head in dx, so
	// we don't mind the high words
#endif

	mov	dh,dl
#if CHECK_CYLINDER
	cmp	ax,tmp_cyls	// Check it is in limits
#else
	cmp	ax,#1023	// Check it is below max.
#endif
	jna	cyl_ok
	br	lin_cyl_exceed

cyl_ok:	xchg	al,ah
	shl	al,#6
	or	cx,ax
	pop	bx		! bx
	pop	ax		! cx - drive
	mov	dl,al
	mov	al,#1
	mov	ah,#INT13H_LOAD
	push	es
	pusha
	int	0x13
	jnc	lin_load_ok
	jmp	retry_read
lin_load_ok:
	// Carry already cleared by int13h 
	popa
	pop	es

lin_load_done:
	pop	dx
	pop	ax
	popa
	ret

// Retry read at most 3 times
///////////////////////////////
retry_read:
	call	reset_drive
	popa
	pusha
	int	#0x13
	jnc	retry_read_ok
	dec	byte ptr retry_cnt
	jnz	retry_read
	br	lin_error
retry_read_ok:
	mov	byte ptr retry_cnt,#3
	jmp	lin_load_ok
	

// Reset drive (DL)
/////////////////////
reset_drive:
	mov	ah,#INT13H_RESET
	int	#0x13
	jnc	reset_ok
	br	lin_error
reset_ok:
	ret


// Get drive geometry (called from read_linear and write_linear)
//////////////////////////////////////////////////////////////////
get_geometry:
	push	es
	mov	dl,cl
	mov	ah,#INT13H_GEO
	int	#0x13
	jnc	cont_getgeo
	br	lin_error
cont_getgeo:	
	inc	dh
	mov	tmp_heads,dh
#if CHECK_CYLINDER	
	push	cx
	xchg	ch,cl
	shr	ch,#6
	mov	tmp_cyls,cx
	pop	cx
#endif	
	and	cx,#0x3f
	mov	tmp_sects,cx
	pop	es

	ret

#ifdef STAGE2

// Write sectors at linear disk address
//  Input:
//	AX,DX = linear address ( sectors )
//      ES:BX = source address
//      CL    = drive
////////////////////////////////////////////
write_linear:
	pusha

	cmp	byte ptr last_drive,cl
	je	do_lin_write
	
	mov	byte ptr last_drive,cl
	
	call	get_geometry

	popa	// Restore registers
	pusha

do_lin_write:
	
	push	ax
	push	dx
	push	cx			! Save stuff
	push	bx

// Calculate sector
/////////////////////
	rol	eax,#16			! ax:dx -> eax
	mov	ax,dx
	rol	eax,#16
	xor	edx,edx
	xor	ecx,ecx
	mov	cx,tmp_sects
	div	ecx
	inc	dx			! sector = 1 to sect/cyl
	mov	cx,dx			! sect/cyl<256, so it fits in dx
	
	// Calculate cylinder and head
	xor	edx,edx
	xor	ebx,ebx
	mov	bl,tmp_heads
	div	ebx
	
	// cylinder number should fit in ax and head in dx, so
	// we don't mind the high words

	mov	dh,dl
#if CHECK_CYLINDER
	cmp	ax,tmp_cyls	// Check it is in limits
#else
	cmp	ax,#1023	// Check it is below max.
#endif
	jna	lw_cyl_ok
	br	lin_cyl_exceed

lw_cyl_ok: xchg	al,ah
	shl	al,#6
	or	cx,ax
	pop	bx		! bx
	pop	ax		! cx - drive
	mov	dl,al		! drive number into DL
	mov	al,#1
	mov	ah,#INT13H_WRITE
	push	es
	pusha
	int	0x13
	jnc	lin_write_ok
	br	lin_error	// retry_read
lin_write_ok:
	// Carry already cleared by int13h 
	popa
	pop	es

lin_write_done:
	pop	dx
	pop	ax
	popa
	ret
#endif /* STAGE2 */

// Data
/////////

tmp_sects:	.word	0		! Sectors per Track
tmp_cyls:	.word	0		! Number of Cylinders
tmp_heads:	.byte	0		! Number of Heads
last_drive:	.byte	0xff
retry_cnt:	.byte 3
