/*
 * gom_file.c: loading and saving files
 */

/* 
 * This file is part of the package
 *
 * gom, Gom is nOt yet another Mixer
 *
 * (c) Stephan Suerken <suerken@fh-worms.de> 1996, 1997
 */

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * INCLUDES
 */
#include "gom.h"

/*
 * MACROS
 */

#define GOM_FILE_OPT_SIZE         100
#define GOM_FILE_OPT_SIZE_QUOTED "100"

/*
 * DECLARATIONS/DEFINTIONS
 */

char gom_file_option[GOM_FILE_OPT_SIZE];

/*
 * FUNCTION PROTOTYPES
 */

/*****************/

/*
 * INTERN HELPERS
 */

char * 
gom_file_fncat(const char * ct)
{
  if (strlen(ct) > FILENAME_MAX-strlen(gom_file_fn)-1)
    gom_info(GOM_INFO_ERROR, "Bug: File name too big (>%i characters).", FILENAME_MAX-1);
  return strncat(gom_file_fn, ct, FILENAME_MAX-strlen(gom_file_fn)-1);
}

int
gom_file_systemconf()
{
  return (geteuid() == 0) || (getuid() == 0);
}

int 
gom_file_fn_make(enum gom_info_types verb,
		 const char * fname)
{
  gom_file_fn[0] = '\0';
  if (fname != NULL && (fname[0] == '/' || (fname[0] == '.' && fname[1] == '/')))
    {
      /* /dir/file or ./dir/file: no overwrite */
      gom_file_fncat(fname);
      return 0;
    }
  else                /* filename relative to goms config dir; overwrite */
    {
      if (gom_file_systemconf())
	{
	  /* superuser privileges: write to system directory */
	  gom_file_fncat(GOM_CONFIG_DIR_SYSTEM);
	}
      else if (getenv("HOME") != NULL)
	{
	  /* normal user; write to ~/.gom */
	  gom_file_fncat(getenv("HOME"));
	  gom_file_fncat("/" GOM_CONFIG_DIR_USER);
	}
      else
        {
	  gom_info(GOM_INFO_QUIET, "Warning: No environment variable HOME found (using current directory).");
	  gom_file_fncat("./" GOM_CONFIG_DIR_USER);
	}
      gom_file_fncat((fname == NULL) || (strlen(fname) == 0) ? GOM_OPTIONS_FILE_DEFAULT : fname);
      return 1;           /* ok to overwrite */
    }
}


/*
 * LOADING
 */


/* intern */
char *
gom_file_optarg_get()
{
  static char nullstr[] = "";

  if (strlen(gom_file_option) < 2)
    /* gom_file_option == "d" or worse */
    return nullstr;
  else
    /* seems to be valid */
    return (char *) &gom_file_option[1];
}


char
gom_file_fprereadc(FILE * f)
{
  char c;
  if (1 == fscanf(f, "%c", &c))
    fseek(f, -1L, SEEK_CUR);
  else
    c = EOF;
  return c;
}


/* loading options from stream: */
void 
gom_file_options_read(enum gom_info_types std_verb,
		      FILE * f, int action_ignore)
{
  char ind, action;
  int count, line = 1, line_valid;

  do
    {
      /* valid lines MUST start with a dash; pre-read first char of line to check */
      line_valid = ((ind = gom_file_fprereadc(f)) == '-');

      if (line_valid)
	{
	  gom_info(std_verb+2, "Line %d: significant, processing...", line);
	  
	  while (1 == (count = fscanf(f,
				      "-%" GOM_FILE_OPT_SIZE_QUOTED "s",
				      (char *) &gom_file_option)))
	    {
	      /* validity check? */

	      action = gom_file_option[0];

	      gom_info(std_verb+2, "Line %d: Option found: %s", line, gom_file_option);

	      if (action != action_ignore)
		gom_action(std_verb, action, 1, gom_file_optarg_get, NULL, NULL);
	      else
		gom_info(std_verb,
			 "Option \"-%s\" ignored.",
			 &gom_file_option);
	      
	      /* a new following space indicates more options */
	      if (isspace(ind = gom_file_fprereadc(f)) && (ind != '\n'))
		/* trennzeichen, aber kein newline */
		fseek(f, +1L, SEEK_CUR);
	      else
		/* kein trennzeichen, evt. newline */
		break;
	    }
	}
      else
	{
	  /* this is only for debugging config files... */
	  if (ind == '#')
	    gom_info(std_verb+2, "Line %d: comment, skipping...", line);
	  else if (ind == '\n')
	    gom_info(std_verb+2, "Line %d: empty, skipping...", line);
	  else
	    gom_info(std_verb+2, "Line %d: ambigious, skipping...", line);
	}
      
      /* skipping to the end of this line */
      while ((1 == fscanf(f, "%c", &ind)) && (ind != EOF) && (ind != '\n'))
	{
	  if (line_valid)
	    gom_info(std_verb+2, "Line %d: Skipping suspicious trailing garbage (%c).", line, ind);
	};

      line++;
    }
  while (feof(f) == 0);
}

/* loading options from file: */
void 
gom_file_options_load(enum gom_info_types std_verb,
		      char * fname,
		      enum gom_info_types open_err_verb, int action_ignore)
{
  gom_file_fn_make(std_verb, fname);
  if ((gom_file_f = fopen(gom_file_fn, "r")) != NULL)
    {
      gom_info(std_verb, "Loading option file from \"%s\"...", gom_file_fn);
      gom_file_options_read(std_verb+1, gom_file_f, action_ignore);
      gom_info(std_verb, "Option file load from \"%s\" completed.", gom_file_fn);
      
      /* ok, no error check here... */
      fclose(gom_file_f);
    }
  else
    {
      gom_info(open_err_verb,
	       "Can't open file \"%s\" for reading: %s.",
	       gom_file_fn, strerror(errno));
    }
}

/* loading gom's configuration: */
/* - either loads user, system or no config */
/* - no errors */
void
gom_file_config_load(enum gom_info_types verb, int action_ignore)
{
  /* check if user config file exists */
  gom_file_fn_make (verb, GOM_CONFIG_FILE);
  if (access (gom_file_fn, F_OK) == 0)
    {
      /* try to load user config */
      gom_info(verb+1, "User configuration file (%s) found.", gom_file_fn);
      gom_file_options_load (verb, GOM_CONFIG_FILE, verb, action_ignore);
    }
  else
    {
      gom_file_fn_make (verb, GOM_CONFIG_DIR_SYSTEM GOM_CONFIG_FILE);
      if (access (gom_file_fn, F_OK) == 0)
	{
	  /* try to load system config */
	  gom_info(verb+1, "System configuration file (%s) found.", gom_file_fn);
	  gom_file_options_load (verb, GOM_CONFIG_DIR_SYSTEM GOM_CONFIG_FILE, verb, action_ignore);
	}
      else
	{
	  gom_info(verb,
		   "Neither user (~/" GOM_CONFIG_DIR_USER GOM_CONFIG_FILE ") nor system ("
		   GOM_CONFIG_DIR_SYSTEM GOM_CONFIG_FILE ") configuration file found.");
	}

    }
}


/*
 * SAVING
 */

/* intern */
int
gom_file_open_save(enum gom_info_types verb,
		   const char * file)
{
  int overwrite;


  overwrite = gom_file_fn_make(verb, file);
  
  /* NEVER write the config file as setting */
  if ((file != NULL) && (strncmp(file, GOM_CONFIG_FILE, strlen(GOM_CONFIG_FILE)) == 0))
    {
      gom_info(GOM_INFO_ERROR,
	       "I won't save files starting with \"" GOM_CONFIG_FILE "\" (like \"%s\") to the configuration dir.",
	       gom_file_fn);
    }
  /* try open file */
  else if (!overwrite && (gom_file_f = fopen(gom_file_fn, "r")) != NULL)  /* open absolute given file */
    {
      fclose(gom_file_f); gom_file_f = NULL;
      gom_info(GOM_INFO_ERROR,
	       "File \"%s\" exists (I refuse to overwrite files elsewhere than in $HOME/.gom).",
	       gom_file_fn);
    }
  else if ((gom_file_f = fopen(gom_file_fn, "w")) == NULL)  /* open relative given file */
    {
      gom_info(GOM_INFO_ERROR,
	       "Can't open file \"%s\" for save: %s.",
	       gom_file_fn, strerror(errno));     
    }
  else
    {
      /* ok to overwrite */
      if (gom_file_systemconf())
	gom_info(GOM_INFO_QUIET, "Warning: Running as root, writing to \"" GOM_CONFIG_DIR_SYSTEM "\".");
      return 1;
    }
  return 0;
}

/* intern helps to stream */
int
gom_file_stropt_write(FILE * f, char option, char * arg)
{
  return fprintf(f, "-%c%s ", option, arg);
}

int 
gom_file_intopt_write(FILE * f, char option, int arg)
{
  return fprintf(f, "-%c%i ", option, arg);
}

/* write ONE CHANNEL */
void 
gom_file_options_c_write(FILE * f, int c)
{
  int cc;

  /* select channel, write record */
  gom_file_intopt_write(f, 'c', c);
  if (gom_driver_c_r(c) > -1)
    gom_file_intopt_write(f, 'r', gom_driver_c_r(c));
  
  /* unlock channel to get the different volumes right when *loading* */
  gom_file_intopt_write(f, 'k', 0);
  
  /* write all channel volumes */
  for (cc=gom_driver_c_c_first(c); cc > -1; cc=gom_driver_c_c_next(c, cc, 1))
    {
      gom_file_intopt_write(f, 'C', cc);
      gom_file_intopt_write(f, 'l', gom_driver_c_c_v(c, cc));
    }
  /* _lock_ channel as bulls neck default when *loading* */
  gom_file_intopt_write(f, 'k', 1);
}

/* settings file to stream */
void 
gom_file_settings_write(FILE * f, int verbose)
{
  int c;

  if (verbose)
    {
      fprintf(f,
	      "# This is a mixer settings file, automatically produced by " GOM_VERSION_STRING ".\n"
	      "# Lines starting with \"-\" are read (must be one-letter style gom options).\n"
	      "# Other lines are comments.\n"
	      "#\n"
	      "# Channels with recording source == 1 first (avoids \"last rec. source error\" when loading):\n");
    }

  for (c=gom_driver_c_first(); c > -1; c=gom_driver_c_next(c, 1))
    {
      if (gom_driver_c_r(c) > 0)
	{
	  gom_file_options_c_write(f, c);
	  if (verbose)
	    fprintf(f, "\n");
	}
    }
  
  if (verbose)
    fprintf(f,
	    "# Other channels:\n");
  for (c=gom_driver_c_first(); c > -1; c=gom_driver_c_next(c, 1))
    {
      if (gom_driver_c_r(c) <= 0)
	{
	  gom_file_options_c_write(f, c);
	  if (verbose)
	    fprintf(f, "\n");
	}
    }
  if (verbose)
    fprintf(f,
	    "# Resetting current channel to first.\n");
  gom_file_intopt_write(f, 'c', gom_driver_c_first());
}

/* settings file to file: */
void 
gom_file_settings_save(enum gom_info_types std_verb,
		       char * fname)
{
  /* try open file */
  if (gom_file_open_save(std_verb, fname))
    {
      /* write a comment header */
      gom_file_settings_write(gom_file_f, 1);

      if (fclose(gom_file_f) == 0)
	gom_info(std_verb, "Settings saved to \"%s\".", gom_file_fn);
      else
	gom_info(GOM_INFO_ERROR, "Problems closing file \"%s\": %s.",
		 gom_file_fn, strerror(errno));
    }
}
