/* filename: rlpr-util.c
 * project: rlpr
 * author: meem  --  meem@sherilyn.wustl.edu
 * version: $Id: rlpr-util.c,v 1.1 1997/02/22 07:19:17 meem Exp meem $
 * contents: useful general-purpose rlpr functions
 *
 * Time-stamp: <1997/04/27 14:44:11 cdt -- meem@sherilyn.wustl.edu>
 */

/* copyright (c) 1996, 1997 meem, meem@gnu.ai.mit.edu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 */

#include "config.h"

#include <sys/types.h>                /* for off_t           */
#include <stdio.h>
#include <unistd.h>		      /* for geteuid(), ..   */
#include <sys/stat.h>                 /* for struct stat     */
#include <sys/utsname.h>              /* for uname()         */
#include <fcntl.h>                    /* for fcntl()         */
#include <string.h>                   /* for memcpy(), ..    */
#include <ctype.h>                    /* for tolower()       */
#include <stdlib.h>                   /* for malloc()        */
#include <netdb.h>                    /* for gethostbyname() */

#include "rlpr-util.h"

/* modified slightly from Stevens, "UNIX Network Programming", p. 279 */

int writen(int fd, const char *ptr, int nbytes) {
  int nleft, nwritten;

  nleft = nbytes;
  while (nleft > 0) {
    nwritten = write(fd, ptr, nleft);
    if (nwritten <= 0) return nwritten;
    nleft -= nwritten;
    ptr   += nwritten;
  }

  return nbytes;
}

/* not used right now but it will be handy when i implement non-blocking
 * reads a writes
 */

int set_fl(int fd, int *flags) {
  int val;
  
  if ((val = fcntl(fd, F_GETFL, 0)) < 0)
    return -1;
  
  if (fcntl(fd, F_SETFL, val | *flags) < 0)
    return -1;
    
  *flags = val;
  return 1;
}

ssize_t read_fd_to_fd(int rfd, int wfd) {
  static char buf[BUFSIZ * 4];
  ssize_t     count, total_count = 0;
  
  for (total_count = 0;; total_count += count) {
    count = read(rfd, buf, sizeof buf);

    if (count == 0) return total_count;
    if (count <  0) return -1;

    safe_writen(wfd, buf, count);
  }
}

int get_local_hostname(char * dest, size_t dest_sz) {
  struct utsname buf;
  if (uname(&buf) < 0) return -1;

  strncpy(dest, buf.nodename, dest_sz);
  return 0;
}

void init_sockaddr(struct sockaddr_in *sin,
                   const char *hostname, u_short port_hbo)
{
  struct hostent *hp;
  
  memset(sin, 0, sizeof(*sin));
  sin->sin_family = AF_INET;

  if (hostname) 
    if ((hp = gethostbyname(hostname)) == NULL)
      rlpr_msg(FATAL, NO_ERRNO,	      /* FIXME this is a policy decision */
	       "hostname \"%s\" does not seem to exist!", hostname);
    else memcpy(&sin->sin_addr, hp->h_addr, hp->h_length);
    
  if (port_hbo)
    sin->sin_port = htons(port_hbo);
}

void toggle_euid(void) {
  static int times_called = 0;
  static uid_t setuid_uid, euid;

  if (!times_called)
    setuid_uid = geteuid();
  euid = (times_called++ % 2) ? setuid_uid : getuid();
  
#ifdef HAVE_SETEUID             /* use POSIX first */
  seteuid(euid);
#elif  HAVE_SETREUID            /* fallback on BSD */
  setreuid(-1, euid);
#elif  HAVE_SETRESUID           /* HP/UX saturday night special */
  setresuid(-1, euid, -1);
#else
#error you do not appear to have seteuid(), setreuid() or setresuid().
       you need one for secure operation of the rlpr package.
#endif
}

char * strlower(char *str) {
  char *tmp = str;
  while ((*str = tolower(*str))) str++;
  return tmp;
}  

/* this works with an fd because it is used with temporary
 * files that have already been unlink()'ed.
 */

off_t filesz(int fd) {
  static struct stat st;
  return (fstat(fd, &st) < 0) ? (off_t)-1 : st.st_size;
}

int bind_try_range(struct sockaddr_in * sin, int lo, int hi, int sock) {
  unsigned short i;

  for (i = lo; i <= hi; i++) {
    sin->sin_port = htons(i);
    if (bind(sock, (struct sockaddr *) sin, sizeof(*sin)) == 0)
      return 0;
  }
  return -1;
}

void get_and_verify_ack(int sockfd, char *caller)
{
  char ack;
  
  if (read(sockfd, &ack, 1) < 0)      /* check acknolwedgement */
    rlpr_msg(FATAL, ERRNO, "%s: while reading ack from lpd", caller);

  if (ack != 0)
    rlpr_msg(FATAL, NO_ERRNO, 
             "%s: lpd refused [%d]: are we in its /etc/hosts.lpd?",
	     caller, ack);
}


const char * h_strerror(void) {

  extern int h_errno;		/* some systems don't declare this in netdb.h */
  
  static struct { int error; const char * msg; }
  errlist[] =
    { { HOST_NOT_FOUND, "specified host is unknown" },
      { NO_ADDRESS    , "the request is valid but has no IP address" },
      { NO_RECOVERY   , "a non-recoverable name server error occurred" },
      { TRY_AGAIN     , "a temporary error occurred. try again" },
      { 0             ,  NULL } };
  size_t i;

  for (i = 0; errlist[i].msg; i++)
    if (h_errno == errlist[i].error)
      return errlist[i].msg;

  return "unknown error";
}

inline void * rlpr_malloc(size_t sz) {
  void *ptr = malloc(sz);
  check_ptr(ptr);
  return ptr;
}

inline void * rlpr_strdup(char * orig) {
  void *ptr = strdup(orig);
  check_ptr(ptr);
  return ptr;
}

/* these are only used if the host platform doesn't have them */

#ifndef HAVE_STRCSPN
size_t strcspn(const char *s, const char *reject) {
  const char *sp;

  for (sp = s; *sp; sp++)
    if (strchr(reject, *sp) != NULL) break;
  return sp - s;
}
#endif /* not HAVE_STRCSPN */

#ifndef HAVE_STRDUP
char * strdup(const char *s) {
  char * ptr = malloc(strlen(s) + 1);
  return ptr ? strcpy(ptr, s) : NULL;
}
#endif /* not HAVE_STRDUP */

#ifndef HAVE_STRSTR
char * strstr(const char *haystack, const char *needle) {
  if (*needle == '\0') return (char *) haystack;

  for (;(haystack = strchr(haystack, *needle)) != NULL; haystack++) {
    /* see if the rest matches */
    const char *hayp = haystack, *needp = needle;
    for (;;)
      if (*++needp == '\0') return (char *) haystack;
      else if (*++hayp != *needp) break;
  }

  return NULL;
}
#endif /* not HAVE_STRSTR */

#ifndef HAVE_VSNPRINTF
int vsnprintf(char *str, size_t n, const char *format, va_list ap) {
  return vsprintf(str, format, ap);
}
#endif  /* not HAVE_VSNPRINTF */

