
/*
 * input.c: does the actual input line stuff... keeps the appropriate stuff
 * on the input line, handles insert/delete of characters/words... the whole
 * ball o wax 
 *
 * Written By Michael Sandrof
 *
 * Copyright(c) 1990 
 *
 * See the COPYRIGHT file, or do a HELP IRCII COPYRIGHT 
 */

#include "irc.h"

#include "alias.h"
#include "commands.h"
#include "exec.h"
#include "history.h"
#include "hook.h"
#include "input.h"
#include "ircaux.h"
#include "keys.h"
#include "screen.h"
#include "server.h"
#include "ircterm.h"
#include "list.h"
#include "vars.h"
#include "misc.h"
#include "screen.h"
#include "output.h"
#include "chelp.h"
#include "dcc.h"
#include "cdcc.h"
#include "whowas.h"
#include "tcl_bx.h"
#include "window.h"
#include "status.h"
#include "hash2.h"

#ifdef TRANSLATE
#include "translat.h"
#endif
#ifdef WANT_HEBREW
#include "hebrew.h"
#endif

#include <sys/ioctl.h>

void get_history _((int));

static char new_nick[NICKNAME_LEN+1] = "";
static char *input_lastmsg = NULL;

extern NickTab *getnextnick _((int, char *, char *, char *));
extern int extended_handled;
extern char *getchannick _((char *, char *));
extern int foreground;

NickTab *tabkey_array = NULL, *autoreply_array = NULL;



const int WIDTH = 10;

/* input_prompt: contains the current, unexpanded input prompt */
static	char	*input_prompt = NULL;

enum I_STATE { 
	STATE_NORMAL = 0, 
	STATE_COMPLETE, 
	STATE_TABKEY, 
	STATE_TABKEYNEXT, 
	STATE_CNICK, 
	STATE_CNICKNEXT 
} in_completion = STATE_NORMAL;

/* These are sanity macros.  The file was completely unreadable before 
 * i put these in here.  I make no apologies for them.
 */
#define current_screen		last_input_screen
#define INPUT_CURSOR 		current_screen->input_cursor
#define INPUT_BUFFER 		current_screen->input_buffer
#define MIN_POS 		current_screen->buffer_min_pos
#define THIS_POS 		current_screen->buffer_pos
#define THIS_CHAR 		INPUT_BUFFER[THIS_POS]
#define MIN_CHAR 		INPUT_BUFFER[MIN_POS]
#define PREV_CHAR 		INPUT_BUFFER[THIS_POS-1]
#define NEXT_CHAR 		INPUT_BUFFER[THIS_POS+1]
#define ADD_TO_INPUT(x) 	strmcat(INPUT_BUFFER, (x), INPUT_BUFFER_SIZE);
#define INPUT_ONSCREEN 		current_screen->input_visible
#define INPUT_VISIBLE 		INPUT_BUFFER[INPUT_ONSCREEN]
#define ZONE			current_screen->input_zone_len
#define START_ZONE 		current_screen->input_start_zone
#define END_ZONE 		current_screen->input_end_zone
#define INPUT_PROMPT 		current_screen->input_prompt
#define INPUT_PROMPT_LEN 	current_screen->input_prompt_len
#define INPUT_LINE 		current_screen->input_line
  
#define BUILT_IN_KEYBINDING(x) void x (char key, char *string)


static int safe_puts (register char *str, int len, int echo) 
{
	int i = 0;
#ifdef WINNT
	i = len;
	nt_write(current_screen->hStdout, str, len);
#else
	while (*str && i < len)
	{
		term_putchar(*str);
		str++; i++;
	}
#endif
	return i;
}

/* cursor_to_input: move the cursor to the input line, if not there already */
extern void cursor_to_input _((void))
{
	Screen *oldscreen = last_input_screen;
	Screen *screen;
	
	if (!foreground)
		return;

	for (screen = screen_list; screen; screen = screen->next)
	{
		if (screen->alive && is_cursor_in_display(screen))
		{
			output_screen = screen;
			last_input_screen = screen;
			term_move_cursor(INPUT_CURSOR, INPUT_LINE);
			cursor_not_in_display();
			term_flush();
		}
	}
	output_screen = last_input_screen = oldscreen;
}

/*
 * update_input: does varying amount of updating on the input line depending
 * upon the position of the cursor and the update flag.  If the cursor has
 * move toward one of the edge boundaries on the screen, update_cursor()
 * flips the input line to the next (previous) line of text. The update flag
 * may be: 
 *
 * NO_UPDATE - only do the above bounds checking. 
 *
 * UPDATE_JUST_CURSOR - do bounds checking and position cursor where is should
 * be. 
 *
 * UPDATE_FROM_CURSOR - does all of the above, and makes sure everything from
 * the cursor to the right edge of the screen is current (by redrawing it). 
 *
 * UPDATE_ALL - redraws the entire line 
 */
extern void	update_input (int update)
{
	int	old_zone;
	char	*ptr, *ptr_free;
	int	len,
		free_it = 0,
		echo = 1,
		max;

	char	*prompt;
#ifdef WANT_HEBREW
	char prehebbuff[2000];
#endif
	if (dumb_mode || !foreground)
		return;
#ifdef WANT_HEBREW
	/* 
	* crisk: hebrew thingy
	*/
	if (get_int_var(HEBREW_TOGGLE_VAR))
	{
		update = UPDATE_ALL;
		strcpy(prehebbuff,get_input());
		hebrew_process(get_input());
	}
#endif

	cursor_to_input();

	if (last_input_screen->promptlist)
		prompt = last_input_screen->promptlist->prompt;
	else
		prompt = input_prompt;
	if (prompt && update != NO_UPDATE)
	{
		int	args_used;	

		if (is_valid_process(get_target_by_refnum(0)) != -1)
			ptr = (char *)get_prompt_by_refnum(0);
		else
		{
			ptr = expand_alias(prompt, empty_string, &args_used, NULL);
			free_it = 1;
		}
		if (last_input_screen->promptlist)
			term_echo(last_input_screen->promptlist->echo);

		ptr_free = ptr;
		ptr = strip_ansi(ptr);
		strcat(ptr, ALL_OFF_STR);	/* Yes, we can do this */
		if (free_it)
			new_free(&ptr_free);
		free_it = 1;
		
		if ((ptr && !INPUT_LINE) || (!ptr && INPUT_LINE) ||
			strcmp(ptr, last_input_screen->input_buffer))
		{
			if (last_input_screen->input_prompt_malloc)
				new_free(&INPUT_PROMPT);

			last_input_screen->input_prompt_malloc = free_it;

			INPUT_PROMPT = ptr;
			len = strlen(INPUT_PROMPT);
			INPUT_PROMPT_LEN = output_with_count(INPUT_PROMPT, 0, 0);
/*			len - count_ansi(INPUT_PROMPT, len);*/
			update = UPDATE_ALL;
		}
		else
		{
			if (free_it)
				new_free(&ptr);
		}
	}

	/*
	 * 
	 * HAS THE SCREEN CHANGED SIZE SINCE THE LAST TIME?
	 *
	 */

	/*
	 * If the screen has resized, then we need to re-compute the
	 * side-to-side scrolling effect.
	 */
	if ((last_input_screen->li != last_input_screen->old_li) || 
	    (last_input_screen->co != last_input_screen->old_co))
	{
		/*
		 * The input line is always the bottom line
		 */
		INPUT_LINE = last_input_screen->li;

		/*
		 * The "zone" is the range in which when you type, the
		 * input line does not scroll.  Its WIDTH chars in from
		 * either side.
		 */
		ZONE = last_input_screen->co - (WIDTH * 2);
		if (ZONE < 10)
			ZONE = 10;		/* Take that! */

		START_ZONE = WIDTH;
		END_ZONE = last_input_screen->co - WIDTH;

		last_input_screen->old_co = last_input_screen->co;
		last_input_screen->old_li = last_input_screen->li;
	}

	/*
	 * About zones:
	 * The input line is divided into "zones".  A "zone" is set above,
	 * and is the width of the screen minus 20 (by default).  The input
	 * line, as displayed, is therefore composed of the current "zone",
	 * plus 10 characters from the previous zone, plus 10 characters 
	 * from the next zone.  When the cursor moves to an adjacent zone,
	 * (by going into column 9 from the right or left of the edge), the
	 * input line is redrawn.  There is one catch.  The first "zone"
	 * includes the first ten characters of the input line.
	 */
	old_zone = START_ZONE;

	/*
	 * The BEGINNING of the current "zone" is a calculated value:
	 *	The number of characters since the origin of the input buffer
	 *	is the number of printable chars in the input prompt plus the
	 *	current position in the input buffer.  We subtract from that
	 * 	the WIDTH delta to take off the first delta, which doesnt
	 *	count towards the width of the zone.  Then we divide that by
	 * 	the size of the zone, to get an integer, then we multiply it
	 * 	back.  This gives us the first character on the screen.  We
	 *	add WIDTH to the result in order to get the start of the zone
	 *	itself.
	 * The END of the current "zone" is just the beginning plus the width.
	 * If we have moved to an adjacent "zone" since last time, we want to
	 * 	completely redraw the input line.
	 */
	START_ZONE = ((INPUT_PROMPT_LEN + THIS_POS - WIDTH) / ZONE) * ZONE + WIDTH;
	END_ZONE = START_ZONE + ZONE;

	if (old_zone != START_ZONE)
		update = UPDATE_ALL;

	/*
	 * Now that we know where the "zone" is in the input buffer, we can
	 * easily calculate where where we want to start displaying stuff
	 * from the INPUT_BUFFER.  If we're in the first "zone", then we will
	 * output from the beginning of the buffer.  If we're not in the first
	 * "zone", then we will begin to output from 10 characters to the
	 * left of the zone, after adjusting for the length of the prompt.
	 */
	if (START_ZONE == WIDTH)
		INPUT_ONSCREEN = 0;
	else
		INPUT_ONSCREEN = START_ZONE - WIDTH - INPUT_PROMPT_LEN;

	/*
	 * And the cursor is simply how many characters away THIS_POS is
	 * from the first column on the screen.
	 */
	if (INPUT_ONSCREEN == 0)
		INPUT_CURSOR = INPUT_PROMPT_LEN + THIS_POS;
	else
		INPUT_CURSOR = THIS_POS - INPUT_ONSCREEN;

	/*
	 * If the cursor moved, or if we're supposed to do a full update,
	 * then redrwa the entire input line.
	 */
	if (update == UPDATE_ALL)
	{
		term_move_cursor(0, INPUT_LINE);

		/*
		 * If the input line is NOT empty, and we're starting the
		 * display at the beginning of the input buffer, then we
		 * output the prompt first.
		 */
		if (INPUT_ONSCREEN == 0 && INPUT_PROMPT && *INPUT_PROMPT)
		{
			/*
			 * Forcibly turn on echo.
			 */
			int	echo = term_echo(1);

			/*
			 * Crop back the input prompt so it does not extend
			 * past the end of the zone.
			 */
			if (INPUT_PROMPT_LEN > (last_input_screen->co - WIDTH))
				INPUT_PROMPT_LEN = last_input_screen->co - WIDTH - 1;

			/*
			 * Output the prompt.
			 */
			output_with_count(INPUT_PROMPT, 0, 1);

			/*
			 * Turn the echo back to what it was before,
			 * and output the rest of the input buffer.
			 */
			term_echo(echo);
			safe_puts(INPUT_BUFFER, last_input_screen->co - INPUT_PROMPT_LEN, echo);
		}

		/*
		 * Otherwise we just output whatever we have.
		 */
		else if (echo)
			safe_puts(&(INPUT_VISIBLE), last_input_screen->co, echo);

		/*
		 * Clear the rest of the input line and reset the cursor
		 * to the current input position.
		 */
		term_clear_to_eol();
	}
	else if (update == UPDATE_FROM_CURSOR)
	{
		/*
		 * Move the cursor to where its supposed to be,
		 * Figure out how much we can output from here,
		 * and then output it.
		 */
		term_move_cursor(INPUT_CURSOR, INPUT_LINE);
		max = last_input_screen->co - (THIS_POS - INPUT_ONSCREEN);
		if (INPUT_ONSCREEN == 0 && INPUT_PROMPT && *INPUT_PROMPT)
			max -= INPUT_PROMPT_LEN;

		if ((len = strlen(&(THIS_CHAR))) > max)
			len = max;
		safe_puts(&(THIS_CHAR), len, echo);
		term_clear_to_eol();
	}
	term_move_cursor(INPUT_CURSOR, INPUT_LINE);
	term_echo(1);
	term_flush();
#ifdef WANT_HEBREW
	/* 
	 * crisk: hebrew thingy 
	 */
	if (get_int_var(HEBREW_TOGGLE_VAR))
		set_input(prehebbuff);
#endif
}

extern void change_input_prompt (int direction)
{
	if (!last_input_screen->promptlist)
	{
		strcpy(INPUT_BUFFER, last_input_screen->saved_input_buffer);
		THIS_POS = last_input_screen->saved_buffer_pos;
		MIN_POS = last_input_screen->saved_min_buffer_pos;
		*last_input_screen->saved_input_buffer = '\0';
		last_input_screen->saved_buffer_pos = 0;
		last_input_screen->saved_min_buffer_pos = 0;
		update_input(UPDATE_ALL);
	}

	else if (direction == -1)
		update_input(UPDATE_ALL);

	else if (!last_input_screen->promptlist->next)
	{
		strcpy(last_input_screen->saved_input_buffer, INPUT_BUFFER);
		last_input_screen->saved_buffer_pos = THIS_POS;
		last_input_screen->saved_min_buffer_pos = MIN_POS;
		*INPUT_BUFFER = '\0';
		THIS_POS = MIN_POS = 0;
		update_input(UPDATE_ALL);
	}
}

/* input_move_cursor: moves the cursor left or right... got it? */
extern void	input_move_cursor (int dir)
{
	cursor_to_input();
	if (dir)
	{
		if (THIS_CHAR)
		{
			THIS_POS++;
			term_cursor_right();
		}
	}
	else
	{
		if (THIS_POS > MIN_POS)
		{
			THIS_POS--;
			term_cursor_left();
		}
	}
	update_input(NO_UPDATE);
}

/*
 * set_input: sets the input buffer to the given string, discarding whatever
 * was in the input buffer before 
 */
void	set_input (char *str)
{
	strmcpy(INPUT_BUFFER + MIN_POS, str, INPUT_BUFFER_SIZE - MIN_POS);
	THIS_POS = strlen(INPUT_BUFFER);
}

/*
 * get_input: returns a pointer to the input buffer.  Changing this will
 * actually change the input buffer.  This is a bad way to change the input
 * buffer tho, cause no bounds checking won't be done 
 */
char	*get_input _((void))
{
	return (&(MIN_CHAR));
}

/* init_input: initialized the input buffer by clearing it out */
extern void init_input _((void))
{
	*INPUT_BUFFER = (char) 0;
	THIS_POS = MIN_POS;
}

/* get_input_prompt: returns the current input_prompt */
extern char	*get_input_prompt _((void))
{ 
	return input_prompt; 
}

/*
 * set_input_prompt: sets a prompt that will be displayed in the input
 * buffer.  This prompt cannot be backspaced over, etc.  It's a prompt.
 * Setting the prompt to null uses no prompt 
 */
void	set_input_prompt (Window *win, char *prompt, int unused)
{
	if (prompt)
		malloc_strcpy(&input_prompt, convert_output_format(prompt, NULL, NULL));
	else if (input_prompt)
		malloc_strcpy(&input_prompt, empty_string);
	else
		return;

	update_input(UPDATE_ALL);
}


/* 
 * Why did i put these in this file?  I dunno.  But i do know that the ones 
 * in edit.c didnt have to be here, and i knew the ones that were here DID 
 * have to be here, so i just moved them all to here, so that they would all
 * be in the same place.  Easy enough. (jfn, june 1995)
 */

/*
 * input_forward_word: move the input cursor forward one word in the input
 * line 
 */
BUILT_IN_KEYBINDING(input_forward_word)
{
	cursor_to_input();

	while ((my_isspace(THIS_CHAR) || ispunct(THIS_CHAR)) && (THIS_CHAR))
			THIS_POS++;
	while (!(ispunct(THIS_CHAR) || my_isspace(THIS_CHAR)) && (THIS_CHAR))
			THIS_POS++;
	update_input(UPDATE_JUST_CURSOR);
}

/* input_backward_word: move the cursor left on word in the input line */
BUILT_IN_KEYBINDING(input_backward_word)
{
	cursor_to_input();
	while ((THIS_POS > MIN_POS) && (my_isspace(PREV_CHAR) || ispunct(PREV_CHAR)))
		THIS_POS--;
	while ((THIS_POS > MIN_POS) && !(ispunct(PREV_CHAR) || my_isspace(PREV_CHAR)))
		THIS_POS--;

	update_input(UPDATE_JUST_CURSOR);
}

/* input_delete_character: deletes a character from the input line */
BUILT_IN_KEYBINDING(input_delete_character)
{
	cursor_to_input();
	if (THIS_CHAR)
	{
		char	*ptr = NULL;
		int	pos;

		malloc_strcpy(&ptr, &(NEXT_CHAR));
		strcpy(&(THIS_CHAR), ptr);
		new_free(&ptr);

		if (termfeatures & TERM_CAN_DELETE)
		{
			term_delete(1);
			update_input(UPDATE_FROM_CURSOR);
		}
		else
		{
			pos = INPUT_ONSCREEN + last_input_screen->co - 1;
			if (pos < strlen(INPUT_BUFFER))
			{
				term_move_cursor(last_input_screen->co - 1, INPUT_LINE);
#ifdef WINNT
				nt_write(last_input_screen->hStdout, INPUT_BUFFER[pos], 1);
#else
				term_putchar(INPUT_BUFFER[pos]);
#endif
				term_move_cursor(INPUT_CURSOR, INPUT_LINE);
			}
			update_input(NO_UPDATE);
		}
	}
	in_completion = STATE_NORMAL;
}

/* input_backspace: does a backspace in the input buffer */
BUILT_IN_KEYBINDING(input_backspace)
{
	cursor_to_input();
	if (THIS_POS > MIN_POS)
	{
		char	*ptr = NULL;
		int	pos;


		malloc_strcpy(&ptr, &(THIS_CHAR));
		strcpy(&(PREV_CHAR), ptr);
		new_free(&ptr);
		THIS_POS--;
		term_cursor_left();
		if (THIS_CHAR)
		{
			if (termfeatures & TERM_CAN_DELETE)
			{
				term_delete(1);
				update_input(UPDATE_FROM_CURSOR);
			}
			else
			{
				pos = INPUT_ONSCREEN + last_input_screen->co - 1;
				if (pos < strlen(INPUT_BUFFER))
				{
					term_move_cursor(last_input_screen->co - 1, INPUT_LINE);
#ifdef WINNT
					nt_write(last_input_screen->hStdout, &(INPUT_BUFFER[pos]), 1);
#else
					term_putchar(INPUT_BUFFER[pos]);
#endif
				}
				update_input(UPDATE_JUST_CURSOR);
			}
		}
		else
		{
#ifdef WINNT
			nt_write(last_input_screen->hStdout, " ", 1);
#else
			term_putchar(' ');
#endif
#ifndef __EMX__
			term_cursor_left();
			update_input(NO_UPDATE);
#else
			update_input(UPDATE_FROM_CURSOR);
#endif
		}
	}
	in_completion = STATE_NORMAL;
	*new_nick = 0;
}

/*
 * input_beginning_of_line: moves the input cursor to the first character in
 * the input buffer 
 */
BUILT_IN_KEYBINDING(input_beginning_of_line)
{
	cursor_to_input();
	THIS_POS = MIN_POS;
	update_input(UPDATE_JUST_CURSOR);
}

/*
 * input_beginning_of_line: moves the input cursor to the first character in
 * the input buffer 
 */
BUILT_IN_KEYBINDING(new_input_beginning_of_line)
{
	cursor_to_input();
	THIS_POS = MIN_POS;
	update_input(UPDATE_JUST_CURSOR);
	extended_handled = 1;
}

/*
 * input_end_of_line: moves the input cursor to the last character in the
 * input buffer 
 */
BUILT_IN_KEYBINDING(input_end_of_line)
{
	cursor_to_input();
	THIS_POS = strlen(INPUT_BUFFER);
	update_input(UPDATE_JUST_CURSOR);
}

BUILT_IN_KEYBINDING(input_delete_to_previous_space)
{
	int	old_pos;
	char	c;

	cursor_to_input();
	old_pos = THIS_POS;
	c = THIS_CHAR;

	while (!my_isspace(THIS_CHAR) && THIS_POS >= MIN_POS)
		THIS_POS--;

	if (THIS_POS < old_pos)
	{
		strcpy(&(NEXT_CHAR), &(INPUT_BUFFER[old_pos]));
		THIS_POS++;
	}

	update_input(UPDATE_FROM_CURSOR);
}


/*
 * input_delete_previous_word: deletes from the cursor backwards to the next
 * space character. 
 */
BUILT_IN_KEYBINDING(input_delete_previous_word)
{
	int	old_pos;
	char	c;

	cursor_to_input();
	old_pos = THIS_POS;
	while ((THIS_POS > MIN_POS) && (my_isspace(PREV_CHAR) || ispunct(PREV_CHAR)))
		THIS_POS--;
	while ((THIS_POS > MIN_POS) && !(ispunct(PREV_CHAR) || my_isspace(PREV_CHAR)))
		THIS_POS--;
	c = INPUT_BUFFER[old_pos];
	INPUT_BUFFER[old_pos] = (char) 0;
	malloc_strcpy(&cut_buffer, &THIS_CHAR);
	INPUT_BUFFER[old_pos] = c;
	strcpy(&(THIS_CHAR), &(INPUT_BUFFER[old_pos]));
	update_input(UPDATE_FROM_CURSOR);
}

/*
 * input_delete_next_word: deletes from the cursor to the end of the next
 * word 
 */
BUILT_IN_KEYBINDING(input_delete_next_word)
{
	int	pos;
	char	*ptr = NULL,
		c;

	cursor_to_input();
	pos = THIS_POS;
	while ((my_isspace(INPUT_BUFFER[pos]) || ispunct(INPUT_BUFFER[pos])) && INPUT_BUFFER[pos])
		pos++;
	while (!(ispunct(INPUT_BUFFER[pos]) || my_isspace(INPUT_BUFFER[pos])) && INPUT_BUFFER[pos])
		pos++;
	c = INPUT_BUFFER[pos];
	INPUT_BUFFER[pos] = (char) 0;
	malloc_strcpy(&cut_buffer, &(THIS_CHAR));
	INPUT_BUFFER[pos] = c;
	malloc_strcpy(&ptr, &(INPUT_BUFFER[pos]));
	strcpy(&(THIS_CHAR), ptr);
	new_free(&ptr);
	update_input(UPDATE_FROM_CURSOR);
}

/*
 * input_add_character: adds the character c to the input buffer, repecting
 * the current overwrite/insert mode status, etc 
 */
BUILT_IN_KEYBINDING(input_add_character)
{
	int	display_flag = NO_UPDATE;

	if (last_input_screen->promptlist)
		term_echo(last_input_screen->promptlist->echo);
		
	cursor_to_input();
        if (THIS_POS >= INPUT_BUFFER_SIZE)
	{
		term_echo(1);
		return;
	}
                                                        
	if (get_int_var(INSERT_MODE_VAR))
	{
		if (THIS_CHAR)
		{
			char	*ptr = NULL;

			ptr = alloca(strlen(&(THIS_CHAR)) + 1);
			strcpy(ptr, &(THIS_CHAR));
			THIS_CHAR = key;
			NEXT_CHAR = 0;
			ADD_TO_INPUT(ptr);
			if (termfeatures & TERM_CAN_INSERT)
				term_insert(key);
			else
			{
#ifdef WINNT
				nt_write(last_input_screen->hStdout, &(THIS_CHAR), 1);
#else
				term_putchar(key);
#endif
				if (NEXT_CHAR)
				    display_flag = UPDATE_FROM_CURSOR;
				else
				    display_flag = NO_UPDATE;
			}
		}
		else
		{
			THIS_CHAR = key;
			NEXT_CHAR = 0;
#ifdef WINNT
			nt_write(last_input_screen->hStdout, &(THIS_CHAR), 1);
#else
			term_putchar(key);
#endif
		}
	}
	else
	{
		if (THIS_CHAR == 0)
			NEXT_CHAR = 0;
		THIS_CHAR = key;
#ifdef WINNT
		nt_write(last_input_screen->hStdout, &(THIS_CHAR), 1);
#else
		term_putchar(key);
#endif
	}
	THIS_POS++;
	update_input(display_flag);
	if (in_completion == STATE_COMPLETE && key == ' ' && input_lastmsg)
	{
		new_free(&input_lastmsg);
		*new_nick = 0;
		in_completion = STATE_NORMAL;
	}
	term_echo(1);
}

/* input_clear_to_eol: erases from the cursor to the end of the input buffer */
BUILT_IN_KEYBINDING(input_clear_to_eol)
{
	cursor_to_input();
	malloc_strcpy(&cut_buffer, &(THIS_CHAR));
	THIS_CHAR = 0;
	term_clear_to_eol();
	update_input(NO_UPDATE);
}

/*
 * input_clear_to_bol: clears from the cursor to the beginning of the input
 * buffer 
 */
BUILT_IN_KEYBINDING(input_clear_to_bol)
{
	char	*ptr = NULL;
	cursor_to_input();
	malloc_strcpy(&cut_buffer, &(MIN_CHAR));
	cut_buffer[THIS_POS - MIN_POS] = (char) 0;
	malloc_strcpy(&ptr, &(THIS_CHAR));
	MIN_CHAR = (char) 0;
	ADD_TO_INPUT(ptr);
	new_free(&ptr);
	THIS_POS = MIN_POS;
	term_move_cursor(INPUT_PROMPT_LEN, INPUT_LINE);
	term_clear_to_eol();
	update_input(UPDATE_FROM_CURSOR);
}

/*
 * input_clear_line: clears entire input line
 */
BUILT_IN_KEYBINDING(input_clear_line)
{
	cursor_to_input();
	malloc_strcpy(&cut_buffer, INPUT_BUFFER + MIN_POS);
	MIN_CHAR = (char) 0;
	THIS_POS = MIN_POS;
	term_move_cursor(INPUT_PROMPT_LEN, INPUT_LINE);
	term_clear_to_eol();
	update_input(NO_UPDATE);
}

/*
 * input_transpose_characters: swaps the positions of the two characters
 * before the cursor position 
 */
BUILT_IN_KEYBINDING(input_transpose_characters)
{
	cursor_to_input();
	if (last_input_screen->buffer_pos > MIN_POS)
	{
		u_char	c1[3] = { 0 };
		int	pos, end_of_line = 0;

		if (THIS_CHAR)
			pos = THIS_POS;
		else if (strlen(get_input()) > MIN_POS + 2)
		{
			pos = THIS_POS - 1;
			end_of_line = 1;
		}
		else
			return;

		c1[0] = INPUT_BUFFER[pos];
		c1[1] = INPUT_BUFFER[pos] = INPUT_BUFFER[pos - 1];
		INPUT_BUFFER[pos - 1] = c1[0];
		term_cursor_left();
		if (end_of_line)
			term_cursor_left();

#ifdef WINNT
		nt_write(last_input_screen->hStdout, c1, 2);
#else
		term_putchar(c1[0]);
		term_putchar(c1[1]);
#endif
		if (!end_of_line)
			term_cursor_left();
		update_input(NO_UPDATE);
	}
}


BUILT_IN_KEYBINDING(refresh_inputline)
{
	update_input(UPDATE_ALL);
}

/*
 * input_yank_cut_buffer: takes the contents of the cut buffer and inserts it
 * into the input line 
 */
BUILT_IN_KEYBINDING(input_yank_cut_buffer)
{
	char	*ptr = NULL;

	if (cut_buffer)
	{
		malloc_strcpy(&ptr, &(THIS_CHAR));
		/* Ooops... */
		THIS_CHAR = 0;
		ADD_TO_INPUT(cut_buffer);
		ADD_TO_INPUT(ptr);
		new_free(&ptr);
		update_input(UPDATE_FROM_CURSOR);
		THIS_POS += strlen(cut_buffer);
		if (THIS_POS > INPUT_BUFFER_SIZE)
			THIS_POS = INPUT_BUFFER_SIZE;
		update_input(UPDATE_JUST_CURSOR);
	}
}


/* used with input_move_cursor */
#define RIGHT 1
#define LEFT 0

/* BIND functions: */
BUILT_IN_KEYBINDING(forward_character)
{
	input_move_cursor(RIGHT);
}

BUILT_IN_KEYBINDING(backward_character)
{
	input_move_cursor(LEFT);
}

BUILT_IN_KEYBINDING(backward_history)
{
	get_history(PREV);
}

BUILT_IN_KEYBINDING(forward_history)
{
	get_history(NEXT);
}

BUILT_IN_KEYBINDING(toggle_insert_mode)
{
	int tog = get_int_var(INSERT_MODE_VAR);
	tog ^= 1;
	set_int_var(INSERT_MODE_VAR, tog);
}


BUILT_IN_KEYBINDING(input_msgreply)
{
char *cmdchar;
char *line, *cmd, *t;
char *snick;
NickTab *nick = NULL;
int got_space = 0;

	if (!(cmdchar = get_string_var(CMDCHARS_VAR))) 
		cmdchar = DEFAULT_CMDCHARS;

	t = line = m_strdup(get_input());
	if (t)
		got_space = strchr(t, ' ') ? 1 : 0;
	cmd = next_arg(line, &line);
	snick = next_arg(line, &line);
	if ((cmd && *cmd == *cmdchar && got_space) || !cmd)
	{

		if (cmd && *cmd == *cmdchar)
			cmd++;
		if (in_completion == STATE_NORMAL && snick)
			strncpy(new_nick, snick, sizeof(new_nick)-1);

		if ((nick = getnextnick(0, new_nick, input_lastmsg, snick)))
		{
			if (nick->nick && *(nick->nick))
			{
				snick = nick->nick;
				malloc_strcpy(&input_lastmsg, nick->nick);
			}
		}
		if (nick)
		{
			char *tmp = NULL;
			input_clear_line('\0', NULL);
			if (fget_string_var(FORMAT_NICK_MSG_FSET))
				malloc_strcpy(&tmp, stripansicodes(convert_output_format(fget_string_var(FORMAT_NICK_MSG_FSET), "%s%s %s %s", cmdchar, nick->type ? nick->type:cmd?cmd:"msg", nick->nick, line?line:empty_string)));
			else
				malloc_sprintf(&tmp, "%s%s %s %s", cmdchar, nick->type?nick->type:cmd?cmd:"msg", nick->nick, line?line:empty_string);
			set_input(tmp);
			new_free(&tmp);
		} else
			command_completion(0, NULL);
	} 
	else
		command_completion(0, NULL);
	update_input(UPDATE_ALL);
	new_free(&t);
}

BUILT_IN_KEYBINDING(input_msgreplyback)
{
#if 0
char *tmp = NULL;
char *cmdchar;
	tmp = gettabkey(-1, 0, NULL);
	if (tmp && *tmp)
	{
		char *tmp1 = NULL;
		input_clear_line('\0', NULL);
		if (!(cmdchar = get_string_var(CMDCHARS_VAR))) 
			cmdchar = DEFAULT_CMDCHARS;
		if (fget_string_var(FORMAT_NICK_MSG_FSET))
			malloc_strcpy(&tmp1, stripansicodes(convert_output_format(fget_string_var(FORMAT_NICK_MSG_FSET), "%cmsg %s", *cmdchar, tmp)));
		else
			malloc_sprintf(&tmp1, "%cmsg %s ", *cmdchar, tmp);
		set_input(tmp1);
		new_free(&tmp1);
	}
#endif
}

void add_autonick_input(char *nick, char *line)
{
char *tmp1 = NULL;
	input_clear_line('\0', NULL);
	if ((do_hook(AR_REPLY_LIST, "%s", nick)))
	{
		if (fget_string_var(FORMAT_NICK_AUTO_FSET))
			malloc_strcpy(&tmp1, stripansicodes(convert_output_format(fget_string_var(FORMAT_NICK_AUTO_FSET), "%s %s", nick, line?line:empty_string)));
		else
			malloc_sprintf(&tmp1, "%s\002:\002 %s" , nick, line);
		set_input(tmp1);
		new_free(&tmp1);
	}
	update_input(UPDATE_ALL);
}

BUILT_IN_KEYBINDING(input_autoreply)
{
char *tmp = NULL, *q;
char *nick = NULL;
char *line = NULL;

	q = line = m_strdup(&last_input_screen->input_buffer[MIN_POS]);
	if ((nick = next_arg(line, &line)))
	{
		if ((tmp = strrchr(nick, ':')))
			*tmp = 0;
		if ((tmp = strrchr(nick, '\002')))
			*tmp = 0;
	}
	if (!input_lastmsg)
	{
		NickTab *t;
		t = gettabkey(1, 1, nick);
		if (*new_nick && !tmp)
			t = gettabkey(1,1,new_nick);
		if (t)
			tmp = t->nick;
	}
	if (tmp && *tmp)
	{
		add_autonick_input(tmp, line);
		strcpy(new_nick, tmp);
	}
	else
	{
		tmp = getchannick(input_lastmsg, nick);	
		if (*new_nick && !tmp)
			tmp = getchannick(input_lastmsg, new_nick);		
		if (tmp)
		{
			add_autonick_input(tmp, line);
			strcpy(new_nick, tmp);
		}
	}
	malloc_strcpy(&input_lastmsg, nick);
	in_completion = STATE_COMPLETE;
	new_free(&q);
}

BUILT_IN_KEYBINDING(input_autoreplyback)
{
#if 0
char *tmp = NULL;
	tmp = gettabkey(-1, 1, NULL);
	if (tmp && *tmp)
	{
		char *tmp1 = NULL;
		input_clear_line('\0', NULL);
		if ((do_hook(AR_REPLY_LIST, "%s", tmp)))
		{
			if (fget_int_var(FORMAT_NICK_AUTO_FSET))
				malloc_strcpy(&tmp1, stripansicodes(convert_output_format(fget_string_var(FORMAT_NICK_AUTO_FSET), "%s", tmp)));
			else
				malloc_sprintf(&tmp1, "%s\002:\002", tmp);
			set_input(tmp1);
			new_free(&tmp1);
		}
	}
#endif
}


BUILT_IN_KEYBINDING(send_line)
{
	int	server;
	WaitPrompt	*OldPrompt;
	
	server = from_server;
	from_server = get_window_server(0);
	unhold_a_window(current_window);
	if (last_input_screen->promptlist && last_input_screen->promptlist->type == WAIT_PROMPT_LINE)
	{
		OldPrompt = last_input_screen->promptlist;
		(*OldPrompt->func)(OldPrompt->data, get_input());
		set_input(empty_string);
		last_input_screen->promptlist = OldPrompt->next;
		new_free(&OldPrompt->data);
		new_free(&OldPrompt->prompt);
		new_free((char **)&OldPrompt);
		change_input_prompt(-1);
	}
	else
	{
		char	*line,
			*tmp = NULL;

		line = get_input();
		if (line && (*line != get_int_var(CMDCHARS_VAR)) && get_int_var(NICK_COMPLETION_VAR))
		{
			char auto_comp_char = ':';
			if (!(auto_comp_char = (char)get_int_var(NICK_COMPLETION_CHAR_VAR)))
				auto_comp_char = ':';
								
			/* possible nick completion */
			if (strchr(line, auto_comp_char))
			{
				char *p;
				ChannelList *chan;
				NickList *nick;
				char *channel;				
				malloc_strcpy(&tmp, line);
				p = strchr(tmp, auto_comp_char);
				*p++ = 0;
				if (*tmp && *p && (channel = get_current_channel_by_refnum(0)))
				{
					chan = lookup_channel(channel, from_server, 0);
					for (nick = next_nicklist(chan, NULL); nick; nick = next_nicklist(chan, nick))
						if (!my_strnicmp(tmp, nick->nick, strlen(tmp)))
							break;

					/* This may not be as efficient, but lets let bx try its method
					 * first due to the order it stores the nick list... This way we can
					 * be sure to take the one it finds over the inside completion...
					 */
					if (!nick)
						for (nick = next_nicklist(chan, NULL); nick; nick = next_nicklist(chan, nick))
							if (my_strnstr(nick->nick, tmp, strlen(tmp)))
								break;
					if (nick)
					{
						if (fget_string_var(FORMAT_NICK_COMP_FSET))
							malloc_strcpy(&tmp, stripansicodes(convert_output_format(fget_string_var(FORMAT_NICK_COMP_FSET), "%s %s", nick->nick, p)));
						else
							malloc_sprintf(&tmp, "%s\002%c\002%s", nick->nick, auto_comp_char, p);
					}
					else
						malloc_strcpy(&tmp, line);
				} else malloc_strcpy(&tmp, line);
			} else malloc_strcpy(&tmp, line);
		} else 
			malloc_strcpy(&tmp, line);
#ifdef WANT_TCL
		
		if (!check_tcl_input(line) && do_hook(INPUT_LIST, "%s", tmp))
#else
		if (do_hook(INPUT_LIST, "%s", tmp))
#endif
		{
			if (get_int_var(INPUT_ALIASES_VAR))
				parse_line(NULL, tmp, empty_string, 1, 0);
			else
				parse_line(NULL, tmp, NULL, 1, 0);
		}
		update_input(UPDATE_ALL);
		new_free(&tmp);
	}
	new_free(&input_lastmsg);
	*new_nick = 0;
	in_completion = STATE_NORMAL;
	from_server = server;
}

#define METAX(x) \
	BUILT_IN_KEYBINDING( meta ## x ## _char ) \
	{ last_input_screen->meta_hit = (x); }
METAX(39) METAX(38) METAX(37) METAX(36) METAX(35) 
METAX(34) METAX(33) METAX(32) METAX(31) METAX(30)
METAX(29) METAX(28) METAX(27) METAX(26) METAX(25) 
METAX(24) METAX(23) METAX(22) METAX(21) METAX(20)
METAX(19) METAX(18) METAX(17) METAX(16) METAX(15) 
METAX(14) METAX(13) METAX(12) METAX(11) METAX(10)
METAX(9)  METAX(8)  METAX(7)  METAX(6)  METAX(5)
METAX(3)  METAX(2)  METAX(1)
                

BUILT_IN_KEYBINDING(meta4_char)
{
	if (last_input_screen->meta_hit == 4)
		last_input_screen->meta_hit = 0;
	else
		last_input_screen->meta_hit = 4;
}

BUILT_IN_KEYBINDING(quote_char)
{
	last_input_screen->quote_hit = 1;
}

/* These four functions are boomerang functions, which allow the highlight
 * characters to be bound by simply having these functions put in the
 * appropriate characters when you press any key to which you have bound
 * that highlight character. >;-)
 */
BUILT_IN_KEYBINDING(insert_bold)
{
	input_add_character (BOLD_TOG, string);
}

BUILT_IN_KEYBINDING(insert_reverse)
{
	input_add_character (REV_TOG, string);
}

BUILT_IN_KEYBINDING(insert_underline)
{
	input_add_character (UND_TOG, string);
}

BUILT_IN_KEYBINDING(highlight_off)
{
	input_add_character (ALL_OFF, string);
}

BUILT_IN_KEYBINDING(insert_blink)
{
	input_add_character (BLINK_TOG, string);
}
  

/* type_text: the BIND function TYPE_TEXT */
BUILT_IN_KEYBINDING(type_text)
{
	if (!string)
		return;
	for (; *string; string++)
		input_add_character(*string, empty_string);
}

/*
 * clear_screen: the CLEAR_SCREEN function for BIND.  Clears the screen and
 * starts it if it is held 
 */
BUILT_IN_KEYBINDING(clear_screen)
{
	hold_mode(NULL, OFF, 1);
	clear_window_by_refnum(0);
}

/* parse_text: the bindable function that executes its string */
BUILT_IN_KEYBINDING(parse_text)
{
	parse_line(NULL, string, empty_string, 0, 0);
}


/*
 * edit_char: handles each character for an input stream.  Not too difficult
 * to work out.
 */
void	edit_char (u_char key)
{
	void	(*func) (char, char *) = NULL;
	char	*ptr = NULL;
	u_char	extended_key;
	WaitPrompt *oldprompt;
	int 	i;

	if (dumb_mode)
	{
#ifdef TIOCSTI
		ioctl(0, TIOCSTI, &key);
#else
		say("Sorry, your system doesnt support 'faking' user input...");
#endif
		return;
	}

	/* were we waiting for a keypress? */
	if (last_input_screen->promptlist && last_input_screen->promptlist->type == WAIT_PROMPT_KEY)
	{
		unsigned char key_[2] = "\0";
		key_[0] = key;
		oldprompt = last_input_screen->promptlist;

		(*oldprompt->func)(oldprompt->data, key_);

		set_input(empty_string);
		last_input_screen->promptlist = oldprompt->next;
		new_free(&oldprompt->data);
		new_free(&oldprompt->prompt);
		new_free((char **)&oldprompt);
		change_input_prompt(-1);
		return;
	}

#if 0
	if (key & 0x80)
	{
		if (meta_mode)
		{
			edit_char('\033');
			key &= ~0x80;
		}
		else if (!term_eight_bit())
			key &= ~0x80;
	}
#endif
                                                                                                                                                                
#ifdef TRANSLATE
	if (translation)
		extended_key = transFromClient[key];
	else
#endif
	extended_key = key;


#ifdef __EMX__
	if (key == 0)
		key = 27;
#endif

	/* did we just hit the quote character? */
	if (last_input_screen->quote_hit)
	{
		last_input_screen->quote_hit = 0;
		input_add_character(extended_key, empty_string);
	}
	else
	{
		/* Check to see if this is a meta-key */
		i = last_input_screen->meta_hit;
		if (i >= 0 && i <= MAX_META)
		{
			if (keys[i] && (*keys[i]) && (*keys[i])[key])
			{
			    func = key_names[(*keys[i])[key]->key_index].func;
			    ptr = (*keys[i])[key]->stuff;
			}
		}

		if (last_input_screen->meta_hit != 4)
			last_input_screen->meta_hit = 0;

		/* nope. none of these.  just a regular character */
		if (func)
			func(extended_key, ptr ? ptr : empty_string);
	}
}

BUILT_IN_KEYBINDING(my_scrollback)
{
	scrollback_backwards(key, string);
	extended_handled = 1;
}

BUILT_IN_KEYBINDING(my_scrollforward)
{
	scrollback_forwards(key, string);
	extended_handled = 1;
}

BUILT_IN_KEYBINDING(my_scrollend)
{
	scrollback_end(key, string);
	extended_handled = 1;
}

BUILT_IN_KEYBINDING(do_chelp)
{
	chelp(NULL, "INDEX", NULL, NULL);
	extended_handled = 1;
}

#ifdef WANT_CDCC
BUILT_IN_KEYBINDING(cdcc_plist)
{
	l_plist(NULL, NULL);
	extended_handled = 1;
}
#endif

BUILT_IN_KEYBINDING(dcc_plist)
{
	dcc_glist(NULL, NULL);
	extended_handled = 1;
}

BUILT_IN_KEYBINDING(toggle_cloak)
{
	if (get_int_var(CLOAK_VAR) == 1 || get_int_var(CLOAK_VAR) == 2)
		set_int_var(CLOAK_VAR, 0);
	else
		set_int_var(CLOAK_VAR, 1);
	put_it("CTCP Cloaking is now [\002%s\002]", on_off(get_int_var(CLOAK_VAR)));
	extended_handled = 1;
}


extern int in_window_command;

static void handle_swap(int windownum)
{
char *p = NULL;
	malloc_sprintf(&p, "SWAP %d", windownum);
	windowcmd(NULL, p, NULL, NULL);
	set_channel_window(current_window, get_current_channel_by_refnum(current_window->refnum), current_window->server);
	new_free(&p);
	set_input_prompt(current_window, get_string_var(INPUT_PROMPT_VAR), 0);
	update_input(UPDATE_ALL);
	update_all_windows();
}

BUILT_IN_KEYBINDING(window_swap1)
{
	handle_swap(1);
	extended_handled = 1;
}

BUILT_IN_KEYBINDING(window_swap2)
{
	handle_swap(2);
	extended_handled = 1;
}

BUILT_IN_KEYBINDING(window_swap3)
{
	handle_swap(3);
	extended_handled = 1;
}

BUILT_IN_KEYBINDING(window_swap4)
{
	handle_swap(4);
	extended_handled = 1;
}

BUILT_IN_KEYBINDING(window_swap5)
{
	handle_swap(5);
	extended_handled = 1;
}

BUILT_IN_KEYBINDING(window_swap6)
{
	handle_swap(6);
	extended_handled = 1;
}

BUILT_IN_KEYBINDING(window_swap7)
{
	handle_swap(7);
	extended_handled = 1;
}
BUILT_IN_KEYBINDING(window_swap8)
{
	handle_swap(8);
	extended_handled = 1;
}
BUILT_IN_KEYBINDING(window_swap9)
{
	handle_swap(9);
	extended_handled = 1;
}
BUILT_IN_KEYBINDING(window_swap10)
{
	handle_swap(10);
	extended_handled = 1;
}

BUILT_IN_KEYBINDING(channel_chops)
{
	users("chops", "-ops", NULL, NULL);
	extended_handled = 1;
}

BUILT_IN_KEYBINDING(channel_nonops)
{
	users("nops", "-nonops", NULL, NULL);
	extended_handled = 1;
}

BUILT_IN_KEYBINDING(w_help)
{
	chelp(NULL, "WINDOW", NULL, NULL);
}

BUILT_IN_KEYBINDING(change_to_split)
{
extern char *last_split_server;
	if (!last_split_server)
		return;
	servercmd(NULL, last_split_server, NULL, NULL);
}

BUILT_IN_KEYBINDING(join_last_invite)
{
	if (invite_channel)
	{
		if (!in_join_list(invite_channel, from_server))
			send_to_server("JOIN %s", invite_channel);
		else
			bitchsay("Already joining %s", invite_channel);
	}
	else
		bitchsay("You haven't been invited to a channel yet");
}

BUILT_IN_KEYBINDING(wholeft)
{
	show_wholeft(NULL);
}

BUILT_IN_KEYBINDING(window_key_balance)
{
	in_window_command = 1;
	message_from(NULL, LOG_CURRENT);
	rebalance_windows(current_window->screen);
	update_all_windows();
	in_window_command = 0;
	message_from(NULL, LOG_CRAP);
}

BUILT_IN_KEYBINDING(window_grow_one)
{
	in_window_command = 1;
	message_from(NULL, LOG_CURRENT);
	resize_window(1, current_window, 1);
	update_all_windows();
	in_window_command = 0;
	message_from(NULL, LOG_CRAP);
}

BUILT_IN_KEYBINDING(window_key_hide)
{
	in_window_command = 1;
	message_from(NULL, LOG_CURRENT);
	hide_window(current_window);
	update_all_windows();
	in_window_command = 0;
	message_from(NULL, LOG_CRAP);
}

BUILT_IN_KEYBINDING(window_key_kill)
{
	in_window_command = 1;
	message_from(NULL, LOG_CURRENT);
	delete_window(current_window);
	update_all_windows();
	in_window_command = 0;
	message_from(NULL, LOG_CRAP);
}

BUILT_IN_KEYBINDING(window_key_list)
{
	in_window_command = 1;
	message_from(NULL, LOG_CURRENT);
	window_list(current_window, NULL, NULL);
	in_window_command = 0;
	message_from(NULL, LOG_CRAP);
}

BUILT_IN_KEYBINDING(window_key_move)
{
	in_window_command = 1;
	message_from(NULL, LOG_CURRENT);
	move_window(current_window, 1);
	update_all_windows();
	in_window_command = 0;
	message_from(NULL, LOG_CRAP);
}

BUILT_IN_KEYBINDING(dcc_ostats)
{
	message_from(NULL, LOG_CURRENT);
	extended_handled = 1;
	dcc_stats(NULL, NULL);
	message_from(NULL, LOG_CRAP);
}

BUILT_IN_KEYBINDING(window_shrink_one)
{
	in_window_command = 1;
	message_from(NULL, LOG_CURRENT);
	resize_window(1, current_window, -1);
	update_all_windows();
	in_window_command = 0;
	message_from(NULL, LOG_CRAP);
}

BUILT_IN_KEYBINDING(cpu_saver_on)
{
	cpu_saver = 1;
	update_all_status(current_window, NULL, 0);
}

BUILT_IN_KEYBINDING(input_unclear_screen)
{
	hold_mode(NULL, OFF, 1);
	unclear_window_by_refnum(0);
}

BUILT_IN_KEYBINDING(ignore_last_nick)
{
NickTab *nick;
char *tmp1;
	if ((nick = gettabkey(1, 0, NULL)))
	{
		set_input(empty_string);
		tmp1 = m_sprintf("%sig %s", get_string_var(CMDCHARS_VAR), nick->nick);
		set_input(tmp1);
		new_free(&tmp1);
	}
	update_input(UPDATE_ALL);
}

BUILT_IN_KEYBINDING(nick_completion)
{
char *q, *line;
int i = -1;
char *nick = NULL, *tmp;

	q = line = m_strdup(&last_input_screen->input_buffer[MIN_POS]);
	if (in_completion == STATE_NORMAL)
	{
		i = word_count(line);
		nick = extract(line, i-1, i);
	}
	if (nick)
		line[strlen(line)-strlen(nick)] = 0;
	else
		*line = 0;
	if ((tmp = getchannick(input_lastmsg, nick && *nick ? nick : NULL)))
	{
		malloc_strcat(&q, tmp);
		set_input(q);
		update_input(UPDATE_ALL);
		malloc_strcpy(&input_lastmsg, tmp);
		in_completion = STATE_COMPLETE;
	}
	new_free(&q);
	new_free(&nick);
}

char *getchannick (char *oldnick, char *nick) 
{
ChannelList *chan; 
char *channel, *tnick = NULL; 
NickList *cnick;
	channel = get_current_channel_by_refnum(0);
	if (channel)
	{
		if (!(chan = lookup_channel(channel, from_server, 0)) || !(cnick = next_nicklist(chan, NULL)))
		{
			in_completion = STATE_NORMAL;
			return NULL;
		}
		/* 
		 * we've never been here before so return first nick 
		 * user hasn't entered anything on the line.
		 */
		if (!oldnick && !nick && cnick)
		{
			in_completion = STATE_CNICK;
			return cnick->nick;
		}
		/*
		 * user has been here before so we attempt to find the correct
		 * first nick to start from.
		 */
		if (oldnick)
		{
			/* find the old nick so we have a frame of reference */
			for(cnick = next_nicklist(chan, NULL); cnick; cnick = next_nicklist(chan, cnick))
			{
				if (!my_strnicmp(cnick->nick, oldnick, strlen(oldnick)))
				{
					tnick = cnick->nick;
					if ((cnick = next_nicklist(chan, cnick)))
						tnick = cnick->nick;
					break;
	
				}
				
			}				
		}
		/*
		 * if the user has put something on the line
		 * we attempt to pattern match here.
		 */
		if (nick && in_completion == STATE_NORMAL)
		{
			/* 
			 * if oldnick was the last one in the channel 
			 * cnick will be NULL;
			 */
			if (!cnick)
			{
				cnick = next_nicklist(chan, NULL);
				tnick = cnick->nick;
			}
			/* we have a new nick */
			else if (next_nicklist(chan, cnick))
			{
				/* 
				 * if there's more than one nick, start 
				 * scanning.
				 */
				for (; cnick; cnick = next_nicklist(chan, cnick))
				{
#if 1
					if (!my_strnicmp(cnick->nick, nick, strlen(nick)) || my_strnstr(cnick->nick, nick, strlen(nick)))
#else
					if (!my_strnicmp(cnick->nick, nick, strlen(nick)))
#endif
					{
						tnick = cnick->nick;
						break;
					}
				}
#if 1				
				/* Same as before, make sure bx gets its try w/o ours interfering... */
				
				if (!tnick)
				{
					for (; cnick; cnick = next_nicklist(chan, cnick))
					{
						if (my_strnstr(cnick->nick, nick, strlen(nick)))
						{
							tnick = cnick->nick;
							break;
						}
					}
				}
#endif
			} 
			else
				tnick = cnick->nick;
		} 
		else if (in_completion == STATE_CNICK)
		{
			/*
			 * else we've been here before so
			 * attempt to continue through the nicks 
			 */
			if (!cnick)
				cnick = next_nicklist(chan, NULL);
			tnick = cnick->nick;
		}
	}
	if (tnick)
		in_completion = STATE_CNICK;

	return tnick;
}

/* 
 * set which to 1 to access autoreply array.
 * 0 for msglist array
 */

void addtabkey(char *nick, char *type, int which)
{
NickTab *tmp, *new;

	tmp = (which == 1) ? autoreply_array : tabkey_array;

	if (!tmp || !(new = (NickTab *)remove_from_list((List **)&tmp, nick)))
	{
		new = (NickTab *)new_malloc(sizeof(NickTab));
		malloc_strcpy(&new->nick, nick);
		if (type)
			malloc_strcpy(&new->type, type);
	}
	/*
	 * most recent nick is at the top of the list 
	 */
	new->next = tmp;
	tmp = new;
	if (which == 1)
		autoreply_array = tmp;
	else
		tabkey_array = tmp;	
}


NickTab *gettabkey(int direction, int which, char *nick)
{
NickTab *tmp, *new;


	new = tmp = (which == 1) ? autoreply_array : tabkey_array;

	if (nick)
	{
		for (; tmp; tmp = tmp->next)
			if (!my_strnicmp(nick, tmp->nick, strlen(nick)))
				return tmp;
		return NULL;
	}
	tmp = new;
	if (!tmp)
		return NULL;
		
	switch(direction)
	{
		case 1:
		default:
		{
			/*
			 * need at least two nicks in the list
			 */
			if (new->next)
			{
				/*
				 * reset top of array
				 */
				if (which == 1)
					autoreply_array = new->next;
				else
					tabkey_array = new->next;
				
				/*
				 * set the current nick next pointer to NULL
				 * and then reset top of list.
				 */
				
				new->next = NULL;
				if (which == 1)
					tmp = autoreply_array;
				else
					tmp = tabkey_array;
				
				/*
				 * find the last nick in the list
				 * so we can make the old top pointer 
				 * point to the item
				 */
				while (tmp)
					if (tmp->next)
						tmp = tmp->next;
					else
						break;
				/* set the pointer and then return. */
				tmp->next = new;		
			}
			break;
		}
		case -1:
		{
			if (new && new->next)
			{
				tmp = new;
				while(tmp)
					if (tmp->next && tmp->next->next)
						tmp = tmp->next;
					else
						break;
				/* 
				 * tmp now points at last two items in list 
				 * now just swap some pointers.
				 */	
				new = tmp->next;
				tmp->next = NULL;
				if (which == 1)
				{
					new->next = autoreply_array;
					autoreply_array = new;
				} else {
					new->next = tabkey_array;
					tabkey_array = new;
				}
			}
			break;
		}
	}
	if (new && new->nick)
		return new;
	return NULL;
}

NickTab *getnextnick(int which, char *input_nick, char *oldnick, char *nick)
{
ChannelList *chan; 
NickList *cnick = NULL;
NickTab  *tmp =  (which == 1) ? autoreply_array : tabkey_array;
int server = from_server;
static NickTab sucks = { NULL };

	if (tmp && (in_completion == STATE_NORMAL || in_completion == STATE_TABKEY))
	{
	

		if (!oldnick && !nick && tmp)
		{
			in_completion = STATE_TABKEY;
			return tmp;
		}	
		if (oldnick)
		{
			for (; tmp; tmp = tmp->next)
			{
				if (!my_strnicmp(oldnick, tmp->nick, strlen(oldnick)))
					break;
			}
			/* nick was not in the list. oops didn't come from here */
			if (!tmp && in_completion == STATE_TABKEY)
				tmp = (which == 1) ? autoreply_array : tabkey_array;
			else if (tmp)
				tmp = tmp->next;
		}
		if (nick && in_completion != STATE_TABKEY)
		{
			if (tmp && tmp->next)
			{
				for (; tmp; tmp = tmp->next)
					if (!my_strnicmp(nick, tmp->nick, strlen(nick)))
						break;
			}
		} 
		if (tmp)
		{
			in_completion = STATE_TABKEY;
			return tmp;
		}
	}

	if ((chan = prepare_command(&server, NULL, 3)))
	{
		cnick = next_nicklist(chan, NULL);
		/* 
		 * we've never been here before so return first nick 
		 * user hasn't entered anything on the line.
		 */
		if (!oldnick && !nick && cnick)
		{
			in_completion = STATE_CNICK;
			sucks.nick = cnick->nick;
			return &sucks;
		}
		/*
		 * user has been here before so we attempt to find the correct
		 * first nick to start from.
		 */
		if (oldnick)
		{
			/* find the old nick so we have a frame of reference */
			for (; cnick; cnick = next_nicklist(chan, cnick))
			{
				if (!my_strnicmp(cnick->nick, oldnick, strlen(oldnick)))
				{
					cnick = next_nicklist(chan, cnick);
					break;
				}
			}
		}
		/*
		 * if the user has put something on the line
		 * we attempt to pattern match here.
		 */
		if (input_nick)
		{
			/* 
			 * if oldnick was the last one in the channel 
			 * cnick will be NULL;
			 */
			if (!cnick && oldnick)
				cnick = next_nicklist(chan, NULL);
			/* we have a new nick */
			else if (cnick)
			{
				/* 
				 * if there's more than one nick, start 
				 * scanning.
				 */
				for (;cnick; cnick = next_nicklist(chan, cnick))
				{
					if (!my_strnicmp(cnick->nick, input_nick, strlen(input_nick)) || !strcasecmp(cnick->nick, input_nick))
						break;
				}
			} 
		} 
		else if (in_completion == STATE_CNICK)
		{
			/*
			 * else we've been here before so
			 * attempt to continue through the nicks 
			 */
/*
			if (!cnick)
				cnick = chan->nicks;
*/
		}
	}
	if (!cnick)
		in_completion = STATE_NORMAL;
	else
		in_completion = STATE_CNICK;
	if (cnick)
		sucks.nick = cnick->nick;
	return sucks.nick ? &sucks : NULL;
}
