/*
   Siag, Scheme In A Grid
   Copyright (C) 1996, 1997  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <signal.h>

#include <unistd.h>
#include <sys/wait.h>
#include <sys/stat.h>

#include <X11/cursorfont.h>
#include <X11/keysym.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xatom.h>
#include <X11/Shell.h>	/* for XtNtitle */
#include <X11/Xmu/Atoms.h>
#include <X11/Xmu/StdSel.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>
#include <X11/Xaw/Paned.h>

#include <OffiX/DragAndDrop.h>

#include "../siod/siod.h"

#include "../common/common.h"
#include "../common/cmalloc.h"
#include "../common/fonts.h"

#include "../common/bitmaps/new.xbm"
#include "../common/bitmaps/fld_open.xbm"
#include "../common/bitmaps/floppy3.xbm"
#include "../common/bitmaps/preview.xbm"
#include "../common/bitmaps/printer.xbm"
#include "../common/bitmaps/plotter.xbm"
#include "../common/bitmaps/sigma.xbm"
#include "../common/bitmaps/info.xbm"
#include "../common/bitmaps/copyright.xbm"
#include "../common/bitmaps/bold.xbm"
#include "../common/bitmaps/italic.xbm"
#include "../common/bitmaps/hleft.xbm"
#include "../common/bitmaps/hcenter.xbm"
#include "../common/bitmaps/hright.xbm"
#include "../common/bitmaps/grid.xbm"
#include "../common/bitmaps/borders.xbm"
#include "../common/bitmaps/uline.xbm"
#include "../common/bitmaps/none.xbm"

#include "../xcommon/Table.h"
#include "../xcommon/xfonts.h"
#include "../xcommon/embed.h"
#include "../xcommon/dialogs.h"

#include "../siag/types.h"
#include "../siag/calc.h"

#include "xsiag.h"
#include "icon.h"

String fallback_resources[] = {
#include "app-defaults.h"
	NULL
};

#define HELPTEXT_INTERVAL 500

window *w_list;

int pr_scr_flag;
int grid_lines = 1;

int macro_flag;

textbuf kbd_macro = {0, 0, 0};

static XtAppContext app_context;

Widget topLevel;

static Widget topbox, gridpane,	text1, label1, label2;

static Widget btnFont, mnuFont, btnSize, mnuSize, btnStyle, mnuStyle, btnColor, mnuColor;
static Widget cmdBold, cmdItalic, cmdHLeft, cmdHCenter, cmdHRight;
static Widget cmdBorders, cmdGrid, cmdUline, cmdNone;

static Widget helpshell, helplabel;

static int helptext_mode = 2;	/* popup */

static int helptext_timeout = None;
static char helptext_text[256];

static unsigned long highlight_color;
static unsigned long unhighlight_color;

static Display *display;

static Cursor rownum_cursor;
static Cursor colnum_cursor;

extern int recalc;	/* number of iterations */

enum {ABORT=0, DONE, WAITING};

static int status;
int input_warp_pointer = 1;	/* move pointer to input field */

static void DialogCancelAction(Widget w, XEvent * event, String * params, Cardinal * n)
{
	status = ABORT;
}

static void DialogDoneAction(Widget w, XEvent * event, String * params, Cardinal * n)
{
	status = DONE;
}

static XtActionsRec input_actions[] =
{
	{"dialog-done", DialogDoneAction},
	{"dialog-cancel", DialogCancelAction}
};

static void draw_input(Display * display, char *text)
{
	int r = w_list->point_pos.row, c = w_list->point_pos.col;
	int type = ret_type(w_list->buf, r, c);
	int intp = ret_interpreter(w_list->buf, r, c);
	char b[1024];
	if (type == EXPRESSION || type == STRING)
		sprintf(b, "[%d,%d] [%s] %s",
			r, c, interpreter2name(intp), text);
	else
		sprintf(b, "[%d,%d] %s",
			r, c, text);
	XtVaSetValues(label1, XtNlabel, b, (char *)NULL);
}

static void draw_status(Display * display, char *text)
{
	XtVaSetValues(label2, XtNlabel, text, (char *)NULL);
	XFlush(display);
}

/* Print the string p on the bottom line of the screen.  If p is empty and
   the last string printed was also empty, the string isn't printed. */
void llpr(char *p)
{
	static int isclear = FALSE;

	if (isclear && p[0] == '\0')
		return;
	isclear = (p[0] == '\0');

	draw_status(display, p);
}

static void draw_vbar(Display *display, window *w)
{
	double float_value;

	float_value = w->top.row;
	float_value /= BUFFER_ROWS;

	XawScrollbarSetThumb(w->ui->vscroll, float_value, 0.0);
}

static void draw_hbar(Display *display, window *w)
{
	double float_value;

	float_value = w->top.col;
	float_value /= BUFFER_COLS;

	XawScrollbarSetThumb(w->ui->hscroll, float_value, 0.0);
}

static void draw_scrollbars(Display *display, window *w)
{
	draw_vbar(display, w);
	draw_hbar(display, w);
}

static char *rownum_text(buffer *buf, int row, int col)
{
	static char b[80];
	sprintf(b, "%d", row);
	return b;
}

static char *colnum_text(buffer *buf, int row, int col)
{
	static char b[80];
	sprintf(b, "%d", col);
	return b;
}

static unsigned long rowcol_format(buffer *buf, int row, int col)
{
	return HELVETICA | SIZE_12 | BOLD | HADJ_CENTER;
}

/* 970427: prot */
/* 970812: Table widget */
static void draw_colnums(Display *display, window *w)
{
	XtVaSetValues(w->ui->colnum,
		XtNtableProtCol, w->prot.col,
		XtNtableTopCol, w->top.col,
		XtNtableData, w->buf,
		XtNtableRedisplay, True,
		(char *)NULL);
}

static void draw_rownums(Display * display, window *w)
{
	XtVaSetValues(w->ui->rownum,
		XtNtableProtRow, w->prot.row,
		XtNtableTopRow, w->top.row,
		XtNtableData, w->buf,
		XtNtableRedisplay, True,
		(char *)NULL);
}

/*
How to make Siag always draw the top row and/or the left column.

This is desirable to keep headers on the screen when the sheet
is scrolled. To make this possible, I need to add a field in
the window structure to specify the first unprotected cell.
Everything above and to the left of this cell (called w->prot)
should always be drawed, regardless of what top is.

While this sounds easy, it profoundly changes the mapping from
cell coordinates to screen coordinates and vice versa. For instance,
answering the question "is the cursor on the screen" is no longer
as easy.

Make top be the first unprotected column. Point can never leave
the unprotected area. Leave mark and other positions for now.
*/

static int cell_type(buffer *b, int row, int col)
{

	if (ret_type(b, row, col) == EMBED)
		return TABLE_EMBED;

	return TABLE_TEXT;
}

static char *cell_text(buffer *b, int row, int col)
{
	static char s[1024];
	s[0] = '\0';
	ret_pvalue(s, b, row, col, -1);
	return s;
}

static unsigned long cell_format(buffer *b, int row, int col)
{
	return ret_format(b, row, col);
}

/* 970427: First shot at implementing this protection */
/* 970812: Use the Table widget's own drawing routines */
static void draw_cells(Display *display, window *w)
{
	int top_row, top_col;

	/* this is more verbose than necessary, but I want to set
	   every resource there is just to see if they work ;-) */
	XtVaSetValues(w->ui->grid,
		XtNtableMaxRow, BUFFER_ROWS,
		XtNtableMaxCol, BUFFER_COLS,
		XtNtableProtRow, w->prot.row,
		XtNtableProtCol, w->prot.col,
		XtNtableTopRow, w->top.row,
		XtNtableTopCol, w->top.col,
		XtNtableSelectTopRow, w->blku.row,
		XtNtableSelectBottomRow, w->blkl.row,
		XtNtableSelectLeftCol, w->blku.col,
		XtNtableSelectRightCol, w->blkl.col,
		XtNtablePointRow, w->point_pos.row,
		XtNtablePointCol, w->point_pos.col,
		XtNtableRowHeight, cell_height,
		XtNtableColWidth, cell_width,
		XtNtableType, cell_type,
		XtNtableText, cell_text,
		XtNtableFormat, cell_format,
		XtNtableData, w->buf,
		XtNtableRedisplay, True,
		XtNtableGridLines, grid_lines,
		(char *)NULL);
	/* there is a chance that this moved top */
	XtVaGetValues(w->ui->grid,
		XtNtableTopRow, &top_row,
		XtNtableTopCol, &top_col,
		(char *)NULL);
	if (top_row != w->top.row) {
		w->top.row = top_row;
		XtVaSetValues(w->ui->rownum,
			XtNtableTopRow, top_row,
			XtNtableRedisplay, !pr_scr_flag,
			(char *)NULL);
	}
	if (top_col != w->top.col) {
		w->top.col = top_col;
		XtVaSetValues(w->ui->colnum,
			XtNtableTopCol, top_col,
			XtNtableRedisplay, !pr_scr_flag,
			(char *)NULL);
	}
}

int ask_for_str_comp(char *prompt, char *buffr, int (*comp)(char *))
{
	XtAppContext app_context = XtWidgetToApplicationContext(topLevel);
	String string;

TRACEME((f,"ask_for_str_comp(%s, %s, %p)\n", prompt, buffr, comp));

	status = WAITING;

	XtVaSetValues(label1,
		XtNlabel, prompt, (char *)NULL);
	XtVaSetValues(text1,
		XtNdisplayCaret, True,
		XtNstring, buffr,
		XtNinsertPosition, strlen(buffr), (char *)NULL);

	XtAddGrab(text1, False, False);

	XtSetKeyboardFocus(topLevel, text1);

	while (status == WAITING) {
		XEvent event_return;
		String string;

/* THIS IS INCREDIBLY ANNOYING!!! WHAT IS GOING ON HERE? */
/* (events are getting generated and changing the label of label1) */
		XtVaGetValues(label1,
			XtNlabel, &string, (char *)NULL);

TRACEME((f,"label=\"%s\"\n", string))

		if (strcmp(string, prompt))
			XtVaSetValues(label1,
				XtNlabel, prompt, (char *)NULL);

		XtAppNextEvent(app_context, &event_return);
		XtDispatchEvent(&event_return);
	}

	XtVaGetValues(text1,
		XtNstring, &string, (char *)NULL);
	strcpy(buffr, string);	/* no check on length... */
	XtVaSetValues(text1,
		XtNdisplayCaret, False,
		XtNstring, "", (char *)NULL);

	XtRemoveGrab(text1);

	activate_window(w_list);
	return status;
}

/* This particular completion function doesn't complete at all, it just
   returns TRUE, making TAB equivalent to RET and LFD. */
static int nocomp(char *b)
{
	return TRUE;
}

/* Calls ask_for_str_comp with nocomp as completion function. */
/* 95-06-29: changed "buffer" to "buffr" to please gcc */
int ask_for_str(char *prompt, char *buffr)
{
	return ask_for_str_comp(prompt, buffr, nocomp);
}

int add_str_to_input_queue(textbuf buf)
{
        return FALSE;
}

static void set_block(int row1, int col1, int row2, int col2)
{
	int ur = w_list->blku.row, uc = w_list->blku.col;
	int lr = w_list->blkl.row, lc = w_list->blkl.col;

	if (row1 < row2) {
		w_list->blku.row = row1;
		w_list->blkl.row = row2;
	} else {
		w_list->blku.row = row2;
		w_list->blkl.row = row1;
	}
	if (col1 < col2) {
		w_list->blku.col = col1;
		w_list->blkl.col = col2;
	} else {
		w_list->blku.col = col2;
		w_list->blkl.col = col1;
	}


	/* Redraw if any change */
	if (ur != w_list->blku.row || uc != w_list->blku.col ||
		lr != w_list->blkl.row || lc != w_list->blkl.col) {
		pr_scr_flag = TRUE;
	}


	/* Become selection owner */
	/* this function should be integrated with the one in cmds.c */
	if (XtOwnSelection(w_list->ui->grid, XA_PRIMARY, CurrentTime, convert_proc,
		lose_ownership_proc, NULL) == False) {
		XtWarning("Siag: failed to become selection owner\n");
		w_list->blku.row = w_list->blku.col = -1;
		w_list->blkl.row = w_list->blkl.col = -1;
	}
}

#ifndef ABS
#define ABS(a) ((a)>0?(a):-(a))
#endif

static int colnum_grab(Widget w)
{
	int x = -1;
	int owner_events = True;
	unsigned int event_mask = ButtonReleaseMask |
				PointerMotionMask;
	int pointer_mode = GrabModeAsync;
	int keyboard_mode = GrabModeAsync;
	Window confine_to = XtWindow(w);
	static Cursor cursor = None;
	Time time = CurrentTime;
	int waiting = True;
	if (cursor == None)
		cursor = XCreateFontCursor(display, XC_right_side);
	XtGrabPointer(w, owner_events, event_mask,
		pointer_mode, keyboard_mode, confine_to, cursor, time);
	while (waiting) {
		XEvent event_return;
		XtAppNextEvent(app_context, &event_return);
		if (event_return.type == ButtonRelease) {
			waiting = False;
		} else if (event_return.type == MotionNotify) {
			x = event_return.xmotion.x;
		} else {
			XtDispatchEvent(&event_return);
		}
	}
	XtUngrabPointer(w, CurrentTime);
	return x;
}

static void ColnumButtonAction(Widget w,
		XEvent *event, String *params, Cardinal *n)
{
	int col, row, x, y;

	x = event->xbutton.x;
	y = 0;
	hide_cur(w_list);
	activate_window(find_window_by_widget(w));
	get_coords_cell(w_list, w_list->top.row, w_list->top.col, 
			&row, &col, x, y);
	if (*n < 1 || !strcmp(params[0], "set")) {
		/* If we are within 4 pixels from a cell boundary, grab the
		   pointer and move the border to wherever we release it. */
		int x1, y1, w1;
		get_cell_coords(w_list, w_list->top.row, w_list->top.col,
			row, col, &x1, &y1);
		w1 = x-x1;
		if (cell_width(w_list->buf, col)-w1 < 5) {
			x = colnum_grab(w);
		w1 = x-x1;
			if (w1 > 5 && w1 < 500)
				set_width(w_list->buf, col, w1);
		} else {
		/* Otherwise do everything as before */
			set_point_col(w_list, col);
			set_block(1, col, BUFFER_ROWS, col);
		}
		pr_scr_flag = TRUE;

	}
	else if (!strcmp(params[0], "left"))
		w_list->blku.col = col;
	else if (!strcmp(params[0], "right"))
		w_list->blkl.col = col;
	else if (!strcmp(params[0], "select")) {
		int r1 = w_list->blku.row, c1 = w_list->blku.col;
		int r2 = w_list->blkl.row, c2 = w_list->blkl.col;
		/* is the selection set to something already? */
		if (r1 == -1 || c1 == -1 || r2 == -1 || c2 == -1)
			set_block(1, col,
				BUFFER_ROWS,
				get_point(w_list).col);
		else {
			/* which corner is closer? */
			if (ABS(c1-col) < ABS(c2-col))
				set_block(1, col, BUFFER_ROWS, c2);
			else
				set_block(1, c1, BUFFER_ROWS, col);
		}
	}
	else if (!strcmp(params[0], "adjust")) {
		set_block(1, col,
			BUFFER_ROWS,
			get_point(w_list).col);
	}
	show_cur(w_list);
}

static int rownum_grab(Widget w)
{
	int y = -1;
	int owner_events = True;
	unsigned int event_mask = ButtonReleaseMask |
				PointerMotionMask;
	int pointer_mode = GrabModeAsync;
	int keyboard_mode = GrabModeAsync;
	Window confine_to = XtWindow(w);
	static Cursor cursor = None;
	Time time = CurrentTime;
	int waiting = True;
	if (cursor == None)
		cursor = XCreateFontCursor(display, XC_bottom_side);
	XtGrabPointer(w, owner_events, event_mask,
		pointer_mode, keyboard_mode, confine_to, cursor, time);
	while (waiting) {
		XEvent event_return;
		XtAppNextEvent(app_context, &event_return);
		if (event_return.type == ButtonRelease) {
			waiting = False;
		} else if (event_return.type == MotionNotify) {
			y = event_return.xmotion.y;
		} else {
			XtDispatchEvent(&event_return);
		}
	}
	XtUngrabPointer(w, CurrentTime);
	return y;
}

static void RownumButtonAction(Widget w,
		XEvent * event, String * params, Cardinal * n)
{
	int col, row, x, y;

	x = 0;
	y = event->xbutton.y;
	hide_cur(w_list);
	activate_window(find_window_by_widget(w));
	get_coords_cell(w_list, w_list->top.row, w_list->top.col, 
			&row, &col, x, y);
	if (*n < 1 || !strcmp(params[0], "set")) {
		/* If we are within 4 pixels from a cell boundary, grab the
		   pointer and move the border to wherever we release it. */
		int x1, y1, h1;
		get_cell_coords(w_list, w_list->top.row, w_list->top.col,
			row, col, &x1, &y1);
		h1 = y-y1;
		if (cell_height(w_list->buf, row)-h1 < 5) {
			y = rownum_grab(w);
		h1 = y-y1;
			if (h1 > 5 && h1 < 500)
				set_height(w_list->buf, row, h1);
		} else {
		/* Otherwise do everything as before */
			set_point_row(w_list, row);
			set_block(row, 1, row, BUFFER_COLS);
		}
		pr_scr_flag = TRUE;

	}
	else if (!strcmp(params[0], "top"))
		w_list->blku.row = row;
	else if (!strcmp(params[0], "bottom"))
		w_list->blkl.row = row;
	else if (!strcmp(params[0], "select")) {
		int r1 = w_list->blku.row, c1 = w_list->blku.col;
		int r2 = w_list->blkl.row, c2 = w_list->blkl.col;
		/* is the selection set to something already? */
		if (r1 == -1 || c1 == -1 || r2 == -1 || c2 == -1)
			set_block(row, 1,
				get_point(w_list).row,
				BUFFER_COLS);
		else {
			/* which corner is closer? */
			if (ABS(r1-row) < ABS(r2-row))
				set_block(row, 1, r2, BUFFER_COLS);
			else
				set_block(r1, 1, row, BUFFER_COLS);
		}
	}
	else if (!strcmp(params[0], "adjust")) {
		set_block(row, 1,
			get_point(w_list).row,
			BUFFER_COLS);
	}
	show_cur(w_list);
}

#define CTRL(c) ((c)&31)
#define CONTROL_MASK 4
#define ALT(c) ((c)|0x80)
#define ALT_MASK 8

static void KeyEventAction(Widget w,
		XEvent *event, String *params, Cardinal *n)
{
	int count, bufsiz = 10;
	char buf[12];
	KeySym keysym;
	XKeyEvent kevent;
	kevent = event->xkey;
	count = XLookupString(&kevent, buf, bufsiz, &keysym, NULL);

	lastc = keysym;
	switch (lastc) {
	case XK_Home:		lastc = CTRL('a'); break;
	case XK_Left:		lastc = CTRL('b'); break;
	case XK_Delete:		lastc = CTRL('d'); break;
	case XK_Up:		lastc = CTRL('p'); break;
	case XK_Right:		lastc = CTRL('f'); break;
	case XK_Down:		lastc = CTRL('n'); break;
	case XK_Page_Up:	lastc = ALT('v'); break;
	case XK_Page_Down:	lastc = CTRL('v'); break;
	case XK_End:		lastc = CTRL('e'); break;
	default:		if (lastc > 255) return;
	}
	if (kevent.state & CONTROL_MASK) lastc = CTRL(lastc);
	if (kevent.state & ALT_MASK) lastc = ALT(lastc);
	do_cmd(lastc);
}

static void GridButtonAction(Widget w,
		XEvent * event, String * params, Cardinal * n)
{
	int col, row;
	int x, y;

	x = event->xbutton.x;
	y = event->xbutton.y;
	hide_cur(w_list);
	activate_window(find_window_by_widget(w));
	get_coords_cell(w_list, w_list->top.row, w_list->top.col, 
			&row, &col, x, y);
	if (*n < 1 || !strcmp(params[0], "point")) {
		set_point_row(w_list, row);
		set_point_col(w_list, col);
	} else if (!strcmp(params[0], "mark")) {
		set_mark_row(w_list, row);
		set_mark_col(w_list, col);
	} else if (!strcmp(params[0], "block")) {
		set_block(row, col,
			get_mark(w_list).row, get_mark(w_list).col);
		pr_scr_flag = TRUE;
	}
	else if (!strcmp(params[0], "paste")) {
		set_point_row(w_list, row);
		set_point_col(w_list, col);
		XtGetSelectionValue(w, XA_PRIMARY, target_atom,
			requestor_callback, event, event->xbutton.time);
		pr_scr_flag = TRUE;
	}
	else if (!strcmp(params[0], "select")) {
		int r1 = w_list->blku.row, c1 = w_list->blku.col;
		int r2 = w_list->blkl.row, c2 = w_list->blkl.col;
		/* is the selection set to something already? */
		if (r1 == -1 || c1 == -1 || r2 == -1 || c2 == -1)
			set_block(row, col,
				get_point(w_list).row,
				get_point(w_list).col);
		else {
			/* which corner is closer? */
			if (ABS(r1-row) < ABS(r2-row)) r1 = r2;
			if (ABS(c1-col) < ABS(c2-col)) c1 = c2;
			set_block(r1, c1, row, col);
		}
	}
	else if (!strcmp(params[0], "adjust")) {
		set_block(row, col,
			get_point(w_list).row,
			get_point(w_list).col);
	}
	show_cur(w_list);
}

static void highlight_action(Widget w, XEvent *event,
	String *params, Cardinal *num_params)
{
	if (w) XtVaSetValues(w,
		XtNbackground, highlight_color,
		(char *)NULL);
}

static void unhighlight_action(Widget w, XEvent *event,
	String *params, Cardinal *num_params)
{
	if (w) XtVaSetValues(w,
		XtNbackground, unhighlight_color,
		(char *)NULL);
}

static void helptext_popup(XtPointer client_data, XtIntervalId *id)
{
	helptext_timeout = None;
	XtPopup(helpshell, XtGrabNone);
}

static void helptext_popdown()
{
	if (helptext_timeout != None)
		XtRemoveTimeOut(helptext_timeout);
	helptext_timeout = None;
	XtPopdown(helpshell);
}

static void draw_input_action(Widget w, XEvent *event,
	String *params, Cardinal *num_params)
{
	if (helptext_mode & 1) {	/* label */
		char *p = ret_text(w_list->buf,
			w_list->point_pos.row, w_list->point_pos.col);
		if (p == NULL)
			p = "";
		draw_input(display, p);
	}
	if (helptext_mode & 2) {	/* popup */
		helptext_popdown();
	}
}

static void helptext_action(Widget w, XEvent *event,
	String *params, Cardinal *num_params)
{
	int i;

	strncpy(helptext_text, params[0], 255);
	for (i = 1; i < *num_params; i++) {
		strncat(helptext_text, " ", 255);
		strncat(helptext_text, params[i], 255);
	}
	if (helptext_mode & 1) {	/* label */
		XtVaSetValues(label1, XtNlabel, helptext_text, (char *)NULL);
	}
	if (helptext_mode & 2) {	/* popup */
		Position x, y;
		Dimension height, width;
		XFontStruct *font;

		XtVaGetValues(w,
			XtNheight, &height, (char *)NULL);
		XtTranslateCoords(w,
			0, height+10, &x, &y);
		XtVaGetValues(helplabel,
			XtNfont, &font, (char *)NULL);
		width = XTextWidth(font, helptext_text, strlen(helptext_text));
		XtVaSetValues(helpshell,
			XtNx, x,
			XtNy, y,
			XtNwidth, width+8, (char *)NULL);
		XtVaSetValues(helplabel, XtNlabel, helptext_text, (char *)NULL);
		helptext_timeout = XtAppAddTimeOut(app_context,
					HELPTEXT_INTERVAL,
					helptext_popup, (XtPointer)w);
	}
}

static LISP lhelptext_mode(LISP newmode)
{
	draw_input_action(NULL, NULL, NULL, NULL);	/* well it works */
	helptext_mode = get_c_long(newmode);
	return NIL;
}

static void siaghelp_action(Widget w, XEvent *event,
	String *params, Cardinal *num_params)
{
	char b[256];

	sprintf(b, "file://localhost%s/siag/docs/%s", siaghome, params[0]);
	if (!fork()) {
		execlp(siaghelp, "Siaghelp", "-geometry", "600x400", b, (char *)NULL);
		exit(0);
	}
}

static XtActionsRec actions[] =
{
	{"key-event", KeyEventAction},
	{"grid-button", GridButtonAction},
	{"colnum-button", ColnumButtonAction},
	{"rownum-button", RownumButtonAction},
	{"siag-highlight", highlight_action},
	{"siag-unhighlight", unhighlight_action},
	{"helptext", helptext_action},
	{"draw-input", draw_input_action},
	{"siaghelp", siaghelp_action}
};

static void execute_callback(Widget w,
		XtPointer client_data, XtPointer call_data)
{
	execute((char *)client_data);
}

static void vscroll_jump(Widget w, XtPointer client_data, XtPointer call_data)
{
	float top;
	int gridtop;
	position p;

	hide_cur(w_list);
	activate_window(find_window_by_widget(w));
	XtVaGetValues(w, XtNtopOfThumb, &top, (char *)NULL);
	gridtop = top*BUFFER_ROWS;
	if (gridtop < 1) gridtop = 1;
	/* make sure protection is respected */
	/* is it necessary to move point here? */
	p.row = gridtop;
	p.col = w_list->point_pos.col;
	set_point(w_list, p);
	p.col = w_list->top.col;
	set_top(w_list, p);
	pr_scr_flag = TRUE;
	show_cur(w_list);
}

static void vscroll_scroll(Widget w, XtPointer client_data, XtPointer call_data)
{
	int i = (int) call_data;
	Dimension length;

	if (i == 0) return;	/* no div by zero */
	activate_window(find_window_by_widget(w));
	XtVaGetValues(w, XtNlength, &length, (char *)NULL);
	if (i < 0) {
		if ((length / -i) > 15)
			execute("(scroll-cell-down)");
		else
			execute("(scroll-down)");
	} else {
		if ((length / i) > 15)
			execute("(scroll-cell-up)");
		else
			execute("(scroll-up)");
	}
}

static void hscroll_jump(Widget w, XtPointer client_data, XtPointer call_data)
{
	float top;
	int gridtop;
	position p;

	hide_cur(w_list);
	activate_window(find_window_by_widget(w));
	XtVaGetValues(w, XtNtopOfThumb, &top, (char *)NULL);
	gridtop = top*BUFFER_COLS;
	if (gridtop < 1) gridtop = 1;
	p.col = gridtop;
	p.row = w_list->point_pos.row;
	set_point(w_list, p);
	p.row = w_list->top.row;
	set_top(w_list, p);
	pr_scr_flag = TRUE;
	show_cur(w_list);
}

static void hscroll_scroll(Widget w, XtPointer client_data, XtPointer call_data)
{
	int i = (int) call_data;
	Dimension length;

	if (i == 0) return;	/* no div by zero */
	activate_window(find_window_by_widget(w));
	XtVaGetValues(w, XtNlength, &length, (char *)NULL);
	if (i < 0) {
		if ((length / -i) > 15)
			execute("(scroll-cell-left)");
		else
			execute("(scroll-left)");
	} else {
		if ((length / i) > 15)
			execute("(scroll-cell-right)");
		else
			execute("(scroll-right)");
	}
}

static struct {
	char *label;
	Widget button, menu;
} menubar[10];

static menucount = 0;

static void make_menu(char *label)
{
	char button_name[80];

	sprintf(button_name, "btn%s", label);
	menubar[menucount].label = cstrdup(label);
	menubar[menucount].button = XtVaCreateManagedWidget(button_name,
		menuButtonWidgetClass, topbox,
		XtNlabel, label,
		XtNtop, XawChainTop,
		XtNleft, XawChainLeft,
		XtNright, XawChainLeft,
		XtNbottom, XawChainTop, (char *)NULL);
	if (menucount) {
		XtVaSetValues(menubar[menucount].button,
			XtNfromHoriz, menubar[menucount-1].button, (char *)NULL);
	}
	menubar[menucount].menu = XtVaCreatePopupShell("menu",
		simpleMenuWidgetClass, menubar[menucount].button, (char *)NULL);

	menucount++;
}

static void init_menu()
{
	make_menu("File");
	make_menu("Edit");
	make_menu("Block");
	make_menu("Format");
	make_menu("Data");
	make_menu("Window");
	make_menu("Plot");
	make_menu("Tools");
	make_menu("Help");
}

static Widget find_menu_by_name(char *label)
{
	int i;

	for (i = 0; i < menucount; i++) {
		if (!cstrcasecmp(menubar[i].label, label))
			return menubar[i].menu;
	}
	return NULL;
}

static LISP add_menu_entry(LISP menu, LISP label, LISP function)
{
	Widget entry;
	Widget menuw = find_menu_by_name(get_c_string(menu));
	if (!menuw) {
		fprintf(stderr, "No menu! Bailing out.\n");
		return NIL;
	}

	if (!strcmp(get_c_string(label), "-")) {	/* line pane */
		entry = XtCreateManagedWidget("-",
				smeLineObjectClass,
				menuw,
				NULL, 0);
	} else {
		entry = XtVaCreateManagedWidget(get_c_string(function),
				smeBSBObjectClass, menuw,
				XtNlabel, get_c_string(label),
				(char *)NULL);
		XtAddCallback(entry,
			XtNcallback, execute_callback, cstrdup(get_c_string(function)));
	}
	return NIL;
}

static Widget tbNew, tbCopy;	/* needed as fromVert and fromHoriz */

static Widget make_toggle(char *name, char *cmd, Widget pw, 
	Widget below, Widget after,
	char *bits, int width, int height)
{
	Pixel fg, bg;
	Display *display = XtDisplay(pw);

	Widget w = XtVaCreateManagedWidget(name,
		toggleWidgetClass, pw, (char *)NULL);
	XtVaGetValues(w,
		XtNforeground, &fg,
		XtNbackground, &bg, (char *)NULL);
	XtVaSetValues(w,
		XtNfromVert, below,
		XtNtop, XawChainTop,
		XtNleft, XawChainLeft,
		XtNright, XawChainLeft,
		XtNbottom, XawChainTop,
		XtNbitmap,
		XCreatePixmapFromBitmapData(display,
			XDefaultRootWindow(display),
			bits, width, height, fg, bg,
			XDefaultDepthOfScreen(XtScreen(w))), NULL);
	if (after) {
		XtVaSetValues(w,
			XtNfromHoriz, after, (char *)NULL);
	}
	XtAddCallback(w,
		XtNcallback, execute_callback, (XtPointer)cmd);
	return w;
}

static Widget make_command(char *name, char *cmd, Widget pw, 
	Widget below, Widget after,
	char *bits, int width, int height)
{
	Pixel fg, bg;
	Display *display = XtDisplay(pw);

	Widget w = XtVaCreateManagedWidget(name,
		commandWidgetClass, pw, (char *)NULL);
	XtVaGetValues(w,
		XtNforeground, &fg,
		XtNbackground, &bg, (char *)NULL);
	XtVaSetValues(w,
		XtNfromVert, below,
		XtNtop, XawChainTop,
		XtNleft, XawChainLeft,
		XtNright, XawChainLeft,
		XtNbottom, XawChainTop,
		XtNbitmap,
		XCreatePixmapFromBitmapData(display,
			XDefaultRootWindow(display),
			bits, width, height, fg, bg,
			XDefaultDepthOfScreen(XtScreen(w))), NULL);
	if (after) {
		XtVaSetValues(w,
			XtNfromHoriz, after, (char *)NULL);
	}
	XtAddCallback(w,
		XtNcallback, execute_callback, (XtPointer)cmd);
	return w;
}

/* The toolbar */

#define BELOW_WIDGET menubar[0].button
static void init_toolbar()
{
	Widget tbOpen, tbSave, tbView, tbPrint, tbPlot, tbSum, tbHelp;

	tbNew = make_command("tbNew", "(new-siag)", topbox,
		BELOW_WIDGET, NULL,
		(char *)new_bits, new_width, new_height);
	tbOpen = make_command("tbOpen", "(load-buffer)", topbox,
		BELOW_WIDGET, tbNew,
		(char *)fld_open_bits, fld_open_width, fld_open_height);
	tbSave = make_command("tbSave", "(save-buffer-as)", topbox,
		BELOW_WIDGET, tbOpen,
		(char *)floppy3_bits, floppy3_width, floppy3_height);
	tbView = make_command("tbView", "(preview)", topbox,
		BELOW_WIDGET, tbSave,
		(char *)preview_bits, preview_width, preview_height);
	tbPrint = make_command("tbPrint", "(print)", topbox,
		BELOW_WIDGET, tbView,
		(char *)printer_bits, printer_width, printer_height);
	tbPlot = make_command("tbPlot", "(plot \"lines\")", topbox,
		BELOW_WIDGET, tbPrint,
		(char *)plotter_bits, plotter_width, plotter_height);
	tbSum = make_command("tbSum", "(block-sum)", topbox,
		BELOW_WIDGET, tbPlot,
		(char *)sigma_bits, sigma_width, sigma_height);
	tbHelp = make_command("tbHelp", "(help-contents)", topbox,
		BELOW_WIDGET, tbSum,
		(char *)info_bits, info_width, info_height);
	tbCopy = make_command("tbCopy", "(help-copyright)", topbox,
		BELOW_WIDGET, tbHelp,
		(char *)copyright_bits, copyright_width, copyright_height);
}
#undef BELOW_WIDGET

static void init_toggle()
{
	cmdBold = make_toggle("cmdBold", "(toggle-fontflag BOLD)", topbox,
		tbNew, btnColor,
		(char *)bold_bits, bold_width, bold_height);
	cmdItalic = make_toggle("cmdItalic", "(toggle-fontflag ITALIC)", topbox,
		tbNew, cmdBold,
		(char *)italic_bits, italic_width, italic_height);
	cmdHLeft = make_toggle("cmdHLeft", "(change-font HADJ_LEFT HADJ_MASK)",
		topbox, tbNew, cmdItalic,
		(char *)hleft_bits, hleft_width, hleft_height);
	cmdHCenter = make_toggle("cmdHCenter", "(change-font HADJ_CENTER HADJ_MASK)",
		topbox, tbNew, cmdHLeft,
		(char *)hcenter_bits, hcenter_width, hcenter_height);
	cmdHRight = make_toggle("cmdHRight", "(change-font HADJ_RIGHT HADJ_MASK)",
		topbox, tbNew, cmdHCenter,
		(char *)hright_bits, hright_width, hright_height);
	cmdBorders = make_command("cmdBorders", "(block-borders 1)", topbox,
		menubar[0].button, tbCopy,
		(char *)borders_bits, borders_width, borders_height);
	cmdGrid = make_command("cmdGrid", "(block-borders 2)", topbox,
		menubar[0].button, cmdBorders,
		(char *)grid_bits, grid_width, grid_height);
	cmdUline = make_command("cmdUline", "(block-borders 3)", topbox,
		menubar[0].button, cmdGrid,
		(char *)uline_bits, uline_width, uline_height);
	cmdNone = make_command("cmdNone", "(block-borders 0)", topbox,
		menubar[0].button, cmdUline,
		(char *)none_bits, none_width, none_height);
}

static void font_menu(Widget w, char *label, char *cmd)
{
	Widget entry = XtVaCreateManagedWidget(cmd,
			smeBSBObjectClass, w,
			XtNlabel, label,
			(char *)NULL);
	XtAddCallback(entry,
		XtNcallback, execute_callback, cmd);
}

static void setup_buttons()
{
	btnFont = XtVaCreateManagedWidget("btnFont",
		menuButtonWidgetClass, topbox,
		XtNtop, XawChainTop,
		XtNleft, XawChainLeft,
		XtNright, XawChainLeft,
		XtNbottom, XawChainTop,
		XtNfromVert, tbNew,
		XtNwidth, 160,
		XtNlabel, "Font", (char *)NULL);

	mnuFont = XtVaCreatePopupShell("menu",
		simpleMenuWidgetClass, btnFont, (char *)NULL);

	font_menu(mnuFont, "Courier",
		"(change-font COURIER FONT_MASK)");
	font_menu(mnuFont, "Helvetica",
		"(change-font HELVETICA FONT_MASK)");
	font_menu(mnuFont, "New Century Schoolbook",
		"(change-font NEW_CENTURY FONT_MASK)");
	font_menu(mnuFont, "Times",
		"(change-font TIMES FONT_MASK)");

	btnSize = XtVaCreateManagedWidget("btnSize",
		menuButtonWidgetClass, topbox,
		XtNtop, XawChainTop,
		XtNleft, XawChainLeft,
		XtNright, XawChainLeft,
		XtNbottom, XawChainTop,
		XtNfromVert, tbNew,
		XtNfromHoriz, btnFont,
		XtNwidth, 40,
		XtNlabel, "Size", (char *)NULL);

	mnuSize = XtVaCreatePopupShell("menu",
		simpleMenuWidgetClass, btnSize, (char *)NULL);

	font_menu(mnuSize, "8",
		"(change-font SIZE_8 SIZE_MASK)");
	font_menu(mnuSize, "10",
		"(change-font SIZE_10 SIZE_MASK)");
	font_menu(mnuSize, "12",
		"(change-font SIZE_12 SIZE_MASK)");
	font_menu(mnuSize, "14",
		"(change-font SIZE_14 SIZE_MASK)");
	font_menu(mnuSize, "18",
		"(change-font SIZE_18 SIZE_MASK)");
	font_menu(mnuSize, "20",
		"(change-font SIZE_20 SIZE_MASK)");
	font_menu(mnuSize, "24",
		"(change-font SIZE_24 SIZE_MASK)");
	font_menu(mnuSize, "30",
		"(change-font SIZE_30 SIZE_MASK)");

	btnStyle = XtVaCreateManagedWidget("btnStyle",
		menuButtonWidgetClass, topbox,
		XtNtop, XawChainTop,
		XtNleft, XawChainLeft,
		XtNright, XawChainLeft,
		XtNbottom, XawChainTop,
		XtNfromVert, tbNew,
		XtNfromHoriz, btnSize,
		XtNwidth, 80,
		XtNlabel, "Style", (char *)NULL);

	mnuStyle = XtVaCreatePopupShell("menu",
		simpleMenuWidgetClass, btnStyle, (char *)NULL);

	font_menu(mnuStyle, "Default",
		"(change-font FMT_DEFAULT FMT_MASK)");
	font_menu(mnuStyle, "Invisible",
		"(change-font FMT_INVISIBLE FMT_MASK)");
	font_menu(mnuStyle, "Integer",
		"(change-font FMT_INTEGER FMT_MASK)");
	font_menu(mnuStyle, "Scientific",
		"(change-font FMT_SCIENTIFIC FMT_MASK)");
	font_menu(mnuStyle, "Fixed",
		"(change-font FMT_FIXED FMT_MASK)");
	font_menu(mnuStyle, "Date",
		"(change-font FMT_DATE FMT_MASK)");
	font_menu(mnuStyle, "Time",
		"(change-font FMT_TIME FMT_MASK)");
	font_menu(mnuStyle, "Comma",
		"(change-font FMT_COMMA FMT_MASK)");
	font_menu(mnuStyle, "Percent",
		"(change-font FMT_PERCENT FMT_MASK)");
	font_menu(mnuStyle, "Hex",
		"(change-font FMT_HEX FMT_MASK)");
	font_menu(mnuStyle, "Currency",
		"(change-font FMT_CURRENCY FMT_MASK)");
	font_menu(mnuStyle, "User 1",
		"(change-font FMT_USER1 FMT_MASK)");
	font_menu(mnuStyle, "User 2",
		"(change-font FMT_USER2 FMT_MASK)");
	font_menu(mnuStyle, "User 3",
		"(change-font FMT_USER3 FMT_MASK)");
	font_menu(mnuStyle, "User 4",
		"(change-font FMT_USER4 FMT_MASK)");
	font_menu(mnuStyle, "User 5",
		"(change-font FMT_USER5 FMT_MASK)");

	btnColor = XtVaCreateManagedWidget("btnColor",
		menuButtonWidgetClass, topbox,
		XtNtop, XawChainTop,
		XtNleft, XawChainLeft,
		XtNright, XawChainLeft,
		XtNbottom, XawChainTop,
		XtNfromVert, tbNew,
		XtNfromHoriz, btnStyle,
		XtNwidth, 80,
		XtNlabel, "Color", (char *)NULL);

	mnuColor = XtVaCreatePopupShell("menu",
		simpleMenuWidgetClass, btnColor, (char *)NULL);

	font_menu(mnuColor, "Black",
		"(change-font COLOR_BLACK COLOR_MASK)");
	font_menu(mnuColor, "Red",
		"(change-font COLOR_RED COLOR_MASK)");
	font_menu(mnuColor, "Green",
		"(change-font COLOR_GREEN COLOR_MASK)");
	font_menu(mnuColor, "Blue",
		"(change-font COLOR_BLUE COLOR_MASK)");
	font_menu(mnuColor, "Yellow",
		"(change-font COLOR_YELLOW COLOR_MASK)");
	font_menu(mnuColor, "Magenta",
		"(change-font COLOR_MAGENTA COLOR_MASK)");
	font_menu(mnuColor, "Cyan",
		"(change-font COLOR_CYAN COLOR_MASK)");
	font_menu(mnuColor, "White",
		"(change-font COLOR_WHITE COLOR_MASK)");
}

static void drop_handler(Widget w, XtPointer data,
		XEvent *event, Boolean *b)
{
	unsigned char *Data;
	unsigned char *filename;
	unsigned long DataSize;
	int DataType = DndDataType(event);

	if (DataType == DndNotDnd) return;
	DndGetData(&Data, &DataSize);
	switch (DataType) {
	case DndRawData:
		/* This should allow dnd within siag(s) */
		break;
	case DndFile:
		/* load file */
		filename = Data;
		if (!fork()) {
			execlp("siag", "Siag", filename, (char *)NULL);
			exit(1);
		}
		break;
	case DndFiles:
		/* load several files */
		filename = Data;
		while (filename[0] != '\0') {
			if (!fork()) {
				execlp("siag", "Siag", filename, (char *)NULL);
				exit(1);
			}
			filename = filename+strlen((char *)filename)+1;
		}
		break;
	case DndText:
		/* insert the text at the drop point */
		break;
	case DndDir:
		/* chdir to Dir and start another Siag */
		filename = Data;
		if (!fork()) {
			chdir((char *)filename);
			execlp("siag", "Siag", (char *)NULL);
			exit(1);
		}
		break;
	case DndLink:
		/* why would anyone want this?! a stale link to nowhere */
		break;
	case DndExe:
		/* possible use: run the program and load the output */
		break;
	case DndURL:
		/* load the document using ftpget or httpget */
		break;
	case DndMIME:
		/* some more conversion needed */
		break;
	case DndUnknown:
		/* according to the docs, this should be treated as Raw */
		/* I think it's better to just bail out */
		break;
	default:
		/* something's wrong */
		;
	}
}

static int abort_siag(Display *foo, XErrorEvent *bar)
{
	abort();
	return 0;
}

void mainloop()
{
	/* keep X from terminating gracefully upon fatal error */
	XSetErrorHandler(abort_siag);

	XtAppMainLoop(app_context);

	exit(0);
}

void init_windows1(int *argc, char **argv)
{
	XtSetLanguageProc(NULL, (XtLanguageProc) NULL, NULL);

	topLevel = XtVaAppInitialize(
		    &app_context,	/* application context */
		    "Siag",		/* application class */
		    NULL, 0,		/* command line options list */
		    argc, argv,		/* command line args */
		    fallback_resources,	/* for missing app-defaults file */
		    (char *)NULL);	/* terminate varargs list */

	XtAppAddActions(app_context, actions, XtNumber(actions));

	topbox = XtCreateManagedWidget("topbox",
		formWidgetClass, topLevel, NULL, 0);

	init_menu();

	init_toolbar();
	setup_buttons();
	init_toggle();

	helpshell = XtVaCreatePopupShell("helpshell",
		overrideShellWidgetClass, topLevel, (char *)NULL);
	helplabel = XtVaCreateManagedWidget("helplabel",
		labelWidgetClass, helpshell, (char *)NULL);

	label1 = XtVaCreateManagedWidget("label1",
		labelWidgetClass, topbox,
		XtNtop, XawChainTop,
		XtNleft, XawChainLeft,
		XtNright, XawChainRight,
		XtNbottom, XawChainTop,
		XtNfromVert, btnFont,
		XtNwidth, 600, (char *)NULL);

	XtAppAddActions(app_context, input_actions, XtNumber(input_actions));

	text1 = XtVaCreateManagedWidget("text1",
		asciiTextWidgetClass, topbox,
		XtNeditType, XawtextEdit,
		XtNdisplayCaret, False,
		XtNtop, XawChainTop,
		XtNleft, XawChainLeft,
		XtNright, XawChainRight,
		XtNbottom, XawChainTop,
		XtNfromVert, label1,
		XtNwidth, 600, (char *)NULL);

	gridpane = XtVaCreateManagedWidget("gridpane",
		panedWidgetClass, topbox,
		XtNtop, XawChainTop,
		XtNleft, XawChainLeft,
		XtNright, XawChainRight,
		XtNbottom, XawChainBottom,
		XtNfromVert, text1,
		XtNallowResize, True,
		XtNmin, 40,
		XtNwidth, 600,
		XtNheight, 200, (char *)NULL);

	label2 = XtVaCreateManagedWidget("label2",
		labelWidgetClass, topbox,
		XtNtop, XawChainBottom,
		XtNleft, XawChainLeft,
		XtNright, XawChainRight,
		XtNbottom, XawChainBottom,
		XtNfromVert, gridpane,
		XtNwidth, 600, (char *)NULL);
}

void activate_window(window *w)
{
	char b[256];

	if (w_list)
		XtVaSetValues(w_list->ui->grid,
			XtNtableVisibleCursor, False,
			(char *)NULL);
	w_list = w;
	strcpy(b, "Siag: ");
	strncat(b, w->buf->name, 200);
	XtVaSetValues(topLevel, XtNtitle, b, (char *)NULL);
	XtVaSetValues(w->ui->grid, XtNtableVisibleCursor, True, (char *)NULL);
	XtSetKeyboardFocus(topLevel, w->ui->grid);
}

Pixmap draw_snapshot()
{
	int x, y;
	unsigned int width, height, border_width, depth;
	Window cell_win, root;
	Pixmap bitmap;
	GC gc;
	unsigned long valuemask = 0;
	XGCValues values;

	cell_win = XtWindow(w_list->ui->grid);
	XGetGeometry(display, cell_win, &root,
		&x, &y, &width, &height, &border_width, &depth);
	bitmap = XCreatePixmap(display, cell_win, width, height, 1);
	gc = XCreateGC(display, bitmap, valuemask, &values);
	XCopyPlane(display, cell_win, bitmap, gc, 0, 0, width, height,
			0, 0, 1);
	XFreeGC(display, gc);

	return bitmap;
}

window *find_window_by_widget(Widget wdg)
{
	window *w = w_list;
	do {
		if (w->ui->viewport == wdg || w->ui->selectall == wdg ||
			w->ui->colnum == wdg || w->ui->rownum == wdg ||
			w->ui->grid == wdg || w->ui->vscroll == wdg ||
			w->ui->hscroll == wdg)
			return w;
		w = w->next;
	} while (w != w_list);
	return NULL;
}

void free_window(window *w)
{
	window *pw;

	for (pw = w_list; pw->next != w && pw->next != pw; pw = pw->next);
	pw->next = w->next;

	if (w_list == w) w_list = w_list->next;
	if (w_list == w) w_list = NULL;
	XtDestroyWidget(w->ui->viewport);
	cfree(w->ui);
	cfree(w);
}

static int width(Widget w)
{
	Dimension x;

	XtVaGetValues(w, XtNwidth, &x, (char *)NULL);
	return x;
}

static int height(Widget w)
{
	Dimension x;

	XtVaGetValues(w, XtNheight, &x, (char *)NULL);
	return x;
}

window *new_window(buffer *b, window *prev)
{
	Dimension totalwidth, formheight, w1, h1;
	int d;	/* distance between widgets in form */
	window *w;

	w = (window *)cmalloc(sizeof(window));

	if (w == NULL) return NULL;

	w->ui = (siag_ui *)cmalloc(sizeof(siag_ui));
	if (w->ui == NULL) {
		cfree(w);
		return NULL;
	}

	w->buf = b;
	w->point_pos.row = 1;
	w->point_pos.col = 1;
	w->blku.row = w->blku.col = 2;
	w->blkl.row = w->blkl.col = 1;
	w->prot = w->top = w->point_pos;

	if (prev == NULL) prev = w;
	else w->next = prev->next;
	prev->next = w;

	/* Figure out how big the new form should be.			*/
	/* The total width must be the width of the gridpane.		*/
	XtVaGetValues(gridpane,
		XtNwidth, &totalwidth, (char *)NULL);

TRACEME((f,"Total width: %d\n", totalwidth))

	/* The form height is whatever we get, but if it is too small
	we cannot create the new window.				*/
	formheight = 100;
	w->ui->viewport = XtVaCreateManagedWidget("viewport",
		formWidgetClass, gridpane,
		XtNwidth, totalwidth,
		/*XtNdefaultDistance, 0,*/
		/*XtNheight, formheight, */
		(char *)NULL);
	XtVaGetValues(w->ui->viewport,
		XtNheight, &formheight, XtNdefaultDistance, &d, (char *)NULL);

TRACEME((f,"Form height: %d\n", formheight))

	/* The selectall button should get its size from the ad */
	w->ui->selectall = XtVaCreateManagedWidget("selectall",
		commandWidgetClass, w->ui->viewport, (char *)NULL);
	XtAddCallback(w->ui->selectall, XtNcallback,
			execute_callback, "(select-all)");

	/* The colnum should take up all the space between the button
	and vscroll							*/
	w1 = totalwidth-width(w->ui->selectall)-20-4*d;	/* just guessing */
	h1 = height(w->ui->selectall);

TRACEME((f,"colnum size: w=%d h=%d\n", w1, h1))

	w->ui->colnum = XtVaCreateManagedWidget("colnum",
		tableWidgetClass, w->ui->viewport,
		XtNwidth, w1, XtNheight, h1,
		XtNtableDefaultHeight, h1,
		XtNtableColWidth, cell_width,
		XtNtableFormat, rowcol_format,
		XtNtableText, colnum_text,
		XtNtableMaxRow, 1,
		XtNtableMaxCol, BUFFER_COLS,
		XtNfromHoriz, w->ui->selectall, (char *)NULL);
/*	XDefineCursor(display, XtWindow(w->ui->colnum), colnum_cursor);
*/
	w1 = width(w->ui->selectall);
	h1 = formheight-height(w->ui->selectall)-20-4*d;

TRACEME((f,"rownum size: w=%d h=%d\n", w1, h1))

	w->ui->rownum = XtVaCreateManagedWidget("rownum",
		tableWidgetClass, w->ui->viewport,
		XtNwidth, w1, XtNheight, h1,
		XtNtableDefaultWidth, w1,
		XtNtableRowHeight, cell_height,
		XtNtableFormat, rowcol_format,
		XtNtableText, rownum_text,
		XtNtableMaxRow, BUFFER_ROWS,
		XtNtableMaxCol, 1,
		XtNfromVert, w->ui->selectall, (char *)NULL);
/*	XDefineCursor(display, XtWindow(w->ui->rownum), rownum_cursor);
*/
	w1 = width(w->ui->colnum);
	h1 = height(w->ui->rownum);

TRACEME((f,"grid size: w=%d h=%d\n", w1, h1))

	w->ui->grid = XtVaCreateManagedWidget("grid",
		tableWidgetClass, w->ui->viewport,
		XtNwidth, w1, XtNheight, h1,
		XtNfromHoriz, w->ui->selectall,
		XtNfromVert, w->ui->selectall, (char *)NULL);
	h1 = height(w->ui->colnum)+height(w->ui->grid)+d;
	w->ui->vscroll = XtVaCreateManagedWidget("vscroll",
		scrollbarWidgetClass, w->ui->viewport,
		XtNheight, h1,
		XtNfromHoriz, w->ui->colnum, (char *)NULL);
	w1 = width(w->ui->rownum)+width(w->ui->grid)+d;
	w->ui->hscroll = XtVaCreateManagedWidget("hscroll",
		scrollbarWidgetClass, w->ui->viewport,
		XtNwidth, w1,
		XtNfromVert, w->ui->rownum, (char *)NULL);
	XtAddCallback(w->ui->vscroll, XtNjumpProc, vscroll_jump, NULL);
	XtAddCallback(w->ui->vscroll, XtNscrollProc, vscroll_scroll, NULL);

	XtAddCallback(w->ui->hscroll, XtNjumpProc, hscroll_jump, NULL);
	XtAddCallback(w->ui->hscroll, XtNscrollProc, hscroll_scroll, NULL);

	XtVaGetValues(w->ui->viewport,
		XtNheight, &formheight, (char *)NULL);

TRACEME((f,"Form height: %d\n", formheight))

	return w;
}

int remove_window(window *w)
{
	
	if (w == w->next) return FALSE;
	free_window(w);
	return TRUE;
}

int split_window(window *w)
{
	window *w2 = new_window(w->buf, w);

	if (w2 == NULL) return FALSE;
	w2->point_pos = w->point_pos;
	w2->top = w->top;
	w2->prot = w->prot;
	return TRUE;
}

/*
   Set up the whole initial window structure and initialize scrupd.
   The window list w_list is set to a list with a single window with the
   buffer b.
*/
/*
 * This function is by no means as transparent as I wold like it to be.
 * There are some very complex interactions going on, many of which
 * span across functions and even libraries. And I frankly don't have
 * a firm grip on them myself.
 * 
 * Here is how things *should* work
 * 1. Create toplevel window, menu et al (init_windows1)
 * 2. Initialize input (req. topLevel). Sets up dialog actions
 * 3. Create text widget and the rest of the main widgets (init_windows2)
 * 4. Initialize parsers? Is it necessary to do that here?
 * 5. Create a dummy buffer
 * 6. Call init_windows (req. a buffer)
 */
void init_windows(buffer * b, int argc, char **argv)
{
	char *p;

	unsigned long foreground;	/* pixel value */
	unsigned long background;
	Window cell_win;
	Atom wm_delete_window;	/* Atom sent to destroy a window */

	XtRealizeWidget(topLevel);

	display = XtDisplay(topLevel);

	rownum_cursor = XCreateFontCursor(display, XC_sb_right_arrow);
	colnum_cursor = XCreateFontCursor(display, XC_sb_down_arrow);

	activate_window(new_window(b, NULL));

	wm_delete_window = XInternAtom(display, "WM_DELETE_WINDOW", False);
	XtOverrideTranslations(topLevel,
		XtParseTranslationTable(
			"<Message>WM_PROTOCOLS: execute(quit-siag)"));

	XSetWMProtocols(display, XtWindow(topLevel), &wm_delete_window, 1);

	init_color(display);

	foreground = BlackPixelOfScreen(XtScreen(topLevel));
	cell_win = XtWindow(w_list->ui->grid);

	XtVaGetValues(w_list->ui->colnum,
		XtNbackground, &background, (char *)NULL);
   	XtVaGetValues(w_list->ui->rownum,
		XtNbackground, &background, (char *)NULL);
   	XtVaGetValues(w_list->ui->grid,
		XtNbackground, &background, (char *)NULL);
	draw_scrollbars(display, w_list);
	p = ret_text(w_list->buf,
		      w_list->point_pos.row, w_list->point_pos.col);
	if (p == NULL)
		p = "";
	draw_input(display, p);
	draw_status(display, "");
	draw_cells(display, w_list);
	draw_colnums(display, w_list);
	draw_rownums(display, w_list);

	/* Set up selection */
	target_atom = XInternAtom(display, "SIAG_BLOCK", False);

        embed_init(topLevel);

        init_subr_3("add-menu-entry", add_menu_entry);
        init_subr_1("helptext-mode", lhelptext_mode);

	init_calc_cmds();

        init_form(topLevel);

        ReadIcons(topLevel);
        DndInitialize(topLevel);        /* initialize drag and drop */
        DndRegisterOtherDrop(drop_handler);     /* in any window */

        font_init(topLevel);

   	highlight_color = WhitePixel(display, DefaultScreen(display));
	XtVaGetValues(label1,	/* or any old widget with a grey background */
		XtNbackground, &unhighlight_color,
		(char *)NULL);

	fsel_init(topLevel);	/* so how come the siag_format hack works? */
}

/* Clean up before exit.  All buffers and windows are freed. */
void exit_windows()
{
	/* free all buffers */
	while (b_list != NULL)
		free_buffer(b_list);
	while (w_list != NULL)
		free_window(w_list);
}

/*
   Prints and refreshes all the windows.
   Sets pr_scr_flag to FALSE.
   970422: also recalculate if needed
*/
static void pr_scr()
{
	window *w;
	buffer *b;

	draw_status(display, "");
	b = b_list;
	do {
		if (b->recalc) {
			int i;
			b->recalc = 0;
			for (i = 0; i < recalc; i++)
				calc_matrix(b);
		}
		b = b->next;
	} while (b != b_list);

	w = w_list;
	do {
		draw_cells(display, w);
		draw_colnums(display, w);
		draw_rownums(display, w);
		draw_scrollbars(display, w);
		w = w->next;
	} while (w != w_list);
	pr_scr_flag = FALSE;
}	/* pr_scr */

/* 970427: the following four functions facilitate navigating in
	the cell area with protection implemented */

static int cell_next_row(window *w, int row)
{
	if (row+1 == w->prot.row) return w->top.row;
	return row+1;
}

static int cell_next_col(window *w, int col)
{
	if (col+1 == w->prot.col) return w->top.col;
	return col+1;
}

static int cell_prev_row(window *w, int row)
{
	if (row == w->top.row) return w->prot.row-1;
	return row-1;
}

static int cell_prev_col(window *w, int col)
{
	if (col == w->top.col) return w->prot.col-1;
	return col-1;
}

/* From (row, col), calculate (x, y) coordinates. This is a little tricker
	now. Rather than starting the calculation in (0, 0) we must
	take the protected cells into account. */
void get_cell_coords(window *w, int top_row, int top_col,
		     int cell_row, int cell_col,
		     int *cell_x, int *cell_y)
{
	int i;

	*cell_y = 0;
	for (i = 1; i < w->prot.row; i++)
		*cell_y += cell_height(w->buf, i);

	while (cell_row < top_row) {
		cell_row = cell_next_row(w, cell_row);
		*cell_y -= cell_height(w->buf, cell_row);
	}
	while (cell_row > top_row) {
		cell_row = cell_prev_row(w, cell_row);
		*cell_y += cell_height(w->buf, cell_row);
	}
	*cell_x = 0;
	for (i = 1; i < w->prot.col; i++)
		*cell_x += cell_width(w->buf, i);

	while (cell_col < top_col) {
		cell_col = cell_next_col(w, cell_col);
		*cell_x -= cell_width(w->buf, cell_col);
	}
	while (cell_col > top_col) {
		cell_col = cell_prev_col(w, cell_col);
		*cell_x += cell_width(w->buf, cell_col);
	}
}

void get_coords_cell(window *w, int top_row, int top_col,
		     int *cur_row, int *cur_col,
		     int cur_x, int cur_y)
{
	int prot_x = 0, prot_y = 0, i;

	for (i = 1; i < w->prot.col; i++)
		cur_x -= cell_width(w->buf, i);
	for (i = 1; i < w->prot.row; i++)
		cur_y -= cell_height(w->buf, i);

	*cur_row = top_row;
	*cur_col = top_col;
	while (cur_y < prot_y && *cur_row > 1) {
		cur_y += cell_height(w->buf, *cur_row);
		(*cur_row) = cell_prev_row(w, *cur_row);
	}
	while (cur_y > cell_height(w->buf, *cur_row) && *cur_row < BUFFER_ROWS) {
		cur_y -= cell_height(w->buf, *cur_row);
		(*cur_row) = cell_next_row(w, *cur_row);
	}
	while (cur_x < prot_x && *cur_col > 1) {
		cur_x += cell_width(w->buf, *cur_col);
		(*cur_col) = cell_prev_col(w, *cur_col);
	}
	while (cur_x > cell_width(w->buf, *cur_col) && *cur_col < BUFFER_COLS) {
		cur_x -= cell_width(w->buf, *cur_col);
		(*cur_col) = cell_next_col(w, *cur_col);
	}
}

void show_format()
{
	static int last_fmt = -1;
	int fmt = ret_format(w_list->buf,
				w_list->point_pos.row,
				w_list->point_pos.col);
	int bold = fmt & BOLD;
	int italic = fmt & ITALIC;
	int hadj = fmt & HADJ_MASK;
	int hadj_left = (hadj == HADJ_LEFT);
	int hadj_center = (hadj == HADJ_CENTER);
	int hadj_right = (hadj == HADJ_RIGHT);

	if (fmt != last_fmt) {
		/* menus */
		XtVaSetValues(btnFont,
			XtNlabel, family2name(fmt), (char *)NULL);
		XtVaSetValues(btnSize,
			XtNlabel, size2name(fmt), (char *)NULL);
		XtVaSetValues(btnStyle,
			XtNlabel, style2name(fmt), (char *)NULL);
		XtVaSetValues(btnColor,
			XtNlabel, color2name(fmt), (char *)NULL);
	}

	last_fmt = fmt;

	/* toggle buttons */
	XtVaSetValues(cmdBold,
		XtNstate, (bold?1:0), (char *)NULL);
	XtVaSetValues(cmdItalic,
		XtNstate, (italic?1:0), (char *)NULL);
	XtVaSetValues(cmdHLeft,
		XtNstate, (hadj_left?1:0), (char *)NULL);
	XtVaSetValues(cmdHCenter,
		XtNstate, (hadj_center?1:0), (char *)NULL);
	XtVaSetValues(cmdHRight,
		XtNstate, (hadj_right?1:0), (char *)NULL);
}

int cursor_visible = FALSE;

/*
   void show_cur(window *w)
   Moves the cursor to reflect the position of point in w.
   If point is not visible, the window is moved so that point is in
   the middle of the screen.
*/
void show_cur(window *w)
{
	char *p;
	int top_row = w->top.row;
	int top_col = w->top.col;

	XtVaSetValues(w->ui->grid,
		XtNtablePointRow, w->point_pos.row,
		XtNtablePointCol, w->point_pos.col,
		XtNtableVisibleCursor, True,
		(char *)NULL);

	if (pr_scr_flag) {
		pr_scr();
	} else {
#if 1
/*
FIXME
This causes confusion when the grid regains focus after having lost
the selection: the colnum widget isn't updated. But if I move this
check to immediately after the point is moved, a newly protected
window is drawn incorrectly: top is apparently not moved at all,
so the protected cells are drawn twice.
Must get back to this!
*/
		/* this may have moved the top, so we must check that */
		XtVaGetValues(w->ui->grid,
			XtNtableTopRow, &(w->top.row),
			XtNtableTopCol, &(w->top.col),
			(char *)NULL);
#endif
		if (top_row != w->top.row) {
			draw_rownums(display, w);
			draw_vbar(display, w);
		}
		if (top_col != w->top.col) {
			draw_colnums(display, w);
			draw_hbar(display, w);
		}
	}
	p = ret_text(w->buf, w->point_pos.row,
		      w->point_pos.col);
	if (p == NULL)
		p = "";
	draw_input(display, p);
	show_format();
	cursor_visible = TRUE;
}	/* show_cur */

void hide_cur(window *w)
{
/*	XtVaSetValues(w->ui->grid,
		XtNtableVisibleCursor, False,
		(char *)NULL);
	cursor_visible = FALSE;
*/
}

