/*
   Siag, Scheme In A Grid
   Copyright (C) 1996, 1997  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * ci.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <locale.h>

#include "../common/cmalloc.h"
#include "../common/common.h"

#include "types.h"
#include "calc.h"

/* C expression evaluator */
#define READCHAR (c=*calcexpr++)
#define READ_RETURN(t) return(READCHAR,(t))

#define EMPTY_STACK (opind == 0)
#define PUSH(e) (opstack[++opind] = (e))
#define POP (opstack[opind--])
#define TOP (opstack[opind])
#define PUSHRESSTACK(x) (resstack[++resind] = (x))

#define ISOPERATOR prio

typedef enum { ADD = 1, PLUS, ADD_ASS, INC,
        SUB, MINUS, SUB_ASS, DEC,
        MULT, MULT_ASS, DIV, DIV_ASS,
        MOD, MOD_ASS, RSHIFT, RSHIFT_ASS,
        LSHIFT, LSHIFT_ASS, BAND, BAND_ASS,
        LAND, BOR, BOR_ASS, LOR,
        XOR, XOR_ASS, LNOT, BNOT,
        ASSIGN, PTRTOMB, MEMBER, LT,
        LE, GT, GE, EQUAL,
        NE, LPAR, RPAR, LBRACK,
        RBRACK, LBRACE, RBRACE, NUMBER,
        ROW, COLUMN, QMARK, COLON,
        COMMA, SEMICOLON, AT_SIGN, IDENT,
        SIZEOF, FOR, WHILE, IF,
        ELSE, DO, SWITCH, RETURN,
        BREAK, CONTINUE, GOTO, NOSYMBOL,
        END
} opcode;

static char *calcexpr;			/* the expression being calculated */
static int calcrow, calccol;		/* the cell being calculated */

static int c;			/* the last character read */

static int prio[END + 1];
static int isinfix[END + 1];

static int opind, resind;
static int opstack[40];
static double resstack[40];

static double symbol_val;

typedef enum {
	REF_START, REF_R, REF_DIG1, REF_C, REF_DIG2, REF_LIMBO, REF_ALNUM
} ref_state;

#define BREAKCHARS " +-*/%=(),;:!&|^<>?\t\r\n"

/* move block between (r1,c1) and (r2,c2), direction (rd,cd) */
static char *update_C_references(char *expr, int r1, int c1,
				int r2, int c2, int rd, int cd)
{
	char *p;
	int rc = ref_counter(expr);

	if (!rc) return expr;

	/* rough upper bound on new length. A few bytes extra don't matter */
	/* strlen("r100000c100000")-strlen("r1c1") = 10 */
	p = cmalloc(strlen(expr)+10*rc+1);
	rc = ref_updater(expr, p, BREAKCHARS, r1, c1, r2, c2, rd, cd);
	if (rc) {
		expr = cstrdup(p);
	}
	cfree(p);
	return expr;
}

/* the new scanner */
static int next_symbol(buffer *b)
{
	char *p;
	int lineno;
	char tokentext[256];
	char *endptr;

Restart:	/* restart efter comment */
	while (isspace(c)) {
		if (c == '\n')
			lineno++;
		READCHAR;
	}
	switch (c) {
	case ',':
		READ_RETURN(COMMA);
	case '=':
		if (READCHAR)
			READ_RETURN(EQUAL);
		return ASSIGN;
	case '+':
		switch (READCHAR) {
		case '=':
			READ_RETURN(ADD_ASS);
		case '+':
			READ_RETURN(INC);
		default:
			return ADD;
		}
	case '-':
		switch (READCHAR) {
		case '=':
			READ_RETURN(SUB_ASS);
		case '-':
			READ_RETURN(DEC);
		case '>':
			READ_RETURN(PTRTOMB);
		default:
			return SUB;
		}
	case '*':
		if (READCHAR == '=')
			READ_RETURN(MULT_ASS);
		return MULT;
	case '/':
		switch (READCHAR) {
		case '=':
			READ_RETURN(DIV_ASS);
		case '*':
			READCHAR;
			while (c != '*' || READCHAR != '/') {
				if (c == '\0') {
					errorflag = TRUE;
					return NOSYMBOL;
				}
				if (c != '*')
					READCHAR;
			}
			READCHAR;
			goto Restart;
		default:
			return DIV;
		}
	case '%':
		if (READCHAR == '=')
			READ_RETURN(MOD_ASS);
		return MOD;
	case '>':
		switch (READCHAR) {
		case '>':
			if (READCHAR == '=')
				READ_RETURN(RSHIFT_ASS);
			return RSHIFT;
		case '=':
			READ_RETURN(GE);
		default:
			return GT;
		}
	case '<':
		switch (READCHAR) {
		case '<':
			if (READCHAR == '=')
				READ_RETURN(LSHIFT_ASS);
			return LSHIFT;
		case '=':
			READ_RETURN(LE);
		default:
			return LT;
		}
	case '&':
		switch (READCHAR) {
		case '=':
			READ_RETURN(BAND_ASS);
		case '&':
			READ_RETURN(LAND);
		default:
			return BAND;
		}
	case '|':
		switch (READCHAR) {
		case '=':
			READ_RETURN(BOR_ASS);
		case '|':
			READ_RETURN(LOR);
		default:
			return BOR;
		}
	case '^':
		if (READCHAR == '=')
			READ_RETURN(XOR_ASS);
		return XOR;
	case '?':
		READ_RETURN(QMARK);
	case ':':
		READ_RETURN(COLON);
	case '!':
		if (READCHAR == '=')
			READ_RETURN(NE);
		return LNOT;
	case '~':
		READ_RETURN(BNOT);
	case '(':
		READ_RETURN(LPAR);
	case ')':
		READ_RETURN(RPAR);
	case '[':
		READ_RETURN(LBRACK);
	case ']':
		READ_RETURN(RBRACK);
	case '{':
		READ_RETURN(LBRACE);
	case '}':
		READ_RETURN(RBRACE);
	case ';':
		READ_RETURN(SEMICOLON);
	case '@':
		READ_RETURN(AT_SIGN);
	case '\0':
		return END;
	case '0':
		/* hack to not treat 0.1 as octal 0 followed by error */
		if (*calcexpr == '.') {
			symbol_val = strtod(calcexpr, &endptr);
		} else {
			calcexpr--;
			symbol_val = strtol(calcexpr, &endptr, 0);
		}
		if (calcexpr == endptr)
			return NOSYMBOL;
		calcexpr = endptr;
		READCHAR;
		return NUMBER;
	case '.':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
		/* read decimal number */
		calcexpr--;
		symbol_val = strtod(calcexpr, &endptr);
		if (calcexpr == endptr)	/* no conversion performed */
			return NOSYMBOL;
		calcexpr = endptr;
		READCHAR;
		return NUMBER;
	default:		/* read identifier or error */
		p = tokentext;
		while (isalnum(c) || c == '_' || c == '.') {
			*p++ = c;
			READCHAR;
		}
		*p = '\0';
		if (!cstrcasecmp(tokentext, "R"))
			return ROW;
		if (!cstrcasecmp(tokentext, "C"))
			return COLUMN;

#if 1	/* let SIOD take care of everything */

	/* preprocessor hack to evaluate functions */
		if (c == '(') {
			char expr[1024];
			int par = 1;
			READCHAR;
			expr[0] = '(';
			strcpy(expr+1, tokentext);
			p = expr+strlen(expr);
			*p++ = ' ';
			while (par > 0) {
				if (!c) return NOSYMBOL;
				if (c == ',') *p++ = ' ';
				else *p++ = c;
				if (c == '(') par++;
				if (c == ')') par--;
				READCHAR;
			}
			*p = '\0';
/*fprintf(stderr, "expr = '%s'\n", expr);
*/
			symbol_val = parse_expr(b, siod_interpreter, expr,
						calcrow, calccol).number;
		} else { /* let SIOD evaluate it as a symbol */
			symbol_val = parse_expr(b, siod_interpreter, tokentext,
						calcrow, calccol).number;
		}
		if (siag_type == EXPRESSION) return NUMBER;
#else	/* handle it ourselves */
		/* Handle Visicalc style references */
		if (*tokentext == 'R' || *tokentext == 'r') {
			long row, col;
			char *p = tokentext;
			p++;	/* skip past 'r' */
			row = strtol(p, &p, 10);
			if (*p != 'C' && *p != 'c') return NOSYMBOL;
			if (row < 1 || row > BUFFER_ROWS) return NOSYMBOL;
			p++;	/* skip past 'c' */
			col = strtol(p, &p, 10);
			if (*p) return NOSYMBOL;
			if (col < 1 || col > BUFFER_COLS) return NOSYMBOL;
			symbol_val = ret_val(b, row, col).number;
			return NUMBER;
		}
#endif
		return IDENT;
	}
	return NOSYMBOL;
}

static void 
calculate(buffer *b, int op)
{
	register int i;
	register double sum;
	register double tmp;
	long n1, n2;

	switch (op) {
	case SEMICOLON:
		sum = 0;
		tmp = resstack[resind--];
		for (i = resstack[resind]; i <= tmp; i++)
			sum += ret_val(b, i, calccol).number;
		resstack[resind] = sum;
		break;
	case COLON:
		sum = 0;
		tmp = resstack[resind--];
		for (i = resstack[resind]; i <= tmp; i++)
			sum += ret_val(b, calcrow, i).number;
		resstack[resind] = sum;
		break;
	case COMMA:
		tmp = resstack[resind--];
		resstack[resind] =
			ret_val(b, (int)resstack[resind], (int)tmp).number;
		break;
	case LOR:
		n1 = resstack[resind--];
		n2 = resstack[resind];
		resstack[resind] = n2 || n1;
		break;
	case LAND:
		n1 = resstack[resind--];
		n2 = resstack[resind];
		resstack[resind] = n2 && n1;
		break;
	case BOR:
		n1 = resstack[resind--];
		n2 = resstack[resind];
		resstack[resind] = n2 | n1;
		break;
	case XOR:
		n1 = resstack[resind--];
		n2 = resstack[resind];
		resstack[resind] = n2 ^ n1;
		break;
	case BAND:
		n1 = resstack[resind--];
		n2 = resstack[resind];
		resstack[resind] = n2 & n1;
		break;
	case EQUAL:
		tmp = resstack[resind--];
		resstack[resind] = resstack[resind] == tmp;
		break;
	case NE:
		tmp = resstack[resind--];
		resstack[resind] = resstack[resind] != tmp;
		break;
	case LT:
		tmp = resstack[resind--];
		resstack[resind] = resstack[resind] < tmp;
		break;
	case LE:
		tmp = resstack[resind--];
		resstack[resind] = resstack[resind] <= tmp;
		break;
	case GT:
		tmp = resstack[resind--];
		resstack[resind] = resstack[resind] > tmp;
		break;
	case GE:
		tmp = resstack[resind--];
		resstack[resind] = resstack[resind] >= tmp;
		break;
	case RSHIFT:
		n1 = resstack[resind--];
		n2 = resstack[resind];
		resstack[resind] = n2 >> n1;
		break;
	case LSHIFT:
		n1 = resstack[resind--];
		n2 = resstack[resind];
		resstack[resind] = n2 << n1;
		break;
	case ADD:
		tmp = resstack[resind--];
		resstack[resind] += tmp;
		break;
	case SUB:
		tmp = resstack[resind--];
		resstack[resind] -= tmp;
		break;
	case MULT:
		tmp = resstack[resind--];
		resstack[resind] *= tmp;
		break;
	case DIV:
		tmp = resstack[resind--];
		resstack[resind] /= tmp;
		break;
	case MOD:
		n1 = resstack[resind--];
		n2 = resstack[resind];
		resstack[resind] = n2 % n1;
		break;
	case LNOT:
		resstack[resind] = !resstack[resind];
		break;
	case BNOT:
		n1 = resstack[resind];
		resstack[resind] = ~n1;
		break;
	case PLUS:
		break;
	case MINUS:
		resstack[resind] = -resstack[resind];
		break;
	default:
		errorflag = TRUE;
	}
	if (resind < 1)
		errorflag = TRUE;
}				/* calculate */

static int operand(buffer *b)
/* Non-zero return value means something went wrong */
{
	register int op;

      Operand:
	if (errorflag)
		return TRUE;
	switch (next_symbol(b)) {
	case NUMBER:
		PUSHRESSTACK(symbol_val);
		break;
	case ROW:
		PUSHRESSTACK((double) calcrow);
		break;
	case COLUMN:
		PUSHRESSTACK((double) calccol);
		break;
	case LPAR:
		PUSH(LPAR);
		goto Operand;
	case LNOT:
		PUSH(LNOT);
		goto Operand;
	case BNOT:
		PUSH(BNOT);
		goto Operand;
	case SUB:
		PUSH(MINUS);
		goto Operand;
	case ADD:
		PUSH(PLUS);
		goto Operand;
	default:
		return errorflag = TRUE;
	}

      Operator:
	if (errorflag)
		return TRUE;
	op = next_symbol(b);
      Restart:
	if (errorflag)
		return TRUE;
	if (isinfix[op]) {
		if (EMPTY_STACK || prio[op] > prio[TOP]) {
			PUSH(op);
			goto Operand;
		} else {
			calculate(b, POP);
			goto Restart;
		}
	} else if (op == END) {
		while (!EMPTY_STACK) {
			if (ISOPERATOR[TOP])
				calculate(b, POP);
			else {
				return errorflag = TRUE;
			}
		}
		return FALSE;
	} else if (op == RPAR) {
		if (EMPTY_STACK) {
			return errorflag = TRUE;
		}
		while (TOP != LPAR) {
			calculate(b, POP);
			if (EMPTY_STACK) {
				return errorflag = TRUE;
			}
		}
		(void)POP;
		goto Operator;
	}
	return errorflag = TRUE;
}

/* End of C expression evaluator */

static cval parse_C_expr(buffer *b, char *expr, int row, int col)
{
	cval value;
	calcexpr = expr;
	READCHAR;
	calcrow = row;
	calccol = col;

	siag_type = EXPRESSION;
	opind = resind = 0;
	/* index 0 will never contain anything, it's used as a safety margin */
	operand(b);
	if (errorflag) siag_type = ERROR;
	value.number = resstack[1];
	return value;
}

static void exec_C_expr(char *s)
{
	if (ok2print) hide_cur(w_list);
	parse_C_expr(buffer_of_window(w_list), s, get_point(w_list).row,
			get_point(w_list).col);
	if (ok2print) show_cur(w_list);
}


int init_C_parser()
{
	static int p = 0;

	setlocale(LC_NUMERIC, "C");

	prio[LPAR] = p++;
	prio[SEMICOLON] = prio[COLON] = prio[COMMA] = p++;
	prio[LOR] = p++;
	prio[LAND] = p++;
	prio[BOR] = p++;
	prio[XOR] = p++;
	prio[BAND] = p++;
	prio[EQUAL] = prio[NE] = p++;
	prio[LT] = prio[LE] = prio[GT] = prio[GE] = p++;
	prio[RSHIFT] = prio[LSHIFT] = p++;
	prio[ADD] = prio[SUB] = p++;
	prio[MULT] = prio[DIV] = prio[MOD] = p++;
	prio[LNOT] = prio[BNOT] = prio[PLUS] = prio[MINUS] = p++;

	isinfix[LPAR] = isinfix[SEMICOLON] = isinfix[COLON] = isinfix[COMMA] =
		isinfix[LOR] = isinfix[LAND] = isinfix[BOR] = isinfix[XOR] = isinfix[BAND] =
		isinfix[EQUAL] = isinfix[NE] =
		isinfix[LT] = isinfix[LE] = isinfix[GT] = isinfix[GE] =
		isinfix[RSHIFT] = isinfix[LSHIFT] =
		isinfix[ADD] = isinfix[SUB] = isinfix[MULT] = isinfix[DIV] = isinfix[MOD] =
		isinfix[LNOT] = isinfix[BNOT] = isinfix[PLUS] = isinfix[MINUS] = TRUE;

	return register_interpreter("C", parse_C_expr, exec_C_expr,
				update_C_references); /* no exec */
}

