fisher.test <- function(x, y = NULL, alternative = "two.sided")
{
  DNAME <- deparse(substitute(x))  
  if (is.matrix(x)) {
    if (any(dim(x) < 2))
      stop("x must have at least 2 rows and columns")
    if (any(x < 0) || any(is.na(x))) 
      stop("all entries of x must be nonnegative and finite")
  }
  else {
    if (is.null(y)) 
      stop("if x is not a matrix, y must be given")
    if (length(x) != length(y)) 
      stop("x and y must have the same length")
    DNAME <- paste(DNAME, "and", deparse(substitute(y)))
    OK <- complete.cases(x, y)
    x <- as.factor(x[OK])
    y <- as.factor(y[OK])
    if ((nlevels(x) < 2) || (nlevels(y) < 2)) 
      stop("x and y must have at least 2 levels")
    x <- table(x, y)
  }

  if (any(dim(x) != c(2, 2)))
    stop("Sorry, only 2 by 2 tables are currently implemented")

  CHOICES <- c("two.sided", "less", "greater")
  alternative <- CHOICES[pmatch(alternative, CHOICES)]
  if (length(alternative) > 1 || is.na(alternative)) 
    stop("alternative must be \"two.sided\", \"less\" or \"greater\"")  

  m <- sum(x[, 1])
  n <- sum(x[, 2])
  k <- sum(x[1, ])
  x <- x[1, 1]
  PVAL <- switch(alternative,
		 less = phyper(x, m, n, k),
		 greater = 1 - phyper(x - 1, m, n, k),
		 two.sided = {
		   eps <- 10^(-6)
		   if ((PVAL <- phyper(x, m, n, k)) < .5)
		     v <- phyper(0:k, m, n, k)
		   else {
		     v <- 1 - phyper(0:k, m, n, k)
		     PVAL <- 1 - phyper(x - 1, m, n, k)
		   }
		   min(1, PVAL + max(v[v <= (1 + eps) * PVAL]))
		 })

  structure(list(p.value = PVAL,
		 alternative = alternative,
		 method = "Fisher's Exact Test for Count Data",
		 data.name = DNAME),
	    class = "htest")
}
