/*           
     YABASIC ---  a tiny integrated Basic Compiler/Interpreter
     written by Marc-Oliver Ihm in 1995-98.
     e-mail: ihm@kph.uni-mainz.de

     Current Version:
*/
#define BASIC_VERSION            "2.13"
/*

     Date of last change: 
*/
#define DOLC                     "March 9, 1998"

char YABLICENSE[]=

"     This program is free software; you can redistribute it and/or       \n"
"     modify it under the terms of the GNU General Public License         \n"
"     as published by the Free Software Foundation; either version        \n"
"     of the License, or (at your option) any later version.              \n"
"                                                                         \n"
"     This program is distributed in the hope that it will be useful,     \n"
"     but WITHOUT ANY WARRANTY; without even the implied warranty of      \n"
"     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the       \n"
"     GNU General Public License for more details.                        \n"
"                                                                         \n"
"     You should have received a copy of the GNU General Public License   \n"
"     along with this program (the file is named COPYING);                \n"
"     if not, write to the Free Software                                  \n"
"     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.           \n"
;


/* ------------- defines ---------------- */

/*
   Define one and only one of the following symbols, depending on your
   System:
            - UNIX: uses some UNIX-features and X11
            - WINDOWS: uses WIN32-features
*/

#define BASIC_NAME "yabasic"

#define DONE {current=current->next;break;}  /* reduces type-work */

#if defined(UNIX) && defined(WINDOWS)
UNIX and WINDOWS are defined at once; check your compiler settings
#endif


/* ------------- includes ---------------- */

#include "yabasic.h"          /* all prototypes and structures */
#ifdef UNIX
#include <curses.h>
#endif


/* ------------- external references ---------------- */

extern int yylineno;   /* current line number */
extern void switch_to_my_file(FILE *); /* switches lex input */ 
extern int yyparse();  /* call bison parser */


/* ------------- enum types ---------------- */

enum stck_or_sym { /* contents of a stack element or type of a symbol */
  STRING,NUMBER,LBL,GTO,JUMP,FREE,FUNC,ARRAY,RETADD,NIL
};

enum cmd_type { /* type of command */
  FIRST_COMMAND, /* no command, just marks start of list */

  LABEL,GOTO,QGOTO,GOSUB,QGOSUB,RETURN,          /* flow control */
  END,DECIDE,SKIPPER,NOP,

  DIM,FUNCTION,DOARRAY,                          /* everything with "()"*/

  DBLADD,DBLMIN,DBLMUL,DBLDIV,DBLPOW,NEGATE,     /* double operations */
  PUSHDBLSYM,POPDBLSYM,PUSHDBL,

  SETINFOLEVEL,SETFONTHEIGHT,SETWINHEIGHT,       /* internal vars */
  SETWINWIDTH,

  AND,OR,NOT,LT,GT,LE,GE,EQ,NE,                  /* comparisons */
  STREQ,STRNE,STRLT,STRLE,STRGT,STRGE,

  PUSHSTRSYM,POPSTRSYM,PUSHSTR,CONCAT,           /* string operations */
  PUSHSTRPTR,CHANGESTRING,

  PRINT,MYREAD,PROMPT,RESTORE,QRESTORE,          /* i/o operations */
  READDATA,DATA,MYOPEN,MYCLOSE,MYSWITCH,TESTEOF,
  WAIT,BELL,MOVE,CLEARSCR,
  
  OPENWIN,DOT,LINE,CIRCLE,TEXT,CLOSEWIN,CLEARWIN,       /* grafics */
  OPENPRN,CLOSEPRN,

  LAST_COMMAND /* no command, just marks end of list */
};

enum states { /* current state of program */
  HATCHED,INITIALIZED,COMPILING,RUNNING,FINISHED
};



/* ------------- global variables ---------------- */

struct symbol *symroot; /* first element in symbol list */
struct symbol *symhead; /* last element ind symbol list */
struct stackentry *stackroot; /* lowest element in double stack */
struct stackentry *stackhead; /* topmost element in double stack */
struct command *current; /* currently executed command */
struct command *cmdroot; /* first command */
struct command *cmdhead; /* last command */
struct command *datapointer; /* current location for read-command */
int infolevel; /* controls issuing of error messages */
int errorlevel; /* highest level of error message seen til now */
int diagnostic_count; /* number of diagnostic messages */
int note_count; /* number of notes */
int warning_count; /* number of warning messages */
int error_count; /* number of error messages */
int interactive; /* true, if commands come from stdin */
char *string; /* for trash-strings */
int labelcount=0; /* count self-generated labels */
int commandcount; /* total number of commands */
int program_state;  /* state of program */
char inputprompt[100]; /* input prompt */
struct command *lastcommand; /* most recently created command */
FILE *streams[10]; /* file streams */
FILE *currentstream; /* current stream for output ... */
int curinized=FALSE; /* true, if curses has been initialized */

/* printer-related */
FILE *printerfile=NULL; /* file to print on */
char *prfilename=NULL; /* filename to print on */
int print_to_file; /* print to file ? */


/* ------------- global variables for Graphics ---------------- */

#ifdef UNIX
Display *display; /* X-Display */
Window window,root; /* ID of window and root window */
GC gc; /* GC for drawing */
XSizeHints sizehints; /* hints for window manager */
static unsigned long f_colour,b_colour; /* colors */
#elif WINDOWS
WNDCLASS myclass; /* window class for my program */
HWND window;   /* handle of my window */
HANDLE this_instance;
HDC devcon; /* device context */
char *my_class="my_class";
struct { /* contains all information for window-thread */
  HANDLE handle; /* handle of thread */
  DWORD id;    /* id of my thread */
  HANDLE winevent; /* handle for window event */
} thread;
LOGFONT logfont; /* structure for font-characteristics */
HFONT myfont; /* handle of font for screen */
HFONT printerfont; /* handle of printer-font */
HPEN printerpen; /* handle of printer-pen */
HDC printer=NULL; /* handle of printer */
float prnscale;  /* scaling factor for printer */
float prnoff;  /* offset for printer */
BOOL Commandline; /* true if launched from command line */
HANDLE ConsoleInput; /* handle for console input */
HANDLE ConsoleOutput; /* handle for console output */
int LINES; /* number of lines on screen */
int COLS; /* number of columns on screen */
#endif

unsigned long winwidth,winheight;  /* size of window */
int winx,winy; /* position of window */
int winopened=FALSE; /* flag if window is open already */
double psscale; /* from pixels to points */
int fontheight; /* height of used font in pixel */
char *foreground=NULL;
char *background=NULL;
char *geometry=NULL;
char *displayname=NULL;
char *font=NULL;

/* ------ array with explanation text ------- */
struct {
  enum cmd_type command;
  char *text;
} explanation[LAST_COMMAND-FIRST_COMMAND+1]={
  {FIRST_COMMAND,"???"},
  {LABEL,"LABEL"},{GOTO,"GOTO"},{QGOTO,"QGOTO"},{GOSUB,"GOSUB"},
  {QGOSUB,"QGOSUB"},{RETURN,"RETURN"},{END,"END"},{DECIDE,"DECIDE"},
  {SKIPPER,"SKIPPER"},{NOP,"NOP"},{DIM,"DIM"},{FUNCTION,"FUNCTION"},
  {DOARRAY,"DOARRAY"},{DBLADD,"DBLADD"},{DBLMIN,"DBLMIN"},{DBLMUL,"DBLMUL"},
  {DBLDIV,"DBLDIV"},{DBLPOW,"DBLPOW"},{NEGATE,"NEGATE"},
  {PUSHDBLSYM,"PUSHDBLSYM"},{POPDBLSYM,"POPDBLSYM"},{PUSHDBL,"PUSHDBL"},
  {SETINFOLEVEL,"SETINFOLEVEL"},{SETFONTHEIGHT,"SETFONTHEIGHT"},
  {SETWINHEIGHT,"SETWINHEIGHT"},{SETWINWIDTH,"SETWINWIDTH"},
  {AND,"AND"},{OR,"OR"},{NOT,"NOT"},{LT,"LT"},{GT,"GT"},{LE,"LE"},
  {GE,"GE"},{EQ,"EQ"},{NE,"NE"},{STREQ,"STREQ"},{STRNE,"STRNE"},
  {PUSHSTRSYM,"PUSHSTRSYM"},{POPSTRSYM,"POPSTRSYM"},
  {PUSHSTR,"PUSHSTR"},{CONCAT,"CONCAT"},{PUSHSTRPTR,"PUSHSTRPTR"},
  {CHANGESTRING,"CHANGESTRING"},{PRINT,"PRINT"},{MYREAD,"MYREAD"},
  {PROMPT,"PROMPT"},{RESTORE,"RESTORE"},{QRESTORE,"QRESTORE"},
  {READDATA,"READDATA"},{DATA,"DATA"},{MYOPEN,"MYOPEN"},
  {MYCLOSE,"MYCLOSE"},{MYSWITCH,"MYSWITCH"},{WAIT,"WAIT"},
  {BELL,"BELL"},{MOVE,"MOVE"},{CLEARSCR,"CLEARSCR"},{OPENWIN,"OPENWIN"},
  {DOT,"DOT"},{LINE,"LINE"},{CIRCLE,"CIRCLE"},{TEXT,"TEXT"},
  {CLOSEWIN,"CLOSEWIN"},{CLEARWIN,"CLEARWIN"},{OPENPRN,"OPENPRN"},
  {CLOSEPRN,"CLOSEPRN"},{TESTEOF,"TESTEOF"},
  {LAST_COMMAND,"???"}
};


/* ------------- main program ---------------- */

int main(int argc,char *argv[])
{
  time_t start,compiled,ended;
#ifdef WINDOWS
  CONSOLE_SCREEN_BUFFER_INFO csbi;
#endif

  string=(char *)malloc(sizeof(char)*1000);

  program_state=HATCHED;
  infolevel=WARNING; /* set the initial Infolevel */
    
#ifdef WINDOWS
  /* get handle of instance */
  this_instance=GetModuleHandle(NULL);

  /* define my window class */
  myclass.style=0;
  myclass.lpfnWndProc=(LPVOID) mywindowproc;
  myclass.cbClsExtra=0; /* no extra bytes */
  myclass.cbWndExtra=0;
  myclass.hInstance=this_instance;
  myclass.hIcon=LoadIcon(this_instance,"yabasicIcon");
  myclass.hCursor=LoadCursor(NULL,IDC_ARROW); /*  standard cursor */
  myclass.hbrBackground=(HBRUSH) COLOR_WINDOW; /* default-background */
  myclass.lpszMenuName=NULL;
  myclass.lpszClassName=my_class;

  RegisterClass(&myclass);

  /* get console handles */
  ConsoleInput=GetStdHandle(STD_INPUT_HANDLE);
  ConsoleOutput=GetStdHandle(STD_OUTPUT_HANDLE);

  /* find out, if launched from commandline */
  GetConsoleScreenBufferInfo(ConsoleOutput,&csbi);
  Commandline=!((csbi.dwCursorPosition.X==0) && (csbi.dwCursorPosition.Y==0));
  if ((csbi.dwSize.X<=0) || (csbi.dwSize.Y <= 0)) Commandline=TRUE;

#endif

  parse_arguments(argc,argv);

  time(&start);
  initialize();
  program_state=INITIALIZED;

  error(NOTE,"Calling parser/compiler");
  if (interactive) {
    printf("Enter your program; type <RETURN> twice, when done.\n");
  }
  program_state=COMPILING;
  yyparse();

  create_myend();
  sprintf(string,"Read %d line(s)",yylineno);
  error(NOTE,string);
  sprintf(string,"Generated %d command(s)",commandcount);
  error(NOTE,string);

  time(&compiled);
          
  if (errorlevel>ERROR) {
    program_state=RUNNING;
    run_it();}
  else {
    program_state=FINISHED;
    error(ERROR,"Program not executed");
  }

  sprintf(string,"%d diagnostic(s), %d note(s), %d warning(s), %d error(s)",
          diagnostic_count,note_count,warning_count,error_count);
  error(NOTE,string);
  time(&ended);
  sprintf(string,"Compilation time = %g second(s)",difftime(compiled,start));
  error(NOTE,string);
  sprintf(string,"Execution time = %g second(s)",difftime(ended,compiled));
  error(NOTE,string);
  end_it();
  if (errorlevel>ERROR) 
    return 1;
  else
    return 0;
}



/* ------------- subroutines ---------------- */


struct symbol *get_sym(char *name,int type,int add) 
     /* gets the value of a symbol, or creates it with initial value type */
{
  struct symbol *curr,*new;

  curr=symroot;
  while(curr!=symhead) {   /* search 'til head of list */
    if (curr->type==type  /* do the types match ? */
        &&!strcmp(name,curr->name))  /* do the names match ? */
      return curr; /* give back address */
    curr=curr->next; /* try next entry */
  }
  /* run (ppp) through all variables. */
  if (!add) return NULL;  /* dont create, if not found */
  /* end of list, create new element */
  new=(struct symbol *)my_malloc(sizeof(struct symbol)); /* ceate new item */
  symhead=new;  /* advance head */
  curr->name=my_strdup(name);  /* store new name */
  curr->next=new;
  curr->value=0.0;
  curr->pointer=NULL;
  curr->type=type;
  curr->value=0.0;
  if (type==STRING) {   /* create empty string */
    curr->pointer=my_malloc(sizeof(char));
    *(char *)(curr->pointer)='\0';
  }
  return curr;
}
    
    
void create_pushdbl(double value) /* create command 'pushdbl' */
{
  struct command *cmd;
  
  cmd=add_command(PUSHDBL);
  if (cmd->pointer==NULL) cmd->pointer=my_malloc(sizeof(double));
  *(double *)(cmd->pointer)=value;
}


void pushdbl(struct command *cmd) 
{
  /* push double onto stack */
  struct stackentry *p;

  p=push();
  p->value= *(double *)cmd->pointer;
  p->type=NUMBER;
}


void create_pushdblsym(char *symbol) /* create command 'pushdblsym' */
{
  struct command *cmd;

  cmd=add_command(PUSHDBLSYM);
  /* get room to store specific information */
  cmd->pointer= &(get_sym(symbol,NUMBER,TRUE)->value);
}


void pushdblsym(struct command *cmd) 
{
  /* push double symbol onto stack */
  struct stackentry *p;
  
  p=push();
  p->value= *(double *)cmd->pointer;
  p->type=NUMBER;
}


void create_popdblsym(char *symbol) /* create command 'popdblsym' */
{
  struct command *cmd;
  struct symbol *s;

  cmd=add_command(POPDBLSYM);
  if (symbol) {
    /* storing specific information: */
    s=get_sym(symbol,NUMBER,TRUE);
    cmd->pointer= &(s->value);
  }
  else {
    cmd->pointer=NULL;
    return;
  }
  
  /* treat internal vars */
  if (!strncmp(symbol,"yab",3)) {
    if (!strcmp(symbol,"yabinfolevel")) {
      cmd=add_command(SETINFOLEVEL);
    }
    if (!strcmp(symbol,"yabwinheight")) {
      cmd=add_command(SETWINHEIGHT);
    }
    if (!strcmp(symbol,"yabwinwidth")) {
      cmd=add_command(SETWINWIDTH);
    }
    if (!strcmp(symbol,"yabfontheight")) {
      cmd=add_command(SETFONTHEIGHT);
    }
  }
}


void setinfolevel(void)
{
  /* set infolevel to content of variable infolevel */
  int i;
  static char *levels[]={"FATAL","ERROR","WARNING","NOTE","DIAGNOSTIC"};
  
  i=get_sym("yabinfolevel",NUMBER,FALSE)->value;
  
  if (i!=DIAGNOSTIC && i!=NOTE && i!=WARNING && 
      i!=ERROR && i!=FATAL) return;
  
  if (infolevel<i) infolevel=i;
  if (infolevel<=DIAGNOSTIC) {
    sprintf(string,"setting infolevel to %s",levels[i-FATAL]);
    error(DIAGNOSTIC,string);
  }
  infolevel=i;
}


void setwinheight(void)
{
  /* set winheight to content of variable winheight */
  
  winheight=get_sym("yabwinheight",NUMBER,FALSE)->value;
  get_sym("yabwinheight",NUMBER,FALSE)->value=winheight;

  if (winheight<1) {
    error(ERROR,"winheight less than 1 pixel");
    return;
  }

  calc_psscale();
}


void setwinwidth(void)
{
  /* set winwidth to content of variable winwidth */
  
  winwidth=get_sym("yabwinwidth",NUMBER,FALSE)->value;
  get_sym("yabwinwidth",NUMBER,FALSE)->value=winwidth;

  if (winwidth<1) {
    error(ERROR,"winwidth less than 1 pixel");
    return;
  }

  calc_psscale();
}


void setfontheight(void)
{
  /* set fontheight to content of variable fontheight */
  
  fontheight=get_sym("yabfontheight",NUMBER,FALSE)->value;
  get_sym("yabfontheight",NUMBER,FALSE)->value=fontheight;

  calc_psscale();
}


void popdblsym(struct command *cmd) 
{
  /* pop double from stack */
  double d;

  d=pop()->value;
  if (cmd->pointer==NULL) return;
  *(double *)(cmd->pointer)=d;
}


void create_dblbin(char c) /* create command for binary double operation */
{
  switch(c) {
  case '+':add_command(DBLADD);break;
  case '-':add_command(DBLMIN);break;
  case '*':add_command(DBLMUL);break;
  case '/':add_command(DBLDIV);break;
  case '^':add_command(DBLPOW);break;
  }
  /* no specific information needed */
}


void dblbin(struct command *cmd) /* compute with two numbers from stack */
{
  struct stackentry *d;
  double a,b,c;

  b=pop()->value;
  a=pop()->value;
  d=push();
  switch(cmd->type) {
  case(DBLADD):c=a+b; break;
  case(DBLMIN):c=a-b; break;
  case(DBLMUL):c=a*b; break;
  case(DBLDIV): 
    if (fabs(b)<DBL_MIN) {
      sprintf(string,"Division by zero, set to %g",DBL_MAX);
      error(NOTE,string);
      c=DBL_MAX;}
    else
      c=a/b;
    break;
  case(DBLPOW):
    if (b==2) 
      c=a*a;
    else if (a<0) {
      error(ERROR,"Power of negative value. Don't now what to do");
      return;}
    else
      c=exp(b*log(a));
    break;
  }
  d->value=c;
  d->type=NUMBER;
}


void create_negate() /* creates command negate */
{
  add_command(NEGATE);
}


void negate() /* negates top of stack */
{
  struct stackentry *a,*b;
  double d;

  a=pop();
  d=a->value;
  b=push();
  b->type=NUMBER;
  b->value= -d;
}


void create_pushstrptr(char *symbol) /* push string-pointer onto stack */
{
  struct command *cmd;

  cmd=add_command(PUSHSTRPTR);
  cmd->pointer=&get_sym(symbol,STRING,TRUE)->pointer;
}


void pushstrptr(struct command *cmd)  /* push string-pointer onto stack */
{
  struct stackentry *p;
  
  p=push();
  p->pointer=*(char **)cmd->pointer;
  p->type=STRING;
}


void create_pushstrsym(char *symbol) /* push string-symbol onto stack */
{
  struct command *cmd;

  cmd=add_command(PUSHSTRSYM);
  /* get room to store specific information */
  cmd->pointer=&get_sym(symbol,STRING,TRUE)->pointer;
}


void pushstrsym(struct command *cmd)  /* push string-symbol onto stack */
{
  struct stackentry *p;
  
  p=push();
  p->pointer=my_strdup(*(char **)cmd->pointer);
  p->type=STRING;
}


void create_popstrsym(char *symbol) /* create command 'popstrsym' */
{
  struct command *cmd;
  struct symbol *s;

  cmd=add_command(POPSTRSYM);
  if (symbol) {
    /* storing specific information: */
    s=get_sym(symbol,STRING,TRUE);
    cmd->pointer=(char **)&(s->pointer);
  }
  else {
    cmd->pointer=NULL;
  }
}


void popstrsym(struct command *cmd) /* pop string from stack */
{
  struct stackentry *p;

  p=pop();
  if (cmd->pointer==NULL) return;
  if (*(char **)cmd->pointer!=NULL) free(*(char **)cmd->pointer);
  *(char **)cmd->pointer=my_strdup(p->pointer);
}


void create_concat() /* creates command concat */
{
  add_command(CONCAT);
}


void concat() /* concetenates two strings from stack */
{
  struct stackentry *a,*b,*c;
  char *aa,*bb,*cc;

  a=pop();
  b=pop();
  if (a->type!=STRING || b->type!=STRING) {
    error(FATAL,"Need strings to concat");
    return;
  }
  aa=a->pointer;
  bb=b->pointer;
  cc=(char *) my_malloc(sizeof(char)*(strlen(aa)+strlen(bb)+1));
  strcpy(cc,bb);
  strcat(cc,aa);
  c=push();
  c->type=STRING;
  c->pointer=cc;
}  


void create_pushstr(char *s) /* creates command pushstr */
{
  struct command *cmd;
  
  cmd=add_command(PUSHSTR);
  cmd->pointer=my_strdup(s); /* store string */
}


void pushstr(struct command *cmd) 
{
  /* push string onto stack */
  struct stackentry *p;
  
  p=push();
  p->pointer=my_strdup((char *)cmd->pointer);
  p->type=STRING;
}


void create_goto(char *label) /* creates command goto */
{
  struct command *cmd;

  cmd=add_command(GOTO);
  /* specific info */
  cmd->pointer=my_strdup(label);
}


void create_gosub(char *label) /* creates command gosub */
{
  struct command *cmd;

  cmd=add_command(GOSUB);
  /* specific info */
  cmd->pointer=my_strdup(label);
}


void jump(struct command *cmd) 
     /* jump to specific Label; used as goto or gosub */
{
  struct command *curr;
  struct stackentry *ret;
  int type;

  type=cmd->type;
  if (type==QGOSUB || type==GOSUB) {
    ret=push();
    ret->pointer=current;
    ret->type=RETADD;
  }
  if (type==QGOSUB || type==QGOTO) {
    current=(struct command *)cmd->pointer;
    return;
  }
  curr=cmdroot;
  while(curr!=cmdhead) {   /* go through all commands */
    if (curr->type==LABEL && !strcmp(curr->pointer,cmd->pointer)) {
      /* found right Label ! */
      current=curr; /* jump to new location */
      /* use the address instead of the name next time ! */
      cmd->pointer=curr;
      cmd->type=(type==GOTO) ? QGOTO:QGOSUB; /* quick jump from now on */
      return;
    }
    curr=curr->next;
  }
  /* label not found */
  sprintf(string,"Can't find label '%s'",(char *)cmd->pointer);
  error(ERROR,string);
}


void create_return() /* creates command return */
{
  add_command(RETURN);
}


void myreturn() /* return from gosub */
{
  struct stackentry *address;

  address=pop();
  if (address->type!=RETADD) {
    error(ERROR,"RETURN without GOSUB");
    return;
  }
  current=(struct command *)address->pointer;
  return;
}


void create_label(char *label) /* creates command label */
{
  struct command *cmd,*curr;

  /* check, if label is duplicate */
  curr=cmdroot;
  while(curr!=cmdhead) {   /* go through all commands */
    if (curr->type==LABEL && !strcmp(curr->pointer,label)) {
      /* found duplicate Label ! */
      sprintf(string,"duplicate label '%s'",(char *)curr->pointer);
      error(ERROR,string);
      return;
    }
    curr=curr->next;
  }

  cmd=add_command(LABEL);
  /* store label */
  cmd->pointer=my_strdup(label);
}


void create_skipper() /* creating command skipper */
{
  add_command(SKIPPER);
}


void skipper()
     /* used for on_goto/gosub, skip specified number of commands */
{
  int i,n;

  n=pop()->value;
  i=1;
  current=current->next;
  while(i<n && (current->next)->type!=NOP) { /* go through all labels */
    current=current->next; /* next label */
    i++;
  }
}


void create_nop() /* does nothing */
{
  add_command(NOP);
}


void create_myend() /* create command 'end' */
{
  add_command(END);
}


void create_print(char type) /* create command 'print' */
{
  struct command *cmd;

  cmd=add_command(PRINT);
  cmd->pointer=my_malloc(sizeof(int));
  /* store type of print  */
  cmd->tag=type;
}


void print(struct command *cmd) /* print on screen */
{
  int type;
  struct stackentry *p;
  FILE *str;
  static int last;
  char *toprint; /* points on chars to print */
#ifdef UNIX
  int x,y;
#else
  DWORD written; /* number of bytes written */
#endif

  if (currentstream==NULL) 
    str=stdout;
  else 
    str=currentstream;

  type=cmd->tag;
  switch(type) {
  case 'n':  /* print newline */
#ifdef UNIX
    if (curinized && str==stdout) {
      getyx(stdscr,y,x);
      if (y==LINES-1) {
	scrl(1);
	move(y,0);
      } 
      else {
	move(y+1,0);
      }
      string[0]='\0';
    } 
    else
#endif
      {
	string[0]='\n';
	string[1]='\0';
      }
    toprint=string;
    break;
  case 'd':  /* print double value */
    p=pop();
    sprintf(string,"%s%g",(last=='d')?" ":"",p->value);
    toprint=string;
    break;
  case 's': 
    p=pop();
    toprint=(char *)p->pointer;
    break;
  }
  if (curinized && str==stdout) {
#ifdef UNIX
      addstr(toprint);
      refresh();
#else
      WriteConsole(ConsoleOutput,toprint,strlen(toprint),&written,NULL);
#endif
  }
  else
    {
      fprintf(str,"%s",toprint);
      fflush(str);
  }
  last=type;
}

void create_mymove() /* create command 'mymove' */
{
  add_command(MOVE);
}

void mymove() /* move to specific position on screen */
{
  int x,y;
#ifdef WINDOWS
  COORD coord;
#endif

  y=pop()->value;
  if (y<0) y=0;
  if (y>LINES-1) y=LINES-1;
  x=pop()->value;
  if (x<0) x=0;
  if (x>COLS-1) x=COLS-1;
  if (!curinized) {
    error(ERROR,"need to call 'clear screen' first");
    return;
  }
#ifdef UNIX
  move(y,x);
#else
  coord.X=x;
  coord.Y=y;
  SetConsoleCursorPosition(ConsoleOutput,coord);
#endif
}

void create_clearscreen() /* create command 'clearscreen' */
{
  add_command(CLEARSCR);
}

void clearscreen() /* clear entire screen */
{
#ifdef WINDOWS
  DWORD written; /* number of chars actually written */
  COORD coord; /* coordinates to start writing */
#endif

  if (!curinized) curinit();
#ifdef UNIX
  clear();
  refresh();
#else
  coord.X=0;
  coord.Y=0;
  FillConsoleOutputCharacter(ConsoleOutput,' ',10000,coord,&written);
  SetConsoleCursorPosition(ConsoleOutput,coord);
  curinized=TRUE;
#endif
}

void curinit() /* initialize curses */
{
  struct symbol *s;
#ifdef WINDOWS
  CONSOLE_SCREEN_BUFFER_INFO coninfo; /* receives console size */
#endif

#ifdef UNIX
  initscr();
  scrollok(stdscr,TRUE);
  idlok(stdscr,TRUE);
#else
  GetConsoleScreenBufferInfo(ConsoleOutput,&coninfo);
  COLS=coninfo.dwSize.X;
  LINES=coninfo.dwSize.Y;
#endif
  s=get_sym("yabscreenwidth",NUMBER,FALSE);
  s->value=COLS;
  s=get_sym("yabscreenheight",NUMBER,FALSE);
  s->value=LINES;
  curinized=TRUE;
}

char *replace(char *string) /* replace \n,\a, etc. */
{
  char *from,*to;

  from=to=string;
  while(*from) {
    if (*from=='\\') {
      from++;
      switch(*from) {
      case 'n':	*to='\n';break;
      case 't':	*to='\t';break;
      case 'v':	*to='\v';break;
      case 'b':	*to='\b';break;
      case 'r':	*to='\r';break;
      case 'f':	*to='\f';break;
      case 'a':	*to='\a';break;
      case '\\': *to='\\';break;
      case '\?': *to='\?';break;
      case '\'': *to='\'';break;
      case '\"': *to='\"';break;
      default:
	*to='\\';
	to++;
	*to=*from;
      }
    }
    else
      *to=*from;
    from++;
    to++;
  }
  *to='\0';
  return string;
}


void create_myopen(double stream,char *mode) /* create command 'myopen' */
{
  struct command *cmd;

  if (badstream((int)stream)) return;
  cmd=add_command(MYOPEN);
  cmd->args=(int) stream;
  cmd->pointer=my_strdup(mode);
}


void myopen(struct command *cmd) /* open specified file for given name */
{
  FILE *handle;
  int stream;
  char *name;
  char *mode;
  
  mode=(char *)cmd->pointer;
  name=pop()->pointer;
  stream=cmd->args;
  if (streams[stream]!=NULL) {
    error(ERROR,"Stream already in use");
    return;
  }
  handle=fopen(name,mode);
  if (handle==NULL) {
    sprintf(string,"Could not open '%s'",name);
    error(ERROR,string);
    return;
  }
  streams[stream]=handle;
}


void create_myclose(double stream) /* create command 'myclose' */
{
  struct command *cmd;

  if (badstream((int)stream)) return;
  cmd=add_command(MYCLOSE);
  cmd->args=(int) stream;
  return;
}


void myclose(struct command *cmd) /* close the specified stream */
{
  int s;
  
  s=cmd->args;
  if (streams[s]==NULL) {
    sprintf(string,"Stream %d already closed",s);
    error(WARNING,string);
    return;
  }
  fclose(streams[s]);
  streams[s]=NULL;
}


void create_myswitch(double stream) /* create command myswitch */
{
  struct command *cmd;

  if (stream!=0.0 && badstream((int)stream)) return;
  cmd=add_command(MYSWITCH);
  cmd->args=(int) stream;
}


void myswitch(struct command *cmd) /* switch to specified stream */
{
  int stream;

  stream=cmd->args;
  if (stream==0) 
    currentstream=NULL;
  else  {
    currentstream=streams[stream]; /* switch to stream */
    if (streams[stream]==NULL) {
      sprintf(string,"Stream %d not opened",stream);
      error(ERROR,string);
      return;
    } 
  }
  return;
}


void create_testeof(double stream) /* create command 'testeof' */
{
  struct command *cmd;

  if (badstream((int)stream)) return;
  cmd=add_command(TESTEOF);
  cmd->args=(int) stream;
  return;
}


void testeof(struct command *cmd) /* close the specified stream */
{
  int s,c;
  struct stackentry *result;
  
  s=cmd->args;
  result=push();
  result->type=NUMBER;
  if (streams[s]==NULL) {
    result->value=TRUE;
    return;
  }
  c=getc(streams[s]);
  if (c==EOF) {
    result->value=TRUE;
    return;
  }
  
  result->value=FALSE;
  ungetc(c,streams[s]);
  return;
}


int badstream(int stream) /* test for valid stream id */
{
  int max;

  max=(9>FOPEN_MAX)?(FOPEN_MAX-3):9;
  if (stream>max || stream<1) {
    sprintf(string,"Can handle only streams from 1 to %d",max);
    error(ERROR,string);
    return TRUE;
  }
  return FALSE;
}


void create_myread(char type) /* create command 'read' */
{
  struct command *cmd;

  cmd=add_command(MYREAD);
  if (lastcommand) /* not first input */
    cmd->args=MYREAD;
  else
    cmd->args=PROMPT;
    
  lastcommand=cmd;
  cmd->tag=type;
}


void myread(struct command *cmd) /* read string or double */
{
  double d;
  static char buffer[1000]; /* buffer with read data */
  static int start=-1; /* points into buffer */
  static int end=-1; /* points into buffer */
  struct stackentry *s;
  FILE *str;
  int tileol; /* TRUE, if input should be read til end of line */
  int skip; /* TRUE, if initial whitesapces should be skipped */
#ifdef WINDOWS
  DWORD wrtn; /* number of bytes written */
  DWORD read; /* number of bytes read */
#endif

  skip=(cmd->args!=-PROMPT);
  tileol=(cmd->args<0);

  if (currentstream==NULL) {
    str=stdin;
    if (curinized) {
#ifdef UNIX
      addstr(inputprompt);
#else
      WriteConsole(ConsoleOutput,inputprompt,strlen(inputprompt),&wrtn,NULL);
#endif
    }
    else {
      printf("%s",inputprompt);
    }
  }
  else {
    str=currentstream;
    fflush(currentstream);
  }

  if (end<0 || !buffer[end]) {
    if (str==stdin && curinized) {
#ifdef UNIX
      getstr(buffer);
#else
      ReadConsole(ConsoleInput,buffer,1000,&read,NULL);
#endif
    }
    else {
      if (!fgets(buffer,1000,str)) buffer[0]='\0';
    }
    /* start at beginning of buffer */
    end=0;
  }

  /* skip leading whitespace */
  while(skip && buffer[end] && tabspc(buffer[end])) end++;

  /* remember start of input */
  start=end;

  /* advance to end of input */
  while(buffer[end] && tabspc(buffer[end])) end++;
  while(buffer[end] && (!tabspc(buffer[end]) || tileol)) end++;

  /* cut off trailing newline */
  if (end>0 && buffer[end-1]=='\n') {
    end--;
    buffer[end]='\0';
  }

  /* put in new prompt */
  if (buffer[end])
    inputprompt[0]='\0';
  else
    strcpy(inputprompt,"?");
  
  if (cmd->tag=='s') { /* read string */
    s=push();
    s->type=STRING;
    s->pointer=my_strndup(buffer+start,end-start);}
  else { /* read double */
    s=push();
    s->type=NUMBER;
    s->value=0.0;
    if (buffer[start] && (sscanf(buffer+start,"%lf",&d)==1)) s->value=d;
  }
}
  
int tabspc(char c) /* true, if char is tab or space, helper for myread */
{
  return (c=='\t' || c==' ');
}

void create_prompt(char *p) /* create command 'prompt' */
{
  struct command *cmd;

  lastcommand=NULL;
  cmd=add_command(PROMPT);
  cmd->pointer=my_strdup(p);
}


void prompt(struct command *cmd) /* set input prompt */
{
  strncpy(inputprompt,cmd->pointer,80);
  inputprompt[80]='\0';
}


void create_restore(char *label) /* create command 'restore' */
{
  struct command *c;
  
  c=add_command(RESTORE);
  c->pointer=my_strdup(label);
}


void restore(struct command *cmd) /* reset data pointer to given label */
{
  struct command *curr;

  if (cmd->type==RESTORE) { /* first time; got to search the label */
    if (*((char *)cmd->pointer)=='\0') {
      cmd->pointer=cmdroot;
      cmd->type=QRESTORE;
      goto found; /* restore to first command */
    }
    curr=cmdroot;
    while(curr!=cmdhead) {   /* go through all commands */
      if (curr->type==LABEL && !strcmp(curr->pointer,cmd->pointer)) {
        /* found right Label ! */
        /* use the address instead of the name next time ! */
        cmd->pointer=curr;
        cmd->type=QRESTORE;
        goto found;
      }
      curr=curr->next;
    }
    /* did not found label */
    sprintf(string,"couldn't found label '%s'",(char *)cmd->pointer);
    error(ERROR,string);
    return;
  }
found:
  datapointer=cmd->pointer;
  return;
}


void create_dbldata(double value)  /* create command dbldata */
{
  struct command *c;

  c=add_command(DATA);
  c->pointer=my_malloc(sizeof(double));
  *((double *)c->pointer)=value;
  c->tag='d'; /* double value */
}


void create_strdata(char *value)  /* create command strdata */
{
  struct command *c;

  c=add_command(DATA);
  c->pointer=my_strdup(value);
  c->tag='s'; /* string value */
}


void create_readdata(char type) /* create command readdata */
{
  struct command *cmd;

  cmd=add_command(READDATA);
  cmd->tag=type;
}


void readdata(struct command *cmd) /* read data items */
{
  struct stackentry *read;
  char type;

  type=cmd->tag;
  while(datapointer->type!=DATA) {
    if (datapointer==cmdhead) {
      error(ERROR,"Run out of data items");
      return;
    }
    datapointer=datapointer->next;
  }
  if (type!=datapointer->tag) {
    error(ERROR,"Type of READ and DATA don't match");
    return;
  }
  read=push();
  if (type=='d') { /* read a double value */
    read->type=NUMBER;
    read->value= *((double *)datapointer->pointer);}
  else {
    read->type=STRING;
    read->pointer=my_strdup(datapointer->pointer);
  }
  datapointer=datapointer->next; /* next item */
}


void create_dblrelop(char c) /* create command dblrelop */ 
{
  int type;

  switch(c) {
  case '=': type=EQ;break;
  case '!': type=NE;break;
  case '<': type=LT;break;
  case '{': type=LE;break;
  case '>': type=GT;break;
  case '}': type=GE;break;
  }
  add_command(type);
}


void dblrelop(struct command *type)  /* compare topmost double-values */
{
  double a,b,c;
  struct stackentry *result;

  b=pop()->value;
  a=pop()->value;
  switch(current->type) {
  case EQ:c=(a==b);break;
  case NE:c=(a!=b);break;
  case LE:c=(a<=b);break;
  case LT:c=(a<b);break;
  case GE:c=(a>=b);break;
  case GT:c=(a>b);break;
  }
  result=push();
  result->value=c;
  result->type=NUMBER;
}    


void create_strrelop(char c) /* create command strrelop */ 
{
  int type;

  switch(c) {
  case '=': type=STREQ;break;
  case '!': type=STRNE;break;
  case '<': type=STRLT;break;
  case '{': type=STRLE;break;
  case '>': type=STRGT;break;
  case '}': type=STRGE;break;
  }
  add_command(type);
}


void strrelop(struct command *type)  /* compare topmost string-values */
{
  char *a,*b;
  double c;
  struct stackentry *result;

  b=pop()->pointer;
  a=pop()->pointer;
  switch(current->type) {
  case STREQ:c=(strcmp(a,b)==0);break;
  case STRNE:c=(strcmp(a,b)!=0);break;
  case STRLT:c=(strcmp(a,b)<0);break;
  case STRLE:c=(strcmp(a,b)<=0);break;
  case STRGT:c=(strcmp(a,b)>0);break;
  case STRGE:c=(strcmp(a,b)>=0);break;
  }
  result=push();
  result->value=c;
  result->type=NUMBER;
}    


void create_boole(char c) /* create command boole */ 
{
  int type;

  switch(c) {
  case '|': type=OR;break;
  case '&': type=AND;break;
  case '!': type=NOT;break;
  }
  add_command(type);
}


void boole(struct command *type)  /* perform and/or/not */
{
  int a,b,c;
  struct stackentry *result;

  a=pop()->value;
  if (current->type==NOT) 
    c=!a;
  else {
    b=pop()->value;
    if (current->type==AND)
      c=a&&b;
    else
      c=a||b;
  }
  result=push();
  result->value=c;
  result->type=NUMBER;
}    


void create_decide() /* creates command decide */
{
  add_command(DECIDE);
}
    

void decide() /*  skips next command, if 0 on stack */
{
  struct stackentry *a;

  a=pop();
  if (a->type!=NUMBER) {
    error(FATAL,"Dont find number to decide");
    return;
  }
  if (a->value!=0) current=current->next; /* skip one command */
}


void create_doarray(char *symbol,int command) /* creates array-commands */ 
{
  struct command *cmd;
  struct symbol *a;
  struct array *ar;
  int dimcount;

  a=get_sym(symbol,ARRAY,FALSE);
  if (a==NULL) {
    sprintf(string,"array '%s' has not been dimed",symbol);
    error(ERROR,string);
    return;
  }

  dimcount=pop()->value;
  ar=a->pointer;
  if (dimcount!=ar->dimension) {
    sprintf(string,"improper array dimension %d for '%s'",dimcount,symbol);
    error(ERROR,string);
    return;
  }

  cmd=add_command(DOARRAY);
  
  switch(command) {
  case CALLARRAY:
    cmd->args=CALLARRAY;
    break;
  case ASSIGNARRAY:
    cmd->args=ASSIGNARRAY;
    break;
  case CALLSTRINGARRAY:
    cmd->args=CALLSTRINGARRAY;
    break;
  case ASSIGNSTRINGARRAY:
    cmd->args=ASSIGNSTRINGARRAY;
    break;
  case GETSTRINGPOINTER:
    cmd->args=GETSTRINGPOINTER;
    break;
  }

  cmd->pointer=ar;
  cmd->args=command;

  return;
}


void doarray(struct command *current) /* call an array */
{
  struct array *ar;
  struct stackentry *stack;
  void *p;
  char **str;
  double *dbl;
  int i,j,bnd,index,call;

  call=(current->args==CALLARRAY || 
	current->args==CALLSTRINGARRAY ||
	current->args==GETSTRINGPOINTER);
  if (!call) stack=pop();

  ar=(struct array *)current->pointer;
  index=0;
  for(i=0;i<ar->dimension;i++) {
    bnd=(ar->bounds[i]);
    index*=bnd;
    j=pop()->value;
    if (j<0 || j>=bnd) {
      sprintf(string,"index %d (=%d) out of range",ar->dimension-i,j);
      error(ERROR,string);
      return;
    }
    index+=j;
  }

  if (call) stack=push();

  p=ar->pointer;
  switch(current->args) {
  case CALLARRAY:
    dbl=(double *)p+index;
    stack->value= *dbl;
    stack->type=NUMBER;
    break;
  case ASSIGNARRAY:
    dbl=(double *)p+index;
    *dbl=stack->value;
    break;
  case CALLSTRINGARRAY:
    str=((char **)p+index);
    stack->pointer=my_strdup(*str);
    stack->type=STRING;
    break;
  case ASSIGNSTRINGARRAY:
    str=((char **)p+index);
    if (*str!=NULL) free(*str);
    *str=my_strdup(stack->pointer);
    break;
  case GETSTRINGPOINTER:
    str=((char **)p+index);
    stack->pointer=*str;
    stack->type=STRING;
    break;
  }
}


void create_changestring(int type) /* create command 'changestring' */
{
  struct command *cmd;

  cmd=add_command(CHANGESTRING);
  cmd->args=type;
}


void changestring(struct command *current) /* changes a string */
{
  int type,a2,a3;
  char *newpart;
  char *oldstring;
  int i,len;
  struct stackentry *a1;

  type=current->args;
  newpart=pop()->pointer;
  if (type>TWOARGS) a3=(int)pop()->value;
  if (type>ONEARGS) a2=(int)pop()->value;
  a1=pop();
  oldstring=a1->pointer; 
  a1->pointer=NULL; /* this prevents push from freeing the memory */
  
  switch(type) {
  case MYMID:
     for(i=1;i<a2+a3;i++) {
      if (!oldstring[i-1]) break;
      if (i>=a2) {
	if (!newpart[i-a2]) break;
	oldstring[i-1]=newpart[i-a2];
      }
    }
    break;
  case MYLEFT:
    for(i=1;i<=a2;i++) {
      if (!oldstring[i-1] || !newpart[i-1]) break;
      oldstring[i-1]=newpart[i-1];
    }
    break;
  case MYRIGHT:
    len=strlen(oldstring);
    for(i=1;i<=len;i++) {
      if (i>len-a2) {
	if (!newpart[i-1-len+a2]) break;
	oldstring[i-1]=newpart[i-1-len+a2];
      }
    }
    break;
  }
}

void create_function(int type) /* create command 'function' */
     /* type can be sin,cos,mid$ ... */
{
  struct command *cmd;
  
  cmd=add_command(FUNCTION);
  cmd->args=type;  
}


void function(struct command *current) /* performs a function */
{
  struct stackentry *stack,*a1,*a2,*a3;
  char *pointer;
  double value;
  int type,result,len,start,i,max;
  char *str,*str2;
  
  type=current->args;
  if (type>TWOARGS) a3=pop();
  if (type>ONEARGS) a2=pop();
  if (type>ZEROARGS) a1=pop();

  switch (type) {
  case MYSIN:
    value=sin(a1->value);
    result=NUMBER;
    break;
  case MYASIN:
    value=asin(a1->value);
    type=NUMBER;
    break;
  case MYCOS:
    value=cos(a1->value);
    result=NUMBER;
    break;
  case MYACOS:
    value=acos(a1->value);
    result=NUMBER;
    break;
  case MYTAN:
    value=tan(a1->value);
    result=NUMBER;
    break;
  case MYATAN:
    value=atan(a1->value);
    result=NUMBER;
    break;
  case MYEXP:
    value=exp(a1->value);
    result=NUMBER;
    break;
  case MYLOG:
    value=log(a1->value);
    result=NUMBER;
    break;
  case MYLEN:
    value=(double) strlen(a1->pointer);
    result=NUMBER;
    break;
  case MYSTR:
    sprintf(string,"%g",a1->value);
    pointer=my_strdup(string);
    result=STRING;
    break;
  case MYSQRT:
    value=sqrt(a1->value);
    result=NUMBER;
    break;
  case MYINT:
    value=(int) a1->value;
    result=NUMBER;
    break;
  case MYFRAC:
    value=a1->value-(int) a1->value;
    result=NUMBER;
    break;
  case MYRAN:
    value=a1->value*(float)rand()/RAND_MAX;
    result=NUMBER;
    break;
  case MYRAN2:
    value=(float)rand()/RAND_MAX;
    result=NUMBER;
    break;
  case MYMIN:
    if (a1->value>a2->value)
      value=a2->value;
    else
      value=a1->value;
    result=NUMBER;
    break;
  case MYMAX:
    if (a1->value>a2->value)
      value=a1->value;
    else
      value=a2->value;
    result=NUMBER;
    break;
  case MYVAL:
    i=sscanf((char *) a1->pointer,"%lf",&value);
    if (i!=1) value=0;
    result=NUMBER;
    break;
  case MYATAN2:
    value=atan2(a1->value,a2->value);
    result=NUMBER;
    break;
  case MYLEFT:
    str=a1->pointer;
    len=a2->value;
    pointer=fromto(str,0,len-1);
    result=STRING;
    break;
  case MYRIGHT:
    str=a1->pointer;
    max=strlen(str);
    len=a2->value;
    pointer=fromto(str,max-len,max-1);
    result=STRING;
    break;
  case MYMID:
    str=a1->pointer;
    start=a2->value;
    len=a3->value;
    pointer=fromto(str,start-1,start+len-2);
    result=STRING;
    break;
  case MYINKEY:
    pointer=inkey();
    result=STRING;
    break;
  case MYCHR:
    pointer=my_malloc(2);
    i=floor(a1->value);
    if (i>255 || i<0) {
      sprintf(string,"cant convert %g to character",a1->value);
      error(ERROR,string);
      return;
    }
    pointer[1]='\0';
    pointer[0]=(unsigned char)i;
    result=STRING;
    break;
  case MYASC:
    value=((unsigned char *)a1->pointer)[0];
    result=NUMBER;
    break;
  case MYUPPER:
    str=a1->pointer;
    pointer=my_malloc(strlen(str)+1);
    i=-1;
    do {
      i++;
      pointer[i]=toupper(str[i]);
    } while(pointer[i]);
    result=STRING;
    break;
  case MYLOWER:
    str=a1->pointer;
    pointer=my_malloc(strlen(str)+1);
    i=-1;
    do {
      i++;
      pointer[i]=tolower(str[i]);
    } while(pointer[i]);
    result=STRING;
    break;
  case MYINSTR:
    str=a1->pointer;
    str2=a2->pointer;
    pointer=strstr(str,str2);
    if (pointer==NULL) 
      value=0;
    else
      value=pointer-str+1;
    result=NUMBER;
    break;   
  case MYSYSTEM:
    str=a1->pointer;
    pointer=do_system(str);
    result=STRING;
    break;
  case MYSYSTEM2:
    str=a1->pointer;
    value=do_system2(str);
    result=NUMBER;
    break;
  default:
    error(ERROR,"function called but not implemented");
    return;
  }
  
  stack=push();
  /* copy result */
  stack->type=result;
  if (result==STRING)
    stack->pointer=pointer;
  else
    stack->value=value;
}


void create_dim(char *name,char type) /* create command 'dim' */
     /* type can be 's'=string or 'd'=double Array */
{ 
  struct command *cmd;
  struct symbol *s;
  struct array *ar;
  int dimcount;

  dimcount=pop()->value;
  cmd=add_command(DIM);
  s=get_sym(name,ARRAY,FALSE); /* search for array */
  if (s!=NULL) {
    sprintf(string,"array '%s' has been dimed already",name);
    error(ERROR,string);
    return;
  }
  s=get_sym(name,ARRAY,TRUE); /* create array */
  ar=my_malloc(sizeof(struct array));
  cmd->pointer=ar;
  s->pointer=ar;
  ar->type=type;
  ar->dimed=FALSE;
  ar->dimension=dimcount;
  if (dimcount>10) {
    error(ERROR,"Dimension larger than 10");
    return;
  }
}


void dim(struct command *cmd) /* get room for array */
{
  struct array *ar;
  struct stackentry *s;
  char *nul,**str;
  double *dbl;
  int total,size,i;
  
  ar=(struct array *)cmd->pointer;
  if (ar->dimed) {
    error(ERROR,"Array has been dimed already");
    return;
  }
  total=1; /* count total amount of memory */
  for(i=0;i<ar->dimension;i++) {
    s=pop();
    if (s->type!=NUMBER) {
      error(ERROR,"Improper index in dim statement");
      return;
    }
    size=(int) s->value;
    if (size<=0) {
      error(ERROR,"One bound is less or equal zero");
      return;
    }
    size++; /* allow for zero-index-element */
    (ar->bounds)[i]=size;
    total*=size;
  }
  ar->total=total;
  if (ar->type=='s')         /* it is a string array */
    ar->pointer=my_malloc(total*sizeof(char *));
  else
    ar->pointer=my_malloc(total*sizeof(double));
  if (ar->pointer==NULL) {
    error(ERROR,"Could not get enough memory for dim");
    return;
  }
  /* initialize Array */
  if (ar->type=='s') { 
    str=ar->pointer;
    for(i=0;i<total;i++) {
      nul=my_malloc(sizeof(char));
      *nul='\0';
      *(str+i)=nul;}}
  else {
    dbl=ar->pointer;
    for(i=0;i<total;i++) *(dbl+i)=0.0;
  }
  ar->dimed=TRUE;
}


char *fromto(char *str,int from,int to) /* gives back portion of string */
     /* from and to can be in the range 1...strlen(str) */
{
  int len,i;
  char *part;
  
  len=strlen(str);
  if (from>to || to<0 || from>len-1) {
    /* give back empty string */
    part=my_malloc(1);
    part[0]='\0';
  }
  else {
    if (from<=0) from=0;
    if (to>=len) to=len-1;
    part=my_malloc(sizeof(char)*(to-from+2)); /* characters and '/0' */
    for(i=from;i<=to;i++) part[i-from]=str[i]; /* copy */
    part[i-from]='\0';
  }
    return part;
}


char *inkey(void) /* gets char from keyboard, blocks and doesnt print */
{
#ifdef UNIX
  struct termios rawterm; /* terminal configured for raw-mode */
  struct termios saveterm; /* saved terminal to be restored */
#elif WINDOWS
  DWORD oflags; /* saves normal state of console input buffer */
  DWORD flags; /* new input mode for console input buffer */
  INPUT_RECORD inrec[8]; /* for reading key-event */
  int num; /* number of valid chars */
#endif
  int i; /* loop index */
  unsigned char *ret; /* string to be returned */
  int numread; /* number of bytes read */
  char prefix='\0'; 

  ret=(char *)malloc(8); /* get memory for key */
#ifdef UNIX
  tcgetattr(0,&rawterm); /* get current state */
  tcgetattr(0,&saveterm); /* remember state */
  rawterm.c_lflag&=~(ECHO|ECHONL|ICANON);
#ifdef VMIN
  rawterm.c_cc[VMIN]=1;
#elif defined(VEOF)
  rawterm.c_cc[VEOF]=1;
#endif
  tcsetattr(0,TCSADRAIN,&rawterm);
  do {
    numread=read(0,ret,7);
  } while(numread==0);
  tcsetattr(0,TCSADRAIN,&saveterm);
#elif WINDOWS
  GetConsoleMode(ConsoleInput,&oflags);
  flags=oflags&~(ENABLE_LINE_INPUT|ENABLE_ECHO_INPUT);
  SetConsoleMode(ConsoleInput,flags);
  do {
    ReadConsoleInput(ConsoleInput,inrec,8,(LPDWORD)&numread);
    num=0;
    for(i=0;i<numread;i++) {
      if (inrec[i].EventType!=KEY_EVENT || 
	  !inrec[i].Event.KeyEvent.bKeyDown ||
	  inrec[i].Event.KeyEvent.wVirtualKeyCode==VK_SHIFT ||
	  inrec[i].Event.KeyEvent.wVirtualKeyCode==VK_CONTROL) continue;
      if (inrec[i].Event.KeyEvent.uChar.AsciiChar) {
	ret[num]=inrec[i].Event.KeyEvent.uChar.AsciiChar;
      }
      else {
	prefix=';';
	ret[num]=inrec[i].Event.KeyEvent.wVirtualKeyCode;
      }
      num++;
    }
  } while(num==0);
  numread=num;
  SetConsoleMode(ConsoleInput,oflags);
#endif
  ret[numread]='\0';
  /* map to printable chars */
  for(i=0;i<numread;i++) {
    if (ret[i]>128) {
      ret[i]-=128;
      if (!prefix) prefix='>';
    }
    if (ret[i]<32) {
      ret[i]+=32;
      if (!prefix) prefix='<';
    }
    if (ret[i]==127) {
      ret[i]=32;
      if (!prefix) prefix='=';
    }
  }
#ifdef WINDOWS
  /* insert prefix */
  if (prefix) {
    for(i=numread;i>=0;i--) ret[i+1]=ret[i];
    ret[0]=prefix;
  }
#endif

  return ret;
}

void create_openwin(int num) /* create Command 'openwin' */
{
  struct command *cmd;
  
  cmd=add_command(OPENWIN);
  cmd->args=num;
}


void openwin(struct command *cmd) /* open a Window */
{
  static first=TRUE; /* flag to decide if initialization is necessary */
#ifdef UNIX
  static XEvent event; /* what has happened ? */
#endif

  if (winopened) {
    error(WARNING,"Window already open");
    return;
  }

  if (cmd->args==3) {
    fontheight=pop()->value;
    get_sym("yabfontheight",NUMBER,FALSE)->value=fontheight;
  }
  if (cmd->args==2 || cmd->args==3) {
    winheight=(unsigned long) pop()->value;
    get_sym("yabwinheight",NUMBER,FALSE)->value=winheight;
    if (winheight<1) {
      error(ERROR,"winheight less than 1 pixel");
      return;
    }

    winwidth=(unsigned long) pop()->value;
    get_sym("yabwinwidth",NUMBER,FALSE)->value=winwidth;
    if (winwidth<1) {
      error(ERROR,"winwidth less than 1 pixel");
      return;
    }
  }

  /* initialize grafics */
  if (first && !grafinit()) return;

#ifdef UNIX

  /* create the window */
  window=XCreateSimpleWindow(display,root,winx,winy,winwidth,winheight,
			     0,0,b_colour);
  if (window==None) {
    error(ERROR,"Could not create window");
    return;
  }

  /* put in name for the window */
  XStoreName(display,window,BASIC_NAME);

  /* set size hints */
  XSetWMNormalHints(display,window,&sizehints);

  /* display it */
  XMapWindow(display,window);

  /* wait for exposure */
  XSelectInput (display,window,ExposureMask); 
  XFlush(display);
  XNextEvent(display,&event);

#else /* WINDOWS */

  thread.winevent=CreateEvent(NULL,FALSE,FALSE,"winevent");
  /* create thread to care for window */
  thread.handle=CreateThread(NULL,0,(LPTHREAD_START_ROUTINE)winthread,
			     0,0,(LPDWORD)&thread.id);
  if (thread.handle==NULL) {
    error(ERROR,"can't create thread for window");
    return;
  }
  
  WaitForSingleObject(thread.winevent,INFINITE);
  DeleteObject(thread.winevent);

#endif
  first=FALSE;
  winopened=TRUE;
  calc_psscale();
}


int grafinit(void)  /* initialize grafics (either X or WIN95) */
{

#ifdef UNIX
  static int screen; /* Number of Screen on Display */
  static XColor asked,got; /* color is complex ... */
  static Colormap map; /* my color map */
  static XGCValues xgcvalues; /* Values for Graphics Context */
  XFontStruct *myfont; /* properties of default font */
  static int w,h; /* width and height of window */
#elif WINDOWS
  int n;
  int f; /* int-value of font */
  char *family; /* font family */
#endif

#ifdef UNIX
 
  /* get display */
  display=XOpenDisplay(displayname);
  if (display==NULL) {
    error(ERROR,"Can't open Display");
    return FALSE;
  }
    
  /* get screen */
  screen=DefaultScreen(display);
  root=RootWindow(display,screen);

  /* care for colors */
  if (DefaultDepth(display,screen)==1) {  /* BW-Screen ? */
    f_colour=BlackPixel(display,screen);
    b_colour=WhitePixel(display,screen); }
  else {
    map=DefaultColormap(display,screen);
    if (foreground==NULL) 
      foreground=XGetDefault(display,BASIC_NAME,"foreground");
    if (foreground==NULL) foreground="black";
      
    if (!XAllocNamedColor(display,map,foreground,&got,&asked)) {
      sprintf(string,"could not get fg-color '%s', trying 'black' instead",
	      foreground);
      error(WARNING,string);
      if (!XAllocNamedColor(display,map,"black",&got,&asked)) {
	error(ERROR,"could not get it");
	return FALSE;
      }
    }
    f_colour=got.pixel;
      
    if (background==NULL) 
      background=XGetDefault(display,BASIC_NAME,"background");
    if (background==NULL) background="white";
    if (!XAllocNamedColor(display,map,background,&got,&asked)) {
      sprintf(string,"could not get bg-color '%s', trying 'white' instead",
	      background);
      error(WARNING,string);
      if (!XAllocNamedColor(display,map,"white", &got,&asked)) {
	error(ERROR,"could not get it");
	return FALSE;
      }
    }
    b_colour=got.pixel;
  }

  /* get size hints */
  if (geometry==NULL) geometry=XGetDefault(display,BASIC_NAME,"geometry");
  XParseGeometry(geometry,&winx,&winy,&w,&h);
  sizehints.x=winx;
  sizehints.y=winy;
  sizehints.flags=USPosition;

  /* get font height */
  if (font==NULL) font=XGetDefault(display,BASIC_NAME,"font");
  if (font==NULL) font="fixed";
  myfont=XLoadQueryFont(display,font);
  if (myfont==NULL) {
    sprintf(string,"could not load font '%s', trying 'fixed' instead",font);
    error(WARNING,string);
    myfont=XLoadQueryFont(display,"fixed");
    if (myfont==NULL) {
      error(ERROR,"could not get it");
      return FALSE;
    }
  }
  xgcvalues.font=myfont->fid;
  fontheight=myfont->ascent;
  get_sym("yabfontheight",NUMBER,FALSE)->value=fontheight;
    
  /* create graphics context, accept defaults ... */
  xgcvalues.foreground=f_colour;
  xgcvalues.background=b_colour;
  gc=XCreateGC(display,root,GCForeground | GCBackground | GCFont,
	       &xgcvalues);
  
#elif WINDOWS

  /* choose font */
  if (!font) font=getreg("font");
  if (!font) font="swiss30";
  f=FF_SWISS;
  fontheight=30;    
  get_sym("yabfontheight",NUMBER,FALSE)->value=fontheight;
    
  family=my_strdup(font);
  for(n=0;*(family+n)!='\0' && !isdigit(*(family+n));n++)
    *(family+n)=tolower(*(family+n));
  if (isdigit(*(family+n))) sscanf(family+n,"%d",&fontheight);
  *(family+n)='\0';
    
  if (!strcmp("decorative",family)) f=FF_DECORATIVE;
  if (!strcmp("dontcare",family)) f=FF_DONTCARE;
  if (!strcmp("modern",family)) f=FF_MODERN;
  if (!strcmp("roman",family)) f=FF_ROMAN;
  if (!strcmp("script",family)) f=FF_SCRIPT;
  if (*family=='\0' || !strcmp("swiss",family)) f=FF_SWISS;
    
  logfont.lfHeight=fontheight;
  logfont.lfWidth=0;
  logfont.lfEscapement=0;
  logfont.lfOrientation=0;
  logfont.lfWeight=FW_DONTCARE;
  logfont.lfItalic=FALSE;
  logfont.lfUnderline=FALSE;
  logfont.lfStrikeOut=FALSE;
  logfont.lfCharSet=DEFAULT_CHARSET;
  logfont.lfOutPrecision=OUT_DEFAULT_PRECIS;
  logfont.lfClipPrecision=CLIP_DEFAULT_PRECIS;
  logfont.lfQuality=DEFAULT_QUALITY;
  logfont.lfPitchAndFamily=DEFAULT_PITCH | f;
  logfont.lfFaceName[0]='\0';
  myfont=CreateFontIndirect(&logfont);

  if (myfont==NULL) {
    sprintf(string,"Could not create font '%s' for screen",font);
    error(ERROR,string);
    return FALSE;
  }
#endif
  return TRUE;
}

void calc_psscale()  /* calculate scale-factor for postscript */
{
  if ((float)winwidth/winheight>(float)18/25)
    psscale=18*0.39*72/winwidth;
  else
    psscale=25*0.39*72/winheight;
}


void create_dot() /* create Command 'dot' */
{
  add_command(DOT);
}


void dot() /* draw a dot */
{
  unsigned long x,y;

  y=(unsigned long) pop()->value;
  x=(unsigned long) pop()->value;
  if (!winopened && !printerfile) {
    error(ERROR,"Got no window to draw");
    return;
  }
  if (winopened) {
#ifdef UNIX
    XDrawPoint(display,window,gc,x,y);
    XFlush(display);
#elif WINDOWS
    startdraw();
    SetPixelV(devcon,x,y,RGB(0,0,0));
    if (printer) {
      MoveToEx(printer,x*prnscale+prnoff,y*prnscale+prnoff,NULL);
      LineTo(printer,x*prnscale+prnoff,y*prnscale+prnoff);
    }
    ReleaseDC(window,devcon);
#endif
  }
  if (printerfile) {
    fprintf(printerfile,"%i %i D\n",
            (int)((x-0.5)*psscale),(int)((winheight-(y+0.5))*psscale));
  }
}


void create_line() /* create Command 'line' */
{
  add_command(LINE);
}


void line() /* draw a line */
{
  unsigned long x1,y1,x2,y2;
  
  y2=(unsigned long) pop()->value;
  x2=(unsigned long) pop()->value;
  y1=(unsigned long) pop()->value;
  x1=(unsigned long) pop()->value;
  if (!winopened && !printerfile) {
    error(ERROR,"Got no window to draw");
    return;
  }
  
  if (winopened) {
#ifdef UNIX
    XDrawLine(display,window,gc,x1,y1,x2,y2);
    XFlush(display);
#elif WINDOWS
    startdraw();
    MoveToEx(devcon,x1,y1,NULL);
    LineTo(devcon,x2,y2);
    if (printer) {
      MoveToEx(printer,x1*prnscale+prnoff,y1*prnscale+prnoff,NULL);
      LineTo(printer,x2*prnscale+prnoff,y2*prnscale+prnoff);
    }
    ReleaseDC(window,devcon);
#endif
  }
#ifdef UNIX
  if(printerfile) {
    fprintf(printerfile,"N\n");
    fprintf(printerfile,"%i %i M\n",
            (int)(x1*psscale),(int)((winheight-y1)*psscale));
    fprintf(printerfile,"%i %i L S\n",
            (int)(x2*psscale),(int)((winheight-y2)*psscale));
    fflush(printerfile);
  }
#endif
}


void create_circle() /* create Command 'circle' */
{
  add_command(CIRCLE);
}


void circle() /* draw a circle */
{
  int x,y,r;
  
  r=pop()->value;
  y=pop()->value;
  x=pop()->value;
  if (!winopened && !printerfile) {
    error(ERROR,"Got no window to draw");
    return;
  }

  if (winopened) {
#ifdef UNIX
    XDrawArc(display,window,gc,x-r,y-r,2*r,2*r,0*64,360*64);
    XFlush(display);
#else /* WINDOWS */
    startdraw();
    Arc(devcon,x-r,y-r,x+r,y+r,0,0,0,0);
    if (printer) {
      Arc(printer,(x-r)*prnscale+prnoff,(y-r)*prnscale+prnoff,
	  (x+r)*prnscale+prnoff,(y+r)*prnscale+prnoff,0,0,0,0); 
    }
    ReleaseDC(window,devcon);
#endif
  }
#ifdef UNIX
  if(printerfile) {
    fprintf(printerfile,"N\n");
    fprintf(printerfile,"%i %i %i C S\n",
            (int)(x*psscale),(int)((winheight-y)*psscale),(int)(r*psscale));
    fflush(printerfile);
  }
#endif
}


void create_text(int str_first) /* create Command 'text' */
{
  struct command *cmd;
  
  cmd=add_command(TEXT);
  cmd->tag=str_first ? 'n' : 's';
}


void text(struct command *cmd) /* write a text */
{
  unsigned long x,y;
  char *text;

  if (cmd->tag=='s') text=(char *)pop()->pointer;
  y=(unsigned long) pop()->value;
  x=(unsigned long) pop()->value;
  if (cmd->tag!='s') text=(char *)pop()->pointer;
  if (!winopened && !printerfile) {
    error(ERROR,"Got no window to draw");
    return;
  }
  if (winopened) {
#ifdef UNIX
    XDrawString(display,window,gc,x,y,text,strlen(text));
    XFlush(display);
#else /* WINDOWS */
    startdraw();
    SelectObject(devcon,myfont);
    SetBkMode(devcon,TRANSPARENT);
    SetTextAlign(devcon,TA_LEFT | TA_BOTTOM);
    TextOut(devcon,x,y,text,strlen(text));
    ReleaseDC(window,devcon);
    if (printer) {
      SelectObject(printer,printerfont);
      SetBkMode(printer,TRANSPARENT);
      SetTextAlign(printer,TA_LEFT | TA_BOTTOM);
      TextOut(printer,x*prnscale+prnoff,y*prnscale+prnoff,text,strlen(text));
    }
#endif
  }
#ifndef WINDOWS
  if (printerfile) {
    fprintf(printerfile,"%i %i M\n",(int)(x*psscale),(int)((winheight-y)*psscale));
    fprintf(printerfile,"(%s) show\n",text);
  }
#endif
}


void create_closewin() /* create Command 'closewin' */
{
  add_command(CLOSEWIN);
}


void closewin() /* close the window */
{
  if (!winopened) {
    error(WARNING,"Got no window to close");
    return;
  }
  winopened=FALSE;
#ifdef UNIX
  XDestroyWindow(display,window); 
  XFlush(display);
#else /* WINDOWS */
  winopened=FALSE;
  PostThreadMessage(thread.id,WM_QUIT,0,0);
#endif
}


void create_clearwin() /* create Command 'clearwin' */
{
  add_command(CLEARWIN);
}


void clearwin() /* clear the window */
{
  if (!winopened && !printerfile) {
    error(WARNING,"Got no window to clear");
    return;
  }
  if (winopened) {
#ifdef UNIX
    XClearWindow(display,window); 
#else /* WINDOWS */
    RECT interior;
    
    startdraw();
    GetClientRect(window,&interior);
    FillRect(devcon,&interior,(HBRUSH) COLOR_WINDOW);
    if (printer) {
      EndPage(printer);
      StartPage(printer);
    }
#endif
  }
  if (printerfile) {
    fprintf(printerfile,"showpage\n");
    fflush(printerfile);
  }
}


void create_mywait() /* create Command 'wait' */
{
  add_command(WAIT);
}


void mywait() /* wait given number of seconds */
{
  double delay;

#if defined(UNIX)
#ifdef HAVE_SETITIMER
  struct itimerval new;
#else
  time_t start,now;
#endif
#endif

#ifdef WINDOWS
  MSG msg;
  int timerid;
#endif
  
  delay=fabs(pop()->value);
#ifdef UNIX
#ifdef HAVE_SETITIMER
  new.it_interval.tv_sec=0.0;
  new.it_interval.tv_usec=0.0;
  new.it_value.tv_sec=floor(delay);
  new.it_value.tv_usec=1000000*(delay-floor(delay));
  setitimer(ITIMER_REAL,&new,NULL);
  signal(SIGALRM,signal_handler);
  pause();
#else /* not HAVE_SETITIMER */
  time(&start);
  do {
    time(&now);
  } while(difftime(now,start)<delay);
#endif
#else /* WINDOWS */
  timerid=SetTimer(NULL,0,delay*1000,(TIMERPROC) NULL);
  GetMessage((LPMSG)&msg,NULL,WM_TIMER,WM_TIMER);
  KillTimer(NULL,timerid);
#endif
}

#if !defined(HAVE_DIFFTIME) && defined(UNIX)
double difftime(long a,long b) /* compute time differences */
{
  return (double)(a-b);
}
#endif


void create_bell() /* create Command 'bell' */
{
  add_command(BELL);
}


void bell() /* ring ascii bell */
{
#ifdef UNIX
  printf("\007");
  fflush(stdout);
#else /* WINDOWS */
  Beep(1000,100);
#endif
}


void pushname(char *name) /* push a name on stack */
{
  struct stackentry *s;

  s=push();
  s->pointer=my_strdup(name);
  s->type=STRING;
}


void pushlabel() /* generate goto and push label on stack */
{
  char *st;
  struct stackentry *en;
  
  st=(char *) my_malloc(sizeof(char)*20);
  sprintf(st,"***%d",labelcount);
  labelcount++;
  create_goto(st);
  en=push();
  en->type=LBL;
  en->pointer=st;
}
  

void poplabel() /* pops a label and generates the matching command */
{
  struct stackentry *en;
  
  en=pop();   /* get name of label */
  if (en->type!=LBL) {
    error(FATAL,"Not a goto on stack");
    return;
  }
  create_label(en->pointer);  /* and create it */
}


void pushgoto() /* generate label and push goto on stack */
{
  char *st;
  struct stackentry *en;
  
  st=(char *) my_malloc(sizeof(char)*20);
  sprintf(st,"***%d",labelcount);
  labelcount++;
  create_label(st);
  en=push();
  en->type=GTO;
  en->pointer=st;
}
  

void popgoto() /* pops a goto and generates the matching command */
{
  struct stackentry *en;
  
  en=pop();   /* get name of goto */
  if (en->type!=GTO) {
    error(FATAL,"Not a goto on stack");
    return;
  }
  create_goto(en->pointer);  /* and create it */
}


void pushcounter(void) 
{
  /* push number '0' on stack, will be used as a counter */

  struct stackentry *p;
  
  p=push();
  p->type=NUMBER;
  p->value=0;
}


void inccounter(void) 
{
  /* increment topmost stack element */

  (stackhead->prev->value)++;
}


void swap() /*swap topmost elements on stack */
{
  struct stackentry *a,*b;
  
  if ((a=stackhead->prev)==NULL || (b=a->prev)==NULL) {
    error(ERROR,"Nothing to swap on stack !");
    return;
  }
  a->prev=b->prev;b->next=a->next;   /* just swap the pointers */
  a->next=b;b->prev=a;
  stackhead->prev=b;
  if (b==stackroot) stackroot=a;  /* treat root special */
  else (a->prev)->next=a;
}


struct stackentry *push() 
     /* push element on stack and enlarge it*/
{
  struct stackentry *new;
  
  if (stackhead->next==NULL) { /* no next element */
    /* create new element */
    new=(struct stackentry *)my_malloc(sizeof(struct stackentry)); 
    /* and initialize it */
    new->next=NULL;  
    new->value=0.0;
    new->type=FREE;
    new->prev=stackhead;
    new->pointer=NULL;
    stackhead->next=new;
  }
  stackhead=stackhead->next; /* advance head */
  /* any content is set free */
  if ((stackhead->prev)->pointer!=NULL && (stackhead->prev)->type==STRING) 
    free((stackhead->prev)->pointer);
  (stackhead->prev)->pointer=NULL;
  return stackhead->prev;
}
    

struct stackentry *pop()
     /* pops element to memory and looks for pop-error */
{
  /* test if there is something on the stack */
  if (stackhead==stackroot) {
    error(FATAL,"Popped too much.");
    return stackhead;
  }
  stackhead=stackhead->prev; /* move down in stack */
  if (stackhead->type==FREE) 
    error(WARNING,"Popped element without content.");
  return stackhead;  /* this is your value; use it quickly ! */
}

    
struct command *add_command(int type) 
     /* get room for new command, and make a link from old one */
{
  struct command *new,*old;

  cmdhead->type=type;  /* store command */
  cmdhead->line=(interactive)?yylineno-1:yylineno;
  commandcount++;
  cmdhead->pointer=NULL;  /* no data yet */ 
  /* no next element, so, create it: */
  new=(struct command *)my_malloc(sizeof(struct command)); 
  /* and initialize */
  new->next=NULL;
  new->pointer=NULL;
  cmdhead->next=new;
  old=cmdhead;
  cmdhead=cmdhead->next;
  return old;
}


void parse_arguments(int argc,char *argv[])
     /* parse arguments from the command line */
{
  int ar;
  FILE *inputfile;
  char *option;
  char info;
  int opened=FALSE;
#ifdef WINDOWS
  char *name;
#endif

  for(ar=1;ar<argc;ar++) {
    option=argv[ar];
    if (!strcmp("-help",option) || !strcmp("--help",option) ||
        !strcmp("-h",option)    || !strcmp("-?",option))
      goto usage;
    else if (!strcmp("-i",option) || !strcmp("-info",option) ||
             !strcmp("-infolevel",option)) {
      ar++;
      info=tolower(*argv[ar]);
      switch(info) {
      case 'd':infolevel=DIAGNOSTIC;break;
      case 'n':infolevel=NOTE;break;
      case 'w':infolevel=WARNING;break;
      case 'e':infolevel=ERROR;break;
      case 'f':infolevel=FATAL;break;
      default:
        fprintf(stderr,"There's  no infolevel '%s'.\n",argv[ar]);
        goto usage;
      }
    }
    else if (!strcmp("-fg",option)) {   
      ar++;
      foreground=my_strdup(argv[ar]);
    }
    else if (!strcmp("-bg",option)) {           
      ar++;
      background=my_strdup(argv[ar]);   
    }
    else if (!strcmp("-geometry",option)) {             
      ar++;
      geometry=my_strdup(argv[ar]);     
    }
    else if (!strcmp("-display",option)) {              
      ar++;
      displayname=my_strdup(argv[ar]);
    }
    else if (!strcmp("-font",option)) {         
      ar++;
      font=my_strdup(argv[ar]);
    }
    else if (!strcmp("-licence",option)) {
      fprintf(stderr,"\n%s\n",YABLICENSE);
      end_it();
      exit(0);
    }
    else if (*option=='-') {
      fprintf(stderr,"Don't know option '%s'.\n",option);
      goto usage;
    }
    else if (!opened) { /* not an option */
      inputfile=fopen(argv[ar],"r");
      if (inputfile==NULL) {
        fprintf(stderr,"Could not open '%s'.\n",argv[ar]);
	end_it();
        exit(1);}
      else {
#ifdef WINDOWS
	name=strrchr(argv[ar],'\\');
	if (!name) name=strrchr(argv[ar],'/');
	if (name)
	  SetConsoleTitle(name+1);
	else
	  SetConsoleTitle(argv[ar]);
#endif
        opened=TRUE;
      }
    }
    else {    /* two filename arguments */
      fprintf(stderr,"Can handle only one file.\n");
      goto usage;
    }
  }
  
  interactive=FALSE;
  if (!opened) {
    interactive=TRUE;
    inputfile=stdin;
  }
    
  /* open a flex buffer for the file */
  switch_to_my_file(inputfile);
  return;

usage: /* print a short usage message and then exit */
  fprintf(stderr,"\n"BASIC_NAME" "BASIC_VERSION" , last change on "DOLC", subject to GNU copyleft.\n");
  fprintf(stderr,"Usage:\n");
  fprintf(stderr,"                "BASIC_NAME" [options] [filename]\n");
  fprintf(stderr,"\n  options:\n");
  fprintf(stderr,"    -i [d|n|w|e|f]  :  set infolevel to diagnostic,note,warning(default),\n                       error or fatal respectively.\n");
  fprintf(stderr,"    -?,-help        :  issue this message.\n");
  fprintf(stderr,"    -geometry       :  e.g. 10+20 to position window at x=10,y=20.\n");
#ifdef UNIX
  fprintf(stderr,"    -fg,-bg         :  colors for grafic.\n");
  fprintf(stderr,"    -display        :  screen, where window will be displayed.\n");
  fprintf(stderr,"    -font           :  font for grafics.\n");
#else /* WINDOWS */
  fprintf(stderr,"    -font           :  font for grafics, specify style and size, e.g swiss10\n");
  fprintf(stderr,"                       style=[decorative|dontcare|modern|roman|script|swiss]\n");
#endif
  fprintf(stderr,"    -licence        :  print licence, then exit.\n");
  fprintf(stderr,"\n  filename: File to read basic-code from. If "BASIC_NAME" is called\n");
  fprintf(stderr,"            without any filename, it will read and execute a program\n");
  fprintf(stderr,"            from STDIN and then exit.\n\n");
  fprintf(stderr,"For further infos on "BASIC_NAME" see the files \""BASIC_NAME".html\" or \""BASIC_NAME".txt\".\n");
#ifdef UNIX
  fprintf(stderr,"\n");
#endif
  end_it();
  exit(0);
}


void chop_command(char *command,int *argc,char ***argv)
     /* chop the WIN95-commandline into seperate strings */
{
  int i,j,count;
  int quote;
  char c,last;
  char *curr;
  char **list;

  /* count, how many arguments */
  count=i=0;
  last=' ';
  quote=FALSE;
  while((c=*(command+i))!='\0') {
    if (!quote && c!=' ' && last==' ') count++;
    if (c=='\"') quote=!quote;
    last=c;
    i++;
  }

  /* fill BASIC_NAME into argv[0] */
  *argv=malloc((count+1)*sizeof(char *));
  list=*argv;
  *argc=count+1;
  *list=malloc(sizeof(char)*strlen(BASIC_NAME));
  strncpy(*list,BASIC_NAME,strlen(BASIC_NAME));

  /* fill in other strings */
  i=0;
  count=1;
  last=' ';
  quote=FALSE;
  do {
    c=*(command+i);
    if (!quote && c!=' ' && last==' ') j=i;
    if (c=='\"') {
      quote=!quote;
      if (quote) j++;
    }
    if (((c==' ' && !quote) || c=='\0') && last!=' ') {
      *(list+count)=malloc((i-j+1)*sizeof(char));
      strncpy(*(list+count),command+j,i-j);
      curr=*(list+count)+i-j;
      *curr='\0';
      if (*(curr-1)=='\"') *(curr-1)='\0';
      count++;
    }
    last=c;
    i++;
  } while(c!='\0');
}

void end_it(void) /* perform shutdown-operations */
{
#ifdef UNIX
  if (curinized) endwin();
#else
  if (!Commandline) {
    printf("Program done --- Press <RETURN>");
    getc(stdin);
  }
  if (printerfont) DeleteObject(printerfont);
  if (myfont) DeleteObject(myfont);
  if (printer) DeleteDC(printer);
#endif
}


int do_system2(char *cmd) /* execute command as system */
{
#ifdef UNIX
  return system(cmd);
#else
  STARTUPINFO start;
  PROCESS_INFORMATION proc;
  DWORD ec; /* exit code */

  GetStartupInfo(&start);
  sprintf(string,"command.com /C %s",cmd);
  if (!CreateProcess(NULL,string,NULL,NULL,TRUE,NORMAL_PRIORITY_CLASS,
		     NULL,NULL,&start,&proc)) {
    sprintf(string,"couldn't execute '%s'",cmd);
    error(ERROR,string);
    return -1;
  }
  WaitForSingleObject(proc.hProcess,INFINITE);
  if (GetExitCodeProcess(proc.hProcess,&ec)) return ec;
  return -1;
#endif  
}


char *do_system(char *cmd) /* executes command via command.com */
{
  static char buff[10000]; /* buffer to store command */
  int len; /* number of bytes read */
#ifdef UNIX
  FILE *p; /* points to pipe */
  int c; /* char read from pipe */

  p=popen(cmd,"r");
  if (p==NULL) {
    sprintf(string,"couldn't execute '%s'",cmd);
    error(ERROR,string);
    return my_strdup("");
  }
  len=0;
  while(len<10000) {
    c=fgetc(p);
    if (c==EOF) break;
    buff[len]=c;
    len++;
  }
  buff[len]='\0';
  pclose(p);
#else
  HANDLE sout; /* saved standard output-handle */
  HANDLE serr; /* saved standard error-handle */
  STARTUPINFO start;
  PROCESS_INFORMATION proc;
  HANDLE piperead,pipewrite; /* both ends of pipes */
  PSECURITY_DESCRIPTOR SD=NULL; /* specifying access to pipe */
  SECURITY_ATTRIBUTES  sa; /* security attributes for pipe */
	

  SD=(PSECURITY_DESCRIPTOR)my_malloc(SECURITY_DESCRIPTOR_MIN_LENGTH);
  InitializeSecurityDescriptor(SD,SECURITY_DESCRIPTOR_REVISION);
  /* add a NULL disc. ACL to the security descriptor. */	
  SetSecurityDescriptorDacl(SD,TRUE,(PACL)NULL,FALSE);
  sa.nLength=sizeof(sa);
  sa.lpSecurityDescriptor=SD;
  sa.bInheritHandle=TRUE;
		
  /* save standard handles */
  sout=GetStdHandle(STD_OUTPUT_HANDLE);
  serr=GetStdHandle(STD_ERROR_HANDLE);

  /* create pipe for writing */
  CreatePipe(&piperead,&pipewrite,&sa,0);

  /* bend standard handles */
  SetStdHandle(STD_OUTPUT_HANDLE,pipewrite);
  SetStdHandle(STD_ERROR_HANDLE,pipewrite);

  /* initialize startupinfo structure */
  GetStartupInfo(&start);
  start.cb=sizeof(STARTUPINFO); 
  start.lpReserved=NULL; 
  start.lpReserved2=NULL; 
  start.cbReserved2=0; 
  start.lpDesktop=NULL;  
  start.dwFlags=STARTF_USESTDHANDLES;
  start.hStdOutput=GetStdHandle(STD_OUTPUT_HANDLE);
  start.hStdError=GetStdHandle(STD_ERROR_HANDLE);
  start.hStdInput=GetStdHandle(STD_INPUT_HANDLE);

  sprintf(string,"command.com /C %s",cmd);
  if (!CreateProcess(NULL,string,NULL,NULL,TRUE,NORMAL_PRIORITY_CLASS,
		     NULL,NULL,&start,&proc)) {
    sprintf(string,"couldn't execute '%s'",cmd);
    error(ERROR,string);
    return my_strdup("");
  }

  /* wait for output to arrive */
  WaitForSingleObject(proc.hProcess,INFINITE);
  ReadFile(piperead,buff,10000,(LPDWORD)&len,NULL);
  buff[len]='\0';
  /* restore saved handle values */
  SetStdHandle(STD_OUTPUT_HANDLE,sout);
  SetStdHandle(STD_ERROR_HANDLE,serr);
  CloseHandle(piperead);
  CloseHandle(pipewrite);  
  free(SD);
#endif
  return my_strdup(buff);
}

#ifdef WINDOWS  /* routines needed only for WIN95 */
/* procedure to process WIN95 messages */
LRESULT CALLBACK mywindowproc(HWND handle,unsigned msg,
                              UINT wparam,DWORD lparam)
{
  switch(msg) {
  default:
    return(DefWindowProc(handle,msg,wparam,lparam));
  }
}


/* procedure for WIN95-thread */
DWORD winthread(LPWORD par) {
  MSG msg;
  int w,h;
  RECT cr; /* client area rectangle */

  if (!geometry) geometry=getreg("geometry");
  if (geometry) 
    if (sscanf(geometry,"%ix%i+%i+%i",&w,&h,&winx,&winy)!=4)
      if (sscanf(geometry,"+%i+%i",&winx,&winy)!=2)
	if (sscanf(geometry,"%i+%i",&winx,&winy)!=2) 
	  winx=winy=30;

  /* get window-size from client-area size */
  cr.left=winx;
  cr.right=winx+winwidth;
  cr.top=winy;
  cr.bottom=winy+winheight;
  AdjustWindowRectEx(&cr,WS_VISIBLE | WS_CAPTION | WS_SYSMENU,
		     FALSE, WS_EX_TOPMOST);

  /* create my window */
  window=CreateWindowEx(WS_EX_TOPMOST,
			my_class, 
			NULL,            /* my style */
			WS_VISIBLE | WS_CAPTION | WS_SYSMENU,/* window style */
			winx,               /* initial x-position */
			winy,               /* initial y-position */
			cr.right-cr.left,      /* initial x-size */
			cr.bottom-cr.top,    /* initial y-size */
			NULL,            /* parent window */
			NULL,            /* menu handle */
			this_instance,   /* my instance */
			(LPVOID) NULL);  /* dont know why */
  
  /* show my window */
  SetWindowText(window,BASIC_NAME" - Grafic Window");
  ShowWindow(this_instance,SW_SHOWDEFAULT);
  UpdateWindow(this_instance);
  winopened=TRUE;
  SetEvent(thread.winevent);

  /* get and dispatch messages */
  while(GetMessage((LPMSG)&msg,NULL,0,0))
    DispatchMessage((LPMSG)&msg);
  
  DestroyWindow(window);
  ExitThread(0);
  return 0;
}


void startdraw() /* prepare for drawing */
{
  RECT interior;
  
  devcon=GetDC(window);
  GetClientRect(window,&interior);
  SelectClipRgn(devcon,NULL);
  IntersectClipRect(devcon,interior.left,interior.top,
		    interior.right,interior.bottom);
}


char *getreg(char *name) /* get default from Registry */
{
  char *keyname="SOFTWARE\\"BASIC_NAME;
  HKEY key;
  char reg[80];
  DWORD n;

  RegOpenKeyEx(HKEY_LOCAL_MACHINE,keyname,0,KEY_ALL_ACCESS,&key);
  n=80;
  reg[0]='\0';
  RegQueryValueEx(key,name,NULL,NULL,reg,&n);
  if (reg[0]=='\0') return NULL;
  return my_strdup(reg);
}
#endif  /* WIN95-stuff ... */


void create_openprinter(int num) /* create command 'openprinter' */
{
  struct command *cmd;
  
  cmd=add_command(OPENPRN);
  cmd->args=num;
}


void openprinter(struct command *cmd) /* opens a printer for WIN95 */
{
  static int first=TRUE;
#ifdef WINDOWS
  char PrinterName[200]; /* Name of default Printer */
  char *p; /* points into PrinterName */
  DOCINFO di;
  float prnwidth,prnheight;
  LOGBRUSH mybrush;
  RECT interior;
#endif  

  /* close file, if already open */
#ifdef UNIX
  if (printerfile) closeprinter();
#endif

  if (cmd->args==1) {
    prfilename=my_strdup((char *)(pop()->pointer));
    print_to_file=TRUE;}
  else {
    prfilename="\0";
    print_to_file=FALSE;
  }
#ifndef WINDOWS
  if (*prfilename=='\0') {
    free(prfilename);
    prfilename=my_strdup("/tmp/yabasic.ps");
  }
  if (*prfilename!='\0') {
    printerfile=fopen(prfilename,"w");
    if (!printerfile) {
      sprintf(string,"could not open file '%s' for printing",prfilename);
      error(ERROR,string);
    }
  }
#endif
  if (!first) return;

#ifdef WINDOWS
  /* query win.ini for defaul printer */
  GetProfileString("windows","device",",,,",
		   PrinterName,200);
  
  /* truncate printer name */
  p=PrinterName;
  while(*p && *p!=',') p++;
  *p='\0';

  printer=CreateDC(NULL,PrinterName,NULL,NULL);
  if (!printer) printer=CreateDC(NULL,"winspool",NULL,NULL);

  if (!printer) {
    error(ERROR,"Couldn't get handle for printer");
    return;
  }
  if (SelectObject(printer,CreateSolidBrush(RGB(0,0,0)))==NULL) {
    error(ERROR,"Could not select brush for printer");
    return;
  }
  
  /* calculate scaling-factors */
  prnwidth=(float)GetDeviceCaps(printer,HORZRES);
  prnheight=(float)GetDeviceCaps(printer,VERTRES);
  if (prnwidth/winwidth>prnheight/winheight) {
    prnscale=0.7*prnheight/winheight;
    prnoff=0.15*prnheight;}
  else {
    prnscale=0.7*prnwidth/winwidth;
    prnoff=0.15*prnwidth;
  }
  
  /* set clipping region */
  GetClientRect(window,&interior);
  SelectClipRgn(printer,NULL);
  IntersectClipRect(printer,
		    interior.left*prnscale+prnoff,
		    interior.top*prnscale+prnoff,
		    interior.right*prnscale+prnoff,
		    interior.bottom*prnscale+prnoff);
  
  /* create printerfont */
  logfont.lfHeight=fontheight*prnscale;
  printerfont=CreateFontIndirect(&logfont);
  if (printerfont==NULL) {
    sprintf(string,"Could not create font for printer");
    error(ERROR,string);
    return;
  }
  
  /* create printerpen */
  mybrush.lbStyle=BS_SOLID;
  mybrush.lbColor=DIB_RGB_COLORS;
  mybrush.lbHatch=HS_DIAGCROSS;
  printerpen=ExtCreatePen(PS_GEOMETRIC,prnscale,&mybrush,0,NULL);
  if (SelectObject(printer,printerpen)==NULL) {
    error(ERROR,"Couldn't select printerpen");
    return;
  }
  
  di.cbSize=sizeof(DOCINFO);
  di.lpszDocName=BASIC_NAME" grafics";
  di.lpszOutput=(print_to_file) ? prfilename : (LPTSTR)NULL;
  di.lpszDatatype=(LPTSTR)NULL;
  di.fwType=0;
  if (StartDoc(printer,&di)==SP_ERROR) {
    error(ERROR,"Couldn't start printing");
    return;
  }
  StartPage(printer);
  first=FALSE;
#else /* UNIX */
  fprintf(printerfile,"%%!PS-Adobe-1.0\n");
  fprintf(printerfile,"%%%%Title: "BASIC_NAME"-Grafics\n");
  fprintf(printerfile,"%%%%BoundingBox: 0 0 %i %i\n",
	  (int)(winwidth*psscale),(int)(winheight*psscale));
  fprintf(printerfile,"%%%%DocumentFonts: Helvetica\n");
  fprintf(printerfile,"%%%%Creator: "BASIC_NAME"\n");
  fprintf(printerfile,"%%%%Pages: (atend)\n");
  fprintf(printerfile,"%%%%EndComments\n");
  fprintf(printerfile,"gsave\n");
  fprintf(printerfile,"/M {moveto} def\n");
  fprintf(printerfile,"/RL {rlineto} def\n");
  fprintf(printerfile,"/L {lineto} def\n");
  fprintf(printerfile,"/N {newpath} def\n");
  fprintf(printerfile,"/S {stroke} def\n");
  fprintf(printerfile,"/D {N M 0 %g RL %g 0 RL 0 %g RL closepath fill} def\n",
	  psscale,psscale,-psscale);
  fprintf(printerfile,"/C {N 0 360 arc} def\n");
  fprintf(printerfile,"/Helvetica findfont\n");
  fprintf(printerfile,"%g scalefont setfont\n",(double)fontheight*psscale);
  fprintf(printerfile,"30 30 translate\n");
  fflush(printerfile);
#endif
}


void create_closeprinter() /* create command 'closeprinter' */
{
  add_command(CLOSEPRN);
}


void closeprinter() /* closes printer for WIN95 */
{
#ifdef WINDOWS
  EndPage(printer);
  EndDoc(printer);
#else /* UNIX */
  if (printerfile) {
    fprintf(printerfile,"showpage\ngrestore\n%%%%Trailer\n");
    fclose(printerfile);
    printerfile=NULL;
    if (!strncmp(prfilename,"/tmp/",5)) {
      sprintf(string,"lpr %s",prfilename);
      if (system(string)) {
	sprintf(string,"couldn't print '%s'",prfilename);
	error(ERROR,string);
	return;
      }
      remove(prfilename);
    }
  }
#endif
  if (prfilename) free(prfilename);
}


void initialize() 
     /* give correct values to pointers etc ... */
{
  int i;
  
  /* install exception handler */
  signal(SIGFPE,signal_handler);
  signal(SIGSEGV,signal_handler);
 
  /* initialize error handling: no errors seen 'til now */
  errorlevel=DIAGNOSTIC;  
  diagnostic_count=0;
  note_count=0;
  warning_count=0;
  error_count=0;

  /* initialize symbol table */
  symroot=(struct symbol *)my_malloc(sizeof(struct symbol)); /* ceate first */
  symroot->type=FREE;
  symroot->pointer=NULL;
  symroot->next=NULL;
  symroot->name=NULL;
  symroot->value=0.0;
  
  /* initialize numeric stack */
  /* create first : */
  stackroot=(struct stackentry *)my_malloc(sizeof(struct stackentry)); 
  stackroot->next=NULL;
  stackroot->prev=NULL;
  stackroot->value=0.0;

  /* initialize command stack */
  /* create first : */
  cmdroot=(struct command *)my_malloc(sizeof(struct command)); 
  cmdroot->next=NULL;

  /* initialize random number generator */
  srand((unsigned int)time(NULL));

  reset();

  datapointer=cmdroot; /* restore for read data */

  /* file stuff */
  for(i=1;i<=9;i++) streams[i]=NULL;
  printerfile=NULL; /* no ps-file yet */
}


void signal_handler(int sig)   /* handle signals */
{
  switch (sig) {
  case SIGFPE:
    error(FATAL,"Floating point exception, cannot proceed.");
    end_it();
    exit(1);
  case SIGSEGV:
    error(FATAL,"Segmentation violation, cannot proceed.");
    end_it();
    exit(1);
#ifdef UNIX
  case SIGALRM: /* ignore */
    break;
#endif
  default:
    break;
  }
}


void reset() 
     /*
   reset pointers to their initial values, 
   initialize variables and functions 
*/
{
  struct symbol *s;
  struct stackentry *base;
  int i;

  symhead=symroot; /* list of double symbols */
  stackhead=stackroot; /* stack of double values */
  base=push();
  base->type=NIL; /* push nil, so that pop will not crash */
  cmdhead=cmdroot; /* list of commands */;
  commandcount=0;

  /* create useful variables */
  s=get_sym("PI",NUMBER,TRUE);
  s->value=3.14159265359;
  s=get_sym("pi",NUMBER,TRUE);
  s->value=3.14159265359;
  s=get_sym("EULER",NUMBER,TRUE);
  s->value=2.71828182864;
  s=get_sym("euler",NUMBER,TRUE);
  s->value=2.71828182864;

  /* add internal variables */
  get_sym("yabinfolevel",NUMBER,TRUE)->value=infolevel;
  get_sym("yabdiagnostic",NUMBER,TRUE)->value=DIAGNOSTIC;
  get_sym("yabnote",NUMBER,TRUE)->value=NOTE;
  get_sym("yabwarning",NUMBER,TRUE)->value=WARNING;
  get_sym("yaberror",NUMBER,TRUE)->value=ERROR;
  get_sym("yabfatal",NUMBER,TRUE)->value=FATAL;
  get_sym("yabfontheight",NUMBER,TRUE)->value=10;
  get_sym("yabwinheight",NUMBER,TRUE)->value=100;
  get_sym("yabwinwidth",NUMBER,TRUE)->value=100;  
  s=get_sym("yabscreenwidth",NUMBER,TRUE);
  s->value=0;
  s=get_sym("yabscreenheight",NUMBER,TRUE);
  s->value=0;

  /* add internal string variables */
  s=get_sym("yabos$",STRING,TRUE);
  free(s->pointer);
#ifdef UNIX
  s->pointer=my_strdup("unix");
#else
  s->pointer=my_strdup("windows");
#endif

  /* set default-scales for grafics */
  fontheight=10;
  get_sym("yabfontheight",NUMBER,FALSE)->value=fontheight;
  winheight=100;
  get_sym("yabwinheight",NUMBER,FALSE)->value=winheight;
  winwidth=100;
  get_sym("yabwinwidth",NUMBER,FALSE)->value=winwidth;
  calc_psscale();

  /* file stuff */
  for(i=1;i<=9;i++) 
    if (streams[i]!=NULL) {
      sprintf(string,"Stream %d not closed; closing it now",i);
      error(NOTE,string);
      fclose(streams[i]);
    }
}


void run_it()
     /* execute the compiled code */
{
  int endflag=FALSE;
  int i;

  current=cmdroot; /* start with first comand */
  while(current!=cmdhead && !endflag && errorlevel>ERROR) {
    /* print command name */
    if (infolevel>=DIAGNOSTIC) {
      for(i=FIRST_COMMAND+1;i<LAST_COMMAND;i++) {
	if (current->type==explanation[i].command) {
	  sprintf(string,"executing command '%s'",explanation[i].text);
	  break;
	}
      }
      if (i==LAST_COMMAND) {
	sprintf(string,"executing unknown command %d",current->type);
      }
      error(DIAGNOSTIC,string);
    }
    switch(current->type) {
    case GOTO:case QGOTO:case GOSUB:case QGOSUB:
      jump(current); DONE;
    case SKIPPER: 
      skipper(); break;
    case LABEL:case DATA:case NOP: 
      DONE;
    case RETURN:
      myreturn(); DONE;
    case PUSHDBLSYM: 
      pushdblsym(current); DONE;
    case PUSHDBL:
      pushdbl(current); DONE;
    case POPDBLSYM:
      popdblsym(current); DONE;
    case POPSTRSYM:
      popstrsym(current); DONE;
    case PUSHSTRSYM: 
      pushstrsym(current); DONE;
    case PUSHSTR:
      pushstr(current); DONE;
    case CONCAT:
      concat(); DONE;
    case PRINT:
      print(current); DONE;
    case MOVE:
      mymove(); DONE;
    case CLEARSCR:
      clearscreen(); DONE;
    case TESTEOF:
      testeof(current); DONE;
    case MYOPEN:
      myopen(current); DONE;
    case MYCLOSE:
      myclose(current); DONE;
    case MYSWITCH:
      myswitch(current); DONE;
    case MYREAD:
      myread(current); DONE;
    case RESTORE:case QRESTORE:
      restore(current); DONE;
    case READDATA:
      readdata(current); DONE;
    case PROMPT:
      prompt(current); DONE;
    case DBLADD:case DBLMIN:case DBLMUL:case DBLDIV:case DBLPOW:
      dblbin(current); DONE;
    case NEGATE:
      negate(); DONE;
    case EQ:case NE:case GT:case GE:case LT:case LE:
      dblrelop(current); DONE;
    case STREQ:case STRNE:case STRLT:case STRLE:case STRGT:case STRGE:
      strrelop(current); DONE;
    case AND:case OR:case NOT:
      boole(current); DONE;
    case FUNCTION:
      function(current); DONE;
    case DOARRAY:
      doarray(current); DONE;
    case CHANGESTRING:
      changestring(current); DONE;
    case PUSHSTRPTR:
      pushstrptr(current); DONE;
    case DIM:
      dim(current); DONE;
    case DECIDE:
      decide(); DONE;
    case OPENWIN:
      openwin(current); DONE;
    case OPENPRN:
      openprinter(current); DONE;
    case CLOSEPRN:
      closeprinter(); DONE;
    case DOT:
      dot(); DONE;
    case LINE:
      line(); DONE;
    case CIRCLE:
      circle(); DONE;
    case TEXT:
      text(current); DONE;
    case CLOSEWIN:
      closewin(); DONE;
    case CLEARWIN:
      clearwin(); DONE;
    case WAIT:
      mywait(); DONE;
    case BELL:
      bell(); DONE;
    case SETINFOLEVEL:
      setinfolevel(); DONE;
    case SETFONTHEIGHT:
      setfontheight(); DONE;
    case SETWINHEIGHT:
      setwinheight(); DONE;
    case SETWINWIDTH:
      setwinwidth(); DONE;
    case END:
      endflag=TRUE; break;
    default:
      sprintf(string,"Unkown Interpreter-Command, Token %d.",current->type);
      error(ERROR,string);
      break;
    }
  }
  program_state=FINISHED;
  switch(errorlevel) {
  case NOTE:case DIAGNOSTIC: 
    error(NOTE,"Program ended normally."); break;
  case WARNING:
    error(WARNING,"Program ended with a warning"); break;
  case ERROR:
    error(ERROR,"Program stopped due to an error"); break;
  case FATAL: /* should not come here ... */
    error(FATAL,"Program terminated due to FATAL error");
    break;
  }
}


void error(int severe, char *message) 
     /* reports an basic error to the user and possibly exits */
{
  
  if (severe<=infolevel) {
    fprintf(stderr,"---");
    switch(severe) {
    case(DIAGNOSTIC): 
      fprintf(stderr,"Diagnostic"); 
      diagnostic_count++;
      break;
    case(NOTE): 
      fprintf(stderr,"Note"); 
      note_count++;
      break;
    case(WARNING): 
      fprintf(stderr,"Warning"); 
      warning_count++;
      break;
    case(ERROR): 
      fprintf(stderr,"Error"); 
      error_count++;
      break;
    case(FATAL): 
      fprintf(stderr,"Fatal"); 
      break;
    }
    if (program_state==COMPILING) {
      if (yylineno<0) 
	fprintf(stderr," at end of file");
      else
	fprintf(stderr," in line %d",(interactive)?yylineno-1:yylineno);
    }
    else if (program_state==RUNNING && current->line>0) 
      fprintf(stderr," in line %d",current->line);

    fprintf(stderr,": %s\n\r",message);
  }
  if (severe<errorlevel) errorlevel=severe;
  if (severe<=FATAL) {
    fprintf(stderr,"---Immediate exit to system, due to a fatal error.\n");
    end_it();
    exit(1);
  }
}


char *my_strndup(char *arg,int len) /*  own version of strndup */
{
  char *copy;

  copy=my_malloc(len+1);
  if (!copy) return NULL;

  strncpy(copy,arg,len);
  copy[len]='\0';

  return copy;
}
  

char *my_strdup(char *arg)  /* my own version of strdup, checks for failure */
{
  int l;

  l=strlen(arg);
  return my_strndup(arg,l);
}


void *my_malloc(unsigned num) /* Alloc memory and issue warning on failure */
{
  void *room;
  
  room=malloc(num);
  if (room==NULL) {
    sprintf(string,"Can't malloc %d bytes of memory",num);
    error(FATAL,string);
  }
  return room;
}
