// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#ifndef _TextRenderer_h_
#define _TextRenderer_h_

#include <qcolor.h>
#include <qfont.h>
#include "SgmlRenderer.h"

class ButtonRenderer;
class CheckBoxRenderer;
class EntryRenderer;
class ImageRenderer;
class ListBoxRenderer;
class OptionRenderer;
class RadioRenderer;
class TextAreaRenderer;

//: The Anchor class associates a name with a physical position in the rendered document.
class AnchorRenderer : public Renderer {
    Q_OBJECT
    QString _name;
public:
    AnchorRenderer( const QString& anchorName,
                    Canvas*        canvas,
                    int            clipWidth=0,
                    QObject*       parent=0,
                    const char*    name=0 );

    const QString& name();
};

class StyleRenderer : public Renderer {
    Q_OBJECT
    QFont   _font;
    QColor  _color;
    QString _url;
    int     _spaceWidth;
    int     _ascent;
public:
    StyleRenderer( const QFont&   font,
                   const QColor&  color,
                   const QString& url,
                   int            spaceWidth,
                   int            ascent,
                   Canvas*        canvas,
                   int            clipWidth=0,
                   QObject*       parent=0,
                   const char*    name=0 );
    ~StyleRenderer();

    const QFont& font();

    int spaceWidth();

    void repaint( QPainter& p, const Rect& r );
};

//: The Word class stores all the data needed to renderer a single word.
class WordRenderer : public Renderer {
    Q_OBJECT
    //. The actual text of the word.  For preformatted text, this can contain
    //. spaces.
    QString _word;

    //. The width of the space before this word.  If this is zero and the
    //. spaceAfter the preceding word is also zero then there is no break
    //. between the two words, just a change in font, color, or url.  Otherwise,
    //. this is the width of a single space in the word's font.
    bool    _spaceBefore;

    //. The width of the space after this word.  If this is zero and the
    //. spaceBefore the next word is also zero then there is no break between
    //. the two words, just a change in font, color, or url.  Otherwise, this
    //. is the width of a single space in the word's font.
    bool    _spaceAfter;

    //. If set to TRUE then this is the last word on the line.  This member has
    //. meaning only for the PreformatRenderer.
    bool    _endOfLine;
public:
    //. Construct a Word.
    WordRenderer( const QString& word,
                  bool           spaceBefore,
                  bool           spaceAfter,
                  bool           endOfLine,
                  Canvas*        canvas,
                  int            clipWidth=0,
                  QObject*       parent=0,
                  const char*    name=0 );

    ~WordRenderer();

    const QString& word();

    bool spaceBefore();

    bool spaceAfter();

    bool endOfLine();

    void repaint( QPainter& p, const Rect& r );
};

//: The Lines class stores a list of Inline elements.
struct Line : public Rect {
    StyleRenderer*  style;
    QList<Renderer> renderers;
};

//: The TextRenderer class is the base class for all block-level renderers.
//. The TextRenderer maintains a list of inlined elements (words and images),
//. and repaints them on demand.  It also keeps a list of mouse zones for
//. clickable hyperlinks.
class TextRenderer : public SgmlRenderer {
    Q_OBJECT
public:
    struct MouseZone {
        Rect    zone;
        QString url;
        MouseZone( const Rect& _zone, const QString& _url );
    };
protected:
    friend class PlainRenderer;
    
    QString           _leftover;
    bool              _done;
    bool              _startOfContent;
    bool              _spaceBefore;
    bool              _spaceAfter;
    QList<Renderer>   _renderers;
    QList<Line>       _lines;
    TextAreaRenderer* _textArea;
    OptionRenderer*   _option;
    ListBoxRenderer*  _listBox;
    int               _prefix;
    QString           _optionText;
    QString           _optionValue;
    bool              _optionSelected;

    const QString findHyperlink( bool endTag=FALSE );
    
    StyleRenderer* makeStyle( Style* style, bool endTag=FALSE );
public:
    //. Create a TextRenderer.  Nothing new here, compared to the SgmlRenderer.
    TextRenderer( Canvas*     canvas,
                  SgmlParser* parser,
                  int         clipWidth=0,
                  QObject*    parent=0,
                  const char* name=0 );

    //. This is a virtual destructor.
    virtual ~TextRenderer();

    bool findAnchor( const QString& name, int& x, int& y );

    void repaint( QPainter& p, const Rect& r );
public slots:
    //. Process the endOfData signal from the SgmlParser.  Finish processing
    //. any previously leftover content.
    virtual void endOfData();

    //. Process the endTag signal from the SgmlParser.  Finish processing any
    //. leftover content.
    virtual void endTag();

    //. Process the startTag signal from the SgmlParser.  Finish processing any
    //. leftover content.
    virtual void startTag();
};

#endif
