// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
// GifDecoder written by Markus Demleitner
// msdemlei@ari.uni-heidelberg.de
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

extern "C"
{
#include <time.h>
}

#include <qpainter.h>
#include <qcolor.h>
#include "Canvas.h"
#include "ConsoleWindow.h"
#include "Options.h"
#include "GifDecoder.h"
#include "Renderer.h"

#define HASHSIZ 4096 //Size of GIF hash table.  Spec has it that codes can't
//be longer than 12 bits
#define UPDATE_INTERVAL 1 // Seconds between two renders of images being
// decoded

GifDecoder::GifDecoder( Renderer* renderer )
        : ImageDecoder( renderer )
{ 
    headerBytes = 0;
    isGif89 = FALSE;
    broken = 0;
    bufPtr = 0;
    lookingForNextBlock = FALSE;
    readingColorTable = FALSE;
    readingImageDescriptor = FALSE;
    readingImageData = FALSE;
    readingExtension = FALSE;
    readingImageBlock = FALSE;
    skippingDataBlocks = FALSE;
    gotOneIm = FALSE;
    canDraw = FALSE;
    bytesLeft = 0;
    drawnToLine = 0;
    transColor = -1;
    im = new QImage();
    mask = new QImage();
    lastUpdate = (int)time( 0 );
}

GifDecoder::~GifDecoder()
{
    delete im;
    if ( mask )
        delete mask;
}

void GifDecoder::data( const char* bytes, int length )
{
    if ( canDraw )  //Let's see if we want to do a redraw
        if ( (int)time( 0 ) - lastUpdate > UPDATE_INTERVAL ) {
            renderProgressive();
	    lastUpdate = (int)time( 0 );
        }
    while ( 1 ) {
        // If we already have an error, forget the whole thing
        if ( broken || gotOneIm ) {
            return;
        }

        // Do we already know the Format?
        if ( headerBytes < 6 ) {
            for ( ; ( headerBytes < 6 ) && ( length > 0 ) ; length-- ) {
                header[headerBytes++] = *bytes++;
            }

            if ( headerBytes < 6 ) {
                return;
            }
        
            // Now I've got 6 Bytes in header.  That's enough to decide if
            // it's Gif89 or Gif87 
            if ( ! strncmp( (char*)header, "GIF89a", 6 ) )
                isGif89 = TRUE;
            else if ( strncmp( (char*)header, "GIF87a", 6 ) ) {
                broken = 1;  // Wrong magic.  Bummer!
                return;
            }
        }

        // Do we already have a screen descriptor?
        if ( headerBytes < 13 ) {
            for ( ; ( headerBytes < 13 ) && ( length > 0 ) ; length-- ) {
                header[headerBytes++] = *bytes++;
            }

            if ( headerBytes < 13 ) {
                return;
            }
        
            width = (header[7]<<8)+header[6];
            height = (header[9]<<8)+header[8];
            bitsPerPixel = (header[10]&0x07);
            hasGlobalColorTable = (header[10]&0x80) != 0;
            //Now we can allocate the image the picture is read into
            im->create( width, height, 8, 0, QImage::IgnoreEndian );
            mask->create ( width, height, 1, 2, QImage::BigEndian );
            mask->fill ( 0 ); // qweb doc says that's done automatically.
            // I don't think so.
            if ( hasGlobalColorTable ) 
                readingColorTable = TRUE;
            else
                lookingForNextBlock = TRUE;
            if ( !length ) 
                return;
        }

        if ( readingColorTable ) {
            //Put bytes into buffer, process them if all colors we expect are there
            while ( bufPtr < 3*(2<<bitsPerPixel) ) {
                buf[bufPtr++] = *bytes++;
                if ( !--length )
                    return;
            }
            //Now the color table is complete and can be processed
            getGifColorTable( buf, 2<<bitsPerPixel );
            readingColorTable = FALSE;
            bufPtr = 0;
            lookingForNextBlock = TRUE;
            if ( !length )
                return;
        }

        if ( lookingForNextBlock ) {
            length --;
            switch ( *bytes++ ) {
                case 0x2c: //Image descriptor
                    readingImageDescriptor = TRUE;
                    lookingForNextBlock = FALSE;
                    break;
                case 0x21: //Extension introducer;
                    readingExtension = TRUE;
                    lookingForNextBlock = FALSE;
                    break;
                case 0x3b: //Trailer
                    return;
                case 0: //Block Terminator, handle according to state
                    if ( skippingDataBlocks )
                        skippingDataBlocks = FALSE;
                    continue;
                default: 
                    broken = 2;
                    return;
            }
            if ( !length ) 
                return;
        }

        if ( readingImageDescriptor ) {
            //Same procedure as for global color table
            while ( bufPtr < 9 ) {
                buf[bufPtr++] = *bytes++;
                if ( !--length )
                    return;
            }
            //I'm ignoring x and y offsets for the time being.
            iwid = (buf[5]<<8)+buf[4];
            ihei = (buf[7]<<8)+buf[6];
            hasLocalColorTable = (buf[8]&0x80) > 0;
            bitsPerPixel = buf[8]&0x7;
            interlaced = (buf[8]&0x40) > 0;
            if ( hasLocalColorTable ) {
                //should save global color table here
                readingColorTable = TRUE;
            } 
            else {
                readingImageData = TRUE;
            }
            readingImageDescriptor = FALSE;
            bufPtr = 0;
            if ( !length )
                return;
            else
                continue;
        }

        if ( readingExtension ) {
            //Read extension into buffer
            while ( bufPtr < 2 ) {
                //don't know block size yet
                buf[bufPtr++] = *bytes++;
                if ( ! --length )
                    return;
            }
            while ( bufPtr <= buf[1]+1 ) {
                buf[bufPtr++] = *bytes++;
                if ( ! --length )
                    return;
            }
            //Now decide what to to
            switch ( buf[0] ) {
                case 0xFF: // Application Extension: Ignore
                case 0xFE: // Comment Extension: Ignore
                case 0x01: // Plain Text Extension: Ignore (is that ok?)
                    skippingDataBlocks = TRUE;
                    break;
                case 0xf9: // Graphic Control Extension: Handle
                    readGCE();
                    lookingForNextBlock = TRUE;
                    break;
                default:
                    broken = 6;
            }
            bufPtr = 0;
            readingExtension = FALSE;
            if ( !length )
                return;
            else
                continue;
        }

        if ( readingImageData ) {
            if ( interlaced ) 
                curDecoder = new LZWDecoder(*bytes++, im, mask, transColor, TRUE );
            else 
                curDecoder = new LZWDecoder(*bytes++, im, mask, transColor );
            bytesLeft = 0;
            readingImageData = FALSE;
            readingImageBlock = TRUE;
            canDraw = TRUE;
            if ( ! --length )
                return;
        }

        if ( readingImageBlock ) {
            while ( 1 ) { //read image blocks as long as we have any
                if ( bytesLeft <= 0 ) { //A new block
                    bytesLeft = *(unsigned char*)bytes++;
                    length--;
                    if ( bytesLeft == 0 ) { //end of data blocks
                        if ( ! curDecoder->done )
                            broken = 5;
                        readingImageBlock = FALSE;
                        lookingForNextBlock = TRUE;
                        gotOneIm = TRUE; //forget the rest for now
                        break;
                    }
                    if ( !length )
                        return;
                }
                while ( bytesLeft > 0 ) {
                    bytesLeft --;
                    curDecoder->LZWdecode( *(unsigned char*)bytes++ );
                    if ( ! --length )
                        return;
                }
            }
            continue;
        }

        if ( skippingDataBlocks )
            while ( 1 ) { //skip blocks as long as we have any
                if ( bytesLeft <= 0 ) {
                    bytesLeft = *(unsigned char*)bytes++;
                    length --;
                    if ( bytesLeft == 0 ) {
                        skippingDataBlocks = FALSE;
                        lookingForNextBlock = TRUE;
                        break;
                    }
                }
                while ( bytesLeft > 0 ) {
                    bytesLeft --;
                    bytes++;
                    if ( ! --length )
                        return;
                }
            }

    }

}

//Read Graphic Control Extension in buf.  Dispoal Method might be interesting
//in the future, but for now it is safe to ignore it.
//User Input Flag should always be ignored
//Delay Time will be needed when we support animated GIFs
//IMPORTANT:  According to spec we are to forget all that is read in here
//            when the next rendering block comes, regardless if it has
//            a GCE or not.  This has yet to be done.
void GifDecoder::readGCE( void )
{
    if ( buf[2]&0x1 > 0 )
        transColor = buf[5];
    else
        transColor = -1;
}
    
void GifDecoder::getGifColorTable(unsigned char *table,int numColors ) 
{
    int i;

    im->setNumColors(numColors);
    for ( i = numColors*3-3 ; i >= 0; i -= 3 ) {
        im->setColor(i/3, qRgb( table[i], table[i+1], table[i+2] ) );
    }
}

//Initialize LZW char and index tables
void LZWDecoder::mktab(void) 
{
    int i;
    unsigned char *c = chtab;
    int *p = pttab;

    for (i=0;i<(int)clrcd;i++) {
        *c++ = i;
	*p++ = HASHSIZ+1;
    }
    for (;i<HASHSIZ;i++) { 
        *c++ = 0;    //strictly speaking, this should not be
	*p++ = 0;    //necessary
    }  
}

//setPixels for interlaced image with transparent color
void LZWDecoder::setPixelsit( unsigned char *c, unsigned char *base )
{ 
    while ( c >= base ) {
        if ( nextPix - curlin < width ) {
            pixbuf <<= 1;
            if ( tInd != *c )
                pixbuf |= 1;
            if ( ++inpixbuf > 7 ) {
                *pixpt++ = pixbuf;
                inpixbuf = pixbuf = 0;
            }
            *nextPix++ = *c--;
        }
        else {
	    curScanline += intlIncrement;
	    if ( curScanline < height ) {
                nextPix = curlin = im->scanLine(curScanline);
                *pixpt = pixbuf<<(8-inpixbuf);
                pixpt = tmask->scanLine( curScanline );
                inpixbuf = pixbuf = 0;
            }
            else {
	        if ( intlIncrement > 2 ) {
		    if ( intlIncrement == 4) { //was 4-lines pass
		        intlIncrement = 2;
			curScanline = 1;
	            } else
                        if ( curScanline%8 == 0 )  // was first 8-lines pass
                            curScanline = 4;
                        else {                     // was second 8-lines pass
                            curScanline = 2;
                            intlIncrement = 4;
                        }
		    nextPix = curlin = im->scanLine( curScanline );
		    if ( inpixbuf > 0 )
                        *pixpt = pixbuf<<(8-inpixbuf);
                    pixpt = tmask->scanLine( curScanline );
                    inpixbuf = pixbuf = 0;
		}
		else  
		    intlIncrement = 0; //shouldn't happen
		if ( intlIncrement == 0 || curScanline >= height ) {
                    nextPix = curlin = im->scanLine( height-1 );
		    if ( inpixbuf > 0 )
                        *pixpt = pixbuf<<(8-inpixbuf);
                    pixpt = tmask->scanLine( height-1 );
                    inpixbuf = pixbuf = 0;
                }
            }
	}
    }
}

//setPixels for interlaced image
void LZWDecoder::setPixelsi( unsigned char *c, unsigned char *base )
{ 
    while ( c >= base ) {
        if ( nextPix - curlin < width ) 
            *nextPix++ = *c--;
        else {
	    curScanline += intlIncrement;
	    if ( curScanline < height ) 
                nextPix = curlin = im->scanLine(curScanline);
            else {
	        if ( intlIncrement > 2 ) {
		    if ( intlIncrement == 4) { //was 4-lines pass
		        intlIncrement = 2;
			curScanline = 1;
	            } else
                        if ( curScanline%8 == 0 )  // was first 8-lines pass
                            curScanline = 4;
                        else { // was second 8-lines pass
                            curScanline = 2;
                            intlIncrement = 4;
                        }
		    nextPix = curlin = im->scanLine( curScanline );
		}
		else 
                    nextPix = curlin = im->scanLine( height-1 );
            }
	}
    }
}

//setPixels for non-interlaced image with transparency
void LZWDecoder::setPixelsnit( unsigned char *c, unsigned char *base )
{  
    while ( c >= base ) {
        if ( nextPix - curlin < width ) {
	    pixbuf <<= 1;
	    if ( tInd != *c )
	        pixbuf |= 1;
	    if ( ++inpixbuf > 7 ) {
	        *pixpt++ = pixbuf;
		inpixbuf = pixbuf = 0;
	    }  
            *nextPix++ = *c--;
	}
        else {
	    if ( ++curScanline < height ) {
                nextPix = curlin = im->scanLine( curScanline );
		if ( inpixbuf > 0 )
		    *pixpt = pixbuf<<(8-inpixbuf);
		pixpt = tmask->scanLine( curScanline );
		inpixbuf = pixbuf = 0;
            }
            else { 
                nextPix = curlin = im->scanLine(--curScanline);
		if ( inpixbuf > 0 )
		    *pixpt = pixbuf<<(8-inpixbuf);
		pixpt = tmask->scanLine( curScanline );
		inpixbuf = pixbuf = 0;
	    }
	}
    }
}

void LZWDecoder::setPixelsni( unsigned char *c, unsigned char *base )
{  
    while ( c >= base ) {
        if ( nextPix - curlin < width ) 
            *nextPix++ = *c--;
        else {
	    if ( ++curScanline < height ) 
                nextPix = curlin = im->scanLine(curScanline);
            else // Better safe than sorry...
                nextPix = curlin = im->scanLine(--curScanline);
	}
    }
}

//decompress an LZW code
int LZWDecoder::LZWdecomp ( unsigned int c ) 
{
    unsigned char *op = outstr;
    unsigned int idx;
			   
    //cout << (int)c<< "-" << (int)firstFree << " ";

    if ( c == endcd ) {
        if ( inpixbuf > 0 ) {
            *pixpt = pixbuf<<(8-inpixbuf);
	    pixbuf = inpixbuf = 0;
	}
        done = TRUE;
	return 1;
    }

    if ( c == clrcd ) {  //Clear code -> reset everything
        currentCodeSize = LZWsize+1;
	mask = (1<<currentCodeSize)-1;
	firstFree = endcd + 1;
	maxCode = 1<<currentCodeSize;
	deferredClear = FALSE;
	afterClear = TRUE;
        return 0;
    }

    if ( afterClear ) { //immidiately after a clear just set ocd
        ocd = c;
	*op = c;
//        setPixels( op, outstr );
        switch ( setPixelFunc ) {
            case GIF_SetPixelSNI:
                setPixelsni( op, outstr );
                break;

            case GIF_SetPixelSI:
                setPixelsi( op, outstr );
                break;

            case GIF_SetPixelSNIT:
                setPixelsnit( op, outstr );
                break;

            case GIF_SetPixelSIT:
                setPixelsit( op, outstr );
                break;
        }
	afterClear = FALSE;
	return 0;
    }

    if ( c < firstFree ) {
        idx = c;
	do {
	    *op++ = chtab[idx];
            idx = pttab[idx];
	}  while ( idx < HASHSIZ );
	if ( !deferredClear ) {
	    chtab[firstFree] = *(op-1); 
	    pttab[firstFree] = ocd;
	}
    }
    else {
        idx = ocd;
        op++; //filled in below
        do {
            *op++ = chtab[idx];
            idx = pttab[idx];
        } while ( idx < HASHSIZ );
        if ( !deferredClear ) {
            *outstr = chtab[firstFree] = *(op-1);
            pttab[firstFree] = ocd;
        }
    }
//     setPixels( op-1, outstr );
    switch ( setPixelFunc ) {
        case GIF_SetPixelSNI:
            setPixelsni( op-1, outstr );
            break;
            
        case GIF_SetPixelSI:
            setPixelsi( op-1, outstr );
            break;
            
        case GIF_SetPixelSNIT:
            setPixelsnit( op-1, outstr );
            break;
            
        case GIF_SetPixelSIT:
            setPixelsit( op-1, outstr );
            break;
    }

    ocd = c;
    firstFree++;
    if ( firstFree ==  maxCode ) { 
        if (currentCodeSize < 12 ) { //increase code size by one
            currentCodeSize ++;
            //cout << "!!!!" << currentCodeSize << "!!!!";
            maxCode = 1<<currentCodeSize;
            mask = (1<<currentCodeSize)-1;
        }  //Gif spec allows deffered clear code -- if this happens, freeze
        //all tables
        else {
            deferredClear = TRUE;
            if ( options->debugging() )
                console->warning( "Warning: GIF uses deferred clear" );
//	         cout << "Warning: GIF uses deferred clear";
        }
    }
    return 0;
}

//get a code and try to make something of it
int LZWDecoder::LZWdecode( unsigned char c ) { 
    unsigned int code;

    fifo |= (unsigned long)c<<inFifo;
    inFifo += 8;
    while ( inFifo >= currentCodeSize && !done ) {
        code = fifo & mask;
        fifo >>= currentCodeSize;
        inFifo -= currentCodeSize;
        if ( LZWdecomp( code ) == -1 && options->debugging() )
            console->warning( "GifDecoder: Panic!" );
//	    cout << "GifDecoder: Panic!\n";
    }
    return 0;
}

//transparencyMask == NULL means image has no transparent color
LZWDecoder::LZWDecoder( unsigned char size, QImage *image, 
                        QImage *transparencyMask, int transColor, bool intlFlag )
{
    done = FALSE;
    im = image;
    tmask = transparencyMask;
    tInd = transColor;
    LZWsize = size;
    width = im->width();
    height = im->height();
    pttab = new int[HASHSIZ];
    chtab = new unsigned char[HASHSIZ];
    outstr = new unsigned char[HASHSIZ+1];
    currentCodeSize = LZWsize+1;
    mask = (1<<currentCodeSize)-1;
    clrcd = 1<<LZWsize;
    endcd = clrcd+1;
    curScanline = 0;
    curlin = im->scanLine( curScanline );
    nextPix = curlin;
    fifo = 0;
    inFifo = 0;
    pixbuf = 0;
    inpixbuf = 0;
    if ( tmask )
        pixpt = tmask->scanLine (curScanline );
    else
        pixpt = 0;
    mktab();
    if ( intlFlag ) {
        if ( tmask ) {
//            setPixels = setPixelsit;
            setPixelFunc = GIF_SetPixelSIT;
        } else {
//            setPixels = setPixelsi;
            setPixelFunc = GIF_SetPixelSI;
        }
	intlIncrement = 8;
    } else {
        if ( tmask ) {
//	    setPixels = setPixelsnit;
            setPixelFunc = GIF_SetPixelSNIT;
	} else {
//            setPixels = setPixelsni;
            setPixelFunc = GIF_SetPixelSNI;
        }
    }
    LZWdecomp(clrcd); 
}


LZWDecoder::~LZWDecoder( void )
{
    delete pttab;
    delete chtab;
    delete outstr;
}

void GifDecoder::renderProgressive()
{
    _pixmap.convertFromImage( *im, QPixmap::Color );
    if ( drawnToLine <= 0 ) { //first redraw
        _renderer->resize( width, height );
    }
    if ( gotOneIm )
        drawnToLine = height;
    else
        drawnToLine = curDecoder->getCurScanline()-1;
    if ( mask ) {
        QBitmap b;
        _pixmap.setMask( b = *mask );
    }

    _renderer->needRedraw();
}


void GifDecoder::endOfData()
{
    if ( canDraw ) {
        renderProgressive();
        delete curDecoder;
    }
}
