# =============================================================================
#
# File:		action.tcl
# Project:	TkDesk
#
# Started:	14.10.94
# Changed:	17.10.94
# Author:	cb
#
# Description:	Implements procs for opening & executing files and for popups.
#
# Copyright (C) 1996  Christian Bolik
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
# See the file "COPYING" in the base directory of this distribution
# for more.
#
# -----------------------------------------------------------------------------
#
# Sections:
#	proc dsk_open {viewer file}
#	proc dsk_openall {{files ""}}
#	proc dsk_exec {args}
#	proc dsk_ask_exec {}
#	proc dsk_ask_dir {}
#	proc dsk_cd {dir}
#	proc dsk_open_dir {dir}
#	proc dsk_popup {lbox file mx my}
#	proc dsk_build_popup {poplist file lbox mx my}
#	proc _expand_pc {cmd {file ""}}
#	itcl_class dsk_Periodic
#	proc dsk_periodic {}
#	proc dsk_jobs {}
#	proc dsk_jobs_fill {}
#	proc dsk_jobs_sig {signal}
#       proc dsk_edit {args}
#       proc dsk_view {args}
#
# =============================================================================

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_open
# Args:		viewer		name of calling file viewer window
#		file		name of file
# Returns: 	
# Desc:		Performs the default action for $file. Gets called after
#		a double click in a file listbox. Or from the Bookmarks
#               menu.
# Side-FX:	none
#

set dsk_open(lastcmd) ""
set tkdesk(dont_add_cmd_to_history) 0

proc dsk_open {viewer file} {
    global tkdesk cmd dsk_open env

    if ![winfo exists $viewer] {
	set viewer .fv0
    }
    
    if ![file exists $file] {
	$viewer config -dir [$viewer curdir]
	return
    }

    if [file isdirectory $file] {
	# DIRECTORY

	set file [subst -nocommands -novariables \
		[_make_fname_safe $file]]
	set dname [file tail $file]
	if {$dname == "."} {
	    set file [file dirname $file]
	} elseif {$dname == ".."} {
	    set file [file dirname $file]
	    set file [file dirname $file]
	}

	if $tkdesk(file_lb,control) {
	    dsk_FileList .dfl[dsk_FileList :: id] -dir "$file"
	    set tkdesk(file_lb,control) 0
	} else {
	    $viewer config -dir "$file"
	}

    } elseif [file_executable $file] {
	# EXECUTABLE

	if !$tkdesk(file_lb,control) {
	    dsk_busy
	    dsk_exec $file
	    dsk_lazy
	} else {
	    set cmd [file tail $file]
	    cb_readString "Enter command:" cmd "Execute"
	    if {$cmd != ""} {
		set owd [pwd]
		cd [file dirname $file]
		eval dsk_exec $cmd
		cd $owd
	    }
	    set tkdesk(file_lb,control) 0
	}

    } else {
	# REGULAR FILE

	if $tkdesk(file_lb,control) {
	    set ft [file tail $file]
	    if {[llength $ft] == 1} {
		set cmd "$dsk_open(lastcmd) $ft"
	    } else {
		set cmd "$dsk_open(lastcmd) \"$ft\""
	    }
	    cb_readString "Enter command:" cmd "Execute"
	    if {$cmd != ""} {
		set dsk_open(lastcmd) [string range $cmd 0 \
			[expr [string last " " $cmd] - 1]]
		set owd [pwd]
		cd [file dirname $file]
		eval dsk_exec $cmd
		cd $owd
	    }
	    set tkdesk(file_lb,control) 0
	} else {
	    # determine default action for file:
	    if ![info exists tkdesk(popup,regulars)] return

	    dsk_busy
	    set action_found 0
	    set fname [file tail $file]
	    foreach entry $tkdesk(popup,regulars) {
		set patlist [lindex $entry 0]

		foreach pat $patlist {
		    if [string match $pat $fname] {
			set elist [lindex $entry 1]
			set action [lindex [lindex $elist 0] 1]
			set action_found 1
			break
		    }
		}

		if $action_found  break
	    }

	    if {$action != "" && $action != "-"} {
		# add file to file history
		dsk_history_file $file
		
		set owd [pwd]
		cd [file dirname $file]
		eval [_expand_pc $action $file]
		cd $owd
	    }
	    dsk_lazy
	}

    }
}

proc dsk_history_file {file} {
    global tkdesk env
    
    set tkdesk(dont_add_cmd_to_history) 1
    if {[string first $env(HOME) $file] == 0} {
	file_history add [string_replace $file $env(HOME) ~]
    } else {
	file_history add $file
    }
    set tkdesk(dont_add_cmd_to_history) 0
}

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_openall
# Args:		files		(opt.) list of files
# Returns: 	""
# Desc:		Opens all selected files (or $files). "Open" means it
#		performs the default action for each file.
# Side-FX:	
#

proc dsk_openall {args} {
    global tkdesk

    set files $args
    if {$files == ""} {
    	set files [$tkdesk(active_viewer) select get]
    }

    if {$files == ""} {
	cb_info "Please select one or more files first."
    }

    dsk_busy
    foreach file $files {
	if ![file isdirectory $file] {
	    dsk_open $tkdesk(active_viewer) $file
	} else {
	    dsk_open_dir $file
	}
    }
    dsk_lazy

    return
}

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_exec
# Args:		args		command line to execute
# Returns: 	pid or ""
# Desc:		Executes $cmd in the background.
# Side-FX:	none
#

set tkdesk(dsk_exec,pids) ""
set tkdesk(dsk_exec,cmds) ""
set dsk_exec(bgcnt) 0

proc dsk_exec {args} {
    global tkdesk env dsk_exec

    if {$args == ""} {
	return
    }

    dsk_busy
    set cmd $args
    if !$tkdesk(dont_add_cmd_to_history) {
	if {[string first $env(HOME) $cmd] == 0} {
	    exec_history add [string_replace $cmd $env(HOME) ~]
	} else {
	    exec_history add $cmd
	}
    }
    set cnt [incr dsk_exec(bgcnt)]
    dsk_sound dsk_exec_launch
    if [file_executable $cmd] {
	# then dsk_exec is called from dsk_open

    	set oldwd [pwd]
    	cd [file dirname $cmd]
    	set err [catch {set pid \
		[blt_bgexec dsk_exec(bgvar,$cnt) $cmd </dev/null &]} errmsg]
    	cd $oldwd
	if $err {
	    dsk_lazy
	    dsk_errbell
	    cb_error $errmsg
	    return 0
	}
    } else {
	set shell_args "</dev/null"
	set cmd ""

	foreach token $args {
	    if [string match {<*} $token] {
		lappend shell_args $token
	    } elseif [string match {>*} $token] {
		lappend shell_args $token
	    } elseif [string match {2>*} $token] {
		lappend shell_args $token
	    } elseif [string match {&} $token] {
		continue
	    } else {
		lappend cmd $token
	    }
	}

	set cmd [string_replace $cmd \{ \"]
	set cmd [string_replace $cmd \} \"]
	dsk_debug "Executing: sh -c \"exec $cmd\" $shell_args &"
	set pid [eval blt_bgexec dsk_exec(bgvar,$cnt) \
		sh -c [list "exec $cmd"] $shell_args &]
	#set pid [exec sh -c \"exec $cmd\" $shell_args &]
    }
    set dsk_exec(bgcmd,$cnt) $cmd
    trace variable dsk_exec(bgvar,$cnt) w dsk_exec_trace
    dsk_status "Launched:  $cmd"
    dsk_lazy

    lappend tkdesk(dsk_exec,pids) $pid
    if {[llength $args] > 1} {
    	lappend tkdesk(dsk_exec,cmds) "$args"
    } else {
    	lappend tkdesk(dsk_exec,cmds) $args
    }

    if [winfo exists .dsk_jobs] {
	dsk_jobs_fill
    }

    return $pid
}

# this proc will be invoked when a dsk_exec'ed command exits:
proc dsk_exec_trace {arr idx op} {
    global dsk_exec

    set num [lindex [split $idx ,] 1]
    set cmd $dsk_exec(bgcmd,$num)
    unset dsk_exec(bgvar,$num)
    unset dsk_exec(bgcmd,$num)

    dsk_sound dsk_exec_exit
    #dsk_status "[lindex $cmd 0] has exited."
    dsk_status "Exit:  $cmd"
}

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_ask_exec
# Args:		none
# Returns: 	pid
# Desc:		Asks the user which command to execute.
# Side-FX:	none
#

if ![info exists tkdesk(geometry,dsk_ask_exec)] {
    set tkdesk(geometry,dsk_ask_exec) ""
}
if ![info exists tkdesk(cmd_history)] {
    set tkdesk(cmd_history) ""
}

proc dsk_ask_exec {{cmd ""}} {
    global tkdesk

    set t .dsk_ask_exec
    if [winfo exists $t] {
	cb_raise $t
	return
    }

    toplevel $t

    frame $t.f -bd 1 -relief raised
    pack $t.f -fill both -expand yes

    frame $t.fl
    pack $t.fl -in $t.f -fill x

    label $t.label -text "Command to execute:"
    pack $t.label -in $t.fl -side left -padx $tkdesk(pad) -pady $tkdesk(pad)

    frame $t.fe
    pack $t.fe -in $t.f -fill both -expand yes

    entry $t.entry -width 40 -bd 2 -relief sunken
    pack $t.entry -in $t.fe -fill x -expand yes -side left \
		-padx $tkdesk(pad) -pady $tkdesk(pad) -ipady 2
    bind $t.entry <Return> "$t.bOK.button flash ; $t.bOK.button invoke"
    bind $t.entry <Escape> "$t.bCancel.button flash ; $t.bCancel.button invoke"
    bind $t.entry <Control-c> "$t.bCancel.button flash
				$t.bCancel.button invoke"
    cb_bindForCompletion $t.entry <Control-Tab>
    if {$cmd != ""} {
	$t.entry insert end $cmd
    }

    blt_drag&drop target $t.entry handler text "dd_handle_text $t.entry"

    menubutton $t.mbHist -bd 2 -relief raised \
		-bitmap @$tkdesk(library)/cb_tools/bitmaps/combo.xbm \
		-menu $t.mbHist.menu
    pack $t.mbHist -in $t.fe -side right \
		-padx $tkdesk(pad) -pady $tkdesk(pad) -ipadx 2 -ipady 2

    menu $t.mbHist.menu \
		-postcommand "cmd_history buildmenu $t.mbHist.menu"
    # add dummy entry to work around bug in pre Tk 4.0p2:
    $t.mbHist.menu add command -label "dummy"

    frame $t.fb -bd 1 -relief raised
    pack $t.fb -fill x

    cb_button $t.bOK -text "   OK   " -default 1 -command {
		set tmpcmd [.dsk_ask_exec.entry get]
		#set tkdesk(geometry,dsk_ask_exec) [wm geometry .dsk_ask_exec]
		#destroy .dsk_ask_exec
		if {$tmpcmd == ""} {
		    unset tmpcmd
		    return
		}
		.dsk_ask_exec.bCancel.button invoke
		cmd_history add $tmpcmd
		set tmpowd [pwd]
		cd [$tkdesk(active_viewer) curdir]
		if [auto_execok [lindex $tmpcmd 0]] {
		    eval dsk_exec $tmpcmd
		} elseif [file isdirectory $tmpcmd] {
		    dsk_open_dir $tmpcmd
		} else {
		    dsk_errbell
		    cb_error "Could not find or execute:\n$tmpcmd"
		}
		cd $tmpowd
		unset tmpcmd tmpowd
	}
    cb_button $t.bCancel -text "  Close  " -command {
		set tkdesk(geometry,dsk_ask_exec) [wm geometry .dsk_ask_exec]
		destroy .dsk_ask_exec }

    pack $t.bOK $t.bCancel -in $t.fb -side left \
		-padx $tkdesk(pad) -pady $tkdesk(pad)

    bind $t <Any-Enter> "focus $t.entry"

    wm minsize $t 326 106
    wm title $t "Execute"
    wm protocol $t WM_DELETE_WINDOW {
		set tkdesk(geometry,dsk_ask_exec) [wm geometry .dsk_ask_exec]
		destroy .dsk_ask_exec }

    if {$tkdesk(geometry,dsk_ask_exec) == ""} {
	cb_centerToplevel $t
    } else {
	wm geometry $t $tkdesk(geometry,dsk_ask_exec)
    }

    grab $t
    set old_focus [focus]
    focus $t.entry
    tkwait window $t
    focus -force $old_focus
}

proc dsk_ask_exec_cb {t cmd} {
    $t.entry delete 0 end
    $t.entry insert end $cmd
}

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_ask_dir
# Args:		none
# Returns: 	""
# Desc:		Asks the user which directory to open.
# Side-FX:	none
#

if ![info exists tkdesk(geometry,dsk_ask_dir)] {
    set tkdesk(geometry,dsk_ask_dir) ""
}

proc dsk_ask_dir {{browser ""}} {
    global tkdesk dsk_ask_dir

    set t .dsk_ask_dir
    if [winfo exists $t] {
	cb_raise $t
	return
    }

    toplevel $t

    frame $t.f -bd 1 -relief raised
    pack $t.f -fill both -expand yes

    frame $t.fl
    pack $t.fl -in $t.f -fill x

    label $t.label -text "Directory to open:"
    pack $t.label -in $t.fl -side left -padx $tkdesk(pad) -pady $tkdesk(pad)

    set dsk_ask_dir(browser) [string match "browser" $browser]
    checkbutton $t.cbBrowser -text "In Browser" -variable dsk_ask_dir(browser)
    pack $t.cbBrowser -in $t.fl -side right \
	    -padx $tkdesk(pad) -pady $tkdesk(pad)

    frame $t.fe
    pack $t.fe -in $t.f -fill both -expand yes

    entry $t.entry -width 40 -bd 2 -relief sunken
    pack $t.entry -in $t.fe -fill x -expand yes -side left \
		-padx $tkdesk(pad) -pady $tkdesk(pad) -ipady 2
    bind $t.entry <Return> "$t.bOK.button flash ; $t.bOK.button invoke"
    bind $t.entry <Escape> "$t.bCancel.button flash ; $t.bCancel.button invoke"
    bind $t.entry <Control-c> "$t.bCancel.button flash
				$t.bCancel.button invoke"
    cb_bindForCompletion $t.entry <Control-Tab>

    blt_drag&drop target $t.entry handler text "dd_handle_text $t.entry"

    menubutton $t.mbHist -bd 2 -relief raised \
		-bitmap @$tkdesk(library)/cb_tools/bitmaps/combo.xbm \
		-menu $t.mbHist.menu
    pack $t.mbHist -in $t.fe -side right \
		-padx $tkdesk(pad) -pady $tkdesk(pad) -ipadx 2 -ipady 2

    menu $t.mbHist.menu -postcommand "dsk_ask_dir_hmenu"
    # add dummy entry to work around bug in pre Tk 4.0p2:
    $t.mbHist.menu add command -label "dummy"

    frame $t.fb -bd 1 -relief raised
    pack $t.fb -fill x

    cb_button $t.bOK -text "   OK   " -default 1 -command \
	    "dsk_ask_dir_ok"
    cb_button $t.bCancel -text "  Close  " -command {
		set tkdesk(geometry,dsk_ask_dir) [wm geometry .dsk_ask_dir]
		destroy .dsk_ask_dir
    }

    pack $t.bOK $t.bCancel -in $t.fb -side left \
	    -padx $tkdesk(pad) -pady $tkdesk(pad)

    bind $t <Any-Enter> "focus $t.entry"

    wm minsize $t 326 117
    wm title $t "Open Directory"
    wm protocol $t WM_DELETE_WINDOW {.dsk_ask_dir.bCancel.button invoke}

    if {$tkdesk(geometry,dsk_ask_dir) == ""} {
	cb_centerToplevel $t
    } else {
	wm geometry $t $tkdesk(geometry,dsk_ask_dir)
    }

    grab $t
    set old_focus [focus]
    focus $t.entry
    tkwait window $t
    focus -force $old_focus
}

proc dsk_ask_dir_ok {} {
    global tkdesk dsk_ask_dir
    
    set tmpdir [.dsk_ask_dir.entry get]
    .dsk_ask_dir.bCancel.button invoke
    update idletasks
    
    if {$tmpdir != ""} {
	dsk_busy
	if $dsk_ask_dir(browser) {
	    dsk_FileViewer .fv[dsk_FileViewer :: id] \
		    -dir $tmpdir -num_lbs $tkdesk(num_lbs)
	} else {
	    dsk_FileList .dfl[dsk_FileList :: id] -dir $tmpdir
	}
	dsk_lazy
    }
}

proc dsk_ask_dir_hmenu {} {
    global tkdesk

    set t .dsk_ask_dir
    catch "$t.mbHist.menu delete 0 last"
    if $tkdesk(sort_history) {
	set l [lsort [history get]]
    } else {
	set l [history get]
    }
    foreach dir $l {
	$t.mbHist.menu add command -label $dir \
			-command "$t.entry delete 0 end
				$t.entry insert end $dir" \
			-font $tkdesk(font,entries)
    }
}

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_cd
# Args:		dir	name of directory
# Returns: 	""
# Desc:		Display directory $dir in the active file viewer.
# Side-FX:	none
#

proc dsk_cd {dir} {
    global tkdesk

    if ![file isdirectory $dir] {
	set dir [file dirname $dir]
    }

    if {$dir != ""} {
    	$tkdesk(active_viewer) config -directory $dir
    }
}

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_open_dir
# Args:		dir		directory to open
# Returns: 	""
# Desc:		Opens a window for directory $dir
# Side-FX:	none
#

proc dsk_open_dir {dir} {

    dsk_FileList .dfl[dsk_FileList :: id] -directory $dir
}

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_popup
# Args:		lbox		name of calling listbox
#		file		name of file
#		mx		mouse x position on root window
#		my		mouse y position on root window
#               opt             additional options
# Returns: 	
# Desc:		Shows the appropriate popup menu for $file.
# Side-FX:	none
#

proc dsk_popup {lbox file mx my {opt ""}} {
    global tkdesk

    if $tkdesk(append_type_char) {
	set file [dsk_striptc $file]
    }
    #dsk_debug "dsk_popup: $file, $viewer, x: $mx, y: $my"

    if [file isdirectory $file] {
	# DIRECTORY
	dsk_build_popup tkdesk(popup,directories) $file $lbox $mx $my $opt
    } elseif [file_executable $file] {
	# EXECUTABLE
	dsk_build_popup tkdesk(popup,executables) $file $lbox $mx $my $opt
    } else {
	# REGULAR (and others)
	dsk_build_popup tkdesk(popup,regulars) $file $lbox $mx $my $opt
    }
}

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_build_popup
# Args:		poplist - list of popup entries
#		file - path and name of file
#		lbox - name of listbox where mousebutton was pressed
#		mx		mouse x position on root window
#		my		mouse y position on root window
#               opt             additional options
# Returns: 	""
# Desc:		builds the appropriate popup menu and brings it on-screen
# Side-FX:	none
#

proc dsk_build_popup {poplist file lbox mx my {opt ""}} {
    global tkdesk

    if ![info exists $poplist] {
	dsk_debug "$poplist does not exist!"
    }

    set fname [file tail $file]
    set dirname [file dirname $file]
    set sfile [_make_fname_safe $file]

    catch "destroy .dsk_popup"
    foreach entry [set $poplist] {
	#
	# the first element of each entry is a list of glob patterns
	#
	set patlist [lindex $entry 0]
	foreach pat $patlist {
	    if [string match $pat $fname] {
		menu .dsk_popup
		if [file isdirectory $file] {
		    set ffnt $tkdesk(font,directories)
		    set ffg $tkdesk(color,directories)
		} elseif [file_executable $file] {
		    set ffnt $tkdesk(font,executables)
		    set ffg $tkdesk(color,executables)
		} else {
		    set ffnt $tkdesk(font,file_lbs)
		    set ffg black
		}
		#.dsk_popup add command -label "$fname " \
			#	 -command "catch {destroy .dsk_popup} ;\
			#	     dsk_fileinfo [_make_fname_safe $file]" \
			#	 -font $ffnt -foreground $ffg \
			#        -activeforeground $ffg
		.dsk_popup add cascade -label "$fname " -menu .dsk_popup.fm \
			-font $ffnt -foreground $ffg -activeforeground $ffg
		menu [set m .dsk_popup.fm]
		$m add command -label "Info " -command \
			"catch {destroy .dsk_popup} ;\
			dsk_fileinfo \"$sfile\""
		$m add command -label "Open with... " -command \
			"catch {destroy .dsk_popup} ;\
			set tkdesk(file_lb,control) 1 ;\
			dsk_open $tkdesk(active_viewer) \"$sfile\""
		$m add separator
		$m add command -label "Copy, Move, ... " -command \
			"catch {destroy .dsk_popup} ;\
			dsk_copy \"$sfile\""
		$m add command -label "Rename... " -command \
			"catch {destroy .dsk_popup} ;\
			dsk_rename \"$sfile\""
		$m add command -label "Delete " -command \
			"catch {destroy .dsk_popup} ;\
			dsk_delete \"$sfile\""

		
		if {$opt != ""} {
		    switch $opt {
			"dir" {
			    .dsk_popup add command -label "Open Directory " \
				    -command "catch {destroy .dsk_popup}
			          dsk_open_dir [file dirname $file]"
			}
		    }
		}

		if {$poplist == "tkdesk(popup,directories)"} {
		    .dsk_popup add cascade -label "Traverse" \
			    -menu [set m .dsk_popup.mc]
		    menu $m -postcommand "dsk_casdirs $file $m 1"
		    $m add command -label "dummy"
		}

		.dsk_popup add separator

		set num_entries 2
		set max_chars [expr [string length $fname] + 2]
		set menu_entries [lindex $entry 1]
		foreach me $menu_entries {
		    if {$poplist == "tkdesk(popup,executables)"} {
			#
			# Skip the "Edit" entry if file is not a script:
			#
			if [string match "Edit" [lindex $me 0]] {
			    set fd [open $file]
			    set sig [read $fd 2]
			    close $fd
			    if {$sig != "#!"} {
				continue
			    }
			}
		    }

		    if {$me != "-"} {
			set l [lindex $me 0]
			set ll [string length $l]
			if {$ll > $max_chars} {
			    set max_chars $ll
			}
			.dsk_popup add command -label $l \
				-command "catch {destroy .dsk_popup}
			dsk_history_file \"$sfile\"
			[_expand_pc [lindex $me 1] $file]"
		    } else {
			.dsk_popup add separator
		    }
		    incr num_entries
		}

		break
	    }
	}
	if [winfo exists .dsk_popup] {
	    break
	}
    }

    if [winfo exists .dsk_popup] {
	set owd [pwd]
	cd $dirname
	#
	# Post the popup
	#
	#cb_MenuPopupAdd $lbox 3 .dsk_popup "" "" 1 $mx $my
	update
	tk_popup .dsk_popup $mx $my 
	tkwait window .dsk_popup
	update idletasks
	cd $owd
    }

    return ""
}

#
# -----------------------------------------------------------------------------
#
# Proc:		_expand_pc
# Args:		cmd	command line with %?
#		file	(opt.) name of selected file (with path)
# Returns: 	cmd with %? expanded
# Desc:		Expands all %? shortcuts in the given command line.
#		Does also some preprocessing of the command.
# Side-FX:	none
#

proc _expand_pc {cmd {file ""}} {
    global tkdesk

    dsk_debug -nonewline "Expanding \"$cmd\" to "

    if {$file == ""} {
	set files [_make_fnames_safe]
	set file [lindex $files 0]
    } else {
	set file [_make_fname_safe $file]
    }

    if {$file == ""} {
	if {[string first "%A" $cmd] > -1} {
	    cb_info "Please select one or more files first."
	    return ""
	} elseif {[string first "%" $cmd] > -1} {
	    if {[string first "%D" $cmd] == -1} {
	    	cb_info "Please select a file first."
	    	return ""
	    }
	}
    }

    # (I disabled the following cause it crashes with auto-loading.)
    # Enabled it again because TkDesk no longer uses auto-loading.
    if {[info commands [lindex $cmd 0]] == ""} {
	if {[info procs [lindex $cmd 0]] == ""} {
	    # use "dsk_exec" as the default command
	    set cmd "dsk_exec $cmd"
	}
    }

    set dir [string trimright [$tkdesk(active_viewer) curdir] /]
    set ocmd $cmd
    set pcmd ""
    foreach cmd [split $ocmd \n] {
	#set l [llength $cmd]
	#for {set i 0} {$i < $l} {incr i} {
	#    set e [lindex $cmd $i]
	#    switch -glob -- $e {
	#	 %s	{set cmd [lreplace $cmd $i $i $file]}
	#	 %d	{set cmd [lreplace $cmd $i $i [file dirname $file]]}
	#	 %f	{set cmd [lreplace $cmd $i $i [file tail $file]]}
	#	 %b	{set cmd [lreplace $cmd $i $i [string range $file 0 \
	#		 [expr [string last "." $file] - 1 ]]]}
	#	 %A	{set cmd [eval lreplace [list $cmd] $i $i \
	#		 [split $files]]}
	#	 %D	{set cmd [lreplace $cmd $i $i $dir]}
	#	 *%s*	{
	#	     set cmd [lreplace $cmd $i $i \
	#		     [string_replace $e %s $file]]
	#	 }
	#	 *%d*	{
	#	     set cmd [lreplace $cmd $i $i \
	#		     [string_replace $e %d [file dirname $file]]]
	#	 }
	#	 *%f*	{
	#	     set cmd [lreplace $cmd $i $i \
	#		     [string_replace $e %f [file tail $file]]]
	#	 }
	#	 *%b*	        {
	#	     set cmd [lreplace $cmd $i $i \
	#		     [string_replace $e %b [string range $file 0 \
	#		     [expr [string last "." $file] - 1 ]]]]
	#	 }
	#	 *%A*	{
	#	     set cmd [lreplace $cmd $i $i \
	#		     [string_replace $e %A [split $files]]]
	#	 }
	#	 *%D*	{
	#	     set cmd [lreplace $cmd $i $i \
	#		     [string_replace $e %D $dir]]
	#	 }
	#    }
	#}
	if {[string first "%s" $cmd] > -1} {
	    set cmd [string_replace $cmd %s \{$file\}]
	}
	if {[string first "%d" $cmd] > -1} {
	    set cmd [string_replace $cmd %d \{[file dirname $file]\}]
	}
	if {[string first "%f" $cmd] > -1} {
	    set cmd [string_replace $cmd %f \{[file tail $file]\}]
	}
	if {[string first "%b" $cmd] > -1} {
	    set cmd [string_replace $cmd %b \{[string range $file 0 \
		    [expr [string last "." $file] - 1 ]]\}]
	}
	if {[string first "%A" $cmd] > -1} {
	    set cmd [string_replace $cmd %A [split $files]]
	}
	if {[string first "%D" $cmd] > -1} {
	    set cmd [string_replace $cmd %D \{$dir\}]
	}
	append pcmd "$cmd\n"
    }

    dsk_debug "\"$pcmd\""
    return [subst -nocommands -novariables $pcmd]
}

#
# -----------------------------------------------------------------------------
#
# Proc:		_make_fnames_safe
# Args:		none
# Returns: 	filenames with all special chars ([ etc.) backslashed
# Desc:		...
# Side-FX:	none
#

proc _make_fnames_safe {} {
    global tkdesk

    set flist [$tkdesk(active_viewer) select get]
    if {$flist == ""} {return ""}

    foreach file $flist {
	#if {[string first "\\" $file] > -1} continue

	set nfile [string_replace $file \[ \\\[]
	set nfile [string_replace $nfile \] \\\]]
	set nfile [string_replace $nfile \{ \\\{]
	set nfile [string_replace $nfile \} \\\}]
	set nfile [string_replace $nfile \$ \\\$]
	set nfile [string_replace $nfile \" \\\"]
	set nfile [string_replace $nfile " " "\\\ "]
	if {$nfile != ""} {
	    lappend rlist $nfile
	}
    }

    #puts $rlist
    if {[llength $rlist] == 1} {
	return [lindex $rlist 0]
    } else {
	return $rlist
    }
}

# -----------------------------------------------------------------------------
#
# Proc:		_make_fname_safe
# Args:		file - filename
# Returns: 	filename with all special chars ([ etc.) backslashed
# Desc:		...
# Side-FX:	none
#

proc _make_fname_safe {{file ""}} {

    if {$file == ""} {return ""}

    set nfile [string_replace $file \[ \\\[]
    set nfile [string_replace $nfile \] \\\]]
    set nfile [string_replace $nfile \{ \\\{]
    set nfile [string_replace $nfile \} \\\}]
    set nfile [string_replace $nfile \$ \\\$]
    set nfile [string_replace $nfile \" \\\"]
    set nfile [string_replace $nfile " " "\\\ "]

    #puts $nfile
    return $nfile
}



#
# =============================================================================
#
# Class:	dsk_Periodic
# Desc:		Implements a window that periodically executes a shell
#		command and collects its output in a text widget.
#
# Methods:	
# Procs:	
# Publics:
#

itcl_class dsk_Periodic {

    constructor {args} {
	global tkdesk

	#
	# Create a toplevel with this object's name
	# (later accessible as $this-top):
	#
        set class [$this info class]
        ::rename $this $this-tmp-
        ::toplevel $this -class $class
	wm withdraw $this
        ::rename $this $this-top
        ::rename $this-tmp- $this

	frame $this.fe -bd 1 -relief raised
	pack $this.fe -fill x

	label $this.lCmd -text "Command:"
	entry $this.eCmd -bd 2 -relief sunken -width 10
	bind $this.eCmd <Return> "$this.bExec.button flash
				$this.bExec.button invoke"
	bind $this.eCmd <Tab> "focus $this.eSec"
    	menubutton $this.mbHist -bd 2 -relief raised \
		-bitmap @$tkdesk(library)/cb_tools/bitmaps/combo.xbm \
		-menu $this.mbHist.menu
    	menu $this.mbHist.menu \
		-postcommand "pcmd_history buildmenu $this.mbHist.menu"
	# add dummy entry to work around bug in pre Tk 4.0p2:
	$this.mbHist.menu add command -label "dummy"

	frame $this.fSep -width 16
	label $this.lu1 -text "Exec every"
	entry $this.eSec -bd 2 -relief sunken -width 4
	bind $this.eSec <Return> "$this.bExec.button flash
				$this.bExec.button invoke"
	bind $this.eSec <Tab> "focus $this.eCmd"
	$this.eSec insert end $period
	label $this.lu2 -text "seconds"

	pack $this.lCmd $this.eCmd $this.mbHist $this.fSep $this.lu1 \
		$this.eSec $this.lu2 \
		-in $this.fe -side left -ipady 2 \
		-padx $tkdesk(pad) -pady $tkdesk(pad)
	pack configure $this.eCmd -fill x -expand yes
	pack configure $this.mbHist  -ipadx 2 -ipady 2

	frame $this.ft -bd 1 -relief raised
	pack $this.ft -fill both -expand yes

	cb_text $this.ftext -vscroll 1 -lborder 1 -wrap none \
		-pad $tkdesk(pad) -width 20 -height 2 -state disabled
	pack $this.ftext -in $this.ft -fill both -expand yes \
		-pady $tkdesk(pad)

	frame $this.fb -bd 1 -relief raised
	pack $this.fb -fill x

	cb_button $this.bExec -text "  Exec  " -default 1 \
		-command "$this config -period \[$this.eSec get\]
				$this config -command \[$this.eCmd get\]
				focus $this
				pcmd_history add \[$this.eCmd get\]"
	cb_button $this.bClose -text "  Close  " -command "$this delete"
	pack $this.bExec $this.bClose -in $this.fb -side left \
		-padx $tkdesk(pad) -pady $tkdesk(pad)

	bind $this <Any-Enter> "+focus $this"
	bind $this <Tab> "focus $this.eCmd"

    	wm minsize $this 354 124
	wm geometry $this 592x278
	wm title $this "Periodic Execution"
	wm protocol $this WM_DELETE_WINDOW "$this delete"

	eval config $args
	wm deiconify $this
    }

    destructor {
        #after 10 rename $this-top {}		;# delete this name
        catch {destroy $this}		;# destroy associated window
    }

    #
    # ----- Methods and Procs -------------------------------------------------
    #

    method config {config} {
    }

    method execute {} {
	global [set this]
	
	if {[grab current] == ""} {
	    dsk_busy dsk_Periodic
	    set err [catch {eval blt_bgexec [set this](var) \
		    -output [set this](result) \
		    -error [set this](error) \
		    $command </dev/null &} m]
	    if $err {
		dsk_lazy dsk_Periodic
	    	set update_started 0
		dsk_errbell
		if {[string length $m] < 100} {
	    	    cb_error "Couldn't execute $command! ($m)"
		} else {
	    	    cb_error \
			"Couldn't execute $command for some strange reason..."
		}
	    	return
	    }

	    tkwait variable [set this](var)
	    set result [set [set this](result)]
	    set errout [set [set this](error)]
	    
	    set oldy [lindex [cb_old_sb_get $this.ftext.vscroll] 2]
	    $this.ftext.text config -state normal
	    $this.ftext.text delete 1.0 end
	    if {$errout == ""} {
		$this.ftext.text insert end $result
	    } else {
		$this.ftext.text insert end $errout
	    }
	    $this.ftext.text yview $oldy
	    $this.ftext.text config -state disabled
	    dsk_lazy dsk_Periodic
	}
    }

    method update {} {
	$this execute

	if $period {
	    set update_started 1
	    after [expr $period * 1000] "catch \"$this update\""
	} else {
	    set update_started 0
	}
    }

    proc id {} {
	set i $id
	incr id
	return $i
    }

    #
    # ----- Variables ---------------------------------------------------------
    #

    public command "" {
	if {$command != ""} {
	    if !$update_started {
	    	$this update
	    } else {
	    	$this execute
	    }
	}
	$this.eCmd delete 0 end
	$this.eCmd insert end $command
    }

    public period 10 {
	$this.eSec delete 0 end
	$this.eSec insert end $period
    }

    protected update_started 0

    common id 0
}

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_periodic (~_cb is the callback for the history menu)
# Args:		none
# Returns: 	""
# Desc:		Creates a window for periodic execution of shell commands.
# Side-FX:	none (hopefully)
#

proc dsk_periodic {{cmd ""} {period 10}} {

    if {$cmd != ""} {
    	dsk_Periodic .pe[dsk_Periodic :: id] -command $cmd -period $period
    } else {
    	dsk_Periodic .pe[dsk_Periodic :: id] -period $period
    }
    return
}

proc dsk_periodic_cb {t cmd} {
    $t.eCmd delete 0 end
    $t.eCmd insert end $cmd
}

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_jobs
# Args:		none
# Returns: 	""
# Desc:		Displays a listing of all started background processes
#		that were started by TkDesk.
# Side-FX:	none
#

set dsk_jobs(pids) ""
if ![info exists tkdesk(geometry,dsk_jobs)] {
    set tkdesk(geometry,dsk_jobs) ""
}

proc dsk_jobs {} {
    global tkdesk dsk_jobs

    set t .dsk_jobs
    if [winfo exists $t] {
	cb_raise $t
	return
    }

    dsk_busy 

    toplevel $t
    wm withdraw $t

    frame $t.fl -bd 1 -relief raised
    pack $t.fl -fill x

    label $t.label -text "Background Jobs of TkDesk:"
    pack $t.label -in $t.fl -side left -padx $tkdesk(pad) -pady $tkdesk(pad)

    frame $t.fj
    pack $t.fj -fill both -expand yes

    cb_listbox $t.flb -vscroll 1 -hscroll 1 -lborder 1 -uborder 1 \
		-pad $tkdesk(pad) -font $tkdesk(font,file_lbs) \
		-width 20 -height 5
    $t.flb config -bd 1 -relief raised
    pack $t.flb -in $t.fj -side left -fill both -expand yes

    frame $t.fjb -bd 1 -relief raised
    pack $t.fjb -in $t.fj -side left -fill y

    button $t.bTerm -text "Terminate" -width 9 -command "dsk_jobs_sig term"
    pack $t.bTerm -in $t.fjb -padx $tkdesk(pad) -pady $tkdesk(pad) -ipady 2

    button $t.bHang -text "Hang up" -width 9 -command "dsk_jobs_sig hangup"
    pack $t.bHang -in $t.fjb -padx $tkdesk(pad) -pady $tkdesk(pad) -ipady 2

    button $t.bKill -text "Kill!" -width 9 -command "dsk_jobs_sig kill"
    pack $t.bKill -in $t.fjb -padx $tkdesk(pad) -pady $tkdesk(pad) -ipady 2

    button $t.bStop -text "Stop" -width 9 -command "dsk_jobs_sig stop"
    pack $t.bStop -in $t.fjb -padx $tkdesk(pad) -pady $tkdesk(pad) -ipady 2

    button $t.bCont -text "Continue" -width 9 -command "dsk_jobs_sig cont"
    pack $t.bCont -in $t.fjb -padx $tkdesk(pad) -pady $tkdesk(pad) -ipady 2

    frame $t.fb -bd 1 -relief raised
    pack $t.fb -fill x

    button $t.bClose -text "  Close  " -command {
		set tkdesk(geometry,dsk_jobs) [wm geometry .dsk_jobs]
		destroy .dsk_jobs }
    pack $t.bClose -in $t.fb -side left \
		-padx $tkdesk(pad) -pady $tkdesk(pad) -ipady 2

    button $t.bUpdate -text " Update " -command "dsk_jobs_fill"
    pack $t.bUpdate -in $t.fb -side left \
		-padx $tkdesk(pad) -pady $tkdesk(pad) -ipady 2

    wm protocol $t WM_DELETE_WINDOW {
		set tkdesk(geometry,dsk_jobs) [wm geometry .dsk_jobs]
		destroy .dsk_jobs }
    wm minsize $t 276 278
    wm title $t "Job Control"
    wm deiconify $t

    if {$tkdesk(geometry,dsk_jobs) == ""} {
	wm geometry $t 422x278
    } else {
	wm geometry $t $tkdesk(geometry,dsk_jobs)
    }

    dsk_jobs_update
    dsk_lazy
}


proc dsk_jobs_fill {} {
    global tkdesk dsk_jobs

    #dsk_busy

    set pslist [split [exec ps] \n]
    set dsk_jobs(pids) ""
    set t .dsk_jobs
    set csel [$t.flb.lbox curselection]
    $t.flb.lbox delete 0 end

    set i 0
    foreach cmd $tkdesk(dsk_exec,cmds) {
	set pid [lindex $tkdesk(dsk_exec,pids) $i]
	if {[lsearch -regexp $pslist "^ *$pid +"] > -1} {
	    lappend dsk_jobs(pids) $pid
	    $t.flb.lbox insert end $cmd
	} else {
	    lreplace $tkdesk(dsk_exec,cmds) $i $i
	    lreplace $tkdesk(dsk_exec,pids) $i $i
	}
	incr i
    }
    if {$csel != ""} {
	$t.flb.lbox selection set [lindex $csel 0]
    }

    set t .dsk_jobs

    #dsk_lazy
}


proc dsk_jobs_sig {signal} {
    global dsk_jobs tkdesk

    switch $signal {
	term	{set signum 15}
	hangup	{set signum 1}
	kill	{set signum 9}
	stop	{set signum 19}
	cont	{set signum 18}
	default {cb_error "Unknown signal $signal!" ; return}
    }

    set sel [.dsk_jobs.flb.lbox curselection]
    if {$sel != ""} {
	foreach i $sel {
	    set pid [lindex $dsk_jobs(pids) $i]
	    set err [catch "exec kill -$signum $pid" m]
	    dsk_debug "exec kill -$signum $pid"
	    if {$err && $tkdesk(in_development)} {
		dsk_errbell
		cb_error "kill $signum failed! ($m)"
	    }
#	    if {$signal == "stop"} {
#		set j [lsearch $tkdesk(dsk_exec,pids) $pid]
#		if {$j > -1} {
#		    set tkdesk(dsk_exec,cmds) [lreplace \
#			$tkdesk(dsk_exec,cmds) $j $j \
#			"[lindex $tkdesk(dsk_exec,cmds) $j] (stopped)"]
#		}
#	    }
	}

	after 500 dsk_jobs_fill
    } else {
	cb_info "Please select one or more jobs first."
    }
}


proc dsk_jobs_update {} {
    global tkdesk

    if [winfo exists .dsk_jobs] {
    	dsk_jobs_fill
	after [expr $tkdesk(update,jobs) * 1000] dsk_jobs_update
    }
}

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_edit
# Args:		args		a list of filenames to edit
# Returns: 	""
# Desc:		Calls the in "System" specified editor on the given files.
#		The editor will run in the background.
# Side-FX:	none
#

proc dsk_edit {args} {
    global tkdesk

    set files ""
    foreach f $args {
	if ![file isdirectory $f] {
	    lappend files $f
	}
    }
    
    if $tkdesk(editor,mfiles) {
	if {$tkdesk(editor,cmd) != "builtin"} {
	    eval dsk_exec $tkdesk(editor,cmd) $files
	} else {
	    if {$files != ""} {
		dsk_Editor .de[dsk_Editor :: id] -files $files
	    } else {
		dsk_editor new
	    }
	}
    } else {
	foreach file $files {
	    if {$tkdesk(editor,cmd) != "builtin"} {
		dsk_exec $tkdesk(editor,cmd) $file
	    } else {
		dsk_Editor .de[dsk_Editor :: id] -files $file
	    }

	}
    }

    return
}

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_view
# Args:		args - shell command
# Returns: 	""
# Desc:		Displays the standard output of command $args in the builtin
#               editor.
# Side-FX:	none
#

proc dsk_view {args} {

    eval dsk_editor cmd $args
}

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_confirm
# Args:         msg - message to display in the dialog box
#               args - tcl script
# Returns: 	""
# Desc:		Evaluates $args after a positive confirmation.
# Side-FX:	none
#

proc dsk_confirm {msg args} {

    if ![cb_okcancel $msg] {
	eval [lindex $args 0]
    }
}

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_cbhelp
# Args:		file, (opt.) mode/regexp
# Returns: 	""
# Desc:		Invokes the cb_Help::show class procedure.
# Side-FX:	none
#

proc dsk_cbhelp {file {regexp ""}} {
    global tkdesk

    dsk_busy
    cb_Help :: setfont $tkdesk(font,mono)
    cb_Help :: textbg $tkdesk(color,basic)
    cb_Help :: show $file $regexp
    dsk_lazy
}

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_sound
# Args:		snd: name of sound event, e.g. welcome, launch.
#               The soundfile is accessed by $tkdesk(sound,$snd).
# Returns: 	""
# Desc:		Plays a sound file via $tkdesk(soundcmd).
# Side-FX:	none
#

proc dsk_sound {snd} {
    global tkdesk

    if ![info exists tkdesk(soundcmd)] return
    if {$tkdesk(soundcmd) == ""} return
    
    if ![info exists tkdesk(sound,$snd)] {
	#puts stderr "tkdesk: no associated sound file for $snd"
	return
    } else {
	if {$tkdesk(sound,$snd) == ""} return
	
	#update
	set errmsg ""
	set cmd [format "$tkdesk(soundcmd)" \
		[cb_tilde $tkdesk(sound,$snd) expand]]
	catch {eval exec $cmd} errmsg
	# prevent the pid in $errmsg from being printed
	if {$errmsg != "" && [llength $errmsg] > 1} {
	    puts stderr $errmsg
	}
    }
}

# ---------------------------------------------------------------------------
# dsk_bell:
# Rings the bell or plays the corresponding sound file:
#
proc dsk_bell {} {
    global tkdesk

    if {![info exists tkdesk(soundcmd)] || \
	    ![info exists tkdesk(sound,dsk_bell)]} {
	bell
    } else {
	dsk_sound dsk_bell
    }
}

# ---------------------------------------------------------------------------
# dsk_errbell:
# Rings the bell or plays the corresponding sound file:
#
proc dsk_errbell {} {
    global tkdesk

    if {![info exists tkdesk(soundcmd)] || \
	    ![info exists tkdesk(sound,dsk_error)]} {
	bell
    } else {
	dsk_sound dsk_error
    }
}

# ---------------------------------------------------------------------------
# dsk_print:
# Asks for the command to use when printing files.
#
proc dsk_print {args} {
    global tkdesk dsk_print_cmd

    set files $args
    if {$files == ""} {
    	set files [_make_fnames_safe]
    }

    if {$files == ""} {
	cb_info "Please select one or more files first."
	return
    }

    if [info exists tkdesk(cmd,print)] {
	set dsk_print_cmd $tkdesk(cmd,print)
    } else {
	set dsk_print_cmd "lpr"
    }

    cb_readString "Print command (file names will be appended):" \
	    dsk_print_cmd "Print Command"

    if {$dsk_print_cmd != ""} {
	eval exec "$dsk_print_cmd $files" &
	set tkdesk(cmd,print) $dsk_print_cmd
    }
}
