/* This software is Copyright 1995 by Karl-Johan Johnsson
 *
 * Permission is hereby granted to copy, reproduce, redistribute or otherwise
 * use this software as long as: there is no monetary profit gained
 * specifically from the use or reproduction of this software, it is not
 * sold, rented, traded or otherwise marketed, and this copyright notice is
 * included prominently in any copy made. 
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. ANY USE OF THIS
 * SOFTWARE IS AT THE USER'S OWN RISK.
 */
#include "global.h"
#include "child.h"
#include "file.h"
#include "kill.h"
#include "pixmaps.h"
#include "resource.h"
#include "util.h"
#include "thread.h"

#define KILL_SIZE	16
#define SKIP_SPACE(c)	while (IS_SPACE(*c)) c++

void parse_kill_file(void)
{
    FILE	*fp = NULL;
    char	buffer[1024];
    char	*file_name = res_kill_file();
    KILL_NODE	**list = NULL;
    int		n, no_alloc = 0;
    regex_t	re;

    global.kill_list = NULL;

    if (file_name)
	fp = fopen_expand(file_name, "r", False);

    if (fp) {
	for (n = 0 ; fgets(buffer, sizeof(buffer), fp) ; n++) {
	    char	*c, *p;
	    
	    c = strchr(buffer, '\n');
	    if (c) *c = '\0';
	    
	    for (c = buffer ; IS_SPACE(*c) ; c++);
	    if (*c == '\0') {
		n--;
		continue;
	    }

	    if (n + 2 > no_alloc) {
		no_alloc += KILL_SIZE;
		list =
		    (KILL_NODE **)XtRealloc((char *)list,
					    no_alloc * sizeof(KILL_NODE*));
	    }

	    list[n] = (KILL_NODE *)XtMalloc(sizeof(KILL_NODE));
	    list[n]->expr_str = NULL;
	    list[n]->expr_re = NULL;
	    list[n]->group_str = NULL;
	    list[n]->group_re = NULL;
	    list[n]->color = NULL;
	    list[n]->hot = 0;
	    list[n]->field = KillFieldSubject;
	    list[n]->scope = KillScopeArticle;
	    list[n]->action = KillActionKill;
	    list[n]->valid = True;
	    list[n]->applicable = False;
	    list[n]->unused = False;
	    list[n]->alloced_pixel = False;
	    list[n]->msgid_valid = False;

	    SKIP_SPACE(c);
	    switch (*c++) {
	    case 'M':
	    case 'm':
		list[n]->field = KillFieldMsgid;
		break;
	    case 'S':
	    case 's':
		list[n]->field = KillFieldSubject;
		break;
	    case 'F':
	    case 'f':
		list[n]->field = KillFieldFrom;
		break;
	    case 'X':
	    case 'x':
		list[n]->field = KillFieldXref;
		break;
	    default:
		fprintf(stderr,
			"knews: Unknown kill field "
			"'%c' in kill file.\n",
			*c);
		list[n]->valid = False;
		break;
	    }

	    SKIP_SPACE(c);
	    switch (*c++) {
	    case 'A':
	    case 'a':
		list[n]->scope = KillScopeArticle;
		break;
	    case 'S':
	    case 's':
		list[n]->scope = KillScopeSubject;
		break;
	    case 'T':
		list[n]->scope = KillScopeThread;
		break;
	    case 't':
		list[n]->scope = KillScopeSubthread;
		break;
	    case '\0':
		fprintf(stderr,
			"knews: Premature end of line in kill file.\n");
		list[n]->valid = False;
		continue;
	    default:
		fprintf(stderr,
			"knews: Unknown kill scope "
			"'%c' in kill file.\n",
			*c);
		list[n]->valid = False;
		break;
	    }

	    SKIP_SPACE(c);
	    switch (*c) {
	    case 'K':
	    case 'k':
		c++;
		list[n]->action = KillActionKill;
		SKIP_SPACE(c);
		if (c[0] == '\0')
		    break;
		else if (c[0] == '|' && c[1] == '|')
		    c += 2;
		else {
		    fprintf(stderr,
			    "knews: Color specified for kill entry"
			    "in kill file, ignored.\n");
		    c = strstr(c, "||");
		    if (c)
			c += 2;
		}
		break;
	    case 'H':
	    case 'h':
		list[n]->action =
		    *c++ == 'H' ? KillActionHot : KillActionHotish;
		SKIP_SPACE(c);
		if (c[0] == '\0')
		    break;
		else if (c[0] == '|' && c[1] == '|') {
		    fprintf(stderr,
			    "knews: No color specified for hot entry"
			    "in kill file.\n");
		    c += 2;
		    list[n]->valid = False;
		} else {
		    p = strstr(c, "||");
		    if (p) {
			*p = '\0';
			list[n]->color = XtNewString(c);
			c = p + 2;
		    } else
			c = NULL;
		}
		break;
	    case '\0':
		break;
	    default:
		list[n]->valid = False;
		fprintf(stderr,
			"knews: Error in kill file expected 'H' or 'K'.\n");
		c = strstr(c + 1, "||");
		break;
	    }

	    if (!c || *c == '\0') {
		fprintf(stderr,
			"knews: Premature end of line in kill file.\n");
		list[n]->valid = False;
		continue;
	    }

	    p = strstr(c, "||");
	    if (p) *p = '\0';
	    list[n]->group_str = XtNewString(c);
	    if (!p) {
		list[n]->valid = False;
		fprintf(stderr,
			"knews: Premature end of line in kill file.\n");
		continue;
	    }
	    if (regcomp(&re, c, REGEXP_COMPILE_FLAGS) == 0) {
		list[n]->group_re = (regex_t *)XtMalloc(sizeof(regex_t));
		memcpy(list[n]->group_re, &re, sizeof(regex_t));
	    } else {
		list[n]->valid = False;
		fprintf(stderr,
			"knews: Parse error in group regexp: %s\n", c);
	    }

	    c = p + 2;
	    SKIP_SPACE(c);
	    if (*c == '\0') {
		list[n]->valid = False;
		fprintf(stderr,
			"knews: Premature end of line in kill file.\n");
		continue;
	    }

	    if (list[n]->field == KillFieldMsgid) {
		Boolean	valid = True;
		
		if (*c != '<')
		    valid = False;
		p = c + strlen(c) - 1;
		while (IS_SPACE(*p))
		    p--;
		if (*p != '>')
		    valid = False;
		*(p+1) = '\0';

		p = strchr(c, '@');
		for (p++ ; *p != '\0' && *p != '>' ; p++)
		    if ('A' <= *p && *p <= 'Z')
			*c += 'a' - 'A';
		if (*p != '>')
		    valid = False;

		list[n]->expr_str = XtNewString(c);
		list[n]->msgid_valid = valid;
		if (!valid) {
		    list[n]->valid = False;
		    fprintf(stderr,
			    "knews: Invalid msgid in kill file: %s\n",
			    c);
		}
	    } else {
		list[n]->expr_str = XtNewString(c);
		if (regcomp(&re, c, REGEXP_COMPILE_FLAGS) == 0) {
		    list[n]->expr_re = (regex_t *)XtMalloc(sizeof(regex_t));
		    memcpy(list[n]->expr_re, &re, sizeof(regex_t));
		} else {
		    list[n]->valid = False;
		    fprintf(stderr,
			    "knews: Parse error in expression regexp: %s\n",
			    c);
		}
	    }
	}

	fclose(fp);

	while (n < no_alloc)
	    list[n++] = NULL;
	global.kill_list = list;
	global.no_alloc_kill_list = no_alloc;
    }
    
    alloc_hot_pixmaps();
}

int update_kill_file(void)
{
    char	*file_name = res_kill_file();
    FILE	*fp;
    KILL_NODE	**loop = global.kill_list;
    KILL_NODE	*node;
    int		expire = res_expire_kills() && !res_ask_how_many();
    int		ok;

    if (!file_name)
	return False;
    if (!loop)
	return True;

    sighup_may_exit = False;

    fp = fopen_expand(file_name, "w", True);
    if (!fp) {
	sighup_may_exit = True;
	return False;
    }

    while ((node = *loop++)) {
	if (expire && node->applicable && node->unused)
	    continue;

	putc("MSFX"[node->field], fp);
	putc("ASTt"[node->scope], fp);
	putc("KHh"[node->action], fp);

	if (node->color)
	    fprintf(fp, "%s", node->color);
	putc('|', fp);
	putc('|', fp);
	if (node->group_str)
	    fprintf(fp, "%s", node->group_str);
	putc('|', fp);
	putc('|', fp);
	if (node->expr_str)
	    fprintf(fp, "%s", node->expr_str);
	putc('\n', fp);
    }

    ok = (fclose(fp) != EOF);

    sighup_may_exit = True;
    if (caught_sighup)
	sighup_handler(0);

    return ok;
}

void free_kill_file(void)
{
    int	i;

    free_hot_pixmaps();

    for (i = 0 ; i < global.no_alloc_kill_list ; i++) {
	KILL_NODE	*node = global.kill_list[i];

	if (!node)
	    break;

	XtFree(node->expr_str);
	XtFree(node->group_str);
	XtFree(node->color);
	if (node->expr_re) {
	    regfree(node->expr_re);
	    XtFree((char *)node->expr_re);
	}
	if (node->group_re) {
	    regfree(node->group_re);
	    XtFree((char *)node->group_re);
	}
	XtFree((char *)node);
    }

    XtFree((char *)global.kill_list);
    global.kill_list = NULL;
    global.no_alloc_kill_list = 0;
}

/*********************************************************************/

static ARTICLE *get_by_msgid(KILL_NODE *node)
{
    ARTICLE	*art;

    art = find_article(node->expr_str + 1, strlen(node->expr_str) - 2);
    node->applicable = True;
    node->unused = art == NULL;

    return art;
}

static long m_a_kill(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    ARTICLE	*art = get_by_msgid(node);

    if (!art || !art->from || art->read || art->hot != 0)
	return 0;

    art->read = True;
    art->killed = True;
    global.curr_group->no_unread--;
    art->subject->no_unread--;

    return 1;
}

static long m_a_hot(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    ARTICLE	*art = get_by_msgid(node);

    if (art && art->from && art->hot == 0) {
	art->hot = node->hot;
	if (!art->read)
	    global.n_hot++;
    }

    return 0;
}

static long m_s_kill(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    ARTICLE	*art = get_by_msgid(node);

    if (!art)
	return 0;

    return mark_subject_read(art->subject, False, True);
}

static long m_s_hot(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    ARTICLE	*art = get_by_msgid(node);

    if (art)
	global.n_hot += mark_subject_hot(art->subject, node->hot);

    return 0;
}

static long m_T_kill(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    ARTICLE	*art = get_by_msgid(node);

    if (!art)
	return 0;

    return mark_thread_read(art->subject->thread, False, True);
}

static long m_T_hot(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    ARTICLE	*art = get_by_msgid(node);

    if (art)
	global.n_hot += mark_thread_hot(art->subject->thread, node->hot);

    return 0;
}

static long m_t_kill(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    ARTICLE	*art = get_by_msgid(node);

    if (!art)
	return 0;

    return mark_subthread_read(art, False, True);
}

static long m_t_hot(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    ARTICLE	*art = get_by_msgid(node);

    if (art)
	global.n_hot += mark_subthread_hot(art, node->hot);

    return 0;
}

/*********************************************************************/

static long s_s_kill(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    long	n = 0;
    SUBJECT	*subj;

    for (subj = subjects ; subj ; subj = subj->next)
	if (regexec(node->expr_re, subj->subject, 0, NULL, 0) == 0)
	    n += mark_subject_read(subj, False, True);

    return n;
}

static long s_s_hot(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    SUBJECT	*subj;

    for (subj = subjects ; subj ; subj = subj->next)
	if (regexec(node->expr_re, subj->subject, 0, NULL, 0) == 0)
	    global.n_hot += mark_subject_hot(subj, node->hot);

    return 0;
}

static long s_T_kill(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    long	n = 0;
    SUBJECT	*subj;

    for (subj = subjects ; subj ; subj = subj->next)
	if (regexec(node->expr_re, subj->subject, 0, NULL, 0) == 0)
	    n += mark_thread_read(subj->thread, False, True);

    return n;
}

static long s_T_hot(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    SUBJECT	*subj;

    for (subj = subjects ; subj ; subj = subj->next)
	if (regexec(node->expr_re, subj->subject, 0, NULL, 0) == 0)
	    global.n_hot += mark_thread_hot(subj->thread, node->hot);

    return 0;
}

static long s_t_kill(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    long	n = 0;
    SUBJECT	*subj;

    for (subj = subjects ; subj ; subj = subj->next)
	if (regexec(node->expr_re, subj->subject, 0, NULL, 0) == 0) {
	    ARTICLE	*art = subj->thread;

	    while (art)
		if (art->subject != subj)
		    art = next_in_thread_preorder(art);
		else {
		    n += mark_subthread_read(art, False, True);
		    art = preorder_skip_subthread(art);
		}
	}

    return n;
}

static long s_t_hot(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    SUBJECT	*subj;

    for (subj = subjects ; subj ; subj = subj->next)
	if (regexec(node->expr_re, subj->subject, 0, NULL, 0) == 0) {
	    ARTICLE	*art = subj->thread;

	    while (art)
		if (art->subject != subj)
		    art = next_in_thread_preorder(art);
		else {
		    global.n_hot += mark_subthread_hot(art, node->hot);
		    art = preorder_skip_subthread(art);
		}
	}

    return 0;
}

/*********************************************************************/

static long f_a_kill(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    long	n = 0;
    ARTICLE	*art;

    for (art = articles ; art ; art = art->next) {
	if (art->from && !art->read && art->hot == 0 &&
	    regexec(node->expr_re, art->from, 0, NULL, 0) == 0) {
	    art->read = True;
	    art->killed = True;
	    n++;
	    global.curr_group->no_unread--;
	    art->subject->no_unread--;
	}
    }

    return 0;
}

static long f_a_hot(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    ARTICLE	*art;

    for (art = articles ; art ; art = art->next)
	if (art->from && art->hot == 0 &&
	    regexec(node->expr_re, art->from, 0, NULL, 0) == 0) {
	    art->hot = node->hot;
	    if (!art->read)
		global.n_hot++;
	}

    return 0;
}

static long f_a_hotish(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    ARTICLE	*art;

    for (art = articles ; art ; art = art->next)
	if (art->from && art->hot == 0 && !art->read &&
	    regexec(node->expr_re, art->from, 0, NULL, 0) == 0) {
	    art->hot = node->hot;
	    if (!art->read)
		global.n_hot++;
	}

    return 0;
}

static long f_s_kill(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    long	n = 0;
    ARTICLE	*art;

    for (art = articles ; art ; art = art->next)
	if (art->from && regexec(node->expr_re, art->from, 0, NULL, 0) == 0)
	    n += mark_subject_read(art->subject, False, True);

    return n;
}

static long f_s_hot(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    ARTICLE	*art;

    for (art = articles ; art ; art = art->next)
	if (art->from && regexec(node->expr_re, art->from, 0, NULL, 0) == 0)
	    global.n_hot += mark_subject_hot(art->subject, node->hot);

    return 0;
}

static long f_T_kill(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    long	n = 0;
    SUBJECT	*subj;
    ARTICLE	*art;

    for (subj = subjects ; subj ; subj = subj->next) {
	if (subj->prev && subj->prev->thread == subj->thread)
	    continue;
	for (art = subj->thread ; art ; art = next_in_thread_preorder(art))
	    if (art->from &&
		regexec(node->expr_re, art->from, 0, NULL, 0) == 0) {
		n += mark_thread_read(subj->thread, False, True);
		break;
	    }
    }

    return n;
}

static long f_T_hot(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    SUBJECT	*subj;
    ARTICLE	*art;

    for (subj = subjects ; subj ; subj = subj->next) {
	if (subj->prev && subj->prev->thread == subj->thread)
	    continue;
	for (art = subj->thread ; art ; art = next_in_thread_preorder(art))
	    if (art->from &&
		regexec(node->expr_re, art->from, 0, NULL, 0) == 0) {
		global.n_hot += mark_thread_hot(subj->thread, node->hot);
		break;
	    }
    }

    return 0;
}

static long f_t_kill(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    long	n = 0;
    SUBJECT 	*subj;

    for (subj = subjects ; subj ; subj = subj->next) {
	ARTICLE	*art = subj->thread;

	if (subj->prev && subj->prev->thread == art)
	    continue;

	while (art)
	    if (art->from &&
		regexec(node->expr_re, art->from, 0, NULL, 0) == 0) {
		n += mark_subthread_read(art, False, True);
		art = preorder_skip_subthread(art);
	    } else
		art = next_in_thread_preorder(art);
    }

    return n;
}

static long f_t_hot(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    SUBJECT 	*subj;

    for (subj = subjects ; subj ; subj = subj->next) {
	ARTICLE	*art = subj->thread;

	if (subj->prev && subj->prev->thread == art)
	    continue;

	while (art)
	    if (art->from &&
		regexec(node->expr_re, art->from, 0, NULL, 0) == 0) {
		global.n_hot += mark_subthread_hot(art, node->hot);
		art = preorder_skip_subthread(art);
	    } else
		art = next_in_thread_preorder(art);
    }

    return 0;
}

/*********************************************************************/

static long x_a_kill(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    long	n = 0;
    ARTICLE	*art;

    for (art = articles ; art ; art = art->next) {
	if (art->xref && !art->read && art->hot == 0 &&
	    regexec(node->expr_re, art->xref, 0, NULL, 0) == 0) {
	    art->read = True;
	    art->killed = True;
	    global.curr_group->no_unread--;
	    art->subject->no_unread--;
	    n++;
	}
    }

    return n;
}

static long x_a_hot(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    ARTICLE	*art;

    for (art = articles ; art ; art = art->next)
	if (art->xref && art->hot == 0 &&
	    regexec(node->expr_re, art->xref, 0, NULL, 0) == 0) {
	    art->hot = node->hot;
	    if (!art->read)
		global.n_hot++;
	}

    return 0;
}

static long x_s_kill(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    long	n = 0;
    SUBJECT	*subj;
    ARTICLE	*art;

    for (subj = subjects ; subj ; subj = subj->next)
	for (art = subj->thread ; art ; art = next_in_thread_preorder(art)) {
	    if (art->subject != subj)
		continue;
	    if (art->xref &&
		regexec(node->expr_re, art->xref, 0, NULL, 0) == 0) {
		n += mark_subject_read(subj, False, True);
		break;
	    }
	}

    return n;
}

static long x_s_hot(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    SUBJECT	*subj;
    ARTICLE	*art;

    for (subj = subjects ; subj ; subj = subj->next)
	for (art = subj->thread ; art ; art = next_in_thread_preorder(art)) {
	    if (art->subject != subj)
		continue;
	    if (art->xref &&
		regexec(node->expr_re, art->xref, 0, NULL, 0) == 0) {
		global.n_hot +=	mark_subject_hot(subj, node->hot);
		break;
	    }
	}

    return 0;
}

static long x_T_kill(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    long	n = 0;
    SUBJECT	*subj;
    ARTICLE	*art;

    for  (subj = subjects ; subj ; subj = subj->next) {
	if (subj->prev && subj->prev->thread == subj->thread)
	    continue;
	for (art = subj->thread ; art ; art = next_in_thread_preorder(art))
	    if (art->xref &&
		regexec(node->expr_re, art->xref, 0, NULL, 0) == 0) {
		n += mark_thread_read(subj->thread, False, True);
		break;
	    }
    }

    return n;
}

static long x_T_hot(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    SUBJECT	*subj;
    ARTICLE	*art;

    for  (subj = subjects ; subj ; subj = subj->next) {
	if (subj->prev && subj->prev->thread == subj->thread)
	    continue;
	for (art = subj->thread ; art ; art = next_in_thread_preorder(art))
	    if (art->xref &&
		regexec(node->expr_re, art->xref, 0, NULL, 0) == 0) {
		global.n_hot +=	mark_thread_hot(subj->thread, node->hot);
		break;
	    }
    }

    return 0;
}

static long x_t_kill(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    long	n = 0;
    SUBJECT	*subj;

    for  (subj = subjects ; subj ; subj = subj->next) {
	ARTICLE	*art = subj->thread;
	
	if (subj->prev && subj->prev->thread == art)
	    continue;

	while (art)
	    if (art->xref &&
		regexec(node->expr_re, art->xref, 0, NULL, 0) == 0) {
		n += mark_subthread_read(art, False, True);
		art = preorder_skip_subthread(art);
	    } else
		art = next_in_thread_preorder(art);
    }

    return n;
}

static long x_t_hot(KILL_NODE *node, ARTICLE *articles, SUBJECT *subjects)
{
    SUBJECT	*subj;

    for  (subj = subjects ; subj ; subj = subj->next) {
	ARTICLE	*art = subj->thread;
	
	if (subj->prev && subj->prev->thread == art)
	    continue;

	while (art)
	    if (art->xref &&
		regexec(node->expr_re, art->xref, 0, NULL, 0) == 0) {
		global.n_hot +=	mark_subthread_hot(art, node->hot);
		art = preorder_skip_subthread(art);
	    } else
		art = next_in_thread_preorder(art);
    }

    return 0;
}

/*********************************************************************/

static long (*funcs[4][4][3])(KILL_NODE*, ARTICLE*, SUBJECT*) = {
    {{m_a_kill, m_a_hot, m_a_hot},
     {m_s_kill, m_s_hot, m_s_hot},
     {m_T_kill, m_T_hot, m_T_hot},
     {m_t_kill, m_t_hot, m_t_hot}},
    {{s_s_kill, s_s_hot, s_s_hot},
     {s_s_kill, s_s_hot, s_s_hot},
     {s_T_kill, s_T_hot, s_T_hot},
     {s_t_kill, s_t_hot, s_t_hot}},
    {{f_a_kill, f_a_hot, f_a_hotish},
     {f_s_kill, f_s_hot, f_s_hot},
     {f_T_kill, f_T_hot, f_T_hot},
     {f_t_kill, f_t_hot, f_t_hot}},
    {{x_a_kill, x_a_hot, x_a_hot},
     {x_s_kill, x_s_hot, x_s_hot},
     {x_T_kill, x_T_hot, x_T_hot},
     {x_t_kill, x_t_hot, x_t_hot}},
};

long kill_articles(void)
{
    KILL_NODE	**loop = global.kill_list;
    ARTICLE	*art, *articles;
    SUBJECT	*subj, *subjects;
    long	n_killed = 0;

    articles = get_articles(main_thr);
    subjects = get_subjects(main_thr);

    global.n_hot = 0;
    for (art = articles ; art ; art = art->next)
	art->hot = 0;

    if (!loop)
	return 0;

    for ( ; *loop ; loop++) {
	KILL_NODE	*node = *loop;

	if (!node->valid || !node->group_re ||
	    regexec(node->group_re, global.curr_group->name, 0, NULL, 0) != 0)
	    continue;

	n_killed +=
	    funcs[node->field][node->scope][node->action]
	    (node, articles, subjects);
    }

    for (subj = subjects ; subj ; subj = subj->next) {
	subj->hot = 0; /* just making sure */
	update_subj_hot_value(subj);
    }

    return n_killed;
}

char *regexp_escape_string(char *src, int anchor)
{
    char	*result, *dest;

    dest = result = XtMalloc(2 * strlen(src) + 3);

    if (anchor)
	*dest++ = '^';
    while (*src != '\0') {
	if (strchr("^.[$()|*+?{\\", *src)) *dest++ = '\\';
	*dest++ = *src++;
    }
    if (anchor)
	*dest++ = '$';
    *dest = '\0';

    return XtRealloc(result, strlen(result) + 1);
}
