/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@medizin.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: pinout.c,v 1.5 94/07/19 14:08:21 nau Exp $";

/* pinout routines
 */
#include <stdio.h>
#include <string.h>

#include "global.h"

#include "create.h"
#include "data.h"
#include "draw.h"
#include "memory.h"
#include "misc.h"
#include "pinout.h"
#include "transform.h"

#include <X11/Shell.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Simple.h>
#include <X11/Xaw/Viewport.h>

/* ---------------------------------------------------------------------------
 * local defines
 */
#define	OFFSET			50		/* offset in mil */

/* ---------------------------------------------------------------------------
 * some local types
 */
typedef struct					/* information of one window */
{
	ElementType		Element;	/* element data to display */
	String			Title;		/* window title */
	Widget			Shell,		/* shell widget */
					Output;		/* output widget managed by a viewport widget */
} PinoutType, *PinoutTypePtr;

/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	void			CB_Dismiss(Widget, XtPointer, XtPointer);
static	XtEventHandler	PinoutEvent(Widget, XtPointer, XEvent *, Boolean *);

/* ---------------------------------------------------------------------------
 * creates a new window to display an elements pinout
 */
void CreatePinoutWindow(Widget Parent, ElementTypePtr Element)
{
	Widget			masterform,
					dismiss,
					viewport;
	PinoutTypePtr	pinout;
	LineTypePtr		line;
	ArcTypePtr		arc;
	Cardinal		i;

		/* allocate memory for title and pinout data */
	if (!Element)
		return;
	pinout = (PinoutTypePtr) MyCalloc(1, sizeof(PinoutType), "CreatePinoutWindow()");
	pinout->Title = (String) MyCalloc(strlen(UNKNOWN(Element->CanonicalName))+
		strlen(UNKNOWN(Element->NameOnPCB))+4, sizeof(char), "CreatePinoutWindow()");
	sprintf(pinout->Title, "%s [%s]", UNKNOWN(Element->CanonicalName),
		UNKNOWN(Element->NameOnPCB));

		/* copy element data 
		 * move element to a 5% offset from zero position
		 * remove both names (the take too much screen space)
		 * set all package lines to zero with
		 */
	CopyElement(NULL, Element, &pinout->Element);
	SetElementInfo(&pinout->Element, 1);
	MoveElement(&pinout->Element,
		-pinout->Element.Rect.X +OFFSET, -pinout->Element.Rect.Y +OFFSET);
	MyFree(&pinout->Element.CanonicalName);
	MyFree(&pinout->Element.NameOnPCB);
	for (i = pinout->Element.LineN, line = pinout->Element.Line; i; i--, line++)
		line->Thickness = 0;
	for (i = pinout->Element.ArcN, arc = pinout->Element.Arc; i; i--, arc++)
		arc->Thickness = 0;

		/* create shell window with viewport and exit button */
	pinout->Shell = XtVaCreatePopupShell("pinout", topLevelShellWidgetClass,
		Parent,
		XtNtitle, pinout->Title,
		XtNallowShellResize, True,
		NULL);
	masterform = XtVaCreateManagedWidget("pinoutMasterForm", formWidgetClass,
		pinout->Shell,
		XtNresizable, True,
		XtNfromHoriz, NULL,
		XtNfromVert, NULL,
		NULL);
	viewport = XtVaCreateManagedWidget("viewport", viewportWidgetClass,
		masterform,
		XtNresizable, True,
		XtNleft, XtChainLeft,
		XtNright, XtChainRight,
		XtNtop, XtChainTop,
		XtNbottom, XtChainBottom,
		XtNallowHoriz, True,
		XtNallowVert, True,
		XtNuseBottom, True,
		XtNforceBars, True,
		NULL);
	pinout->Output = XtVaCreateManagedWidget("output", simpleWidgetClass,
		viewport,
		XtNwidth, (pinout->Element.Rect.Width +2*OFFSET) >> Settings.PinoutZoom,
		XtNheight, (pinout->Element.Rect.Height +2*OFFSET) >> Settings.PinoutZoom,
		NULL);
	dismiss = XtVaCreateManagedWidget("dismiss", commandWidgetClass,
		masterform,
		XtNlabel, "Dismiss",
		XtNfromVert, viewport,
		XtNleft, XtChainLeft,
		XtNright, XtChainRight,
		XtNtop, XtChainBottom,
		XtNbottom, XtChainBottom,
		NULL);

		/* add event handler for viewport and callback for 'dismiss' button
		 * the pointer to the pinout structure is passed to these functions
		 * as 'ClientData'
		 */
	XtAddCallback(dismiss, XtNcallback, CB_Dismiss, (XtPointer) pinout);
	XtAddEventHandler(pinout->Output,
		ExposureMask,
		True, (XtEventHandler) PinoutEvent, (XtPointer) pinout);

		/* bring all stuff to the screen */
	XtPopup(pinout->Shell, XtGrabNone);
}

/* ---------------------------------------------------------------------------
 * event handler for all pinout windows
 * a pointer to the pinout struct is passed as ClientData
 */
static XtEventHandler PinoutEvent(Widget W, XtPointer ClientData, XEvent *Event, Boolean *Flag)
{
	PinoutTypePtr	pinout = (PinoutTypePtr) ClientData;
	Window			window = XtWindow(pinout->Output);
	PinTypePtr		pin;
	Cardinal		i;

	switch(Event->type)
	{
		case Expose:				/* just redraw the complete window */
			if (window)				/* check for valid ID */
			{
				SwitchDrawingWindow(Settings.PinoutZoom, window);
				DrawNewElement(&pinout->Element);

					/* draw pin names, offset is included in (x,y) */
				for (i = pinout->Element.PinN, pin = pinout->Element.Pin; i; i--, pin++)
				{
					XDrawString(Dpy, window, Output.fgGC,
						pin->X  >> Settings.PinoutZoom,
						((pin->Y -pin->Thickness/2) >> Settings.PinoutZoom)
							-Settings.PinoutFont->max_bounds.descent,
						UNKNOWN(pin->Name),
						MIN(Settings.PinoutNameLength, strlen(UNKNOWN(pin->Name))));
				}
			}
			break;
	}
}

/* ---------------------------------------------------------------------------
 * callback routine of the dismiss buttons of all pinout windows
 * a pointer to the pinout struct is passed as ClientData
 */
static void CB_Dismiss(Widget W, XtPointer ClientData, XtPointer CallData)
{
	PinoutTypePtr	pinout = (PinoutTypePtr) ClientData;

		/* release memory */
	XtDestroyWidget(pinout->Shell);
	SaveFree(pinout->Title);
	FreeElementMemory(&pinout->Element);
	SaveFree(pinout);
}
