/* oswin32/osprg.c */

#include "elvis.h"
#include <sys/types.h>
#include <dos.h>
#include <fcntl.h>
#include <io.h>

static char	*cmd;		/* the command to run */
static char	tempread[100];	/* name of temp file for prog's stdout */
static char	tempwrite[100];	/* name of temp file for prog's stdout */
static int	fd;		/* file descriptor for reading or writing */
static int	status;		/* exit status of the program */

/* Declares which program we'll run, and what we'll be doing with it.
 * This function should return True if successful.  If there is an error,
 * it should issue an error message via msg(), and return False.
 */
BOOLEAN prgopen(char *command, BOOLEAN willwrite, BOOLEAN willread)
{
	/* remember the command we're supposed to run */
	cmd = command;

	/* if we're going to be reading, then choose a tempread name now */
	if (willread)
	{
		strcpy(tempread, o_directory ? dirpath(tochar8(o_directory), "rdXXXXXX") : "rdXXXXXX");
		_mktemp(tempread);

		/* open it and close it, just to make sure it exists */
		fd = _open(tempread, _O_WRONLY|_O_CREAT|_O_EXCL, 0600);
		if (fd < 0)
		{
			msg(MSG_ERROR, "[s]can't create temp file $1", tempread);
			return False;
		}
		_close(fd);
	}
	else
		tempread[0] = '\0';

	/* if we'll be writing, then choose a tempwrite name and open it */
	if (willwrite)
	{
		strcpy(tempwrite, o_directory ? dirpath(tochar8(o_directory), "wrXXXXXX") : "wrXXXXXX");
		_mktemp(tempwrite);
		fd = _open(tempwrite, _O_WRONLY|_O_CREAT|_O_EXCL, 0600);
		if (fd < 0)
		{
			msg(MSG_ERROR, "[s]can't create temp file $1", tempwrite);
			if (tempread[0])
				_unlink(tempread);
			return False;
		}
	}
	else
		tempwrite[0] = '\0';

	return True;
}

/* Write the contents of buf to the program's stdin, and return nbytes
 * if successful, or -1 for error.  Note that this text should
 * be subjected to the same kind of transformations as textwrite().
 * In fact, it may use textwrite() internally.
 *
 * For DOS, this is simply a write() to the temp file.
 */
int prgwrite(CHAR *buf, int nbytes)
{
	return _write(fd, buf, nbytes);
}

/* Marks the end of writing.  Returns True if all is okay, or False if
 * error.
 */
BOOLEAN prggo(void)
{
	int	old0;	/* elvis' stdin */
	int	old1;	/* elvis' stdout */
	int	old2;	/* elvis' stderr */
	CHAR	*arg[3];/* arguments when evaluating command string */

	/* are we supposed to redirect stdin? */
	if (tempwrite[0])
	{
		/* close the file we've been writing to */
		_close(fd);

		/* save the old stdin as another fd so we can switch back later */
		old0 = _dup(0);
		assert(old0 > 2);

		/* open the temp file as stdin */
		close(0);
#ifdef NDEBUG
		(void)_open(tempwrite, _O_RDONLY);
#else
		assert(_open(tempwrite, _O_RDONLY) == 0);
#endif
	}

	/* are we supposed to redirect stdout/stderr ? */
	if (tempread[0])
	{
		/* save the old stdout and stderr as other fds */
		old1 = _dup(1);
		old2 = _dup(2);
		assert(old1 > 2 && old2 > 2);

		/* open the temp file as stdout/stderr */
		_close(1);
		_close(2);
#ifdef NDEBUG
		(void)_open(tempread, _O_WRONLY);
		_dup(1);
#else
		assert(_open(tempread, _O_WRONLY) == 1);
		assert(_dup(1) == 2);
#endif
	}

	/* if redirecting anything, then evaluate the command string like a
	 * message, substituting the input and/or output files for $1 and $2
	 */
	if ((tempwrite[0] || tempread[0]) && CHARchr(cmd, '$'))
	{
		arg[0] = toCHAR(tempwrite);
		arg[1] = toCHAR(tempread);
		arg[2] = NULL;
		cmd = tochar8(calculate(toCHAR(cmd), arg, True));
		if (!cmd)
		{
			if (tempwrite[0])
				remove(tempwrite);
			if (tempread[0])
				remove(tempread);
			return False;
		}
	}

	/* run the program */
	status = system(cmd);

	/* if we redirected stdin, undo it now */
	if (tempwrite[0])
	{
		/* undo the redirection */
		_close(0);
#ifdef NDEBUG
		(void)_dup(old0);
#else
		assert(_dup(old0) == 0);
#endif
		_close(old0);

		/* delete the temp file */
		_unlink(tempwrite);
	}

	/* if we redirected stdout/stderr, undo it now and open the temp file */
	if (tempread[0])
	{
		/* undo the redirection */
		_close(1);
		_close(2);
#ifdef NDEBUG
		(void)_dup(old1);
		(void)_dup(old2);
#else
		assert(_dup(old1) == 1);
		assert(_dup(old2) == 2);
#endif
		_close(old1);
		_close(old2);

		/* open the temp file */
		fd = _open(tempread, _O_RDONLY);
		assert(fd > 0);
	}

	return True;
}


/* Reads text from the program's stdout, and returns the number of
 * characters read.  At EOF, it returns 0.  Note that this text
 * should be subjected to the same kinds of transformations as
 * textread().
 */
int prgread(CHAR *buf, int nbytes)
{
	return _read(fd, buf, nbytes);
}

/* Clean up, and return the program's exit status.  The exit status
 * should be 0 normally.
 */
int prgclose(void)
{
	char	dummy[1];

	/* if we were reading stdout, then close & delete that temp file */
	if (tempread[0])
	{
		_close(fd);
		_unlink(tempread);
	}

	/* for some reason, Windows/NT seems to get confused about where the
	 * console is, after doing any redirection.  We'd better call ttyraw()
	 * just to be on the safe side.
	 */
	ttyraw(dummy);

	/* return the program's exit status */
	return status;
}
