/* dmc.c */
/* Copyright 1995 by Steve Kirkendall */

char id_dmc[] = "$Id: dmc.c,v 2.24 1996/05/06 05:28:44 steve Exp $";

#include "elvis.h"
#ifdef DISPLAY_C

#if USE_PROTOTYPES
static BOOLEAN iskeyword(char *word);
static DMINFO *init(WINDOW win);
static void term(DMINFO *info);
static MARK setup(MARK top, long cursor, MARK bottom, DMINFO *info);
static MARK image(WINDOW w, MARK line, DMINFO *info, void (*draw)(CHAR *p, long qty, _char_ font, long offset));
#endif

/* These are the font names, used in the option descriptions below */
char fontnames[] = "normal bold emphasized italics underlined fixed";

/* These are the descriptions and values of some global options */
static OPTDESC globdesc[] =
{
	{"commentfont","cfont",	opt1string,	optisoneof,	fontnames },
	{"stringfont","sfont",	opt1string,	optisoneof,	fontnames },
	{"keywordfont","kfont",	opt1string,	optisoneof,	fontnames },
	{"functionfont","ffont",opt1string,	optisoneof,	fontnames },
	{"variablefont","vfont",opt1string,	optisoneof,	fontnames },
	{"prepfont", "pfont",	opt1string,	optisoneof,	fontnames },
	{"otherfont", "ofont",	opt1string,	optisoneof,	fontnames },
};
static OPTVAL globval[QTY(globdesc)];
#define o_commentfont	globval[0].value.character
#define o_stringfont	globval[1].value.character
#define o_keywordfont	globval[2].value.character
#define o_functionfont	globval[3].value.character
#define o_variablefont	globval[4].value.character
#define o_prepfont	globval[5].value.character
#define o_otherfont	globval[6].value.character

/* This data type is used to denote a token type.  Values of this type will
 * be used as indicies into the cfont[] array, below, to determine which
 * font each language element should use.  The last symbol in the list must
 * be PUNCT, because the declaration of cfont[] depends on this.
 */
typedef enum { COMMENT, COMMENT2, STRING, CHARACTER, KEYWORD, FUNCTION, VARIABLE, PREP, PREPWORD, OTHER, PUNCT } TOKENTYPE;

/* This array stores the fonts to be used with the TOKENTYPES, above.  It is
 * initialized by the setup() function each time the screen is redrawn,
 * to reflect the values of the "font" options above.
 */
static char cfont[PUNCT + 1];

/* This function returns True if the given word is a C keyword, or False
 * otherwise.
 */
static BOOLEAN iskeyword(word)
	char	*word;	/* pointer to word */
{
	static char *keyword[] =
	{
		"auto", "break", "case", "char", "class", "const", "continue",
		"default", "delete", "do", "double", "else", "enum", "extern",
		"far", "float", "friend", "for", "goto", "if", "inline",
		"int", "long", "near", "new", "operator", "private",
		"protected", "public", "register", "return", "short",
		"signed", "sizeof", "static", "struct", "switch", "this",
		"typedef", "union", "unsigned", "virtual", "void", "volatile",
		"while"
	};
	int	i;

	/* try to find the word in the list of keywords */
	for (i = 0; i < QTY(keyword) && (*keyword[i] != *word || strcmp(keyword[i], word)); i++)
	{
	}

	/* did we find it? */
	return (i < QTY(keyword)) ? True : False;
}


/* start the mode, and allocate dminfo */
static DMINFO *init(win)
	WINDOW	win;
{
	/* if this is the first-ever time a window has been initialized to
	 * this mode, then we have some extra work to do...
	 */
	if (!dmc.mark2col)
	{
		/* Inherit some functions from normal mode. */
		dmc.mark2col = dmnormal.mark2col;
		dmc.move = dmnormal.move;
		dmc.wordmove = dmnormal.wordmove;
		dmc.indent = dmnormal.indent; /* !!! really a good idea? */
		dmc.tagatcursor = dmnormal.tagatcursor;
		dmc.tagload = dmnormal.tagload;
		dmc.tagnext = dmnormal.tagnext;

		/* initialize the mode's global options */
		optpreset(o_commentfont, 'i', OPT_REDRAW);
		optpreset(o_stringfont, 'u', OPT_REDRAW);
		optpreset(o_keywordfont, 'b', OPT_REDRAW);
		optpreset(o_functionfont, 'n', OPT_REDRAW);
		optpreset(o_variablefont, 'n', OPT_REDRAW);
		optpreset(o_prepfont, 'e', OPT_REDRAW);
		optpreset(o_otherfont, 'b', OPT_REDRAW);

		/* if no real window, then we're done! */
		if (!win)
			return NULL;
	}

	/* allocate a single TOKENTYPE pointer */
	return safealloc(1, sizeof(TOKENTYPE));
}

/* end the mode, and free the modeinfo */
static void term(info)
	DMINFO	*info;	/* window-specific information about mode */
{
	safefree(info);
}

/* Choose a line to appear at the top of the screen, and return its mark.
 * Also, initialize the info for the next line.
 */
static MARK setup(top, cursor, bottom, info)
	MARK	top;	/* where the image drawing began last time */
	long	cursor;	/* cursor's offset into buffer */
	MARK	bottom;	/* where the image drawing ended last time */
	DMINFO	*info;	/* window-specific information about mode */
{
	MARK	newtop;
	BOOLEAN	oddquotes;
	CHAR	*cp;
	CHAR	following;
	BOOLEAN	knowstr, knowcom;

	/* copy the values of the "font" options into the cfont[] array */
	cfont[COMMENT] = cfont[COMMENT2] = o_commentfont;
	cfont[STRING] = cfont[CHARACTER] = o_stringfont;
	cfont[KEYWORD] = o_keywordfont;
	cfont[FUNCTION] = o_functionfont;
	cfont[VARIABLE] = o_variablefont;
	cfont[PREP] = cfont[PREPWORD] = o_prepfont;
	cfont[OTHER] = o_otherfont;
	cfont[PUNCT] = 'n';

	/* use the normal mode's setup function to choose the screen top */
	newtop = (*dmnormal.setup)(top, cursor, bottom, info);
	if (!newtop || markoffset(newtop) >= o_bufchars(markbuffer(newtop)))
		return newtop;

	/* The top line could be a continuation of a COMMENT or STRING.
	 * (Other token can't span a newline, so we can ignore them.)
	 * Scan backward for clues about comments or strings.
	 *
	 * This isn't perfect.  To do the job perfectly, we'd need to start
	 * at the top of the buffer, and scan *forward* to the top of the
	 * screen, but that could take far too long.
	 */
	oddquotes = knowstr = knowcom = False;
	*(TOKENTYPE *)info = PUNCT;
	for (following = *scanalloc(&cp, newtop);
	     scanprev(&cp) && (!knowstr || !knowcom);
	     following = *cp)
	{
		if (*cp != '\\' && following == '"' && !knowstr)
		{
			/* a " which isn't preceded by a \ toggles the quote state */
			oddquotes = (BOOLEAN)!oddquotes;
		}
		else if (*cp != '\\' && following == '\n')
		{
			/* strings can span a newline unless preceded by a backslash */
			knowstr = True;
		}
		else if (*cp == '/' && following == '*')
		{
			/* We'll assume that slash-asterisk always starts a
			 * comment (i.e., that it never occurs inside a string).
			 * However, some C++ programmers like to begin comments
			 * with slash-slash and a bunch of asterisks; we need
			 * to watch out for that.
			 */
			knowstr = knowcom = True;
			if (!scanprev(&cp) || *cp != '/')
				*(TOKENTYPE *)info = COMMENT;
		}
		else if (*cp == '*' && following == '/')
		{
			/* We'll assume that asterisk-slash always ends a comment.
			 * (I.e., that it never occurs inside a string.)
			 */
			knowstr = knowcom = True;
		}
		else if (*cp == '/' && following == '/')
		{
			/* We'll assume that slash-slash always indicates a single-
			 * line comment.  (I.e., that it never occurs in a string or
			 * slash-asterisk type comment.
			 */
			knowstr = knowcom = True;
			oddquotes = False;
		}
	}
	scanfree(&cp);

	/* If it isn't a comment, then it might be a string... check oddquotes */
	if (*(TOKENTYPE *)info == PUNCT && oddquotes)
	{
		*(TOKENTYPE *)info = STRING;
	}

	return newtop;
}

/* generate the image of a line, and return the mark of the next line */
static MARK image(w, line, info, draw)
	WINDOW	w;		/* window where drawing will take place */
	MARK	line;		/* line to be drawn */
	DMINFO	*info;		/* window-specific information amount mode */
	void	(*draw)P_((CHAR *p, long qty, _char_ font, long offset));
				/* function for drawing a single character */
{
	int	col;
	CHAR	*cp;
	CHAR	tmpchar;
	long	offset;
	static MARKBUF tmp;
	TOKENTYPE *ttptr;
	char	undec[40];	/* characters of undecided font */
	CHAR	*up;		/* pointer used for scanning chars */
	CHAR	prev, prev2;	/* the preceding two characters */
	BOOLEAN	quote;		/* True after a backslash in STRING or CHARACTER */
	BOOLEAN	anylower;	/* True if a word contains any lowercase letters */
	int	i;

	/* initially, we'll assume we continue the font of the previous line */
	ttptr = (TOKENTYPE *)info;
	quote = False;

	/* for each character in the line... */
	for (prev = ' ', col = 0, offset = markoffset(line), scanalloc(&cp, line);
	     cp && *cp != '\n';
	     prev = *cp, offset++, scannext(&cp))
	{
		/* some characters are handled specially */
		if (*cp == '\t' && !o_list(w))
		{
			/* tab ends any symbol */
			if (*ttptr == KEYWORD || *ttptr == FUNCTION || *ttptr == VARIABLE || *ttptr == OTHER || *ttptr == PREPWORD)
			{
				*ttptr = PUNCT;
			}

			/* display the tab character as a bunch of spaces */
			tmpchar = ' ';
			i = o_tabstop(markbuffer(w->cursor));
			i -= col % i;
			(*draw)(&tmpchar, -i, cfont[*ttptr], offset);
			col += i;
		}
		else if (*cp < ' ' || *cp == 127)
		{
			/* any control character ends any symbol */
			if (*ttptr == KEYWORD || *ttptr == FUNCTION || *ttptr == VARIABLE || *ttptr == OTHER || *ttptr == PREPWORD)
			{
				*ttptr = PUNCT;
			}

			/* control characters */
			tmpchar = '^';
			(*draw)(&tmpchar, -1, cfont[*ttptr], offset);
			tmpchar = *cp ^ 0x40;
			(*draw)(&tmpchar, -1, cfont[*ttptr], offset);
			col += 2;
		}
		else if (col == 0 && *cp == '#' && *ttptr == PUNCT)
						/* preprocessor directive */
		{
			/* output the '#' in prepfont */
			*ttptr = PREP;
			(*draw)(cp, 1, cfont[PREP], offset);
			col++;
		}
		else /* normal printable character */
		{
			/* starting a keyword/function/variable? */
			if (*ttptr == PUNCT && (*cp == '_' || isalpha(*cp)))
			{
				/* collect lowercase letters */
				for (i = 0, scanalloc(&up, marktmp(tmp, markbuffer(line), offset));
				     i < 9 && up && islower(*up);
				     i++, scannext(&up))
				{
					undec[i] = *up;
				}
				undec[i] = '\0';

				/* did we find a keyword? */
				if (i > 0 && (!up || (*up != '_' && !isalnum(*up))) && iskeyword(undec))
				{
					*ttptr = KEYWORD;
				}
				else /* must be function, variable, or other */
				{
					/* if we got any letters at all previously, they were lowercase */
					anylower = (BOOLEAN)(i > 0);

					/* continue on to the end of the word */
					for (prev2 = prev = '\0';
					     up && (*up == '_' || isalnum(*up));
					     prev2 = prev, prev = *up, scannext(&up))
					{
						if (islower(*up))
						{
							anylower = True;
						}
					}

					/* skip any following whitespace */
					for (; up && *up == ' '; scannext(&up))
					{
					}

					/* is the word followed by a '(' ? */
					if (up && *up == '(')
					{
						*ttptr = FUNCTION;
					}
					else
					/* does the word end with "_t", or
					 * is it uppercase & more than 2 chars long? */
					if ((prev2 != '\0' && !anylower) || (prev2 == '_' && prev == 't'))
					{
						*ttptr = OTHER;
					}
					else /* plain names are usually variables */
					{
						*ttptr = VARIABLE;
					}
				}
				scanfree(&up);
			}
			else if (*ttptr == PREP && !isspace(*cp))
			{
				*ttptr = PREPWORD;
			}
			else if (*ttptr == PREPWORD && !isalnum(*cp))
			{
				*ttptr = PUNCT;
			}
			else if (!isalnum(*cp) && *cp != '_' && (*ttptr == FUNCTION || *ttptr == VARIABLE || *ttptr == KEYWORD || *ttptr == OTHER))
			{
				*ttptr = PUNCT;
			}

			/* start of a string? */
			if (*ttptr == PUNCT && *cp == '"')
			{
				*ttptr = STRING;

				/* make sure the initial quote character
				 * isn't going to be mistaken for the
				 * terminating quote character.
				 */
				quote = True;
			}

			/* start of a character literal? */
			if (*ttptr == PUNCT && *cp == '\'')
			{
				*ttptr = CHARACTER;

				/* make sure the initial quote character
				 * isn't going to be mistaken for the
				 * terminating quote character.
				 */
				quote = True;
			}

			/* start of a comment? */
			if (*ttptr == PUNCT && *cp == '/')
			{
				scanalloc(&up, marktmp(tmp, markbuffer(line), offset + 1));
				if (up && *up == '*')
				{
					*ttptr = COMMENT;
				}
				else if (up && *up == '/')
				{
					*ttptr = COMMENT2;
				}
				scanfree(&up);
			}

			/* draw the character */
			(*draw)(cp, 1, cfont[*ttptr], offset);
			col++;

			/* end of a string? */
			if (*ttptr == STRING && *cp == '"' && !quote)
			{
				*ttptr = PUNCT;
			}

			/* end of a character? */
			if (*ttptr == CHARACTER && *cp == '\'' && !quote)
			{
				*ttptr = PUNCT;
			}

			/* end of a comment? */
			if (*ttptr == COMMENT && prev == '*' && *cp == '/')
			{
				*ttptr = PUNCT;
			}

			/* in a STRING or CHARACTER constant, backslash
			 * is used to quote the following character.
			 */
			if (*cp == '\\' && !quote)
			{
				quote = True;
			}
			else
			{
				quote = False;
			}
		}
	}

	/* end the line */
	if (o_list(w))
	{
		tmpchar = '$';
		(*draw)(&tmpchar, -1, 'n', -1);
	}
	tmpchar = '\n';
	(*draw)(&tmpchar, 1, 'n', offset);
	if (cp)
	{
		offset++;
	}
	else
	{
		offset = o_bufchars(markbuffer(w->cursor));
	}

	/* Strings can span a newline if the newline is preceded by a
	 * backslash.  Old-style C comments can span a newline.  Everything
	 * else ends here.
	 */
	if ((*ttptr != STRING || prev != '\\') && *ttptr != COMMENT)
	{
		*ttptr = PUNCT;
	}

	/* clean up & return the MARK of the next line */
	scanfree(&cp);
	return marktmp(tmp, markbuffer(w->cursor), offset);
}

DISPMODE dmc =
{
	"c",
	"C Source",
	True,	/* can optimize */
	True,	/* can use normal wordwrap */
	0,	/* no window options */
	NULL,
	QTY(globdesc),
	globdesc,
	globval,
	init,
	term,
	NULL,	/* init() sets this to be identical to dmnormal's mark2col() */
	NULL,	/* init() sets this to be identical to dmnormal's move() */
	NULL,	/* init() sets this to be identical to dmnormal's moveword() */
	setup,
	image,
	NULL,	/* doesn't need a header */
	NULL	/* init() sets this to be identical to dmnormal's indent() */
};
#endif /* DISPLAY_C */
