;**************************************************************************
;*
;* Boot-ROM-Code to load an operating system across a TCP/IP network.
;*
;* Module:  process.asm
;* Purpose: Process management functions for DOS simulator
;* Entries: dos00, dos31, dos4C, dos4D, loadprog
;*
;**************************************************************************
;*
;* Copyright (C) 1995,1996 Gero Kuhlmann <gero@gkminix.han.de>
;*
;*  This program is free software; you can redistribute it and/or modify
;*  it under the terms of the GNU General Public License as published by
;*  the Free Software Foundation; either version 2 of the License, or
;*  any later version.
;*
;*  This program is distributed in the hope that it will be useful,
;*  but WITHOUT ANY WARRANTY; without even the implied warranty of
;*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;*  GNU General Public License for more details.
;*
;*  You should have received a copy of the GNU General Public License
;*  along with this program; if not, write to the Free Software
;*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;*


;
;**************************************************************************
;
; Include assembler macros:
;
include ..\..\headers\asm\macros.inc
include ..\..\headers\asm\layout.inc
include ..\..\headers\asm\memory.inc
include .\dospriv.inc


;
;**************************************************************************
;
; Data segment
;
data_start

		public	curr_psp

		extrn	dos_active:byte	; DOS active flag

ret_code	dw	0		; program return code
curr_psp	dw	0		; current PSP
psp_tmpl	psp	<>		; PSP template

data_end


;
;**************************************************************************
;
; Start code segment.
;
text_start

	public	dos00, dos31, dos4C		; define entry poimts
	public	dos4C, dos4D
	public	loadprog

	extrn	dos48:near
	extrn	dos4A:near
	extrn	freeall:near
	extrn	checkmcb:near
	extrn	dosfatal:near


;
;**************************************************************************
;
; Terminate program.
; Input:  none
; Output: none (function does not return)
; Registers changed: all
;
dos00		proc	near

	les	bx,old_stack[bp]
	mov	dx,es:[bx + 2]		; load caller's code segment
	mov	curr_psp,dx
	xor	al,al
	jmp	short dos4C

dos00		endp


;
;**************************************************************************
;
; Terminate current process, but keep it resident in memory.
; Input:  AL  -  return code
;         DX  -  number of paragraphs to keep in memory
; Output: none (function does not return)
; Registers changed: all
;
dos31		proc	near

	mov	ah,3
	mov	ret_code,ax		; set return code
	mov	ax,curr_psp
	or	ax,ax			; is there a program running at all?
	jnz	short dos311
	mov	ax,FATAL_NOPROC		; nope, can't continue
	jmp	short dos315

dos311:	mov	bx,dx
	mov	dx,ax
	dec	ax			; compute MCB address
	mov	es,ax
	call	checkmcb		; check if MCB is valid
	jc	short dos313

dos312:	mov	word ptr es:[mcb_owner],0FFFFh	; this saves the block from
	call	freeall				; being freed
	jnc	short dos314
dos313:	mov	ax,FATAL_MEMORY			; unable to continue
dos315:	jmp	dosfatal

dos314:	mov	word ptr es:[mcb_owner],dx	; restore PSP address in MCB
	mov	es,dx
	call	dos4A				; resize the memory block
	jc	short dos313
	jmp	short terminate

dos31		endp


;
;**************************************************************************
;
; Terminate current program.
; Input:  AL  -  return code
; Output: none (function does not return)
; Registers changed: all
;
dos4C		proc	near

	xor	ah,ah
	mov	ret_code,ax		; set return code
	mov	ax,curr_psp
	or	ax,ax			; is there a program running at all?
	jnz	short dos4C1
	mov	ax,FATAL_NOPROC		; nope, can't continue
	jmp	short dos4C4

dos4C1:	mov	dx,ax
	dec	ax			; compute MCB address
	mov	es,ax
	call	checkmcb		; check if MCB is valid
	jnc	short dos4C2
dos4C3:	mov	ax,FATAL_MEMORY		; unable to continue
dos4C4:	jmp	dosfatal

dos4C2:	call	freeall			; free all memory used by this process
	jc	short dos4C3		; at least one block (PSP) has to be
	jcxz	short dos4C3		; freed
	jmp	short terminate

dos4C		endp


;
;**************************************************************************
;
; Get exit code.
; Input:  none
; Output: AL  -  exit code
;         AH  -  exit cause
; Registers changed: AX
;
dos4D		proc	near

	mov	ax,ret_code
	ret

dos4D		endp


;
;**************************************************************************
;
; Terminate current process. This routine does the actual job, except
; memory management.
; Input:  none
; Output: none (routine does not return)
; Registers changed: all
;
terminate	proc	near

; Restore the stack of the parent process, reset the PSP of the
; parent process, reset the DOS active flag, and return to the parent
; calling address.

	cli
	mov	es,curr_psp
	mov	ax,es:[psp_parent]	; reset current PSP
	mov	curr_psp,ax
	mov	dos_active,0		; reset DOS active flag
	mov	ss,word ptr es:[psp_oldstack+2]
	mov	sp,word ptr es:[psp_oldstack+0]
	retf				; return to loadprog

terminate	endp


;
;**************************************************************************
;
; Start a new process. This routine gets called by the DOS initialization
; code, which passes a pointer to process image and to the command line.
; Note that both have to reside outside of the memory space which is used
; and maintained by the DOS simulator. Also note, that the simulator can
; only handle COM-type programs, not EXE-type programs.
; The program image must begin with a magic cookie followed by it's
; size in paragraphs.
; Input:  ES:DI  -  pointer to program code image
;         ES:SI  -  pointer to command line
; Output: none
; Registers changed: AX, BX, CX, DX
;
loadprog	proc	near

	cld
	push	es
	push	si
	push	di

; First check that the program code image has the correct magic cookie
; at it's beginning.

	cmp	word ptr es:[di],COM_MAGIC
	mov	ax,FATAL_PROG
	jne	short load9

; Next allocate 64 kB of memory for the program, or as much as possible
; if 64 kB are not available. Then check if the program will fit into
; that memory space together with the PSP and some safety margin for
; the stack.

	mov	bx,1000h		; try to acquire 64 kB of memory
	call	dos48
	jnc	short load1
	cmp	ax,ERR_NOMEM
	jne	short load8		; couldn't get that much, so try
	call	dos48			; to get as much as possible
	jc	short load8
load1:	sub	bx,COM_MIN_FREE
	jbe	short load7		; check if the program will fit into
	cmp	es:[di+COM_SIZE_OFS],bx	; the reserved memory area
	jb	short load2
load7:	mov	ax,FATAL_NOMEM
	jmp	short load9
load8:	mov	ax,FATAL_MEMORY
load9:	jmp	dosfatal

; Copy the program code into the new memory block

load2:	push	si
	push	ds
	mov	bx,es			; preserve original ES in BX
	mov	es,ax
	mov	ds,bx
	assume	ds:nothing
	assume	es:nothing
	mov	si,di
	mov	ax,[si+COM_SIZE_OFS]	; size is in paragraphs, convert it
	shift	shl,ax,3		; to number of words
	mov	cx,ax
	lea	si,[si+COM_CODE_OFS]	; copy program code image into new
	mov	di,PSP_SIZE		; place
	rep	movsw

	shift	shr,di,4		; compute segment of first usable
	inc	di			; address after the program end
	mov	dx,es
	add	dx,di

; Copy the PSP template into the new memory block and setup with some
; miscellaneous values.

	xor	ax,ax
	xor	di,di
	mov	cx,PSP_SIZE/2		; first clear the whole PSP area
	rep	stosw

	pop	ds
	assume	ds:dgroup
	xor	di,di
	mov	si,offset dgroup:psp_tmpl
	mov	cx,size psp_tmpl
	rep	movsb			; copy PSP template into memory area

	mov	es:psp_seghndl,es
	mov	ax,curr_psp
	mov	es:psp_parent,ax
	mov	es:psp_endmem,dx

; Next copy the command line into the PSP. The FCB's remain empty because we
; don't use FCB's anyway.

	pop	si
	push	ds
	mov	ds,bx
	assume	ds:nothing
	or	si,si
	jz	short load4
ifdef IS386
	movzx	cx,byte ptr ds:[si]	; save size of command line
else
	mov	cl,ds:[si]		; save size of command line
	xor	ch,ch
endif
	jcxz	short load4
	cmp	cx,PSP_MAXCMDL
	jbe	short load3
	mov	cx,PSP_MAXCMDL
load3:	mov	es:[psp_cmdlsize],cl
	inc	si
	mov	di,psp_cmdl
	rep	movsb			; save command line
	mov	al,0Dh			; terminate command line with CR
	stosb
	xor	al,al			; and with a zero byte
	stosb
load4:	pop	ds
	assume	ds:dgroup

; Now prepare the stack for the return from the called program. This involves
; pushing all relevant registers and a far return address.

	cli
	push	bp
	push	ds
	push	cs
ifdef IS186
	push	offset _text:loadr	; push return address
else
	mov	ax,offset _text:loadr
	push	ax			; push return address
endif
	mov	word ptr es:[psp_oldstack + 0],sp
	mov	word ptr es:[psp_oldstack + 2],ss

; The PSP has been setup. Now set the owner of the program's memory
; block and switch the stack to the new program.

	mov	ax,es
	mov	curr_psp,ax		; save new PSP pointer
	dec	ax
	mov	ds,ax
	assume	ds:nothing		; get size of memory block from
	mov	ax,ds:[mcb_size]	; MCB
	mov	ds:[mcb_owner],es	; set owner of memory block
	shift	shl,ax,4		; convert paragraph size into byte
	dec	ax			; offset for top of stack
	dec	ax
	mov	dx,es
	mov	ss,dx			; set new stack
	mov	sp,ax
	sti

; Finally make the new program current, and call it with the return pointer
; set correctly on the new stack.

	mov	ds,dx			; set data segment register correctly
	xor	ax,ax
	push	ax			; save near pointer to terminate fctn
	push	es
ifdef IS186
	push	PSP_SIZE
else
	mov	ax,PSP_SIZE		; save pointer to the new program's
	push	ax			; entry point
endif
	retf				; call the new program

; The called program will return here. Simply pop all registers from the
; stack and return to the DOS initialization routine.

loadr:	pop	ds
	pop	bp
	pop	di
	pop	si
	pop	es
	sti
	ret

loadprog	endp


;
;**************************************************************************
;
text_end

	end

