/*
 **************************************************************************
 *
 * Boot-ROM-Code to load an operating system across a TCP/IP network.
 *
 * Module:  tftp.c
 * Purpose: Get a file with TFTP protocol
 * Entries: tftp_open, tftp_get
 *
 **************************************************************************
 *
 * Copyright (C) 1995,1996 Gero Kuhlmann <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "../../headers/general.h"
#include "../public/net.h"
#include "../public/arpa.h"
#include "../public/romlib.h"
#include "./arpapriv.h"
#include "./tftp.h"



/*
 **************************************************************************
 * 
 * Error codes private to this file:
 */
#define ERR_INV		-1		/* invalid packet size		*/
#define ERR_ERR		-2		/* error packet received	*/
#define ERR_OP		-3		/* invalid opcode		*/
#define ERR_BLOCK	-4		/* invalid block number		*/
#define ERR_TIMEOUT	-5		/* timeout while receiving	*/



/*
 **************************************************************************
 * 
 * Global variables:
 */
static struct tftphdr *inpbuf;		/* TFTP input buffer		*/
static struct tftphdr *outbuf;		/* TFTP output buffer		*/
static int tid;				/* Transaction ID		*/
static int currblock;			/* Current data block		*/
static int ibuflen;			/* Size of data in input buffer	*/
static int isopen;			/* TRUE if connection is open	*/



/*
 **************************************************************************
 * 
 * Send a tftp request packet
 */
static void send_req(int request, unsigned char *fname, int fnamlen)
{
  unsigned char *cp;
  char *octet = OCTET_STR;
  int len;

  /*
   * The output buffer is setup with the request code, the file name,
   * and the name of the data format.
   */
  memset(outbuf, 0, sizeof(struct tftphdr));
  outbuf->th_op = htons(request);
  len = SEGSIZE - sizeof(outbuf->th_op) - sizeof(OCTET_STR) - 2;
  cp = (unsigned char *)&(outbuf->th_block);
  for (; *fname && len > 0 && fnamlen > 0; len--, fnamlen--)
	*cp++ = *fname++;
  *cp++ = '\0';
  for (; *octet; )
	*cp++ = *((unsigned char *)octet++);
  *cp++ = '\0';

  /* Finally send the request */
  len = (int)(cp - (unsigned char *)outbuf);
  (void)udp_write((char *)outbuf, len);
}



/*
 **************************************************************************
 * 
 * Send a tftp acknowledge packet
 */
static void send_ack(int block)
{
  int len = (int)(outbuf->th_data - (unsigned char *)outbuf);

  memset(outbuf, 0, sizeof(struct tftphdr));
  outbuf->th_op = htons(TFTP_ACK);
  outbuf->th_block = htons(block);
  (void)udp_write((char *)outbuf, len);
}



/*
 **************************************************************************
 * 
 * Receive a TFTP data packet
 */
static int rcv_packet(int block)
{
  int len = 0;

  /* Read packet with timeout */
  len = udp_read((char *)inpbuf, sizeof(struct tftphdr), TFTP_TIMEOUT, CHR_ESC);
  if (len == 0) {
	printf("TFTP: Timeout\n");
	return(ERR_TIMEOUT);
  } else if (len < 0) {
	printf("TFTP: Aborted\n");
	return(ERR_ERR);
  }

  /* Check that the packet has a correct length */
  len -= (int)(inpbuf->th_data - (unsigned char *)inpbuf);
  if (len < 0 || len > SEGSIZE) {
	printf("TFTP: Invalid packet\n");
	return(ERR_INV);
  }

  /* Check if we got an error packet */
  if (ntohs(inpbuf->th_op) == TFTP_ERROR) {
	printf("TFTP: Error %d: %s\n",
				(int)ntohs(inpbuf->th_error), inpbuf->th_data);
	return(ERR_ERR);
  }

  /* Check if we got a valid data packet at all */
  if (ntohs(inpbuf->th_op) != TFTP_DATA) {
	printf("TFTP: Invalid opcode\n");
	return(ERR_OP);
  }

  /* Check if the block number of the data packet is correct */
  if (ntohs(inpbuf->th_block) != block) {
	printf("TFTP: Block %d != %d\n", (int)ntohs(inpbuf->th_block), block);
	return(ERR_BLOCK);
  }

  return(len);
}



/*
 **************************************************************************
 * 
 * Open a TFTP connection
 */
char *tftp_open(t_ipaddr server, unsigned char *fname, int fnamlen)
{
  int retry;

  /* Try to open a TFTP connection to the server */
  currblock = 0;
  for (retry = 0; retry < TFTP_RETRY; retry++) {
	/*
	 * First open a new socket. The local port will be the same as the
	 * transaction ID, which should be a random number and different
	 * between each retry.
	 */
	tid = TFTP_C_PORT + (random() & 0x03f) + ((retry * 64) & 0x1c0);
	if (!udp_open(server, tid, TFTP_S_PORT)) {
		printf("TFTP: ARP timeout\n");
		break;
	}

	/*
	 * Send the file request block, and then wait for the first data
	 * block. If there is no response to the query, retry it with
	 * another transaction ID, so that all old packets get discarded
	 * automatically.
	 */
	send_req(TFTP_RRQ, fname, fnamlen);
	if ((ibuflen = rcv_packet(1)) >= 0) {
		isopen = TRUE;
		(void)send_ack(1);
		return((char *)inpbuf->th_data);
	}

	/* If an error occurred, retries are useless */
	if (ibuflen == ERR_ERR)
		break;
  }

  return(NULL);
}



/*
 **************************************************************************
 *
 * Read the next data packet from a TFTP connection
 */
int tftp_get(void)
{
  int retry;

  /* Don't do anything if no TFTP connection is active. */
  if (!isopen)
	return(0);

  /*
   * If the block number is 0 then we are still dealing with the first
   * data block after opening a connection. If the data size is smaller
   * than SEGSIZE just close the connection again.
   */
  if (currblock == 0) {
	currblock++;
	if (ibuflen < SEGSIZE)
		isopen = FALSE;
	return(ibuflen);
  }

  /*
   * Wait for the next data packet. If no data packet is coming in,
   * resend the ACK for the last packet to restart the sender. Maybe
   * he didn't get our first ACK.
   */
  for (retry = 0; retry < TFTP_RETRY; retry++) {
	if ((ibuflen = rcv_packet(currblock + 1)) >= 0) {
		currblock++;
		send_ack(currblock);
		if (ibuflen < SEGSIZE)
			isopen = FALSE;
		return(ibuflen);
	}
	if (ibuflen == ERR_ERR)
		break;
	else
		send_ack(currblock);
  }

  isopen = FALSE;
  return(-1);
}



/*
 **************************************************************************
 * 
 * Initialize TFTP protocol.
 */
int init_tftp(void)
{
  /* Set name of module for error messages */
  arpa_module_name = "tftp";

  /* Assign a new set of buffers */
  if ((inpbuf = (struct tftphdr *)malloc(sizeof(struct tftphdr) * 2)) == NULL)
	return(FALSE);
  outbuf = (struct tftphdr *)((char *)inpbuf + sizeof(struct tftphdr));
  return(TRUE);
}
