
/* Solarize 1.0 -- image filter plug-in for the GIMP image manipulation program
 * Copyright (C) 1996 Ian Tester
 *
 * You can contact me at 94024831@postoffice.csu.edu.au
 * You can contact the original The GIMP authors at gimp@xcf.berkeley.edu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* This plug-in applies the "solarize" effect.
   All intensities are doubled. Any values above 255 are made to 
   "reverse" back down to 0.
   i.e   0->127 gets mapped to   0->255
       128->255 gets mapped to 255->0

    or      /\
           /  \  out
            in

   supports indexed images (i.e it only has to do the palette!)
   any changes don't "undo" on indexed images though. strange

*/

#include <stdlib.h>
#include <stdio.h>
#include "gimp.h"

/* Declare a local function.
 */

static void solarize_image (Image, Image);
static void solarize_indexed (Image, Image);

static char *prog_name;

int main (int argc, char **argv)
{
  Image input, output;

  /* Save the program name so we can use it later in reporting errors
   */
  prog_name = argv[0];

  /* Call 'gimp_init' to initialize this filter.
   * 'gimp_init' makes sure that the filter was properly called and
   *  it opens pipes for reading and writing.
   */
  if (gimp_init (argc, argv))
    {
      /* This is a regular filter. What that means is that it operates
       *  on the input image. Output is put into the ouput image. The
       *  filter should not worry, or even care where these images come
       *  from. The only guarantee is that they are the same size and
       *  depth.
       */
      input = gimp_get_input_image (0);
      output = gimp_get_output_image (0);

      /* If both an input and output image were available, then do some
       *  work. (Blur). Then update the output image.
       */
      if (input && output)
	{
	  if ((gimp_image_type (input) == RGB_IMAGE) ||
	      (gimp_image_type (input) == GRAY_IMAGE))
	    {
	      gimp_init_progress ("Solarize");
	      solarize_image (input, output);
	      gimp_update_image (output);
	    }
	  else
	    {
	      gimp_init_progress ("Solarize");
	      solarize_indexed (input, output);
	      gimp_update_image (output);
	    }
	}
      
      /* Free both images.
       */
      gimp_free_image (input);
      gimp_free_image (output);

      /* Quit
       */
      gimp_quit ();
    }

  return 0;
}

static void solarize_image (Image input, Image output)
{
  long width, height;
  long channels, rowstride;
  long val;
  unsigned char *src,*dest;
  short row, col;
  int x1,y1,x2,y2;

  gimp_image_area(input,&x1,&y1,&x2,&y2);

  /* Get the size of the input image. (This will/must be the same
   *  as the size of the output image.
   */
  width = gimp_image_width (input);
  height = gimp_image_height (input);
  channels = gimp_image_channels (input);
  rowstride = width * channels;

  x1 *= channels;
  x2 *= channels;

  src = gimp_image_data (input);
  dest = gimp_image_data (output);

  for (row = y1; row < y2; row++)
    {
      for (col = x1; col < x2; col++)
	{
	  val = (long) src[col];
	  val *= 2;
	  if (val < 256)
	    *dest++ = val;
          else
	    *dest++ = (511 - val);
	}

      src += rowstride;

      if ((row % 5) == 0)
	gimp_do_progress (row, y2 - y1);
    }
  
}

static void solarize_indexed(Image input, Image output)
{
 unsigned char *icmap,*ocmap,*in,*out;
 int num,c,val;

/* first copy the image data over */

 memcpy(gimp_image_data(output), gimp_image_data(input), \
        gimp_image_width(input) * gimp_image_height(input));

 num = gimp_image_colors(input) * 3;

 icmap = gimp_image_cmap(input);

 ocmap = (unsigned char *)malloc(num);

 in = icmap;
 out = ocmap;

/*
 gimp_set_image_colors(output,ocmap,num);

 printf("solarizing a palette with %d entries...\n",num / 3);
*/

 for (c = 0; c < num; c++)
 {
  val = (int)*in++;

  val *= 2;
  if (val > 255)
   val = (511 - val);

  *out++ = (unsigned char)val;
 }

 gimp_set_image_colors(output,ocmap,num);

}
