;;
;; hyperlatex.el  v 1.3
;;
;; A common input format for LaTeX and Html documents
;; This file realizes the translation to Html format.
;;
;; $Modified: Wednesday, June 28, 1995 by otfried $
;;
;; This file is part of Hyperlatex
;; Copyright (C) 1994 Otfried Schwarzkopf	
;;  
;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or (at
;; your option) any later version.
;;      
;; This program is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.
;;     
;; A copy of the GNU General Public License is available on the World
;; Wide web at "http://www.cs.ruu.nl/people/otfried/txt/copying.txt".
;; You can also obtain it by writing to the Free Software Foundation,
;; Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;;
;; for site INSTALLATION, you might want to change
;;  `hyperlatex-button-url'
;; Hyperlatex looks for bitmaps for the page puttons at an url
;; which is obtained by concatenating this with the strings
;; "previous.xbm", "up.xbm", "next.xbm"
;;
;; You can also change hyperlatex-known-document-styles
;;

(defvar hyperlatex-known-document-styles
      '(hyperlatex 11pt 12pt twoside titlepage a4 ipe CCR amssymb
		   9pt 8pt 7pt Otwo Z))
;;(defvar hyperlatex-button-url "http://www.cs.ruu.nl/icons/latex2html/")

;; Debian changes
;; Local icons
(defvar hyperlatex-button-url "file:///usr/doc/examples/hyperlatex/")
;; World wide icons (change this and run make afterwards)
;;(defvar hyperlatex-button-url "http://www.yoursite/icons/latex2html/")

(defvar hyperlatex-default-depth 4)

(defvar hyperlatex-index nil)
(defvar hyperlatex-labels nil)
(defvar hyperlatex-sections nil)
(defvar hyperlatex-new-commands nil)

;; The list of accents was taken from
;; http://info.cern.ch/hypertext/WWW/MarkUp/ISOlat1.html

(defvar hyperlatex-html-accents
  '(( "'A" "Aacute"	)
    ( "^A" "Acirc"	)
    ( "`A" "Agrave"	)
    ( "~A" "Atilde"	)
    ( "\"A" "Auml"	)
    ( "'E" "Eacute"	)
    ( "^E" "Ecirc"	)
    ( "`E" "Egrave"	)
    ( "\"E" "Euml"	)
    ( "'I" "Iacute"	)
    ( "^I" "Icirc"	)
    ( "`I" "Igrave"	)
    ( "\"I" "Iuml"	)
    ( "~N" "Ntilde"	)
    ( "'O" "Oacute"	)
    ( "^O" "Ocirc"	)
    ( "`O" "Ograve"	)
    ( "~O" "Otilde"	)
    ( "\"O" "Ouml"	)
    ( "'U" "Uacute"	)
    ( "^U" "Ucirc"	)
    ( "`U" "Ugrave"	)
    ( "\"U" "Uuml"	)
    ( "'Y" "Yacute"	)
    ( "'a" "aacute"	)
    ( "^a" "acirc"	)
    ( "`a" "agrave"	)
    ( "~a" "atilde"	)
    ( "\"a" "auml"	)
    ( "'e" "eacute"	)
    ( "^e" "ecirc"	)
    ( "`e" "egrave"	)
    ( "\"e" "euml"	)
    ( "'\\i" "iacute"	)
    ( "^\\i" "icirc"	)
    ( "`\\i" "igrave"	)
    ( "\"\\i" "iuml"	)
    ( "~n" "ntilde"	)
    ( "'o" "oacute"	)
    ( "^o" "ocirc"	)
    ( "`o" "ograve"	)
    ( "~o" "otilde"	)
    ( "\"o" "ouml"	)
    ( "'u" "uacute"	)
    ( "^u" "ucirc"	)
    ( "`u" "ugrave"	)
    ( "\"u" "uuml"	)
    ( "'y" "yacute"	)
    ( "\"y" "yuml"	)))

(defvar hyperlatex-format-syntax-table nil)

;;;
;;; Hmm. I will have to think about the syntax table
;;;
(progn
  (setq hyperlatex-format-syntax-table (copy-syntax-table))
  (modify-syntax-entry ?\\ "\\" hyperlatex-format-syntax-table)
  
  (modify-syntax-entry ?{ "(}" hyperlatex-format-syntax-table)
  (modify-syntax-entry ?} "){" hyperlatex-format-syntax-table)
  (modify-syntax-entry ?\[ "_" hyperlatex-format-syntax-table)
  (modify-syntax-entry ?\] "_" hyperlatex-format-syntax-table)
  
  (modify-syntax-entry ?!  "." hyperlatex-format-syntax-table)
  (modify-syntax-entry ?#  "_" hyperlatex-format-syntax-table)
  (modify-syntax-entry ?$  "_" hyperlatex-format-syntax-table)
  (modify-syntax-entry ?%  "_" hyperlatex-format-syntax-table)
  (modify-syntax-entry ?&  "_" hyperlatex-format-syntax-table)
  (modify-syntax-entry ?.  "." hyperlatex-format-syntax-table)

  (modify-syntax-entry ?0  "_" hyperlatex-format-syntax-table)
  (modify-syntax-entry ?1  "_" hyperlatex-format-syntax-table)
  (modify-syntax-entry ?2  "_" hyperlatex-format-syntax-table)
  (modify-syntax-entry ?3  "_" hyperlatex-format-syntax-table)
  (modify-syntax-entry ?4  "_" hyperlatex-format-syntax-table)
  (modify-syntax-entry ?5  "_" hyperlatex-format-syntax-table)
  (modify-syntax-entry ?6  "_" hyperlatex-format-syntax-table)
  (modify-syntax-entry ?7  "_" hyperlatex-format-syntax-table)
  (modify-syntax-entry ?8  "_" hyperlatex-format-syntax-table)
  (modify-syntax-entry ?9  "_" hyperlatex-format-syntax-table)

  (modify-syntax-entry ?:  "." hyperlatex-format-syntax-table)
  (modify-syntax-entry ?@  "_" hyperlatex-format-syntax-table)
  (modify-syntax-entry ?\" " " hyperlatex-format-syntax-table)
  (modify-syntax-entry ?\' " " hyperlatex-format-syntax-table)

  (modify-syntax-entry ?\( " " hyperlatex-format-syntax-table)
  (modify-syntax-entry ?\) " " hyperlatex-format-syntax-table)
)

(defun hyperlatex-format-buffer ()
  "Process the current buffer as hyperlatex code, into a Html document.
The Html file output is generated in a directory specified in the
 \\sethtmldirectory command."
  (interactive)
  (let ((lastmessage "Formatting Html file..."))
    (message lastmessage)
    (hyperlatex-format-buffer-1)
    (message "%s done." lastmessage)))

(defun hyperlatex-format-buffer-1 ()
  (let (hyperlatex-command-start
	hyperlatex-command-name
	hyperlatex-stack
	hyperlatex-directory
	hyperlatex-basename
	hyperlatex-title
	hyperlatex-address
	hyperlatex-mathitalics
	;; used to set levels of headings correctly:
	hyperlatex-is-article
	;; the depth of automatic menus, 0 for none
	(hyperlatex-auto-menu 1)
	;; did we create a menu in this section?
	hyperlatex-menu-in-section
	(hyperlatex-max-depth hyperlatex-default-depth)
	(hyperlatex-table-number 0)
	(hyperlatex-figure-number 0)
	(hyperlatex-node-number 0)
	(hyperlatex-sect-number 0)
	(hyperlatex-node-section 0)
	hyperlatex-rev-sections
	(hyperlatex-label-number 0)
	(input-buffer (current-buffer))
	(input-directory default-directory))
    (set-buffer (get-buffer-create "*Hyperlatex Html output*"))
    (fundamental-mode)
    (set-syntax-table hyperlatex-format-syntax-table)
    (erase-buffer)
    (insert-buffer-substring input-buffer)
    (setq case-fold-search nil)
    ;; hyperlatex-run-documentstyle-hooks are first to have a look
    ;; it also figures out whether this is `article' style
    (goto-char (point-min))
    (hyperlatex-run-documentstyle-hooks)
    ;; now figure out parameters
    (goto-char (point-min))
    (re-search-forward "^[ \t]*\\\\begin[ \t]*{document}")
    (beginning-of-line)
    (let ((end (point)))
      ;; first \\htmldirectory
      (goto-char (point-min))
      (re-search-forward "^\\\\htmldirectory")
      (setq hyperlatex-directory (hyperlatex-get-arg-here))
      ;; first \\htmlname
      (goto-char (point-min))
      (re-search-forward "^\\\\htmlname")
      (setq hyperlatex-basename (hyperlatex-get-arg-here))
      (message "Using filename %s/%s.html" hyperlatex-directory
	       hyperlatex-basename)
      ;; \\htmltitle
      (goto-char (point-min))
      (re-search-forward "^\\\\htmltitle")
      (setq hyperlatex-title (hyperlatex-get-arg-here))
      (message "Title of work is %s" hyperlatex-title)
      ;; \\htmldepth
      (goto-char (point-min))
      (if (re-search-forward "^\\\\htmldepth" end t)
	  (progn
	    (setq hyperlatex-max-depth 
		  (string-to-int (hyperlatex-get-arg-here)))
	    (if (<= hyperlatex-max-depth 0)
		(error "Illegal value of \\htmldepth"))
	    (message "Maximum depth for single documents is %d"
		     hyperlatex-max-depth)))
      ;; \\htmlmathitalics
      (goto-char (point-min))
      (if (re-search-forward "^\\\\htmlmathitalics" end t)
	  (setq hyperlatex-mathitalics t))
      ;; \\htmlautomenu
      (goto-char (point-min))
      (if (re-search-forward "^\\\\htmlautomenu" end t)
	  (setq hyperlatex-auto-menu
		(string-to-int (hyperlatex-get-arg-here))))
      ;; \\htmlpanel
      (goto-char (point-min))
      (if (re-search-forward "^\\\\htmlpanel" end t)
	  (setq hyperlatex-make-panel
		(string= (hyperlatex-get-arg-here) "1")))
      ;; \\htmlbuttonurl --- this is undocumented
      (goto-char (point-min))
      (if (re-search-forward "^\\\\htmlbuttonurl" end t)
	  (setq hyperlatex-button-url (hyperlatex-get-arg-here)))
      ;; \\htmladdress
      (goto-char (point-min))
      (if (re-search-forward "^\\\\htmladdress" end t)
	  (progn
	    (setq hyperlatex-address (hyperlatex-get-arg-here))
	    (setq hyperlatex-address
		  (hyperlatex-format-expand-string hyperlatex-address))))
      ;; set and clear
      (goto-char (point-min))
      (while (re-search-forward "^\\(\\\\H[ \t]*\\)?\\\\set[ \t]*{" end t)
	(setq hyperlatex-command-start (1- (point)))
	(hyperlatex-set))
      (goto-char (point-min))
      (while (re-search-forward "^\\(\\\\H[ \t]*\\)?\\\\clear[ \t]*{" end t)
	(setq hyperlatex-command-start (1- (point)))
	(hyperlatex-clear))
      ;; newcommand and newenvironment
      (goto-char (point-min))
      (while (re-search-forward "^\\(\\\\H[ \t]*\\)?\\\\newcommand" end t)
	(setq hyperlatex-command-start (point))
	(let ((name (hyperlatex-parse-required-argument))
	      (nbargs (hyperlatex-parse-optional-argument))
	      (expansion (hyperlatex-parse-required-argument)))
	  (if nbargs
	      (error "Cannot handle new commands with arguments"))
	  (setq hyperlatex-new-commands
		(cons (cons (cons (substring name 1) 'hyperlatex-format)
			    expansion)
		      hyperlatex-new-commands))))
      (goto-char (point-min))
      (while (re-search-forward "^\\(\\\\H[ \t]*\\)?\\\\newenvironment" end t)
	(setq hyperlatex-command-start (point))
	(let ((name (hyperlatex-parse-required-argument))`
	      (nbargs (hyperlatex-parse-optional-argument))
	      (beginexp (hyperlatex-parse-required-argument))
	      (endexp   (hyperlatex-parse-required-argument)))
	  (if nbargs
	      (error "Cannot handle new environments with arguments"))
	  (setq hyperlatex-new-commands
		(cons (cons (cons name 'hyperlatex-format) beginexp)
		      (cons (cons (cons name 'hyperlatex-end) endexp)
			    hyperlatex-new-commands))))))
    ;; Remove \end{document} and what follows 
    (goto-char (point-max))
    (if (search-backward "\\end{document}" nil t)
	(delete-region (point) (point-max))
      (error "Missing \\end{document}"))
    ;; Make sure buffer ends in a newline.
    (or (= (preceding-char) ?\n)
	(insert "\n"))
    ;; Scan the buffer, start from \\topnode
    (goto-char (point-min))
    (re-search-forward "^[ \t]*\\\\topnode[ \t]*{")
    (delete-region (point-min) (1- (match-beginning 0)))
    ;; init indices
    (setq hyperlatex-index nil)
    (setq hyperlatex-labels nil)
    (setq hyperlatex-sections nil)
    ;; now convert
    (hyperlatex-prelim-substitutions)
    (hyperlatex-format-scan)
    ;; finish last node
    (goto-char (point-max))
    (hyperlatex-finish-node)
    (setq hyperlatex-rev-sections (reverse hyperlatex-sections))
    (hyperlatex-insert-links)
    (hyperlatex-insert-node-links)
    (hyperlatex-insert-menus)
    (hyperlatex-final-substitutions)
    ;; now generate real Html and save nodes 
    (hyperlatex-convert-html)
    (hyperlatex-write-files)))

(defun batch-hyperlatex-format ()
  "Runs  hyperlatex-format-buffer  on the files remaining on the command line.
Must be used only with -batch, and kills emacs on completion.
Each file will be processed even if an error occurred previously.
For example, invoke
  \"emacs -batch -funcall batch-hyperlatex-format $docs/ ~/*.tex\"."
  (if (not noninteractive)
      (error "batch-hyperlatex-format may only be used -batch."))
  (let ((auto-save-default nil)
	(find-file-run-dired nil))
    (let ((error 0)
	  file
	  (files ()))
      (while command-line-args-left
	(setq file (expand-file-name (car command-line-args-left)))
	(cond ((not (file-exists-p file))
	       (message ">> %s does not exist!" file)
	       (setq error 1
		     command-line-args-left (cdr command-line-args-left)))
	      ((file-directory-p file)
	       (setq command-line-args-left
		     (nconc (directory-files file)
			    (cdr command-line-args-left))))
	      (t
	       (setq files (cons file files)
		     command-line-args-left (cdr command-line-args-left)))))
      (while files
	(setq file (car files)
	      files (cdr files))
	(condition-case err
	    (progn
	      (if buffer-file-name (kill-buffer (current-buffer)))
	      (find-file file)
	      (buffer-flush-undo (current-buffer))
	      (message "Hyperlatex formatting %s..." file)
	      (hyperlatex-format-buffer))
	  (error
	   (message ">> Error: %s" (prin1-to-string err))
	   (message ">>  point at")
	   (let ((s (buffer-substring (point)
				      (min (+ (point) 100)
					   (point-max))))
		 (tem 0))
	     (while (setq tem (string-match "\n+" s tem))
	       (setq s (concat (substring s 0 (match-beginning 0))
			       "\n>>  "
			       (substring s (match-end 0)))
		     tem (1+ tem)))
	     (message ">>  %s" s))
	   (setq error 1))))
      (kill-emacs error))))

;;;
;;; ----------------------------------------------------------------------
;;;
;;; These functions actually insert the Html commands and deal with magic.
;;; This is called after all formatting has been done.
;;;

(put 'html	'hyperlatex-format 'hyperlatex-format-html)
(put 'htmlsym	'hyperlatex-format 'hyperlatex-format-htmlsym)
(put 'htmlrule	'hyperlatex-format 'hyperlatex-format-htmlrule)
(put 'htmlimage 'hyperlatex-format 'hyperlatex-format-image)
(put 'gif	'hyperlatex-format 'hyperlatex-format-gif)

(defvar hyperlatex-magic-header	"\M-XHtml:")

(defun hyperlatex-gen (str &optional nopar after)
  "Inserts Html command STR. If optional second argument NOPAR is t,
surrounds it by magic markers that inhibit an automatic \\par before or
after this command. Optional third argument AFTER is inserted behind
the marker after the command."
  (let ((noparstr (if nopar "\M-n" ""))
	(afterstr (if after after "")))
    (insert noparstr "\M-<" str "\M->" noparstr afterstr)))

(defun hyperlatex-gensym (str)
  "Inserts Html command to generate special characters. Use
`(hyperlatex-gensym \"amp\")' to generate `&amp;'."
  (insert "\M-&" str ";"))

(defun hyperlatex-format-html ()
  (let ((arg (hyperlatex-parse-required-argument)))
    (hyperlatex-gen arg)))

(defun hyperlatex-format-htmlsym ()
  (let ((arg (hyperlatex-parse-required-argument)))
    (hyperlatex-gensym arg)))

(defun hyperlatex-format-htmlrule ()
  (let ((oarg (hyperlatex-parse-optional-argument)))
    (if (not (bolp)) (insert "\n"))
    (hyperlatex-gen (if oarg (concat "HR " oarg) "HR") t)))

(defvar hyperlatex-html-alignments
  '( ( "t" "ALIGN=TOP"    )
     ( "c" "ALIGN=MIDDLE" )
     ( "b" "ALIGN=BOTTOM" )
     ( "l" "ALIGN=LEFT"   )
     ( "r" "ALIGN=RIGHT"  )
     ))

(defun hyperlatex-format-image ()
  (let* ((opt (hyperlatex-parse-optional-argument))
	 (reqopt (if opt opt "b"))
	 (align (assoc reqopt hyperlatex-html-alignments))
	 (tags (if align (nth 1 align) reqopt))
	 (url (hyperlatex-parse-required-argument)))
    (hyperlatex-gen (format "IMG SRC=\"%s\" %s" url tags))))

(defun hyperlatex-format-gif ()
  (let* ((opt (hyperlatex-parse-optional-argument))
	 (reqopt (if opt opt "b"))
	 (align (assoc reqopt hyperlatex-html-alignments))
	 (tags (if align (nth 1 align) reqopt))
	 (resolution (hyperlatex-parse-optional-argument))
	 (dpi (hyperlatex-parse-optional-argument))
	 (url (hyperlatex-parse-required-argument)))
    (delete-region hyperlatex-command-start
		   (progn (search-forward "\\end{gif}") (point)))
    (hyperlatex-delete-whitespace)
    (hyperlatex-gen (format "IMG SRC=\"%s.gif\" %s" url tags))))

(defun hyperlatex-convert-html ()
  "Scan buffer and replace the characters special for Html.
Replace PAR entries by \\html{P}, unless there is a magic NOPAR
next to it. Finally, remove or convert all magic entries."
  ;; replace PAR entries by <P>, if okay
  (goto-char (point-min))
  (while (search-forward "\M-p" nil t)
    (replace-match "")
    (if (= (following-char) ?\n)
	(delete-char 1))
    (or (progn
	  (goto-char (match-beginning 0))
	  (skip-chars-backward " \n")
	  (equal (preceding-char) (+ 128 ?n)))
	(progn
	  (goto-char (match-beginning 0))
	  (skip-chars-forward " \n")
	  (equal (following-char) (+ 128 ?n)))
	(progn
	  (goto-char (match-beginning 0))
	  (hyperlatex-gen "P" t))))
  ;; remove magic NOPAR, LABEL
  (goto-char (point-min))
  (while (re-search-forward "[\M-n\M-l]" nil t)
    (replace-match ""))
  ;; fixup &, <, >, ~
  (goto-char (point-min))
  (while (search-forward "~" nil t)
    (replace-match "\M-&#32;"))
  (goto-char (point-min))
  (while (search-forward "&" nil t)
    (replace-match "\M-&amp;"))
  (goto-char (point-min))
  (while (search-forward ">" nil t)
    (replace-match "\M-&gt;"))
  (goto-char (point-min))
  (while (search-forward "<" nil t)
    (replace-match "\M-&lt;"))
  ;; now squeeze out senseless empty lines
  (goto-char (point-min))
  (while (search-forward "\n\n" nil t)
    (forward-char -1)
    (delete-char -1))
  ;; finally, convert the magic chars to the real HTML control sequences
  (goto-char (point-min))
  (while (re-search-forward
	  "[\M-&\M-<\M->\M-%\M-{\M-}\M-~\M- \M--\M-'\M-`]" nil t)
    (replace-match (char-to-string (- (preceding-char) 128))))
  ;; and last, put back ISO characters
  (goto-char (point-min))
  (while (re-search-forward "\M-C{\\([0-9]+\\)}" nil t)
    (replace-match (char-to-string
		    (string-to-int
		     (buffer-substring (match-beginning 1)
				       (match-end 1)))))))
  
(defun hyperlatex-write-files ()
  "Saves the different Html documents in the corresponding files."
  (goto-char (point-max))
  (let ((fin (point))
	(magic-regexp (concat "^" (regexp-quote hyperlatex-magic-header)
			      "\\([^\n]+\\)\n")))
    (while (re-search-backward magic-regexp nil t)
      (let ((fname (concat hyperlatex-directory "/"
			   (buffer-substring (match-beginning 1)
					     (match-end 1))))
	    (beg (match-end 0))
	    (end fin))
	(setq fin (match-beginning 0))
	(write-region beg end fname)))))

;;;
;;; ----------------------------------------------------------------------
;;;
;;; Scanning and Parsing
;;;

(defun hyperlatex-format-scan ()
  "This function formats a Hyperlatex buffer into a Html buffer."
  ;; Scan for \\-commands.
  (goto-char (point-min))
  (let ((foochar nil))
    (while (re-search-forward "[\\%]" nil t)
      (if (= ?% (preceding-char))
	  ;; % comment
	  (progn
	    (forward-char -1)
	    (delete-region (point) (progn (forward-line 1) (point)))
	    (hyperlatex-delete-whitespace))
	;; Otherwise: \command
	;; Handle a few special \-followed-by-one-char commands.
	(setq foochar (following-char))
	(cond ((memq foochar '(?, ?- ?/ ))
	       ;; \\, \\- \\/ are ignored
	       (delete-region (1- (point)) (1+ (point))))
	      ((memq foochar '(?^ ?' ?` ?\" ?~))
	       ;; replace general accents
	       (forward-char 1)
	       (let* ((arg (hyperlatex-get-arg-here t))
		      (match (assoc (concat (char-to-string foochar) arg)
				    hyperlatex-html-accents)))
		 (if (not match) (error "Unknown or unsupported accent"))
		 (delete-char -2)
		 (hyperlatex-gensym (nth 1 match))))
	      ((memq foochar '(?\{ ?}))
	       ;; replace \\{ and \\} by magic braces
	       (delete-region (1- (point)) (1+ (point)))
	       (insert (char-to-string (+ foochar 128))))
	      ((memq foochar '(?\{ ?} ?\  ?\. ?% ?_ ?& ))
	       ;; These characters are simply quoted.  Delete the \.
	       (delete-char -1)
	       (forward-char 1))
	      ;; it is a command. parse it
	      (t
	       (setq hyperlatex-command-start (1- (point)))
	       (if (/= (char-syntax foochar) ?w)
		   ;; a single letter command
		   (forward-char 1)
		 ;; \ is followed by a command-word; find the end of the word.
		 (forward-word 1)
		 ;; and delete white space
		 (hyperlatex-delete-whitespace))
	       (setq hyperlatex-command-name
		     (intern (buffer-substring (1+ hyperlatex-command-start)
					       (point))))
	       ;; remove command
	       (delete-region hyperlatex-command-start (point))
	       (let ((cmd (get hyperlatex-command-name 'hyperlatex-format)))
		 (if cmd (funcall cmd)
		   (hyperlatex-unsupported 'hyperlatex-format))))))))
  (cond (hyperlatex-stack
	 (goto-char (nth 2 (car hyperlatex-stack)))
	 (error "Unterminated \\begin{%s}" (car (car hyperlatex-stack))))))

(defun hyperlatex-prelim-substitutions ()
  "Replaces some Tex idioms by magic markers. This is called before formatting
starts."
  (untabify (point-min) (point-max))
  (goto-char (point-min))
  (while (re-search-forward "[\200-\377]" nil t)
    (replace-match (format "\M-C\M-{%d\M-}" (preceding-char))))
  (goto-char (point-min))
  (while (re-search-forward "\n[ ]*\n"  nil t)
    (replace-match "\n\M-p\n")
    (forward-char -1)))

(defun hyperlatex-final-substitutions ()
  "Replaces some LaTeX idioms by Html expressions. We assume that verbatim
environments have been protected before by making them magic."
  ;; replace dashes
  (goto-char (point-min))
  (while (search-forward "---"  nil t)
    (replace-match "--"))
  ;; Convert left and right quotes to typewriter font quotes.
  (goto-char (point-min))
  (while (search-forward "``" nil t)
    (replace-match "\""))
  (goto-char (point-min))
  (while (search-forward "''" nil t)
    (replace-match "\""))
  ;; remove braces
  (goto-char (point-min))
  (while (re-search-forward "[{}]" nil t)
    (replace-match "")))

(defun hyperlatex-format-expand-string (str)
  "Expands the STRING by doing the Hyperlatex to Html conversions.
  This is pretty crazy, and probably will break if you use \\section or
so in the string."
  (save-excursion
    (set-buffer (get-buffer-create "*Hyperlatex String Expansion Buffer"))
    (erase-buffer)
    (insert str)
    (goto-char (point-max))
    (or (= (preceding-char) ?\n)
	(insert "\n"))
    (let (hyperlatex-command-start
	  hyperlatex-command-name
	  hyperlatex-stack)
      (hyperlatex-format-scan))
    (buffer-substring (point-min) (point-max))))

;;;
;;; ----------------------------------------------------------------------
;;;
;;; Structural commands
;;;

(put 'begin	'hyperlatex-format 'hyperlatex-format-begin)
(put 'end	'hyperlatex-format 'hyperlatex-format-end)
(put 'par	'hyperlatex-format 'hyperlatex-format-par)
(put 'C		'hyperlatex-format 'hyperlatex-discard-commment-line)
(put 'T		'hyperlatex-format 'hyperlatex-discard-commment-line)
(put 'H		'hyperlatex-format 'hyperlatex-format-ignore)
(put 'texonly	'hyperlatex-format 'hyperlatex-parse-required-argument)
(put 'htmlonly	'hyperlatex-format 'hyperlatex-insert-required-argument)
(put 'input	'hyperlatex-format  'hyperlatex-format-input)

(defun hyperlatex-format-ignore ()
  "Function that does not do anything.")

(defun hyperlatex-format-par ()
  (hyperlatex-gen "P" t))

(defun hyperlatex-format-begin ()
  (hyperlatex-format-begin-end 'hyperlatex-format))

(defun hyperlatex-format-end ()
  (hyperlatex-format-begin-end 'hyperlatex-end))

(defun hyperlatex-format-begin-end (prop)
  (setq hyperlatex-command-name (intern (hyperlatex-parse-required-argument)))
  (if (eq prop 'hyperlatex-end)
      (hyperlatex-delete-whitespace))
  (setq cmd (get hyperlatex-command-name prop))
  (if cmd (funcall cmd) (hyperlatex-unsupported prop)))

(defun hyperlatex-unsupported (prop)
  "Called for \\commands not defined in Hyperlatex. Looks them up in
`hyperlatex-new-commands' and complains if not found."
  (let ((match (assoc (cons (symbol-name hyperlatex-command-name) prop)
		      hyperlatex-new-commands)))
    (if match
	(save-excursion (insert (cdr match)))
      (error "Unknown command: %s" (symbol-name hyperlatex-command-name)))))

(defun hyperlatex-format-input ()
  (save-excursion
    (let* ((arg (hyperlatex-parse-required-argument))
	   (file-name (cond
		      ((file-readable-p (expand-file-name arg input-directory))
		       (expand-file-name arg input-directory))
		      ((file-readable-p (expand-file-name
					 (concat arg ".tex") input-directory))
		       (expand-file-name (concat arg ".tex") input-directory))
		      (t (error "I can't find the file %s" file-name)))))
      (message "Inserting file %s..." file-name) (sit-for 1)
      (insert-file file-name)
      (exchange-point-and-mark)
      (narrow-to-region hyperlatex-command-start (point))
      (hyperlatex-prelim-substitutions)
      (goto-char (point-min))
      (widen)
      (message "Inserting file %s...done" file-name))))

;;;
;;; ----------------------------------------------------------------------
;;;
;;; Parse arguments to commands
;;;

(defun hyperlatex-get-arg-here (&optional char)
  (setq hyperlatex-command-start (point))
  (hyperlatex-delete-whitespace)
  (hyperlatex-parse-required-argument char))
	
(defun hyperlatex-parse-required-argument (&optional char)
  "Parses the argument enclosed in braces after the commands.
Deletes command and returns argument.

If optional argument CHAR is not nil, also accept a single char."
  (goto-char hyperlatex-command-start)
  (cond ((looking-at "{") (forward-sexp 1))
	(char (forward-char 1))
	(t (error "hyperlatex-parse-required-argument: no argument provided")))
  (prog1
      (buffer-substring (1+ hyperlatex-command-start) (1- (point)))
    (delete-region hyperlatex-command-start (point))))

(defun hyperlatex-parse-optional-argument ()
  "Parses the argument enclosed in brackets after the commands.
Deletes command and returns argument (nil if none)."
  (goto-char hyperlatex-command-start)
  (if (= (following-char) ?\[ )
      (progn
	(goto-char (1+ (point)))
	(while (/= (following-char) ?\])
	  (if (= (following-char) ?\{)
	      (forward-sexp 1)
	    (goto-char (1+ (point)))))
	(prog1
	    (buffer-substring (1+ hyperlatex-command-start) (point))
	  (delete-region hyperlatex-command-start (1+ (point)))))))

(defun hyperlatex-starred-p ()
  "Is current command starred? Remove star, and skip whitespace."
  (cond ((= (following-char) ?*)
	 (delete-char 1)
	 (hyperlatex-delete-whitespace)
	 t)))

(defvar hyperlatex-deleted-newline nil
  "True if 'hyperlatex-delete-whitespace' deleted a newline while skipping
white space.")

(defun hyperlatex-delete-whitespace ()
  (if (assoc 'example hyperlatex-stack)
      ;; if space is active, we should not skip it
      ()
    (let ((beg (point)))
      (skip-chars-forward " \t")
      (setq hyperlatex-deleted-newline 
	    (cond ((looking-at "\n")
		   (forward-char 1)
		   (skip-chars-forward " \t\n")
		   t)))
      (delete-region beg (point)))))

(defun hyperlatex-insert-required-argument ()
  (save-excursion (insert (hyperlatex-parse-required-argument))))

(defun hyperlatex-discard-commment-line ()
  "Discards commment lines, but tries to be more TeX like by deleting 
the following whitespace."
  (goto-char hyperlatex-command-start)
  (if hyperlatex-deleted-newline
      ()
    (delete-region (point) (progn (forward-line 1) (point)))
    (hyperlatex-delete-whitespace)))

;;;
;;; ----------------------------------------------------------------------
;;;
;;; Make sections and nodes
;;;

(put 'topnode		'hyperlatex-format 'hyperlatex-format-topnode)
(put 'chapter		'hyperlatex-format 'hyperlatex-format-chapter)
(put 'section		'hyperlatex-format 'hyperlatex-format-section)
(put 'subsection	'hyperlatex-format 'hyperlatex-format-subsection)
(put 'subsubsection	'hyperlatex-format 'hyperlatex-format-subsubsection)
(put 'paragraph		'hyperlatex-format 'hyperlatex-format-paragraph)
(put 'subparagraph	'hyperlatex-format 'hyperlatex-format-subparagraph)
(put 'xname		'hyperlatex-format 'hyperlatex-format-xname)
(put 'htmlpanel		'hyperlatex-format 'hyperlatex-format-htmlpanel)

(defvar hyperlatex-node-names nil)
(defvar hyperlatex-make-panel t)
(defvar hyperlatex-made-panel nil)

(defun hyperlatex-format-xname ()
  (setq hyperlatex-node-names
	(cons (cons (1+ hyperlatex-node-number)
		    (format "%s.html" (hyperlatex-parse-required-argument)))
	      hyperlatex-node-names)))

(defun hyperlatex-format-htmlpanel ()
  (setq hyperlatex-make-panel
	(string= (hyperlatex-parse-required-argument) "1")))

(defun hyperlatex-format-chapter ()
  (hyperlatex-format-chapter-1 1))

(defun hyperlatex-format-section ()
  (hyperlatex-format-chapter-1 2))

(defun hyperlatex-format-subsection ()
  (hyperlatex-format-chapter-1 3))

(defun hyperlatex-format-subsubsection ()
  (hyperlatex-format-chapter-1 4))

(defun hyperlatex-format-paragraph ()
  (hyperlatex-format-chapter-1 5))

(defun hyperlatex-format-subparagraph ()
  (hyperlatex-format-chapter-1 6))

(defun hyperlatex-make-header (head)
  "Creates header for new node, with filename, title etc."
  (insert "\n" hyperlatex-magic-header
	  (hyperlatex-fullname hyperlatex-node-number) "\n")
  (setq hyperlatex-made-panel hyperlatex-make-panel)
  (hyperlatex-gen (concat "!-- HTML file produced from "
			  (if (buffer-file-name input-buffer)
			      (concat "file: "
				      (file-name-sans-versions
				       (file-name-nondirectory
					(buffer-file-name input-buffer))))
			    (concat "buffer " (buffer-name input-buffer)))
			  "--")
		  nil "\n")
  (hyperlatex-gen "!-- using Hyperlatex v 1.3 (c) Otfried Schwarzkopf --"
		  nil "\n")
  (hyperlatex-gen "HTML")
  (hyperlatex-gen "HEAD")
  (hyperlatex-gen "TITLE")
  (insert hyperlatex-title (if head (concat " -- " head) ""))
  (hyperlatex-gen "/TITLE" nil "\n")
  (if hyperlatex-made-panel
      (insert (format "\M-XLinks{%d}\n" hyperlatex-sect-number)))
  (hyperlatex-gen "/HEAD")
  (hyperlatex-gen "BODY" nil "\n")
  (setq hyperlatex-label-number 1)
;;;  (hyperlatex-drop-label)
  (setq hyperlatex-node-section hyperlatex-sect-number)
  (if hyperlatex-made-panel
      (insert (format "\n\M-XButtonsNames{%d}\n" hyperlatex-sect-number))))

(defun hyperlatex-fullname (node-number)
  (if (zerop node-number)
      (concat hyperlatex-basename ".html")
    (let ((m (assoc node-number hyperlatex-node-names)))
      (if m
	  (cdr m)
	(format "%s_%d.html" hyperlatex-basename node-number)))))

(defun hyperlatex-format-topnode ()
  "Create the Top node of the Html document."
  (let ((arg (hyperlatex-parse-required-argument)))
    (setq hyperlatex-node-number 0)
    (setq hyperlatex-sect-number 0)
    (setq hyperlatex-menu-in-section nil)
    (setq hyperlatex-node-names nil)
    (hyperlatex-make-header nil)
    (setq hyperlatex-sections (cons (list 0 0 "Top" 0 0) hyperlatex-sections))
    (let ((start (point)))
      (hyperlatex-gen "H1" t)
      (insert arg)
      (hyperlatex-gen "/H1" t "\M-l\n")
      (goto-char start))))

(defun hyperlatex-finish-node ()
  "Finish up the previous node."
  ;; insert automatic menu, if desired
  (if (and (not hyperlatex-menu-in-section)
	   (not (zerop hyperlatex-auto-menu)))
      (insert (format "\n\M-XMenu{%d}{%d}\n"
		      hyperlatex-sect-number
		      hyperlatex-auto-menu)))
  ;; insert address, if one exists
  (if hyperlatex-address
      (progn
	(hyperlatex-gen "ADDRESS" t "\n")
	(insert hyperlatex-address "\n")
	(hyperlatex-gen "/ADDRESS" t "\n")))
  ;; and finish with buttons
  (if hyperlatex-made-panel
      (insert (format "\n\M-XButtons{%d}\n" hyperlatex-node-section)))
  (hyperlatex-gen "/BODY")
  (hyperlatex-gen "/HTML" nil "\n"))

(defun hyperlatex-new-node (level head)
  "Finish up the previous node, and start a new node.
Assumes that the command starting the new node has already been removed,
and we are at the beginning of a new line."
  ;; finish up old node
  (hyperlatex-finish-node)
  (setq hyperlatex-node-number (1+ hyperlatex-node-number))
  (setq hyperlatex-sect-number (1+ hyperlatex-sect-number))
  (hyperlatex-make-header head))

;;
;; section headings are ended with a magic `l', so we reuse the section label
;;

(defun hyperlatex-format-chapter-1 (in-level)
  (hyperlatex-starred-p)
  (let* ((optarg (hyperlatex-parse-optional-argument))
	 (reqarg (hyperlatex-parse-required-argument))
	 (level  (if hyperlatex-is-article (1- in-level) in-level))
	 (head   (hyperlatex-format-expand-string (if optarg optarg reqarg)))
	 (new-node (< level hyperlatex-max-depth)))
    (hyperlatex-delete-whitespace)
    ;; So we can see where we are.
    (message "Formatting: %s ... " reqarg)
    ;; if level is high enough, start new node
    (if new-node
	(hyperlatex-new-node level head)
      ;; otherwise add a new label
      (setq hyperlatex-sect-number (1+ hyperlatex-sect-number))
      (setq hyperlatex-label-number (1+ hyperlatex-label-number)))      
    ;; finally, add new heading
    (setq hyperlatex-sections
	  (cons (list hyperlatex-sect-number
		      hyperlatex-node-number
		      head
		      level
		      (1- hyperlatex-label-number))
		hyperlatex-sections))
    (setq hyperlatex-menu-in-section nil)
    (if new-node
	(save-excursion
	  (hyperlatex-gen (format "H%d" level) t reqarg)
	  (hyperlatex-gen (format "/H%d" level) t "\M-l\n"))
      (save-excursion
	(hyperlatex-gen (format "H%d" level) t)
	(insert "\M-<A NAME=\""
		(format "%d" (1- hyperlatex-label-number))
		"\"\M->" reqarg "\M-</A\M->\M-l")
	(hyperlatex-gen (format "/H%d" level) t "\M-l\n")))))
      

;;;
;;; Finally, we will insert buttons and Document links
;;;

(defun hyperlatex-sect-head (sect)
  "Returns heading of SECT, a pointer into either list."
  (nth 2 sect))

(defun hyperlatex-sect-level (sect)
  "Returns level of SECT, a pointer into either list."
  (nth 3 sect))

(defun hyperlatex-sect-node (sect)
  "Returns node number of SECT, a pointer into either list."
  (nth 1 sect))

(defun hyperlatex-sect-num (sect)
  "Returns section number of SECT, a pointer into either list."
  (car sect))

(defun hyperlatex-sect-label (sect)
  "Returns label of SECT, a pointer into either list."
  (nth 4 sect))

(defun hyperlatex-prev-node (sect)
  "Returns the previous node of section number SECT."
  (if (zerop sect)
      ()
    (let ((sp hyperlatex-sections))
      (while (/= (hyperlatex-sect-num (car sp)) sect)
	(setq sp (cdr sp)))
      ;; sp points to section 
      (let ((lev (hyperlatex-sect-level (car sp))))
	(setq sp (cdr sp))
	(while (> (hyperlatex-sect-level (car sp)) lev)
	  (setq sp (cdr sp)))
	;; now sp points at previous section with level equal or higher
	(if (= (hyperlatex-sect-level (car sp)) lev)
	    (car sp)
	  ())))))

(defun hyperlatex-up-node (sect)
  "Returns the up node of section number SECT."
  (if (zerop sect)
      ()
    (let ((sp hyperlatex-sections))
      (while (/= (hyperlatex-sect-num (car sp)) sect)
	(setq sp (cdr sp)))
      ;; sp points to section 
      (let ((lev (hyperlatex-sect-level (car sp))))
	(setq sp (cdr sp))
	(while (and sp (>= (hyperlatex-sect-level (car sp)) lev))
	  (setq sp (cdr sp)))
	;; now sp points at previous section with higher level
	(car sp)))))
  
(defun hyperlatex-next-node (sect)
  "Returns the next node of section number SECT."
  (if (zerop sect)
      (if (zerop hyperlatex-node-number)
	  ()
	(assoc 1 hyperlatex-sections))
    (let ((sp hyperlatex-rev-sections))
      (while (/= (hyperlatex-sect-num (car sp)) sect)
	(setq sp (cdr sp)))
      ;; sp points to section 
      (let ((lev (hyperlatex-sect-level (car sp))))
	(setq sp (cdr sp))
	(while (and sp (> (hyperlatex-sect-level (car sp)) lev))
	  (setq sp (cdr sp)))
	;; now sp points at next section with higher or same level, or is nil
	(if (and sp (= (hyperlatex-sect-level (car sp)) lev))
	    (car sp)
	  ())))))

(defun hyperlatex-insert-buttons (secnum)
  "Inserts buttons for node SECNUM."
  (let ((prev (hyperlatex-prev-node secnum))
	 (next (hyperlatex-next-node secnum))
	 (up   (hyperlatex-up-node secnum)))
    ;; now insert buttons
    (if prev
	(progn
	  (hyperlatex-gen
	   (format "A HREF=\"%s\""
		   (hyperlatex-fullname (hyperlatex-sect-node prev))))
	  (hyperlatex-gen (concat "IMG SRC=\""
				  hyperlatex-button-url
				  "previous.xbm"
				  "\" ALT=\"[Previous]\""))
	  (hyperlatex-gen "/A" nil "\n")))
    (if up
	(progn
	  (hyperlatex-gen
	   (format "A HREF=\"%s\""
		   (hyperlatex-fullname (hyperlatex-sect-node up))))
	  (hyperlatex-gen (concat "IMG SRC=\""
				  hyperlatex-button-url
				  "up.xbm"
				  "\" ALT=\"[Up]\""))
	  (hyperlatex-gen "/A" nil "\n")))
    (if next
	(progn
	  (hyperlatex-gen
	   (format "A HREF=\"%s\""
		   (hyperlatex-fullname (hyperlatex-sect-node next))))
	  (hyperlatex-gen (concat "IMG SRC=\""
				  hyperlatex-button-url
				  "next.xbm"
				  "\" ALT=\"[Next]\""))
	  (hyperlatex-gen "/A" nil "\n")))
    (hyperlatex-gen "BR" nil "\n")))

(defun hyperlatex-insert-named-links (secnum)
  "Inserts named links for node SECNUM."
  (let ((prev (hyperlatex-prev-node secnum))
	 (next (hyperlatex-next-node secnum))
	 (up   (hyperlatex-up-node secnum)))
    ;; now insert links
    (if prev
	(progn
	  (hyperlatex-gen "B" nil "Go backward to ")
	  (hyperlatex-gen
	   (format "A HREF=\"%s\""
		   (hyperlatex-fullname (hyperlatex-sect-node prev))))
	  (insert (hyperlatex-sect-head prev))
	  (hyperlatex-gen "/A")
	  (hyperlatex-gen "/B")
	  (hyperlatex-gen "BR" nil "\n")))
    (if up
	(progn
	  (hyperlatex-gen "B" nil "Go up to ")
	  (hyperlatex-gen
	   (format "A HREF=\"%s\""
		   (hyperlatex-fullname (hyperlatex-sect-node up))))
	  (insert (hyperlatex-sect-head up))
	  (hyperlatex-gen "/A")
	  (hyperlatex-gen "/B")
	  (hyperlatex-gen "BR" nil "\n")))
    (if next
	(progn
	  (hyperlatex-gen "B" nil "Go forward to ")
	  (hyperlatex-gen
	   (format "A HREF=\"%s\""
		   (hyperlatex-fullname (hyperlatex-sect-node next))))
	  (insert (hyperlatex-sect-head next))
	  (hyperlatex-gen "/A")
	  (hyperlatex-gen "/B")
	  (hyperlatex-gen "BR" nil "\n")))
    (if (or prev up next)
	(hyperlatex-gen "HR" nil "\n"))))
      
    
(defun hyperlatex-insert-html-links (secnum)
  "Inserts Html `link' entries for node SECNUM."
  (let ((prev (hyperlatex-prev-node secnum))
	 (next (hyperlatex-next-node secnum))
	 (up   (hyperlatex-up-node secnum)))
    (if prev
	(progn
	  (hyperlatex-gen
	   (format "LINK REV=\"Precedes\" HREF=\"%s\""
		   (hyperlatex-fullname (hyperlatex-sect-node prev)))
	   nil "\n")))
    (if up
	(progn
	  (hyperlatex-gen
	   (format "LINK REV=\"Subdocument\" HREF=\"%s\""
		   (hyperlatex-fullname (hyperlatex-sect-node up)))
    	   nil "\n")))
    (if next
	(progn
	  (hyperlatex-gen
	   (format "LINK REL=\"Precedes\" HREF=\"%s\""
		   (hyperlatex-fullname (hyperlatex-sect-node next)))
	   nil "\n")))))

(defun hyperlatex-insert-node-links ()
  "Inserts buttons and HTML Document Links."
  (message "Inserting buttons ... ")
  (goto-char (point-min))
  (while (search-forward "\M-XButtonsNames" nil t)
    (replace-match "")
    (let ((secnum (string-to-int (hyperlatex-get-arg-here))))
      (hyperlatex-insert-buttons secnum)
      (hyperlatex-insert-named-links secnum)))
  (goto-char (point-min))
  (while (search-forward "\M-XButtons" nil t)
    (replace-match "")
    (let ((secnum (string-to-int (hyperlatex-get-arg-here))))
      (hyperlatex-gen "P" nil "\n")
      (hyperlatex-insert-buttons secnum)))
  (goto-char (point-min))
  (while (search-forward "\M-XLinks" nil t)
    (replace-match "")
    (let ((secnum (string-to-int (hyperlatex-get-arg-here))))
      (hyperlatex-insert-html-links secnum))))

;;;
;;; ----------------------------------------------------------------------
;;;
;;; Make menus
;;;

(put 'htmlmenu	'hyperlatex-format 'hyperlatex-format-makemenu)

(defun hyperlatex-format-makemenu ()
  "We want a menu here, with given depth."
  (let ((depth (string-to-int (hyperlatex-parse-required-argument))))
    (setq hyperlatex-menu-in-section t)
    (insert (format "\M-XMenu{%d}{%d}\n"
		    hyperlatex-sect-number
		    depth))))

(defun hyperlatex-close-menus (newlev lastlev)
  "Inserts enough MENU or /MENU tags to get to NEWLEV (from LASTLEV)."
  (let ((oldlev lastlev))
    (while (> newlev oldlev)
      (hyperlatex-gen "MENU" nil "\n")
      (setq oldlev (1+ oldlev)))
    (while (< newlev oldlev)
      (hyperlatex-gen "/MENU" nil "\n")
      (setq oldlev (1- oldlev)))))

(defun hyperlatex-insert-menus ()
  "Really insert the menus now."
  (message "Inserting menus ... ")
  (goto-char (point-min))
  (while (search-forward "\M-XMenu" nil t)
    (replace-match "")
    (let ((secnum (string-to-int (hyperlatex-get-arg-here)))
	  (depth  (string-to-int (hyperlatex-get-arg-here)))
	  (sp     hyperlatex-rev-sections))
      (while (/= (hyperlatex-sect-num (car sp)) secnum)
	(setq sp (cdr sp)))
      ;; sp points to section 
      (let* ((lev (hyperlatex-sect-level (car sp)))
	     (nodenum (hyperlatex-sect-node (car sp)))
	     (lastlev lev))
	(setq sp (cdr sp))
	(while (and sp (> (hyperlatex-sect-level (car sp)) lev))
	  ;; sp points to a subsection of mine!
	  (if (<= (hyperlatex-sect-level (car sp)) (+ lev depth))
	      ;; make a menu entry
	      (let ((newlev (hyperlatex-sect-level (car sp))))
		(hyperlatex-close-menus newlev lastlev)
		(setq lastlev newlev)
		(hyperlatex-gen "LI")
		(hyperlatex-gen (format "A HREF=\"%s\""
					(hyperlatex-gen-url
					 (hyperlatex-sect-node (car sp))
					 (hyperlatex-sect-label (car sp))
					 nodenum)))
		(insert (hyperlatex-sect-head (car sp)))
		(hyperlatex-gen "/A" nil "\n")))
	  (setq sp (cdr sp)))
	(hyperlatex-close-menus lev lastlev)))))

;;;
;;; ----------------------------------------------------------------------
;;;
;;; Cross referencing, hypertext links
;;;

(put 'label	'hyperlatex-format 'hyperlatex-format-label)
(put 'xlabel	'hyperlatex-format 'hyperlatex-format-xlabel)
(put 'link	'hyperlatex-format 'hyperlatex-format-link)
(put 'xlink	'hyperlatex-format 'hyperlatex-format-xlink)

(defun hyperlatex-canon-label (str)
  "Makes string a canonical label name, by removing all strange chars."
  (let ((mystr (copy-sequence str)))
    (while (string-match "[&<>/#% [\"\\?]\\|]" mystr)
      (aset mystr (string-match "[&<>/#% [\"\\?]\\|]" mystr) ?_))
    mystr))

(defun hyperlatex-drop-label ()
  "Drop a label at the current position and return its number. Reuse last label
if there is one."
  (if (save-excursion
	(skip-chars-backward " \t\n\M-p\M-n")
	(= (preceding-char) (+ ?l 128)))
      ()
    ;; else make a new label at current position
    (insert "\M-<A NAME=\"" (format "%d" hyperlatex-label-number)
	    "\"\M->\M-&#160;\M-</A\M->\M-l")
    (setq hyperlatex-label-number (1+ hyperlatex-label-number)))
  (1- hyperlatex-label-number))

(defun hyperlatex-format-label ()
  "Creates a label at current position... But if we are directly behind
a section heading, uses section's label instead."
  (let ((label (hyperlatex-parse-required-argument))
	(number (hyperlatex-drop-label)))
    (setq hyperlatex-labels
	  (cons (list label number hyperlatex-node-number)
		hyperlatex-labels))))

(defun hyperlatex-format-xlabel ()
  "Creates an external label at current position."
  (let ((label (hyperlatex-parse-required-argument)))
    (insert "\M-<A NAME=\"" label "\"\M->\M-&#160;\M-</A\M->")))

(defun hyperlatex-gen-url (label-node label-number &optional current)
  "Generates a URL for a label in NODE with NUMBER. If node is the same as the
CURRENT node, simply returns `#NUMBER', else returns `NAME#NUMBER', unless
NUMBER is zero, in which case the returned url is `NAME`.
CURRENT is optional, and defaults to the current node."
  (if (zerop label-number)
      (hyperlatex-fullname label-node)
    (format "%s#%d"
	    (if (= (if current current hyperlatex-node-number) label-node)
		""
	      (hyperlatex-fullname label-node))
	    label-number)))

(defun hyperlatex-format-link ()
  (let ((text (hyperlatex-parse-required-argument))
	(latex-text (hyperlatex-parse-optional-argument))
	(label (hyperlatex-parse-required-argument)))
    (hyperlatex-gen (format "A HREF=\"\M-L{%s}{%d}\"" label
			    hyperlatex-node-number))
    (let ((here (point)))
      (insert text)
      (hyperlatex-gen "/A")
      (goto-char here))))

(defun hyperlatex-format-xlink ()
  (let ((text (hyperlatex-parse-required-argument))
	(latex-text (hyperlatex-parse-optional-argument))
	(url (hyperlatex-parse-required-argument)))
    (hyperlatex-gen (concat "A HREF=\"" url "\""))
    (insert text)
    (hyperlatex-gen "/A"))
  (goto-char hyperlatex-command-start))

(defun hyperlatex-insert-links ()
  "Turns the magic L entries into real links."
  (message "Inserting labels ... ")
  (goto-char (point-min))
  (while (search-forward "\M-L" nil t)
    (replace-match "")
    (let* ((label (hyperlatex-get-arg-here))
	   (node-number (hyperlatex-get-arg-here))
	   (match (assoc label hyperlatex-labels)))
      (if (null match)
	  (if (not noninteractive)
	      (error "Unknown label %s" label)
	    (message "WARNING: Unknown label %s " label))
	(insert (hyperlatex-gen-url (nth 2 match) (nth 1 match)
				    (string-to-int node-number)))))))

;;;
;;; ----------------------------------------------------------------------
;;;
;;; Environments
;;;

;; \begin{itemize} pushes (itemize "COMMANDS" STARTPOS) on hyperlatex-stack.
;; \begin{enumerate} pushes (enumerate 0 STARTPOS).
;; \item dispatches to the hyperlatex-item prop of the first elt of the list.
;; For itemize, this puts in and rescans the COMMANDS.

(put 'item	  'hyperlatex-format	'hyperlatex-item)

(put 'itemize	  'hyperlatex-format	'hyperlatex-itemize)
(put 'itemize	  'hyperlatex-end	'hyperlatex-end-itemize)
(put 'itemize	  'hyperlatex-item	'hyperlatex-itemize-item)
(put 'menu	  'hyperlatex-format	'hyperlatex-menu)
(put 'menu	  'hyperlatex-end	'hyperlatex-end-menu)
(put 'menu	  'hyperlatex-item	'hyperlatex-itemize-item)
(put 'enumerate   'hyperlatex-format	'hyperlatex-enumerate)
(put 'enumerate   'hyperlatex-end	'hyperlatex-end-enumerate)
(put 'enumerate	  'hyperlatex-item	'hyperlatex-itemize-item)
(put 'description 'hyperlatex-format	'hyperlatex-description)
(put 'description 'hyperlatex-end	'hyperlatex-end-description)
(put 'description 'hyperlatex-item	'hyperlatex-description-item)


(defvar hyperlatex-stack-depth 0
  "Count of number of unpopped hyperlatex-push-stack calls.
 Used by \\par to avoid paragraphing within verbatim and example
 environments." )

(defun hyperlatex-push-stack (check)
  (setq hyperlatex-stack-depth (1+ hyperlatex-stack-depth))
  (setq hyperlatex-stack
	(cons (list check hyperlatex-command-start)
	      hyperlatex-stack)))

(defun hyperlatex-pop-stack (check)
  (setq hyperlatex-stack-depth (1- hyperlatex-stack-depth))
  (if (null hyperlatex-stack)
      (error "Unmatched \\end{%s}" check))
  (if (not (eq (car (car hyperlatex-stack)) check))
      (error "\\end{%s} matches \\begin{%s}"
	     check (car (car hyperlatex-stack))))
  (prog1 (cdr (car hyperlatex-stack))
	 (setq hyperlatex-stack (cdr hyperlatex-stack))))
;;
;; dispatch for \\item
;;

(defun hyperlatex-item ()
  (funcall (get (car (car hyperlatex-stack)) 'hyperlatex-item)))

;;;
;;; ITEMIZE
;;;

(defun hyperlatex-itemize ()
  (hyperlatex-push-stack 'itemize)
  (hyperlatex-delete-whitespace)
  (hyperlatex-gen "UL" t))

(defun hyperlatex-end-itemize ()
  (hyperlatex-pop-stack 'itemize)
  (hyperlatex-gen "/UL" t "\n"))

(defun hyperlatex-itemize-item ()
  (hyperlatex-gen "LI" t))

;;;
;;; MENU
;;;

(defun hyperlatex-menu ()
  (hyperlatex-push-stack 'menu)
  (hyperlatex-delete-whitespace)
  (hyperlatex-gen "MENU" t))

(defun hyperlatex-end-menu ()
  (hyperlatex-pop-stack 'menu)
  (hyperlatex-gen "/MENU" t "\n"))

;;;
;;; ENUMERATE
;;;

(defun hyperlatex-enumerate ()
  (hyperlatex-push-stack 'enumerate)
  (hyperlatex-delete-whitespace)
  (hyperlatex-gen "OL" t))

(defun hyperlatex-end-enumerate ()
  (hyperlatex-pop-stack 'enumerate)
  (hyperlatex-gen "/OL" t "\n"))

;;;
;;; DESCRIPTION
;;; 

(defun hyperlatex-description ()
  (hyperlatex-push-stack 'description)
  (hyperlatex-delete-whitespace)
  (hyperlatex-gen "DL" t))

(defun hyperlatex-end-description ()
  (hyperlatex-pop-stack 'description)
  (hyperlatex-gen "/DL" t "\n"))

(defun hyperlatex-description-item ()
  (let ((arg (hyperlatex-parse-optional-argument)))
    (if (null arg)
	(error "Missing argument for \\item in description environment"))
    (hyperlatex-gen "DT" t)
    (hyperlatex-gen "B" t arg)
    (hyperlatex-gen "/B" t "\n")
    (hyperlatex-gen "DD" t))
  (goto-char hyperlatex-command-start))

;;;
;;; ----------------------------------------------------------------------
;;;
;;; Tables and figures
;;;

(put 'tabular	'hyperlatex-format 'hyperlatex-format-tabular)
(put 'tabular	'hyperlatex-end	   'hyperlatex-end-tabular)
(put 'hline	'hyperlatex-format 'hyperlatex-format-htmlrule)
(put 'S		'hyperlatex-format 'hyperlatex-format-ignore)
(put 'table	'hyperlatex-format 'hyperlatex-format-table)
(put 'table*	'hyperlatex-format 'hyperlatex-format-table)
(put 'table	'hyperlatex-end    'hyperlatex-end-table)
(put 'figure	'hyperlatex-format 'hyperlatex-format-figure)
(put 'figure*	'hyperlatex-format 'hyperlatex-format-figure)
(put 'figure	'hyperlatex-end    'hyperlatex-end-figure)
(put 'caption	'hyperlatex-format 'hyperlatex-format-caption)

;;;
;;; The tabular environment
;;;

(defun hyperlatex-format-tabular ()
  (hyperlatex-parse-required-argument)
  (hyperlatex-push-stack 'tabular)
  (hyperlatex-gen "PRE" t))

(defun hyperlatex-end-tabular ()
  (save-excursion
    (narrow-to-region (car (hyperlatex-pop-stack 'tabular)) (point))
    (goto-char (point-min))
    (while (search-forward "\M-p" nil t)
      (replace-match "\M- "))
    (goto-char (point-max))
    (widen))
  (hyperlatex-gen "/PRE" t "\n"))

;;;
;;; The table and figure environments
;;;

(defun hyperlatex-format-table ()
  (hyperlatex-parse-optional-argument)
  (hyperlatex-delete-whitespace)
  (hyperlatex-push-stack 'table)
  (setq hyperlatex-table-number (1+ hyperlatex-table-number)))

(defun hyperlatex-end-table ()
  (hyperlatex-pop-stack 'table))

(defun hyperlatex-format-figure ()
  (hyperlatex-parse-optional-argument)
  (hyperlatex-delete-whitespace)
  (hyperlatex-push-stack 'figure)
  (setq hyperlatex-figure-number (1+ hyperlatex-figure-number)))

(defun hyperlatex-end-figure ()
  (hyperlatex-pop-stack 'figure))

(defun hyperlatex-format-caption ()
  (let ((caption (hyperlatex-parse-required-argument)))
    (hyperlatex-gen "BLOCKQUOTE" t "\n")
    (insert
     (if (assoc 'table hyperlatex-stack)
	 (format "Table %d : %s\n" hyperlatex-table-number caption)
       (format "Figure %d : %s\n" hyperlatex-figure-number caption)))
    (hyperlatex-gen "/BLOCKQUOTE" t)
    (goto-char hyperlatex-command-start)))

;;;
;;; ----------------------------------------------------------------------
;;;
;;; Quotations, examples and verbatim environments
;;;

(put '\\	'hyperlatex-format	'hyperlatex-format-\\)
(put 'example	'hyperlatex-format	'hyperlatex-format-example)
(put 'example	'hyperlatex-end		'hyperlatex-end-example)
(put 'quotation 'hyperlatex-format	'hyperlatex-format-quote)
(put 'quotation 'hyperlatex-end		'hyperlatex-end-quote)
(put 'quote	'hyperlatex-format	'hyperlatex-format-quote)
(put 'quote	'hyperlatex-end		'hyperlatex-end-quote)
(put 'verse	'hyperlatex-format	'hyperlatex-format-quote)
(put 'verse	'hyperlatex-end		'hyperlatex-end-quote)
(put 'center	'hyperlatex-format	'hyperlatex-format-center)
(put 'center	'hyperlatex-end		'hyperlatex-end-center)
(put 'verbatim	'hyperlatex-format	'hyperlatex-format-verbatim)
(put 'verbatim	'hyperlatex-end		'hyperlatex-end-verbatim)
(put 'verb	'hyperlatex-format	'hyperlatex-format-verb)
(put '+		'hyperlatex-format	'hyperlatex-format-+)

(defun hyperlatex-format-\\ ()
  "Insert a <BR> tag, except in example and tabular, where it does nothing."
  (hyperlatex-starred-p)
  (if (assoc 'example hyperlatex-stack)
      ()
    (if (assoc 'tabular hyperlatex-stack)
	()
      (hyperlatex-gen "BR"))))
  
(defun hyperlatex-format-example ()
  (hyperlatex-push-stack 'example)
  (hyperlatex-gen "BLOCKQUOTE" t)
  (hyperlatex-gen "PRE" t))

(defun hyperlatex-end-example ()
  ;; protect empty lines and remove initial blanks
  (let ((from (car (hyperlatex-pop-stack 'example))))
    (narrow-to-region from (point))
    (goto-char (point-min))
    (while (search-forward "\M-p" nil t)
      (replace-match "\M- "))
    (goto-char (point-max))
    (widen))
  (hyperlatex-gen "/PRE" t)
  (hyperlatex-gen "/BLOCKQUOTE" t "\n"))

(defun hyperlatex-format-quote ()
  (hyperlatex-push-stack 'quote)
  (hyperlatex-delete-whitespace)
  (hyperlatex-gen "BLOCKQUOTE" t))

(defun hyperlatex-end-quote ()
  (hyperlatex-pop-stack 'quote)
  (hyperlatex-gen "/BLOCKQUOTE" t "\n"))

(defun hyperlatex-format-center ()
  (hyperlatex-push-stack 'center)
  (hyperlatex-delete-whitespace)
  (hyperlatex-gen "CENTER" t))

(defun hyperlatex-end-center ()
  (hyperlatex-pop-stack 'center)
  (hyperlatex-gen "/CENTER" t "\n"))

;;;
;;; The following characters have to be protected in verbatim stuff:
;;;   -  '  ` { } ~ %
;;;

(defun hyperlatex-format-verb ()
  "Handle the LaTeX \\verb command."
  (hyperlatex-gen "CODE")
  (let ((the-char (following-char))
	(from (point)))
    (delete-char 1)
    (search-forward (char-to-string the-char))
    (delete-char -1)
    (hyperlatex-protect-verbatim from (point)))
  (hyperlatex-gen "/CODE"))

(defun hyperlatex-format-+ ()
  "Handle the LaTeX \\+ command."
  (hyperlatex-gen "CODE")
  (let ((from (point)))
    (search-forward "+")
    (delete-char -1)
    (hyperlatex-protect-verbatim from (point)))
  (hyperlatex-gen "/CODE"))

(defun hyperlatex-format-verbatim ()
  (hyperlatex-gen "PRE" t)
  (let ((beg (point))
	(end (progn (search-forward  "\\end{verbatim}" nil nil)
		    (point))))
    (hyperlatex-protect-verbatim beg end)
    ;; get rid of \\end{verbatim}
    (goto-char end)
    (delete-char -14))
  (hyperlatex-gen "/PRE" t))

(defun hyperlatex-end-verbatim ()
  (error "Nested verbatim environments do not make sense and do not work"))

(defun hyperlatex-protect-verbatim (beg end)
  "In region from BEGIN to END, protect verbatim characters by making
them magic. Also remove magic PAR markers. Point is left at end of region."
  (narrow-to-region beg end)
  (goto-char (point-min))
  (while (re-search-forward "[-'`{}~]" nil t)
    (replace-match (char-to-string (+ (preceding-char) 128))))
  (goto-char (point-min))
  (while (search-forward "\M-p" nil t)
    (replace-match "\M- "))
  (goto-char (point-max))
  (widen))

;;;
;;; ----------------------------------------------------------------------
;;;
;;; Emphasize, some little macros, and things that are ignored in HTML
;;;

(put 'LaTeX	'hyperlatex-format 'hyperlatex-format-LaTeX)
(put 'LaTeXe	'hyperlatex-format 'hyperlatex-format-LaTeXe)
(put 'TeX	'hyperlatex-format 'hyperlatex-format-TeX)
(put 'back	'hyperlatex-format 'hyperlatex-format-backslash)
(put '=		'hyperlatex-format 'hyperlatex-format-backslash)
(put 'c 	'hyperlatex-format 'hyperlatex-format-c)
(put 'copyright 'hyperlatex-format 'hyperlatex-format-copyright)
(put 'ldots	'hyperlatex-format 'hyperlatex-format-ldots)
(put 'minus	'hyperlatex-format 'hyperlatex-format-minus)
(put 'qquad	'hyperlatex-format 'hyperlatex-format-qquad)
(put 'quad	'hyperlatex-format 'hyperlatex-format-quad)
(put 'sim	'hyperlatex-format 'hyperlatex-format-sim)
(put 'ss 	'hyperlatex-format 'hyperlatex-format-ss)
(put 'today	'hyperlatex-format 'hyperlatex-format-today)

(put 'em	'hyperlatex-format 'hyperlatex-format-em)
(put 'it	'hyperlatex-format 'hyperlatex-format-it)
(put 'bf	'hyperlatex-format 'hyperlatex-format-bf)

(put 'normalsize 'hyperlatex-format 'hyperlatex-format-normalsize)
(put 'large	'hyperlatex-format 'hyperlatex-format-large)
(put 'Large	'hyperlatex-format 'hyperlatex-format-Large)
(put 'LARGE	'hyperlatex-format 'hyperlatex-format-LARGE)
(put 'huge	'hyperlatex-format 'hyperlatex-format-huge)
(put 'Huge	'hyperlatex-format 'hyperlatex-format-Huge)
(put 'small	'hyperlatex-format 'hyperlatex-format-small)
(put 'footnotesize 'hyperlatex-format 'hyperlatex-format-footnotesize)
(put 'scriptsize 'hyperlatex-format 'hyperlatex-format-scriptsize)
(put 'tiny	'hyperlatex-format 'hyperlatex-format-tiny)

(put 'textbf	'hyperlatex-format 'hyperlatex-format-bold)
(put 'bold	'hyperlatex-format 'hyperlatex-format-bold)
(put 'cit	'hyperlatex-format 'hyperlatex-format-cit)
(put 'code	'hyperlatex-format 'hyperlatex-format-code)
(put 'dfn	'hyperlatex-format 'hyperlatex-format-italic)
(put 'dmn	'hyperlatex-format 'hyperlatex-insert-required-argument)
(put 'emph	'hyperlatex-format 'hyperlatex-format-emph)
(put 'file	'hyperlatex-format 'hyperlatex-format-var)
(put 'textit	'hyperlatex-format 'hyperlatex-format-italic)
(put 'italic	'hyperlatex-format 'hyperlatex-format-italic)
(put 'kbd	'hyperlatex-format 'hyperlatex-format-kbd)
(put 'samp	'hyperlatex-format 'hyperlatex-format-samp)
(put 'textsc	'hyperlatex-format 'hyperlatex-format-scap)
(put 'scap	'hyperlatex-format 'hyperlatex-format-scap)
(put 'strong	'hyperlatex-format 'hyperlatex-format-strong)
(put 'texttt	'hyperlatex-format 'hyperlatex-format-typew)
(put 'typew	'hyperlatex-format 'hyperlatex-format-typew)
(put 'underline	'hyperlatex-format 'hyperlatex-format-underline)
(put 'var	'hyperlatex-format 'hyperlatex-format-var)

(put 'math	'hyperlatex-format 'hyperlatex-format-math)

(put 'protect	'hyperlatex-format 'hyperlatex-format-ignore)
(put 'noindent	'hyperlatex-format 'hyperlatex-format-ignore)


(defun hyperlatex-format-TeX ()
  (insert "TeX"))

(defun hyperlatex-format-copyright ()
  (hyperlatex-gensym "#169"))

(defun hyperlatex-format-minus ()
  (insert "-"))

(defun hyperlatex-format-ldots ()
  (insert "..."))

(defun hyperlatex-format-backslash ()
  "replace \\back and \\- by \\"
  (insert "\\"))

(defun hyperlatex-format-sim ()
  "Replace \\sim by magic `~'."
  (insert "\M-~"))

(defun hyperlatex-format-LaTeX ()
  (insert "LaTeX"))

(defun hyperlatex-format-LaTeXe ()
  (insert "LaTeX2e"))

(defun hyperlatex-format-quad ()
  (insert "    "))

(defun hyperlatex-format-qquad ()
  (insert "      "))

(defun hyperlatex-format-today ()
  (let* ((date-string (current-time-string))
	 (month-alist   '(("Jan" . "January") ("Feb" . "February") 
			  ("Mar" . "March") ("Apr" . "April")
			  ("May" . "May") ("Jun" . "June") 
			  ("Jul" . "July") ("Aug" . "August")
			  ("Sep" . "September") ("Oct" . "October")
			  ("Nov" . "November") ("Dec" . "December")))
	 )
    (string-match "\\(...\\) \\(...\\) \\(..\\).*\\(19..\\)"
		  (current-time-string) nil)
    (insert
     (concat (cdr (assoc (substring date-string 
				    (match-beginning 2) (match-end 2))
			 month-alist))
	     " " (substring date-string (match-beginning 3) (match-end 3))
	     ", " (substring date-string (match-beginning 4) (match-end 4))))))

(defun hyperlatex-format-ss ()
  (hyperlatex-gensym "szlig"))

(defun hyperlatex-format-c ()
  (let ((arg (hyperlatex-parse-required-argument t)))
    (if (or (string= arg "c")
	    (string= arg "C"))
	(hyperlatex-gensym (concat arg "cedil"))
      (error "Invalid cedilla accent"))))

;;
;; font changes
;;

(defun hyperlatex-format-bf ()
  (hyperlatex-format-font-1 "B"))

(defun hyperlatex-format-em ()
  (hyperlatex-format-font-1 "EM"))

(defun hyperlatex-format-it ()
  (hyperlatex-format-font-1 "I"))

(defun hyperlatex-format-bold ()
  (hyperlatex-format-font "B"))

(defun hyperlatex-format-cit ()
  (hyperlatex-format-font "CITE"))

(defun hyperlatex-format-code ()
  (hyperlatex-format-font "CODE"))

(defun hyperlatex-format-emph ()
  (hyperlatex-format-font "EM"))

(defun hyperlatex-format-italic ()
  (hyperlatex-format-font "I"))

(defun hyperlatex-format-kbd ()
  (hyperlatex-format-font "KBD"))

(defun hyperlatex-format-samp ()
  (hyperlatex-format-font "SAMP"))

(defun hyperlatex-format-scap ()
  (insert (upcase (hyperlatex-parse-required-argument)))
  (goto-char hyperlatex-command-start))

(defun hyperlatex-format-strong ()
  (hyperlatex-format-font "STRONG"))

(defun hyperlatex-format-typew ()
  (hyperlatex-format-font "TT"))

(defun hyperlatex-format-underline ()
  (hyperlatex-format-font "U"))

(defun hyperlatex-format-var ()
  (hyperlatex-format-font "VAR"))

(defun hyperlatex-format-font (font)
  (let ((arg (hyperlatex-parse-required-argument)))
    (hyperlatex-gen font)
    (insert arg)
    (hyperlatex-gen (concat "/" font)))
  (goto-char hyperlatex-command-start))

(defun hyperlatex-format-font-1 (font)
  (skip-chars-backward " \t")
  (if (/= (preceding-char) ?\{)
      (error "Illegal use of font changing command"))
  (setq hyperlatex-command-start (1- (point)))
  (let ((arg (hyperlatex-parse-required-argument)))
    (hyperlatex-gen font)
    (insert arg)
    (hyperlatex-gen (concat "/" font)))
  (goto-char hyperlatex-command-start))

(defun hyperlatex-format-math ()
  "Format \\math{} and \\math[]{}."
  (let ((opt (hyperlatex-parse-optional-argument))
	(req (hyperlatex-parse-required-argument)))
    (if hyperlatex-mathitalics
	(hyperlatex-gen "I"))
    (insert (if opt opt req))
    (if hyperlatex-mathitalics
	(hyperlatex-gen "/I")))
  (goto-char hyperlatex-command-start))

(defun hyperlatex-format-normalsize ()
  (hyperlatex-format-fontsize "+0"))

(defun hyperlatex-format-large ()
  (hyperlatex-format-fontsize "+1"))

(defun hyperlatex-format-Large ()
  (hyperlatex-format-fontsize "+2"))

(defun hyperlatex-format-LARGE ()
  (hyperlatex-format-fontsize "+3"))

(defun hyperlatex-format-huge ()
  (hyperlatex-format-fontsize "+4"))

(defun hyperlatex-format-Huge ()
  (hyperlatex-format-fontsize "+5"))

(defun hyperlatex-format-small ()
  (hyperlatex-format-fontsize "-1"))

(defun hyperlatex-format-footnotesize ()
  (hyperlatex-format-fontsize "-2"))

(defun hyperlatex-format-scriptsize ()
  (hyperlatex-format-fontsize "-3"))

(defun hyperlatex-format-tiny ()
  (hyperlatex-format-fontsize "-4"))

(defun hyperlatex-format-fontsize (size)
  (skip-chars-backward " \t")
  (if (/= (preceding-char) ?\{)
      (error "Illegal use of font size changing command"))
  (setq hyperlatex-command-start (1- (point)))
  (let ((arg (hyperlatex-parse-required-argument)))
    (hyperlatex-gen (format "font size=%s" size))
    (insert arg)
    (hyperlatex-gen "/font size=+0"))
  (goto-char hyperlatex-command-start))

;;;
;;; ----------------------------------------------------------------------
;;;
;;; Index generation
;;;

(put 'index		'hyperlatex-format 'hyperlatex-format-index)
(put 'cindex		'hyperlatex-format 'hyperlatex-format-index)
(put 'htmlprintindex	'hyperlatex-format 'hyperlatex-format-printindex)

(defun hyperlatex-single-line (str)
  "Replaces newlines in STRING by spaces."
  (if str
      (let ((mystr (copy-sequence str)))
	(while (string-match "\n" mystr)
	  (aset mystr (string-match "\n" mystr) 32))
	mystr)
    nil))
  
(defun hyperlatex-format-index ()
  "Adds an index entry."
  (let ((opt (hyperlatex-single-line (hyperlatex-parse-optional-argument)))
	(arg (hyperlatex-single-line (hyperlatex-parse-required-argument)))
	(label (hyperlatex-drop-label)))
    (setq hyperlatex-index
	  (cons (list arg (if opt opt arg) hyperlatex-node-number label)
		hyperlatex-index))))

(defun hyperlatex-format-printindex ()
  (let ((indexelts hyperlatex-index)
	opoint)
    (hyperlatex-gen "MENU" t "\n")
    (setq opoint (point))
    (while indexelts
      ;; put search-key 
      (insert (nth 1 (car indexelts)) " \M-I")
      (hyperlatex-gen (concat "A HREF="
			      (hyperlatex-gen-url (nth 2 (car indexelts))
						  (nth 3 (car indexelts)))
			      ""))
      (insert (car (car indexelts)))
      (hyperlatex-gen "/A" nil "\n")
      (setq indexelts (cdr indexelts)))
    (shell-command-on-region opoint (point) "sort -f" 1)
    (narrow-to-region opoint (point))
    (goto-char (point-min))
    (while (re-search-forward "^.* \M-I" nil t)
      (replace-match "")
      (hyperlatex-gen "LI"))
    (goto-char (point-max))
    (widen)
    (hyperlatex-gen "/MENU" t "\n")
    (goto-char opoint)))
  
;;;
;;; ----------------------------------------------------------------------
;;;
;;; iftex, ifhtml, tex, \set, \clear, \ifset, \ifclear
;;;

;; If a flag is set with \set FLAG, then text between \ifset and \end
;; ifset is formatted normally, but if the flag is is cleared with
;; \clear FLAG, then the text is not formatted; it is ignored.

;; If a flag is cleared with \clear FLAG, then text between \ifclear
;; and \end ifclear is formatted normally, but if the flag is is set with
;; \set FLAG, then the text is not formatted; it is ignored.  \ifclear
;; is the opposite of \ifset.

(put 'clear  'hyperlatex-format 'hyperlatex-clear)
(put 'set    'hyperlatex-format	'hyperlatex-set)
(put 'ifset  'hyperlatex-format 'hyperlatex-if-set)
(put 'ifset  'hyperlatex-end	'hyperlatex-format-ignore)
(put 'ifclear 'hyperlatex-format 'hyperlatex-if-clear)
(put 'ifclear 'hyperlatex-end	'hyperlatex-format-ignore)
(put 'ifhtml 'hyperlatex-format 'hyperlatex-format-ignore)
(put 'ifhtml 'hyperlatex-end	'hyperlatex-format-ignore)
(put 'iftex  'hyperlatex-format 'hyperlatex-format-iftex)
(put 'tex    'hyperlatex-format 'hyperlatex-format-tex)

(defun hyperlatex-ifset-flag (&optional test-value)
  (let* ((arg (hyperlatex-parse-required-argument))
	 (flag (intern (concat "hyperlatex-FLAG-" arg))))
    (if test-value
	;; return value of flag, nil if not bound
	(and (boundp flag) (symbol-value flag))
      ;; return flag itself
      flag)))

(defun hyperlatex-clear ()
  "Clear the value of the flag."
  (let ((flag (hyperlatex-ifset-flag)))
    (make-local-variable flag)
    (set flag nil)))

(defun hyperlatex-set ()
  "Set the value of the flag."
  (let ((flag (hyperlatex-ifset-flag)))
    (make-local-variable flag)
    (set flag t)))

(defun hyperlatex-if-set ()
  "If set, continue formatting; else do not format region up to \\end{ifset}"
  (if (hyperlatex-ifset-flag t)
      ;; flag is set, don't do anything
      () 
    (delete-region hyperlatex-command-start
		   (progn (search-forward "\\end{ifset}") (point)))
    (hyperlatex-delete-whitespace)))

(defun hyperlatex-if-clear ()
  "If clear, continue formatting; else do not format region up
 to \\end{ifclear}."
  (if (not (hyperlatex-ifset-flag t))
      ;; flag is clear, don't do anything
      () 
    (delete-region hyperlatex-command-start
		   (progn (search-forward "\\end{ifclear}") (point)))
    (hyperlatex-delete-whitespace)))

(defun hyperlatex-format-iftex ()
  (delete-region hyperlatex-command-start
		 (progn (search-forward "\\end{iftex}") (point)))
  (hyperlatex-delete-whitespace))

(defun hyperlatex-format-tex ()
  (delete-region hyperlatex-command-start
		 (progn (search-forward "\\end{tex}") (point)))
  (hyperlatex-delete-whitespace))

;;;
;;; ----------------------------------------------------------------------
;;;
;;; Bibliography support, for included .bbl files
;;;

(put 'bibliography	'hyperlatex-format 'hyperlatex-format-bibliography)
(put 'thebibliography	'hyperlatex-format 'hyperlatex-thebibliography)
(put 'thebibliography	'hyperlatex-end	   'hyperlatex-format-ignore)
(put 'bibitem		'hyperlatex-format 'hyperlatex-format-bibitem)
(put 'newblock		'hyperlatex-format 'hyperlatex-format-ignore)
(put 'bibliographystyle 'hyperlatex-format 'hyperlatex-parse-required-argument)

(defun hyperlatex-format-bibliography ()
  (let* ((tex-name (buffer-file-name input-buffer))
	 (base-name (progn
		      (if (string-match "^.*\\(\\.[a-zA-Z0-9]+\\)$" tex-name)
			  (substring tex-name 0 (match-beginning 1))
			tex-name)))
	 (hyperlatex-bbl-filename (concat base-name ".bbl")))
    (hyperlatex-parse-required-argument)
    (if (file-exists-p hyperlatex-bbl-filename)
	(progn
	  (insert-file hyperlatex-bbl-filename)
	  (goto-char hyperlatex-command-start))
      (message "Formatted bibliography file not found: %s"
	       hyperlatex-bbl-filename))))

(defun hyperlatex-thebibliography ()
  (hyperlatex-parse-required-argument)
  (save-excursion (insert "\\section*{References}\n")))

(defun hyperlatex-format-bibitem ()
  (hyperlatex-parse-optional-argument)
  (hyperlatex-format-label)
  (insert "\n\M-p\n"))

;;;
;;; ----------------------------------------------------------------------
;;;
;;; Extending HyperLaTeX
;;;

(defun hyperlatex-run-documentstyle-hooks ()
  "Foreach \\documentstyle-argument DOCSTYLE, look for
hyperlatex-DOCSTYLE.[el,elc] in the load-path. If hyperlatex-DOCSTYLE-hook
is bound, run it."
  (goto-char (point-min))
  (re-search-forward "\\\\document\\(style\\|class\\)")
  (if (looking-at "\\[")
      (let ((begin (1+ (point)))
	    (end (save-excursion (search-forward "]") (point)))
	    (options-list nil))
	(while (re-search-forward ",\\|]" end t)
	  (setq options-list (cons (buffer-substring begin (1- (point)))
				   options-list))
	  (setq begin (point)))
	(if (looking-at "[ \t\n]*{article}")
	    (setq hyperlatex-is-article t))
	(setq options-list (nreverse options-list))
	(while options-list
	  (let ((option (car options-list))
		(filename nil))
	    (if (not (memq (intern option) hyperlatex-known-document-styles))
		(progn
		  (message "Checking formatting option %s" option)
		  (if (load (concat "hyperlatex-" option) t) ;dont report errs
		      (let ((option-symbol
			     (intern (concat "hyperlatex-" option "-hook"))))
			(if (fboundp option-symbol)
			    (progn
			      (message "Running %s formatting hooks" option)
			      (sit-for 1) 
			      (funcall option-symbol)))
			(message (concat "Done loading file %s" option))
			)
		    ;; (message (concat option "-html not found"))
		    )
		  )
	      )
	    (setq options-list (cdr options-list)))))))

(provide 'hyperlatex1)

(defun hyperlatex-insert-hyperlatex ()
  (interactive)
  (insert "hyperlatex-"))
;;; (local-set-key "\C-s\C-?" 'hyperlatex-insert-hyperlatex)

(defun hyperlatex-compile ()
  "Byte compile Hyperlatex. 
Unix usage:
     emacs -batch -l hyperlatex1.el -f hyperlatex-compile."
  (setq byte-compile-verbose t)
  (if (not noninteractive)
      (error "This command must be used in batch mode."))
  (byte-compile-file "hyperlatex1.el"))

;;;
;;; ----------------------------------------------------------------------
;;;
;;; Local Variables:
;;; mode: emacs-lisp
;;; update-last-edit-date: t
;;; End:
;;;
