/* 
 * update.c	Common update routines for normal passwd file and
 *		J.F. Haugh's shadow password suite.
 *
 * Copyright 1994, 1995 Olaf Kirch, <okir@monad.swb.de>
 *
 * This program is covered by the GNU General Public License, version 2.
 * It is provided in the hope that it is useful. However, the author
 * disclaims ALL WARRANTIES, expressed or implied. See the GPL for details.
 */

#include <sys/types.h>
#include <sys/errno.h>
#include <sys/stat.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>
#include <pwd.h>
#ifdef SHADOWPWD
#include <shadow.h>
#endif

#include <syslog.h>
#include <stdio.h>
#include <string.h>

#include <rpc/rpc.h>
#include <rpc/pmap_clnt.h>
#include "yppasswd.h"


/* How often to retry locking the passwd file...
 */
#define MAX_RETRIES 5

#ifdef DEBUG
# define RPC_SVC_FG
# define _PATH_PASSWD    	"/tmp/passwd"
# define _PATH_OLDPASSWD 	"/tmp/passwd.OLD"
# define _PATH_PTMP  	 	"/tmp/ptmp"
#else
# ifndef _PATH_PASSWD
#  define _PATH_PASSWD   	"/etc/passwd"
# endif
# ifndef _PATH_OLDPASSWD
#  define _PATH_OLDPASSWD 	"/etc/passwd.OLD"
# endif
# ifdef SHADOWPWD
#  ifndef _PATH_SHADOW
#   define _PATH_SHADOW		"/etc/shadow"
#  endif
# endif
# ifndef _PATH_PTMP
#  define _PATH_PTMP	 	"/etc/ptmp"
# endif
#endif

#define xprt_addr(xprt)	(svc_getcaller(xprt)->sin_addr)
#define xprt_port(xprt)	ntohs(svc_getcaller(xprt)->sin_port)
void reaper( int sig );

/*===============================================================*
 * Argument validation. Avoid \n... (ouch).
 * We can't use isprint, because people may use 8bit chars which
 * aren't recognized as printable in the default locale.
 *===============================================================*/
static int
validate_string(char *str)
{
    while (*str && *str != ':' && !iscntrl(*str)) str++;
    return (*str == '\0');
}

static int
validate_args(struct xpasswd *pw)
{
    if (pw->pw_name[0] == '-' || pw->pw_name[0] == '+') {
	syslog(LOG_ALERT, "attempt to modify NIS passwd entry \"%s\"",
			pw->pw_name);
    }

    return validate_string(pw->pw_passwd)
       &&  validate_string(pw->pw_shell)
       &&  validate_string(pw->pw_gecos);
}

/*===============================================================*
 * The /etc/passwd update handler
 *===============================================================*/
int *
yppasswdproc_pwupdate_1(yppasswd *yppw, struct svc_req *rqstp)
{
    struct xpasswd *newpw;	/* passwd struct passed by the client */
    struct passwd *pw;		/* passwd struct obtained from getpwent() */
    int		retries, gotit, fd, c;
    int		chsh = 0, chfn = 0;
    FILE	*oldfp, *newfp;
    static int	res;
    char	logbuf[255];

    newpw = &yppw->newpw;
    res = 1;

    sprintf( logbuf, "update %.12s (uid=%d) from host %s",
			    yppw->newpw.pw_name,
			    yppw->newpw.pw_uid,
			    inet_ntoa(xprt_addr(rqstp->rq_xprt)));

    if (!validate_args(newpw)) {
        syslog ( LOG_ALERT, "%s failed", logbuf );
        syslog ( LOG_ALERT, "Invalid characters in argument. "
        		    "Possible spoof attempt?" );
        return &res;
    }

    /* Lock the passwd file. We retry several times. Maybe it would be
     * better to just return an error condition and have the client reattempt
     * instead? This procedure is already slow enough...
     */
    retries = 0;
    while ((fd = open(_PATH_PTMP, O_CREAT|O_WRONLY|O_EXCL)) < 0
      && errno == EEXIST && retries < MAX_RETRIES) {
        sleep (1);
        retries++;
    }

    if (retries == MAX_RETRIES) {
        syslog ( LOG_NOTICE, "%s failed", logbuf );
        syslog ( LOG_NOTICE, "password file locked" );
        return &res;
    }

    if (fd < 0 || (newfp = fdopen(fd, "w")) == NULL) {
        syslog ( LOG_ERR, "%s failed", logbuf );
        syslog ( LOG_ERR, "Can't create %s. %m", _PATH_PTMP );
        close (fd);
        return &res;
    }

    /* Open the passwd file for reading. We can't use getpwent and friends
     * here, because they go through the YP maps, too.
     */
    if ((oldfp = fopen(_PATH_PASSWD, "r")) == NULL) {
        syslog ( LOG_ERR, "%s failed", logbuf );
        syslog ( LOG_ERR, "Can't open %s: %m", _PATH_PASSWD );
        fclose (newfp);
        return &res;
    }

    gotit = 0;

    /*
     * Loop over all passwd entries
     */
    while((pw = fgetpwent(oldfp)) != NULL) {

        /* check if this is the uid we want to change. A few
         * sanity checks added for consistency.
         */
        if (newpw->pw_uid == pw->pw_uid && newpw->pw_gid == pw->pw_gid
         && !strcmp(newpw->pw_name, pw->pw_name) && !gotit) {

            /* Check the password.
             */
            if (pw->pw_passwd[0] != '\0' &&
		strcmp(crypt(yppw->oldpass, pw->pw_passwd), pw->pw_passwd)) {
        	syslog ( LOG_WARNING, "%s rejected", logbuf );
                syslog ( LOG_WARNING, "Invalid password." );
                sleep(1);
                break;
            }

            /* set the new passwd, shell, and full name
             */
            pw->pw_passwd = newpw->pw_passwd;
#ifdef ALLOW_CHSH
	    chsh = (strcmp(pw->pw_shell, newpw->pw_shell) != 0);
	    pw->pw_shell = newpw->pw_shell;
#endif
#ifdef ALLOW_CHFN
	    chfn = (strcmp(pw->pw_gecos, newpw->pw_gecos) != 0);
	    pw->pw_gecos = newpw->pw_gecos;
#endif
            gotit++;
        }

        /* write the passwd entry to /etc/ptmp
         */
        if (putpwent(pw, newfp) < 0) {
            syslog ( LOG_ERR, "%s failed", logbuf );
            syslog ( LOG_ERR, "Error while writing new password file: %m" );
            break;
        }
        /* fflush (newfp); */
    }
    fclose (newfp);
    fclose (oldfp);

    /* Check if we dropped out of the loop because of an error.
     * If so, return an error to the client.
     */
    if (pw != NULL) {
        unlink (_PATH_PTMP);
        return (&res);
    }

    /* Check whether we actually changed anything
     */
    if (!gotit) {
        syslog ( LOG_WARNING, "%s failed", logbuf );
        syslog ( LOG_WARNING, "User not in password file." );
        unlink (_PATH_PTMP);
        return (&res);
    }

    unlink (_PATH_OLDPASSWD);
    link (_PATH_PASSWD, _PATH_OLDPASSWD);
    unlink (_PATH_PASSWD);
    link (_PATH_PTMP, _PATH_PASSWD);
    unlink (_PATH_PTMP);
    chmod (_PATH_PASSWD, 0644);

    /* Fork off process to rebuild NIS passwd.* maps. If the fork
     * fails, restore old passwd file and return an error.
     */
    if ((c = fork()) < 0) {
    	unlink( _PATH_PASSWD );
    	link( _PATH_OLDPASSWD, _PATH_PASSWD );
    	syslog( LOG_ERR, "%s failed", logbuf );
    	syslog( LOG_ERR, "Couldn't fork map update process: %m" );
    	return (&res);
    }
    if (c == 0) {
    	execlp(MAP_UPDATE_PATH, MAP_UPDATE, NULL);
    	syslog( LOG_ERR, "Error: couldn't exec map update process: %m" );
    	exit(1);
    }

    syslog ( LOG_INFO, "%s successful. Password changed.", logbuf );
    if (chsh || chfn) {
    	syslog ( LOG_INFO, "Shell %schanged (%s), GECOS %schanged (%s).",
    			chsh? "" : "un", newpw->pw_shell,
    			chfn? "" : "un", newpw->pw_gecos );
    }
    res = 0;

    return (&res);
}

#ifdef SHADOWPWD
/*===============================================================*
 * The /etc/shadow update handler
 *===============================================================*/
int *
yppasswdproc_spwupdate_1(yppasswd *yppw, struct svc_req *rqstp)
{
    struct xpasswd *newpw;	/* passwd struct passed by the client */
    struct spwd *spw;		/* shadow struct obtained from spw_locate */
    static int	res;		/* return value */
    int		retries;	/* number of retries to lock shadow file */
    int		c;
    int		chsh = 0, chfn = 0;
    char	logbuf[255];

    newpw = &yppw->newpw;
    res = 1;

    sprintf( logbuf, "update %.12s (uid=%d) from host %s",
			    yppw->newpw.pw_name,
			    yppw->newpw.pw_uid,
			    inet_ntoa(xprt_addr(rqstp->rq_xprt)));

    if (!validate_args(newpw)) {
        syslog ( LOG_ALERT, "%s failed", logbuf );
        syslog ( LOG_ALERT, "Invalid characters in argument. "
        		    "Possible spoof attempt?" );
        return &res;
    }

    /* Lock the passwd file. We retry several times. Maybe it would be
     * better to just return an error condition and have the client reattempt
     * instead? This procedure is already slow enough...
     */
    retries = 0;
    while (!spw_lock() && retries < MAX_RETRIES) {
        sleep (1);
        retries++;
    }

    if (retries == MAX_RETRIES) {
        syslog ( LOG_NOTICE, "%s failed", logbuf );
        syslog ( LOG_NOTICE, "shadow password file locked" );
        return &res;
    }


    if (!spw_open(O_RDWR)) {
        syslog ( LOG_ERR, "%s failed", logbuf );
        syslog ( LOG_ERR, "Can't open %s: %m", _PATH_SHADOW);
        spw_unlock();
        return &res;
    }


    /*
     * Get old shadow password entry
     */
    if ((spw=spw_locate(newpw->pw_name)) == NULL) {
        syslog ( LOG_WARNING, "%s failed", logbuf );
        syslog ( LOG_WARNING, "User not in shadow file." );
        spw_close(); spw_unlock();
        return (&res);
    }

    /*
     * Check the password.
     */
    if (spw->sp_pwdp[0] != '\0' &&
        strcmp(crypt(yppw->oldpass, spw->sp_pwdp), spw->sp_pwdp)) {
	syslog ( LOG_WARNING, "%s rejected", logbuf );
	syslog ( LOG_WARNING, "Invalid password." );
	spw_close(); spw_unlock();
	return (&res);
    }

    /* Update GECOS and/or shell field. We don't support any fancy things
     * like /etc/shells or checking for restricted shells. Shadow passwords
     * are already messy enough.
     */
#if defined(ALLOW_CHSH) || defined(ALLOW_CHFN)
    {
        struct passwd  *pw;		/* passwd struct from getpwnam */

        if ((pw = getpwnam(newpw->pw_name)) == NULL) {
	    syslog ( LOG_WARNING, "%s failed", logbuf );
	    syslog ( LOG_WARNING, "User not in passwd file." );
            spw_close(); spw_unlock();
            return (&res);
        }

        /* Check that GECOS or shell have indeed changed. Otherwise,
         * we won't bother with it.
         */
        chsh = (strcmp(newpw->pw_shell, pw->pw_shell) != 0);
        chfn = (strcmp(newpw->pw_gecos, pw->pw_gecos) != 0);
        if (chsh || chfn) {
            retries = 0;
            while (!pw_lock() && retries < MAX_RETRIES) {
                sleep (1);
                retries++;
            }

            if (retries == MAX_RETRIES) {
                syslog ( LOG_NOTICE, "%s failed", logbuf );
                syslog ( LOG_NOTICE, "password file locked" );
                spw_close(); spw_unlock();
                return &res;
            }


            if (!pw_open(O_RDWR)) {
                syslog ( LOG_ERR, "%s failed", logbuf );
                syslog ( LOG_ERR, "Can't open %s: %m", _PATH_SHADOW);
                spw_close(); spw_unlock();
                pw_unlock();
                return &res;
            }

            /*
             * Get old passwd entry
             */
            if ((pw = pw_locate(newpw->pw_name)) == NULL) {
	        syslog ( LOG_ERR, "%s failed", logbuf );
	        syslog ( LOG_ERR, "getpwnam succeeds but pw_locate fails?!" );
                spw_close(); spw_unlock();
                pw_close();  pw_unlock();
                return (&res);
            }

#ifdef ALLOW_CHSH
            pw->pw_shell = newpw->pw_shell;
#endif
#ifdef ALLOW_CHFN
            pw->pw_gecos = newpw->pw_gecos;
#endif
            if (!pw_update(pw)) {
	        syslog ( LOG_ERR, "%s failed", logbuf );
	        syslog ( LOG_ERR, "Error while updating %s",  _PATH_SHADOW);
                pw_close();  pw_unlock();
	        spw_close(); spw_unlock();
	        return (&res);
            }
    
            pw_close();
            pw_unlock();
        }
    }
#endif

    /* Finally, update the password.
     */
    spw->sp_pwdp=newpw->pw_passwd;

    if (!spw_update(spw)) {
	syslog ( LOG_ERR, "%s failed", logbuf );
	syslog ( LOG_ERR, "Error while updating %s",  _PATH_SHADOW);
	spw_close(); spw_unlock();
	return (&res);
    }

    spw_close();
    spw_unlock();


    /* Fork off process to rebuild NIS passwd.* maps. If the fork
     * fails, restore old passwd file and return an error.
     */
    if ((c = fork()) < 0) {
    	syslog( LOG_ERR, "%s failed", logbuf );
    	syslog( LOG_ERR, "Couldn't fork map update process: %m" );
    	return (&res);
    }
    if (c == 0) {
    	execlp(MAP_UPDATE_PATH, MAP_UPDATE, NULL);
    	syslog( LOG_ERR, "Error: couldn't exec map update process: %m" );
    	exit(1);
    }

    syslog ( LOG_INFO, "%s successful. Password changed", logbuf );
    if (chsh || chfn) {
    	syslog ( LOG_INFO, "Shell %schanged (%s), GECOS %schanged (%s).",
    			chsh? "" : "un", newpw->pw_shell,
    			chfn? "" : "un", newpw->pw_gecos );
    }
    res = 0;

    return (&res);
}
#endif
