/* Copyright (c) 1994 by Sanjay Ghemawat */
/*
 * Behaves like Tk main routine if display can be opened, otherwise
 * like the Tcl main routine.  The following control whether or not
 * Tk is used --
 *
 *	$DISPLAY in environment			Use Tk
 *	-display				Use Tk
 *	-list					Do not use Tk
 *	-show					Do not use Tk
 *	-nodisplay				Do not use Tk
 *
 * The "-f" flag can be used to pass in an initialization script regardless
 * of whether or not Tk is used.
 *
 * All .tcl files from Tcl/Tk libraries are linked into the executable
 * as well to avoid depending on external files being installed correctly.
 */

#include <stdlib.h>
#include <string.h>
#include <tcl.h>
#include <tk.h>
#include "ical.h"

/* Include various libraries converted to strings. */

#ifdef STANDALONE
static char* tcl_lib_str[] = {
#include "tcl_lib.gen"
0
};

static char* tk_lib_str[] = {
#include "tk_lib.gen"
0
};

static char* ical_lib_str[] = {
#include "ical_lib.gen"
0
};

static char* tcllib_str[] = {
#include "tcllib.gen"
0
};
#endif

static char* ical_startup[] = {
#include "ical_start.gen"
0
};

static char* psheader_str[] = {
"set ical(psheader) {%!PS-Adobe-",
#include "psheader.gen"
"}",
0
};

static char* ical_doc_str[] = {
"set ical(doc) {",
#include "icaldoc.gen"
"}",
0
};

#include "left.xbm"
#include "right.xbm"

// Is Tk available?
static int have_tk;

// Was a script specified on the command line?
static int have_script;

/* The following are defined in "tclmain.c" and "tkmain.c". */

extern "C" {
    extern char* tcl_RcFileName;
    extern char* tk_RcFileName;
    extern int tk_main(int, char*[]);
    extern int tcl_main(int, char*[]);
    extern int Tcl_Init(Tcl_Interp*);
    extern int Tk_Init(Tcl_Interp*);
}

static int eval_list(Tcl_Interp*, char** list);

int
main(int argc, char* argv[]) {
    // XXX Hacky scanning of argument list to figure out whether
    // or not Tk is needed, and also if a script is specified on the
    // command line.

    have_script = 0;
    have_tk = (getenv("DISPLAY") != 0);

    for (int i = 1; i < argc; i++) {
	if (strcmp(argv[i], "-display") == 0) {
	    have_tk = 1;
	    continue;
	}
	if (strcmp(argv[i], "-list") == 0) {
	    have_tk = 0;
	    continue;
	}
	if (strcmp(argv[i], "-show") == 0) {
	    have_tk = 0;
	    continue;
	}
	if (strcmp(argv[i], "-nodisplay") == 0) {
	    have_tk = 0;
	    continue;
	}
	if (strcmp(argv[i], "-f") == 0) {
	    have_script = 1;
	    continue;
	}
    }

    // Strip out processed "-nodisplay" arguments
    int j = 1;
    for (i = 1; i < argc; i++) {
	if (strcmp(argv[i], "-nodisplay") == 0) continue;
	argv[j++] = argv[i];
    }
    argv[j] = 0;
    argc = j;

    if (!have_tk && have_script) {
	// If a "-f <script>" is present on the command line,
	// strip out the "-f" because tclMain does not understand it.
	for (i = 1; i < argc-1; i++) {
	    if (strcmp(argv[i], "-f") != 0) continue;

	    /* Slide the rest of the arguments over */
	    /* (including the NULL in argv[argc].   */

	    for (int j = i+1; j <= argc; j++)
		argv[j-1] = argv[j];
	    argc--;
	    break;
	}
    }

    tcl_RcFileName = "~/.tclrc";
    tk_RcFileName  = "~/.wishrc";

    if (have_tk)
	return tk_main(argc, argv);
    else
	return tcl_main(argc, argv);
}

int MyTcl_Init(Tcl_Interp* tcl) {
#ifdef STANDALONE
    return (eval_list(tcl, tcl_lib_str));
#else
    return (Tcl_Init(tcl));
#endif
}

int MyTk_Init(Tcl_Interp* tcl) {
    if (!have_tk) return TCL_OK;
#ifdef STANDALONE
    return (eval_list(tcl, tk_lib_str));
#else
    return (Tk_Init(tcl));
#endif
}

/*
 * Beep.
 *
 *	wbeep [-100..100]
 */
static int beep_proc(ClientData c, Tcl_Interp* tcl,int argc,char* argv[]){
    Tk_Window tkwin = (Tk_Window) c;

    if (argc != 2) {
	Tcl_SetResult(tcl, "illegal number of arguments to wbeep", TCL_STATIC);
	return TCL_ERROR;
    }

    int i;
    if (Tcl_GetInt(tcl, argv[1], &i) != TCL_OK) {
	return TCL_ERROR;
    }

    if ((i < -100) || (i > 100)) {
	Tcl_SetResult(tcl, "invalid volume", TCL_STATIC);
	return TCL_ERROR;
    }

    XBell(Tk_Display(tkwin), i);

    Tcl_SetResult(tcl, "", TCL_STATIC);
    return TCL_OK;
}

int Ical_Init(Tcl_Interp* tcl) {
    if (have_tk) {
	/* Load necessary Tk support code */
	Tk_Window mainWindow = Tk_MainWindow(tcl);

	// Set application class to Ical.
	Tk_SetClass(mainWindow, "Ical");

	// Create command to beep.
	Tcl_CreateCommand(tcl, "wbeep", beep_proc, (ClientData) mainWindow, 0);

	// Create "left" bitmap.
	if (Tk_DefineBitmap(tcl, Tk_GetUid("left_arrow"),
			    left_bits, left_width, left_height) != TCL_OK)
	    return TCL_ERROR;

	// Create "right" bitmap.
	if (Tk_DefineBitmap(tcl, Tk_GetUid("right_arrow"),
			    right_bits, right_width, right_height) != TCL_OK)
	    return TCL_ERROR;
    }

    // Set-up postscript prolog
    if (eval_list(tcl, psheader_str) != TCL_OK)
	return TCL_ERROR;

    // Set-up documentation
    if (eval_list(tcl, ical_doc_str) != TCL_OK)
	return TCL_ERROR;

    // Non-Tk ical commands
    Tcl_CreateCommand(tcl, "calendar",     Cmd_CreateCalendar,	NULL, NULL);
    Tcl_CreateCommand(tcl, "notice",       Cmd_CreateNotice,	NULL, NULL);
    Tcl_CreateCommand(tcl, "appointment",  Cmd_CreateAppt,	NULL, NULL);
    Tcl_CreateCommand(tcl, "date",         Cmd_Date,		NULL, NULL);
    Tcl_CreateCommand(tcl, "time",         Cmd_Time,		NULL, NULL);
    Tcl_CreateCommand(tcl, "de_monthdays", Cmd_MonthDays,	NULL, NULL);
    Tcl_CreateCommand(tcl, "de_interest",  Cmd_InterestTags,	NULL, NULL);
    Tcl_CreateCommand(tcl, "hilite_loop",  Cmd_HiliteLoop,	NULL, NULL);

#ifdef STANDALONE
    // Load tcllib files
    if (eval_list(tcl, tcllib_str) != TCL_OK)
	return TCL_ERROR;

    // Load ical library files
    if (eval_list(tcl, ical_lib_str) != TCL_OK)
	return TCL_ERROR;
#endif

    // Initialize ical stuff
    if (eval_list(tcl, ical_startup) != TCL_OK)
	return TCL_ERROR;

    if (Tcl_Eval(tcl, "ical_init") == TCL_ERROR)
	return TCL_ERROR;

    return TCL_OK;
}

int Tcl_AppInit(Tcl_Interp* interp) {
    if (MyTcl_Init(interp) == TCL_ERROR) return TCL_ERROR;
    if (MyTk_Init(interp) == TCL_ERROR) return TCL_ERROR;
    if (Ical_Init(interp) == TCL_ERROR) return TCL_ERROR;

    if (!have_script) {
	// Perform default initialization
	if (have_tk) {
	    if (Tcl_Eval(interp, "ical_tk_script") == TCL_ERROR)
		return TCL_ERROR;

	    // Do not bother returning to tkMain because it
	    // will try to read from standard input.

	    Tk_MainLoop();
	    Tcl_Eval(interp, "exit");
	    exit(1);
	}

	// Default tcl code
	return Tcl_Eval(interp, "ical_no_tk_script");
    }

    return TCL_OK;
}

// Concatenate list of lines into one string and "Tcl_Eval" it.
static int eval_list(Tcl_Interp* tcl, char** list) {
    // Get buffer size
    int count = 0;
    for (int i = 0; list[i] != 0; i++) {
	count += strlen(list[i]);
	count++;			// Space for newline
    }

    // Copy lines into buffer
    int index = 0;
    char* buf = new char[count+1];
    for (i = 0; list[i] != 0; i++) {
	strcpy(buf+index, list[i]);
	index += strlen(list[i]);
	buf[index] = '\n';
	index++;
    }
    buf[index] = '\0';

    int result = Tcl_Eval(tcl, buf);
    delete [] buf;
    return result;
}
