/*!
    @file     SharedSQL_Command.hpp
    @ingroup  SharedSQL
    @author   DirkT
    @brief    SQLCommands cached by the SharedSQL_CommandCache
    @see            

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
\endif
*/

#ifndef SHAREDSQL_COMMAND_HPP
#define SHAREDSQL_COMMAND_HPP

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "SQLManager/SharedSQL/SharedSQL_ICommand.hpp"
#include "SQLManager/SharedSQL/SharedSQL_CommandStatistics.hpp"
#include "SQLManager/SharedSQL/SharedSQL_ICommandCache.hpp"

#include "SQLManager/SharedSQL/SharedSQL_Plan.hpp"
#include "SQLManager/SharedSQL/SharedSQL_Types.hpp"

#include "SAPDBCommon/SAPDB_Types.hpp"

/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

/*---------------------------------------------------------------------------*/
/*!
    @class          SharedSQL_CachedCommand
    @brief          CachedCommand
    @copydoc        SharedSQL_ICachedCommand
 */
class SharedSQL_CachedCommand : public SharedSQL_ICachedCommand
{
public:
    /*!
        @brief  Constructor
        @param  ID           [in]
        @param  Hash         [in]
        @param  Slot         [in]
        @param  Cache        [in]
        @param  Allocator    [in]
        @param  Context      [in]
        @param  Stmt         [in]
        @param  Internal     [in]
        @param  SpinLockPool [in]
    */
	SharedSQL_CachedCommand( SharedSQL_CommandID ID, SharedSQL_HashValue Hash, SharedSQL_HashValue Slot, 
							    SharedSQL_ICommandCache& Cache, SAPDBMem_IRawAllocator& Allocator,
                                SharedSQL_SQLContext& Context, SharedSQL_SQLStmt& Stmt, SAPDB_Bool Internal,
                                RTESync_BaseSpinlockPool& SpinLockPool );

    /*!
        @brief  Destructor
    */
    ~SharedSQL_CachedCommand( void );


    /*!
        @brief  CompletelyConstructed
        @return ok
    */
    SAPDB_Bool CompletelyConstructed( void );

    virtual SAPDB_Int4 GetPrepareCount();
    virtual void IncPrepareCount();
    virtual void DecPrepareCount();

    virtual SAPDB_Int4 GetParseIDCount();
    virtual void IncParseIDCount();
    virtual void DecParseIDCount();

    virtual SAPDB_Int4 GetExecuteCount();
    virtual void IncExecuteCount();
    virtual void DecExecuteCount();

    virtual SharedSQL_CommandStatus GetStatus( void );
    virtual void SetStatus( SharedSQL_CommandStatus Status );

    virtual SAPDB_Byte GetFlgExtern( void );
    virtual void SetFlgExtern( SAPDB_Byte FlgExtern );

    virtual SAPDB_Bool GetFlgLRUListOld( void );
    virtual void SetFlgLRUListOld( SAPDB_Bool Flg );

    virtual SharedSQL_SQLStmt&               SQLStmt( void );
    virtual SharedSQL_SQLContext&            SQLContext( void );
    virtual SharedSQL_IPlan& Plan( void ) ;

    virtual SharedSQL_ICommandCache& CommandCache( void );

    virtual SharedSQL_CommandStatistics&     Statistics( void );
    virtual void GetCommandInfo( SharedSQL_CommandInfo& I );
    virtual void GetCommandInfoReset( SharedSQL_CommandInfo& I );
    virtual void ResetStatisticsInfo( void );

    virtual SharedSQL_CommandID GetCommandID( void );
    virtual SharedSQL_HashValue	GetHashValue( void );
    virtual SAPDB_Bool IsInternal( void );
    virtual void SetInternal(SAPDB_Bool Internal);

    virtual const SharedSQL_SQLStmt GetSQLStmt( void );
    virtual void EnterLock( bool exclusive );
    virtual bool TryEnterLock( bool exclusive );
    virtual void LeaveLock( bool exclusive );

    virtual SharedSQL_ICachedCommand*        NextHashCommand( void ) const ;

    virtual void IncCacheAccessCountExecute( void );
    virtual void IncCacheSuccessfulAccessCountExecute( void );
    virtual void IncCacheUnSuccessfulAccessCountExecute( void );


    /*!
        @brief  NextHashCommand
        @return
    */
    SharedSQL_CachedCommand*&        NextHashCommand( void );

    /*!
        @brief  PrevHashCommand
        @return
    */
    SharedSQL_CachedCommand*&        PrevHashCommand( void );

    /*!
        @brief  NextLRUCommand
        @return
    */
    SharedSQL_CachedCommand*&        NextLRUCommand( void );

    /*!
        @brief  PrevLRUCommand
        @return
    */
    SharedSQL_CachedCommand*&        PrevLRUCommand( void );

private:
    /// ...
    void FillCommandInfo( SharedSQL_CommandInfo& I );

    /// Pointer to the next command within the same hash table slot 
	SharedSQL_CachedCommand*	        mNextHashCommand;
    /// Pointer to the previous command within the same hash table slot 
	SharedSQL_CachedCommand*	        mPrevHashCommand;
    /// Pointer to the next command within the lru list 
	SharedSQL_CachedCommand*	        mNextLRUCommand;
    /// Pointer to the previous command within the lru list 
	SharedSQL_CachedCommand*	        mPrevLRUCommand;
    /// Unique ID to identify the command
	SharedSQL_CommandID					mCommandID;
    /// Hash value of the command
	SharedSQL_HashValue					mHashValue;
    /// Hash table slot that the command can be found in
    SharedSQL_HashValue                 mSlot;

    /// Reference to the command cache the command is stored in
	SharedSQL_ICommandCache&			mCommandCache;

    /// ...
    RTESync_IRWRegion                  *mpCommandLock;          

    /// Status of the command (New, Invalid, Dropped, Preparing or Prepared) 
    volatile SharedSQL_CommandStatus	mStatus;
    /// Flag to indicated wheater or not the statement string is unloaded to the temporary file
	volatile SAPDB_Byte			    	mFlgExtern;
    /// Number of SharedSQL_PrepareHandle that refer to this command; 
    SAPDB_Int4                          mPrepareCount; 	
    /// Number of SharedSQL_ParseIDHandle that refer to this command; if this is >0 the command must not be removed from the cache
    SAPDB_Int4                          mParseIDCount;
    /// Number of SharedSQL_ExecuteHandle that refer to this command; if the is >0 the plan must not be dropped
    SAPDB_Int4                          mExecuteCount;

    /// The SQL statement
    SharedSQL_SQLStmt					mSQLStmt;
    /// The SQL context
	SharedSQL_SQLContext				mSQLContext;
    /// The prepared execution plan
    SharedSQL_Plan                      mPlan;

    /// Flag to indicate wheather or not the command is an internal one
    SAPDB_Bool                          mInternal;

    /// Flag for LRU list to indicate that this command belongs to the OLD part of the list
    SAPDB_Bool                          mLRUOldFlg;

    /// Statistical information about the command (used by SystemView COMMANDSTATISTICS)
    SharedSQL_CommandStatistics        	mStatistics;
    /// Reset version of the statistical information (used by SystemView COMMANDSTATISTICSRESET)
    SharedSQL_CommandStatistics        	mStatisticsReset;
    //
};  // SharedSQL_CachedCommand

/* ------------------------------------------------------------------------- */

#endif  /* SHAREDSQL_COMMAND_HPP */

