/*!
 ========== licence begin GPL
    Copyright (C) 2002-2003 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
 */
package com.sap.dbtech.jdbc.translators;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.SQLException;

import com.sap.dbtech.jdbc.exceptions.JDBCDriverException;
import com.sap.dbtech.jdbc.exceptions.SQLExceptionSapDB;
import com.sap.dbtech.jdbc.exceptions.StreamIOException;
import com.sap.dbtech.jdbc.packet.DataPart;
import com.sap.dbtech.util.MessageKey;
import com.sap.dbtech.util.MessageTranslator;
import com.sap.dbtech.util.RawByteReader;
import com.sap.dbtech.util.StructuredMem;
import com.sap.dbtech.vsp00.DataType;
import com.sap.dbtech.vsp00.LongDesc;

/**
 * Translator for LONG arguments of DB Procedures written in the internal
 * procedural language.
 */
public class ProcedureStreamTranslator extends DBTechTranslator
{
    public ProcedureStreamTranslator(int mode, int ioType, int dataType,
            int len, int ioLen, int bufpos)
    {
        super(mode, ioType, dataType, len, ioLen, bufpos);
    }

    public int getColumnDisplaySize()
    {
       return Integer.MAX_VALUE;
    }

    public Object transBinaryStreamForInput(InputStream stream, int length)
            throws SQLException
    {
        if (isBinary()) {
            if (stream == null) {
                return null;
            }
            return new BinaryProcedurePutval(this, stream, length);
        } else {
            throw new SQLExceptionSapDB(MessageTranslator
                    .translate(MessageKey.ERROR_CONVERSION_BYTESTREAM));
        }
    }
    
    public Object transStringForInput(String val)
	throws SQLException	{
	if (val == null) {
		return null;
	} else {
	    if(isAscii()) {
			return new AsciiProcedurePutval(this, com.sap.dbtech.util.StringUtil.bytes_iso8859_1(val));
		} else {
		    throw new SQLExceptionSapDB(MessageTranslator.translate(MessageKey.ERROR_CONVERSION_STRINGSTREAM));
		}
	}
}

    public Object transCharacterStreamForInput (java.io.Reader reader, int length) 
		throws SQLException 
	{
        if(isAscii()) {
            if(reader ==  null) {
                return null;
            } else {
                return new AsciiProcedurePutval(this, new ReaderInputStream(reader, false), length);            
            }
        } else {
            throw new SQLExceptionSapDB(MessageTranslator.translate(MessageKey.ERROR_CONVERSION_STRINGSTREAM));            
        }
	}
    
    public Object transBytesForInput(byte[] val) throws SQLException {
		if(val == null) {
			return transBinaryStreamForInput(null, -1);
		} else {
			return transBinaryStreamForInput(new ByteArrayInputStream(val), -1);
		}
	}

    protected void putSpecific(DataPart dataPart, Object data)
            throws SQLException
    {
        AbstractProcedurePutval putval = (AbstractProcedurePutval) data;
        putval.putDescriptor(dataPart);
    }

    protected Object transSpecificForInput(Object obj) throws SQLException
    {
        return null;
    }

    private boolean isAscii()
    {
        return dataType == DataType.Dstra || dataType == DataType.Dlonga;
    }

    private boolean isBinary()
    {
        return dataType == DataType.Dstrb || dataType == DataType.Dlongb;
    }

    
    private InputStream getStream(SQLParamController controller,
    							  StructuredMem mem,
								  StructuredMem longdata) throws SQLException
    {
        java.io.InputStream result = null;
        AbstractGetval getval = null;
        byte [] descriptor;

        if (!this.isNull (controller, mem)) {
            descriptor = mem.getBytes (this.bufpos, this.logicalLength);
            // return also NULL if the LONG hasn't been touched.
			if(descriptorIsNull(descriptor)) {
				return null;
			}
            if (com.sap.dbtech.jdbc.DriverSapDB.isJDBC20) {
                getval = new GetvalLob (controller.getConnectionSapDB (),
                    descriptor, longdata, this.dataType);
            } else {
                getval = new Getval (controller.getConnectionSapDB (),
                    descriptor, longdata, this.dataType);
            }
            result = getval.getAsciiStream ();
        }
        return result;
    }

	public InputStream getAsciiStream(SQLParamController controller,
			StructuredMem mem, StructuredMem longData) throws SQLException {
			return getStream(controller, mem, longData);
	}
	
	public InputStream getBinaryStream(SQLParamController controller,
			StructuredMem mem, StructuredMem longData) throws SQLException {
		if(isBinary()) {
			return getStream(controller, mem, longData);
		} else {
	        throw new ConversionExceptionSapDB (
	                MessageTranslator.translate(MessageKey.ERROR_BINARYREADFROMLONG)
	             );
		}
	}

	private GetvalLob getLob(SQLParamController controller, StructuredMem mem,
			StructuredMem longData) throws SQLException {
		GetvalLob result = null;
		byte[] descriptor;

		if (!this.isNull(controller, mem)) {
			descriptor = mem.getBytes(this.bufpos, this.logicalLength);
			if(descriptorIsNull(descriptor)) {
				return null;
			}
			result = new GetvalLob(controller.getConnectionSapDB(), descriptor,
					longData, this.dataType);
		}
		return result;
	}
	
	public Blob getBlob(SQLParamController controller, StructuredMem mem,
			StructuredMem longData) throws SQLException {
		return this.getLob (controller, mem, longData);
	}
	
	public byte getByte(SQLParamController controller, StructuredMem mem)
			throws SQLException {
		byte[] result = null;
		if (this.isNull(controller, mem))
			return 0;
		else
			result = this.getBytes(controller, mem);
		return result[0];
	}

	/**
	 * 
	 * @return byte []
	 */
	public byte[] getBytes(SQLParamController controller, StructuredMem mem)
			throws SQLException {
		java.io.InputStream blobStream;
		java.io.ByteArrayOutputStream tmpStream;

		blobStream = this.getBinaryStream(controller, mem, controller
				.getReplyData());
		if (blobStream == null) {
			return null;
		}
		try {
			final int bufSizeC = 4096;
			byte[] buf = new byte[bufSizeC];
			int readLen;

			tmpStream = new java.io.ByteArrayOutputStream();
			readLen = blobStream.read(buf);
			while (readLen > 0) {
				tmpStream.write(buf, 0, readLen);
				if (readLen < bufSizeC) {
					break;
				}
				readLen = blobStream.read(buf);
			}
		} catch (com.sap.dbtech.jdbc.exceptions.StreamIOException sqlExc) {
			throw sqlExc.getSqlException();
		} catch (java.io.IOException ioExc) {
			throw new com.sap.dbtech.jdbc.exceptions.JDBCDriverException(ioExc
					.getMessage(), controller);
		}
		return tmpStream.toByteArray();
	}

	public Reader getCharacterStream(SQLParamController controller,
			StructuredMem mem, StructuredMem longData) throws SQLException {
			InputStream byteStream = this.getAsciiStream(controller, mem,
					longData);
			if (byteStream == null) {
				return null;
			} else {
				return new RawByteReader(byteStream);
			}
	}
	public Clob getClob(SQLParamController controller, StructuredMem mem,
			StructuredMem longData) throws SQLException {
			return this.getLob(controller, mem, longData);
	}

	public String getString(SQLParamController controller, StructuredMem mem)
			throws SQLException {
        final int bufSizeC = 4096;
        Reader reader;
        StringBuffer result = new StringBuffer ();

        reader = this.getCharacterStream (controller, mem, controller.getReplyData());
        if (reader == null) {
            return null;
        }
        try {
            char [] buf = new char [4096];
            int charsRead;

            while ((charsRead = reader.read (buf)) > 0) {
                result.append (new String (buf, 0, charsRead));
                if (charsRead < bufSizeC) {
                    break;
                }
            }
        }
        catch (StreamIOException streamExc) {
            throw streamExc.getSqlException ();
        }
        catch (IOException exc) {
            throw new JDBCDriverException (exc.getMessage (), controller);
        }
        return result.toString ();
	}

	protected boolean descriptorIsNull(byte[] descriptor) {
        if(descriptor[LongDesc.State_O] == LongDesc.StateStream_C) {
        	return true;
       } else {
       		return false;
       }
	}
}
