/*----------------------------------------------------------------------------

   libtunepimp -- The MusicBrainz tagging library.
                  Let a thousand taggers bloom!

   Copyright (C) Robert Kaye 2003

   This file is part of libtunepimp.

   libtunepimp is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   libtunepimp is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with libtunepimp; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   $Id: lookuptools.cpp,v 1.22 2004/03/12 00:08:19 robert Exp $

----------------------------------------------------------------------------*/
#ifdef WIN32
#if _MSC_VER == 1200
#pragma warning(disable:4786)
#endif
#endif

#include <stdio.h>
#include <musicbrainz/mb_c.h>
#include "lookuptools.h"
#include "tunepimp.h"
#include "metadata.h"

#define DB printf("%s:%d\n", __FILE__, __LINE__);

Lookup::Lookup(TunePimp *pimp)
{
    this->pimp = pimp;

    pimp->getProxy(proxyServer, proxyPort);
    pimp->getServer(server, port);
    useUTF8 = pimp->getUseUTF8();
}

LookupStatus LookupTRM::lookup(void)
{
    musicbrainz_t  o;
    char          *args[7];
    int            ret, trackNum;
    char           error[255], data[255], trackURI[256],
                   artistURI[256], albumURI[256];
    char           temp[100], duration[100], status[100];

    result.clear();

    // Create the musicbrainz object, which will be needed for subsequent calls
    o = mb_New();

    // Tell the client library to return data in ISO8859-1 and not UTF-8
    mb_UseUTF8(o, useUTF8);

    // Tell the server to return 1 levels of data
    mb_SetDepth(o, 1);
    mb_SetDebug(o, (int)pimp->context.getDebug());

    if (proxyServer.size() > 0 && proxyPort != 0)
        mb_SetProxy(o, (char *)proxyServer.c_str(), proxyPort);

    if (server.size() > 0 && port != 0)
        mb_SetServer(o, (char *)server.c_str(), port);

    sprintf(temp, "%d", mdata.trackNum);
    sprintf(duration, "%ld", mdata.duration);
    args[0] = (char *)trmId.c_str();
    args[1] = (char *)mdata.artist.c_str();
    args[2] = (char *)mdata.album.c_str();
    args[3] = (char *)mdata.track.c_str();
    args[4] = temp;
    args[5] = duration;
    args[6] = NULL;

    ret = mb_QueryWithArgs(o, MBQ_TrackInfoFromTRMId, args);
    if (!ret)
    {
        mb_GetQueryError(o, error, 256);
        err = string(error);
        mb_Delete(o);
        return eLookupError;
    }

    mb_GetResultData(o, MBE_GetStatus, status, 20);
    isFuzzy = !strcmp(status, "Fuzzy");

    for(int i = 1;; i++)
    {
        // Select the first track from the track list
        mb_Select(o, MBS_Rewind);
        if (!mb_Select1(o, MBS_SelectTrack, i))
            break;

        mdata.clear();

        mb_GetResultData(o, MBE_TrackGetTrackId, trackURI, 256);
        mb_GetIDFromURL(o, trackURI, data, 256);
        mdata.trackId = string(data);

        mb_GetResultData(o, MBE_TrackGetArtistId, artistURI, 256);
        mb_GetIDFromURL(o, artistURI, data, 256);
        mdata.artistId = string(data);

        // Extract the artist name from the track
        if (mb_GetResultData(o, MBE_TrackGetArtistName, data, 256))
           mdata.artist = string(data);

        // Extract the artist name from the track
        if (mb_GetResultData(o, MBE_TrackGetArtistSortName, data, 256))
           mdata.sortName = string(data);

        // Extract the track name
        if (mb_GetResultData(o, MBE_TrackGetTrackName, data, 256))
           mdata.track = string(data);

        // Extract the track duration
        mdata.duration = mb_GetResultInt(o, MBE_TrackGetTrackDuration);
        mdata.numTRMIds = mb_GetResultInt(o, MBE_GetNumTrmids);

        mb_Select(o, MBS_SelectTrackAlbum);

        // Extract the track number
        trackNum = mb_GetOrdinalFromList(o, MBE_AlbumGetTrackList, trackURI);
        if (trackNum > 0 && trackNum < 100)
           mdata.trackNum = trackNum;

        // Extract the album name from the track
        if (mb_GetResultData(o, MBE_AlbumGetAlbumName, data, 256))
           mdata.album = string(data);

        if (mb_GetResultData(o, MBE_AlbumGetAlbumType, data, 256))
        {
            char temp[256];

            mb_GetFragmentFromURL(o, data, temp, 256);
            mdata.albumType = convertToAlbumType(temp);
        }
        if (mb_GetResultData(o, MBE_AlbumGetAlbumStatus, data, 256))
        {
            char temp[256];

            mb_GetFragmentFromURL(o, data, temp, 256);
            mdata.albumStatus = convertToAlbumStatus(temp);
        }
       
        int numDates = mb_GetResultInt(o, MBE_AlbumGetNumReleaseDates);
        for(int j = 1; j <= numDates; j++)
        {
            // Select the first release date
            if (mb_Select1(o, MBS_SelectReleaseDate, j))
            {
                // Pull back the release date and release country
                if (mb_GetResultData(o, MBE_ReleaseGetDate, data, 256))
                {
                    int month, day, year;

                    if (sscanf(data, "%d-%d-%d", &year, &month, &day) == 3)
                    {
                        if (mdata.releaseYear == 0 || 
                            mdata.releaseYear > year ||
                           (mdata.releaseYear == year && mdata.releaseMonth > month) ||
                           (mdata.releaseYear == year && mdata.releaseMonth == month && mdata.releaseDay > day))
                        {
                            mdata.releaseYear = year;
                            mdata.releaseMonth = month;
                            mdata.releaseDay = day;
                            mb_GetResultData(o, MBE_ReleaseGetCountry, data, 256);
                            mdata.releaseCountry = data;
                        }
                    }
                }
                mb_Select(o, MBS_Back);
            }
            else
                break;
        }

        mb_GetResultData(o, MBE_AlbumGetAlbumArtistId, albumURI, 256);
        mb_GetIDFromURL(o, albumURI, data, 256);
        mdata.variousArtist = strcmp(MBI_VARIOUS_ARTIST_ID, data) == 0;

        mb_GetResultData(o, MBE_AlbumGetAlbumId, albumURI, 256);
        mb_GetIDFromURL(o, albumURI, data, 256);
        mdata.albumId = string(data);

        result.push_back(mdata);
    }

	 // and clean up the musicbrainz object
    mb_Delete(o);

    return (result.size() > 0) ? eFound : eNotFound;
}

//---------------------------------------------------------------------------

LookupStatus  LookupTrack::lookup(void)
{
    musicbrainz_t  o;
    char          *args[3];
    int            ret;
    char           error[255], data[255];

    // -----------------------------------------------------------------------------------------
    // NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE
    // This class is currently not being used and therefore it is not up to date!
    // NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE NOTE
    // -----------------------------------------------------------------------------------------

    result.clear();

    // Create the musicbrainz object, which will be needed for subsequent calls
    o = mb_New();

    // Tell the client library to return data in ISO8859-1 and not UTF-8
    mb_UseUTF8(o, useUTF8);

    // Tell the server to return 1 levels of data
    mb_SetDepth(o, 1);
    mb_SetDebug(o, (int)pimp->context.getDebug());

    if (proxyServer.size() > 0 && proxyPort != 0)
        mb_SetProxy(o, (char *)proxyServer.c_str(), proxyPort);

    if (server.size() > 0 && port != 0)
        mb_SetServer(o, (char *)server.c_str(), port);

    args[0] = (char *)trackId.c_str();
    args[1] = (char *)albumId.c_str();
    args[2] = NULL;

    ret = mb_QueryWithArgs(o, MBQ_QuickTrackInfoFromTrackId, args);
    if (!ret)
    {
        mb_GetQueryError(o, error, 256);
        err = string(error);
        mb_Delete(o);
        return eLookupError;
    }

    // Did we get what we wanted?
    if (!mb_DoesResultExist(o, MBE_QuickGetArtistName))
    {
        mb_Delete(o);
        return eNotFound;
    }

    // Extract the artist name from the track
    if (mb_GetResultData(o, MBE_QuickGetArtistName, data, 256))
       result.artist = string(data);

    // Extract the album name from the track
    if (mb_GetResultData(o, MBE_QuickGetAlbumName, data, 256))
       result.album = string(data);

    // Extract the track name
    if (mb_GetResultData(o, MBE_QuickGetTrackName, data, 256))
       result.track = string(data);

    // Extract the track id
    if (mb_GetResultData(o, MBE_QuickGetTrackId, data, 256))
       result.trackId = string(data);

    // Extract the track number
    result.trackNum = mb_GetResultInt(o, MBE_QuickGetTrackNum);
    if (result.trackNum < 1 && result.trackNum > 99)
       result.trackNum = 0;

    // Extract the track duration
    result.duration = mb_GetResultInt(o, MBE_QuickGetTrackDuration);
    result.variousArtist = false;

	if (mb_GetResultData(o, "http://musicbrainz.org/mm/mm-2.1#albumArtist", data, 256))
        result.variousArtist = strcmp(MBI_VARIOUS_ARTIST_ID, data) == 0;
	else
		result.variousArtist = false;

    // and clean up the musicbrainz object
    mb_Delete(o);

    return eFound;
}
//---------------------------------------------------------------------------

LookupStatus SubmitTRM::lookup(void)
{
    musicbrainz_t  o;
    int            ret;
    char           error[255];

    if (user.empty() || passwd.empty())
    {
        err = "Incomplete user/password provided.";
        return eSubmitError;
    }

    // Create the musicbrainz object, which will be needed for subsequent calls
    o = mb_New();

    // Tell the client library to return data in ISO8859-1 and not UTF-8
    mb_UseUTF8(o, useUTF8);

    if (proxyServer.size() > 0 && proxyPort != 0)
        mb_SetProxy(o, (char *)proxyServer.c_str(), proxyPort);

    if (server.size() > 0 && port != 0)
        mb_SetServer(o, (char *)server.c_str(), port);

    ret = mb_Authenticate(o, (char *)user.c_str(), (char *)passwd.c_str());
    if (!ret)
    {
        mb_GetQueryError(o, error, 256);
        err = string(error);
        mb_Delete(o);
        return eSubmitError;
    }

    ret = mb_Query(o, (char *)query.c_str());
    if (!ret)
    {
        mb_GetQueryError(o, error, 256);
        err = string(error);
        mb_Delete(o);
        return eSubmitError;
    }

    // and clean up the musicbrainz object
    mb_Delete(o);

    return eSubmitOk;
}
