/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-digest-offset: 8 -*- */
/*
 * soup-auth-digest.c: HTTP Digest Authentication
 *
 * Copyright (C) 2001-2003, Ximian, Inc.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "soup-auth-digest.h"
#include "soup-headers.h"
#include "soup-md5-utils.h"
#include "soup-message.h"
#include "soup-misc.h"
#include "soup-uri.h"

static void construct (SoupAuth *auth, const char *header);
static GSList *get_protection_space (SoupAuth *auth, const SoupUri *source_uri);
static const char *get_realm (SoupAuth *auth);
static void authenticate (SoupAuth *auth, const char *username, const char *password);
static gboolean is_authenticated (SoupAuth *auth);
static char *get_authorization (SoupAuth *auth, SoupMessage *msg);

typedef enum {
	QOP_NONE     = 0,
	QOP_AUTH     = 1 << 0,
	QOP_AUTH_INT = 1 << 1
} QOPType;

typedef enum {
	ALGORITHM_MD5      = 1 << 0,
	ALGORITHM_MD5_SESS = 1 << 1
} AlgorithmType;

struct SoupAuthDigestPrivate {
	char          *user;
	guchar         hex_a1[33];

	/* These are provided by the server */
	char          *realm;
	char          *nonce;
	QOPType        qop_options;
	AlgorithmType  algorithm;
	char          *domain;

	/* These are generated by the client */
	char          *cnonce;
	int            nc;
	QOPType        qop;
};

#define PARENT_TYPE SOUP_TYPE_AUTH
static SoupAuthClass *parent_class;

static void
init (GObject *object)
{
	SoupAuthDigest *digest = SOUP_AUTH_DIGEST (object);

	digest->priv = g_new0 (SoupAuthDigestPrivate, 1);
}

static void
finalize (GObject *object)
{
	SoupAuthDigest *digest = SOUP_AUTH_DIGEST (object);

	if (digest->priv->user)
		g_free (digest->priv->user);
	if (digest->priv->realm)
		g_free (digest->priv->realm);
	if (digest->priv->nonce)
		g_free (digest->priv->nonce);
	if (digest->priv->domain)
		g_free (digest->priv->domain);
	if (digest->priv->cnonce)
		g_free (digest->priv->cnonce);

	g_free (digest->priv);

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
class_init (GObjectClass *object_class)
{
	SoupAuthClass *auth_class = SOUP_AUTH_CLASS (object_class);

	parent_class = g_type_class_ref (PARENT_TYPE);

	auth_class->scheme_name = "Digest";

	auth_class->get_protection_space = get_protection_space;
	auth_class->get_realm = get_realm;
	auth_class->construct = construct;
	auth_class->authenticate = authenticate;
	auth_class->is_authenticated = is_authenticated;
	auth_class->get_authorization = get_authorization;

	object_class->finalize = finalize;
}

SOUP_MAKE_TYPE (soup_auth_digest, SoupAuthDigest, class_init, init, PARENT_TYPE)

typedef struct {
	char *name;
	guint type;
} DataType;

static DataType qop_types[] = {
	{ "auth",     QOP_AUTH     },
	{ "auth-int", QOP_AUTH_INT }
};

static DataType algorithm_types[] = {
	{ "MD5",      ALGORITHM_MD5      },
	{ "MD5-sess", ALGORITHM_MD5_SESS }
};

static guint
decode_data_type (DataType *dtype, const char *name)
{
        int i;

	if (!name)
		return 0;

        for (i = 0; dtype[i].name; i++) {
                if (!g_strcasecmp (dtype[i].name, name))
			return dtype[i].type;
        }

	return 0;
}

static inline guint
decode_qop (const char *name)
{
	return decode_data_type (qop_types, name);
}

static inline guint
decode_algorithm (const char *name)
{
	return decode_data_type (algorithm_types, name);
}

static void
construct (SoupAuth *auth, const char *header)
{
	SoupAuthDigest *digest = SOUP_AUTH_DIGEST (auth);
	GHashTable *tokens;
	char *tmp, *ptr;

	header += sizeof ("Digest");

	tokens = soup_header_param_parse_list (header);
	if (!tokens)
		return;

	digest->priv->nc = 1;
	/* We're just going to do qop=auth for now */
	digest->priv->qop = QOP_AUTH;

	digest->priv->realm = soup_header_param_copy_token (tokens, "realm");
	digest->priv->domain = soup_header_param_copy_token (tokens, "domain");
	digest->priv->nonce = soup_header_param_copy_token (tokens, "nonce");

	tmp = soup_header_param_copy_token (tokens, "qop");
	ptr = tmp;

	while (ptr && *ptr) {
		char *token;

		token = soup_header_param_decode_token ((char **)&ptr);
		if (token)
			digest->priv->qop_options |= decode_qop (token);
		g_free (token);

		if (*ptr == ',')
			ptr++;
	}
	g_free (tmp);

	tmp = soup_header_param_copy_token (tokens, "algorithm");
	digest->priv->algorithm = decode_algorithm (tmp);
	g_free (tmp);

	soup_header_param_destroy_hash (tokens);
}

static GSList *
get_protection_space (SoupAuth *auth, const SoupUri *source_uri)
{
	SoupAuthDigest *digest = SOUP_AUTH_DIGEST (auth);
	GSList *space = NULL;
	SoupUri *uri;
	char *domain, *d, *lasts, *dir, *slash;

	if (!digest->priv->domain || !*digest->priv->domain) {
		/* If no domain directive, the protection space is the
		 * whole server.
		 */
		return g_slist_prepend (NULL, g_strdup (""));
	}

	domain = g_strdup (digest->priv->domain);
	for (d = strtok_r (domain, " ", &lasts); d; d = strtok_r (NULL, " ", &lasts)) {
		if (*d == '/')
			dir = g_strdup (d);
		else {
			uri = soup_uri_new (d);
			if (uri && uri->protocol == source_uri->protocol &&
			    uri->port == source_uri->port &&
			    !strcmp (uri->host, source_uri->host))
				dir = g_strdup (uri->path);
			else
				dir = NULL;
			if (uri)
				soup_uri_free (uri);
		}

		if (dir) {
			slash = strrchr (dir, '/');
			if (slash && !slash[1])
				*slash = '\0';

			space = g_slist_prepend (space, dir);
		}
	}
	g_free (domain);

	return space;
}

static const char *
get_realm (SoupAuth *auth)
{
	SoupAuthDigest *digest = SOUP_AUTH_DIGEST (auth);

	return digest->priv->realm;
}

static void
digest_hex (guchar *digest, guchar hex[33])
{
	guchar *s, *p;

	/* lowercase hexify that bad-boy... */
	for (s = digest, p = hex; p < hex + 32; s++, p += 2)
		sprintf (p, "%.2x", *s);
}

static void
authenticate (SoupAuth *auth, const char *username, const char *password)
{
	SoupAuthDigest *digest = SOUP_AUTH_DIGEST (auth);
	SoupMD5Context ctx;
	guchar d[16];
	char *bgen;

	g_return_if_fail (username != NULL);

	bgen = g_strdup_printf ("%p:%lu:%lu",
			       auth,
			       (unsigned long) getpid (),
			       (unsigned long) time (0));
	digest->priv->cnonce = soup_base64_encode (bgen, strlen (bgen));
	g_free (bgen);

	digest->priv->user = g_strdup (username);

	/* compute A1 */
	soup_md5_init (&ctx);

	soup_md5_update (&ctx, username, strlen (username));

	soup_md5_update (&ctx, ":", 1);
	if (digest->priv->realm) {
		soup_md5_update (&ctx, digest->priv->realm,
				 strlen (digest->priv->realm));
	}

	soup_md5_update (&ctx, ":", 1);
	if (password)
		soup_md5_update (&ctx, password, strlen (password));

	if (digest->priv->algorithm == ALGORITHM_MD5_SESS) {
		soup_md5_final (&ctx, d);

		soup_md5_init (&ctx);
		soup_md5_update (&ctx, d, 16);
		soup_md5_update (&ctx, ":", 1);
		soup_md5_update (&ctx, digest->priv->nonce,
				 strlen (digest->priv->nonce));
		soup_md5_update (&ctx, ":", 1);
		soup_md5_update (&ctx, digest->priv->cnonce,
				 strlen (digest->priv->cnonce));
	}

	/* hexify A1 */
	soup_md5_final (&ctx, d);
	digest_hex (d, digest->priv->hex_a1);
}

static gboolean
is_authenticated (SoupAuth *auth)
{
	SoupAuthDigest *digest = SOUP_AUTH_DIGEST (auth);

	return digest->priv->cnonce != NULL;
}

static char *
compute_response (SoupAuthDigest *digest, SoupMessage *msg)
{
	guchar hex_a2[33], o[33];
	guchar d[16];
	SoupMD5Context md5;
	char *url;
	const SoupUri *uri;

	uri = soup_message_get_uri (msg);
	g_return_val_if_fail (uri != NULL, NULL);
	url = soup_uri_to_string (uri, TRUE);

	/* compute A2 */
	soup_md5_init (&md5);
	soup_md5_update (&md5, msg->method, strlen (msg->method));
	soup_md5_update (&md5, ":", 1);
	soup_md5_update (&md5, url, strlen (url));

	g_free (url);

	if (digest->priv->qop == QOP_AUTH_INT) {
		/* FIXME: Actually implement. Ugh. */
		soup_md5_update (&md5, ":", 1);
		soup_md5_update (&md5, "00000000000000000000000000000000", 32);
	}

	/* now hexify A2 */
	soup_md5_final (&md5, d);
	digest_hex (d, hex_a2);

	/* compute KD */
	soup_md5_init (&md5);
	soup_md5_update (&md5, digest->priv->hex_a1, 32);
	soup_md5_update (&md5, ":", 1);
	soup_md5_update (&md5, digest->priv->nonce,
			 strlen (digest->priv->nonce));
	soup_md5_update (&md5, ":", 1);

	if (digest->priv->qop) {
		char *tmp;

		tmp = g_strdup_printf ("%.8x", digest->priv->nc);

		soup_md5_update (&md5, tmp, strlen (tmp));
		g_free (tmp);
		soup_md5_update (&md5, ":", 1);
		soup_md5_update (&md5, digest->priv->cnonce,
				 strlen (digest->priv->cnonce));
		soup_md5_update (&md5, ":", 1);

		if (digest->priv->qop == QOP_AUTH)
			tmp = "auth";
		else if (digest->priv->qop == QOP_AUTH_INT)
			tmp = "auth-int";
		else
			g_assert_not_reached ();

		soup_md5_update (&md5, tmp, strlen (tmp));
		soup_md5_update (&md5, ":", 1);
	}

	soup_md5_update (&md5, hex_a2, 32);
	soup_md5_final (&md5, d);

	digest_hex (d, o);

	return g_strdup (o);
}

static char *
get_authorization (SoupAuth *auth, SoupMessage *msg)
{
	SoupAuthDigest *digest = (SoupAuthDigest *) auth;
	char *response;
	char *qop = NULL;
	char *nc;
	char *url;
	char *out;
	const SoupUri *uri;

	uri = soup_message_get_uri (msg);
	g_return_val_if_fail (uri != NULL, NULL);
	url = soup_uri_to_string (uri, TRUE);

	response = compute_response (digest, msg);

	if (digest->priv->qop == QOP_AUTH)
		qop = "auth";
	else if (digest->priv->qop == QOP_AUTH_INT)
		qop = "auth-int";
	else
		g_assert_not_reached ();

	nc = g_strdup_printf ("%.8x", digest->priv->nc);

	out = g_strdup_printf (
		"Digest username=\"%s\", realm=\"%s\", nonce=\"%s\", %s%s%s "
		"%s%s%s %s%s%s uri=\"%s\", response=\"%s\"",
		digest->priv->user,
		digest->priv->realm,
		digest->priv->nonce,

		digest->priv->qop ? "cnonce=\"" : "",
		digest->priv->qop ? digest->priv->cnonce : "",
		digest->priv->qop ? "\"," : "",

		digest->priv->qop ? "nc=" : "",
		digest->priv->qop ? nc : "",
		digest->priv->qop ? "," : "",

		digest->priv->qop ? "qop=" : "",
		digest->priv->qop ? qop : "",
		digest->priv->qop ? "," : "",

		url,
		response);

	g_free (response);
	g_free (url);
	g_free (nc);

	digest->priv->nc++;

	return out;
}
