/*
 *  Copyright (C) 2001 Jorn Baayen 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "mozilla.h"
#include "stylesheets.h"
#include "embed.h"
#include "history.h"
#include "eel-gconf-extensions.h"
#include "context.h"

#include <string.h>
#include <stdlib.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkmenuitem.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-util.h>
#include <libgnomeui/gnome-preferences.h>

/* local function prototypes */
static void stylesheets_free_stylesheet_menu_item (GtkMenuItem *mi,
						   gpointer data);
static void stylesheets_change_stylesheet_cb (GtkWidget *mi,
					      AlternateStyleSheet *css);
static void stylesheets_change_usersheet_cb (GtkWidget *mi,
					     GaleonEmbed *embed);
static gboolean stylesheets_free (gpointer key, gpointer val, gpointer data);
static void stylesheets_list (gchar *key, gpointer val, void **data);


/**
 * stylesheets_menu_build: initialize stylesheets menu
 */
void
stylesheets_menu_build (GaleonEmbed *embed)
{
	GSList *radiogroup = NULL, *usersheets, *l;
	gboolean ticked = FALSE, usercss = FALSE;
	AlternateStyleSheet *current;
	GtkWidget *css_menu, *item;
	GList *node, *list = NULL;
	GaleonWindow *window;
	gint i;

	/* sanity checks */
	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);
	
	if (!embed->mozembed || embed->load_started > 0 ||
	    !embed->wrapper || !embed->is_active)
	{
		return;
	}

	/* get stylesheet data */
	list = mozilla_get_alternate_stylesheets (embed);
	current = mozilla_get_selected_stylesheet (embed);
	usersheets =
		eel_gconf_get_string_list (CONF_RENDERING_STYLESHEETS);

	/* create menu */
	css_menu = gtk_menu_new ();
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (window->csslist), css_menu);

	/* tearoff.. */
	if (gnome_preferences_get_menus_have_tearoff ())
	{
		GtkWidget *tearoff = gtk_tearoff_menu_item_new ();
		gtk_menu_append (GTK_MENU (css_menu), tearoff);
		gtk_widget_show (tearoff);
	}

	/* process all usersheets */
	for (l = usersheets; l; l = g_slist_next (l))
	{
		gchar *str = (gchar *) l->data;
		
		if (!g_file_exists (str)) continue;

		usercss = TRUE;

		item = gtk_check_menu_item_new_with_label (g_basename (str));
		gtk_menu_append (GTK_MENU (css_menu), item);
		gtk_widget_lock_accelerators (item);
		gtk_widget_show (item);

		gtk_object_set_data_full (GTK_OBJECT (item), "filename",
					  g_strdup (str), g_free);
		gtk_signal_connect (GTK_OBJECT (item), "activate", 
				    stylesheets_change_usersheet_cb, embed);

		/* tick if active */
		if (g_hash_table_lookup (embed->usersheets, str) != NULL)
		{
			GTK_CHECK_MENU_ITEM (item)->active = TRUE;
		}
	}
	
	/* add sep. if necessary */
	if (g_list_length (list) > 0 && usercss)
	{
		context_menu_add_seperator (GTK_MENU (css_menu));
	}
	
	/* process all sheets */
	for (i = 1, node = list; node; node = node->next, i++)
	{
		AlternateStyleSheet *a = (AlternateStyleSheet *) node->data;

		if (strcmp (a->name, "") == 0)
		{
			item = gtk_radio_menu_item_new_with_label
					(radiogroup, _("Untitled"));
		}
		else
		{
			item = gtk_radio_menu_item_new_with_label
					(radiogroup, a->name);
		}
		radiogroup = gtk_radio_menu_item_group 
				(GTK_RADIO_MENU_ITEM (item));
		gtk_menu_append (GTK_MENU(css_menu), item);
		gtk_widget_lock_accelerators (item);
		gtk_widget_show (item);
		
		gtk_object_set_data (GTK_OBJECT (item), "embed", embed);
		gtk_object_set_data (GTK_OBJECT (item), "index",
				     GINT_TO_POINTER (i));
		gtk_signal_connect (GTK_OBJECT (item), "activate", 
				    stylesheets_change_stylesheet_cb, a);
		gtk_signal_connect (GTK_OBJECT (item), "destroy", 
				    stylesheets_free_stylesheet_menu_item, a);
		  
		/* tick if this is the current sheet */
		if      (current != NULL && a->sheet == current->sheet &&
		         a->type == STYLESHEET_ALTERNATE && ticked == FALSE)
		{
			GTK_CHECK_MENU_ITEM (item)->active = ticked = TRUE;
		}
		else if (current != NULL && a->type == STYLESHEET_DEFAULT &&
		         ticked == FALSE)
		{
			GTK_CHECK_MENU_ITEM (item)->active = ticked = TRUE;
		}
		else if (current == NULL && a->type == STYLESHEET_NONE &&
		         ticked == FALSE)
		{
			GTK_CHECK_MENU_ITEM (item)->active = ticked = TRUE;
		}
		else
		{
			GTK_CHECK_MENU_ITEM (item)->active = FALSE;
		}
	}

	/* set sensitivity according to whether any sheets are found */
	gtk_widget_set_sensitive (window->csslist, (g_list_length (list) > 0 || 
				   usercss));

	/* cleanup */
	if (current)
	{
		g_free (current->name);
		g_free (current);
	}
	g_list_free (list);
	g_slist_foreach (usersheets, (GFunc) g_free, NULL);
	g_slist_free (usersheets);
}

/**
 * stylesheets_free_stylesheet_menu_item: free a stylesheet menu item
 */
static void
stylesheets_free_stylesheet_menu_item (GtkMenuItem *mi, gpointer data)
{
	AlternateStyleSheet *a = (AlternateStyleSheet *) data;
	g_free (a->name);
	g_free (a);
}

/**
 * stylesheets_change_stylesheet_cb: handle a user request for loading a
 * different author stylesheet
 */
static void
stylesheets_change_stylesheet_cb (GtkWidget *mi, AlternateStyleSheet *css)
{
	GaleonEmbed *embed =
		(GaleonEmbed*) gtk_object_get_data (GTK_OBJECT (mi), "embed");
	gint index =
	       GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (mi), "index"));
	return_if_not_embed (embed);

	history_set_selected_authorcss (embed->location, index);
	mozilla_set_alternate_stylesheet (embed, css);
}

/**
 * stylesheets_load_authorsheet: load the author stylesheet saved in the 
 * history
 */
void
stylesheets_load_authorsheet (GaleonEmbed *embed)
{
	gint index = history_get_selected_authorcss (embed->location);

	if (index != -1)
	{
		GList *csslist = mozilla_get_alternate_stylesheets (embed);
		gboolean found = FALSE;
		AlternateStyleSheet *a = NULL;
		GList *l;
		gint i;

		for (i = 1, l = csslist; l; l = g_list_next (l), i++)
		{
			a = (AlternateStyleSheet *) l->data;
			if (index == i)
			{
				found = TRUE;
				break;
			}
			g_free (a->name);
			g_free (a);
		}

		if (index == 0 && g_list_length (csslist) > 0)
		{
			a = g_new0 (AlternateStyleSheet, 1);
			a->name = g_strdup ("");
			a->sheet = NULL;
			found = TRUE;
		}
			
		g_list_free (csslist);

		if (found)
		{
			mozilla_set_alternate_stylesheet (embed, a);
			g_free (a->name);
			g_free (a);
		}
	}
}

/**
 * stylesheets_load_usersheet: load the stylesheet specified by the user
 * in the prefswin
 */
void
stylesheets_load_usersheet (GaleonEmbed *embed)
{

	GSList *defaults, *l;
	gboolean keep = TRUE;
	
	if (embed->usercss_applied) return;
	embed->usercss_applied = TRUE;
	
	if (keep)
	{
		GList *list = NULL, *l;
		g_hash_table_foreach (embed->usersheets,
				      (GHFunc) stylesheets_list, (void **) &list);
		g_hash_table_foreach_remove (embed->usersheets,
					     stylesheets_free, NULL);
		for (l = list; l; l = g_list_next (l))
		{
			gchar *s = (gchar *) l->data;
			g_hash_table_insert (embed->usersheets, s,
					     mozilla_set_user_sheet (embed, s));
		}
		g_list_free (list);
	}
	else
	{
		g_hash_table_foreach_remove (embed->usersheets,
					     stylesheets_free, NULL);
	}

	defaults = eel_gconf_get_string_list
		(CONF_RENDERING_DEFAULT_STYLESHEETS);

	for (l = defaults; l; l = g_slist_next (l))
	{
		gchar *str = (gchar *) l->data;

		if (g_file_exists (str))
		{
			g_hash_table_insert (embed->usersheets, g_strdup (str),
				 mozilla_set_user_sheet (embed, str));
		}
	}
	
	g_slist_foreach (defaults, (GFunc) g_free, NULL);
	g_slist_free (defaults);
}

/**
 * stylesheets_change_usersheet_cb: usersheet item in css menu clicked
 */
static void
stylesheets_change_usersheet_cb (GtkWidget *mi,
				 GaleonEmbed *embed)
{
	gchar *filename = gtk_object_get_data (GTK_OBJECT (mi), "filename");

	if (GTK_CHECK_MENU_ITEM (mi)->active)
	{
		gpointer sheet = mozilla_set_user_sheet (embed, filename);
		g_hash_table_insert (embed->usersheets, g_strdup (filename),
				     sheet);
	}
	else
	{
		gpointer val, key;
		g_hash_table_lookup_extended (embed->usersheets, filename,
					      &key, &val);
		g_hash_table_remove (embed->usersheets, filename);
		mozilla_remove_user_sheet (embed, val);
		g_free (key);
	}
}

/**
 * stylesheets_free_usersheets: util func for freeing the usersheet hash table
 */
void
stylesheets_free_usersheets (GaleonEmbed *embed)
{
	g_hash_table_foreach_remove (embed->usersheets, stylesheets_free, NULL);
	g_hash_table_destroy (embed->usersheets);
	embed->usersheets = NULL;
}

/**
 * stylesheets_free: hash table cleanup helper func
 */
static gboolean
stylesheets_free (gpointer key, gpointer val, gpointer data)
{
	g_free (key);
	return TRUE;
}	

/**
 * list: helper func that lists a hashtable
 */
static void
stylesheets_list (gchar *key, gpointer val, void **data)
{
	GList **ret = (GList **) data;
	*ret = g_list_append (*ret, g_strdup (key));
}
