"""Mailing list manager.

This is a simple mailing list manager that mimicks the ezmlm-idx mail
address commands (see manual page for more info).

Some of the command addresses use MD5 hash functions for security.
In these cases the command address is of the form

    listname-command-id-hash@domain
    
where "listname@domain" is the name of the list, "command" is the command
word, and "id" identifies the object of the command (e.g., which address
to subscribe) and is used to retrieve the relevant information from
persistent storage. "hash" is the MD5 checksum computed from

    listname-command-id@domainSECRET

where "SECRET" is a secret key stored in persistent storage.

This allows a replay attack, of course. Thus, as soon as "id" has been
processed, it is removed from storage and then the replay attack won't
work. Also, "id" will be valid only for a limited time (ezmlm uses
1e7 seconds, I think).

"id" can be anything that can be in a mail address and doesn't contain
a dash. When it is generated, needs to be unique.

Bounce handling
===============

Subscribers are lumped together into groups using some suitable heuristic.
Each group gets a unique identifier. When messages are sent to the list,
each group gets its own copy, with its own bounce address:

    listname-bounce-ID-HASH@example.com
    
where ID is the identifier of the group and HASH is explained above. If
any address in the group bounces (i.e., the bounce address gets any mail),
the group is split into groups of one address each.  

Single-address groups that have existed for long enough without
bouncing will be joined into larger groups. The natural way to group
them is according to the domain: different domains will (typically)
require sending out different copies of the e-mail anyway. (MX records
complicate the issue somewhat, but it's not worth it to do DNS lookups
to do grouping.)

This way, we won't be sending out as many e-mails, which saves on
bandwidth and other resources.

XXX Fill in bounce states

XXX the bounce to cause a state change from ok to bounced is saved


Subscriber database
===================

The subscriber database is indexed by a group identifier, as decribed
above. Each record in the database contains:

    * bounce status of the group
    * timestamp for when the group was created
    * timestamp for the oldest bounce we care about
    * an identifier for the oldest bounce (so it can be looked up)
    * e-mail addresses for the subscribers

The subscriber database is stored in a text file, one line per group:

    id status timestamp-create timestamp-bounce id-of-bounce address ...



Lars Wirzenius <liw@iki.fi>
"""


import getopt
import md5
import os
import shutil
import smtplib
import string
import sys
import time
import ConfigParser


# The following values will be overriden by "make install".
TEMPLATE_DIRS = ["./templates"]
DOTDIR = "dot-eoc"


class UnknownList(Exception):
    pass

class BadCommandAddress(Exception):
    pass

class BadSignature(Exception):
    pass

class ListExists(Exception):
    pass

class ListDoesNotExist(Exception):
    pass

class MissingEnvironmentVariable(Exception):
    pass

class MissingTemplate(Exception):
    pass



def md5sum_as_hex(s):
    hash = md5.new(s).digest()
    return string.join(map(lambda c: "%02x" % ord(c), hash), "")


environ = None

def set_environ(new_environ):
    global environ
    environ = new_environ

def get_from_environ(key):
    global environ
    if environ:
	env = environ
    else:
	env = os.environ
    if env.has_key(key):
	return env[key]
    raise MissingEnvironmentVariable(key)


class MailingListManager:

    def __init__(self, dotdir, sendmail="/usr/sbin/sendmail", lists=[],
    	    	 smtp_server=None):
    	self.dotdir = dotdir
	self.sendmail = sendmail
	self.smtp_server = smtp_server

    	secret_name = os.path.join(self.dotdir, "secret")
    	if not os.path.isdir(self.dotdir):
	    os.makedirs(self.dotdir, 0700)

    	if not os.path.isfile(secret_name):
	    f = open("/dev/urandom", "r")
	    self.secret = f.read(32)
	    f.close()
	    f = open(secret_name, "w")
	    f.write(self.secret)
	    f.close()
    	else:
	    f = open(secret_name, "r")
	    self.secret = f.read()
	    f.close()

    	if not lists:
	    lists = filter(lambda s: "@" in s, os.listdir(dotdir))
    	self.set_lists(lists)

    	self.simple_commands = ["help", "list", "owner", "setlist",
				"setlistsilently", "ignore"]
	self.sub_commands = ["subscribe", "unsubscribe"]
	self.hash_commands = ["subyes", "subapprove", "subreject", "unsubyes",
	    	    	      "bounce", "probe", "approve", "reject",
			      "setlistyes", "setlistsilentyes"]
	self.commands = self.simple_commands + self.sub_commands + \
	    	        self.hash_commands

    	self.environ = None

    def set_lists(self, lists):
    	# The list of lists needs to be sorted in length order so that
	# test@example.com is matched before test-list@example.com
	temp = map(lambda s: (len(s), s), lists)
	temp.sort()
	self.lists = map(lambda t: t[1], temp)

    def decode_address(self, parts):
	return string.join(string.join(parts, "-").split("="), "@")

    def is_list_name(self, local_part, domain):
	return ("%s@%s" % (local_part, domain)) in self.lists

    def compute_hash(self, address):
    	return md5sum_as_hex(address + self.secret)

    def signature_is_bad(self, dict, hash):
	local_part, domain = dict["name"].split("@")
	address = "%s-%s-%s@%s" % (local_part, dict["command"], dict["id"], 
	    	    	    	   domain)
	correct = self.compute_hash(address)
	return correct != hash

    def try_parsing(self, listname, local_part, domain):
    	list_local_part, list_domain = listname.split("@")
	n = len(list_local_part)
	if local_part[:n] != list_local_part:
	    return None, None
    	if len(local_part) != n and local_part[n] != "-":
	    return None, None

	dict = {}
	dict["name"] = listname
	if len(local_part) > n:
	    parts = local_part[n + 1:].split("-")
	    if parts[0] not in self.commands:
	    	return None, None
	    dict["command"] = parts[0]
	    parts = parts[1:]
	else:
	    parts = []
	    dict["command"] = "post"
	
	if dict["command"] in self.sub_commands:
	    dict["sender"] = self.decode_address(parts)
	elif dict["command"] in self.hash_commands:
	    if len(parts) != 2:
		return None, \
		    BadCommandAddress("%s needs id and hash: %s@%s" %
					(dict["command"], local_part, domain))
	    dict["id"] = parts[0]
	    hash = parts[1]
	    if self.signature_is_bad(dict, hash):
		return None, BadSignature
    
	return dict, None

    def parse_command_address(self, local_part, domain):
	if not local_part or not domain:
	    raise BadCommandAddress("local_part or domain missing")
    
    	exception = None
    	for list in self.lists:
	    dict, ex = self.try_parsing(list, local_part, domain)
	    if ex:
	    	if not exception:
		    exception = ex
	    elif dict:
	    	return dict

    	if exception:
    	    raise exception
    	raise UnknownList("%s@%s is not a known list" % (local_part, domain))

    def cleanup_name(self, name, skip_prefix, domain):
    	if skip_prefix and name[:len(skip_prefix)] == skip_prefix:
	    name = name[len(skip_prefix):]
    	if domain:
	    parts = name.split("@", 1)
	    return "%s@%s" % (parts[0], domain)
	return name

    def is_list(self, name, skip_prefix=None, domain=None):
    	name = self.cleanup_name(name, skip_prefix, domain)
    	parts = name.split("@")
	if len(parts) != 2:
	    return 0
    	try:
	    self.parse_command_address(parts[0], parts[1])
    	except BadCommandAddress:
	    return 0
    	except BadSignature:
	    return 0
	except UnknownList:
	    return 0
	return 1

    def create_list(self, name):
    	if self.is_list(name):
	    raise ListExists(name)
	self.set_lists(self.lists + [name])
    	return MailingList(self, name)

    def open_list(self, name):
    	if self.is_list(name):
	    return self.open_list_exact(name)
	else:
	    x = name + "@"
	    for list in self.lists:
	    	if list[:len(x)] == x:
		    return self.open_list_exact(list)
	    raise ListDoesNotExist(name)

    def open_list_exact(self, name):
    	return MailingList(self, name)

    def incoming_message(self, skip_prefix, domain, moderate, post):
	debug("Processing incoming message.")
	debug("$SENDER = <%s>" % get_from_environ("SENDER"))
	debug("$RECIPIENT = <%s>" % get_from_environ("RECIPIENT"))
	dict = self.parse_recipient_address(skip_prefix, domain)
	dict["force-moderation"] = moderate
	dict["force-posting"] = post
	debug("List is <%(name)s>, command is <%(command)s>." % dict)
	list = self.open_list_exact(dict["name"])
	list.obey(dict)

    def parse_recipient_address(self, skip_prefix, domain):
    	recipient = get_from_environ("RECIPIENT")
	recipient = self.cleanup_name(recipient, skip_prefix, domain)
	parts = recipient.split("@")
	if len(parts) != 2:
	    raise UnknownList(recipient)
	return self.parse_command_address(parts[0], parts[1])

    def cleaning_woman(self, send_mail=None):
	now = time.time()
    	for listname in self.lists:
	    list = self.open_list_exact(listname)
	    if send_mail:
	    	list.send_mail = send_mail
	    list.cleaning_woman(now)

    def get_lists(self):
    	return self.lists


class MailingList:

    posting_opts = ["auto", "free", "moderated"]

    def __init__(self, mlm, name):
    	self.mlm = mlm
    	self.dotdir = mlm.dotdir
	self.name = name
	self.dirname = os.path.join(self.dotdir, name)

	self.cp = ConfigParser.ConfigParser()
	self.cp.add_section("list")
	self.cp.set("list", "owners", "")
	self.cp.set("list", "subscription", "free")
	self.cp.set("list", "posting", "free")
	self.cp.set("list", "archived", "no")
	self.cp.set("list", "mail-on-subscription-changes", "no")
	self.cp.set("list", "mail-on-forced-unsubscribe", "no")
	self.cp.set("list", "language", "")

	if os.path.isdir(self.dirname):
	    self.cp.read(self.mkname("config"))
	else:
	    os.mkdir(self.dirname, 0700)
	    self.save_config()
	    f = open(self.mkname("subscribers"), "w")
	    f.close()

    	self.subscribers = SubscriberDatabase(self.dirname, "subscribers")
	self.moderation_box = MessageBox(self.dirname, "moderation-box")
	self.subscription_box = MessageBox(self.dirname, "subscription-box")
	self.bounce_box = MessageBox(self.dirname, "bounce-box")

    def mkname(self, relative):
    	return os.path.join(self.dirname, relative)

    def save_config(self):
	f = open(self.mkname("config"), "w")
	self.cp.write(f)
	f.close()

    def read_stdin(self):
	data = sys.stdin.read()
	# Skip Unix mbox "From " mail start indicator
	if data[:5] == "From ":
	    data = string.split(data, "\n", 1)[1]
	return data

    def send_mail(self, envelope_sender, recipients, text):
	debug("send_mail: sender=%s recipients=%s text=\n%s" % 
	      (envelope_sender, str(recipients), text[:text.find("\n\n")]))
	if recipients:
	    if self.mlm.smtp_server:
	    	smtp = smtplib.SMTP(self.mlm.smtp_server)
		smtp.sendmail(envelope_sender, recipients, text)
		smtp.quit()
	    else:
		recipients = string.join(recipients, " ")
		f = os.popen("%s -oi -f '%s' %s" % 
				(self.mlm.sendmail, envelope_sender, recipients), 
			     "w")
		f.write(text)
		f.close()
	else:
	    debug("send_mail: no recipients, not sending")

    def command_address(self, command):
    	local_part, domain = self.name.split("@")
	return "%s-%s@%s" % (local_part, command, domain)

    def signed_address(self, command, id):
    	unsigned = self.command_address("%s-%s" % (command, id))
	hash = self.mlm.compute_hash(unsigned)
	return self.command_address("%s-%s-%s" % (command, id, hash))

    def ignore(self):
    	return self.command_address("ignore")

    def template(self, template_name, dict):
    	lang = self.cp.get("list", "language")
	if lang:
	    template_name_lang = template_name + "." + lang
    	else:
	    template_name_lang = template_name

	if not dict.has_key("list"):
	    dict["list"] = self.name
	    dict["local"], dict["domain"] = self.name.split("@")
	if not dict.has_key("list"):
	    dict["list"] = self.name

	for dir in [os.path.join(self.dirname, "templates")] + TEMPLATE_DIRS:
	    pathname = os.path.join(dir, template_name_lang)
	    if not os.path.exists(pathname):
	    	pathname = os.path.join(dir, template_name)
	    if os.path.exists(pathname):
		f = open(pathname, "r")
		data = f.read()
		f.close()
		return data % dict

    	raise MissingTemplate(template_name)

    def send_template(self, envelope_sender, sender, recipients,
    	    	      template_name, dict):
    	dict["From"] = sender
	dict["To"] = string.join(recipients, ", ")
	text = self.template(template_name, dict)
	self.send_mail(envelope_sender, recipients, text)

    def owners(self):
	return self.cp.get("list", "owners").split()

    def is_list_owner(self, address):
    	return address in self.cp.get("list", "owners").split()

    def obey_help(self):
	recipient = get_from_environ("SENDER")
	sender = self.command_address("help")
	self.send_template(self.ignore(), sender, [recipient], "help", {})

    def obey_list(self):
	sender = self.command_address("help")
	recipient = get_from_environ("SENDER")
	if self.is_list_owner(recipient):
	    addr_list = self.subscribers.get_all()
	    addr_text = string.join(addr_list, "\n")
	    self.send_template(self.ignore(), sender, [recipient], "list",
			       {
				  "addresses": addr_text,
				  "count": len(addr_list),
			       })
	else:
	    self.send_template(self.ignore(), sender, [recipient], 
	    	    	       "list-sorry", {})

    def obey_setlist(self, origmail):
	sender = self.command_address("help")
	recipient = get_from_environ("SENDER")
	if self.is_list_owner(recipient):
	    id = self.moderation_box.add(recipient, origmail)
    	    if self.parse_setlist_addresses(origmail) == None:
	    	self.send_bad_addresses_in_setlist(id)
		self.moderation_box.remove(id)
	    else:
		confirm = self.signed_address("setlistyes", id)
		self.send_template(self.ignore(), sender, self.owners(), 
		    	    	   "setlist-confirm",
				   {
				      "confirm": confirm,
				      "origmail": origmail,
				   })
	else:
	    self.send_template(self.ignore(), sender, [recipient], 
	    	    	       "setlist-sorry", {})

    def obey_setlistsilently(self, origmail):
	sender = self.command_address("help")
	recipient = get_from_environ("SENDER")
	if self.is_list_owner(recipient):
	    id = self.moderation_box.add(recipient, origmail)
    	    if self.parse_setlist_addresses(origmail) == None:
	    	self.send_bad_addresses_in_setlist(id)
		self.moderation_box.remove(id)
	    else:
		confirm = self.signed_address("setlistsilentyes", id)
		self.send_template(self.ignore(), sender, self.owners(), 
		    	    	   "setlist-confirm",
				   {
				      "confirm": confirm,
				      "origmail": origmail,
				   })
	else:
	    self.send_template(self.ignore(), sender, [recipient], 
	    	    	       "setlist-sorry", {})

    def parse_setlist_addresses(self, text):
    	body = text.split("\n\n", 1)[1]
	lines = body.split("\n")
	lines = filter(lambda line: line != "", lines)
	badlines = filter(lambda line: "@" not in line, lines)
	if badlines:
	    return None
	else:
	    return lines

    def send_bad_addresses_in_setlist(self, id):
	sender = self.command_address("help")
	addr = self.moderation_box.get_address(id)
	origmail = self.moderation_box.get(id)
	self.send_template(self.ignore(), sender, [addr], "setlist-badlist",
			   {
			    "origmail": origmail,
			   })

    def obey_setlistyes(self, dict):
    	if self.moderation_box.has(dict["id"]):
	    sender = self.command_address("help")
	    text = self.moderation_box.get(dict["id"])
	    addresses = self.parse_setlist_addresses(text)
	    if addresses == None:
    	    	self.send_bad_addresses_in_setlist(id)
    	    else:
		removed_subscribers = []
	    	self.subscribers.lock()
		old = self.subscribers.get_all()
		for address in old:
		    if address not in addresses:
			self.subscribers.remove(address)
			removed_subscribers.append(address)
    	    	    else:
		    	addresses.remove(address)
		self.subscribers.add_many(addresses)
		self.subscribers.save()
    	    	
		sender = self.command_address("help")
		for recipient in addresses:
		    self.send_template(self.ignore(), sender, [recipient], 
				       "sub-welcome", {})
    	    	for recipient in removed_subscribers:
		    self.send_template(self.ignore(), sender, [recipient], 
				       "unsub-goodbye", {})

		sender = self.command_address("help")
		self.send_template(self.ignore(), sender, self.owners(), 
		    	    	   "setlist-done", {})



	    self.moderation_box.remove(dict["id"])

    def obey_setlistsilentyes(self, dict):
    	if self.moderation_box.has(dict["id"]):
	    sender = self.command_address("help")
	    text = self.moderation_box.get(dict["id"])
	    addresses = self.parse_setlist_addresses(text)
	    if addresses == None:
    	    	self.send_bad_addresses_in_setlist(id)
    	    else:
	    	self.subscribers.lock()
		old = self.subscribers.get_all()
		for address in old:
		    if address not in addresses:
			self.subscribers.remove(address)
    	    	    else:
		    	addresses.remove(address)
		self.subscribers.add_many(addresses)
		self.subscribers.save()
    	    	
		sender = self.command_address("help")
		self.send_template(self.ignore(), sender, self.owners(), 
		    	    	   "setlist-done", {})
	    self.moderation_box.remove(dict["id"])

    def obey_owner(self, text):
	sender = get_from_environ("SENDER")
	recipients = self.cp.get("list", "owners").split()
	self.send_mail(sender, recipients, text)

    def obey_subscribe_or_unsubscribe(self, dict, template_name, command, 
    	    	    	    	      origmail):

    	requester  = get_from_environ("SENDER")
	subscriber = dict["sender"]
	if not subscriber:
	    subscriber = requester
	    
	if requester in self.owners():
	    confirmers = self.owners()
	else:
	    confirmers = [subscriber]

	sender = self.command_address("help")
	id = self.subscription_box.add(subscriber, origmail)
	confirm = self.signed_address(command, id)
	self.send_template(self.ignore(), sender, confirmers, template_name,
			   {
				"confirm": confirm,
				"origmail": origmail,
			   })

    def obey_subscribe(self, dict, origmail):
    	self.obey_subscribe_or_unsubscribe(dict, "sub-confirm", "subyes", 
	    	    	    	    	   origmail)

    def obey_unsubscribe(self, dict, origmail):
    	self.obey_subscribe_or_unsubscribe(dict, "unsub-confirm", "unsubyes",
	    	    	    	    	   origmail)

    def obey_subyes(self, dict):
    	if self.subscription_box.has(dict["id"]):
	    if self.cp.get("list", "subscription") == "free":
		recipient = self.subscription_box.get_address(dict["id"])
	    	self.subscribers.lock()
		self.subscribers.add(recipient)
		self.subscribers.save()
		sender = self.command_address("help")
		self.send_template(self.ignore(), sender, [recipient], 
				   "sub-welcome", {})
		self.subscription_box.remove(dict["id"])
		if self.cp.get("list", "mail-on-subscription-changes")=="yes":
		    self.send_template(self.ignore(), sender, self.owners(),
				       "sub-owner-notification",
				       {
					"address": recipient,
				       })
	    else:
		sender = self.command_address("help")
		recipients = self.cp.get("list", "owners").split()
		confirm = self.signed_address("subapprove", dict["id"])
		deny = self.signed_address("subreject", dict["id"])
		subscriber = self.subscription_box.get_address(dict["id"])
		origmail = self.subscription_box.get(dict["id"])
		self.send_template(self.ignore(), sender, recipients, 
		    	    	   "sub-moderate", 
		    	    	   {
				       "confirm": confirm,
				       "deny": deny,
				       "subscriber": subscriber,
				       "origmail": origmail,
				   })
		recipient = self.subscription_box.get_address(dict["id"])
		self.send_template(self.ignore(), sender, [recipient], 
		    	    	   "sub-wait", {})

    def obey_subapprove(self, dict):
    	if self.subscription_box.has(dict["id"]):
	    recipient = self.subscription_box.get_address(dict["id"])
	    self.subscribers.lock()
	    self.subscribers.add(recipient)
	    self.subscribers.save()
	    sender = self.command_address("help")
	    self.send_template(self.ignore(), sender, [recipient], 
	    	    	       "sub-welcome", {})
	    self.subscription_box.remove(dict["id"])
	    if self.cp.get("list", "mail-on-subscription-changes")=="yes":
		self.send_template(self.ignore(), sender, self.owners(),
				   "sub-owner-notification",
				   {
				    "address": recipient,
				   })

    def obey_subreject(self, dict):
    	if self.subscription_box.has(dict["id"]):
	    sender = self.command_address("help")
	    recipient = self.subscription_box.get_address(dict["id"])
	    self.send_template(self.ignore(), sender, [recipient], 
	    	    	       "sub-reject", {})
	    self.subscription_box.remove(dict["id"])

    def obey_unsubyes(self, dict):
    	if self.subscription_box.has(dict["id"]):
	    sender = self.command_address("help")
	    recipient = self.subscription_box.get_address(dict["id"])
	    self.subscribers.lock()
	    self.subscribers.remove(recipient)
	    self.subscribers.save()
	    self.send_template(self.ignore(), sender, [recipient], 
	    	    	       "unsub-goodbye", {})
	    self.subscription_box.remove(dict["id"])
	    if self.cp.get("list", "mail-on-subscription-changes")=="yes":
		self.send_template(self.ignore(), sender, self.owners(),
				   "unsub-owner-notification",
				   {
				    "address": recipient,
				   })

    def store_into_archive(self, text):
    	if self.cp.get("list", "archived") == "yes":
	    archdir = os.path.join(self.dirname, "archive")
	    if not os.path.exists(archdir):
	    	os.mkdir(archdir, 0700)
	    id = md5sum_as_hex(text)
	    f = open(os.path.join(archdir, id), "w")
	    f.write(text)
	    f.close()

    def list_headers(self):
    	local, domain = self.name.split("@")
	list = []
	list.append("List-Id: <%s.%s>" % (local, domain))
	list.append("List-Help: <mailto:%s-help@%s>" % (local, domain))
	list.append("List-Unsubscribe: <mailto:%s-unsubscribe@%s>" % 
		    (local, domain))
	list.append("List-Subscribe: <mailto:%s-subscribe@%s>" % 
	    	    (local, domain))
	list.append("List-Post: <mailto:%s@%s>" % (local, domain))
	list.append("List-Owner: <mailto:%s-owner@%s>" % (local, domain))
	list.append("Precedence: bulk");
	return string.join(list, "\n") + "\n"

    def send_mail_to_subscribers(self, text):
    	text = self.list_headers() + text + self.template("footer", {})
    	self.store_into_archive(text)
	for group in self.subscribers.groups():
	    bounce = self.signed_address("bounce", group)
	    addresses = self.subscribers.in_group(group)
	    self.send_mail(bounce, addresses, text)

    def post_into_moderate(self, poster, dict, text):
	id = self.moderation_box.add(poster, text)
	sender = self.command_address("help")
	recipients = self.cp.get("list", "owners").split()
	confirm = self.signed_address("approve", id)
	deny = self.signed_address("reject", id)
	self.send_template(self.ignore(), sender, recipients, "msg-moderate",
	                   {
			    "confirm": confirm,
			    "deny": deny,
			    "origmail": text,
			   })
	self.send_template(self.ignore(), sender, [poster], "msg-wait",
	                   {
			   })
    
    def should_be_moderated(self, posting, poster):
    	if posting == "moderated":
	    return 1
	if posting == "auto" and poster not in self.subscribers.get_all():
	    return 1
    	return 0

    def obey_post(self, dict, text):
     	if dict.has_key("force-moderation") and dict["force-moderation"]:
 	    force_moderation = 1
     	else:
 	    force_moderation = 0
     	if dict.has_key("force-posting") and dict["force-posting"]:
 	    force_posting = 1
     	else:
 	    force_posting = 0
	posting = self.cp.get("list", "posting")
	if posting not in self.posting_opts:
	    error("You have a weird 'posting' config. Please, review it")
	poster = get_from_environ("SENDER")
    	if force_moderation:
	    self.post_into_moderate(poster, dict, text)
    	elif force_posting:
	    self.send_mail_to_subscribers(text)
    	elif self.should_be_moderated(posting, poster):
	    self.post_into_moderate(poster, dict, text)
    	else:
	    self.send_mail_to_subscribers(text)
 
    def obey_approve(self, dict):
    	if self.moderation_box.has(dict["id"]):
	    text = self.moderation_box.get(dict["id"])
	    self.send_mail_to_subscribers(text)
	    self.moderation_box.remove(dict["id"])

    def obey_reject(self, dict):
    	if self.moderation_box.has(dict["id"]):
	    self.moderation_box.remove(dict["id"])

    def split_address_list(self, addrs):
    	domains = {}
	for addr in addrs:
	    userpart, domain = addr.split("@")
	    if domains.has_key(domain):
	    	domains[domain].append(addr)
	    else:
	    	domains[domain] = [addr]
	result = []
    	if len(domains.keys()) == 1:
	    for addr in addrs:
	    	result.append([addr])
    	else:
    	    result = domains.values()
    	return result

    def obey_bounce(self, dict, text):
    	if self.subscribers.has_group(dict["id"]):
	    self.subscribers.lock()
	    addrs = self.subscribers.in_group(dict["id"])
	    if len(addrs) == 1:
    	    	debug("Address <%s> bounced, setting state to bounce." %
		      addrs[0])
    	    	bounce_id = self.bounce_box.add(addrs[0], text[:4096])
		self.subscribers.set(dict["id"], "status", "bounced")
		self.subscribers.set(dict["id"], "timestamp-bounced", 
				     "%f" % time.time())
    	    	self.subscribers.set(dict["id"], "bounce-id",
		    	    	     bounce_id)
    	    else:
    	    	debug("Group %s bounced, splitting." % dict["id"])
		for new_addrs in self.split_address_list(addrs):
		    self.subscribers.add_many(new_addrs)
		self.subscribers.remove_group(dict["id"])
	    self.subscribers.save()
    	else:
	    debug("Ignoring bounce, group %s doesn't exist (anymore?)." %
	    	  dict["id"])

    def obey_probe(self, dict, text):
    	id = dict["id"]
    	if self.subscribers.has_group(id):
	    self.subscribers.lock()
	    if self.subscribers.get(id, "status") == "probed":
		self.subscribers.set(id, "status", "probebounced")
    	    self.subscribers.save()

    def obey(self, dict):
    	text = self.read_stdin()

    	if dict["command"] in ["help", "list", "subscribe", "unsubscribe",
	    	    	       "subyes", "subapprove", "subreject",
			       "unsubyes", "post", "approve"]:
    	    sender = get_from_environ("SENDER")
	    if not sender:
	    	debug("Ignoring bounce message for %s command." % 
		    	dict["command"])
	    	return

    	if dict["command"] == "help":
	    self.obey_help()
    	elif dict["command"] == "list":
	    self.obey_list()
    	elif dict["command"] == "owner":
	    self.obey_owner(text)
    	elif dict["command"] == "subscribe":
	    self.obey_subscribe(dict, text)
    	elif dict["command"] == "unsubscribe":
	    self.obey_unsubscribe(dict, text)
    	elif dict["command"] == "subyes":
	    self.obey_subyes(dict)
    	elif dict["command"] == "subapprove":
	    self.obey_subapprove(dict)
    	elif dict["command"] == "subreject":
	    self.obey_subreject(dict)
    	elif dict["command"] == "unsubyes":
	    self.obey_unsubyes(dict)
    	elif dict["command"] == "post":
	    self.obey_post(dict, text)
    	elif dict["command"] == "approve":
	    self.obey_approve(dict)
    	elif dict["command"] == "reject":
	    self.obey_reject(dict)
    	elif dict["command"] == "bounce":
	    self.obey_bounce(dict, text)
    	elif dict["command"] == "probe":
	    self.obey_probe(dict, text)
    	elif dict["command"] == "setlist":
	    self.obey_setlist(text)
    	elif dict["command"] == "setlistsilently":
	    self.obey_setlistsilently(text)
    	elif dict["command"] == "setlistyes":
	    self.obey_setlistyes(dict)
    	elif dict["command"] == "setlistsilentyes":
	    self.obey_setlistsilentyes(dict)
    	elif dict["command"] == "ignore":
	    pass

    def get_bounce_text(self, id):
    	bounce_id = self.subscribers.get(id, "bounce-id")
	if self.bounce_box.has(bounce_id):
	    bounce_text = self.bounce_box.get(bounce_id)
	    bounce_text = string.join(map(lambda s: "> " + s + "\n",
					  bounce_text.split("\n")), "")
    	else:
	    bounce_text = "Bounce message not available."
    	return bounce_text

    one_week = 7.0 * 24.0 * 60.0 * 60.0

    def handle_bounced_groups(self, now):
	for id in self.subscribers.groups():
	    status = self.subscribers.get(id, "status") 
	    t = float(self.subscribers.get(id, "timestamp-bounced")) 
	    if status == "bounced":
		if now - t > self.one_week:
		    sender = self.signed_address("probe", id) 
		    recipients = self.subscribers.in_group(id) 
		    self.send_template(sender, sender, recipients,
				       "bounce-warning", {
					"bounce": self.get_bounce_text(id),
				       })
		    self.subscribers.set(id, "status", "probed")
	    elif status == "probed":
		if now - t > 2 * self.one_week:
		    debug(("Cleaning woman: probe didn't bounce " + 
			  "for group <%s>, setting status to ok.") % id)
		    self.subscribers.set(id, "status", "ok")
		    self.bounce_box.remove(
			    self.subscribers.get(id, "bounce-id"))
    	    elif status == "probebounced":
		sender = self.command_address("help") 
		for address in self.subscribers.in_group(id):
		    if self.cp.get("list", "mail-on-forced-unsubscribe") \
			== "yes":
			self.send_template(sender, sender,
				       self.owners(),
				       "bounce-owner-notification",
				       {
					"address": address,
					"bounce": self.get_bounce_text(id),
				       })

		    self.bounce_box.remove(
			    self.subscribers.get(id, "bounce-id"))
		    self.subscribers.remove(address) 
		    debug("Cleaning woman: removing <%s>." % address)
		    self.send_template(sender, sender, [address],
				       "bounce-goodbye", {})

    def join_nonbouncing_groups(self, now):
    	to_be_joined = []
    	for id in self.subscribers.groups():
	    status = self.subscribers.get(id, "status")
	    age1 = now - float(self.subscribers.get(id, "timestamp-bounced"))
	    age2 = now - float(self.subscribers.get(id, "timestamp-created"))
	    if status == "ok":
	    	if age1 > self.one_week and age2 > self.one_week:
		    to_be_joined.append(id)
    	if to_be_joined:
    	    addrs = []
	    for id in to_be_joined:
		addrs = addrs + self.subscribers.in_group(id)
	    self.subscribers.add_many(addrs)
    	    for id in to_be_joined:
		self.bounce_box.remove(self.subscribers.get(id, "bounce-id"))
		self.subscribers.remove_group(id)

    def cleaning_woman(self, now):
    	if self.subscribers.lock():
	    self.handle_bounced_groups(now)
	    self.join_nonbouncing_groups(now)
	    self.subscribers.save()

class SubscriberDatabase:

    def __init__(self, dirname, name):
    	self.dict = {}
	self.filename = os.path.join(dirname, name)
	self.lockname = os.path.join(dirname, "lock")
	self.loaded = 0
	self.locked = 0

    def lock(self):
	if os.system("lockfile -l 60 %s" % self.lockname) == 0:
	    self.locked = 1
	    self.load()
    	return self.locked
    
    def unlock(self):
	os.remove(self.lockname)
	self.locked = 0
    
    def load(self):
    	if not self.loaded and not self.dict:
	    f = open(self.filename, "r")
	    for line in f.xreadlines():
	    	parts = line.split()
		self.dict[parts[0]] = {
		    "status": parts[1],
		    "timestamp-created": parts[2],
		    "timestamp-bounced": parts[3],
		    "bounce-id": parts[4],
		    "addresses": parts[5:],
		}
	    f.close()
	    self.loaded = 1

    def save(self):
    	assert self.locked
	assert self.loaded
    	f = open(self.filename + ".new", "w")
	for id in self.dict.keys():
	    f.write("%s " % id)
	    f.write("%s " % self.dict[id]["status"])
	    f.write("%s " % self.dict[id]["timestamp-created"])
	    f.write("%s " % self.dict[id]["timestamp-bounced"])
	    f.write("%s " % self.dict[id]["bounce-id"])
	    f.write("%s\n" % string.join(self.dict[id]["addresses"], " "))
	f.close()
	os.remove(self.filename)
	os.rename(self.filename + ".new", self.filename)
	self.unlock()

    def get(self, id, attribute):
    	self.load()
    	if self.dict.has_key(id) and self.dict[id].has_key(attribute):
	    return self.dict[id][attribute]
	return None

    def set(self, id, attribute, value):
    	assert self.locked
    	self.load()
    	if self.dict.has_key(id) and self.dict[id].has_key(attribute):
	    self.dict[id][attribute] = value

    def add(self, address):
    	return self.add_many([address])

    def add_many(self, addresses):
    	assert self.locked
    	assert self.loaded
	for id in self.dict.keys():
	    old_ones = self.dict[id]["addresses"]
	    for addr in addresses:
	    	if addr in old_ones:
		    old_ones.remove(addr)
    	    self.dict[id]["addresses"] = old_ones
	id = self.new_group()
	self.dict[id] = {
	    "status": "ok",
	    "timestamp-created": self.timestamp(),
	    "timestamp-bounced": "0",
	    "bounce-id": "..notexist..",
	    "addresses": addresses,
	}
	return id

    def new_group(self):
    	keys = self.dict.keys()
	if keys:
	    keys = map(lambda x: int(x), keys)
	    keys.sort()
	    return "%d" % (keys[-1] + 1)
	else:
	    return "0"

    def timestamp(self):
    	return "%.0f" % time.time()

    def get_all(self):
    	self.load()
	list = []
	for values in self.dict.values():
	    list = list + values["addresses"]
    	return list

    def groups(self):
    	self.load()
    	return self.dict.keys()

    def has_group(self, id):
    	self.load()
    	return self.dict.has_key(id)

    def in_group(self, id):
    	self.load()
	return self.dict[id]["addresses"]

    def remove(self, address):
    	assert self.locked
	self.load()
	for id in self.dict.keys():
	    group = self.dict[id]
	    if address in group["addresses"]:
	    	group["addresses"].remove(address)
		if len(group["addresses"]) == 0:
		    del self.dict[id]
		break

    def remove_group(self, id):
    	assert self.locked
	self.load()
	del self.dict[id]


class MessageBox:

    def __init__(self, dirname, boxname):
    	self.boxdir = os.path.join(dirname, boxname)
	if not os.path.isdir(self.boxdir):
	    os.mkdir(self.boxdir, 0700)

    def filename(self, id):
    	return os.path.join(self.boxdir, id)

    def add(self, address, message_text):
    	id = self.make_id(message_text)
	filename = self.filename(id)
	f = open(filename + ".address", "w")
	f.write(address)
	f.close()
	f = open(filename + ".new", "w")
	f.write(message_text)
	f.close()
	os.rename(filename + ".new", filename)
	return id

    def make_id(self, message_text):
    	return md5sum_as_hex(message_text)
	# XXX this might be unnecessarily long

    def remove(self, id):
    	filename = self.filename(id)
    	if os.path.isfile(filename):
	    os.remove(filename)
	    os.remove(filename + ".address")

    def has(self, id):
	return os.path.isfile(self.filename(id))

    def get_address(self, id):
    	f = open(self.filename(id) + ".address", "r")
	data = f.read()
	f.close()
	return data.strip()

    def get(self, id):
    	f = open(self.filename(id), "r")
	data = f.read()
	f.close()
	return data


class DevNull:

    def write(self, str):
    	pass


log_file_handle = None
def log_file():
    global log_file_handle
    if log_file_handle is None:
    	try:
	    log_file_handle = open(os.path.join(DOTDIR, "logfile.txt"), "a")
	except:
	    log_file_handle = DevNull()
    return log_file_handle

def timestamp():
    tuple = time.localtime(time.time())
    return time.strftime("%Y-%m-%d %H:%M:%S", tuple) + " [%d]" % os.getpid()


quiet = 0


def debug(msg):
    if not quiet:
	sys.stderr.write(msg + "\n")
    log_file().write(timestamp() + " " + msg + "\n")


def info(msg):
    sys.stderr.write(msg + "\n")
    log_file().write(timestamp() + " " + msg + "\n")


def error(msg):
    info(msg)
    sys.exit(1)


def usage():
    sys.stdout.write("""\
Usage: enemies-of-carlotta [options] command
Mailing list manager.

Options:
  --name=listname@domain
  --owner=address@domain
  --subscription=free/moderated
  --posting=free/moderated/auto
  --archived=yes/no
  --skip-prefix=string
  --domain=domain.name
  --smtp-server=domain.name
  --quiet
  --moderate

Commands:
  --help
  --create
  --subscribe
  --unsubscribe
  --list
  --is-list
  --edit
  --incoming
  --cleaning-woman
  --show-lists

For more detailed information, please read the enemies-of-carlotta(1)
manual page.
""")
    sys.exit(0)



def set_list_options(list, owners, subscription, posting, archived):
    if owners:
	list.cp.set("list", "owners", string.join(owners, " "))
    if subscription != None:
	list.cp.set("list", "subscription", subscription)
    if posting != None:
	list.cp.set("list", "posting", posting)
    if archived != None:
	list.cp.set("list", "archived", archived)


def main(args):
    opts, args = getopt.getopt(args, "h",
    	    	    	       ["name=",
			        "owner=",
				"subscription=",
				"posting=",
				"archived=",
				"skip-prefix=",
				"domain=",
				"sendmail=",
				"smtp-server=",
				"quiet",
				"moderate",
				"post",

    	    	    	    	"help",
			        "create",
				"destroy",
				"subscribe",
				"unsubscribe",
				"list",
				"is-list",
				"edit",
				"incoming",
				"cleaning-woman",
				"show-lists",
			       ])

    operation = None
    list_name = None
    owners = []
    subscription = None
    posting = None
    archived = None
    skip_prefix = None
    domain = None
    sendmail = "/usr/sbin/sendmail"
    smtp_server = None
    moderate = 0
    post = 0
    global quiet

    for opt, arg in opts:
    	if opt == "--name":
	    list_name = arg
	elif opt == "--owner":
	    owners.append(arg)
    	elif opt == "--subscription":
	    subscription = arg
    	elif opt == "--posting":
	    posting = arg
    	elif opt == "--archived":
	    archived = arg
    	elif opt == "--skip-prefix":
	    skip_prefix = arg
    	elif opt == "--domain":
	    domain = arg
    	elif opt == "--sendmail":
	    sendmail = arg
    	elif opt == "--smtp-server":
	    smtp_server = arg
    	elif opt == "--moderate":
	    moderate = 1
    	elif opt == "--post":
	    post = 1
    	elif opt == "--quiet":
	    quiet = 1
    	else:
	    operation = opt

    if operation is None:
    	error("No operation specified, see --help.")

    if list_name is None and operation not in ["--incoming", "--help", "-h",
    	    	    	    	    	       "--cleaning-woman",
					       "--show-lists"]:
	error("%s requires a list name specified with --name" % operation)

    if operation in ["--help", "-h"]:
    	usage()

    mlm = MailingListManager(DOTDIR, sendmail=sendmail, 
    	    	    	     smtp_server=smtp_server)
    if operation == "--create":
    	if not owners:
	    error("You must give at least one list owner with --owner.")
    	list = mlm.create_list(list_name)
	set_list_options(list, owners, subscription, posting, archived)
	list.save_config()
	debug("Created list %s." % list_name)
    elif operation == "--destroy":
    	shutil.rmtree(os.path.join(DOTDIR, list_name))
	debug("Removed list %s." % list_name)
    elif operation == "--edit":
    	list = mlm.open_list(list_name)
	set_list_options(list, owners, subscription, posting, archived)
	list.save_config()
    elif operation == "--subscribe":
    	list = mlm.open_list(list_name)
	list.subscribers.lock()
	for address in args:
	    list.subscribers.add(address)
	    debug("Added subscriber <%s>." % address)
	list.subscribers.save()
    elif operation == "--unsubscribe":
    	list = mlm.open_list(list_name)
	list.subscribers.lock()
	for address in args:
	    list.subscribers.remove(address)
	    debug("Removed subscriber <%s>." % address)
	list.subscribers.save()
    elif operation == "--list":
    	list = mlm.open_list(list_name)
	for address in list.subscribers.get_all():
	    print address
    elif operation == "--is-list":
    	if mlm.is_list(list_name, skip_prefix, domain):
	    debug("Indeed a mailing list: <%s>" % list_name)
	else:
	    debug("Not a mailing list: <%s>" % list_name)
	    sys.exit(1)
    elif operation == "--incoming":
    	mlm.incoming_message(skip_prefix, domain, moderate, post)
    elif operation == "--cleaning-woman":
    	mlm.cleaning_woman()
    elif operation == "--show-lists":
    	for listname in mlm.get_lists():
	    print listname
    else:
    	error("Internal error: unimplemented option <%s>." % operation)

if __name__ == "__main__":
    main(sys.argv[1:])
