/*
 *  apcsetup.c	-- configuration parts for apcupsd package
 *
 *  apcupsd.c	-- Simple Daemon to catch power failure signals from a
 *		   BackUPS, BackUPS Pro, or SmartUPS (from APCC).
 *		-- Now SmartMode support for SmartUPS and BackUPS Pro.
 *
 *  Copyright (C) 1996-99 Andre M. Hedrick <andre@suse.com>
 *  All rights reserved.
 *
 */

/*
 *		       GNU GENERAL PUBLIC LICENSE
 *			  Version 2, June 1991
 *
 *  Copyright (C) 1989, 1991 Free Software Foundation, Inc.
 *			     675 Mass Ave, Cambridge, MA 02139, USA
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 *  IN NO EVENT SHALL ANY AND ALL PERSONS INVOLVED IN THE DEVELOPMENT OF THIS
 *  PACKAGE, NOW REFERRED TO AS "APCUPSD-Team" BE LIABLE TO ANY PARTY FOR
 *  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
 *  OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ANY OR ALL
 *  OF THE "APCUPSD-Team" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  THE "APCUPSD-Team" SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 *  ON AN "AS IS" BASIS, AND THE "APCUPSD-Team" HAS NO OBLIGATION TO PROVIDE
 *  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *  THE "APCUPSD-Team" HAS ABSOLUTELY NO CONNECTION WITH THE COMPANY
 *  AMERICAN POWER CONVERSION, "APCC".  THE "APCUPSD-Team" DID NOT AND
 *  HAS NOT SIGNED ANY NON-DISCLOSURE AGREEMENTS WITH "APCC".  ANY AND ALL
 *  OF THE LOOK-A-LIKE ( UPSlink(tm) Language ) WAS DERIVED FROM THE
 *  SOURCES LISTED BELOW.
 *
 */

#include "apc.h"

/*********************************************************************/
static char *get_ups_model_V_codes (char *s, UPSINFO *ups)
{

    switch(s[0]) {
    case '0': return("APC Matrix-UPS 3000");
    case '2': return("APC Smart-UPS 250");
    case '3': return("APC Smart-UPS 370ci");
    case '4': return("APC Smart-UPS 400");
    case '5': return("APC Matrix-UPS 5000");
    case '6': return("APC Smart-UPS 600");
    case '7': return("APC Smart-UPS 900");
    case '8': return("APC Smart-UPS 1250");
    case '9': return("APC Smart-UPS 2000");
    case 'A': return("APC Smart-UPS 1400");
    case 'B': return("APC Smart-UPS 1000");
    case 'C': return("APC Smart-UPS 650");
    case 'D': return("APC Smart-UPS 420");
    case 'E': return("APC Smart-UPS 280");
    case 'F': return("APC Smart-UPS 450");
    case 'G': return("APC Smart-UPS 700");
    case 'H': return("APC Smart-UPS 700 XL");
    case 'I': return("APC Smart-UPS 1000");
    case 'J': return("APC Smart-UPS 1000 XL");
    case 'K': return("APC Smart-UPS 1400");
    case 'L': return("APC Smart-UPS 1400 XL");
    case 'M': return("APC Smart-UPS 2200");
    case 'N': return("APC Smart-UPS 2200 XL");
    case 'P': return("APC Smart-UPS 3000");
    case 'O': return("APC Smart-UPS 5000");
    default:
	break;
    }
    return(ups->mode.long_name);
}

/*********************************************************************/
static char *get_ups_model_b_codes (char *s, UPSINFO *ups)
{
	char *cp;

/*
 *	New Firmware revison and model ID String in NN.M.D is the format
 *         This returns two numbers and a character "D"
 *	     where NN == UPS ID Code.
 *		 12 == Back-UPS Pro 650
 *		 13 == Back-UPS Pro 1000
 *		 52 == Smart-UPS 700
 *		 72 == Smart-UPS 1400
 *	     where M == Possible Case Style, unknown???
 *		 1 == Stand Alone
 *		 8 == Rack Mount
 *		 9 == Rack Mount
 *	     where D == unknown???
 *
 */

/*      fprintf(stderr, "\n%s: 'b' %s", argvalue, s); */

	return(cp = ups->mode.long_name);
}

/*********************************************************************/
static void get_ups_model(UPSINFO *ups)
{
    char response[32];
    char *cp;
    unsigned char b;
    int i;
    response[0] = '\0';

    for (i=0; i<4; i++) {
	b=0x01;
	write(ups->fd, &b, 1);
	sleep(1);
    }
    getline(response, sizeof response, ups);

    if (strlen(response)) {
        ups->mode.long_name[0] = '\0';
        sprintf(ups->mode.long_name, "%s", response);
	return;
    }

    response[0] = '\0';
    strcpy(response, smart_poll(ups->UPS_Cmd[CI_UPSMODEL], ups));

    if (strlen(response)) {
	cp = get_ups_model_V_codes(response, ups);
	if (cp != ups->mode.long_name)
            sprintf(ups->mode.long_name, "%s", cp);
	return;
    }

    response[0] = '\0';
    strcpy(response, smart_poll(ups->UPS_Cmd[CI_REVNO], ups));

    if (strlen(response))
        fprintf(stderr, "\n%s: 'b' %s", argvalue,
		get_ups_model_b_codes(response, ups));

}

/*********************************************************************/
void setup_ups_name(UPSINFO *ups)
{
    char *n;
    char response[32];
    char name[9];
    char a = ups->UPS_Cmd[CI_CYCLE_EPROM];
    char c = ups->UPS_Cmd[CI_IDEN];
    int i;
    int j = strlen(ups->upsname);
    name[0] = '\0';

    if (j == 0) {
        fprintf(stderr, "Error, new name of zero length.\n");
	return;
    } else if (j > 8)		     
	j = 8;		      /* maximum size */

    strncpy(name, ups->upsname, j);
    /* blank fill to 8 chars */
    while (j < 8) {
        name[j] = ' ';
	j++;
    }

    /* Ask for name */
    write(ups->fd, &c, 1);  /* c = 'c' */
    getline(response, sizeof(response), ups);
    fprintf(stderr, "The old UPS name is: %s\n", response);

    /* Tell UPS we will change name */
    write(ups->fd, &a, 1);  /* a = '-' */
    sleep(1);

    n = name;
    for (i=0; i<8; i++) {
	write(ups->fd, n++, 1);
	sleep(1);
    }

    /* Expect OK after successful name change */
    *response = 0;
    getline(response, sizeof(response), ups);
    if (strcmp(response, "OK") != 0)
        fprintf(stderr, "\nError changing UPS name\n");

    ups->upsname[0] = '\0';
    smart_poll(ups->UPS_Cmd[CI_IDEN], ups);
    strcpy(ups->upsname, smart_poll(ups->UPS_Cmd[CI_IDEN], ups));

    fprintf(stderr, "The new UPS name is: %s\n", ups->upsname);
}

/********************************************************************* 
 * update date battery replaced
 */
void setup_ups_replace(UPSINFO *ups)
{
    char *n;
    char response[32];
    char battdat[9];
    char a = ups->UPS_Cmd[CI_CYCLE_EPROM];
    char c = ups->UPS_Cmd[CI_BATTDAT];
    int i;
    int j = strlen(ups->battdat);
    battdat[0] = '\0';

    if (j != 8) {
        fprintf(stderr, "Error, new battery date must be 8 characters long.\n");
	return;
    }

    strcpy(battdat, ups->battdat);

    /* Ask for battdat */
    write(ups->fd, &c, 1);  /* c = 'x' */
    getline(response, sizeof(response), ups);
    fprintf(stderr, "The old UPS battery date is: %s\n", response);

    /* Tell UPS we will change battdat */
    write(ups->fd, &a, 1);  /* a = '-' */
    sleep(1);

    n = battdat;
    for (i=0; i<8; i++) {
	write(ups->fd, n++, 1);
	sleep(1);
    }

    /* Expect OK after successful battdat change */
    *response = 0;
    getline(response, sizeof(response), ups);
    if (strcmp(response, "OK") != 0)
        fprintf(stderr, "\nError changing UPS battery date\n");

    ups->battdat[0] = '\0';
    smart_poll(ups->UPS_Cmd[CI_BATTDAT], ups);
    strcpy(ups->battdat, smart_poll(ups->UPS_Cmd[CI_BATTDAT], ups));

    fprintf(stderr, "The new UPS battery date is: %s\n", ups->battdat);
}

/*********************************************************************/
int setup_ups_item(UPSINFO *ups, char *title, char cmd, char *setting)
{
    char response[32];
    char response1[32];
    char oldvalue[32];
    char lastvalue[32];
    char allvalues[256];
    char a = ups->UPS_Cmd[CI_CYCLE_EPROM];
    int i;


    /* Ask for old value */
    write(ups->fd, &cmd, 1);
    if (getline(oldvalue, sizeof(oldvalue), ups) == FAILURE) {
        fprintf(stderr, "Could not get old value of %s.\n", title);
	return FAILURE;
    }
    if (strcmp(oldvalue, setting) == 0) {
        fprintf(stderr, "The UPS %s remains unchanged as: %s\n", title, oldvalue);
	return SUCCESS;
    }
    fprintf(stderr, "The old UPS %s is: %s\n", title, oldvalue);
    strcpy(allvalues, oldvalue);
    strcat(allvalues, " ");
    strcpy(lastvalue, oldvalue);

    /* Try a second time to ensure that it is a stable value */
    write(ups->fd, &cmd, 1);
    *response = 0;
    getline(response, sizeof(response), ups);
    if (strcmp(oldvalue, response) != 0) {
        fprintf(stderr, "\nEEPROM value of %s is not stable\n", title);
	return FAILURE;
    }

    /* Just before entering this loop, the last command sent
     * to the UPS MUST be to query the old value.   
     */
    for (i=0; i<10; i++) {
	write(ups->fd, &cmd, 1);
	getline(response1, sizeof(response1), ups);

	/* Tell UPS to cycle to next value */
        write(ups->fd, &a, 1);  /* a = '-' */

	/* Expect OK after successful change */
	*response = 0;
	getline(response, sizeof(response), ups);
        if (strcmp(response, "OK") != 0) {
            fprintf(stderr, "\nError changing UPS %s\n", title);
            fprintf(stderr, "Got %s instead of OK\n\n", response);
	    sleep(10);
	    return FAILURE;
	}

	/* get cycled value */
	write(ups->fd, &cmd, 1);
	getline(response1, sizeof(response1), ups);

	/* get cycled value again */
	write(ups->fd, &cmd, 1);
	if (getline(response, sizeof(response), ups) == FAILURE ||
	     strcmp(response1, response) != 0) {
            fprintf(stderr, "Error cycling values.\n");
	    getline(response, sizeof(response), ups); /* eat any garbage */
	    return FAILURE;
	}
	if (strcmp(setting, response) == 0) {
            fprintf(stderr, "The new UPS %s is: %s\n", title, response);
	    sleep(10);	     /* allow things to settle down */
	    return SUCCESS;
	}

	/* Check if we cycled back to the same value, but permit
	 * a duplicate because the L for sensitivy appears
	 * twice in a row, i.e. H M L L.
	 */
	if (strcmp(oldvalue, response) == 0 && i > 0)
	    break;
	if (strcmp(lastvalue, response) != 0) {
	    strcat(allvalues, response);
            strcat(allvalues, " ");
	    strcpy(lastvalue, response);
	}
        sleep(5);             /* don't cycle too fast */
    }
    fprintf(stderr, "Unable to change %s to: %s\n", title, setting);
    fprintf(stderr, "Permitted values are: %s\n", allvalues);
    getline(response, sizeof(response), ups); /* eat any garbage */
    return FAILURE;
}


/********************************************************************* 
 *
 * This subroutine polls the UPS to find out what capabilities
 * it has.	  
 *
 */
void get_UPS_capabilities (UPSINFO *ups)
{
    char answer[1000];		      /* keep this big to handle big string */
    char caps[1000], *cmds, *p;
    int i;

    /* Get UPS capabilities string */
    strcpy(caps, apc_chat(ups->UPS_Cmd[CI_UPS_CAPS], ups));
    if (strlen(caps) && (strcmp(caps,"NA") != 0)) {
	ups->UPS_Cap[CI_UPS_CAPS] = TRUE;
	/* skip version */
        for (p = caps; *p && *p != '.'; p++) {
	}
	/* skip alert characters */
        for (; *p && *p != '.'; p++) {
	}
	cmds = p;		      /* point to commands */
	if (!*cmds) {		      /* oops, none */
	   cmds = NULL;
	   ups->UPS_Cap[CI_UPS_CAPS] = FALSE;
	}
    } else {
	cmds = NULL;		      /* No commands string */
    }

    /*
     * Try all the possible UPS capabilities and mark the ones supported.
     * If we can get the eprom caps list, use them to jump over the
     * unsupported caps, if we can't get the cap list try every known
     * capability.
     */
    for (i = 0; i <= CI_MAX_CAPS; i++) {
	if (!cmds || strchr(cmds, ups->UPS_Cmd[i]) != NULL) {
	    strcpy(answer, apc_chat(ups->UPS_Cmd[i], ups));
            if (answer && *answer && (strcmp(answer, "NA") != 0)) {
		ups->UPS_Cap[i] = TRUE;
	    }
	}
    }
}

/********************************************************************* 
 *
 * Read data held by UPS
 *
 */
void read_extended (UPSINFO *ups)
{
    get_UPS_capabilities(ups);

    read_static_ups_data(ups);

    get_ups_model(ups);
}

/********************************************************************* 
 *
 * Set new values in UPS memory.  Change the UPS !!!!!!!!
 *
 */
void setup_extended (UPSINFO *ups)
{
    char setting[20];

    get_UPS_capabilities(ups);

    /* Note, a value of -1 in the variable at the beginning
     * means that the user did not put a configuration directive
     * in /etc/apcupsd/apcupsd.conf. Consequently, if no value
     * was given, we won't attept to change it.
     */

    /* SENSITIVITY */
    if (ups->UPS_Cap[CI_SENS] && strcmp(ups->sensitivity, "-1") != 0) {
        sprintf(setting, "%.1s", ups->sensitivity);
        setup_ups_item(ups, "sensitivity", ups->UPS_Cmd[CI_SENS], setting);
    }
     
    /* WAKEUP_DELAY */
    if (ups->UPS_Cap[CI_DWAKE] && ups->dwake != -1) {
        sprintf(setting, "%03d", (int)ups->dwake);
        setup_ups_item(ups, "wakeup delay", ups->UPS_Cmd[CI_DWAKE], setting);
    }

   
    /* SLEEP_DELAY */
    if (ups->UPS_Cap[CI_DSHUTD] && ups->dshutd != -1) {
        sprintf(setting, "%03d", (int)ups->dshutd);
        setup_ups_item(ups, "shutdown delay", ups->UPS_Cmd[CI_DSHUTD], setting);
    }

    /* LOW_TRANSFER_LEVEL */
    if (ups->UPS_Cap[CI_LTRANS] && ups->lotrans != -1) {
        sprintf(setting, "%03d", (int)ups->lotrans);
        setup_ups_item(ups, "lower transfer voltage",
	       ups->UPS_Cmd[CI_LTRANS], setting);
    }

    /* HIGH_TRANSFER_LEVEL */
    if (ups->UPS_Cap[CI_HTRANS] && ups->hitrans != -1) {
        sprintf(setting, "%03d", (int)ups->hitrans);
        setup_ups_item(ups, "upper transfer voltage",
	       ups->UPS_Cmd[CI_HTRANS], setting);
    }

    /* UPS_BATT_CAP_RETURN */
    if (ups->UPS_Cap[CI_RETPCT] && ups->rtnpct != -1) {
        sprintf(setting, "%02d", (int)ups->rtnpct);
        setup_ups_item(ups, "return threshold percent",
	       ups->UPS_Cmd[CI_RETPCT], setting);
    }

    /* ALARM_STATUS */
    if (ups->UPS_Cap[CI_DALARM] && strcmp(ups->beepstate, "-1") != 0) {
        sprintf(setting, "%.1s", ups->beepstate);
        setup_ups_item(ups, "alarm delay",
	       ups->UPS_Cmd[CI_DALARM], setting);
    }

    /* LOWBATT_SHUTDOWN_LEVEL */
    if (ups->UPS_Cap[CI_DLBATT] && ups->dlowbatt != -1) {
        sprintf(setting, "%02d", (int)ups->dlowbatt);
        setup_ups_item(ups, "low battery warning delay",
	       ups->UPS_Cmd[CI_DLBATT], setting);
    }

    /* UPS_SELFTEST */
    if (ups->UPS_Cap[CI_STESTI] && strcmp(ups->selftest, "-1") != 0) {
        sprintf(setting, "%.3s", ups->selftest);
        /* Make sure "ON" is 3 characters */
	if (setting[2] == 0) {
            setting[2] = ' ';
	    setting[3] = 0;
	}
        setup_ups_item(ups, "self test interval", ups->UPS_Cmd[CI_STESTI], setting);
    }

    /* OUTPUT_VOLTAGE */
    if (ups->UPS_Cap[CI_NOMOUTV] && ups->NomOutputVoltage != -1) {
        sprintf(setting, "%03d", (int)ups->NomOutputVoltage);
        setup_ups_item(ups, "output voltage on batteries",
	       ups->UPS_Cmd[CI_NOMOUTV], setting);
    }
}
