/*
 * pgpFixedKey.c
 *
 * Copyright (C) 1996,1997 Pretty Good Privacy, Inc.  All rights reserved.
 *
 * A single, fixed PGP key for decryption operations.
 * Also contains fixed primes and generators for discrete log keys which
 * can share these.
 *
 * $Id: pgpFixedKey.c,v 1.5.2.2 1997/06/09 23:46:27 quark Exp $
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "pgpFixedKey.h"

unsigned char const fixedKeyID[8] = {
	0xB8, 0xE1, 0x21, 0x0E, 0x88, 0x0C, 0x13, 0x99
};

#if 1
#include "pgpESK.h"
#include "pgpRSAKey.h"
/*
 * A (secret) key is:
 *  0                1    Version (= 2 or 3)
 *  1                4    Timestamp
 *  5                2    Validity (= 0 at present)
 *  7                1    Algorithm (=1 for RSA)
 *  8                2+u  MPI for modulus
 * 10+u              2+v  MPI for exponent
 * 12+u+v            1    Encryption algorithm (0 for none, 1 for IDEA)
 * 13+u+v            t    Encryption IV: 0 or 8 bytes
 * 13+t+u+v          2+w  MPI for d
 * 15+t+u+v+w        2+x  MPI for p
 * 17+t+u+v+w+x      2+y  MPI for q
 * 19+t+u+v+w+x+y    2+z  MPI for u
 * 21+t+u+v+w+x+y+z  2    Checksum
 * 23+t+u+v+w+x+y+z
 */
static byte const fixedkey[] = {
	PGPVERSION_2,	/* version */
	0, 0, 0, 0,	/* timestamp */
	0, 0,		/* validity */
	1,		/* algorithm = RSA */
	0x02, 0x00,	/* bits in n (512) */
	0xDB, 0x24, 0x36, 0x7B, 0x91, 0x82, 0x8E, 0x4F,
	0x17, 0xD4, 0xF7, 0x32, 0xD8, 0x0A, 0x59, 0x7C,
	0x96, 0xD8, 0xCB, 0xF7, 0x29, 0xCA, 0xB1, 0xA5,
	0xE2, 0x09, 0x27, 0xD1, 0xA4, 0xCD, 0xD7, 0x6D,
	0xFD, 0x6A, 0x1A, 0xD4, 0x80, 0x4D, 0x52, 0xD6,
	0x2B, 0x29, 0x99, 0xF9, 0x31, 0x3F, 0x65, 0x99,
	0x59, 0x19, 0xDF, 0x62, 0x58, 0x88, 0x31, 0xC9,
	0xB8, 0xE1, 0x21, 0x0E, 0x88, 0x0C, 0x13, 0x99,
	0x00, 0x06,	/* bits in e (512) */
	0x11,
#define FIXEDKEY_PUBLEN (12+64+1)
	0,		/* Unencrypted */
	0x01,0xf8,	/* bits in d */
	      0xEB, 0xB7, 0x16, 0x08, 0x24, 0x0B, 0x5A,
	0xAF, 0x6D, 0x88, 0x91, 0x6C, 0x78, 0x83, 0x9A,
	0x56, 0xAF, 0x29, 0xBB, 0x1F, 0x5E, 0x6E, 0x75,
	0xF0, 0xD0, 0xB5, 0xF2, 0xE5, 0xCD, 0x3C, 0x10,
	0x96, 0xBC, 0xF0, 0x12, 0x26, 0x10, 0x7D, 0xDA,
	0x4F, 0x11, 0xF1, 0x4A, 0xAA, 0xAC, 0xBB, 0xD4,
	0x74, 0xAC, 0x9E, 0xBD, 0x7B, 0xF3, 0xBA, 0xBE,
	0x58, 0x15, 0xBA, 0xE2, 0x77, 0x75, 0xE0, 0xEF,
	0x01,0x00,	/* bits in p */
	0xE9, 0x61, 0x7D, 0xF2, 0x29, 0x47, 0x80, 0x09,
	0x44, 0xDF, 0x0D, 0x7B, 0xD9, 0x01, 0x67, 0x47,
	0x34, 0xD1, 0x5F, 0xD1, 0xC2, 0xB7, 0xAA, 0x72,
	0xBE, 0x4E, 0xA6, 0xD2, 0xD4, 0x2C, 0xCE, 0x4B,
	0x01,0x00,	/* bits in q */
	0xF0, 0x61, 0x6C, 0x02, 0xF3, 0xB0, 0xD1, 0xD7,
	0x63, 0x9C, 0x39, 0x12, 0xAB, 0xA7, 0x5E, 0xCD,
	0xAB, 0xCC, 0xEB, 0x67, 0x59, 0x38, 0xEA, 0x61,
	0x16, 0x5E, 0xBB, 0xB0, 0xA4, 0x48, 0x27, 0x2B,
	0x01,0x00,	/* bits in u */
	0xD7, 0x58, 0xDF, 0x29, 0x01, 0x82, 0xAA, 0x2E,
	0x2A, 0x5A, 0xF9, 0x82, 0x8F, 0x4D, 0x94, 0x55,
	0x50, 0x7A, 0xAD, 0xB6, 0xC0, 0xA6, 0xB8, 0xC8,
	0x57, 0x47, 0x45, 0x2F, 0x55, 0x3D, 0xDC, 0x50,
	0x52, 0xC7	/* Checksum */
};

#if 0 /*Old code - BAT 19970609*/
struct PgpPubKey *fixedKeyPub(void)
{
	int err;	/* Always 0 */

	return rsaPubFromBuf(fixedkey, FIXEDKEY_PUBLEN, &err);
}

struct PgpSecKey *fixedKeySec(void)
{
	int err;	/* Always 0 */

	return rsaSecFromBuf(fixedkey, sizeof(fixedkey), &err);
}

#endif /*0 on old code*/

/* Fixed p, q, g values for DSA keys */

#if 0
static byte const DSA1024p[] = {
	0xF4, 0x88, 0xFD, 0x58, 0x4E, 0x49, 0xDB, 0xCD, 0x20, 0xB4,
	0x9D, 0xE4, 0x91, 0x07, 0x36, 0x6B, 0x33, 0x6C, 0x38, 0x0D,
	0x45, 0x1D, 0x0F, 0x7C, 0x88, 0xB3, 0x1C, 0x7C, 0x5B, 0x2D,
	0x8E, 0xF6, 0xF3, 0xC9, 0x23, 0xC0, 0x43, 0xF0, 0xA5, 0x5B,
	0x18, 0x8D, 0x8E, 0xBB, 0x55, 0x8C, 0xB8, 0x5D, 0x38, 0xD3,
	0x34, 0xFD, 0x7C, 0x17, 0x57, 0x43, 0xA3, 0x1D, 0x18, 0x6C,
	0xDE, 0x33, 0x21, 0x2C, 0xB5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1,
	0x29, 0x40, 0x18, 0x11, 0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72,
	0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8, 0x07, 0x29, 0x7A, 0xCA,
	0x95, 0x0C, 0xD9, 0x96, 0x9F, 0xAB, 0xD0, 0x0A, 0x50, 0x9B,
	0x02, 0x46, 0xD3, 0x08, 0x3D, 0x66, 0xA4, 0xB0, 0xB6, 0x51,
	0xE2, 0xBA, 0xE5, 0x4E, 0x44, 0x39, 0x7C, 0x9F, 0x2A, 0x5E,
	0x31, 0x2F, 0xF6, 0xDB, 0x5C, 0x75, 0xAE, 0x81

};
static byte const DSA1024q[] = {
	0xFF, 0x9F, 0x9C, 0x7C, 0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26,
	0xBA, 0xAB, 0xA2, 0x5E, 0xC3, 0x55, 0xE9, 0x27, 0xC2, 0xB1
};
#endif
static byte const DSA1536p[] = {
	0xF8, 0xF5, 0x59, 0x5F, 0xDC, 0x76, 0x52, 0x4F, 0x3D, 0x3D,
	0x8D, 0xDB, 0xCE, 0x99, 0xE1, 0x57, 0x92, 0x59, 0xCD, 0xFD,
	0xB8, 0xAE, 0x74, 0x4F, 0xC5, 0xFC, 0x76, 0xBC, 0x83, 0xC5,
	0x47, 0x30, 0x61, 0xCE, 0x7C, 0xC9, 0x66, 0xFF, 0x15, 0xF9,
	0xBB, 0xFD, 0x91, 0x5E, 0xC7, 0x01, 0xAA, 0xD3, 0x5B, 0x9E,
	0x8D, 0xA0, 0xA5, 0x72, 0x3A, 0xD4, 0x1A, 0xF0, 0xBF, 0x46,
	0x00, 0x58, 0x2B, 0xE5, 0xF4, 0x88, 0xFD, 0x58, 0x4E, 0x49,
	0xDB, 0xCD, 0x20, 0xB4, 0x9D, 0xE4, 0x91, 0x07, 0x36, 0x6B,
	0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D, 0x0F, 0x7C, 0x88, 0xB3,
	0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6, 0xF3, 0xC9, 0x23, 0xC0,
	0x43, 0xF0, 0xA5, 0x5B, 0x18, 0x8D, 0x8E, 0xBB, 0x55, 0x8C,
	0xB8, 0x5D, 0x38, 0xD3, 0x34, 0xFD, 0x7C, 0x17, 0x57, 0x43,
	0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33, 0x21, 0x2C, 0xB5, 0x2A,
	0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40, 0x18, 0x11, 0x8D, 0x7C,
	0x84, 0xA7, 0x0A, 0x72, 0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8,
	0x07, 0x29, 0x7A, 0xCA, 0x95, 0x0C, 0xD9, 0x96, 0x9F, 0xAB,
	0xD0, 0x0A, 0x50, 0x9B, 0x02, 0x47, 0x2C, 0xCB, 0xA6, 0x82,
	0xA5, 0xEB, 0xA2, 0x84, 0x3E, 0xE3, 0x3D, 0x56, 0xE7, 0xA9,
	0xF3, 0xE1, 0xC4, 0x3D, 0x22, 0x98, 0x80, 0xF3, 0xC3, 0x08,
	0x49, 0x8F
};
static byte const DSA1536q[] = {
	0x3F, 0xFD, 0x66, 0xA4, 0x5D, 0x41, 0x9F, 0x9C, 0x7C, 0xBD,
	0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA, 0xAB, 0xA2, 0x5E, 0xC3,
	0x55, 0xE9, 0x27, 0xC2, 0x43
};
static byte const DSA2048p[] = {
	0xF6, 0x42, 0x57, 0xB7, 0x08, 0x7F, 0x08, 0x17, 0x72, 0xA2,
	0xBA, 0xD6, 0xA9, 0x42, 0xF3, 0x05, 0xE8, 0xF9, 0x53, 0x11,
	0x39, 0x4F, 0xB6, 0xF1, 0x6E, 0xB9, 0x4B, 0x38, 0x20, 0xDA,
	0x01, 0xA7, 0x56, 0xA3, 0x14, 0xE9, 0x8F, 0x40, 0x55, 0xF3,
	0xD0, 0x07, 0xC6, 0xCB, 0x43, 0xA9, 0x94, 0xAD, 0xF7, 0x4C,
	0x64, 0x86, 0x49, 0xF8, 0x0C, 0x83, 0xBD, 0x65, 0xE9, 0x17,
	0xD4, 0xA1, 0xD3, 0x50, 0xF8, 0xF5, 0x59, 0x5F, 0xDC, 0x76,
	0x52, 0x4F, 0x3D, 0x3D, 0x8D, 0xDB, 0xCE, 0x99, 0xE1, 0x57,
	0x92, 0x59, 0xCD, 0xFD, 0xB8, 0xAE, 0x74, 0x4F, 0xC5, 0xFC,
	0x76, 0xBC, 0x83, 0xC5, 0x47, 0x30, 0x61, 0xCE, 0x7C, 0xC9,
	0x66, 0xFF, 0x15, 0xF9, 0xBB, 0xFD, 0x91, 0x5E, 0xC7, 0x01,
	0xAA, 0xD3, 0x5B, 0x9E, 0x8D, 0xA0, 0xA5, 0x72, 0x3A, 0xD4,
	0x1A, 0xF0, 0xBF, 0x46, 0x00, 0x58, 0x2B, 0xE5, 0xF4, 0x88,
	0xFD, 0x58, 0x4E, 0x49, 0xDB, 0xCD, 0x20, 0xB4, 0x9D, 0xE4,
	0x91, 0x07, 0x36, 0x6B, 0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D,
	0x0F, 0x7C, 0x88, 0xB3, 0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6,
	0xF3, 0xC9, 0x23, 0xC0, 0x43, 0xF0, 0xA5, 0x5B, 0x18, 0x8D,
	0x8E, 0xBB, 0x55, 0x8C, 0xB8, 0x5D, 0x38, 0xD3, 0x34, 0xFD,
	0x7C, 0x17, 0x57, 0x43, 0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33,
	0x21, 0x2C, 0xB5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40,
	0x18, 0x11, 0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72, 0xD6, 0x86,
	0xC4, 0x03, 0x19, 0xC8, 0x07, 0x29, 0x7A, 0xCA, 0x95, 0x0C,
	0xD9, 0x96, 0x9F, 0xAB, 0xD0, 0x0A, 0x51, 0xB8, 0x4F, 0xCF,
	0x7E, 0xC9, 0x6A, 0x96, 0xFA, 0x9E, 0x4E, 0xDC, 0x9F, 0x93,
	0x37, 0x21, 0x13, 0x1D, 0xE3, 0xDD, 0x3D, 0xE0, 0x7D, 0x1D,
	0x6B, 0xCE, 0x09, 0x83, 0x11, 0xB5
};
static byte const DSA2048q[] = {
	0x01, 0xFE, 0x46, 0xD3, 0x08, 0x3D, 0x66, 0xA4, 0x5D, 0x41,
	0x9F, 0x9C, 0x7C, 0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA,
	0xAB, 0xA2, 0x5E, 0xC3, 0x55, 0xE9, 0x27, 0xC2, 0xCF
};
static byte const DSA3072p[] = {
	0xCC, 0x1D, 0x77, 0x57, 0x24, 0x38, 0x4A, 0xE2, 0xC4, 0xF0,
	0xE8, 0x8E, 0x13, 0x67, 0x97, 0x4E, 0x92, 0x13, 0x61, 0xF6,
	0xDB, 0xEB, 0x25, 0x0E, 0x17, 0xFD, 0xF6, 0x98, 0xF7, 0xC8,
	0x7C, 0x79, 0xB0, 0x72, 0x1D, 0x38, 0x75, 0xFB, 0xF6, 0xC1,
	0x73, 0xC4, 0x83, 0x11, 0x26, 0x2B, 0x43, 0x60, 0xC3, 0xE3,
	0xE8, 0xD6, 0x0A, 0xFD, 0xA1, 0x28, 0x26, 0x0B, 0xAE, 0xA9,
	0xAE, 0xB3, 0x65, 0x0F, 0xA2, 0x00, 0x53, 0x01, 0xA0, 0x7C,
	0xD6, 0xAB, 0xA3, 0x12, 0x1E, 0xFA, 0x0F, 0x2A, 0xCE, 0x1F,
	0x74, 0x84, 0x4F, 0xCA, 0xF3, 0x17, 0xF3, 0xA4, 0x40, 0xE9,
	0xD7, 0xD2, 0x77, 0xB6, 0x42, 0x2D, 0x02, 0x36, 0xC1, 0x26,
	0xCB, 0x68, 0x5E, 0x9D, 0x7C, 0x98, 0x09, 0x0A, 0x8D, 0x7E,
	0x2D, 0xED, 0xE4, 0x5D, 0x79, 0xF5, 0xD4, 0x92, 0x4F, 0x9B,
	0x18, 0x8E, 0xFC, 0x2A, 0xA7, 0x4B, 0x7C, 0x32, 0xF6, 0x42,
	0x57, 0xB7, 0x08, 0x7F, 0x08, 0x17, 0x72, 0xA2, 0xBA, 0xD6,
	0xA9, 0x42, 0xF3, 0x05, 0xE8, 0xF9, 0x53, 0x11, 0x39, 0x4F,
	0xB6, 0xF1, 0x6E, 0xB9, 0x4B, 0x38, 0x20, 0xDA, 0x01, 0xA7,
	0x56, 0xA3, 0x14, 0xE9, 0x8F, 0x40, 0x55, 0xF3, 0xD0, 0x07,
	0xC6, 0xCB, 0x43, 0xA9, 0x94, 0xAD, 0xF7, 0x4C, 0x64, 0x86,
	0x49, 0xF8, 0x0C, 0x83, 0xBD, 0x65, 0xE9, 0x17, 0xD4, 0xA1,
	0xD3, 0x50, 0xF8, 0xF5, 0x59, 0x5F, 0xDC, 0x76, 0x52, 0x4F,
	0x3D, 0x3D, 0x8D, 0xDB, 0xCE, 0x99, 0xE1, 0x57, 0x92, 0x59,
	0xCD, 0xFD, 0xB8, 0xAE, 0x74, 0x4F, 0xC5, 0xFC, 0x76, 0xBC,
	0x83, 0xC5, 0x47, 0x30, 0x61, 0xCE, 0x7C, 0xC9, 0x66, 0xFF,
	0x15, 0xF9, 0xBB, 0xFD, 0x91, 0x5E, 0xC7, 0x01, 0xAA, 0xD3,
	0x5B, 0x9E, 0x8D, 0xA0, 0xA5, 0x72, 0x3A, 0xD4, 0x1A, 0xF0,
	0xBF, 0x46, 0x00, 0x58, 0x2B, 0xE5, 0xF4, 0x88, 0xFD, 0x58,
	0x4E, 0x49, 0xDB, 0xCD, 0x20, 0xB4, 0x9D, 0xE4, 0x91, 0x07,
	0x36, 0x6B, 0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D, 0x0F, 0x7C,
	0x88, 0xB3, 0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6, 0xF3, 0xC9,
	0x23, 0xC0, 0x43, 0xF0, 0xA5, 0x5B, 0x18, 0x8D, 0x8E, 0xBB,
	0x55, 0x8C, 0xB8, 0x5D, 0x38, 0xD3, 0x34, 0xFD, 0x7C, 0x17,
	0x57, 0x43, 0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33, 0x21, 0x2C,
	0xB5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40, 0x18, 0x11,
	0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72, 0xD6, 0x86, 0xC4, 0x03,
	0x19, 0xC8, 0x07, 0x29, 0x7A, 0xCA, 0x95, 0x0C, 0xDB, 0x13,
	0xD8, 0xA2, 0x1A, 0x8F, 0x32, 0x6B, 0x11, 0xEB, 0xA3, 0xF2,
	0x71, 0x5E, 0xF5, 0xC5, 0x43, 0x6D, 0x84, 0x4B, 0xF6, 0xFE,
	0x52, 0xB9, 0x60, 0x5A, 0x8E, 0x09, 0xCE, 0xCB, 0xFE, 0x94,
	0xBC, 0xEF, 0x23, 0x5B
};
static byte const DSA3072q[] = {
	0x3F, 0xEB, 0xD0, 0x0A, 0x50, 0x9B, 0x02, 0x46, 0xD3, 0x08,
	0x3D, 0x66, 0xA4, 0x5D, 0x41, 0x9F, 0x9C, 0x7C, 0xBD, 0x89,
	0x4B, 0x22, 0x19, 0x26, 0xBA, 0xAB, 0xA2, 0x5E, 0xC3, 0x55,
	0xE9, 0x27, 0xC3, 0x1D

};
static byte const DSA4096p[] = {
	0xF9, 0x18, 0xA0, 0x7E, 0x5A, 0x06, 0x61, 0x7A, 0x43, 0x90,
	0x95, 0xDC, 0x05, 0x6C, 0x87, 0x86, 0xEC, 0x61, 0xEC, 0xCD,
	0x45, 0x1F, 0x0E, 0xD8, 0xE0, 0xA3, 0x79, 0xC6, 0xC9, 0xDC,
	0x7A, 0x0B, 0xAC, 0xE4, 0x3F, 0xE3, 0x46, 0x94, 0xB6, 0x30,
	0x4A, 0x53, 0xD7, 0x7C, 0x02, 0x16, 0x48, 0x80, 0xB5, 0x15,
	0xE5, 0x29, 0x99, 0xA9, 0x9F, 0x07, 0x74, 0xD3, 0xFF, 0xE3,
	0xA1, 0xC5, 0x96, 0x20, 0x4E, 0x98, 0x65, 0xB8, 0xD8, 0x0D,
	0xEE, 0x10, 0x5D, 0xAB, 0xB6, 0x17, 0x1C, 0x51, 0xD8, 0x50,
	0xCA, 0x22, 0x57, 0x43, 0x29, 0xBE, 0x95, 0xE8, 0x56, 0x2B,
	0x38, 0x78, 0x5C, 0x0B, 0xDB, 0xF8, 0x4C, 0x4D, 0xD5, 0xE3,
	0xAA, 0x46, 0xCC, 0xFB, 0xCE, 0x17, 0xE8, 0x2A, 0x9D, 0x14,
	0x61, 0xE3, 0x84, 0xA9, 0x4F, 0xD1, 0x83, 0x84, 0xA8, 0x79,
	0xB6, 0xEF, 0x8F, 0xA7, 0x43, 0x46, 0x08, 0xC6, 0xCC, 0x1D,
	0x77, 0x57, 0x24, 0x38, 0x4A, 0xE2, 0xC4, 0xF0, 0xE8, 0x8E,
	0x13, 0x67, 0x97, 0x4E, 0x92, 0x13, 0x61, 0xF6, 0xDB, 0xEB,
	0x25, 0x0E, 0x17, 0xFD, 0xF6, 0x98, 0xF7, 0xC8, 0x7C, 0x79,
	0xB0, 0x72, 0x1D, 0x38, 0x75, 0xFB, 0xF6, 0xC1, 0x73, 0xC4,
	0x83, 0x11, 0x26, 0x2B, 0x43, 0x60, 0xC3, 0xE3, 0xE8, 0xD6,
	0x0A, 0xFD, 0xA1, 0x28, 0x26, 0x0B, 0xAE, 0xA9, 0xAE, 0xB3,
	0x65, 0x0F, 0xA2, 0x00, 0x53, 0x01, 0xA0, 0x7C, 0xD6, 0xAB,
	0xA3, 0x12, 0x1E, 0xFA, 0x0F, 0x2A, 0xCE, 0x1F, 0x74, 0x84,
	0x4F, 0xCA, 0xF3, 0x17, 0xF3, 0xA4, 0x40, 0xE9, 0xD7, 0xD2,
	0x77, 0xB6, 0x42, 0x2D, 0x02, 0x36, 0xC1, 0x26, 0xCB, 0x68,
	0x5E, 0x9D, 0x7C, 0x98, 0x09, 0x0A, 0x8D, 0x7E, 0x2D, 0xED,
	0xE4, 0x5D, 0x79, 0xF5, 0xD4, 0x92, 0x4F, 0x9B, 0x18, 0x8E,
	0xFC, 0x2A, 0xA7, 0x4B, 0x7C, 0x32, 0xF6, 0x42, 0x57, 0xB7,
	0x08, 0x7F, 0x08, 0x17, 0x72, 0xA2, 0xBA, 0xD6, 0xA9, 0x42,
	0xF3, 0x05, 0xE8, 0xF9, 0x53, 0x11, 0x39, 0x4F, 0xB6, 0xF1,
	0x6E, 0xB9, 0x4B, 0x38, 0x20, 0xDA, 0x01, 0xA7, 0x56, 0xA3,
	0x14, 0xE9, 0x8F, 0x40, 0x55, 0xF3, 0xD0, 0x07, 0xC6, 0xCB,
	0x43, 0xA9, 0x94, 0xAD, 0xF7, 0x4C, 0x64, 0x86, 0x49, 0xF8,
	0x0C, 0x83, 0xBD, 0x65, 0xE9, 0x17, 0xD4, 0xA1, 0xD3, 0x50,
	0xF8, 0xF5, 0x59, 0x5F, 0xDC, 0x76, 0x52, 0x4F, 0x3D, 0x3D,
	0x8D, 0xDB, 0xCE, 0x99, 0xE1, 0x57, 0x92, 0x59, 0xCD, 0xFD,
	0xB8, 0xAE, 0x74, 0x4F, 0xC5, 0xFC, 0x76, 0xBC, 0x83, 0xC5,
	0x47, 0x30, 0x61, 0xCE, 0x7C, 0xC9, 0x66, 0xFF, 0x15, 0xF9,
	0xBB, 0xFD, 0x91, 0x5E, 0xC7, 0x01, 0xAA, 0xD3, 0x5B, 0x9E,
	0x8D, 0xA0, 0xA5, 0x72, 0x3A, 0xD4, 0x1A, 0xF0, 0xBF, 0x46,
	0x00, 0x58, 0x2B, 0xE5, 0xF4, 0x88, 0xFD, 0x58, 0x4E, 0x49,
	0xDB, 0xCD, 0x20, 0xB4, 0x9D, 0xE4, 0x91, 0x07, 0x36, 0x6B,
	0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D, 0x0F, 0x7C, 0x88, 0xB3,
	0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6, 0xF3, 0xC9, 0x23, 0xC0,
	0x43, 0xF0, 0xA5, 0x5B, 0x18, 0x8D, 0x8E, 0xBB, 0x55, 0x8C,
	0xB8, 0x5D, 0x38, 0xD3, 0x34, 0xFD, 0x7C, 0x17, 0x57, 0x43,
	0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33, 0x21, 0x2C, 0xB5, 0x2A,
	0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40, 0x18, 0x11, 0x8D, 0x7C,
	0x84, 0xA7, 0x0A, 0x72, 0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8,
	0x07, 0x29, 0x88, 0x50, 0x05, 0x4B, 0xFE, 0x24, 0x36, 0x96,
	0x47, 0xAB, 0x40, 0x4A, 0xA2, 0xE5, 0xB3, 0xCB, 0x0E, 0x0F,
	0x2A, 0x1C, 0x45, 0x17, 0x3D, 0xD9, 0x5D, 0xED, 0x9F, 0x0D,
	0x33, 0xDA, 0x04, 0xC9, 0x19, 0xA9, 0x9D, 0x3F, 0x40, 0xF3,
	0xF4, 0x2D
};
static byte const DSA4096q[] = {
	0x01, 0xFF, 0x0C, 0xD9, 0x96, 0x9F, 0xAB, 0xD0, 0x0A, 0x50,
	0x9B, 0x02, 0x46, 0xD3, 0x08, 0x3D, 0x66, 0xA4, 0x5D, 0x41,
	0x9F, 0x9C, 0x7C, 0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA,
	0xAB, 0xA2, 0x5E, 0xC3, 0x55, 0xE9, 0x27, 0xC2, 0x65

};

static struct {
	unsigned	bits;
	byte const *p;
	unsigned	psize;
	byte const *q;
	unsigned	qsize;
} DSAtab[] = {
#if 0
	{1024, DSA1024p, sizeof(DSA1024p),
	 	DSA1024q, sizeof(DSA1024q)},
#endif
	{1536, DSA1536p, sizeof(DSA1536p),
	 	DSA1536q, sizeof(DSA1536q)},
	{2048, DSA2048p, sizeof(DSA2048p),
	 	DSA2048q, sizeof(DSA2048q)},
	{3072, DSA3072p, sizeof(DSA3072p),
	 	DSA3072q, sizeof(DSA3072q)},
	{4096, DSA4096p, sizeof(DSA4096p),
	 	DSA4096q, sizeof(DSA4096q)}
};


/* Return >0 if we find one, and fill in pointers, else return 0 */
int
pgpDSAfixed (unsigned bits, byte const **p, size_t *plen,
	byte const **q, size_t *qlen)
{
#if 0
	(void)bits;		/* Avoid warnings */
	(void)p;
	(void)plen;
	(void)q;
	(void)qlen;
#else
	unsigned i;

	for (i=0; i<sizeof(DSAtab)/sizeof(DSAtab[0]); ++i) {
		if (DSAtab[i].bits > bits)
			return 0;
		if (DSAtab[i].bits == bits) {
			if (p)
				*p = DSAtab[i].p;
			if (plen)
				*plen = DSAtab[i].psize;
			if (q)
				*q = DSAtab[i].q;
			if (qlen)
				*qlen = DSAtab[i].qsize;
			return 1;
		}
	}
#endif
	return 0;
}



/*
 * Fixed p, g values for ElG keys.  The p values are all "strong" primes,
 * such that (p-1)/2 is also prime.  This makes 2 an appropriate generator
 * in each case.
 */

#if 0
static byte const ElG1024p[] = {
	0xF4, 0x88, 0xFD, 0x58, 0x4E, 0x49, 0xDB, 0xCD,
	0x20, 0xB4, 0x9D, 0xE4, 0x91, 0x07, 0x36, 0x6B,
	0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D, 0x0F, 0x7C,
	0x88, 0xB3, 0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6,
	0xF3, 0xC9, 0x23, 0xC0, 0x43, 0xF0, 0xA5, 0x5B,
	0x18, 0x8D, 0x8E, 0xBB, 0x55, 0x8C, 0xB8, 0x5D,
	0x38, 0xD3, 0x34, 0xFD, 0x7C, 0x17, 0x57, 0x43,
	0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33, 0x21, 0x2C,
	0xB5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40,
	0x18, 0x11, 0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72,
	0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8, 0x07, 0x29,
	0x7A, 0xCA, 0x95, 0x0C, 0xD9, 0x96, 0x9F, 0xAB,
	0xD0, 0x0A, 0x50, 0x9B, 0x02, 0x46, 0xD3, 0x08,
	0x3D, 0x66, 0xA4, 0x5D, 0x41, 0x9F, 0x9C, 0x7C,
	0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA, 0xAB,
	0xA2, 0x5E, 0xC3, 0x55, 0xE9, 0x2F, 0x78, 0xC7
};
static byte const ElG1024g[] = {
	0x02
};
#endif

static byte const ElG1536p[] = {
	0xF8, 0xF5, 0x59, 0x5F, 0xDC, 0x76, 0x52, 0x4F,
	0x3D, 0x3D, 0x8D, 0xDB, 0xCE, 0x99, 0xE1, 0x57,
	0x92, 0x59, 0xCD, 0xFD, 0xB8, 0xAE, 0x74, 0x4F,
	0xC5, 0xFC, 0x76, 0xBC, 0x83, 0xC5, 0x47, 0x30,
	0x61, 0xCE, 0x7C, 0xC9, 0x66, 0xFF, 0x15, 0xF9,
	0xBB, 0xFD, 0x91, 0x5E, 0xC7, 0x01, 0xAA, 0xD3,
	0x5B, 0x9E, 0x8D, 0xA0, 0xA5, 0x72, 0x3A, 0xD4,
	0x1A, 0xF0, 0xBF, 0x46, 0x00, 0x58, 0x2B, 0xE5,
	0xF4, 0x88, 0xFD, 0x58, 0x4E, 0x49, 0xDB, 0xCD,
	0x20, 0xB4, 0x9D, 0xE4, 0x91, 0x07, 0x36, 0x6B,
	0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D, 0x0F, 0x7C,
	0x88, 0xB3, 0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6,
	0xF3, 0xC9, 0x23, 0xC0, 0x43, 0xF0, 0xA5, 0x5B,
	0x18, 0x8D, 0x8E, 0xBB, 0x55, 0x8C, 0xB8, 0x5D,
	0x38, 0xD3, 0x34, 0xFD, 0x7C, 0x17, 0x57, 0x43,
	0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33, 0x21, 0x2C,
	0xB5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40,
	0x18, 0x11, 0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72,
	0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8, 0x07, 0x29,
	0x7A, 0xCA, 0x95, 0x0C, 0xD9, 0x96, 0x9F, 0xAB,
	0xD0, 0x0A, 0x50, 0x9B, 0x02, 0x46, 0xD3, 0x08,
	0x3D, 0x66, 0xA4, 0x5D, 0x41, 0x9F, 0x9C, 0x7C,
	0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA, 0xAB,
	0xA2, 0x5E, 0xC3, 0x55, 0xE9, 0x58, 0x7C, 0x47
};
static byte const ElG1536g[] = {
	0x02
};
static byte const ElG2048p[] = {
	0xF6, 0x42, 0x57, 0xB7, 0x08, 0x7F, 0x08, 0x17,
	0x72, 0xA2, 0xBA, 0xD6, 0xA9, 0x42, 0xF3, 0x05,
	0xE8, 0xF9, 0x53, 0x11, 0x39, 0x4F, 0xB6, 0xF1,
	0x6E, 0xB9, 0x4B, 0x38, 0x20, 0xDA, 0x01, 0xA7,
	0x56, 0xA3, 0x14, 0xE9, 0x8F, 0x40, 0x55, 0xF3,
	0xD0, 0x07, 0xC6, 0xCB, 0x43, 0xA9, 0x94, 0xAD,
	0xF7, 0x4C, 0x64, 0x86, 0x49, 0xF8, 0x0C, 0x83,
	0xBD, 0x65, 0xE9, 0x17, 0xD4, 0xA1, 0xD3, 0x50,
	0xF8, 0xF5, 0x59, 0x5F, 0xDC, 0x76, 0x52, 0x4F,
	0x3D, 0x3D, 0x8D, 0xDB, 0xCE, 0x99, 0xE1, 0x57,
	0x92, 0x59, 0xCD, 0xFD, 0xB8, 0xAE, 0x74, 0x4F,
	0xC5, 0xFC, 0x76, 0xBC, 0x83, 0xC5, 0x47, 0x30,
	0x61, 0xCE, 0x7C, 0xC9, 0x66, 0xFF, 0x15, 0xF9,
	0xBB, 0xFD, 0x91, 0x5E, 0xC7, 0x01, 0xAA, 0xD3,
	0x5B, 0x9E, 0x8D, 0xA0, 0xA5, 0x72, 0x3A, 0xD4,
	0x1A, 0xF0, 0xBF, 0x46, 0x00, 0x58, 0x2B, 0xE5,
	0xF4, 0x88, 0xFD, 0x58, 0x4E, 0x49, 0xDB, 0xCD,
	0x20, 0xB4, 0x9D, 0xE4, 0x91, 0x07, 0x36, 0x6B,
	0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D, 0x0F, 0x7C,
	0x88, 0xB3, 0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6,
	0xF3, 0xC9, 0x23, 0xC0, 0x43, 0xF0, 0xA5, 0x5B,
	0x18, 0x8D, 0x8E, 0xBB, 0x55, 0x8C, 0xB8, 0x5D,
	0x38, 0xD3, 0x34, 0xFD, 0x7C, 0x17, 0x57, 0x43,
	0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33, 0x21, 0x2C,
	0xB5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40,
	0x18, 0x11, 0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72,
	0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8, 0x07, 0x29,
	0x7A, 0xCA, 0x95, 0x0C, 0xD9, 0x96, 0x9F, 0xAB,
	0xD0, 0x0A, 0x50, 0x9B, 0x02, 0x46, 0xD3, 0x08,
	0x3D, 0x66, 0xA4, 0x5D, 0x41, 0x9F, 0x9C, 0x7C,
	0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA, 0xAB,
	0xA2, 0x5E, 0xC3, 0x55, 0xE9, 0x32, 0x0B, 0x3B
};
static byte const ElG2048g[] = {
	0x02
};
static byte const ElG3072p[] = {
	0xCC, 0x1D, 0x77, 0x57, 0x24, 0x38, 0x4A, 0xE2,
	0xC4, 0xF0, 0xE8, 0x8E, 0x13, 0x67, 0x97, 0x4E,
	0x92, 0x13, 0x61, 0xF6, 0xDB, 0xEB, 0x25, 0x0E,
	0x17, 0xFD, 0xF6, 0x98, 0xF7, 0xC8, 0x7C, 0x79,
	0xB0, 0x72, 0x1D, 0x38, 0x75, 0xFB, 0xF6, 0xC1,
	0x73, 0xC4, 0x83, 0x11, 0x26, 0x2B, 0x43, 0x60,
	0xC3, 0xE3, 0xE8, 0xD6, 0x0A, 0xFD, 0xA1, 0x28,
	0x26, 0x0B, 0xAE, 0xA9, 0xAE, 0xB3, 0x65, 0x0F,
	0xA2, 0x00, 0x53, 0x01, 0xA0, 0x7C, 0xD6, 0xAB,
	0xA3, 0x12, 0x1E, 0xFA, 0x0F, 0x2A, 0xCE, 0x1F,
	0x74, 0x84, 0x4F, 0xCA, 0xF3, 0x17, 0xF3, 0xA4,
	0x40, 0xE9, 0xD7, 0xD2, 0x77, 0xB6, 0x42, 0x2D,
	0x02, 0x36, 0xC1, 0x26, 0xCB, 0x68, 0x5E, 0x9D,
	0x7C, 0x98, 0x09, 0x0A, 0x8D, 0x7E, 0x2D, 0xED,
	0xE4, 0x5D, 0x79, 0xF5, 0xD4, 0x92, 0x4F, 0x9B,
	0x18, 0x8E, 0xFC, 0x2A, 0xA7, 0x4B, 0x7C, 0x32,
	0xF6, 0x42, 0x57, 0xB7, 0x08, 0x7F, 0x08, 0x17,
	0x72, 0xA2, 0xBA, 0xD6, 0xA9, 0x42, 0xF3, 0x05,
	0xE8, 0xF9, 0x53, 0x11, 0x39, 0x4F, 0xB6, 0xF1,
	0x6E, 0xB9, 0x4B, 0x38, 0x20, 0xDA, 0x01, 0xA7,
	0x56, 0xA3, 0x14, 0xE9, 0x8F, 0x40, 0x55, 0xF3,
	0xD0, 0x07, 0xC6, 0xCB, 0x43, 0xA9, 0x94, 0xAD,
	0xF7, 0x4C, 0x64, 0x86, 0x49, 0xF8, 0x0C, 0x83,
	0xBD, 0x65, 0xE9, 0x17, 0xD4, 0xA1, 0xD3, 0x50,
	0xF8, 0xF5, 0x59, 0x5F, 0xDC, 0x76, 0x52, 0x4F,
	0x3D, 0x3D, 0x8D, 0xDB, 0xCE, 0x99, 0xE1, 0x57,
	0x92, 0x59, 0xCD, 0xFD, 0xB8, 0xAE, 0x74, 0x4F,
	0xC5, 0xFC, 0x76, 0xBC, 0x83, 0xC5, 0x47, 0x30,
	0x61, 0xCE, 0x7C, 0xC9, 0x66, 0xFF, 0x15, 0xF9,
	0xBB, 0xFD, 0x91, 0x5E, 0xC7, 0x01, 0xAA, 0xD3,
	0x5B, 0x9E, 0x8D, 0xA0, 0xA5, 0x72, 0x3A, 0xD4,
	0x1A, 0xF0, 0xBF, 0x46, 0x00, 0x58, 0x2B, 0xE5,
	0xF4, 0x88, 0xFD, 0x58, 0x4E, 0x49, 0xDB, 0xCD,
	0x20, 0xB4, 0x9D, 0xE4, 0x91, 0x07, 0x36, 0x6B,
	0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D, 0x0F, 0x7C,
	0x88, 0xB3, 0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6,
	0xF3, 0xC9, 0x23, 0xC0, 0x43, 0xF0, 0xA5, 0x5B,
	0x18, 0x8D, 0x8E, 0xBB, 0x55, 0x8C, 0xB8, 0x5D,
	0x38, 0xD3, 0x34, 0xFD, 0x7C, 0x17, 0x57, 0x43,
	0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33, 0x21, 0x2C,
	0xB5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40,
	0x18, 0x11, 0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72,
	0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8, 0x07, 0x29,
	0x7A, 0xCA, 0x95, 0x0C, 0xD9, 0x96, 0x9F, 0xAB,
	0xD0, 0x0A, 0x50, 0x9B, 0x02, 0x46, 0xD3, 0x08,
	0x3D, 0x66, 0xA4, 0x5D, 0x41, 0x9F, 0x9C, 0x7C,
	0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA, 0xAB,
	0xA2, 0x5E, 0xC3, 0x55, 0xE9, 0x4C, 0x32, 0x6F
};
static byte const ElG3072g[] = {
	0x02
};
static byte const ElG4096p[] = {
	0xF9, 0x18, 0xA0, 0x7E, 0x5A, 0x06, 0x61, 0x7A,
	0x43, 0x90, 0x95, 0xDC, 0x05, 0x6C, 0x87, 0x86,
	0xEC, 0x61, 0xEC, 0xCD, 0x45, 0x1F, 0x0E, 0xD8,
	0xE0, 0xA3, 0x79, 0xC6, 0xC9, 0xDC, 0x7A, 0x0B,
	0xAC, 0xE4, 0x3F, 0xE3, 0x46, 0x94, 0xB6, 0x30,
	0x4A, 0x53, 0xD7, 0x7C, 0x02, 0x16, 0x48, 0x80,
	0xB5, 0x15, 0xE5, 0x29, 0x99, 0xA9, 0x9F, 0x07,
	0x74, 0xD3, 0xFF, 0xE3, 0xA1, 0xC5, 0x96, 0x20,
	0x4E, 0x98, 0x65, 0xB8, 0xD8, 0x0D, 0xEE, 0x10,
	0x5D, 0xAB, 0xB6, 0x17, 0x1C, 0x51, 0xD8, 0x50,
	0xCA, 0x22, 0x57, 0x43, 0x29, 0xBE, 0x95, 0xE8,
	0x56, 0x2B, 0x38, 0x78, 0x5C, 0x0B, 0xDB, 0xF8,
	0x4C, 0x4D, 0xD5, 0xE3, 0xAA, 0x46, 0xCC, 0xFB,
	0xCE, 0x17, 0xE8, 0x2A, 0x9D, 0x14, 0x61, 0xE3,
	0x84, 0xA9, 0x4F, 0xD1, 0x83, 0x84, 0xA8, 0x79,
	0xB6, 0xEF, 0x8F, 0xA7, 0x43, 0x46, 0x08, 0xC6,
	0xCC, 0x1D, 0x77, 0x57, 0x24, 0x38, 0x4A, 0xE2,
	0xC4, 0xF0, 0xE8, 0x8E, 0x13, 0x67, 0x97, 0x4E,
	0x92, 0x13, 0x61, 0xF6, 0xDB, 0xEB, 0x25, 0x0E,
	0x17, 0xFD, 0xF6, 0x98, 0xF7, 0xC8, 0x7C, 0x79,
	0xB0, 0x72, 0x1D, 0x38, 0x75, 0xFB, 0xF6, 0xC1,
	0x73, 0xC4, 0x83, 0x11, 0x26, 0x2B, 0x43, 0x60,
	0xC3, 0xE3, 0xE8, 0xD6, 0x0A, 0xFD, 0xA1, 0x28,
	0x26, 0x0B, 0xAE, 0xA9, 0xAE, 0xB3, 0x65, 0x0F,
	0xA2, 0x00, 0x53, 0x01, 0xA0, 0x7C, 0xD6, 0xAB,
	0xA3, 0x12, 0x1E, 0xFA, 0x0F, 0x2A, 0xCE, 0x1F,
	0x74, 0x84, 0x4F, 0xCA, 0xF3, 0x17, 0xF3, 0xA4,
	0x40, 0xE9, 0xD7, 0xD2, 0x77, 0xB6, 0x42, 0x2D,
	0x02, 0x36, 0xC1, 0x26, 0xCB, 0x68, 0x5E, 0x9D,
	0x7C, 0x98, 0x09, 0x0A, 0x8D, 0x7E, 0x2D, 0xED,
	0xE4, 0x5D, 0x79, 0xF5, 0xD4, 0x92, 0x4F, 0x9B,
	0x18, 0x8E, 0xFC, 0x2A, 0xA7, 0x4B, 0x7C, 0x32,
	0xF6, 0x42, 0x57, 0xB7, 0x08, 0x7F, 0x08, 0x17,
	0x72, 0xA2, 0xBA, 0xD6, 0xA9, 0x42, 0xF3, 0x05,
	0xE8, 0xF9, 0x53, 0x11, 0x39, 0x4F, 0xB6, 0xF1,
	0x6E, 0xB9, 0x4B, 0x38, 0x20, 0xDA, 0x01, 0xA7,
	0x56, 0xA3, 0x14, 0xE9, 0x8F, 0x40, 0x55, 0xF3,
	0xD0, 0x07, 0xC6, 0xCB, 0x43, 0xA9, 0x94, 0xAD,
	0xF7, 0x4C, 0x64, 0x86, 0x49, 0xF8, 0x0C, 0x83,
	0xBD, 0x65, 0xE9, 0x17, 0xD4, 0xA1, 0xD3, 0x50,
	0xF8, 0xF5, 0x59, 0x5F, 0xDC, 0x76, 0x52, 0x4F,
	0x3D, 0x3D, 0x8D, 0xDB, 0xCE, 0x99, 0xE1, 0x57,
	0x92, 0x59, 0xCD, 0xFD, 0xB8, 0xAE, 0x74, 0x4F,
	0xC5, 0xFC, 0x76, 0xBC, 0x83, 0xC5, 0x47, 0x30,
	0x61, 0xCE, 0x7C, 0xC9, 0x66, 0xFF, 0x15, 0xF9,
	0xBB, 0xFD, 0x91, 0x5E, 0xC7, 0x01, 0xAA, 0xD3,
	0x5B, 0x9E, 0x8D, 0xA0, 0xA5, 0x72, 0x3A, 0xD4,
	0x1A, 0xF0, 0xBF, 0x46, 0x00, 0x58, 0x2B, 0xE5,
	0xF4, 0x88, 0xFD, 0x58, 0x4E, 0x49, 0xDB, 0xCD,
	0x20, 0xB4, 0x9D, 0xE4, 0x91, 0x07, 0x36, 0x6B,
	0x33, 0x6C, 0x38, 0x0D, 0x45, 0x1D, 0x0F, 0x7C,
	0x88, 0xB3, 0x1C, 0x7C, 0x5B, 0x2D, 0x8E, 0xF6,
	0xF3, 0xC9, 0x23, 0xC0, 0x43, 0xF0, 0xA5, 0x5B,
	0x18, 0x8D, 0x8E, 0xBB, 0x55, 0x8C, 0xB8, 0x5D,
	0x38, 0xD3, 0x34, 0xFD, 0x7C, 0x17, 0x57, 0x43,
	0xA3, 0x1D, 0x18, 0x6C, 0xDE, 0x33, 0x21, 0x2C,
	0xB5, 0x2A, 0xFF, 0x3C, 0xE1, 0xB1, 0x29, 0x40,
	0x18, 0x11, 0x8D, 0x7C, 0x84, 0xA7, 0x0A, 0x72,
	0xD6, 0x86, 0xC4, 0x03, 0x19, 0xC8, 0x07, 0x29,
	0x7A, 0xCA, 0x95, 0x0C, 0xD9, 0x96, 0x9F, 0xAB,
	0xD0, 0x0A, 0x50, 0x9B, 0x02, 0x46, 0xD3, 0x08,
	0x3D, 0x66, 0xA4, 0x5D, 0x41, 0x9F, 0x9C, 0x7C,
	0xBD, 0x89, 0x4B, 0x22, 0x19, 0x26, 0xBA, 0xAB,
	0xA2, 0x5E, 0xC3, 0x55, 0xEB, 0x3D, 0xD6, 0x17
};
static byte const ElG4096g[] = {
	0x02
};

static struct {
	unsigned	bits;
	byte const *p;
	unsigned	psize;
	byte const *g;
	unsigned	gsize;
} ElGtab[] = {
#if 0
	{1024, ElG1024p, sizeof(ElG1024p),
		ElG1024g, sizeof(ElG1024g)},
#endif
	{1536, ElG1536p, sizeof(ElG1536p),
		ElG1536g, sizeof(ElG1536g)},
	{2048, ElG2048p, sizeof(ElG2048p),
		ElG2048g, sizeof(ElG2048g)},
	{3072, ElG3072p, sizeof(ElG3072p),
		ElG3072g, sizeof(ElG3072g)},
	{4096, ElG4096p, sizeof(ElG4096p),
		ElG4096g, sizeof(ElG4096g)}
};


/* Return >0 if we find one, and fill in pointers, else return 0 */
int
pgpElGfixed (unsigned bits, byte const **p, size_t *plen,
	byte const **g, size_t *glen)
{
	unsigned i;

	for (i=0; i<sizeof(ElGtab)/sizeof(ElGtab[0]); ++i) {
		if (ElGtab[i].bits > bits)
			return 0;
		if (ElGtab[i].bits == bits) {
			if (p)
				*p = ElGtab[i].p;
			if (plen)
				*plen = ElGtab[i].psize;
			if (g)
				*g = ElGtab[i].g;
			if (glen)
				*glen = ElGtab[i].gsize;
			return 1;
		}
	}
	return 0;
}





#else /* Obsolete code */

#include "keys.h"

static unsigned char const n[512/8] = {
	0xDB, 0x24, 0x36, 0x7B, 0x91, 0x82, 0x8E, 0x4F,
	0x17, 0xD4, 0xF7, 0x32, 0xD8, 0x0A, 0x59, 0x7C,
	0x96, 0xD8, 0xCB, 0xF7, 0x29, 0xCA, 0xB1, 0xA5,
	0xE2, 0x09, 0x27, 0xD1, 0xA4, 0xCD, 0xD7, 0x6D,
	0xFD, 0x6A, 0x1A, 0xD4, 0x80, 0x4D, 0x52, 0xD6,
	0x2B, 0x29, 0x99, 0xF9, 0x31, 0x3F, 0x65, 0x99,
	0x59, 0x19, 0xDF, 0x62, 0x58, 0x88, 0x31, 0xC9,
	0xB8, 0xE1, 0x21, 0x0E, 0x88, 0x0C, 0x13, 0x99
};

static unsigned char const d[512/8-1] = {
	      0xEB, 0xB7, 0x16, 0x08, 0x24, 0x0B, 0x5A,
	0xAF, 0x6D, 0x88, 0x91, 0x6C, 0x78, 0x83, 0x9A,
	0x56, 0xAF, 0x29, 0xBB, 0x1F, 0x5E, 0x6E, 0x75,
	0xF0, 0xD0, 0xB5, 0xF2, 0xE5, 0xCD, 0x3C, 0x10,
	0x96, 0xBC, 0xF0, 0x12, 0x26, 0x10, 0x7D, 0xDA,
	0x4F, 0x11, 0xF1, 0x4A, 0xAA, 0xAC, 0xBB, 0xD4,
	0x74, 0xAC, 0x9E, 0xBD, 0x7B, 0xF3, 0xBA, 0xBE,
	0x58, 0x15, 0xBA, 0xE2, 0x77, 0x75, 0xE0, 0xEF
};

static unsigned char const p[256/8] = {
	0xE9, 0x61, 0x7D, 0xF2, 0x29, 0x47, 0x80, 0x09,
	0x44, 0xDF, 0x0D, 0x7B, 0xD9, 0x01, 0x67, 0x47,
	0x34, 0xD1, 0x5F, 0xD1, 0xC2, 0xB7, 0xAA, 0x72,
	0xBE, 0x4E, 0xA6, 0xD2, 0xD4, 0x2C, 0xCE, 0x4B
};


static unsigned char const q[256/8] = {
	0xF0, 0x61, 0x6C, 0x02, 0xF3, 0xB0, 0xD1, 0xD7,
	0x63, 0x9C, 0x39, 0x12, 0xAB, 0xA7, 0x5E, 0xCD,
	0xAB, 0xCC, 0xEB, 0x67, 0x59, 0x38, 0xEA, 0x61,
	0x16, 0x5E, 0xBB, 0xB0, 0xA4, 0x48, 0x27, 0x2B
};

static unsigned char const u[256/8] = {
	0xD7, 0x58, 0xDF, 0x29, 0x01, 0x82, 0xAA, 0x2E,
	0x2A, 0x5A, 0xF9, 0x82, 0x8F, 0x4D, 0x94, 0x55,
	0x50, 0x7A, 0xAD, 0xB6, 0xC0, 0xA6, 0xB8, 0xC8,
	0x57, 0x47, 0x45, 0x2F, 0x55, 0x3D, 0xDC, 0x50
};

void
fixedKeyDestroy(struct PgpPubKey *pub, struct PgpSecKey *sec)
{
	if (pub) {
		bnEnd(&pub->n);
		bnEnd(&pub->e);
	}
	if (sec) {
		bnEnd(&sec->d);
		bnEnd(&sec->p);
		bnEnd(&sec->q);
		bnEnd(&sec->u);
	}
}

int
fixedKeyGet(struct PgpPubKey *pub, struct PgpSecKey *sec)
{
	if (pub) {
		bnBegin(&pub->n);
		bnBegin(&pub->e);
		 if (bnInsertBigBytes(&pub->n, n, 0, sizeof(n)) < 0 ||
		     bnSetQ(&pub->e, 17) < 0) {
			fixedKeyDestroy(pub, (struct PgpSecKey *)0);
			return -1;
		}
	}

	if (sec) {
		bnBegin(&sec->d);
		bnBegin(&sec->p);
		bnBegin(&sec->q);
		bnBegin(&sec->u);
		if (bnInsertBigBytes(&sec->d, d, 0, sizeof(d)) < 0 ||
	            bnInsertBigBytes(&sec->p, p, 0, sizeof(p)) < 0 ||
	            bnInsertBigBytes(&sec->q, q, 0, sizeof(q)) < 0 ||
	            bnInsertBigBytes(&sec->u, u, 0, sizeof(u)) < 0) {
			fixedKeyDestroy(pub, sec);
			return -1;
		}
	}

	return 0;
}

#endif	/* End of obsolete code */
