/* keyrings.c - Seahorse
 * Copyright (C) 2000, Anthony Mulcahy
 * Copyright 2000, 2001 Jos Carlos Garca Sogo <jsogo@arrakis.es>
 * 
 * Some of the code has been 'borrowed' from
 * GPAPA, The GNU Privacy Pipe Access
 * Copyright (C) 2000 G-N-U GmbH.
 * 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>

#include "gpg-interface.h"
#include "keyrings.h"

Key *
key_new (gchar * localID)
{
  Key *key = (Key *) g_malloc (sizeof (Key));
  memset (key, 0, sizeof (Key));
  key->LocalID = g_strdup (localID);
  return (key);
}


GDate *
extract_date (gchar * date)
{
  gchar **field = NULL;
  gchar delimiter[] = "-";
  GDate *newdate;

  if (date == NULL || strcmp (date, "") == 0)
    return (NULL);

  field = g_strsplit (date, delimiter, 3);

  newdate =
    g_date_new_dmy (atoi (field[2]), atoi (field[1]), atoi (field[0]));
  g_strfreev (field);
  return (newdate);
}				/* extract_date */


gchar *
extract_fingerprint (gchar *line)
{
	gchar **field = NULL;
	gchar delimiter[] = ":";
		
	field = g_strsplit (line, delimiter, 10);
	
	return field[9];
} /* extract_fingerprint */


Key *
extract_key (gchar * line)
{
  gchar **field;
  gchar delimiter[] = ":";
  Key *key;
  Uid *uid;

  field = g_strsplit (line, delimiter, 10);

  key = key_new (field[7]);
  key->KeyTrust = field[1][0];
  key->bits = atoi (field[2]);
  key->algorithm = atoi (field[3]);
  key->KeyID = g_strdup (field[4]);
  key->CreationDate = extract_date (field[5]);
  key->ExpirationDate = extract_date (field[6]);
  key->OwnerTrust = field[8][0];
  key->UserID = g_strdup (field[9]);
  key->secret = 0;

  /* Now, we add an UID to this key */

  uid = uid_new (key->UserID);
  uid->KeyTrust = key->KeyTrust;

  key->uids = g_list_append (key->uids, uid);
  g_strfreev (field);
  return (key);

}				/* extract_key */


Signature *
signature_new (gchar * keyID)
{
  Signature *sig = (Signature *) g_malloc (sizeof (Signature));
  memset (sig, 0, sizeof (Signature));
  sig->KeyID = g_strdup (keyID);
  return (sig);
}				/* signature_new */

Signature *
extract_sig (gchar * line)
{
  gchar **field;
  gchar delimiter[] = ":";
  Signature *sig;

  field = g_strsplit (line, delimiter, 10);

  sig = signature_new (field[4]);
  if (strcmp (field[1], "!") == 0)
    sig->validity = SIG_VALID;
  else if (strcmp (field[1], "-") == 0)
    sig->validity = SIG_INVALID;
  else
    sig->validity = SIG_UNKNOWN;
    
  sig->algorithm = atoi(field[3]);
  sig->CreationDate = extract_date (field[5]);
  sig->UserID = field[9][0] ? g_strdup (field[9]) : NULL;
  g_strfreev (field);
  return (sig);

}				/* extract_sig */

Uid *
uid_new (gchar * userID)
{
  Uid *uid = (Uid *) g_malloc (sizeof (Uid));
  memset (uid, 0, sizeof (Uid));
  uid->UserID = g_strdup (userID);
  return (uid);
}				/* uid_new */


Uid *
extract_uid (gchar * line)
{
  gchar **field;
  gchar delimiter[] = ":";
  Uid *uid;

  field = g_strsplit (line, delimiter, 10);

  uid = uid_new (field[9]);
  uid->KeyTrust = field[1][0];
  g_strfreev (field);
  return (uid);
}				/* extract_uid */



SubKey *
subkey_new (gchar * localID)
{
  SubKey *sub = (SubKey *) g_malloc (sizeof (SubKey));
  memset (sub, 0, sizeof (SubKey));  
  sub->LocalID = g_strdup (localID);
  return (sub);
}				/* sub_new */


SubKey *
extract_subkey (gchar * line)
{
  const gchar delimiter[] = ":";
  gchar **field;
  SubKey *subkey;

  field = g_strsplit (line, delimiter, 9);

  subkey = subkey_new (field[7]);
  subkey->bits = atoi (field[2]);
  subkey->algorithm = atoi (field[3]);
  subkey->KeyID = g_strdup (field[4]);
  subkey->CreationDate = extract_date (field[5]);
  subkey->ExpirationDate = extract_date (field[6]);

  g_strfreev (field);
  return (subkey);
}


GList *
refresh_keyring (GList * PubKeyring)
{
	
  GList * keyring;

  if (PubKeyring != NULL)
  	keyring_release (PubKeyring, NULL);
  keyring = get_public_keyring ();
  get_secret_keyring (keyring);
  return (keyring);
}				/* refresh_keyring */


GList *
get_public_keyring (void)
{
  GList *PubKeyring = NULL;
  GList *sig_list;
  gchar *KeyList;
  gchar *diagnostics;
  gchar *line;
  gint uid_cnt = 0;
  gint sub_cnt = 0;
  gint key_cnt = 0;
  const gchar delimiter[] = "\n";
  Key *key = NULL;

  gpg_list_pubkeys (&KeyList, &diagnostics);
  g_free (diagnostics);

  line = strtok (KeyList, delimiter);
  while (line != NULL)
    {
      if (strncmp (line, "pub", 3) == 0)
	{
	  key = extract_key (line);
	  PubKeyring = g_list_append (PubKeyring, key);
	  uid_cnt = 0;
	  sub_cnt = 0;
	  key_cnt++;
	}
      else if (strncmp (line, "sig", 3) == 0 && sub_cnt == 0)
	{
	  Signature *sig = extract_sig (line);
	  key = g_list_nth_data (PubKeyring, key_cnt - 1);
	  sig_list = get_uid_siglist (key, uid_cnt);
	  sig_list = g_list_append (sig_list, sig);
	  set_uid_siglist (key, uid_cnt, sig_list);
	}
      else if (strncmp (line, "uid", 3) == 0)
	{
	  Uid *uid = extract_uid (line);
	  key = g_list_nth_data (PubKeyring, key_cnt - 1);
	  key->uids = g_list_append (key->uids, uid);
	  uid_cnt++;
	}
      else if (strncmp (line, "sub", 3) == 0)
	{
	  SubKey *sub = extract_subkey (line);
	  key = g_list_nth_data (PubKeyring, key_cnt - 1);
	  key->subs = g_list_append (key->subs, sub);
	  sub_cnt++;
	}
	else if (strncmp (line, "fpr", 3) == 0)
	{
		key->fpr = g_strdup (extract_fingerprint (line));
	}
      line = strtok (NULL, delimiter);
    }

  
  return (PubKeyring);
}				/* get_public_keyring */


GList *
get_uid_siglist (Key * key, gint uid)
{
  Uid *uid_data;

  uid_data = g_list_nth_data (key->uids, uid);
  if (uid_data == NULL)
    show_error ("get_uid_siglist: No such uid");

  return (uid_data->sigs);
}				/* get_uid_siglist */


void
set_uid_siglist (Key * key, gint uid, GList * siglist)
{
  Uid *uid_data;

  uid_data = g_list_nth_data (key->uids, uid);
  if (uid_data == NULL)
    show_error ("set_uid_siglist: No such uid");

  uid_data->sigs = siglist;

}				/* set_uid_siglist */



void
signature_release (Signature * sig, gpointer user_data)
{
  g_free (sig->KeyID);
  g_free (sig->UserID);
  g_date_free (sig->CreationDate);
  g_free (sig);
}				/*signature_release */

void
uid_release (Uid * uid, gpointer user_data)
{
  g_free (uid->UserID);
  g_list_foreach (uid->sigs, (GFunc) signature_release, NULL);
  g_free (uid);
}				/* uid_release */

void
subkey_release (SubKey * sub, gpointer user_data)
{
  g_free (sub->KeyID);
  g_free (sub->LocalID);
  g_date_free (sub->CreationDate);
  if (sub->ExpirationDate != NULL)
    g_date_free (sub->ExpirationDate);
  g_free (sub);
}				/* subkey_release */

void
key_release (Key * key, gpointer user_data)
{
  g_free (key->KeyID);
  g_free (key->LocalID);
  g_free (key->UserID);

  g_date_free (key->CreationDate);

  if (key->ExpirationDate != NULL)
    g_date_free (key->ExpirationDate);

  g_list_foreach (key->uids, (GFunc) uid_release, NULL);
  g_list_foreach (key->subs, (GFunc) subkey_release, NULL);
}				/* key_release */

void
keyring_release (GList * keyring, gpointer user_data)
{
  g_list_foreach (keyring, (GFunc) key_release, NULL);
}				/* keyring_release */

void
get_secret_keyring (GList * PubKeyring)
{
  /* This function doesn't store the secret keyring in memory,
   * but it sets to SECRET_KEY_YES the Key->secret field of
   * the keys found in the
   * public keyring */

  gchar *KeyList;
  gchar delimiter[] = "\n";
  gchar *diagnostics;
  gchar *line;
  gchar **field;
  Key *key;

  gpg_list_seckeys (&KeyList, &diagnostics);
  g_free (diagnostics);

  line = strtok (KeyList, delimiter);
  while (line != NULL)
    {
      if (strncmp (line, "sec", 3) == 0)
	{
	  field = g_strsplit (line, ":", 5);
	  key = keyring_get_key_with_keyid (PubKeyring, field[4]);
	  key->secret = 1;
	  g_strfreev (field);
	}

      line = strtok (NULL, delimiter);
    }
}				/* get_secret_keyring */


Key *
keyring_get_key_with_keyid (GList * PubKeyring, gchar * keyid)
{
  GList *keydata;
  keydata =
    g_list_find_custom (PubKeyring, keyid, (GCompareFunc) key_compare_keyids);
  if (keydata == NULL)
  	return NULL;
  	
  return (keydata->data);
} /* keyring_get_key_with_keyid */


gint
key_compare_keyids (const Key * data, const gchar * keyid)
{
  gint returnvalue;

  returnvalue = strcmp (data->KeyID, keyid);
  return (returnvalue);
} /* key_compare_keyids */


Key *
keyring_get_key_with_userid (GList * keyring, gchar * userid)
{
	GList *keydata;
	keydata =
 	  g_list_find_custom (keyring, userid, (GCompareFunc) key_compare_userids);
	if (keydata == NULL)
		return NULL;
	return (keydata->data);
} /* keyring_get_key_with_userid */


gint
key_compare_userids (const Key * data, const gchar * userid)
{
  gint returnvalue;

  returnvalue = strcmp (data->UserID, userid);
  return (returnvalue);
} /* key_compare_userids */



gint 
keyring_get_key_uid_position_with_userid (Key *key, gchar * userid)
{
	GList *uiddata;
	gint position;
	
	uiddata = g_list_find_custom (key->uids, userid, (GCompareFunc) uid_compare_userids);
	if (uiddata != NULL) {
		position = g_list_position(key->uids, uiddata);
		return (position + 1);
	}
	else return -1;
	
} /* keyring_get_key_uid_position_with_userid */


gint
uid_compare_userids (const Uid *data, const gchar * userid)
{
	gint returnvalue;
	
	returnvalue = strcmp (data->UserID, userid);
	return (returnvalue);
} /* uid_compare_userids */




gchar *
algorithm_to_string (gint num_alg)
{
	gchar *str_alg;
	
	switch (num_alg)
		{
			case 1:
				str_alg = "RSA";
				break;
			
			case 16:
				str_alg = _("ElGamal (encypt only)");
				break;
				
			case 17:
				str_alg = ("DSA");
				break;
				
			case 20:
				str_alg = ("ElGamal");
				break;
				
			default:
				str_alg = _("Unknown");
		}
	return (str_alg);
} /* algorithm_to_string */


gchar *
trust_to_string (gchar trust)
{
	gchar *str_trust;

	switch (trust) {
		case 'f':
			str_trust = _("Full");
			break;
		
		case 'u':
			str_trust = _("Ultimate");
			break;
			
		case 'o':
			str_trust = _("Unknown");
			break;
			
		case 'd':
			str_trust = _("Disabled Key");
			break;
		
		case 'r':
			str_trust = _("REVOKED Key");
			break;
			
		case 'e':
			str_trust = _("Expired Key");
			break;
			
		case 'q':
			str_trust = _("Undefined");
			break;
			
		case 'n':
			str_trust = _("Don't Trust");
			break;
			
		case 'm':
			str_trust = _("Marginal Trust");
			break;
			
		default:
			str_trust = _("ERROR!");
	}
	
	return (str_trust);
	
} /* trust_to_string */


gint
trust_to_int (gchar ctrust)
{
	gfloat trust;

	switch (ctrust) {
		case 'f':
			trust = 10;
			break;
		
		case 'u':
			trust = 10;
			break;
			
		case 'o':
			trust = 0;
			break;
			
		case 'd':
			trust = 0;
			break;
		
		case 'r':
			trust = 0;
			break;
			
		case 'e':
			trust = 0;
			break;
			
		case 'q':
			trust = 0;
			break;
			
		case 'n':
			trust = 0;
			break;
			
		case 'm':
			trust = 5;
			break;
			
		default:
			trust = 0;
	}
	
	return (trust);
	
} /* trust_to_float */
