/*
** Copyright (C) 2000 Thierry MICHEL <thierry@nekhem.com>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
** As a special exception to the GNU General Public License,
** permission is granted for additional uses of the text contained in
** its release of PoPy.
**  
** The exception is that, if you link the PoPy library with other
** files to produce an executable, this does not by itself cause the
** resulting executable to be covered by the GNU General Public License
**  
** This exception applies only to the code released under the name
** PoPy. If you copy code from other releases into a copy of
** PoPy, as the General Public License permits, the exception does
** not apply to the code that you add in this way. To avoid
** misleading anyone as to the status of such modified files, you
** must delete this exception notice from them.
**  
** If you write modifications of your own for PoPy, it is your
** choice whether to permit this exception to apply to your
** modifications. If you do not wish that, delete this exception notice.
**  
*/

#include "PoPymodule.h"

PyObject *PoPy_array_handler(const char *array,int flag);

/* pthread mutex declaration */
pthread_mutex_t global_mutex;

/****************************/
/* Postgres types converter */
/****************************/
static PyObject *
PoPy_converter(PGresult * result, int tuple, int field, PGconn * conx)
{
	int ftype;
	long true_val = 1;
	long false_val = 0;
	PyObject *res = NULL;

	ftype = PQftype(result, field);

	if(PQgetisnull(result, tuple, field)) {
		Py_INCREF(Py_None);
		return Py_None;
	}
	
	/*	ftype = (ftype>1000 && ftype < 1028?POPY_ARRAY:ftype); */
	switch(ftype) {
		/*	case POPY_ARRAY_BOOL:
		   res = PoPy_array_handler(PQgetvalue(result,tuple,field),0);
		break; */
	case POPY_INT2VECTOR:
	case POPY_TID:
	case POPY_OIDVECTOR:
	case POPY_POINTID:
	case POPY_LSEG:
	case POPY_PATH:
	case POPY_BOX:
	case POPY_POLYGON:
	case POPY_LINE:
	case POPY_CIRCLE:
		res = PoPy_array_handler(PQgetvalue(result,tuple,field),1);
		break;
	case POPY_BOOLEAN:
		if(!strcmp("t",PQgetvalue(result, tuple, field)))
			res = PyInt_FromLong(true_val);
		else
			res = PyInt_FromLong(false_val);
		break;

	case POPY_SHORT2:
	case POPY_SHORT4: 			
	case POPY_INTEGER:
	case POPY_XID:
	case POPY_CID:
	case POPY_NUMERIC:
		res = PyInt_FromLong(atoi
				     (PQgetvalue
				      (result, tuple, field)));
		break;

	case POPY_DATE:
	case POPY_TIME:
	case POPY_TIMESTAMP:
	case POPY_ABSTIME:
	case POPY_RELTIME:
	case POPY_TINTERVAL:
	case POPY_INTERVAL:
	case POPY_TIMETZ:
		res = PyString_FromString(PQgetvalue
					  (result, tuple, field));
		break;
#ifdef BLOB
	case POPY_BINARY:
	case POPY_ZPBIT:
	case POPY_VARBIT:
	  if(strcasecmp(PQfname(result,field),"oid")) {
	    Py_BEGIN_ALLOW_THREADS
		    pthread_mutex_lock(&global_mutex);
	    res = 
		    PoPy_binary_object_create(
					      atoi(PQgetvalue
						   (result, tuple, field)),
					      conx);
	    pthread_mutex_unlock(&global_mutex);
	    Py_END_ALLOW_THREADS
		    } else {
			    res = PyInt_FromLong(atoi
						 (PQgetvalue
						  (result, tuple, field)));
		    }
	  break;
#endif
	case POPY_FLOAT:
	case POPY_DOUBLE:
		res = PyFloat_FromDouble(atof
					 (PQgetvalue
					  (result, tuple, field)));
		break;
	default:
		res = PyString_FromString(PQgetvalue
					  (result, tuple, field));
		break;
	}
	return res;
}

#ifdef BLOB
int
PoPy_lobDetector(PGconn * conx, char *query, PyObject * dict)
{
	int i;
	int n_items, flag = 0, res;
	regex_t *regex, regex_buf;
	PyObject *list_keys, *key, *value, *list_values;
	PoPy_BinaryObject *bo;

	char pattern[MAX_BUFFER];
	char tmpl[] = "%(%s)s";
	int fd;

	regex = &regex_buf;

	list_keys = PyDict_Keys(dict);
	list_values = PyDict_Values(dict);
	n_items = PyList_Size(list_keys);

	DBG("query: <%s>\n", query);

	for(i = 0; i < n_items; i++) {
		key = PyList_GetItem(list_keys, i);
		value = PyList_GetItem(list_values, i);
		if(Binary_Check(value)) {
			bo = (PoPy_BinaryObject *) value;
			sprintf(pattern, tmpl,
				PyString_AS_STRING(key));
			regcomp(regex, pattern, REG_ICASE | REG_NOSUB);
			if(!(res = regexec(regex, query, 0, NULL,
					   REG_NOTBOL))) {
				if(bo->oid == 0) {
					Py_BEGIN_ALLOW_THREADS
						pthread_mutex_lock(&global_mutex);
						sem_wait(&mutex);
					bo->oid = lo_creat(conx,
							   INV_READ |
							   INV_WRITE);
					fd = lo_open(conx,
						     bo->oid,
						     INV_READ |
						     INV_WRITE);
					lo_write(conx, fd,
						 bo->buffer,
						 bo->size);
					lo_close(conx, fd);
					pthread_mutex_unlock(&global_mutex);
					Py_END_ALLOW_THREADS
				}
				flag++;
			}
			regfree(regex);
		}
	}
	
	Py_DECREF(list_keys);
	Py_DECREF(list_values);
	
	return flag;
}
#endif

/* Array handler */
PyObject *
PoPy_array_handler(const char *array,int flag)
{
	PyObject *var,*ret,*list;
	char *expr;

	var = PyDict_New();
	if(var == NULL)
		return NULL;

	PyDict_SetItemString(var, "string",PyImport_ImportModule("string"));
	PyDict_SetItemString(var, "list", PyString_FromString(array));
	if(flag == 0) {
		expr = "list = string.replace(string.replace(list,'t','1'),'f','0')";
		ret = PyRun_String(expr, Py_single_input, var, var);
		list = PyDict_GetItemString(var, "list");
	}

	expr = "exec 'list = %s' % string.replace(string.replace(list,'{','['),'}',']')";

	/*   execute the expression and get the result */
	ret = PyRun_String(expr, Py_single_input, var, var);

	if(ret == NULL) {
		Py_DECREF(var);
		return NULL;
	}
	Py_DECREF(ret);
	list = PyDict_GetItemString(var, "list");
	Py_INCREF(list);
	Py_DECREF(var);
	return list;
}

#ifdef BLOB
/* ************************************************ */
/*         BINARY OBJECT METHODS DECLARATION        */
/* ************************************************ */

PyObject *
PoPy_binary_object_repr(PoPy_BinaryObject * self)
{
	char tmp[MAX_BUFFER];

	sprintf(tmp, "%d", self->oid);
	return Py_BuildValue("s", tmp);
}

void
PoPy_binary_object_dealloc(PoPy_BinaryObject * self)
{
	PyMem_Free(self->buffer);
	PyMem_Free((char *) self);
}

PyObject *
PoPy_binary_object_size(PoPy_BinaryObject * self)
{
	return PyInt_FromLong(self->size);
}

static PyObject *
PoPy_binary_object_create(Oid oid, PGconn * conx)
{
	PoPy_BinaryObject *bobj;
	int size;
	int fd;

	if(!
       (bobj =
        PyObject_NEW(PoPy_BinaryObject, &PoPy_BinaryObject_Type)))
		return NULL;

	if((fd = lo_open(conx, oid, INV_READ)) == -1) {
		PyErr_SetString(PoPy_OperationalError,
				"cannot open Large Object");
		Py_DECREF(bobj);
		return NULL;
	}

	lo_lseek(conx, fd, SEEK_CUR, SEEK_END);
	bobj->size = lo_tell(conx, fd) - 1;
	lo_lseek(conx, fd, 0, SEEK_SET);
	bobj->buffer =(void *) PyMem_Malloc(bobj->size);
	if(bobj->buffer == NULL) {
		lo_close(conx, fd);
		Py_DECREF(bobj);
		return PyErr_NoMemory();
	}

	if((size = lo_read(conx, fd, bobj->buffer, bobj->size)) < 0) {
		PyErr_SetString(PoPy_OperationalError,
				"cannot read Large Object");
		lo_close(conx, fd);
		Py_DECREF(bobj);
		return NULL;
	}
	lo_close(conx, fd);
	bobj->ob_type = &PoPy_BinaryObject_Type;
	bobj->oid = oid;

	return(PyObject *) bobj;
}

static struct memberlist PoPy_binary_object_memberlist[] = {
	{"size", T_INT, 0, RO},
	{"buffer", T_OBJECT, 0, RO},
	{NULL}			/* Sentinel */
};

static PyObject *
PoPy_binary_object_getattr(PoPy_BinaryObject * self, char *att_name)
{
	if(!strcmp(att_name, "size"))
		return PyInt_FromLong(self->size);
	if(!strcmp(att_name, "buffer"))
		return PyString_FromStringAndSize(self->buffer, self->size);
	return PyMember_Get((char *) self, PoPy_binary_object_memberlist,
                        att_name);
}

static PyObject *
PoPy_binary_object_NEW(PyObject * self, PyObject * args)
{
	PoPy_BinaryObject *bobj;
	char *data = NULL;
	int size;

	if(!PyArg_ParseTuple(args, "s#", &data, &size))
		return NULL;

	if(!
       (bobj =
        PyObject_NEW(PoPy_BinaryObject, &PoPy_BinaryObject_Type)))
		return NULL;

	bobj->buffer = (void *) PyMem_Malloc(size);
	if(bobj->buffer == NULL) {
		Py_DECREF(bobj);
		return PyErr_NoMemory();
	}

	memcpy(bobj->buffer, data, size);

	bobj->ob_type = &PoPy_BinaryObject_Type;
	bobj->oid = 0;
	bobj->size = size;
	return(PyObject *) bobj;
}
#endif

/* ************************************************ */
/*         CURSOR OBJECT METHODS DECLARATION        */
/* ************************************************ */

static PyObject *
PoPy_cursor_object_close(PoPy_CursorObject * self)
{
	if(self->opened == POPY_FALSE) {
		PyErr_SetString(PoPy_OperationalError, "Cursor closed");
		return NULL;
	}

	if(self->result != NULL) {
		PQclear(self->result);
		self->result = NULL;
	}

	self->opened = POPY_FALSE;
	self->executed = POPY_FALSE;
	self->current = 0;
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
PoPy_cursor_object_nextset(PoPy_CursorObject * self)
{
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
PoPy_cursor_object_setinputsizes(PoPy_CursorObject * self, PyObject * args)
{
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
PoPy_cursor_object_setoutputsize(PoPy_CursorObject * self, PyObject * args)
{
	Py_INCREF(Py_None);
	return Py_None;
}


static PyObject *
PoPy_cursor_object_description(PoPy_CursorObject * self)
{
	int i;

	Py_XDECREF(self->description);
	if(PQresultStatus(self->result) == PGRES_TUPLES_OK) {
		if(!(self->description = PyList_New(self->nfields)))
			return NULL;

		for(i = 0; i < self->nfields; i++) {
			PyObject *val_ftype;
			PyObject *fd = PyList_New(7);
			int j, len, fsize, mlength=0;

			PyList_SET_ITEM(fd,
					0,
					PyString_FromString(PQfname
							    (self->result, 
							     i)));
			
			switch(PQftype(self->result, i)) {
			case POPY_BOOLEAN:
			case POPY_SHORT2:
			case POPY_SHORT4: 			
			case POPY_INTEGER:
			case POPY_XID:
			case POPY_CID:
			case POPY_NUMERIC:
			case POPY_FLOAT:
			case POPY_DOUBLE:
				val_ftype = PyString_FromString("NUMBER");
				break;
			case POPY_ROWID:
				val_ftype = PyString_FromString("ROWID");
				break;
			case POPY_DATE:
			case POPY_TIME:
			case POPY_TIMESTAMP:
			case POPY_ABSTIME:
			case POPY_RELTIME:
			case POPY_TINTERVAL:
			case POPY_INTERVAL:
			case POPY_TIMETZ:
				val_ftype =
					PyString_FromString
					("DATETIME"); break;
#ifdef BLOB
			case POPY_BINARY:
			case POPY_ZPBIT:
			case POPY_VARBIT:
				val_ftype = PyString_FromString("BINARY");
				break;
#endif
			case POPY_TEXT:
				val_ftype = PyString_FromString("BINARY");
				break;
			case POPY_NULL:
			case POPY_CHAR:
			case POPY_VARCHAR:

			case POPY_BYTE:
			case POPY_NAME:
			case POPY_UNKNOWN:
			case POPY_INET:
			case POPY_CIDR:
			case POPY_BPCHAR:
			case POPY_CASH:
				val_ftype =
					PyString_FromString
					("STRING"); break;
			default:
				val_ftype = PyString_FromString("MISSING");
				break;
			}

			PyList_SET_ITEM(fd,1,val_ftype);
			for(j=0; j<self->rowcount; j++) {
				len = PQgetlength(self->result, j, i);
				mlength = ( mlength > len ? mlength : len);
			}
			PyList_SET_ITEM(fd,2,PyInt_FromLong(mlength));
			fsize = PQfsize(self->result, i);
			if (fsize == -1 && (PQfmod(self->result,i) != -1)) 
			  PyList_SET_ITEM(fd,
					  3,
					  PyInt_FromLong(PQfmod(self->result,i)-4));
			else PyList_SET_ITEM(fd,
					     3,
					     PyInt_FromLong(fsize));

			Py_INCREF(Py_None);
			PyList_SET_ITEM(fd,4,Py_None);
			Py_INCREF(Py_None);
			PyList_SET_ITEM(fd,5,Py_None);
			Py_INCREF(Py_None);
			PyList_SET_ITEM(fd,6,Py_None);

			PyList_SET_ITEM(self->description,i,fd);
		}
		DBG("End of description creation");
	}
	else {
		Py_INCREF(Py_None);
		self->description = Py_None;
	}
	Py_INCREF(Py_None);
	return Py_None;
}


static PyObject *
PoPy_cursor_object_fetchone(PoPy_CursorObject * self)
{
	int i;
	PyObject *row;


	if((*self->conn) == NULL) {
		/* ProgrammingError: FIXME */
		PyErr_SetString(PoPy_ProgrammingError,
				"Connection closed");
		return NULL;
	}

	if(self->opened == POPY_FALSE) {
		PyErr_SetString(PoPy_InternalError, "Cursor closed");
		return NULL;
	}

	if(self->executed == POPY_FALSE) {
		PyErr_SetString(PoPy_OperationalError,
				"select query not executed");
		return NULL;
	}
	

    /* 	If the rowcount==0 return an empty list !!! */
    /* 	  Not DB-API 2.0 compliant */
	if(self->rowcount == 0)
        return (PyObject *) PyTuple_New(0);


	if(self->current > (self->rowcount-1)) {
		PyErr_SetString(PoPy_DatabaseError,
				"No more results to fetch\n");
		return NULL;
	}


	if((row = (PyObject *) PyTuple_New(self->nfields)) == NULL)
		return NULL;
	
	for(i = 0; i < self->nfields; i++) {
		PyObject *value;
		value =
			PoPy_converter(self->result, self->current, 
				       i, (*self->conn));
		/*       value = PyString_FromString(PQgetvalue(res,0,i)); */
		PyTuple_SET_ITEM(row, i, value);
	}
	self->current++;
	return row;
}

static PyObject *
PoPy_cursor_object_fetchall(PoPy_CursorObject * self)
{
	PyObject *r, *v;
	int i, t;

	if((*self->conn) == NULL) {
		/* ProgrammingError: FIXME */
		PyErr_SetString(PoPy_ProgrammingError,
				"Connection closed");
		return NULL;
	}

	if(self->opened == POPY_FALSE) {
		PyErr_SetString(PoPy_InternalError, "Cursor closed");
		return NULL;
	}

	if(self->executed == POPY_FALSE) {
		PyErr_SetString(PoPy_OperationalError,
				"select query not executed");
		Py_INCREF(Py_None);
		return Py_None;
	}

	if(self->rowcount == 0)
		return(PyObject *) PyTuple_New(0);


	if(self->current > (self->rowcount-1)) {
		PyErr_SetString(PoPy_DatabaseError,
				"No more results to fetch\n");
		return NULL;
	}


	if(!(r = PyList_New(0)))
		return NULL;

	for(t = 0; t < self->rowcount; t++) {
		if((v = (PyObject *) PyTuple_New(self->nfields)) == NULL) {
			Py_DECREF(r);
			return NULL;
		}

		for(i = 0; i < self->nfields; i++) {
			PyObject *value;
			value =
				PoPy_converter(self->result, t, i,
					       (*self->conn));
			PyTuple_SET_ITEM(v, i, value);
		}
		PyList_Append(r, v);
		Py_DECREF(v);
	}
	self->current = self->rowcount;
	return r;
}

static PyObject *
PoPy_cursor_object_fetchmany(PoPy_CursorObject * self, PyObject * args)
{
	PyObject *r, *v;
	int i;
	unsigned int n_tuples, t;

	if((*self->conn) == NULL) {
		/* ProgrammingError: FIXME */
		PyErr_SetString(PoPy_ProgrammingError,
				"Connection closed");
		return NULL;
	}
	
	if(self->opened == POPY_FALSE) {
		PyErr_SetString(PoPy_InternalError, "Cursor closed");
		return NULL;
	}

	if(self->executed == POPY_FALSE) {
		PyErr_SetString(PoPy_OperationalError,
				"select query not executed");
		return NULL;
	}

	n_tuples = self->arraysize;
	
	if(!PyArg_ParseTuple(args, "|i", &n_tuples))
		return NULL;

	if(self->rowcount == 0)
        return (PyObject *) PyTuple_New(0);

	if(self->current > (self->rowcount - 1)) {
		PyErr_SetString(PoPy_DatabaseError,
				"No more results to fetch\n");
		return NULL;
	}

	if(!(r = PyList_New(0)))
		return NULL;
	
	n_tuples += self->current;

	for(t = self->current; t < n_tuples; t++) {
		if(self->current > (self->rowcount - 1))
			return r;
		if((v = (PyObject *) PyTuple_New(self->nfields)) == NULL) {
			Py_DECREF(r);
			return NULL;
		}

		for(i = 0; i < self->nfields; i++) {
			PyObject *value;
			value =
				PoPy_converter(self->result, t, i,
					       (*self->conn));
			PyTuple_SET_ITEM(v, i, value);
		}
		PyList_Append(r, v);
		Py_DECREF(v);
		self->current++;
	}
	return r;
}

static PyObject *
PoPy_cursor_object_callproc(PoPy_CursorObject * self, PyObject * args)
{
	char *buffer;
	PyObject *string = NULL, *dict = NULL, *query = NULL;
	PyObject *var, *ret;
	char *expr;

	if((*self->conn) == NULL) {
		/* ProgrammingError: FIXME */
		PyErr_SetString(PoPy_ProgrammingError,
				"Connection closed");
		return NULL;
	}

	if(self->opened == POPY_FALSE) {
		PyErr_SetString(PoPy_InternalError, "Cursor closed");
		return NULL;
	}

	/*   Get objects in the argument  */
	if(!PyArg_ParseTuple(args, "S|O!", &string, &PyTuple_Type, &dict))
		return NULL;

	/*   Set globals variables to eval the expression */
	var = PyDict_New();
	if(var == NULL)
		return NULL;

	self->current = 0;

	PyDict_SetItemString(var, "__builtins__", PyEval_GetBuiltins());
	PyDict_SetItemString(var, "procedure", string);

	if(dict) {
		PyDict_SetItemString(var, "parameters", dict);
		
		if(PyTuple_Size(dict) > 1)
			expr =
				"procedure = 'select %s%s' % (procedure,parameters)";
		else
			expr =
				"procedure = \"select %s('%s')\" % (procedure,parameters[0])";
	}
	else {
		PyDict_SetItemString(var, "procedure", string);
		expr = "procedure = 'select %s()' % (procedure)";
	}

	/*   execute the expression and get the result */
	ret = PyRun_String(expr, Py_single_input, var, var);
	if(ret == NULL) {
		Py_DECREF(var);
		return NULL;
	}
	Py_DECREF(ret);

	query = PyDict_GetItemString(var, "procedure");
	
	Py_INCREF(query);
	Py_DECREF(var);

	/* Cancel last results */
	if(self->result != NULL) {
		PQclear(self->result);
		self->result = NULL;
	}

	buffer = (char *) PyMem_Malloc(PyString_GET_SIZE(query)+1);
	sprintf(buffer, "%s",PyString_AS_STRING(query));

	Py_DECREF(query);

	Py_BEGIN_ALLOW_THREADS 
		pthread_mutex_lock(&global_mutex);
		self->result = PQexec((*self->conn), buffer);
		pthread_mutex_unlock(&global_mutex);
	Py_END_ALLOW_THREADS 
		if(PQresultStatus(self->result) != PGRES_TUPLES_OK) {
			SET_POPY_EXCEPTION((*self->conn), self->result,
					   PoPy_OperationalError);
			PQclear(self->result);

			Py_BEGIN_ALLOW_THREADS
				pthread_mutex_lock(&global_mutex);
			self->result = 
				PQexec((*self->conn), "END");
			PQclear(self->result);
			self->result = 
				PQexec((*self->conn), "BEGIN");
			pthread_mutex_unlock(&global_mutex);
			Py_END_ALLOW_THREADS 
			  PQclear(self->result);
			self->result = NULL;
			return NULL;
		}

	self->rowcount = PQntuples(self->result);
	self->nfields = PQnfields(self->result);

	PyMem_Free(buffer);
	self->executed = POPY_TRUE;
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
PoPy_cursor_object_execute(PoPy_CursorObject * self, PyObject * args)
{
	PyObject *string = NULL, *dict = NULL, *query = NULL;
	PyObject *var = NULL, *ret = NULL;
	char *tcount;

	if((*self->conn) == NULL) {
		/* ProgrammingError: FIXME */
		PyErr_SetString(PoPy_ProgrammingError,
				"Connection closed");
		return NULL;
	}

	if(self->opened == POPY_FALSE) {
		PyErr_SetString(PoPy_InternalError, "Cursor closed");
		return NULL;
	}

	/*   Get objects in the argument  */
	if(!PyArg_ParseTuple(args, "S|O!", &string, &PyDict_Type, &dict))
		return NULL;

	/* Cancel last results */
	if(self->result != NULL) {
		PQclear(self->result);
		self->result = NULL;
	}

	if(self->description != Py_None) {
		Py_XDECREF(self->description);
		Py_INCREF(Py_None);
		self->description = Py_None;
	}

	if(dict) {
#ifdef BLOB
		PoPy_lobDetector((*self->conn),
				 PyString_AS_STRING(string),
				 dict);
#endif
		/*   Set globals variables to eval the expression */
		var = PyDict_New();
		PyDict_SetItemString(var, "__builtins__",
				     PyEval_GetBuiltins());
		PyDict_SetItemString(var, "format", string);
		PyDict_SetItemString(var, "dict", dict);
		
		/*   execute the expression and get the result */
		
		ret =
			PyRun_String("format = format % dict",
				     Py_single_input, var, var);
		if(ret == NULL) {
			Py_DECREF(var);
			return NULL;
		}
		Py_DECREF(ret);
		
		query = PyDict_GetItemString(var, "format");
		Py_INCREF(query);
		Py_DECREF(var);
	}

	Py_BEGIN_ALLOW_THREADS
		pthread_mutex_lock(&global_mutex);
		self->result = 
		PQexec((*self->conn), 
		       PyString_AS_STRING(
					  query ?query :string));
		pthread_mutex_unlock(&global_mutex);
	Py_END_ALLOW_THREADS 

		if(PQresultStatus(self->result) ==
		   PGRES_TUPLES_OK) {
			self->rowcount = PQntuples(self->result); 
			
			self->nfields = PQnfields(self->result); 
			
			ret = PoPy_cursor_object_description(self);
			if(ret == NULL) {
				PQclear(self->result);
				return NULL;
			}
			self->current = 0;
			Py_DECREF(ret);
		}
		else if(PQresultStatus(self->result) != PGRES_COMMAND_OK) {
			SET_POPY_EXCEPTION((*self->conn), self->result,
					   PoPy_ProgrammingError);
			PQclear(self->result);
			Py_BEGIN_ALLOW_THREADS
				pthread_mutex_lock(&global_mutex);
			self->result = 
				PQexec((*self->conn), "END");
			PQclear(self->result);
			self->result = 
				PQexec((*self->conn), "BEGIN");
			pthread_mutex_unlock(&global_mutex);
			Py_END_ALLOW_THREADS 
			  PQclear(self->result);
			self->result = NULL;
			return NULL;
		}
		else  {
			tcount = PQcmdTuples(self->result);
			
			if(tcount == NULL || *tcount == '\0')
				self->rowcount = -1;
			else
				self->rowcount = atoi(tcount);
		}

	self->executed = POPY_TRUE;
	
	if(query) {
		Py_DECREF(query);
	}
	
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *
PoPy_cursor_object_executemany(PoPy_CursorObject * self, PyObject * args)
{
	PyObject *list, *item, *query, *ret;
	int i, n;

	if((*self->conn) == NULL) {
		/* ProgrammingError: FIXME */
		PyErr_SetString(PoPy_ProgrammingError,
				"Connection closed");
		return NULL;
	}

	if(self->opened == POPY_FALSE) {
		PyErr_SetString(PoPy_InternalError, "Cursor closed");
		return NULL;
	}

	if(!(PyArg_ParseTuple(args, "SO", &query, &list))) {
		PyErr_SetString(PoPy_ProgrammingError,
				"Bad argument type");
		return NULL;
	}

	if(!PySequence_Check(list)) {
		PyErr_SetString(PoPy_ProgrammingError,
				"Bad argument type");
		return NULL;
	}

	n = PySequence_Length(list);

	for(i = 0; i < n; i++) {
		item = PyList_GetItem(list, i);

		if(!PyMapping_Check(item)) {
			PyErr_SetString(PoPy_ProgrammingError,
					"Item is not a mapping");
			return NULL;
		}

		ret =
			PyObject_CallMethod((PyObject *) self, "execute",
					    "SO", query, item);
		if(ret == NULL)
			return NULL;
		Py_DECREF(ret);
	}
	
	Py_INCREF(Py_None);
	return Py_None;
}


static PoPy_CursorObject *
PoPy_cursor_object_NEW(PGconn ** conn)
{
	PoPy_CursorObject *c;

	if(!
	   (c = PyObject_NEW(PoPy_CursorObject, &PoPy_CursorObject_Type)))
		return NULL;

	c->conn = conn;
	c->rowcount = -1;
	c->current = 0;
	c->nfields = 0;
	Py_INCREF(Py_None);
	c->description = Py_None;
	c->result = NULL;
	c->opened = POPY_TRUE;
	c->arraysize = 5;
	c->executed = POPY_FALSE;
	Py_INCREF(Py_None);
	c->datestyle = Py_None;
	c->ob_type = &PoPy_CursorObject_Type;
	return c;
}

/* Freeing the _ConnectionObject and close the connexion */
/* if it's opened                                       */
static void
PoPy_cursor_object_dealloc(PoPy_CursorObject * self)
{
	if(self->result != NULL)
		PQclear(self->result);

	Py_XDECREF(self->description);
	PyMem_DEL(self);
}


/* String representation of the _ConnectionObject */
PyObject *
PoPy_cursor_object_repr(PoPy_CursorObject * self)
{
	char buffer[MAX_BUFFER];

	sprintf(buffer, "<_CursorObject at %lx>\n", (long) self);
	return PyString_FromString(buffer);
}

static PyMethodDef PoPy_cursor_object_methods[] = {
	
	{"callproc", (PyCFunction) PoPy_cursor_object_callproc,
	 METH_VARARGS, NULL},
	{"execute", (PyCFunction) PoPy_cursor_object_execute, METH_VARARGS,
	 NULL},
	{"executemany", (PyCFunction) PoPy_cursor_object_executemany,
	 METH_VARARGS, NULL},
	{"fetchone", (PyCFunction) PoPy_cursor_object_fetchone, 0, NULL},
	{"fetchall", (PyCFunction) PoPy_cursor_object_fetchall, 0, NULL},
	
	{"fetchmany", (PyCFunction) PoPy_cursor_object_fetchmany,
	 METH_VARARGS, NULL},
	{"nextset", (PyCFunction) PoPy_cursor_object_nextset,0, NULL},
	{"setinputsizes", (PyCFunction) PoPy_cursor_object_setinputsizes,
	 METH_VARARGS, NULL},
	{"setoutputsize", (PyCFunction) PoPy_cursor_object_setoutputsize,
	 METH_VARARGS, NULL},
	{"close", (PyCFunction) PoPy_cursor_object_close, 0, NULL},
	{NULL, NULL}
};

static struct memberlist PoPy_cursor_object_memberlist[] = {
	
	{"description", T_OBJECT,
	 offsetof(PoPy_CursorObject, description), 0},
	{"arraysize", T_INT, 0, 0},
	{"rowcount", T_INT, 0, RO},
	{"datestyle", T_OBJECT, 0, RO},
	{NULL}			/* Sentinel */
};

static int
PoPy_cursor_object_setattr(PoPy_CursorObject * self, char *att_name,
                           PyObject * v)
{
	if(v == NULL) {
		PyErr_SetString(PyExc_AttributeError,
				"can't delete cursor attributes");
		return -1;
	}
	if(!strcmp(att_name, "arraysize")) {
		if(!PyInt_Check(v)) {
			PyErr_SetString(PyExc_TypeError,
					"arraysize must be integer.");
			return -1;
		}
		self->arraysize = PyInt_AsLong(v);
		return 0;
	}

	/* unknown attribute */
	PyErr_SetString(PyExc_TypeError, "not a writable attribute.");
	return -1;
}


static PyObject *
PoPy_cursor_object_getattr(PoPy_CursorObject * self, char *att_name)
{
	PyObject *res;
	
	res =
		Py_FindMethod(PoPy_cursor_object_methods, (PyObject *) self,
			      att_name);
	if(res != NULL)
		return res;
	PyErr_Clear();
	if(strcmp(att_name, "rowcount") == 0)
		return PyInt_FromLong((long)(self->rowcount));
	if(strcmp(att_name, "arraysize") == 0)
		return PyInt_FromLong((long)(self->arraysize));
	if(strcmp(att_name, "datestyle") == 0)
		return self->datestyle;
	return PyMember_Get((char *) self, PoPy_cursor_object_memberlist,
			    att_name);
}

/* ************************************************ */
/*      CONNECTION  OBJECT METHODS DECLARATION      */
/* ************************************************ */

PoPy_ConnectionObject *
PoPy_connection_object_NEW(PyObject * args)
{
	char *conx_str;
	PGresult *res;
	PoPy_ConnectionObject *c;

	if(!PyArg_ParseTuple(args, "s", &conx_str))
		return NULL;

	if(!
	   (c =
	    PyObject_NEW(PoPy_ConnectionObject,
			 &PoPy_ConnectionObject_Type)))
		return NULL;

	c->conn = PQconnectdb (conx_str);

	if (PQstatus (c->conn) == CONNECTION_BAD)
	  {
		  PyErr_SetString (PoPy_OperationalError,
				   PQerrorMessage (c->conn));
		  Py_DECREF (c);
		  return NULL;
	  }

	c->committed = POPY_FALSE;
	c->opened = POPY_TRUE;
	c->ob_type = &PoPy_ConnectionObject_Type;

	Py_BEGIN_ALLOW_THREADS
		pthread_mutex_lock(&global_mutex);
		res = PQexec(c->conn, "BEGIN");
		pthread_mutex_unlock(&global_mutex);
	Py_END_ALLOW_THREADS
		if(PQresultStatus(res) != PGRES_COMMAND_OK) {
			SET_POPY_EXCEPTION(c->conn, res, 
					   PoPy_OperationalError);
			PQclear(res);
			Py_DECREF(c);
			return NULL;
		}
	PQclear(res);
	
	Py_BEGIN_ALLOW_THREADS
		pthread_mutex_lock(&global_mutex);
		res =
		PQexec(c->conn,
		       "SET TRANSACTION ISOLATION LEVEL SERIALIZABLE");
		pthread_mutex_unlock(&global_mutex);
	Py_END_ALLOW_THREADS 
		
		if(PQresultStatus(res) != PGRES_COMMAND_OK) {
			SET_POPY_EXCEPTION(c->conn, res, PoPy_OperationalError);
			PQclear(res);
			Py_DECREF(c);
			return NULL;
		}
	PQclear(res);


	return c;
}

/* String representation of the ConnectionObject */
PyObject *
PoPy_connection_object_repr(PoPy_ConnectionObject * self)
{
	char buffer[MAX_BUFFER];

	if(self->opened == POPY_TRUE && PQstatus(self->conn) == CONNECTION_OK) {
		char *h = PQhost(self->conn);
		sprintf(buffer,
			"%s connection to '%.256s' at %lx\n",
			self->opened == POPY_TRUE ? "open" : "closed",
			h == NULL ? "localhost" : h,
			(long) self);
		return PyString_FromString(buffer);
	}
	sprintf(buffer, "<Instance at %lx state: closed>",(long) self);
	return PyString_FromString(buffer);
}

/* Freeing the ConnectionObject and close the connexion */
/* if it's opened                                       */
static void
PoPy_connection_object_dealloc(PoPy_ConnectionObject * self)
{
	if(self->opened == POPY_TRUE) {
		PQfinish(self->conn);
	}
	pthread_mutex_destroy(&global_mutex);
	PyMem_DEL(self);
}

PyObject *
PoPy_connection_object_rollback(PoPy_ConnectionObject * self)
{
	PGresult *result = NULL;

	if(self->opened == POPY_TRUE
	   && self->committed == POPY_FALSE) {
		Py_BEGIN_ALLOW_THREADS
			pthread_mutex_lock(&global_mutex);
			result = PQexec(self->conn, "rollback");
			pthread_mutex_unlock(&global_mutex);
		Py_END_ALLOW_THREADS
			if(PQresultStatus(result) != PGRES_COMMAND_OK) {
				SET_POPY_EXCEPTION(self->conn, result,
						   PoPy_OperationalError);
				PQclear(result);
				return NULL;
			}
		PQclear(result);
		
		Py_BEGIN_ALLOW_THREADS
			pthread_mutex_lock(&global_mutex);
			result = PQexec(self->conn, "BEGIN");
			pthread_mutex_unlock(&global_mutex);
		Py_END_ALLOW_THREADS
			if(PQresultStatus(result) != PGRES_COMMAND_OK) {
				SET_POPY_EXCEPTION(self->conn, result,
						   PoPy_OperationalError);
				PQclear(result);
				return NULL;
			}
		PQclear(result);
		
		Py_BEGIN_ALLOW_THREADS
			pthread_mutex_lock(&global_mutex);
			result =
			PQexec(self->conn,
			       "SET TRANSACTION ISOLATION LEVEL SERIALIZABLE");
			pthread_mutex_unlock(&global_mutex);
		Py_END_ALLOW_THREADS 
			
			if(PQresultStatus(result) !=
			   PGRES_COMMAND_OK) {
				SET_POPY_EXCEPTION(self->conn, result,
						   PoPy_OperationalError);
				PQclear(result);
				return NULL;
			}
		self->committed = POPY_FALSE;
		PQclear(result);
	}
	else {
		PyErr_SetString(PoPy_NotSupportedError,
				"Cannot perform rollback");
		return NULL;
	}
	Py_INCREF(Py_None);
	return Py_None;
}


PyObject *
PoPy_connection_object_commit(PoPy_ConnectionObject * self)
{
	PGresult *result = NULL;

	if(self->opened == POPY_TRUE) {
		Py_BEGIN_ALLOW_THREADS
			pthread_mutex_lock(&global_mutex);
			result = PQexec(self->conn, "commit");
			pthread_mutex_unlock(&global_mutex);
		Py_END_ALLOW_THREADS
			if(PQresultStatus(result) != PGRES_COMMAND_OK) {
				SET_POPY_EXCEPTION(self->conn, result,
						   PoPy_OperationalError);
				PQclear(result);
				return NULL;
			}
		PQclear(result);
		
		Py_BEGIN_ALLOW_THREADS
			pthread_mutex_lock(&global_mutex);
			result = PQexec(self->conn, "BEGIN");
			pthread_mutex_unlock(&global_mutex);
		Py_END_ALLOW_THREADS
			if(PQresultStatus(result) != PGRES_COMMAND_OK) {
				SET_POPY_EXCEPTION(self->conn, result,
						   PoPy_OperationalError);
				PQclear(result);
				return NULL;
			}
		PQclear(result);
		
		Py_BEGIN_ALLOW_THREADS
			pthread_mutex_lock(&global_mutex);
			result =
			PQexec(self->conn,
			       "SET TRANSACTION ISOLATION LEVEL SERIALIZABLE");
			pthread_mutex_unlock(&global_mutex);
		Py_END_ALLOW_THREADS
	       
			if(PQresultStatus(result) !=
			   PGRES_COMMAND_OK) {
				SET_POPY_EXCEPTION(self->conn, result,
						   PoPy_OperationalError);
				PQclear(result);
				return NULL;
			}
		self->committed = POPY_FALSE;
		PQclear(result);
	}

	Py_INCREF(Py_None);
	return Py_None;
}

PyObject *
PoPy_connection_object_close(PoPy_ConnectionObject * self)
{
	if(self->opened == POPY_TRUE) {
		PoPy_connection_object_rollback(self);
		PQfinish(self->conn);
		self->conn = NULL;
		self->opened = POPY_FALSE;
		self->committed = POPY_FALSE;
	}

	Py_INCREF(Py_None);
	return Py_None;
}

PyObject *
PoPy_connection_object_cursor(PoPy_ConnectionObject * self)
{
	PyObject *cursor = NULL;
	if(self->opened == POPY_FALSE)
		PyErr_SetString(PoPy_OperationalError,
                        "Connection is closed");
	else
		cursor =
			(PyObject *) PoPy_cursor_object_NEW(&self->conn);

	return cursor;
}


static PyMethodDef PoPy_connection_object_methods[] = {
	
	{"close",(PyCFunction) PoPy_connection_object_close, 0, 0},
	{"commit",(PyCFunction) PoPy_connection_object_commit, 0, 0},
	{"cursor",(PyCFunction) PoPy_connection_object_cursor, 0, 0},
	{"rollback",(PyCFunction) PoPy_connection_object_rollback, 0, 0},
	{NULL, NULL}
};

static struct memberlist PoPy_connection_object_memberlist[] = {
	{"opened", T_INT, 0, RO},
	{"closed", T_INT, 0, RO},
	{NULL}			/* Sentinel */
};

static int
PoPy_connection_object_setattr(PoPy_ConnectionObject * c, char *att_name,
                               PyObject * v)
{
	if(v == NULL) {
		PyErr_SetString(PyExc_AttributeError,
				"can't delete connection attributes");
		return -1;
	}
	return PyMember_Set((char *) c, PoPy_connection_object_memberlist,
			    att_name, v);
}


static PyObject *
PoPy_connection_object_getattr(PoPy_ConnectionObject * self, char *att_name)
{
	PyObject *result;
	result =
		Py_FindMethod(PoPy_connection_object_methods,
			      (PyObject *) self, att_name);
	if(result != NULL)
		return result;
	PyErr_Clear();
	if(strcmp(att_name, "opened") == 0)
		return PyInt_FromLong((long)(self->opened));
	if(strcmp(att_name, "closed") == 0)
		return PyInt_FromLong((long) !(self->opened));
	return PyMember_Get((char *) self, PoPy_connection_object_memberlist,
			    att_name);
}

/* To create new exceptions */
static PyObject *
PoPy_exception_NEW(PyObject * dict, char *name, PyObject * base)
{
	PyObject *e;
	char longname[MAX_BUFFER];

	sprintf(longname, "PoPy.%s", name);
	if((e = PyErr_NewException(longname, base, NULL)) == NULL)
		return NULL;
	if(PyDict_SetItemString(dict, name, e))
		return NULL;
	return e;
}


/****************************************/
/*   DATE / TIME  METHODS DECLARATION   */
/****************************************/

PyObject *
Date(PyObject * self, PyObject * args)
{
	PyObject *date;
	int year, month, day;

	if(!PyArg_ParseTuple(args, "iii", &year, &month, &day)) {
		PyErr_SetString(PoPy_ProgrammingError, "Bad Arguments");
		return NULL;
	}

	if(mxDateTime_ImportModuleAndAPI() == -1) {
		PyErr_SetString(PoPy_InterfaceError,
				"Cannot load module DateTime");
		return NULL;
	}

	date =
		mxDateTime.DateTime_FromDateAndTime(year, month, day, 0,
						    0, 0.0);
	
	return date;
}

PyObject *
Time(PyObject * self, PyObject * args)
{
	PyObject *time;
	int hours, minutes;
	double seconds;

	if(!PyArg_ParseTuple(args, "iid", &hours, &minutes, &seconds)) {
		PyErr_SetString(PoPy_ProgrammingError, "Bad Arguments");
		return NULL;
	}

	if(mxDateTime_ImportModuleAndAPI() == -1) {
		PyErr_SetString(PoPy_InterfaceError,
				"Cannot load module DateTime");
		return NULL;
	}
	time =
		mxDateTime.DateTimeDelta_FromTime(hours, minutes,
						  seconds);
	
	return time;
}

PyObject *
Timestamp(PyObject * self, PyObject * args)
{
	PyObject *timestamp;
	int year, month, day, hours, minutes;
	double seconds;

	if(!PyArg_ParseTuple
	   (args, "iiiiid", &year, &month, &day, &hours, &minutes,
        &seconds)) {
		PyErr_SetString(PoPy_ProgrammingError, "Bad Arguments");
		return NULL;
	}

	if(mxDateTime_ImportModuleAndAPI() == -1) {
		PyErr_SetString(PoPy_ProgrammingError,
				"Cannot load module DateTime");
		return NULL;
	}
	timestamp =
		mxDateTime.DateTime_FromDateAndTime(year, month, day,
						    hours, minutes,
						    seconds);
	
	return timestamp;
}

PyObject *
DateFromTicks(PyObject * self, PyObject * args)
{
	PyObject *date;
	double ticks;
	struct tm *tm;
	time_t tticks;

	if(!PyArg_ParseTuple(args, "d", &ticks)) {
		PyErr_SetString(PoPy_ProgrammingError, "Bad Arguments");
		return NULL;
	}

	if(mxDateTime_ImportModuleAndAPI() == -1) {
		PyErr_SetString(PoPy_ProgrammingError,
				"Cannot load module DateTime");
		return NULL;
	}

	tticks =(time_t) ticks;
	tm = localtime(&tticks);

	tm->tm_sec = 0;
	tm->tm_min = 0;
	tm->tm_hour = 0;

	date = mxDateTime.DateTime_FromTmStruct(tm);
	return date;
}

PyObject *
TimeFromTicks(PyObject * self, PyObject * args)
{
	PyObject *date;
	double ticks;
	struct tm *tm;
	time_t tticks;

	if(!PyArg_ParseTuple(args, "d", &ticks)) {
		PyErr_SetString(PoPy_ProgrammingError, "Bad Arguments");
		return NULL;
	}

	if(mxDateTime_ImportModuleAndAPI() == -1) {
		PyErr_SetString(PoPy_ProgrammingError,
				"Cannot load module DateTime");
		return NULL;
	}

	tticks =(time_t) ticks;
	tm = localtime(&tticks);

	date =
		mxDateTime.DateTimeDelta_FromTime(tm->tm_hour, tm->tm_min,
						  tm->tm_sec);
	return date;
}

PyObject *
TimestampFromTicks(PyObject * self, PyObject * args)
{
	PyObject *date;
	double ticks;

	if(!PyArg_ParseTuple(args, "d", &ticks)) {
		PyErr_SetString(PoPy_ProgrammingError, "Bad Arguments");
		return NULL;
	}

	if(mxDateTime_ImportModuleAndAPI() == -1) {
		PyErr_SetString(PoPy_ProgrammingError,
				"Cannot load module DateTime");
		return NULL;
	}

	date = mxDateTime.DateTime_FromTicks(ticks);
	return date;
}



/* ***************************************************** */
/*            -- MODULE FUNCTIONS DEFINITION --          */
/* ***************************************************** */

/* Implementation of the main function of the module */
/* It allows to connect to a PostgreSQL database */
static PyObject *
PoPy_connect(PyObject * self, PyObject * args)
{
	return(PyObject *) PoPy_connection_object_NEW(args);
}


static PyMethodDef PoPy_methods[] = {
	{"connect", PoPy_connect, METH_VARARGS, NULL},
#ifdef BLOB
	{"Binary",(PyCFunction) PoPy_binary_object_NEW, METH_VARARGS, NULL},
#endif
	{"Date", Date, METH_VARARGS, NULL},
	{"Time", Time, METH_VARARGS, NULL},
	{"Timestamp", Timestamp, METH_VARARGS, NULL},
	{"DateFromTicks", DateFromTicks, METH_VARARGS, NULL},
	{"TimeFromTicks", TimeFromTicks, METH_VARARGS, NULL},
	{"TimestampFromTicks", TimestampFromTicks, METH_VARARGS, NULL},
	{NULL, NULL}
};


/***********************************/
/*    OBJECTS TYPES DEFINITIONS    */
/***********************************/

PyTypeObject PoPy_ConnectionObject_Type = {
	PyObject_HEAD_INIT(&PyType_Type) 0,
	"ConnectionObject",
	sizeof(PoPy_ConnectionObject),
	0,
	(destructor) PoPy_connection_object_dealloc,	/* tp_dealloc */
	0,			/*tp_print */
	(getattrfunc) PoPy_connection_object_getattr,	/* tp_getattr */
	(setattrfunc) PoPy_connection_object_setattr,	/* tp_setattr */
	0,			/*tp_compare */
	(reprfunc) PoPy_connection_object_repr,	/* tp_repr */
	/*  , */
};


PyTypeObject PoPy_CursorObject_Type = {
	PyObject_HEAD_INIT(&PyType_Type) 0,
	"_CursorObject",
	sizeof(PoPy_CursorObject),
	0,
	(destructor) PoPy_cursor_object_dealloc,	/* tp_dealloc */
	0,			/*tp_print */
	(getattrfunc) PoPy_cursor_object_getattr,	/* tp_getattr */
	(setattrfunc) PoPy_cursor_object_setattr,	/* tp_setattr */
	0,			/*tp_compare */
	(reprfunc) PoPy_cursor_object_repr,	/* tp_repr */
	0,			/* tp_as_number */
	0,			/* tp_as_sequence */
	0,			/* tp_as_mapping */
	0,			/* tp_hash */
	0,			/* tp_call */
	0,			/* tp_str */
	0,			/* tp_getattro */
	0,			/* tp_setattro */
	0,			/* tp_as_buffer */
	0,			/* tp_flags */
	0,      	/* tp_doc */
};

#ifdef BLOB
PyTypeObject PoPy_BinaryObject_Type = {
	PyObject_HEAD_INIT(&PyType_Type) 0,
	"binary",
	sizeof(PoPy_BinaryObject),
	0,
	(destructor) PoPy_binary_object_dealloc,	/*tp_dealloc */
	0,			/*tp_print */
	(getattrfunc) PoPy_binary_object_getattr,	/*tp_getattr */
	0,			/*tp_setattr */
	0,			/*tp_compare */
	(reprfunc) PoPy_binary_object_repr,	/*tp_repr */
	0,			/*tp_as_number */
	0,			/*tp_as_sequence */
	0,			/*tp_as_mapping */
	0,			/*tp_hash */
	0,			/*tp_call */
	(reprfunc) PoPy_binary_object_repr,	/*tp_str */
	0,			/*tp_getattro */
	0,			/*tp_setattro */
	0,			/*tp_as_buffer */
	Py_TPFLAGS_DEFAULT,	/*tp_flags */
	0,			/*tp_doc */
};
#endif



/***************************************/
/*    PoPy MODULE INITIALIZATION    */
/***************************************/

void
initPoPy(void)
{
	PyObject *dict, *module;

	module = Py_InitModule3("PoPy", PoPy_methods, "");
	dict = PyModule_GetDict(module);

	pthread_mutex_init(&global_mutex,NULL);

	PyDict_SetItemString(dict, "__version__",
                         PyString_FromString(VERSION));
	PyDict_SetItemString(dict, "STRING", PyString_FromString("STRING"));
	PyDict_SetItemString(dict, "BINARY", PyString_FromString("BINARY"));
	PyDict_SetItemString(dict, "NUMBER", PyString_FromString("NUMBER"));
	PyDict_SetItemString(dict, "DATETIME", 
			     PyString_FromString("DATETIME"));
	PyDict_SetItemString(dict, "ROWID", PyString_FromString("ROWID"));
	PyDict_SetItemString(dict, "MISSING", PyString_FromString("MISSING"));

	PyDict_SetItemString(dict, "apilevel", PyString_FromString("2.0"));
	PyDict_SetItemString(dict, "threadsafety", PyInt_FromLong(2L));
	PyDict_SetItemString(dict, "paramstyle",
                         PyString_FromString("pyformat"));
	PyDict_SetItemString(dict, "INV_READ", PyLong_FromLong(INV_READ));
	PyDict_SetItemString(dict, "INV_WRITE", PyLong_FromLong(INV_WRITE));
    /*   PyDict_SetItemString(dict,"INV_ARCHIVE",PyLong_FromLong(INV_ARCHIVE)); */

	if(!
	   (PoPy_Error =
        PoPy_exception_NEW(dict, "Error", PyExc_StandardError)))
		goto error;
	if(!
	   (PoPy_Warning =
        PoPy_exception_NEW(dict, "Warning",
                           PyExc_StandardError))) goto error;
	if(!
	   (PoPy_InterfaceError =
        PoPy_exception_NEW(dict, "InterfaceError",
                           PoPy_Error))) goto error;
	if(!
	   (PoPy_DatabaseError =
        PoPy_exception_NEW(dict, "DatabaseError",
                           PoPy_Error))) goto error;
	if(!
	   (PoPy_DataError =
        PoPy_exception_NEW(dict, "DataError",
                           PoPy_DatabaseError))) goto error;
	if(!
	   (PoPy_OperationalError =
        PoPy_exception_NEW(dict, "OperationalError",
                           PoPy_DatabaseError))) goto error;
	if(!
	   (PoPy_IntegrityError =
        PoPy_exception_NEW(dict, "IntegrityError",
                           PoPy_DatabaseError))) goto error;
	if(!
	   (PoPy_InternalError =
        PoPy_exception_NEW(dict, "InternalError",
                           PoPy_DatabaseError))) goto error;
	if(!
	   (PoPy_ProgrammingError =
        PoPy_exception_NEW(dict, "ProgrammingError",
                           PoPy_DatabaseError))) goto error;
	if(!
	   (PoPy_NotSupportedError =
        PoPy_exception_NEW(dict, "NotSupportedError",
                           PoPy_DatabaseError))) goto error;
 error:
	if(PyErr_Occurred())
		PyErr_SetString(PyExc_ImportError, "PoPy: init failed");
	return;
}
