/*
 * main.h
 *
 * PWLib application header file for Voxilla
 *
 * A H.323 "net telephone" application.
 *
 * Copyright (c) 1993-2001 Equivalence Pty. Ltd.
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Portable Windows Library.
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Portions of this code were written with the assisance of funding from
 * Vovida Networks, Inc. http://www.vovida.com.
 *
 * Portions are Copyright (C) 1993 Free Software Foundation, Inc.
 * All Rights Reserved.
 *
 * Contributor(s): ______________________________________.
 *
 * $Log: main.h,v $
 * Revision 1.25  2001/10/15 14:10:01  rogerh
 * Add new definitions
 *
 * Revision 1.24  2001/09/29 07:11:48  rogerh
 * Delete ogmChanel in destructor - Patrick Koorevaar <pkoorevaar@hotmail.com>
 * Only call conn.ClearCall() once after we pass the record time limit.
 * Reported by Patrick.
 *
 * Revision 1.23  2001/09/28 00:13:48  robertj
 * Changed BYTE* to PBYTEArray so get automatic memory management.
 *
 * Revision 1.22  2001/09/24 22:39:42  craigs
 * Added commands to play and record data files, esp G.723.1
 *
 * Revision 1.21  2001/07/23 09:17:36  rogerh
 * Add the LPC10 codec, a low quality and low bitrate codec.
 *
 * Revision 1.20  2001/07/17 14:33:01  rogerh
 * Support writing of .wav audio files from PCM codecs (eg G711 and GSM).
 * .wav files are written out by default now. If you still want to record
 * to raw audio files with a .sw extension, use the --recordraw option.
 *
 * Revision 1.19  2001/07/01 07:38:58  rogerh
 * Add Microsoft GSM codec. Also allocate memory for frameBuffer dynamically
 * as different codecs can have different frame sizes.
 *
 * Revision 1.18  2001/06/29 11:13:15  rogerh
 * Add AudioDelay class which removes the jitter in recorded files.
 *
 * Revision 1.17  2001/06/29 06:34:57  rogerh
 * Add mutex locks in PCM_Recordfile. This solves the race condition where
 * Close() was called while Write() was still running.
 *
 * Revision 1.16  2000/08/29 12:32:08  craigs
 * Fixed problems with recording messages
 *
 * Revision 1.15  2000/08/28 16:42:59  craigs
 * Finally fixed problems with G.723.1. All codecs now working
 *
 * Revision 1.14  2000/08/28 07:49:51  craigs
 * Added extra stuff
 *
 * Revision 1.13  2000/08/28 00:38:37  craigs
 * Added support for setting listening port number
 *
 * Revision 1.12  2000/08/27 23:42:24  craigs
 * Fixed problem with playback of messages
 * Fixed problem with recording messages
 *
 * Revision 1.11  2000/05/25 12:06:17  robertj
 * Added PConfigArgs class so can save program arguments to config files.
 *
 * Revision 1.10  2000/05/10 05:14:25  robertj
 * Changed capabilities so has a function to get name of codec, instead of relying on PrintOn.
 *
 * Revision 1.9  2000/05/09 11:22:15  craigs
 * Fixed problems caused by new jitter buffer code
 * and fixed OGM problems
 *
 * Revision 1.8  2000/04/25 23:34:22  craigs
 * Added lots of new code, including outgoing and incoming
 * multiplexors, and the start of an IVR system
 *
 * Revision 1.7  2000/01/13 04:03:45  robertj
 * Added video transmission
 *
 * Revision 1.6  1999/11/11 00:27:49  robertj
 * Changed OnAnswerCall() call back function  to allow for asyncronous response.
 *
 * Revision 1.5  1999/11/06 13:27:49  craigs
 * Added extra output and changed for new library changes
 *
 * Revision 1.4  1999/10/24 12:50:37  craigs
 * Fixed G723.1 capability, and added ability for discrete OGMs
 *
 * Revision 1.3  1999/10/24 03:08:49  craigs
 * Fixed problem with recording zero length messages, and added autodelete of files
 *
 * Revision 1.2  1999/10/22 09:56:41  craigs
 * Fixed compile warning
 *
 * Revision 1.1  1999/10/11 00:15:18  craigs
 * Initial version
 *
 *
 */

#ifndef _Voxilla_MAIN_H
#define _Voxilla_MAIN_H

#include <h323.h>
#include <h323pdu.h>
#include <h245.h>
#include <gsmcodec.h>
#include <lid.h>

class AudioDelay : public PObject
{ 
  PCLASSINFO(AudioDelay, PObject);
  
  public:
    AudioDelay();
    BOOL Delay(int time);
    void Restart();
    int  GetError();
 
  protected:
    PTime  previousTime;
    BOOL   firstTime;
    int    error;
};

class OpenAm : public PProcess
{
  PCLASSINFO(OpenAm, PProcess)

  public:
    OpenAm();
    ~OpenAm();

    void Main();
    void RecordFile(PArgList & args);
    void PlayFile(PArgList & args);

  protected:
    long GetCodec(const PString & codecname);
    OpalLineInterfaceDevice * GetDevice(const PString & device);
};

class MyH323EndPoint : public H323EndPoint
{
  PCLASSINFO(MyH323EndPoint, H323EndPoint);

  public:
    MyH323EndPoint(unsigned callLimit,
                   const PString & runCmd,
                   const PDirectory & dir,
                   int flags);

    // overrides from H323EndPoint
    virtual H323Connection * CreateConnection(unsigned callReference);
    BOOL OnIncomingCall(H323Connection &, const H323SignalPDU &, H323SignalPDU &);

    // new functions
    BOOL Initialise(PConfigArgs & args);

    PString    GetGSMOGM() const            { return gsmOgm; }
    void       SetGSMOGM(const PString & s) { gsmOgm = s; }

    PString    GetG711OGM() const            { return g711Ogm; }
    void       SetG711OGM(const PString & s) { g711Ogm = s; }

    PString    GetLPC10OGM() const            { return lpc10Ogm; }
    void       SetLPC10OGM(const PString & s) { lpc10Ogm = s; }

    PString    GetG7231OGM() const            { return g7231Ogm; }
    void       SetG7231OGM(const PString & s) { g7231Ogm = s; }

    unsigned   GetCallLimit() const { return callLimit; }
    PString    GetRunCmd() const    { return runCmd; }
    PDirectory GetDirectory() const { return dir; }

    void       SetRecordWav(const BOOL rec){ recordWav = rec; }
    BOOL       GetRecordWav() const        { return recordWav; }

    enum {
      DeleteAfterRecord = 0x01,
      NoRecordG7231     = 0x02
    };

    BOOL       GetDeleteAfterRecord() const   { return flags & DeleteAfterRecord; }
    BOOL       GetNoRecordG7231() const       { return flags & NoRecordG7231; }

  protected:
    unsigned callLimit;
    PString pcmOgm, g711Ogm, gsmOgm, lpc10Ogm, g7231Ogm, runCmd;
    PDirectory dir;
    int flags;
    BOOL recordWav;
};

class PCM_RecordFile;
class MyH323Connection;
PQUEUE(PStringQueue, PString);

class PCM_OGMChannel : public PIndirectChannel
{
  PCLASSINFO(PCM_OGMChannel, PIndirectChannel);

  public:
    enum {
      Min_Headroom = 90,
      Max_Headroom = 150
    };

    PCM_OGMChannel(MyH323Connection & conn);

    BOOL Read(void * buffer, PINDEX amount);
    void PlayFile(PFile * chan);

    BOOL Close();

    void QueueFile(const PString & cmd);
    void FlushQueue();

    void SetRecordTrigger();
    void SetHangupTrigger();

  protected:
    virtual BOOL ReadFrame(PINDEX amount);
    virtual void CreateSilenceFrame(PINDEX amount);
    virtual void Synchronise(PINDEX amount);
    virtual BOOL IsWAVFileValid(PWAVFile *chan);

    void FrameDelay(int delay);
    BOOL AdjustFrame(void * buffer, PINDEX amount);

    PStringQueue playQueue;

    MyH323Connection & conn;
    PMutex chanMutex;
    int silentCount;
    int totalData;
    BOOL recordTrigger, hangupTrigger;
    BOOL closed;

    BOOL firstRead;
    PTime lastReadTime;
    int headRoom;

    PBYTEArray frameBuffer;
    PINDEX frameLen, frameOffs;
};

class G7231_OGMChannel : public PCM_OGMChannel
{
  PCLASSINFO(G7231_OGMChannel, PCM_OGMChannel);
  public:
    G7231_OGMChannel(MyH323Connection & conn);

  protected:
    BOOL ReadFrame(PINDEX amount);
    void CreateSilenceFrame(PINDEX amount);
    void Synchronise(PINDEX amount);
    BOOL IsWAVFileValid(PWAVFile *chan);
};


class MyH323Connection : public H323Connection
{
  PCLASSINFO(MyH323Connection, H323Connection);

  public:
    MyH323Connection(MyH323EndPoint &, unsigned);
    ~MyH323Connection();

    // overrides from H323Connection
    BOOL OpenAudioChannel(BOOL, unsigned, H323AudioCodec & codec);
    AnswerCallResponse OnAnswerCall(const PString &, const H323SignalPDU &, H323SignalPDU &);
    BOOL OnStartLogicalChannel(H323Channel & channel);
    void OnUserInputString(const PString & value);

    // new functions
    void StartRecording();
    void Hangup();

    void SetE164Number(const PString & _num)
      { e164Number = _num; }

    PString GetE164Number() const
      { return e164Number; }

  protected:
    void OnUserInputChar(char ch);
    BOOL StartMenu(int menuNumber);
    BOOL ProcessMenuCmd(const PString & cmdStr);

    const MyH323EndPoint & ep;
    PString product;
    PTime   callStartTime;
    PTime   recordStartTime;
    PString basename;
    PFilePath recordFn;
    PString transmitCodecName, receiveCodecName;
    BOOL    recordTrigger;
    PMutex  connMutex;

    PCM_RecordFile * recordFile;
    PCM_OGMChannel * ogmChannel;

    PString digits, lastDigits;
    int currentMenu;
    PStringList menuNames;

    PString securityToken, e164Number;
};

class PCM_RecordFile : public PIndirectChannel
{
  PCLASSINFO(PCM_RecordFile, PIndirectChannel)

  public:
    PCM_RecordFile(MyH323Connection & conn, const PFilePath & fn, unsigned callLimit);
    ~PCM_RecordFile();

    BOOL Write(const void * buf, PINDEX len);
    BOOL Close();
    void StartRecording();

    virtual void DelayFrame(PINDEX len);
    virtual BOOL WriteFrame(const void * buf, PINDEX len);

    BOOL WasRecordStarted() const { return recordStarted; }

  protected:
    MyH323Connection & conn;
    PTime finishTime;
    PFilePath fn;
    unsigned callLimit;
    BOOL recordStarted;
    BOOL timeLimitExceeded;
    BOOL closed;
    BOOL isPCM;
    BOOL dataWritten;
    AudioDelay delay;
    PMutex pcmrecordMutex;
    PFile *fileclass; // will point to a PWAVFile or PFile class
};

class G7231_RecordFile : public PCM_RecordFile
{
  PCLASSINFO(G7231_RecordFile, PCM_RecordFile);

  public:
    G7231_RecordFile(MyH323Connection & conn, const PFilePath & fn, unsigned callLimit);
    void DelayFrame(PINDEX len);
    BOOL WriteFrame(const void * buf, PINDEX len);
};

class G7231_File_Codec : public H323AudioCodec
{
  PCLASSINFO(G7231_File_Codec, H323AudioCodec);

  public:
    G7231_File_Codec(Direction dir);

    unsigned GetBandwidth() const;
    static int GetFrameLen(int val);
      
    BOOL Read(BYTE * buffer, unsigned & length, RTP_DataFrame &);
    BOOL Write(const BYTE * buffer, unsigned length, const RTP_DataFrame & rtp, unsigned & frames);
};  

class G7231_File_Capability : public H323AudioCapability
{
  PCLASSINFO(G7231_File_Capability, H323AudioCapability)

  public:
    G7231_File_Capability();

    unsigned GetSubType() const { return H245_AudioCapability::e_g7231; }
    PString GetFormatName() const { return "G.723.1"; }

    H323Codec * CreateCodec(H323Codec::Direction direction) const;

    BOOL OnSendingPDU(H245_AudioCapability & cap, unsigned packetSize) const;
    BOOL OnReceivedPDU(const H245_AudioCapability & pdu, unsigned & packetSize);
    PObject * Clone() const;
};


#endif  // _Voxilla_MAIN_H


// End of File ///////////////////////////////////////////////////////////////
