// Copyright (c) 2000-2001 Brad Hughes <bhughes@trolltech.com>
//
// Use, modification and distribution is allowed without limitation,
// warranty, or liability of any kind.
//

#include "equalizer.h"
#include "eqpreset.h"
#include "icons.h"
#include "logscale.h"

#include <qdatastream.h>
#include <qdir.h>
#include <qfile.h>
#include <qgroupbox.h>
#include <qlistview.h>
#include <qmessagebox.h>
#include <qpainter.h>
#include <qsettings.h>
#include <qslider.h>

static Equalizer *INSTANCE = 0;


class EqPresetItem : public QListViewItem
{
public:
    EqPresetItem(Equalizer *, const EqPreset &);

    QString text(int) const;
    void setText(int, const QString &);

    void paintCell(QPainter *, const QColorGroup &, int, int, int);

    static int RTTI;
    int rtti() const;

    Equalizer *equalizer;
    EqPreset preset;
};

int EqPresetItem::RTTI = 141835;

EqPresetItem::EqPresetItem(Equalizer *eq, const EqPreset &pre)
    : QListViewItem(eq->presetView), equalizer(eq), preset(pre)
{
    setPixmap(0, equalizer->drawArea);

    if (! preset.isDefault())
	setRenameEnabled(0, true);
}

QString EqPresetItem::text(int c) const
{
    if (c != 0)
	return QString::null;
    return preset.name();
}

void EqPresetItem::setText(int c, const QString & t)
{
    if (c != 0)
	return;
    // TODO: ensure that we have a unique name...
    preset.setName(t);
    equalizer->setModified(true);
}

void EqPresetItem::paintCell(QPainter *p, const QColorGroup &cg, int a, int b, int c)
{

    QListViewItem::paintCell(p, cg, a, b, c);

    p->setPen(cg.foreground());

    double dx = double(equalizer->drawArea.width()) / 10.0,
	   dy = double(equalizer->drawArea.height()) / 40.0;
    int i, x1, y1, x2, y2, xmarg = listView()->itemMargin(), ymarg = xmarg;

    xmarg += (equalizer->drawArea.width() - int(dx * 10.0)) / 2;
    ymarg += (equalizer->drawArea.height() - int(dy * 40.0)) / 2;

    for (i = 0; i < 9; i++) {
	x1 = int((i * dx)) + xmarg;
	y1 = int((-preset.band(i) + 20) * dy) + ymarg;
	x2 = int(((i + 1) * dx)) + xmarg;
	y2 = int((-preset.band(i + 1) + 20) * dy) + ymarg;

	p->drawLine(x1, y1, x2, y2);
    }
}

int EqPresetItem::rtti() const
{
    return EqPresetItem::RTTI;
}


Equalizer::Equalizer(QWidget *parent)
    : EqualizerBase(parent, "equalizer"),
      autosave(false), modified(false), ignorechange(false)
{
    INSTANCE = this;

    levelsFrame->setFixedSize(levelsFrame->sizeHint());

    QFontMetrics fm(font());
    drawArea.resize(fm.height() * 2, fm.height() * 3 / 2);
    drawArea.fill(colorGroup().base());

    load();
}

Equalizer::~Equalizer()
{
}


void Equalizer::deinit()
{
    if (isModified()) {
	if (autosave) {
	    save();
	} else {
	    if (! QMessageBox::warning(0,
				       tr("Equalizer Modified"),
				       tr("Do you want to save your Equalizer changes?"),
				       tr("&Yes"),
				       tr("&No"),
				       QString::null, 0, 1))
		save();
	}
    }
}

Equalizer *Equalizer::instance()
{
    return INSTANCE;
}

void Equalizer::configChanged(QSettings &settings)
{
    autosave = settings.readBoolEntry("/MQ3/Equalizer/saveChangesOnExit", false);
    QIconSet iconset = IconLoader::load("equalizer");
    setIcon(iconset.pixmap(QIconSet::Large, QIconSet::Normal));
}

void Equalizer::load()
{
    QValueList<EqPreset> presets;

    QFile file(QDir::homeDirPath() + "/.mq3/equalizers");
    if (file.open(IO_ReadOnly)) {
	QString header("MQ3 Equalizer Presets"), s;

	// make sure we can read the header from the stream - this tells us we are
	// using the correct file format...
	QDataStream stream(&file);
	stream >> s;
	if (s == header) {
	    // read the presets
	    while (! stream.atEnd()) {
		EqPreset preset;
		stream >> preset;
		presets.append(preset);
	    }
	}
    }

    if (! presets.count()) {
	// nothing loaded, create a single default preset
	EqPreset preset("Default");
	preset.def = TRUE;
	presets.append(preset);
	setModified(true);
    } else
	setModified(false);

    // add the presets into the listview
    QValueList<EqPreset>::Iterator it = presets.begin();
    while (it != presets.end())
	(void) new EqPresetItem(this, (*it++));

    presetView->setSelected(presetView->firstChild(), true);
}

void Equalizer::save()
{
    if (! isModified())
	return;

    QFileInfo fileinfo(QDir::homeDirPath() + "/.mq3");
    if (! fileinfo.exists()) {
	QDir::home().mkdir(".mq3");
    }

    QFile file (QDir::homeDirPath() + "/.mq3/equalizers");
    if (! file.open(IO_WriteOnly)) {
	qWarning("MQ3: error writing equalizer presets");
	return;
    }

    QDataStream stream(&file);

    // write out header
    QString header("MQ3 Equalizer Presets");
    stream << header;

    // write out presets
    QListViewItem *item = presetView->firstChild();
    while (item) {
	if (item->rtti() == EqPresetItem::RTTI) {
	    EqPresetItem *epi = (EqPresetItem *) item;
	    stream << epi->preset;
	}
	item = item->nextSibling();
    }
    file.close();

    setModified(false);
}

void Equalizer::addPreset()
{
    EqPresetItem *epi = new EqPresetItem(this, EqPreset("New Preset"));
    presetView->setFocus();
    epi->startRename(0);

    setModified(true);
}

void Equalizer::copyPreset()
{
    QListViewItem *item = presetView->selectedItem();
    if (! item || item->rtti() != EqPresetItem::RTTI)
	return;

    EqPreset preset(((EqPresetItem *) item)->preset);
    preset.def = FALSE;
    preset.setName(preset.name() + tr(" (Copy)"));

    EqPresetItem *epi = new EqPresetItem(this, preset);
    presetView->setFocus();
    epi->startRename(0);

    setModified(true);
}

void Equalizer::removePreset()
{
    QListViewItem *item = presetView->selectedItem();
    if (! item || item->rtti() != EqPresetItem::RTTI)
	return;

    EqPresetItem *epi = (EqPresetItem *) item;
    // can't remove the default preset
    if (epi->preset.isDefault())
	return;

    delete epi;

    setModified(true);
}

void Equalizer::bandChanged(int value)
{
    if (ignorechange)
	return;
    QListViewItem *item = presetView->selectedItem();
    if (! item || item->rtti() != EqPresetItem::RTTI)
	return;

    EqPresetItem *epi = (EqPresetItem *) item;
    value = -value;

    if (sender() == preampSlider) {
	epi->preset.setPreamp(value);
    } else {
        int band;

	if (sender() == band60)
	    band = 0;
	else if (sender() == band170)
	    band = 1;
	else if (sender() == band310)
	    band = 2;
	else if (sender() == band600)
	    band = 3;
	else if (sender() == band1k)
	    band = 4;
	else if (sender() == band3k)
	    band = 5;
	else if (sender() == band6k)
	    band = 6;
	else if (sender() == band12k)
	    band = 7;
	else if (sender() == band14k)
	    band = 8;
	else if (sender() == band16k)
	    band = 9;
	else
	    return;

	epi->preset.setBand(band, value);
	epi->repaint();
    }

    setModified(true);

    emit presetChanged(epi->preset);
}

void Equalizer::presetHighlighted(QListViewItem *item)
{
    if (! item || item->rtti() != EqPresetItem::RTTI) {

	return;
    }

    ignorechange = true;

    EqPreset preset(((EqPresetItem *) item)->preset);
    band60->setValue(-preset.band(0));
    band170->setValue(-preset.band(1));
    band310->setValue(-preset.band(2));
    band600->setValue(-preset.band(3));
    band1k->setValue(-preset.band(4));
    band3k->setValue(-preset.band(5));
    band6k->setValue(-preset.band(6));
    band12k->setValue(-preset.band(7));
    band14k->setValue(-preset.band(8));
    band16k->setValue(-preset.band(9));

    preampSlider->setValue(-preset.preamp());

    ignorechange = false;
}

QStringList Equalizer::presets() const
{
    QStringList list;
    QListViewItem *item = presetView->firstChild();
    while (item) {
	if (item->rtti() == EqPresetItem::RTTI) {
	    EqPresetItem *epi = (EqPresetItem *) item;
	    list << epi->preset.name();
	}
	item = item->nextSibling();
    }
    return list;
}

EqPreset Equalizer::preset(const QString &name) const
{
    // find the right preset
    EqPreset def;
    QListViewItem *item = presetView->firstChild();
    while (item) {
	if (item->rtti() == EqPresetItem::RTTI) {
	    EqPresetItem *epi = (EqPresetItem *) item;
	    if (epi->preset.name() == name)
		return epi->preset;
	    if (epi->preset.name() == "Default")
		def = epi->preset;
	}
	item = item->nextSibling();
    }

    return def;
}

