/* TCPTransmitter.cc
 * TCP connection
 *
 * Copyright (C) 1999-2001 Dave Smith & Julian Missig
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contributor(s): Konrad Podloucky, Matthias Wimmer, Brandon Lees, 
 *  JP Sugarbroad, Julian Missig
 */

/*
 * This file contains code from GNet by David A. Helder
 * http://www.eecs.umich.edu/~dhelder/misc/gnet/
 */

#include <config.h>

#include "TCPTransmitter.hh"

#include <iostream>
#include <string>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <errno.h>
#include <netdb.h>
#include <sys/wait.h>
#include <unistd.h>
#include <fcntl.h>
#ifdef WITH_IPV6
// results in a redefinition of MIN and MAX ...
// would be better to include resolv.h before glib.h
#include <resolv.h>
#endif
#include <glib.h>

using namespace std;

#ifdef HAVE_GETHOSTBYNAME_R_GLIB_MUTEX
     GStaticMutex TCPTransmitter::_gethostbyname_mutex = G_STATIC_MUTEX_INIT;
#endif

TCPTransmitter::TCPTransmitter():
     _socketfd(-1), _IOChannel(NULL), _reconnect_timeoutid(0),
     _reconnect_delay(5), _reconnect_timeval(new GTimeVal),
     _state(Offline), _hostResolved(false), _use_ssl(false),
     _autoreconnect(false), _need_send(false),
     _resolver_watchid(0), _poll_eventid(0),
     _send_retry_eventid(0), _retries(0), _socket_flags(0),
     _resolver_pid(-1), _socket_watchid(0)
{
     _reconnect_timeval->tv_sec = 0;
     
#ifdef WITH_SSL
     _adapter = NULL;
#endif
     
#ifdef WITH_IPV6
     // get an AAAA record from gethostbyname if possible
     res_init();
     _res.options |= RES_USE_INET6;
#endif
}

TCPTransmitter::~TCPTransmitter()
{
     disconnect();
     delete _reconnect_timeval;
}

void TCPTransmitter::connect(const string& host, guint port, bool use_ssl, bool autoreconnect)
{
     // Set autoreconnect and use_ssl
     _autoreconnect = autoreconnect;
     _use_ssl = use_ssl;
     
#ifdef TRANSMITTER_DEBUG
     cout << "Connecting to " << host << ":" << port << endl;
#endif
     if (_state == Offline || _state == Reconnecting)
     {
#ifdef WITH_SSL
	  if (use_ssl && _adapter == NULL)
	  {
	       _adapter = new SSLAdapter();
	  }
	  else if (!use_ssl && _adapter != NULL)
	  {
	       delete _adapter;
	       _adapter = NULL;
	  }
#else
	  if (use_ssl)
	  {
	       cerr << "SSL support not compiled in, but attempt was made to use SSL." << endl;
	       cout << "SSL disabled." << endl;
	  }
#endif
	  if (_reconnect_timeoutid != 0) 
	  {
	       g_source_remove(_reconnect_timeoutid);
	       _reconnect_timeoutid = 0;
	  }
	  g_get_current_time(_reconnect_timeval);
	  
	  _state = Connecting;
	  if (!_hostResolved) 
	  {
#ifdef WITH_IPV6
	       _host_sockaddr.sin6_family = AF_INET6;
	       _host_sockaddr.sin6_port = g_htons(port);
#else
	       _host_sockaddr.sin_family = AF_INET;
	       _host_sockaddr.sin_port = g_htons(port);
#endif
	       // resolve host
	       _async_resolve(host.c_str());
	  }
	  else 
	  {
	       // we don't need to resolve the host again
	       on_host_resolved(NULL, G_IO_NVAL, this);
	  }
     }
     else if(_state == Connected) 
     {
	  // get current time
	  g_get_current_time(_reconnect_timeval);
	  evtConnected();
     }
}

void TCPTransmitter::listen(const string& host, guint port)
{
    int fd = socket(PF_INET, SOCK_STREAM, 0);

    struct sockaddr_in name;

	name.sin_family = AF_INET;
	name.sin_port = g_htons(port);
	name.sin_addr.s_addr = g_htonl(INADDR_ANY);

    socklen_t length = sizeof(name);
    if(bind(fd, (struct sockaddr*)&name, length) < 0)
    {
        handleError("Can not bind socket");
        return;
    }

    if (port == 0)
    {
        struct sockaddr_in sa;
        length = sizeof(sa);
        if(getsockname(fd, (struct sockaddr*)&sa, &length) < 0)
        {
            handleError("Can not retrieve local port number");
            close(fd);
            return;
        }
        _port = ntohs(sa.sin_port);
    }

    if(std::listen(fd, 10) < 0)
    {
        handleError("Could not listen with socket");
        close(fd);
        return;
    }

    _state = Listening;
    _socketfd = fd;
    
    _socket_watchid = g_io_add_watch(g_io_channel_unix_new(_socketfd),
            GIOCondition (G_IO_IN | G_IO_PRI | G_IO_ERR | G_IO_HUP | G_IO_NVAL),
            &TCPTransmitter::on_socket_accept, this);
}

void TCPTransmitter::disconnect()
{
#ifdef TRANSMITTER_DEBUG
     cout << "disconnect() called!" << endl;
#endif
     
    if (_state == Offline)
    {
        // Lame
        return;
    }
    if (_socket_watchid > 0)
    {
        g_source_remove(_socket_watchid);
        _socket_watchid = 0;
    }
	// remove the poll function
	if (_poll_eventid > 0) 
	{
	    g_source_remove(_poll_eventid);
	    _poll_eventid = 0;
	}

	if (_send_retry_eventid > 0)
    {
	    g_source_remove(_send_retry_eventid);
	    _send_retry_eventid = 0;
	}
    send_data_buf* data;
    while (!_sendBuffer.empty())
    {
        data = _sendBuffer.front();
        cout << "Deleteing " << data->sz << endl;
        delete data->data;
        delete data;
        _sendBuffer.pop_front();
    }
	       
#ifdef WITH_SSL
    if (_adapter) 
	{
	    _adapter->disconnect();
	    delete _adapter;
	    _adapter = NULL;
    }
#endif
	  
	// close socket
	if (_socketfd != -1) 
	{
	    close(_socketfd);
	}
	_socketfd = -1;
     
    if (_state != Reconnecting)
    {
        // close IOChannel
        if (_IOChannel != NULL) 
        {
            g_io_channel_close(_IOChannel);
        }
        _state = Offline;
        if (_reconnect_timeoutid) 
        {
            g_source_remove(_reconnect_timeoutid);
            _reconnect_timeoutid = 0;
            _reconnect_delay = 5;
        }
    	evtDisconnected();
    }
}

void TCPTransmitter::startPoll()
{
    // add poll function to event loop
    // poll every 15sec
    _poll_eventid = g_timeout_add(150000, &TCPTransmitter::_connection_poll, this);
#ifdef TRANSMITTER_DEBUG
    cout << "Poll function added to event loop" << endl;
#endif
}

// Transmitter methods

void TCPTransmitter::sendsz(const gchar* data, guint sz)
{
     send_data_buf* send_data = new send_data_buf;
     send_data->data = new char[sz];
     memcpy(send_data->data, data, sz);
     send_data->sz = sz;
     send_data->cur_pos = 0;

     _send(send_data);
}

void TCPTransmitter::send(const gchar* data)
{
     send_data_buf* send_data = new send_data_buf;
     send_data->data = g_strdup(data);
     send_data->sz = strlen(data);
     send_data->cur_pos = 0;

     _send(send_data);
}

void TCPTransmitter::needSend(bool notify)
{
    _need_send = notify;
    if (notify)
    {
        if (_socket_watchid)
            g_source_remove(_socket_watchid);
        _socket_watchid = g_io_add_watch(_IOChannel, GIOCondition (G_IO_IN | G_IO_PRI | G_IO_OUT | G_IO_ERR | G_IO_HUP | G_IO_NVAL), &TCPTransmitter::on_socket_event, this);
    } else {
        if (_socket_watchid)
            g_source_remove(_socket_watchid);
        _socket_watchid = g_io_add_watch(_IOChannel, GIOCondition (G_IO_IN | G_IO_PRI | G_IO_ERR | G_IO_HUP | G_IO_NVAL), &TCPTransmitter::on_socket_event, this);
    }
}

void TCPTransmitter::_send(send_data_buf* data)
{
     guint bytes_written = 0;
     // Write the data to the socket using glib
     GIOError result = G_IO_ERROR_NONE;

    if ((_state != Offline) && (_state != Error))
    {
        if (data != NULL)
        {
            _sendBuffer.push_back(data);
        }
    }
    else
    {
        delete data->data;
        delete data;

        // XXX Bounce?
        handleError("Sending data while offline");
        return;
     }
     
     
    send_data_buf* send_data = NULL;
    while (!_sendBuffer.empty())
    {
        if(send_data == NULL)
            send_data = _sendBuffer.front();

#ifdef WITH_SSL
        if(_adapter) 
        {
            result = _adapter->send(send_data->data + send_data->cur_pos, 
                                    send_data->sz - send_data->cur_pos,
				                    &bytes_written);
        }
        else 
        {
#endif
        g_assert(_IOChannel != NULL);
        result = g_io_channel_write(_IOChannel, 
                                    send_data->data + send_data->cur_pos, 
                                    send_data->sz - send_data->cur_pos,
				                    &bytes_written);
#ifdef WITH_SSL
        }
#endif // WITH_SSL
        if ((result == G_IO_ERROR_NONE || result == G_IO_ERROR_AGAIN) &&
            bytes_written > 0)
        {
            // Tell the client that we sent some data
            evtDataSent(bytes_written);
            // Update the sent block and delete it necessary
            send_data->cur_pos += bytes_written;
            // This really shouldn't be great than, but we should check anyway
            if (send_data->cur_pos >= send_data->sz)
            {
                // It's done, destroy it
                delete send_data->data;
                delete send_data;
                send_data = NULL;
                _sendBuffer.pop_front();
            }

            if (result == G_IO_ERROR_AGAIN)
            {
                needSend(true);
                return;
            }
        } else {
#ifdef WITH_SSL
	        if(_adapter) 
	        {
	            handleError(_adapter->getError());
	        } else {
#endif
	        handleError(errSocket);
#ifdef WITH_SSL
            }
#endif // WITH_SSL
            disconnect();
            return;
        }
    }

    // We're done with all the data
    needSend(_need_send);
}

// Socket related callbacks
gboolean TCPTransmitter::on_socket_accept(GIOChannel* iochannel, GIOCondition cond, gpointer data) 
{
    TCPTransmitter& transmitter = *(static_cast<TCPTransmitter*>(data));
     
    g_source_remove(transmitter._socket_watchid);
     
    if (cond != G_IO_IN)
    {
        transmitter.handleError("Accepting socket died");
        return (FALSE);
    }

    transmitter._state = Accepting;

    // Ok we're ready to accept
    struct sockaddr_in clientname;
	size_t size;

    size = sizeof(clientname);
    int fd = accept(transmitter._socketfd, 
                    (struct sockaddr*)&clientname, &size);
    if (fd < 0)
    {
        transmitter.handleError("Error accepting new socket");
        transmitter.disconnect();
        return (FALSE);
    }

    // Replace socketfd with the new one and cleanup the old
    close(transmitter._socketfd);
    transmitter._socketfd = fd;

    transmitter._IOChannel = g_io_channel_unix_new(transmitter._socketfd);
    transmitter._socket_watchid = g_io_add_watch(transmitter._IOChannel,
            GIOCondition (G_IO_IN | G_IO_PRI | G_IO_ERR | G_IO_HUP | G_IO_NVAL),
            &TCPTransmitter::on_socket_event, data);

    // Finally fire our event
    transmitter.evtAccepted();

    return (FALSE);
}

gboolean TCPTransmitter::on_socket_event(GIOChannel* iochannel, GIOCondition cond, gpointer data) 
{
    TCPTransmitter& transmitter = *(static_cast<TCPTransmitter*>(data));

    if (!transmitter._state == Connected)
    {
        // If we aren't connected don't do anything and return false to get 
        //rid of the socket callback
        return FALSE;
    }
     
    switch(cond) 
    {
    case G_IO_IN:
    {
	    // Read the data from the socket and push it into the session
        guint bytes_read = 2047;
        char buf[2048];
        while (bytes_read == 2047) 
        {
            // Ensure buffer is empty
            memset(buf, 0, 2048);
            // Read data from the channel
            GIOError result;
#ifdef WITH_SSL
            if (transmitter._adapter) 
            {
                result = transmitter._adapter->read(buf, 2047, &bytes_read);
            } else {
#endif
            g_assert(iochannel != NULL);
            result = g_io_channel_read(iochannel, buf, 2047, &bytes_read);
#ifdef WITH_SSL
            }
#endif
	        if (((result == G_IO_ERROR_AGAIN) || (result == G_IO_ERROR_NONE)) &&
                 (bytes_read > 0)) 
	        {
		        transmitter.evtDataAvailable(buf, bytes_read);
	        } else {
                if (bytes_read == 0)
                {
                    transmitter.disconnect();
                    return (FALSE);
                }
#ifdef WITH_SSL
		        if(transmitter._adapter) 
		        {
			        transmitter.handleError(transmitter._adapter->getError());
		        } else {
#endif
		        transmitter.handleError(errSocket);
#ifdef WITH_SSL
                }
#endif
		        return (FALSE);
	        }
        }
	    return (TRUE);
    }
    // We can write, send out our data
    case G_IO_OUT:
        if (transmitter._sendBuffer.empty())
        {
            transmitter.evtCanSendMore();
        } else {
            // Attempt to clear out our buffer
            transmitter._send(NULL);
        }
        return (TRUE);
    default:
	    transmitter.handleError(errSocket);
        return (FALSE);
    }
     
    return (FALSE);
}

gboolean TCPTransmitter::on_socket_connect(GIOChannel* iochannel, GIOCondition cond, gpointer data)
{
     TCPTransmitter& transmitter = *(static_cast <TCPTransmitter*>(data));
   
     // remove watch, in case we don't return immediately to avoid being
     // called more than once
     // (since we always return false, we can do it here already)
     g_source_remove(transmitter._socket_watchid);
     
     if (!transmitter._state == Connecting)
     {
          // Occurs when the Transmitter is disconnected in the process of connecting.  Return to prevent
          // an error from being triggered 
          return FALSE;
     }
     
     if ((cond & G_IO_IN) || (cond & G_IO_OUT)) 
     {
	  // was there an error?
	  gint error;
	  socklen_t len;
	  len = sizeof(error);
	  if (getsockopt(transmitter._socketfd, SOL_SOCKET, SO_ERROR,
			 static_cast <void*>(&error), &len) >= 0) 
	  {
	       if (!error) 
	       {

		    
#ifdef TRANSMITTER_DEBUG
		    cout << "Async connect worked!" << endl;
#endif
		    
            transmitter._state = Connected;
		    
		    // register socket for notification
#ifdef WITH_SSL
		    if (transmitter._adapter) 
		    {
			 // FIXME: reset the socket flags to blocking 
			 // (we currently need that for OpenSSL to work properly)
			 if ((fcntl(transmitter._socketfd, F_SETFL, transmitter._socket_flags)) == -1 ) 
			 {
#ifdef TRANSMITTER_DEBUG
			      cout << "Couldn't reset socket flags!" << endl;
#endif
			      transmitter.handleError(strerror(errno));
			      return(FALSE);
			 }
			 bool success =
			      transmitter._adapter->registerSocket(transmitter._socketfd);
			 if(!success) 
			 {
			      transmitter.handleError(transmitter._adapter->getError());
			 }
		    }
#endif
		    
		    transmitter._socket_watchid = g_io_add_watch(transmitter._IOChannel,
								 GIOCondition (G_IO_IN | G_IO_PRI | G_IO_ERR | G_IO_HUP | G_IO_NVAL),
								 &TCPTransmitter::on_socket_event, data);
		    
		    // get current time
		    g_get_current_time(transmitter._reconnect_timeval);
		    transmitter.evtConnected();
		    return (FALSE);
	       }
	       else /*if(!error)*/ 
	       {
		    transmitter.handleError(strerror(error));
	       }
	  }
	  else /*if(getsockopt() >= 0)*/ 
	  {
	       transmitter.handleError(strerror(errno));
	  }
     }
     else /*if ((cond & G_IO_IN) || (cond & G_IO_OUT))*/ 
     {
	  // there was some nasty error...
	  
#ifdef TRANSMITTER_DEBUG
	  cout << "Async connect failed: " << cond << endl;
#endif
	  
	  // close the socket
	  transmitter.handleError(errSocket);
	  g_io_channel_close(transmitter._IOChannel);
	  transmitter._socketfd = -1;
     }
     
     return (FALSE);
}

gboolean TCPTransmitter::on_host_resolved(GIOChannel* iochannel, GIOCondition cond, gpointer data)
{
     TCPTransmitter& transmitter = *(static_cast <TCPTransmitter*>(data));
     bool error = false;
     
     // remove the watch on the IO channel
     if (transmitter._resolver_watchid > 0)
          g_source_remove(transmitter._resolver_watchid);

     // if _hostResolved == true it means the address was already resolved
     if (!transmitter._hostResolved && transmitter._state == Connecting) 
     {
	  if (cond & G_IO_IN) 
	  {
	       // read from the channel
	       
#ifdef TRANSMITTER_DEBUG
	       cout << "reading from pipe" << endl;
#endif
	       
#ifdef WITH_IPV6
	       guint buf_length = sizeof(guchar) + sizeof(struct in6_addr);
#else
	       guint buf_length = sizeof(guchar) + sizeof(struct in_addr);
#endif
	       gchar * buffer = new gchar[buf_length];
	       guint bytes_read = 0;
	       GIOError result = g_io_channel_read(iochannel, buffer, buf_length, &bytes_read);
	       if (result != G_IO_ERROR_NONE) 
	       {

#ifdef TRANSMITTER_DEBUG
		    cout << "Error while reading from pipe" << endl;
#endif
		    error = true;
	       }
	       else if (buffer[0] == 0) 
	       {
		    // the hostname couldn't be resolved properly
		    
#ifdef TRANSMITTER_DEBUG
		    cout << "read from pipe got me a 0!" << endl;
#endif
		    error = true;
	       }
	       else if (bytes_read != buf_length) 
	       {
		    // something nasty happened, bail out!
#ifdef TRANSMITTER_DEBUG
		    cout << "Didn't receive enough data from resolver process!\n" << "Received: " << bytes_read << " Wanted: " << buf_length << endl;
#endif
		    //when this happens wait a second and try again
		    sleep(1);
		    guint next_bytes_read;
		    result = g_io_channel_read(iochannel, &buffer[bytes_read], buf_length - bytes_read, &next_bytes_read);
		    if (result == G_IO_ERROR_NONE) 
		    {
			 bytes_read += next_bytes_read;
			 if (bytes_read != buf_length) 
			 {
			      error = true;
#ifdef TRANSMITTER_DEBUG
			      cout << "Not enough data on retry. Received: " << bytes_read << " Wanted: " << buf_length << endl;
#endif
			 }
		    }
		    else 
		    {
			 error = true;
		    }
		    
	       }
	       
	       if (!error) 
	       {
		    // copy the received struct in_addr into _host_sockaddr.sin_addr
#ifdef WITH_IPV6
		    memcpy(static_cast <void*>(&transmitter._host_sockaddr.sin6_addr),
			   static_cast <void*>(&buffer[sizeof(guchar)]), sizeof(struct in6_addr));
#else
		    memcpy(static_cast <void*>(&transmitter._host_sockaddr.sin_addr),
			   static_cast <void*>(&buffer[sizeof(guchar)]), sizeof(struct in_addr));
#endif
#ifdef TRANSMITTER_DEBUG
		    cout << "wrote in_addr into _host_sockaddr" << endl;
		    cout << "got the following from the pipe:" << endl;
#ifdef WITH_IPV6
		    cout << "sin6_family: " << transmitter._host_sockaddr.sin6_family << endl;
		    cout << "sin6_port:   " << transmitter._host_sockaddr.sin6_port << endl;
		    char name_buffer[INET6_ADDRSTRLEN];
		    if (inet_ntop(AF_INET6, transmitter._host_sockaddr.sin6_addr.s6_addr, name_buffer, INET6_ADDRSTRLEN))
			 cout << "sin6_addr:   " << name_buffer << endl;
#else
		    cout << "sin_family: " << transmitter._host_sockaddr.sin_family << endl;
		    cout << "sin_port  : " << transmitter._host_sockaddr.sin_port << endl;
		    cout << "sin_addr  : " << transmitter._host_sockaddr.sin_addr.s_addr << endl;
#endif
#endif
		    transmitter._hostResolved = true;
	       }
	       
	       delete buffer;
	  }
	  else 
	  {
	       // should not happen...
	       error = true;
	  }
     }
     
     // close the pipes
     if (iochannel != NULL)
	  g_io_channel_close(iochannel);

     // wait for the resolver process to exit to avoid zombies
     if (transmitter._resolver_pid != -1) 
     {
	  waitpid(transmitter._resolver_pid, NULL, 0);
	  transmitter._resolver_pid = -1;
     }
     
     if(error) 
     {
	  transmitter.handleError(errAddressLookup);
     } 
     else if (transmitter._state == Connecting) 
     {
	  transmitter._async_connect();
     }

     return (FALSE);
}

gboolean TCPTransmitter::reconnect(gpointer data)
{
     TCPTransmitter & t = *(static_cast<TCPTransmitter *>(data));
     if (t._reconnect_timeoutid) 
     { 
	  // this means we haven't reconnected yet
	  t._reconnect_timeoutid = 0;
	  t._state = Reconnecting;
	  t.connect("", 0, t._use_ssl, t._autoreconnect);
     }
     return (FALSE);
}

void TCPTransmitter::handleError(const TransmitterError e) 
{
     string emsg;

     if (_state == Offline || _state == Error)
        return;
     
     switch(e) 
     {
     case errAddressLookup:
	  emsg = "Couldn't Resolve Hostname";
	  break;
     case errSocket:
	  emsg = getSocketError();
	  break;
     default:
	  // shouldn't happen
	  emsg = "Unknown Error";
     }
     
     handleError(emsg);
}

void TCPTransmitter::handleError(const string & emsg)
{
#ifdef TRANSMITTER_DEBUG
     cout << "A Transmitter Error occurred: " << emsg << endl;
#endif
     if (_state == Offline || _state == Error)
        return;
     
     // check if we should auto-reconnect: 
     if (_autoreconnect) 
     {
	  GTimeVal now;
	  g_get_current_time(&now);
	  glong connected_time = now.tv_sec - _reconnect_timeval->tv_sec;
	  if (connected_time < 60) 
	  {
	       // we've been only connected for less than a minute
	       _reconnect_delay *= 2; // double the delay
	  } 
	  else 
	  {
	       _reconnect_delay = 10; // 10 secs 
	  }
	       
#ifdef TRANSMITTER_DEBUG
	  cout << "Disconnected. Attempting auto-reconnect in " << _reconnect_delay << " seconds" << endl;
#endif
	  evtReconnect();
	  _reconnect_timeoutid = g_timeout_add(_reconnect_delay * 1000, &TCPTransmitter::reconnect, static_cast<void*>(this));
	  // set _reconnecting to true to prevent evtDisconnected from being called by disconnect ()
	  _state = Reconnecting;
     } 
     else 
     {
      _state = Error;
	  evtError(emsg);
     }
	 disconnect();
}

const string TCPTransmitter::getSocketError() 
{
     gint error;
     socklen_t len = sizeof(error);
     string emsg;
     if (_socketfd != -1) 
     {
	  if (getsockopt(_socketfd, SOL_SOCKET, SO_ERROR,
			 static_cast <void*>(&error), &len) >= 0) {
	       if (error) 
	       {
		    return (strerror(error));
	       }
	  }
     }
     return("Unknown error");
}

void TCPTransmitter::_async_resolve(const gchar* hostname)
{
     g_assert(hostname != NULL);
     // check if hostname is in dotted decimal notation
#ifdef WITH_IPV6
     if (inet_pton(AF_INET6, hostname, &_host_sockaddr.sin6_addr) != 0) 
#else
     if (inet_aton(hostname, &_host_sockaddr.sin_addr) != 0) 
#endif
     {
	  // all done
	  _hostResolved = true;
	  
#ifdef TRANSMITTER_DEBUG
	  cout << "dot-dec resolved" << endl;
#endif
	  
	  on_host_resolved(NULL, G_IO_NVAL, this);
	  
	  return;
     }
     
     // didn't work, we need to do a lookup
     _resolver_pid = -1;
     gint pipes[2];
     
     // create a pipe
     if (pipe(pipes) == -1)
     {
	  handleError(errAddressLookup);
	  return;
     }
     
     errno = 0;
     // try to fork as long as errno == EAGAIN
     do 
     {
	  // fork the lookup process
	  if ((_resolver_pid = fork()) == 0) 
	  {
#ifdef WITH_IPV6
	       struct in6_addr ia;
#else
	       struct in_addr ia;
#endif
	       if (!_gethostbyname(hostname, &ia)) 
	       {
		    guchar zero = 0;
		    if (write(pipes[1], &zero, sizeof(zero)) == -1) 
		    {
			 g_warning("Problem with writing to pipe");
		    }
#ifdef TRANSMITTER_DEBUG
		    cout << "host couldn't be resolved, wrote NULL " << endl;
#endif
	       }
	       else 
	       {
#ifdef WITH_IPV6
		    guchar size = sizeof(struct in6_addr);
#else
		    guchar size = sizeof(struct in_addr);
#endif
		    
#ifdef TRANSMITTER_DEBUG
#ifdef WITH_IPV6
		    char name_buffer[INET6_ADDRSTRLEN];
		    if (inet_ntop(AF_INET6, ia.s6_addr, name_buffer, INET6_ADDRSTRLEN))
			 cout << "wrote to pipe: sin6_addr:   " << name_buffer << endl;
#else
		    cout << "wrote to pipe: sin_addr: " << ia.s_addr << endl;
#endif
#endif
		    
#ifdef WITH_IPV6
		    if ( (write(pipes[1], &size, sizeof(guchar))) == -1 ||
			 (write(pipes[1], &ia, sizeof(struct in6_addr)) == -1)) 
		    {
			 g_warning("Problem with writing to pipe");
		    }
#else 
		    if ( (write(pipes[1], &size, sizeof(guchar))) == -1 ||
			 (write(pipes[1], &ia, sizeof(struct in_addr)) == -1)) 
		    {
			 g_warning("Problem with writing to pipe");
		    }
#endif
		    
		    
#ifdef TRANSMITTER_DEBUG
		    cout << "lookup process done!" << endl;
#endif
		    
	       }
	       // close our end of the pipe
	       close(pipes[1]);
	       
	       // exit (call _exit() to avoid atexit being called)	
	       _exit(EXIT_SUCCESS);
	       
	  }
	  else if (_resolver_pid > 0) 
	  {
	       // parent process creates an IOChannel to read from the pipe
	       _resolver_watchid = g_io_add_watch(g_io_channel_unix_new(pipes[0]),
						  GIOCondition (G_IO_IN | G_IO_ERR | G_IO_HUP | G_IO_NVAL),
						  &TCPTransmitter::on_host_resolved,
						  this);
	       return;
	  }

	  else if (errno != EAGAIN) 
	  {
	       // nasty error
	       g_warning("Resolver fork error: %s (%d)\n", g_strerror(errno), errno);
	       handleError(errAddressLookup);
	       return;
	  }
	  
     }
     while (errno == EAGAIN);
	  
     return;
}

void TCPTransmitter::_async_connect()
{
     // connect non-blocking
     
     // create socket
#ifdef WITH_IPV6
     _socketfd = socket(PF_INET6, SOCK_STREAM, 0);
#else
     _socketfd = socket(PF_INET, SOCK_STREAM, 0);
#endif
     if (_socketfd < 0) 
     {
	  // something nasty happened
#ifdef TRANSMITTER_DEBUG
	  cout << "socket() failed: " << strerror(errno) << endl;
#endif
	  handleError(strerror(errno));
	  return;
     }
     _socket_flags = fcntl(_socketfd, F_GETFL, 0);
     if (_socket_flags == -1) 
     {
	  // not good
#ifdef TRANSMITTER_DEBUG
	  cout << "fcntl F_GETFL failed on socket: " << strerror(errno) << endl;
#endif
	  handleError(errSocket);
	  return;
     }
     if (fcntl(_socketfd, F_SETFL, _socket_flags | O_NONBLOCK) == -1) 
     {
	  // damn!
#ifdef TRANSMITTER_DEBUG
	  cout << "fcntl F_SETFL failed on socket: " << strerror(errno) << endl;
#endif
	  handleError(strerror(errno));
	  return;
     }
     
     int one = 1;
     if (setsockopt(_socketfd, SOL_SOCKET, SO_KEEPALIVE, &one, sizeof(one)) < 0) 
     {
#ifdef TRANSMITTER_DEBUG
          cout << "setsockopt failed: " << strerror(errno) << endl;
#endif
     }
     
     // try to connect non-blocking
     if (::connect(_socketfd, (struct sockaddr*) (&_host_sockaddr), sizeof(_host_sockaddr)) < 0) 
     {
	  if (errno != EINPROGRESS) 
	  {
	       // Yikes!
#ifdef TRANSMITTER_DEBUG
	       cout << "connect failed: " << strerror(errno) << endl;
#endif	       
	       handleError(strerror(errno));
	       return;
	  }
     }
     
     _IOChannel = g_io_channel_unix_new(_socketfd);
     _socket_watchid = g_io_add_watch(_IOChannel,
				      GIOCondition(G_IO_IN | G_IO_OUT | G_IO_PRI | G_IO_ERR | G_IO_HUP | G_IO_NVAL),
				      &TCPTransmitter::on_socket_connect, this);

#ifdef TRANSMITTER_DEBUG
     cout << "IOChannel watch added!" << endl;
#endif
}

#ifdef WITH_IPV6
bool TCPTransmitter::_gethostbyname(const gchar* hostname, struct in6_addr* addr_result)
#else
bool TCPTransmitter::_gethostbyname(const gchar* hostname, struct in_addr* addr_result)
#endif
{
#ifdef HAVE_GETHOSTBYNAME_R_GLIBC
     struct hostent result_buf, *result;
     size_t len;
     char* buf;
     int herr;
     int res;
     
     len = 1024;
     buf = g_new(gchar, len);
     
     while ((res = gethostbyname_r (hostname, &result_buf, buf, len,
				    &result, &herr)) == ERANGE) 
     {
	  len *= 2;
	  buf = g_renew (gchar, buf, len);
     }
     
     if (res || result == NULL || result->h_addr_list[0] == NULL) 
     {
	  g_free(buf);
	  return false;
     }
     
     if (addr_result)
     {
	  memcpy(addr_result, result->h_addr_list[0], result->h_length);
     }
     
     g_free(buf);

#else
#ifdef HAVE_GET_HOSTBYNAME_R_SOLARIS
     struct hostent result;
     size_t len;
     char* buf;
     int herr;
     int res;
     
     len = 1024;
     buf = g_new(gchar, len);
     
     while ((res = gethostbyname_r (hostname, &result, buf, len,
				    &herr)) == ERANGE)
     {
	  len *= 2;
	  buf = g_renew (gchar, buf, len);
     }
     
     if (res || hp == NULL || hp->h_addr_list[0] == NULL) 
     {
	  g_free(buf);
	  return false;
     }
     
     if (addr_result)
     {
	  memcpy(addr_result, result->h_addr_list[0], result->h_length);
     }
     
     g_free(buf);
#else
#ifdef HAVE_GETHOSTBYNAME_R_HPUX
     struct hostent result;
     struct hostent_data buf;
     int res;
     
     res = gethostbyname_r (hostname, &result, &buf);
     
     if (res == 0) 
     {
	  if (addr_result) 
	  {
	       memcpy(addr_result, result.h_addr_list[0], result.h_length);
	  }
     }
     else
	  return false;
#else 
#ifdef HAVE_GETHOSTBYNAME_R_GLIB_MUTEX
     struct hostent* he;
     
     g_static_mutex_lock(&_gethostbyname_mutex);
     he = gethostbyname(hostname);
     g_static_mutex_unlock(&_gethostbyname_mutex);
     
     if (he != NULL && he->h_addr_list[0] != NULL) 
     {
	  if (addr_result)
	  {
	       memcpy(addr_result, he->h_addr_list[0], he->h_length);
	  }
     }
     else
	  return false;
#else
     struct hostent* he;
     
     he = gethostbyname(hostname);
     if (he != NULL && he->h_addr_list[0] != NULL)
     {
	  if (addr_result)
	  {
	       memcpy(addr_result, he->h_addr_list[0], he->h_length);
	  }
     }
     else
	  return false;
#endif // HAVE_GETHOSTBYNAME_R_GLIB_MUTEX
#endif // HAVE_GETHOSTBYNAME_R_HPUX
#endif // HAVE_GET_HOSTBYNAME_R_SOLARIS
#endif // HAVE_GETHOSTBYNAME_R_GLIBC

     return true;
}

gboolean TCPTransmitter::_connection_poll(gpointer data) 
{
     static struct sockaddr addr;
     static socklen_t addr_length = 1;
     
     TCPTransmitter& transmitter = *(static_cast<TCPTransmitter*> (data));
     
#ifdef TRANSMITTER_DEBUG
     cout << "_connection_poll called" << endl;
#endif     
     
     if (transmitter._state == Connected) 
     {
	  transmitter.send(" ");
     }
     
#ifdef POLL_DEBUG     
     if (getpeername(transmitter._socketfd, &addr, &addr_length) != 0) 
     {
	  if (transmitter._state == Connected) 
	  {
	       //an error occurred
	       cout << "_connection_poll: Error: " << strerror(errno) << endl;
	       transmitter.handleError(strerror(errno));
	  } 
	  return (FALSE);  
     }
#endif
     
#ifdef TRANSMITTER_DEBUG
     cout << "_connection_poll: Everything fine" << endl;
#endif
     return (TRUE);
}
