/* 
   Copyright  1998, 1999 Enbridge Pipelines Inc. 
   Copyright  1999 Dave Carrigan
   All rights reserved.

   This module is free software; you can redistribute it and/or modify
   it under the same terms as Apache itself. This module is
   distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE. The copyright holder of this
   module can not be held liable for any general, special, incidental
   or consequential damages arising out of the use of the module.

   $Id: auth_ldap.c,v 1.2 1999/08/27 05:05:07 dave Exp $
*/

const char *auth_ldap_version = "1.4.0";

#include "auth_ldap.h"

module MODULE_VAR_EXPORT auth_ldap_module;

/* Backwards compatibility for Apache 1.2 */
#if APACHE_RELEASE < 1030000
int ldap_loglevel = 4; 
void ap_log_error(const char *file, int line, int level,
		  const server_rec *s, const char *fmt, ...)
{
  va_list args;
  if (level >= ldap_loglevel)
    return;
  fprintf(stderr, "(%s:%d) ", file, line);
  fflush(stderr);
  va_start(args, fmt);
  vfprintf(stderr, fmt, args);
  fprintf(stderr, "\n");
  fflush(stderr);
  va_end(args);
}
#endif /* if APACHE_RELEASE */

/* 
 * Compatibility for OpenLDAP libraries. Differences between OpenLDAP and 
 * Netscape LDAP, as they affect this module
 * 
 *  Netscape uses ldap_memcache_*; OpenLDAP uses ldap_enable_cache, et. al.
 *
 *  Netscape uses ldap_search_ext_s; OpenLDAP uses only basic ldap_search_s
 *
 *  Netscape uses ldap_memfree; OpenLDAP just uses free().
 *
 *  Netscape uses ldap_init; OpenLDAP uses ldap_open which calls an internal 
 *  ldap_init function.
 *
 * In this section, we just implement the Netscape SDK functions that are 
 * missing in OpenLDAP. The exception is ldap_init, which already exists 
 * internally in OpenLDAP, so we define an auth_ldap_init, which calls 
 * ldap_init under Netscape and ldap_open under OpenLDAP.
 * 
 */
#ifdef WITH_OPENLDAP
/*
 * Netscape separates cache creation from attaching the cache to the LDAP 
 * connection; OpenLDAP does not. Under OpenLDAP, we use memcache_init to
 * save the ttl and size, and then use memcache_set to call OpenLDAP's
 * enable_cache with the saved values
 */
int
ldap_memcache_init(unsigned long ttl, unsigned long size, char **baseDNs, 
		   void *thread_fns, LDAPMemCache **cachep)
{
  *cachep = (LDAPMemCache *)malloc(sizeof(LDAPMemCache));
  if (*cachep) {
    (*cachep)->ttl = ttl;
    (*cachep)->size = size;
    return LDAP_SUCCESS;
  } else {
    return LDAP_NO_MEMORY;
  }
}

int
ldap_memcache_set(LDAP *ld, LDAPMemCache *c)
{
  return ldap_enable_cache(ld, c->ttl, c->size);
}

/*
 * OpenLDAP doesn't support extended search. Since auth_ldap doesn't use
 * it anyway, we just translate the extended search into a normal search.
 */
int
ldap_search_ext_s(LDAP *ldap, char *base, int scope, char *filter,
		  char **attrs, int attrsonly, void *servertrls, void *clientctrls,
		  void *timeout, int sizelimit, LDAPMessage **res)
{
  return ldap_search_s(ldap, base, scope, filter, attrs, attrsonly, res);
}

void
ldap_memfree(void *p)
{
  free(p);
}

LDAP *
auth_ldap_init(char *host, int port)
{
  LDAP *ret;
  fprintf(stderr, "ldap_init\n");
  ret = ldap_open(host, port);
  if (ret) ret->ld_options = 0;
  return ret;
}
#else
LDAP *
auth_ldap_init(char *host, int port)
{
  fprintf(stderr, "auth_ldap_init\n");
  return ldap_init(host, port);
}
#endif /* WITH_OPENLDAP */

/* This function calls log_reason in 1.2, while it calls ap_log_error in 1.3+ */
void auth_ldap_log_reason(request_rec *r, const char *fmt, ...)
{
  va_list args;
  char buf[MAX_STRING_LEN];

  va_start(args, fmt);
  ap_vsnprintf(buf, sizeof(buf), fmt, args);
  va_end(args);

  #if APACHE_RELEASE < 1030000
  log_reason(buf, r->uri, r);
  #else
  ap_log_rerror(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, r, buf);
  #endif
}


/*
 * Closes an LDAP connection by unbinding. Sets the needbind flag for the
 * http connection config record and clears the bound dn string in the
 * global connection record. The next time connect_to_server is called, the
 * connection will be recreated.
 *
 * Note: the LDAP op cache is not deleted; the assumption is that the op
 * cache should be valid on the failover server.
 *
 * If the log parameter is set, adds a debug entry to the log that the
 * server was down and it's reconnecting.
 *
 * The mutex for the LDAPConnection should already be acquired.
 */
void
auth_ldap_free_connection(request_rec *r, int log)
{
  auth_ldap_config_rec *sec =
    (auth_ldap_config_rec *)ap_get_module_config(r->per_dir_config, &auth_ldap_module);

  if (log)
    ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		  "{%d} Server is down; reconnecting and starting over", (int)getpid());

  if (sec->ldc->ldap) {
    ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		  "{%d} Freeing connection to ldap server(s) `%s'", 
		  (int)getpid(), sec->host);
    ldap_unbind_s(sec->ldc->ldap);
    sec->ldc->ldap = NULL;
    sec->needbind = 1;
    if (sec->ldc->bounddn) {
      free(sec->ldc->bounddn);
      sec->ldc->bounddn = NULL;
    }
  }
}

/*
 * Connect to the LDAP server and binds. Does not connect if already
 * connected (i.e.sec->ldc->ldap is non-NULL.) Does not bind if already bound.
 * Returns 1 on success; 0 on failure
 *
 * The mutex for the LDAPConnection should already be acquired.
 */
int
auth_ldap_connect_to_server(request_rec *r)
{
  int result;
  auth_ldap_config_rec *sec;
  auth_ldap_server_conf *conf;
  int failures = 0;

  conf = (auth_ldap_server_conf *)ap_get_module_config(r->server->module_config,
						       &auth_ldap_module);
  sec = (auth_ldap_config_rec *)ap_get_module_config(r->per_dir_config, 
						     &auth_ldap_module);

 start_over:
  if (failures++ > 10) {
    auth_ldap_log_reason(r, "Too many failures connecting to LDAP server");
    return 0;
  }
  if (!sec->ldc->ldap) {
    sec->needbind = 1;
    if (sec->ldc->bounddn) {
      free(sec->ldc->bounddn);
      sec->ldc->bounddn = NULL;
    }
    ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		  "{%d} Opening connection to ldap server(s) `%s'", 
		  (int)getpid(), sec->host);
    ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		  "{%d} LDAP OP: init", (int)getpid());
    
    if ((sec->ldc->ldap = auth_ldap_init(sec->host, sec->port)) == NULL) {
      extern int errno;
      auth_ldap_log_reason(r, "Could not connect to LDAP server: %s", strerror(errno));
      return 0;
    }

#ifdef WITH_SSL
    if (sec->secure) {
      ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		    "{%d} Initializing SSL for this connection", 
		    (int)getpid());
      if (!conf->have_certdb) {
	auth_ldap_log_reason(r, 
			     "Secure LDAP requested, but no "
			     "AuthLDAPCertDBPath directive in config");
	return 0;
      } else {
	result = ldapssl_install_routines(sec->ldc->ldap);
	if (result != LDAP_SUCCESS) {
	  auth_ldap_log_reason(r, "SSL initialization failed: %s", 
			       ldap_err2string(result));
	  return 0;
	}
	result = ldap_set_option(sec->ldc->ldap, LDAP_OPT_SSL, LDAP_OPT_ON);
	if (result != LDAP_SUCCESS) {
	  auth_ldap_log_reason(r, "SSL option failed: %s", 
			       ldap_err2string(result));
	  return 0;
	}
      }
    } 
#endif
  }

  /* At this point the LDAP connection is now alive. */
  if (!sec->needbind) {
    return 1;
  }

  ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		"{%d} Binding to server `%s' as %s/%s", (int)getpid(),
		sec->host, 
		sec->binddn? sec->binddn : "(null)",
		sec->bindpw? sec->bindpw : "(null)");
  /* 
   * Now bind with the username/password provided by the
   * configuration. It will be an anonymous bind if no u/p was
   * provided. 
   */
  ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		"{%d} LDAP OP: simple bind", (int)getpid());
  if ((result = ldap_simple_bind_s(sec->ldc->ldap, sec->binddn, sec->bindpw))
      == LDAP_SERVER_DOWN) {
    auth_ldap_free_connection(r, 1);
    goto start_over;
  }

  if (result != LDAP_SUCCESS) {
    auth_ldap_free_connection(r, 0);
    auth_ldap_log_reason(r, "Could not bind to LDAP server `%s' as %s: %s", 
			 sec->host, 
			 sec->binddn? sec->binddn : "(null)", 
			 ldap_err2string(result));
    return 0;
  } 

  if (sec->ldc->cache) {
    int result;
    ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		  "{%d} Attaching ldap connection to ldap cache", (int)getpid());
    result = ldap_memcache_set(sec->ldc->ldap, sec->ldc->cache);
    if (result != LDAP_SUCCESS) {
      ap_log_rerror(APLOG_MARK, APLOG_WARNING|APLOG_NOERRNO, r,
		    "Setting cache to ldap server failed: %s", 
		    ldap_err2string(result));
    }
  }

  sec->ldc->bounddn = sec->binddn? strdup(sec->binddn) : NULL;
  sec->needbind = 0;

  return 1;
}

/*
 * Find an existing ldap connection struct that matches the
 * host. Create a new one if none are found.
 * Will set the ldc field of the sec struct and create a memcache.
 */
void
auth_ldap_find_connection(auth_ldap_config_rec *sec, request_rec *r)
{
  struct LDAPconnection *l, *p;	/* To traverse the linked list */
  auth_ldap_server_conf *conf;

  conf = (auth_ldap_server_conf *)ap_get_module_config(r->server->module_config,
						       &auth_ldap_module);

  /* Find if this connection exists in the linked list yet */
  if (ap_acquire_mutex(conf->mtx) != MULTI_OK)
    ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
		  "Could not acquire configuration mutex. Expect deadlocks.");

  for (l=conf->ldapconnections,p=NULL; l; l=l->next) {
    if (l->port == sec->port && strcmp(l->host, sec->host) == 0)
      break;
    p = l;
  }

  if (l) {
    /* 
     * Existing connection. Now determine if we need to rebind because
     * the last time this connection was used, it bound as a differert
     * dn than this time.
     */
    if ((sec->binddn && !l->bounddn) ||
	(!sec->binddn && l->bounddn) ||
	(sec->binddn && l->bounddn && strcmp(sec->binddn, l->bounddn) != 0))
      sec->needbind = 1;
    else
      sec->needbind = 0;
  } else {
    /* 
       Create a new connection entry in the linked list. Note that we
       don't actually establish an LDAP connection yet; that happens
       the first time authentication is requested.

       We allocate these connections out of the real heap, not from a
       pool. This is because the connections should last as long as
       possible.
    */
    l = (struct LDAPconnection *)malloc(sizeof(struct LDAPconnection));
    if (!l) {
      if (!ap_release_mutex(conf->mtx))
	ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
		      "Could not release configuration mutex. Expect deadlocks.");

      return;
    }

    l->ldap = NULL;
    if (conf->cache_size >= 0) {
      int result;

      ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
		    "{%d} Creating LDAP cache", (int)getpid());
      result = ldap_memcache_init(conf->cache_ttl, conf->cache_size,
				  NULL, NULL, &(l->cache));
      if (result != LDAP_SUCCESS) {
	ap_log_rerror(APLOG_MARK, APLOG_WARNING|APLOG_NOERRNO, r,
		      "LDAP cache initialization failed: %s", ldap_err2string(result));
	
	l->cache = NULL;
      }
    } else {
      l->cache = NULL;
    }
    if (conf->opcache_size > 0) {
      ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
		    "{%d} Creating bind cache", (int)getpid());
      l->ldapopcache = auth_ldap_new_cache(conf->opcache_size);
    } else {
      l->ldapopcache = NULL;
    }
    l->host = strdup(sec->host);
    l->port = sec->port;
    l->bounddn = NULL;
    l->next = NULL;
    l->mtx = ap_create_mutex(NULL);
    if (p) {
      p->next = l;
    } else {
      conf->ldapconnections = l;
    }
    sec->needbind = 1;
  }
  sec->ldc = l;
  if (!ap_release_mutex(conf->mtx))
    ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
		  "Could not release configuration mutex. Expect deadlocks.");
}

int
ldap_authenticate_basic_user(request_rec *r)
{
  auth_ldap_config_rec *sec =
    (auth_ldap_config_rec *)ap_get_module_config(r->per_dir_config, &auth_ldap_module);
  const char *sent_pw;
  int result;
  LDAPMessage *res, *entry;
  char filtbuf[MAX_STRING_LEN];
  int count;
  char *dn;
  int ret;
  int failures = 0;

  /* 
     Basic sanity checks before any LDAP operations even happen.
  */
  if (!sec->have_ldap_url)
    return DECLINED;

  /* There is a good AuthLDAPURL, right? */
  if (sec->ldc == NULL) {
    auth_ldap_find_connection(sec, r);
    if (sec->ldc == NULL) {
      auth_ldap_log_reason(r, "Could not find/create LDAPconnection struct");
      return sec->auth_authoritative? AUTH_REQUIRED : DECLINED;
    }
  }

  ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		"{%d} authenticate: using URL %s", (int)getpid(), sec->url);

  /* Get the password that the client sent */
  if ((result = ap_get_basic_auth_pw(r, &sent_pw))) {
    ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		 "{%d} authenticate: result is %d", (int)getpid(), result);
    return result;
  }

  if (ap_acquire_mutex(sec->ldc->mtx) != MULTI_OK)
    ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
		  "Could not acquire connection mutex. Expect deadlocks.");

  /*
   * If any LDAP operation fails due to LDAP_SERVER_DOWN, control returns here.
   */
 start_over:
  if (failures++ > 10) {
    auth_ldap_log_reason(r, "Too many failures connecting to LDAP server");
    return 0;
  }
  if (!auth_ldap_connect_to_server(r))
    return sec->auth_authoritative? AUTH_REQUIRED : DECLINED;

  /* 
     Now, search for this user in the directory. The search filter
     consists of the filter provided with the URL, combined with a
     filter made up of the attribute provided with the URL, and the
     actual username passed by the HTTP client. For example, assume
     that the LDAP URL is 
     
       ldap://ldap.airius.com/ou=People, o=Airius?uid??(posixid=*)

     Further, assume that the userid passed by the client was `userj'.
     The search filter will be (&(posixid=*)(uid=userj))
       
     We don't worry about the buffer not being able to hold the entire
     filter. If ap_snprintf couldn't hold the entire filter, then
     ldap_search_s will fail with an invalid filter error. Since the
     filter would have to be pathologically long for this to happen, I
     don't consider it a problem that the error message won't be very
     meaningful.
  */
  ap_snprintf(filtbuf, sizeof(filtbuf), "(&(%s)(%s=%s))", 
	      sec->filter, sec->attribute, r->connection->user);

  /* 
     Now do a search with the resulting filter, starting from the base
     dn. We do attrs only, because we just want to find the entry so
     that we can get its DN, not to actually retrieve any of its
     attributes.
  */
  
  ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		"{%d} Peforming a search with filter %s", (int)getpid(), filtbuf);

  ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		"{%d} LDAP OP: search", (int)getpid());
  
  if ((result = ldap_search_ext_s(sec->ldc->ldap,
				  sec->basedn, sec->scope, 
				  filtbuf, NULL, 1, 
				  NULL, NULL, NULL, -1, &res)) == LDAP_SERVER_DOWN) {
    ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		  "{%d} Server is down; reconnecting and starting over", (int)getpid());
    auth_ldap_free_connection(r, 1);
    goto start_over;
  }

  if (result != LDAP_SUCCESS) {
    auth_ldap_log_reason(r, "LDAP search for %s failed: LDAP error: %s; URI %s", 
			 filtbuf, ldap_err2string(result), r->uri);
    if (!ap_release_mutex(sec->ldc->mtx))
      ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
		    "Could not release connection mutex. Expect deadlocks.");

    return sec->auth_authoritative? AUTH_REQUIRED : DECLINED;
  }

  /* 
     We should have found exactly one entry; to find a different
     number is an error.
  */
  count = ldap_count_entries(sec->ldc->ldap, res);
  if (count != 1) {
    auth_ldap_log_reason(r, 
			 "Search must return exactly 1 entry; found "
			 "%d entries for search %s: URI %s", 
			 count, filtbuf, r->uri);
    if (!ap_release_mutex(sec->ldc->mtx))
      ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
		    "Could not release connection mutex. Expect deadlocks.");
    return sec->auth_authoritative? AUTH_REQUIRED: DECLINED;
  }

  entry = ldap_first_entry(sec->ldc->ldap, res);

  /* Grab the dn, copy it into the pool, and free it again */
  dn = ldap_get_dn(sec->ldc->ldap, entry);
  sec->dn = ap_pstrdup(r->pool, dn);
  ldap_memfree(dn);
    
  ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		"{%d} DN returned from search is %s", (int)getpid(), sec->dn);
    
  /* 
     A bind to the server with an empty password always succeeds, so
     we check to ensure that the password is not empty. This implies
     that users who actually do have empty passwords will never be
     able to authenticate with this module. I don't see this as a big
     problem.
  */
  if (strlen(sent_pw) <= 0) {
    auth_ldap_log_reason(r, "AuthLDAP: user %s provided an empty password: %s", 
			 r->connection->user, r->uri);
    ap_note_basic_auth_failure(r);
    if (!ap_release_mutex(sec->ldc->mtx))
      ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
		    "Could not release connection mutex. Expect deadlocks.");
    return AUTH_REQUIRED;
  }

  ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		"{%d} Validating possibly cached user %s via bind", 
		(int)getpid(), sec->dn);

  ret = auth_ldap_authbind(sent_pw, r);
  if (!ap_release_mutex(sec->ldc->mtx))
    ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
		  "Could not release connection mutex. Expect deadlocks.");
  return ret;
}



#if APACHE_RELEASE >= 1030000
#ifdef AUTH_LDAP_FRONTPAGE_HACK
/* 
 * In apache 1.2.x, we could just call mod_auth's get_pw. In 1.3.x, mod_auth could
 * be a DSO, and the get_pw symbol may not be available to this module, so we have
 * to replicate mod_auth's get_pw here.
 */
static char *auth_ldap_get_pw(request_rec *r, char *user, char *auth_pwfile)
{
    configfile_t *f;
    char l[MAX_STRING_LEN];
    const char *rpw, *w;

    if (!(f = ap_pcfg_openfile(r->pool, auth_pwfile))) {
	ap_log_rerror(APLOG_MARK, APLOG_ERR, r,
		    "Could not open password file: %s", auth_pwfile);
	return NULL;
    }
    while (!(ap_cfg_getline(l, MAX_STRING_LEN, f))) {
	if ((l[0] == '#') || (!l[0]))
	    continue;
	rpw = l;
	w = ap_getword(r->pool, &rpw, ':');

	if (!strcmp(user, w)) {
	    ap_cfg_closefile(f);
	    return ap_getword(r->pool, &rpw, ':');
	}
    }
    ap_cfg_closefile(f);
    return NULL;
}
#endif
#endif

int
ldap_check_auth(request_rec *r)
{
  auth_ldap_config_rec *sec =
    (auth_ldap_config_rec *)ap_get_module_config(r->per_dir_config, &auth_ldap_module);

  int m = r->method_number;

  const array_header *reqs_arr = ap_requires(r);
  require_line *reqs = reqs_arr ? (require_line *)reqs_arr->elts : NULL;

  register int x;
  const char *t;
  char *w;
  int method_restricted = 0;

  if (sec->ldc == NULL) {
    ap_log_rerror(APLOG_MARK, APLOG_WARNING|APLOG_NOERRNO, r,
		 "No URL defined for LDAP authentication");
    return DECLINED;
  }

  if (sec->user_is_dn) 
    r->connection->user = sec->dn;

  if (!reqs_arr) {
    ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r,
		  "{%d} No requirements array", (int)getpid());
    return sec->auth_authoritative? AUTH_REQUIRED : DECLINED;
  }

  /* Loop through the requirements array until there's no elements
     left, or something causes a return from inside the loop */
  for(x=0; x < reqs_arr->nelts; x++) {
    if (! (reqs[x].method_mask & (1 << m))) continue;

    method_restricted = 1;
	
    t = reqs[x].requirement;
    w = ap_getword(r->pool, &t, ' ');
    
    if (strcmp(w, "valid-user") == 0) {
#ifdef AUTH_LDAP_FRONTPAGE_HACK
      /*
	Valid user will always be true if we authenticated with ldap,
	but when using front page, valid user should only be true if
	he exists in the frontpage password file. This hack will get
	auth_ldap to look up the user in the the pw file to really be
	sure that he's valid. Naturally, it requires mod_auth to be
	compiled in, but if mod_auth wasn't in there, then the need
	for this hack wouldn't exist anyway.
      */
      if (sec->frontpage_hack && sec->frontpage_hack_pwfile != NULL) {
	if (auth_ldap_get_pw(r, r->connection->user, sec->frontpage_hack_pwfile) != NULL) {
	  ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
			"{%d} agreeing to authenticate because user is "
			"valid-user (FP Hack)", (int)getpid());
	  return OK;
	} else {
	  auth_ldap_log_reason(r, "Denying user %s because user is valid in ldap, not in FrontPage", r->connection->user);
	  ap_note_basic_auth_failure(r);
	  return AUTH_REQUIRED;
	}
      } else {
#endif
	ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
		      "{%d} agreeing to authenticate because user "
		      "is valid-user", (int)getpid());
	return OK;
#ifdef AUTH_LDAP_FRONTPAGE_HACK
      }
#endif
    } else if (strcmp(w, "user") == 0) {
      /* 
       * First do a whole-line compare, in case it's something like
       *   require user Babs Jensen
       */
      if (auth_ldap_compare(sec->dn, sec->attribute, t, r)) { 
	ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
		      "{%d} agreeing to authenticate because of "
		      "require user directive", (int)getpid());
	return OK;
      }
      /* 
       * Now break apart the line and compare each word on it 
       */
      while (t[0]) {
	w = ap_getword_conf(r->pool, &t);
	if (auth_ldap_compare(sec->dn, sec->attribute, t, r)) {
	  ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
			"{%d} agreeing to authenticate because of"
			"require user directive", (int)getpid());
	  return OK;
	}
      }
    } else if (strcmp(w, "group") == 0) {
      if (auth_ldap_compare(t, "member", sec->dn, r) ||
	  auth_ldap_compare(t, "uniquemember", sec->dn, r)) {
	ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
		      "{%d} agreeing to authenticate because "
		      "of group membership", (int)getpid());
	return OK;
      }
    }
  }

  if (!method_restricted) {
    ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
		  "{%d} agreeing to authenticate because non-restricted", 
		  (int)getpid());
    return OK;
  }

  if (!sec->auth_authoritative) {
    ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
		  "{%d} declining to authenticate", (int)getpid());
    return DECLINED;
  }

  ap_log_rerror(APLOG_MARK, APLOG_DEBUG|APLOG_NOERRNO, r, 
		"{%d} denying authentication", (int)getpid());
  ap_note_basic_auth_failure (r);
  return AUTH_REQUIRED;
}

command_rec auth_ldap_cmds[] = {
  {"AuthLDAPURL", parse_auth_ldap_url, NULL, OR_AUTHCFG, RAW_ARGS, 
   "URL to define LDAP connection. This should be an RFC 2255 complaint\n"
   "URL of the form ldap://host[:port]/basedn[?attrib[?scope[?filter]]].\n"
   "<ul>\n"
   "<li>Host is the name of the LDAP server. Use a space separated list of hosts \n"
   "to specify redundant servers.\n"
   "<li>Port is optional, and specifies the port to connect to.\n"
   "<li>basedn specifies the base DN to start searches from\n"
   "<li>Attrib specifies what attribute to search for in the directory. If not "
   "provided, it defaults to <b>uid</b>.\n"
   "<li>Scope is the scope of the search, and can be either <b>sub</b> or "
   "<b>one</b>. If not provided, the default is <b>sub</b>.\n"
   "<li>Filter is a filter to use in the search. If not provided, "
   "defaults to <b>(objectClass=*)</b>.\n"
   "</ul>\n"
   "Searches are performed using the attribute and the filter combined. "
   "For example, assume that the\n"
   "LDAP URL is <b>ldap://ldap.airius.com/ou=People, o=Airius?uid?sub?(posixid=*)</b>. "
   "Searches will\n"
   "be done using the filter <b>(&((posixid=*))(uid=<i>username</i>))</b>, "
   "where <i>username</i>\n"
   "is the user name passed by the HTTP client. The search will be a subtree "
   "search on the branch <b>ou=People, o=Airius</b>."
  },

  {"AuthLDAPBindDN", ap_set_string_slot,
   (void *)XtOffsetOf(auth_ldap_config_rec, binddn), OR_AUTHCFG, TAKE1, 
   "DN to use to bind to LDAP server. If not provided, will do an anonymous bind."},

  {"AuthLDAPBindPassword", ap_set_string_slot,
   (void *)XtOffsetOf(auth_ldap_config_rec, bindpw), OR_AUTHCFG, TAKE1, 
   "Password to use to bind to LDAP server. If not provided, will do an anonymous bind "
   "will be done."},

  {"AuthLDAPRemoteUserIsDN", ap_set_flag_slot,
   (void *)XtOffsetOf(auth_ldap_config_rec, user_is_dn), OR_AUTHCFG, FLAG,
   "Set to 'on' to set the REMOTE_USER environment variable to be the full "
   "DN of the remote user. By default, this is set to off, meaning that "
   "the REMOTE_USER variable will contain whatever value the remote user sent."},

  {"AuthLDAPAuthoritative", ap_set_flag_slot,
   (void *)XtOffsetOf(auth_ldap_config_rec, auth_authoritative), OR_AUTHCFG, FLAG,
   "Set to 'off' to allow access control to be passed along to lower modules if "
   "the UserID and/or group is not known to this module"},

#ifdef WITH_SSL
  {"AuthLDAPCertDBPath", auth_ldap_set_certdbpath, NULL, RSRC_CONF, TAKE1,
   "Specifies the file containing Certificate Authority certificates "
   "for validating secure LDAP server certificates. This file must be the "
   "cert7.db database used by Netscape Communicator"},
#endif

  {"AuthLDAPCacheSize", auth_ldap_set_cache_size, NULL, RSRC_CONF, TAKE1,
   "Sets the maximum amount of memory in bytes that the LDAP search cache will use. "
   "Zero means no limit; -1 disables the cache. Defaults to 10KB."},

  {"AuthLDAPCacheTTL", auth_ldap_set_cache_ttl, NULL, RSRC_CONF, TAKE1,
   "Sets the maximum time (in seconds) that an item can be cached in the LDAP "
   "search cache. Zero means no limit. Defaults to 600 seconds (10 minutes)."},

  {"AuthLDAPOpCacheSize", auth_ldap_set_opcache_size, NULL, RSRC_CONF, TAKE1,
   "Sets the initial size of the LDAP operation cache (for bind and compare "
   "operations). Set to 0 to disable the cache. The default is 1024, which is "
   "sufficient to efficiently cache approximately 3000 distinct DNs."},

  {"AuthLDAPOpCacheTTL", auth_ldap_set_opcache_ttl, NULL, RSRC_CONF, TAKE1,
   "Sets the maximum time (in seconds) that an item is cached in the LDAP "
   "operation cache. Zero means no limit. Defaults to 600 seconds (10 minutes)."},

  {"AuthLDAPCacheCompareOps", auth_ldap_set_compare_flag, NULL, RSRC_CONF, TAKE1,
   "Set to no to disable caching of LDAP compare operations. Defaults to yes."},
 
#ifdef AUTH_LDAP_FRONTPAGE_HACK
  {"AuthLDAPFrontPageHack", ap_set_flag_slot,
   (void *)XtOffsetOf(auth_ldap_config_rec, frontpage_hack), OR_AUTHCFG, FLAG,
   "Set to 'on' to support Microsoft FrontPage"},

  { "AuthUserFile", auth_ldap_snarf_pwfile, NULL, OR_AUTHCFG, TAKE12, 
    "Not processed by auth_ldap; passed directly to other auth modules"},
#endif

  {NULL}
};

module auth_ldap_module = {
   STANDARD_MODULE_STUFF,
   NULL,			/* initializer */
   create_auth_ldap_dir_config,	/* dir config creater */
   NULL,			/* dir merger --- default is to override */
   create_auth_ldap_config,	/* server config */
   NULL,			/* merge server config */
   auth_ldap_cmds,		/* command table */
   NULL,			/* handlers */
   NULL,			/* filename translation */
   ldap_authenticate_basic_user, /* check_user_id */
   ldap_check_auth,		/* check auth */
   NULL,			/* check access */
   NULL,			/* type_checker */
   NULL,			/* fixups */
   NULL,			/* logger */
   NULL,			/* header parser */
   NULL,			/* child_init */
   NULL,			/* child_exit */
   NULL				/* post read-request */
};

#ifdef NEED_EPRINTF
/* More subroutines needed by GCC output code on some machines.  */
/* Compile this one with gcc.  */
/* Copyright (C) 1989, 1992, 1993, 1994, 1995 Free Software Foundation, Inc.

This file is part if libgcc2.c, which is part of GNU CC. Auth_ldap just
needs the _eprintf function.

GNU CC is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU CC; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#undef NULL /* Avoid errors if stdio.h and our stddef.h mismatch.  */
#include <stdio.h>
/* This is used by the `assert' macro.  */
void
__eprintf (string, expression, line, filename)
     const char *string;
     const char *expression;
     int line;
     const char *filename;
{
  fprintf (stderr, string, expression, line, filename);
  fflush (stderr);
  abort ();
}

#endif
